/**
 * @module ol/interaction/DragBox
 */
// FIXME draw drag box
import Event from '../events/Event.js';
import {always, mouseOnly, mouseActionButton} from '../events/condition.js';
import {VOID} from '../functions.js';
import PointerInteraction from './Pointer.js';
import RenderBox from '../render/Box.js';


/**
 * A function that takes a {@link module:ol/MapBrowserEvent} and two
 * {@link module:ol/pixel~Pixel}s and returns a `{boolean}`. If the condition is met,
 * true should be returned.
 * @typedef {function(this: ?, import("../MapBrowserEvent.js").default, import("../pixel.js").Pixel, import("../pixel.js").Pixel):boolean} EndCondition
 */


/**
 * @typedef {Object} Options
 * @property {string} [className='ol-dragbox'] CSS class name for styling the box.
 * @property {import("../events/condition.js").Condition} [condition] A function that takes an {@link module:ol/MapBrowserEvent~MapBrowserEvent} and returns a boolean
 * to indicate whether that event should be handled.
 * Default is {@link ol/events/condition~always}.
 * @property {number} [minArea=64] The minimum area of the box in pixel, this value is used by the default
 * `boxEndCondition` function.
 * @property {EndCondition} [boxEndCondition] A function that takes a {@link module:ol/MapBrowserEvent~MapBrowserEvent} and two
 * {@link module:ol/pixel~Pixel}s to indicate whether a `boxend` event should be fired.
 * Default is `true` if the area of the box is bigger than the `minArea` option.
 * @property {function(this:DragBox, import("../MapBrowserEvent.js").default)} onBoxEnd Code to execute just
 * before `boxend` is fired.
 */


/**
 * @enum {string}
 */
var DragBoxEventType = {
  /**
   * Triggered upon drag box start.
   * @event DragBoxEvent#boxstart
   * @api
   */
  BOXSTART: 'boxstart',

  /**
   * Triggered on drag when box is active.
   * @event DragBoxEvent#boxdrag
   * @api
   */
  BOXDRAG: 'boxdrag',

  /**
   * Triggered upon drag box end.
   * @event DragBoxEvent#boxend
   * @api
   */
  BOXEND: 'boxend'
};


/**
 * @classdesc
 * Events emitted by {@link module:ol/interaction/DragBox~DragBox} instances are instances of
 * this type.
 */
var DragBoxEvent = /*@__PURE__*/(function (Event) {
  function DragBoxEvent(type, coordinate, mapBrowserEvent) {
    Event.call(this, type);

    /**
     * The coordinate of the drag event.
     * @const
     * @type {import("../coordinate.js").Coordinate}
     * @api
     */
    this.coordinate = coordinate;

    /**
     * @const
     * @type {import("../MapBrowserEvent.js").default}
     * @api
     */
    this.mapBrowserEvent = mapBrowserEvent;

  }

  if ( Event ) DragBoxEvent.__proto__ = Event;
  DragBoxEvent.prototype = Object.create( Event && Event.prototype );
  DragBoxEvent.prototype.constructor = DragBoxEvent;

  return DragBoxEvent;
}(Event));


/**
 * @classdesc
 * Allows the user to draw a vector box by clicking and dragging on the map,
 * normally combined with an {@link module:ol/events/condition} that limits
 * it to when the shift or other key is held down. This is used, for example,
 * for zooming to a specific area of the map
 * (see {@link module:ol/interaction/DragZoom~DragZoom} and
 * {@link module:ol/interaction/DragRotateAndZoom}).
 *
 * This interaction is only supported for mouse devices.
 *
 * @fires DragBoxEvent
 * @api
 */
var DragBox = /*@__PURE__*/(function (PointerInteraction) {
  function DragBox(opt_options) {

    PointerInteraction.call(this);

    var options = opt_options ? opt_options : {};

    /**
    * @type {import("../render/Box.js").default}
    * @private
    */
    this.box_ = new RenderBox(options.className || 'ol-dragbox');

    /**
    * @type {number}
    * @private
    */
    this.minArea_ = options.minArea !== undefined ? options.minArea : 64;

    /**
     * Function to execute just before `onboxend` is fired
     * @type {function(this:DragBox, import("../MapBrowserEvent.js").default)}
     * @private
     */
    this.onBoxEnd_ = options.onBoxEnd ? options.onBoxEnd : VOID;

    /**
    * @type {import("../pixel.js").Pixel}
    * @private
    */
    this.startPixel_ = null;

    /**
    * @private
    * @type {import("../events/condition.js").Condition}
    */
    this.condition_ = options.condition ? options.condition : always;

    /**
    * @private
    * @type {EndCondition}
    */
    this.boxEndCondition_ = options.boxEndCondition ?
      options.boxEndCondition : this.defaultBoxEndCondition;
  }

  if ( PointerInteraction ) DragBox.__proto__ = PointerInteraction;
  DragBox.prototype = Object.create( PointerInteraction && PointerInteraction.prototype );
  DragBox.prototype.constructor = DragBox;

  /**
   * The default condition for determining whether the boxend event
   * should fire.
   * @param {import("../MapBrowserEvent.js").default} mapBrowserEvent The originating MapBrowserEvent
   *     leading to the box end.
   * @param {import("../pixel.js").Pixel} startPixel The starting pixel of the box.
   * @param {import("../pixel.js").Pixel} endPixel The end pixel of the box.
   * @return {boolean} Whether or not the boxend condition should be fired.
   */
  DragBox.prototype.defaultBoxEndCondition = function defaultBoxEndCondition (mapBrowserEvent, startPixel, endPixel) {
    var width = endPixel[0] - startPixel[0];
    var height = endPixel[1] - startPixel[1];
    return width * width + height * height >= this.minArea_;
  };

  /**
  * Returns geometry of last drawn box.
  * @return {import("../geom/Polygon.js").default} Geometry.
  * @api
  */
  DragBox.prototype.getGeometry = function getGeometry () {
    return this.box_.getGeometry();
  };

  /**
   * @inheritDoc
   */
  DragBox.prototype.handleDragEvent = function handleDragEvent (mapBrowserEvent) {
    if (!mouseOnly(mapBrowserEvent)) {
      return;
    }

    this.box_.setPixels(this.startPixel_, mapBrowserEvent.pixel);

    this.dispatchEvent(new DragBoxEvent(DragBoxEventType.BOXDRAG,
      mapBrowserEvent.coordinate, mapBrowserEvent));
  };

  /**
   * @inheritDoc
   */
  DragBox.prototype.handleUpEvent = function handleUpEvent (mapBrowserEvent) {
    if (!mouseOnly(mapBrowserEvent)) {
      return true;
    }

    this.box_.setMap(null);

    if (this.boxEndCondition_(mapBrowserEvent, this.startPixel_, mapBrowserEvent.pixel)) {
      this.onBoxEnd_(mapBrowserEvent);
      this.dispatchEvent(new DragBoxEvent(DragBoxEventType.BOXEND,
        mapBrowserEvent.coordinate, mapBrowserEvent));
    }
    return false;
  };

  /**
   * @inheritDoc
   */
  DragBox.prototype.handleDownEvent = function handleDownEvent (mapBrowserEvent) {
    if (!mouseOnly(mapBrowserEvent)) {
      return false;
    }

    if (mouseActionButton(mapBrowserEvent) &&
        this.condition_(mapBrowserEvent)) {
      this.startPixel_ = mapBrowserEvent.pixel;
      this.box_.setMap(mapBrowserEvent.map);
      this.box_.setPixels(this.startPixel_, this.startPixel_);
      this.dispatchEvent(new DragBoxEvent(DragBoxEventType.BOXSTART,
        mapBrowserEvent.coordinate, mapBrowserEvent));
      return true;
    } else {
      return false;
    }
  };

  return DragBox;
}(PointerInteraction));


export default DragBox;

//# sourceMappingURL=DragBox.js.map