var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
/**
 * @module ol/geom/Geometry
 */
import { abstract } from '../util.js';
import BaseObject from '../Object.js';
import { createEmpty, getHeight, returnOrUpdate } from '../extent.js';
import { transform2D } from './flat/transform.js';
import { get as getProjection, getTransform } from '../proj.js';
import Units from '../proj/Units.js';
import { create as createTransform, compose as composeTransform } from '../transform.js';
/**
 * @type {import("../transform.js").Transform}
 */
var tmpTransform = createTransform();
/**
 * @classdesc
 * Abstract base class; normally only used for creating subclasses and not
 * instantiated in apps.
 * Base class for vector geometries.
 *
 * To get notified of changes to the geometry, register a listener for the
 * generic `change` event on your geometry instance.
 *
 * @abstract
 * @api
 */
var Geometry = /** @class */ (function (_super) {
    __extends(Geometry, _super);
    function Geometry() {
        var _this = _super.call(this) || this;
        /**
         * @private
         * @type {import("../extent.js").Extent}
         */
        _this.extent_ = createEmpty();
        /**
         * @private
         * @type {number}
         */
        _this.extentRevision_ = -1;
        /**
         * @protected
         * @type {Object<string, Geometry>}
         */
        _this.simplifiedGeometryCache = {};
        /**
         * @protected
         * @type {number}
         */
        _this.simplifiedGeometryMaxMinSquaredTolerance = 0;
        /**
         * @protected
         * @type {number}
         */
        _this.simplifiedGeometryRevision = 0;
        return _this;
    }
    /**
     * Make a complete copy of the geometry.
     * @abstract
     * @return {!Geometry} Clone.
     */
    Geometry.prototype.clone = function () {
        return abstract();
    };
    /**
     * @abstract
     * @param {number} x X.
     * @param {number} y Y.
     * @param {import("../coordinate.js").Coordinate} closestPoint Closest point.
     * @param {number} minSquaredDistance Minimum squared distance.
     * @return {number} Minimum squared distance.
     */
    Geometry.prototype.closestPointXY = function (x, y, closestPoint, minSquaredDistance) {
        return abstract();
    };
    /**
     * @param {number} x X.
     * @param {number} y Y.
     * @return {boolean} Contains (x, y).
     */
    Geometry.prototype.containsXY = function (x, y) {
        var coord = this.getClosestPoint([x, y]);
        return coord[0] === x && coord[1] === y;
    };
    /**
     * Return the closest point of the geometry to the passed point as
     * {@link module:ol/coordinate~Coordinate coordinate}.
     * @param {import("../coordinate.js").Coordinate} point Point.
     * @param {import("../coordinate.js").Coordinate=} opt_closestPoint Closest point.
     * @return {import("../coordinate.js").Coordinate} Closest point.
     * @api
     */
    Geometry.prototype.getClosestPoint = function (point, opt_closestPoint) {
        var closestPoint = opt_closestPoint ? opt_closestPoint : [NaN, NaN];
        this.closestPointXY(point[0], point[1], closestPoint, Infinity);
        return closestPoint;
    };
    /**
     * Returns true if this geometry includes the specified coordinate. If the
     * coordinate is on the boundary of the geometry, returns false.
     * @param {import("../coordinate.js").Coordinate} coordinate Coordinate.
     * @return {boolean} Contains coordinate.
     * @api
     */
    Geometry.prototype.intersectsCoordinate = function (coordinate) {
        return this.containsXY(coordinate[0], coordinate[1]);
    };
    /**
     * @abstract
     * @param {import("../extent.js").Extent} extent Extent.
     * @protected
     * @return {import("../extent.js").Extent} extent Extent.
     */
    Geometry.prototype.computeExtent = function (extent) {
        return abstract();
    };
    /**
     * Get the extent of the geometry.
     * @param {import("../extent.js").Extent=} opt_extent Extent.
     * @return {import("../extent.js").Extent} extent Extent.
     * @api
     */
    Geometry.prototype.getExtent = function (opt_extent) {
        if (this.extentRevision_ != this.getRevision()) {
            this.extent_ = this.computeExtent(this.extent_);
            this.extentRevision_ = this.getRevision();
        }
        return returnOrUpdate(this.extent_, opt_extent);
    };
    /**
     * Rotate the geometry around a given coordinate. This modifies the geometry
     * coordinates in place.
     * @abstract
     * @param {number} angle Rotation angle in radians.
     * @param {import("../coordinate.js").Coordinate} anchor The rotation center.
     * @api
     */
    Geometry.prototype.rotate = function (angle, anchor) {
        abstract();
    };
    /**
     * Scale the geometry (with an optional origin).  This modifies the geometry
     * coordinates in place.
     * @abstract
     * @param {number} sx The scaling factor in the x-direction.
     * @param {number=} opt_sy The scaling factor in the y-direction (defaults to
     *     sx).
     * @param {import("../coordinate.js").Coordinate=} opt_anchor The scale origin (defaults to the center
     *     of the geometry extent).
     * @api
     */
    Geometry.prototype.scale = function (sx, opt_sy, opt_anchor) {
        abstract();
    };
    /**
     * Create a simplified version of this geometry.  For linestrings, this uses
     * the the {@link
     * https://en.wikipedia.org/wiki/Ramer-Douglas-Peucker_algorithm
     * Douglas Peucker} algorithm.  For polygons, a quantization-based
     * simplification is used to preserve topology.
     * @param {number} tolerance The tolerance distance for simplification.
     * @return {Geometry} A new, simplified version of the original geometry.
     * @api
     */
    Geometry.prototype.simplify = function (tolerance) {
        return this.getSimplifiedGeometry(tolerance * tolerance);
    };
    /**
     * Create a simplified version of this geometry using the Douglas Peucker
     * algorithm.
     * See https://en.wikipedia.org/wiki/Ramer-Douglas-Peucker_algorithm.
     * @abstract
     * @param {number} squaredTolerance Squared tolerance.
     * @return {Geometry} Simplified geometry.
     */
    Geometry.prototype.getSimplifiedGeometry = function (squaredTolerance) {
        return abstract();
    };
    /**
     * Get the type of this geometry.
     * @abstract
     * @return {import("./GeometryType.js").default} Geometry type.
     */
    Geometry.prototype.getType = function () {
        return abstract();
    };
    /**
     * Apply a transform function to each coordinate of the geometry.
     * The geometry is modified in place.
     * If you do not want the geometry modified in place, first `clone()` it and
     * then use this function on the clone.
     * @abstract
     * @param {import("../proj.js").TransformFunction} transformFn Transform.
     */
    Geometry.prototype.applyTransform = function (transformFn) {
        abstract();
    };
    /**
     * Test if the geometry and the passed extent intersect.
     * @abstract
     * @param {import("../extent.js").Extent} extent Extent.
     * @return {boolean} `true` if the geometry and the extent intersect.
     */
    Geometry.prototype.intersectsExtent = function (extent) {
        return abstract();
    };
    /**
     * Translate the geometry.  This modifies the geometry coordinates in place.  If
     * instead you want a new geometry, first `clone()` this geometry.
     * @abstract
     * @param {number} deltaX Delta X.
     * @param {number} deltaY Delta Y.
     * @api
     */
    Geometry.prototype.translate = function (deltaX, deltaY) {
        abstract();
    };
    /**
     * Transform each coordinate of the geometry from one coordinate reference
     * system to another. The geometry is modified in place.
     * For example, a line will be transformed to a line and a circle to a circle.
     * If you do not want the geometry modified in place, first `clone()` it and
     * then use this function on the clone.
     *
     * @param {import("../proj.js").ProjectionLike} source The current projection.  Can be a
     *     string identifier or a {@link module:ol/proj/Projection~Projection} object.
     * @param {import("../proj.js").ProjectionLike} destination The desired projection.  Can be a
     *     string identifier or a {@link module:ol/proj/Projection~Projection} object.
     * @return {Geometry} This geometry.  Note that original geometry is
     *     modified in place.
     * @api
     */
    Geometry.prototype.transform = function (source, destination) {
        /** @type {import("../proj/Projection.js").default} */
        var sourceProj = getProjection(source);
        var transformFn = sourceProj.getUnits() == Units.TILE_PIXELS ?
            function (inCoordinates, outCoordinates, stride) {
                var pixelExtent = sourceProj.getExtent();
                var projectedExtent = sourceProj.getWorldExtent();
                var scale = getHeight(projectedExtent) / getHeight(pixelExtent);
                composeTransform(tmpTransform, projectedExtent[0], projectedExtent[3], scale, -scale, 0, 0, 0);
                transform2D(inCoordinates, 0, inCoordinates.length, stride, tmpTransform, outCoordinates);
                return getTransform(sourceProj, destination)(inCoordinates, outCoordinates, stride);
            } :
            getTransform(sourceProj, destination);
        this.applyTransform(transformFn);
        return this;
    };
    return Geometry;
}(BaseObject));
export default Geometry;
//# sourceMappingURL=Geometry.js.map