/**
 * @module ol/pointer/MouseSource
 */
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
// Based on https://github.com/Polymer/PointerEvents
// Copyright (c) 2013 The Polymer Authors. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
// * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
// * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
import EventSource from './EventSource.js';
/**
 * @type {number}
 */
export var POINTER_ID = 1;
/**
 * @type {string}
 */
export var POINTER_TYPE = 'mouse';
/**
 * Radius around touchend that swallows mouse events.
 *
 * @type {number}
 */
var DEDUP_DIST = 25;
/**
 * Handler for `mousedown`.
 *
 * @this {MouseSource}
 * @param {MouseEvent} inEvent The in event.
 */
function mousedown(inEvent) {
    if (!this.isEventSimulatedFromTouch_(inEvent)) {
        // TODO(dfreedman) workaround for some elements not sending mouseup
        // http://crbug/149091
        if (POINTER_ID.toString() in this.pointerMap) {
            this.cancel(inEvent);
        }
        var e = prepareEvent(inEvent, this.dispatcher);
        this.pointerMap[POINTER_ID.toString()] = inEvent;
        this.dispatcher.down(e, inEvent);
    }
}
/**
 * Handler for `mousemove`.
 *
 * @this {MouseSource}
 * @param {MouseEvent} inEvent The in event.
 */
function mousemove(inEvent) {
    if (!this.isEventSimulatedFromTouch_(inEvent)) {
        var e = prepareEvent(inEvent, this.dispatcher);
        this.dispatcher.move(e, inEvent);
    }
}
/**
 * Handler for `mouseup`.
 *
 * @this {MouseSource}
 * @param {MouseEvent} inEvent The in event.
 */
function mouseup(inEvent) {
    if (!this.isEventSimulatedFromTouch_(inEvent)) {
        var p = this.pointerMap[POINTER_ID.toString()];
        if (p && p.button === inEvent.button) {
            var e = prepareEvent(inEvent, this.dispatcher);
            this.dispatcher.up(e, inEvent);
            this.cleanupMouse();
        }
    }
}
/**
 * Handler for `mouseover`.
 *
 * @this {MouseSource}
 * @param {MouseEvent} inEvent The in event.
 */
function mouseover(inEvent) {
    if (!this.isEventSimulatedFromTouch_(inEvent)) {
        var e = prepareEvent(inEvent, this.dispatcher);
        this.dispatcher.enterOver(e, inEvent);
    }
}
/**
 * Handler for `mouseout`.
 *
 * @this {MouseSource}
 * @param {MouseEvent} inEvent The in event.
 */
function mouseout(inEvent) {
    if (!this.isEventSimulatedFromTouch_(inEvent)) {
        var e = prepareEvent(inEvent, this.dispatcher);
        this.dispatcher.leaveOut(e, inEvent);
    }
}
var MouseSource = /** @class */ (function (_super) {
    __extends(MouseSource, _super);
    /**
     * @param {import("./PointerEventHandler.js").default} dispatcher Event handler.
     */
    function MouseSource(dispatcher) {
        var _this = this;
        var mapping = {
            'mousedown': mousedown,
            'mousemove': mousemove,
            'mouseup': mouseup,
            'mouseover': mouseover,
            'mouseout': mouseout
        };
        _this = _super.call(this, dispatcher, mapping) || this;
        /**
         * @const
         * @type {!Object<string, Event|Object>}
         */
        _this.pointerMap = dispatcher.pointerMap;
        /**
         * @const
         * @type {Array<import("../pixel.js").Pixel>}
         */
        _this.lastTouches = [];
        return _this;
    }
    /**
     * Detect if a mouse event was simulated from a touch by
     * checking if previously there was a touch event at the
     * same position.
     *
     * FIXME - Known problem with the native Android browser on
     * Samsung GT-I9100 (Android 4.1.2):
     * In case the page is scrolled, this function does not work
     * correctly when a canvas is used (WebGL or canvas renderer).
     * Mouse listeners on canvas elements (for this browser), create
     * two mouse events: One 'good' and one 'bad' one (on other browsers or
     * when a div is used, there is only one event). For the 'bad' one,
     * clientX/clientY and also pageX/pageY are wrong when the page
     * is scrolled. Because of that, this function can not detect if
     * the events were simulated from a touch event. As result, a
     * pointer event at a wrong position is dispatched, which confuses
     * the map interactions.
     * It is unclear, how one can get the correct position for the event
     * or detect that the positions are invalid.
     *
     * @private
     * @param {MouseEvent} inEvent The in event.
     * @return {boolean} True, if the event was generated by a touch.
     */
    MouseSource.prototype.isEventSimulatedFromTouch_ = function (inEvent) {
        var lts = this.lastTouches;
        var x = inEvent.clientX;
        var y = inEvent.clientY;
        for (var i = 0, l = lts.length, t = void 0; i < l && (t = lts[i]); i++) {
            // simulated mouse events will be swallowed near a primary touchend
            var dx = Math.abs(x - t[0]);
            var dy = Math.abs(y - t[1]);
            if (dx <= DEDUP_DIST && dy <= DEDUP_DIST) {
                return true;
            }
        }
        return false;
    };
    /**
     * Dispatches a `pointercancel` event.
     *
     * @param {Event} inEvent The in event.
     */
    MouseSource.prototype.cancel = function (inEvent) {
        var e = prepareEvent(inEvent, this.dispatcher);
        this.dispatcher.cancel(e, inEvent);
        this.cleanupMouse();
    };
    /**
     * Remove the mouse from the list of active pointers.
     */
    MouseSource.prototype.cleanupMouse = function () {
        delete this.pointerMap[POINTER_ID.toString()];
    };
    return MouseSource;
}(EventSource));
/**
 * Creates a copy of the original event that will be used
 * for the fake pointer event.
 *
 * @param {Event} inEvent The in event.
 * @param {import("./PointerEventHandler.js").default} dispatcher Event handler.
 * @return {Object} The copied event.
 */
export function prepareEvent(inEvent, dispatcher) {
    var e = dispatcher.cloneEvent(inEvent, inEvent);
    // forward mouse preventDefault
    var pd = e.preventDefault;
    e.preventDefault = function () {
        inEvent.preventDefault();
        pd();
    };
    e.pointerId = POINTER_ID;
    e.isPrimary = true;
    e.pointerType = POINTER_TYPE;
    return e;
}
export default MouseSource;
//# sourceMappingURL=MouseSource.js.map