var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
/**
 * @module ol/layer/BaseVector
 */
import Layer from './Layer.js';
import RBush from 'rbush';
import { assign } from '../obj.js';
import { createDefaultStyle, toFunction as toStyleFunction, } from '../style/Style.js';
/**
 * @template {import("../source/Vector.js").default|import("../source/VectorTile.js").default} VectorSourceType
 * @typedef {Object} Options
 * @property {string} [className='ol-layer'] A CSS class name to set to the layer element.
 * @property {number} [opacity=1] Opacity (0, 1).
 * @property {boolean} [visible=true] Visibility.
 * @property {import("../extent.js").Extent} [extent] The bounding extent for layer rendering.  The layer will not be
 * rendered outside of this extent.
 * @property {number} [zIndex] The z-index for layer rendering.  At rendering time, the layers
 * will be ordered, first by Z-index and then by position. When `undefined`, a `zIndex` of 0 is assumed
 * for layers that are added to the map's `layers` collection, or `Infinity` when the layer's `setMap()`
 * method was used.
 * @property {number} [minResolution] The minimum resolution (inclusive) at which this layer will be
 * visible.
 * @property {number} [maxResolution] The maximum resolution (exclusive) below which this layer will
 * be visible.
 * @property {number} [minZoom] The minimum view zoom level (exclusive) above which this layer will be
 * visible.
 * @property {number} [maxZoom] The maximum view zoom level (inclusive) at which this layer will
 * be visible.
 * @property {import("../render.js").OrderFunction} [renderOrder] Render order. Function to be used when sorting
 * features before rendering. By default features are drawn in the order that they are created. Use
 * `null` to avoid the sort, but get an undefined draw order.
 * @property {number} [renderBuffer=100] The buffer in pixels around the viewport extent used by the
 * renderer when getting features from the vector source for the rendering or hit-detection.
 * Recommended value: the size of the largest symbol, line width or label.
 * @property {VectorSourceType} [source] Source.
 * @property {import("../PluggableMap.js").default} [map] Sets the layer as overlay on a map. The map will not manage
 * this layer in its layers collection, and the layer will be rendered on top. This is useful for
 * temporary layers. The standard way to add a layer to a map and have it managed by the map is to
 * use {@link import("../PluggableMap.js").default#addLayer map.addLayer()}.
 * @property {boolean} [declutter=false] Declutter images and text. Decluttering is applied to all
 * image and text styles of all Vector and VectorTile layers that have set this to `true`. The priority
 * is defined by the z-index of the layer, the `zIndex` of the style and the render order of features.
 * Higher z-index means higher priority. Within the same z-index, a feature rendered before another has
 * higher priority.
 *
 * As an optimization decluttered features from layers with the same `className` are rendered above
 * the fill and stroke styles of all of those layers regardless of z-index.  To opt out of this
 * behavior and place declutterd features with their own layer configure the layer with a `className`
 * other than `ol-layer`.
 * @property {import("../style/Style.js").StyleLike|null} [style] Layer style. When set to `null`, only
 * features that have their own style will be rendered. See {@link module:ol/style/Style~Style} for the default style
 * which will be used if this is not set.
 * @property {import("./Base.js").BackgroundColor} [background] Background color for the layer. If not specified, no background
 * will be rendered.
 * @property {boolean} [updateWhileAnimating=false] When set to `true`, feature batches will
 * be recreated during animations. This means that no vectors will be shown clipped, but the
 * setting will have a performance impact for large amounts of vector data. When set to `false`,
 * batches will be recreated when no animation is active.
 * @property {boolean} [updateWhileInteracting=false] When set to `true`, feature batches will
 * be recreated during interactions. See also `updateWhileAnimating`.
 * @property {Object<string, *>} [properties] Arbitrary observable properties. Can be accessed with `#get()` and `#set()`.
 */
/**
 * @enum {string}
 * @private
 */
var Property = {
    RENDER_ORDER: 'renderOrder',
};
/**
 * @classdesc
 * Vector data that is rendered client-side.
 * Note that any property set in the options is set as a {@link module:ol/Object~BaseObject}
 * property on the layer object; for example, setting `title: 'My Title'` in the
 * options means that `title` is observable, and has get/set accessors.
 *
 * @template {import("../source/Vector.js").default|import("../source/VectorTile.js").default} VectorSourceType
 * @template {import("../renderer/canvas/VectorLayer.js").default|import("../renderer/canvas/VectorTileLayer.js").default|import("../renderer/canvas/VectorImageLayer.js").default|import("../renderer/webgl/PointsLayer.js").default} RendererType
 * @extends {Layer<VectorSourceType, RendererType>}
 * @api
 */
var BaseVectorLayer = /** @class */ (function (_super) {
    __extends(BaseVectorLayer, _super);
    /**
     * @param {Options<VectorSourceType>} [opt_options] Options.
     */
    function BaseVectorLayer(opt_options) {
        var _this = this;
        var options = opt_options ? opt_options : {};
        var baseOptions = assign({}, options);
        delete baseOptions.style;
        delete baseOptions.renderBuffer;
        delete baseOptions.updateWhileAnimating;
        delete baseOptions.updateWhileInteracting;
        _this = _super.call(this, baseOptions) || this;
        /**
         * @private
         * @type {boolean}
         */
        _this.declutter_ =
            options.declutter !== undefined ? options.declutter : false;
        /**
         * @type {number}
         * @private
         */
        _this.renderBuffer_ =
            options.renderBuffer !== undefined ? options.renderBuffer : 100;
        /**
         * User provided style.
         * @type {import("../style/Style.js").StyleLike}
         * @private
         */
        _this.style_ = null;
        /**
         * Style function for use within the library.
         * @type {import("../style/Style.js").StyleFunction|undefined}
         * @private
         */
        _this.styleFunction_ = undefined;
        _this.setStyle(options.style);
        /**
         * @type {boolean}
         * @private
         */
        _this.updateWhileAnimating_ =
            options.updateWhileAnimating !== undefined
                ? options.updateWhileAnimating
                : false;
        /**
         * @type {boolean}
         * @private
         */
        _this.updateWhileInteracting_ =
            options.updateWhileInteracting !== undefined
                ? options.updateWhileInteracting
                : false;
        return _this;
    }
    /**
     * @return {boolean} Declutter.
     */
    BaseVectorLayer.prototype.getDeclutter = function () {
        return this.declutter_;
    };
    /**
     * Get the topmost feature that intersects the given pixel on the viewport. Returns a promise
     * that resolves with an array of features. The array will either contain the topmost feature
     * when a hit was detected, or it will be empty.
     *
     * The hit detection algorithm used for this method is optimized for performance, but is less
     * accurate than the one used in {@link import("../PluggableMap.js").default#getFeaturesAtPixel}: Text
     * is not considered, and icons are only represented by their bounding box instead of the exact
     * image.
     *
     * @param {import("../pixel.js").Pixel} pixel Pixel.
     * @return {Promise<Array<import("../Feature").default>>} Promise that resolves with an array of features.
     * @api
     */
    BaseVectorLayer.prototype.getFeatures = function (pixel) {
        return _super.prototype.getFeatures.call(this, pixel);
    };
    /**
     * @return {number|undefined} Render buffer.
     */
    BaseVectorLayer.prototype.getRenderBuffer = function () {
        return this.renderBuffer_;
    };
    /**
     * @return {function(import("../Feature.js").default, import("../Feature.js").default): number|null|undefined} Render
     *     order.
     */
    BaseVectorLayer.prototype.getRenderOrder = function () {
        return /** @type {import("../render.js").OrderFunction|null|undefined} */ (this.get(Property.RENDER_ORDER));
    };
    /**
     * Get the style for features.  This returns whatever was passed to the `style`
     * option at construction or to the `setStyle` method.
     * @return {import("../style/Style.js").StyleLike|null|undefined} Layer style.
     * @api
     */
    BaseVectorLayer.prototype.getStyle = function () {
        return this.style_;
    };
    /**
     * Get the style function.
     * @return {import("../style/Style.js").StyleFunction|undefined} Layer style function.
     * @api
     */
    BaseVectorLayer.prototype.getStyleFunction = function () {
        return this.styleFunction_;
    };
    /**
     * @return {boolean} Whether the rendered layer should be updated while
     *     animating.
     */
    BaseVectorLayer.prototype.getUpdateWhileAnimating = function () {
        return this.updateWhileAnimating_;
    };
    /**
     * @return {boolean} Whether the rendered layer should be updated while
     *     interacting.
     */
    BaseVectorLayer.prototype.getUpdateWhileInteracting = function () {
        return this.updateWhileInteracting_;
    };
    /**
     * Render declutter items for this layer
     * @param {import("../PluggableMap.js").FrameState} frameState Frame state.
     */
    BaseVectorLayer.prototype.renderDeclutter = function (frameState) {
        if (!frameState.declutterTree) {
            frameState.declutterTree = new RBush(9);
        }
        /** @type {*} */ (this.getRenderer()).renderDeclutter(frameState);
    };
    /**
     * @param {import("../render.js").OrderFunction|null|undefined} renderOrder
     *     Render order.
     */
    BaseVectorLayer.prototype.setRenderOrder = function (renderOrder) {
        this.set(Property.RENDER_ORDER, renderOrder);
    };
    /**
     * Set the style for features.  This can be a single style object, an array
     * of styles, or a function that takes a feature and resolution and returns
     * an array of styles. If set to `null`, the layer has no style (a `null` style),
     * so only features that have their own styles will be rendered in the layer. Call
     * `setStyle()` without arguments to reset to the default style. See
     * {@link module:ol/style/Style~Style} for information on the default style.
     * @param {import("../style/Style.js").StyleLike|null} [opt_style] Layer style.
     * @api
     */
    BaseVectorLayer.prototype.setStyle = function (opt_style) {
        this.style_ = opt_style !== undefined ? opt_style : createDefaultStyle;
        this.styleFunction_ =
            opt_style === null ? undefined : toStyleFunction(this.style_);
        this.changed();
    };
    return BaseVectorLayer;
}(Layer));
export default BaseVectorLayer;
//# sourceMappingURL=BaseVector.js.map