import { Host, h } from '@stencil/core';
/**
 * Component to manage a text field that can be edited. The component has two view modes: "Preview" and "Edit".
 * In preview mode, it simply displays the {@link OntotextEditableTextField#value}.
 * In edit mode, the component displays an input element with the value {@link OntotextEditableTextField#value}, along with save and cancel buttons.
 */
export class OntotextEditableTextField {
  constructor() {
    /**
     * There is an issue when the component is opened programmatically, for instance, when it is opened from a context menu. In such cases,
     * the handler {@link OntotextEditableTextField#handleClickOutside} checks if the event is outside the component and takes an action.
     * This action is normally executed when the component is open and the user clicks outside the component.
     * This parameter is used to control when the first click event should be skipped.
     */
    this.skipFirstClickOutside = true;
    this.value = undefined;
    this.edit = false;
    this.translationService = undefined;
  }
  /**
   * Handler for changing view mode of component. When this happened a componentModeChanged even will be fired.
   *
   * @param edit - the new view mode.
   */
  handleEditModeChange(edit) {
    this.edit = edit;
    this.editedValue = this.value;
    this.componentModeChanged.emit(this.edit);
  }
  /**
   * Click handler. Saves {@link OntotextEditableTextField#value} if the click is outside component and component is in edit mode.
   */
  handleClickOutside(event) {
    // Nothing to do if the component is in preview mode or the component is not opened from a double click on host element.
    if (!this.edit || this.skipFirstClickOutside) {
      this.skipFirstClickOutside = false;
      return;
    }
    const clickedElement = event.target;
    // Check if the clicked element is outside the component
    if (!this.hostElement.contains(clickedElement) && this.inputElement && !this.inputElement.contains(clickedElement)) {
      this.save();
      this.cancel();
    }
  }
  keydownListener(ev) {
    if (!this.edit) {
      return;
    }
    if ('Enter' === ev.key) {
      this.save();
    }
    else if ('Escape' === ev.key) {
      this.cancel();
    }
  }
  /**
   * Initializes the field that holds edited value when component is loaded.
   */
  componentDidLoad() {
    this.editedValue = this.value;
  }
  /**
   * Sets the input element on focus if element is on edit mode when component is updated.
   */
  componentDidUpdate() {
    if (this.edit && this.inputElement) {
      this.inputElement.focus();
      this.inputElement.select();
    }
  }
  /**
   * Saves changed value of the text component for eventual future saving.
   */
  handleValueChanged(event) {
    this.editedValue = event.target.value;
  }
  /**
   * Switches component to edit mode.
   */
  openEditMode() {
    this.edit = true;
    this.editedValue = this.value;
    this.skipFirstClickOutside = true;
  }
  /**
   * Saves the value of text field and emits "valueChanged" event with the new value.
   */
  save() {
    const isValueChanged = this.value !== this.editedValue;
    this.value = this.editedValue;
    this.edit = false;
    if (isValueChanged) {
      this.valueChanged.emit(this.value);
    }
  }
  /**
   * Switches the component in preview mode.
   */
  cancel() {
    this.edit = false;
    this.editedValue = this.value;
  }
  render() {
    return (h(Host, { onDblClick: this.openEditMode, class: 'editable-text-field-wrapper' }, !this.edit && h("div", { class: 'preview-value' }, this.value), this.edit && (h("div", { class: 'edit-mode-container' }, h("input", { class: 'editable-text-field', type: "text", value: this.editedValue, onInput: (event) => this.handleValueChanged(event), ref: (el) => (this.inputElement = el) }), h("button", { onClick: () => this.save(), class: 'save-btn', title: (this.translationService ? this.translationService.translate('yasqe.tab_list.tab_rename.save.btn.label') : '') }, h("span", { class: 'save-btn-label icon-tick' })), h("button", { onClick: () => this.cancel(), class: 'cancel-btn', title: (this.translationService ? this.translationService.translate('yasqe.tab_list.tab_rename.cancel.btn.label') : '') }, h("span", { class: 'cancel-btn-label icon-close' }))))));
  }
  static get is() { return "ontotext-editable-text-field"; }
  static get originalStyleUrls() {
    return {
      "$": ["ontotext-editable-text-field.scss"]
    };
  }
  static get styleUrls() {
    return {
      "$": ["ontotext-editable-text-field.css"]
    };
  }
  static get properties() {
    return {
      "value": {
        "type": "string",
        "mutable": true,
        "complexType": {
          "original": "string",
          "resolved": "string",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "The value of the text field."
        },
        "attribute": "value",
        "reflect": false
      },
      "edit": {
        "type": "boolean",
        "mutable": true,
        "complexType": {
          "original": "boolean",
          "resolved": "boolean",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "Controls the view mode of component. If true the component will be in Edit mode."
        },
        "attribute": "edit",
        "reflect": false,
        "defaultValue": "false"
      },
      "translationService": {
        "type": "unknown",
        "mutable": false,
        "complexType": {
          "original": "TranslationService",
          "resolved": "TranslationService",
          "references": {
            "TranslationService": {
              "location": "import",
              "path": "../../services/translation.service"
            }
          }
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": ""
        }
      }
    };
  }
  static get events() {
    return [{
        "method": "valueChanged",
        "name": "valueChanged",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "The \"valueChanged\" event is fired when the text field value changes."
        },
        "complexType": {
          "original": "string",
          "resolved": "string",
          "references": {}
        }
      }, {
        "method": "componentModeChanged",
        "name": "componentModeChanged",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "The \"componentModeChanged\" event is fired when the view mode changes."
        },
        "complexType": {
          "original": "boolean",
          "resolved": "boolean",
          "references": {}
        }
      }];
  }
  static get elementRef() { return "hostElement"; }
  static get watchers() {
    return [{
        "propName": "edit",
        "methodName": "handleEditModeChange"
      }];
  }
  static get listeners() {
    return [{
        "name": "click",
        "method": "handleClickOutside",
        "target": "window",
        "capture": false,
        "passive": false
      }, {
        "name": "keydown",
        "method": "keydownListener",
        "target": "document",
        "capture": false,
        "passive": false
      }];
  }
}
