import { h, Host } from '@stencil/core';
import { defaultOntotextYasguiConfig, RenderingMode } from '../../models/yasgui-configuration';
import { YASGUI_MIN_SCRIPT } from '../yasgui/yasgui-script';
import { YasguiBuilder } from '../../services/yasgui/yasgui-builder';
import { VisualisationUtils } from '../../services/utils/visualisation-utils';
import { HtmlElementsUtil } from '../../services/utils/html-elements-util';
import { OntotextYasguiService } from '../../services/yasgui/ontotext-yasgui-service';
import { TranslationService } from '../../services/translation.service';
import { ServiceFactory } from '../../services/service-factory';
import { YasguiConfigurationBuilder } from '../../services/yasgui/configuration/yasgui-configuration-builder';
import { toOutputEvent } from '../../models/output-events/output-event';
import { NotificationMessageService } from '../../services/notification-message-service';
import { MessageCode } from '../../models/internal-events/internal-notification-message-event';
import { YasqeService } from '../../services/yasqe/yasqe-service';
import { YasrService } from '../../services/yasr/yasr-service';
import { PivotTablePlugin } from '../../plugins/yasr/pivot-table/pivot-table-plugin';
import { ChartsPlugin } from "../../plugins/yasr/charts/charts-plugin";
import { SavedQueryOpened } from '../../models/output-events/saved-query-opened';
import { Debounce } from "../../services/utils/debounce";
/**
 * This is the custom web component which is adapter for the yasgui library. It allows as to
 * configure and extend the library without potentially breaking the component clients.
 *
 * The component have some sane defaults for most of its configurations. So, in practice, it can be
 * used as is by providing just the sparql endpoint config.
 * For other customizations, the default configurations can be overridden by providing an external
 * configuration object compliant with the <code>ExternalYasguiConfiguration</code> interface to the
 * component.
 *
 * There is a configuration watcher which triggers the initialization again after a change is
 * detected.
 *
 * During the component initialization, the provided external configuration is passed down to a
 * configuration builder which use it to override and extend the yasgui library defaults.
 *
 * After the configuration is ready, then a yasgui instance is created with it.
 *
 * After the yasgui instance is ready, then a post initialization phase begins. During the phase the
 * yasgui can be tweaked using the values from the configuration.
 */
export class OntotextYasguiWebComponent {
  constructor() {
    this.defaultViewMode = RenderingMode.YASGUI;
    this.subscriptions = [];
    this.config = undefined;
    this.language = undefined;
    this.savedQueryConfig = undefined;
    this.saveQueryData = undefined;
    this.savedQueryData = undefined;
    this.deleteQueryData = undefined;
    this.isVerticalOrientation = true;
    this.renderingMode = this.getRenderMode();
    this.showSaveQueryDialog = false;
    this.showSavedQueriesPopup = false;
    this.showSavedQueriesPopupTarget = undefined;
    this.showConfirmationDialog = false;
    this.showShareQueryDialog = false;
    this.showCopyResourceLinkDialog = false;
    this.copiedResourceLink = undefined;
    this.loaderMessage = undefined;
    this.additionalLoaderMessage = undefined;
    this.yasqeBtnLabel = undefined;
    this.yasguiBtnLabel = undefined;
    this.yasrBtnLabel = undefined;
    this.serviceFactory = new ServiceFactory(this.hostElement);
    this.translationService = this.serviceFactory.get(TranslationService);
    this.yasguiConfigurationBuilder = this.serviceFactory.get(YasguiConfigurationBuilder);
    this.yasguiBuilder = this.serviceFactory.get(YasguiBuilder);
    this.ontotextYasguiService = this.serviceFactory.get(OntotextYasguiService);
    this.notificationMessageService = this.serviceFactory.get(NotificationMessageService);
    this.initComponentTranslationHandlers();
  }
  configurationChanged(newConfig) {
    this.init(newConfig);
  }
  languageChanged(newLang) {
    this.translationService.setLanguage(newLang);
    this.getOntotextYasgui()
      .then((ontotextYasgui) => {
      ontotextYasgui.refresh();
    });
  }
  savedQueryConfigChanged() {
    this.shouldShowSaveQueryDialog();
    this.shouldShowSavedQueriesPopup();
    this.saveQueryData = this.initSaveQueryData();
  }
  onOrientationChanged() {
    if (this.isVerticalOrientation) {
      this.unregisterTabsListResizeObserver();
      this.updateYasrTopMargin();
    }
    else {
      this.registerTabsListResizeObserver();
    }
  }
  /**
   * Changes rendering mode of component.
   *
   * @param newRenderMode - then new render mode of component.
   * @param editorHeight - the height for yasqe.
   */
  changeRenderMode(newRenderMode, editorHeight) {
    return this.getOntotextYasgui()
      .then(() => {
      VisualisationUtils.changeRenderMode(this.hostElement, newRenderMode, this.isVerticalOrientation, editorHeight);
    });
  }
  /**
   * Allows the client to set a query in the current opened tab.
   * The cursor position is preserved.
   *
   * @param query The query that should be set in the current focused tab.
   */
  setQuery(query) {
    return this.getOntotextYasgui()
      .then(() => {
      this.ontotextYasgui.setQuery(query);
    });
  }
  /**
   * Executes the YASQE query from the currently opened tab and switches to the specified <code>renderingMode</code> when the query is executed.
   *
   * @param renderingMode - specifies the new view mode of the component when the query is executed.
   */
  query(renderingMode = RenderingMode.YASGUI) {
    return this.getOntotextYasgui()
      .then((ontotextYasgui) => {
      this.defaultViewMode = renderingMode;
      return ontotextYasgui.query();
    });
  }
  /**
   * Fetches the query from YASQE editor.
   */
  getQuery() {
    return this.getOntotextYasgui()
      .then(() => {
      return this.ontotextYasgui.getQuery();
    });
  }
  /**
   * Checks if query is valid.
   */
  isQueryValid() {
    return this.getOntotextYasgui()
      .then(() => {
      return this.ontotextYasgui.isQueryValid();
    });
  }
  /**
   * Allows the client to init the editor using a query model. When the query and query name are
   * found in any existing opened tab, then it'd be focused. Otherwise a new tab will be created and
   * initialized using the provided query model.
   * @param queryModel The query model.
   */
  openTab(queryModel) {
    // While this does the job in this particular method, we definitely need a more general approach
    // for handling the fact that there is a chance for the client to hit the problem where when the
    // OntotextYasgui instance is created and returned the wrapped Yasgui instance might not be yet
    // initialized.
    return this.getOntotextYasgui()
      .then((ontotextYasgui) => ontotextYasgui.openTab(queryModel));
  }
  /**
   * Utility method allowing the client to get the mode of the query which is written in the current
   * editor tab.
   * The query mode can be either `query` or `update` regarding the query mode. This method just
   * exposes the similar utility method from the yasqe component.
   *
   * @return A promise which resolves with a string representing the query mode.
   */
  getQueryMode() {
    return this.getOntotextYasgui().then((ontotextYasgui) => {
      return ontotextYasgui.getQueryMode();
    });
  }
  /**
   * Utility method allowing the client to get the type of the query which is written in the current
   * editor tab.
   * The query mode can be `INSERT`, `LOAD`, `CLEAR`, `DELETE`, etc. This method just exposes the
   * similar utility method from the yasqe component.
   *
   * @return A promise which resolves with a string representing the query type.
   */
  getQueryType() {
    return this.getOntotextYasgui().then((ontotextYasgui) => {
      return ontotextYasgui.getQueryType();
    });
  }
  /**
   * Fetches the query result and return it as JSON.
   */
  getEmbeddedResultAsJson() {
    return this.getOntotextYasgui().then((ontotextYasgui) => {
      return ontotextYasgui.getEmbeddedResultAsJson();
    });
  }
  /**
   * Fetches the query result and return it as CSV.
   */
  getEmbeddedResultAsCSV() {
    return this.getOntotextYasgui().then((ontotextYasgui) => {
      return ontotextYasgui.getEmbeddedResultAsCSV();
    });
  }
  /**
   * Hides the YASQE action button with the name <code>yasqeActionButtonNames</code>.
   *
   * @param yasqeActionButtonNames - the name of the action that needs to be hidden.
   */
  hideYasqeActionButton(yasqeActionButtonNames) {
    return this.getOntotextYasgui().then(() => YasqeService.hideYasqeActionButtons(yasqeActionButtonNames));
  }
  /**
   * Shows the YASQE action button with the name <code>yasqeActionButtonNames</code>.
   *
   * @param yasqeActionButtonNames - the name of the action that needs to be displayed.
   */
  showYasqeActionButton(yasqeActionButtonNames) {
    return this.getOntotextYasgui().then(() => YasqeService.showYasqeActionButtons(yasqeActionButtonNames));
  }
  /**
   * Fetches info about ongoing requests.
   */
  getOngoingRequestsInfo() {
    return this.getOntotextYasgui()
      .then((ontotextYasgui) => {
      return ontotextYasgui.getOngoingRequestsInfo();
    });
  }
  /**
   * Aborts the running query if any.
   */
  abortQuery() {
    return this.getOntotextYasgui()
      .then((ontotextYasgui) => {
      ontotextYasgui.abortQuery();
    });
  }
  /**
   * Aborts all running requests.
   */
  abortAllRequests() {
    return this.getOntotextYasgui()
      .then((ontotextYasgui) => {
      ontotextYasgui.abortAllRequests();
    });
  }
  /**
   * Reinitializes the Yasgui component by resetting the defined flags to those in the configuration.
   * Clears the results of the query.
   * @param resetFlags - object defining the flags for what to reset
   */
  reInitYasgui(resetFlags) {
    return this.getOntotextYasgui()
      .then((ontotextYasgui) => {
      ontotextYasgui.reInitYasgui(resetFlags);
    });
  }
  /**
   * Checks whether the query has been modified after the initialization of the YASQE editor.
   */
  isQueryDirty() {
    return this.getOntotextYasgui()
      .then((ontotextYasgui) => ontotextYasgui.isQueryDirty());
  }
  /**
   * There are rendering problems when the window size is changed. To address this, we added a listener to the window resize event.
   * When the event occurs, we refresh the component to recalculate and resolve the rendering issues.
   */
  onResize() {
    this.getOntotextYasgui()
      .then((ontotextYasgui) => {
      ontotextYasgui.refresh();
      const editorHeight = this.ontotextYasgui.getEditorHeight(ontotextYasgui.getTabId());
      VisualisationUtils.setYasqeFullHeight(this.renderingMode, VisualisationUtils.resolveOrientation(this.isVerticalOrientation), editorHeight);
    });
  }
  onBeforeunloadHandler() {
    this.getOntotextYasgui()
      .then((ontotextYasgui) => {
      if (ontotextYasgui.isQueryDirty()) {
        ontotextYasgui.saveQuery();
      }
    });
  }
  /**
   * Handler for the event fired when the save query button in the yasqe toolbar is triggered.
   */
  saveQueryHandler() {
    this.showSaveQueryDialog = true;
    this.saveQueryData = this.getDefaultSaveQueryData();
  }
  /**
   * Handler for the event fired when the query should be saved by the component client.
   */
  createSavedQueryHandler(event) {
    this.createSavedQuery.emit(event.detail);
  }
  /**
   * Handler for the event fired when the query should be saved by the component client.
   */
  updateSavedQueryHandler(event) {
    this.updateSavedQuery.emit(event.detail);
  }
  /**
   * Handler for the event fired when the saveQueryDialog gets closed.
   */
  closeSaveDialogHandler() {
    this.showSaveQueryDialog = false;
  }
  /**
   * Handler for the event fired when the show saved queries button in the yasqe toolbar is triggered.
   */
  showSavedQueriesHandler(event) {
    this.loadSavedQueries.emit(true);
    this.showSavedQueriesPopup = true;
    this.showSavedQueriesPopupTarget = event.detail.buttonInstance;
  }
  /**
   * Handles the event fired when the saved query is selected from the saved queries list. Tries to
   * find a tab with the same name and query inside and opens it. Otherwise opens a new tab with
   * provided parameters. When a saved query is loaded into a tab, emit the 'saveQueryOpened' event with the tab instance.
   */
  savedQuerySelectedHandler(event) {
    const queryData = event.detail;
    this.showSavedQueriesPopup = false;
    this.openTab({
      queryName: queryData.queryName,
      query: queryData.query,
      owner: undefined
    })
      .then((tab) => this.saveQueryOpened.emit(new SavedQueryOpened(tab)));
  }
  /**
   * Handler for the event fired when the edit saved query button is triggered.
   */
  editSavedQueryHandler(event) {
    this.savedQueryData = event.detail;
    this.showSavedQueriesPopup = false;
    this.showSaveQueryDialog = true;
  }
  /**
   * Handler for the event fired when the delete saved query button is triggered.
   */
  savedQuerySelectedForEditHandler(event) {
    this.showConfirmationDialog = true;
    this.deleteQueryData = event.detail;
  }
  /**
   * Handler for confirmation event fired when deletion of saved query is approved.
   */
  deleteSavedQueryHandler() {
    this.showConfirmationDialog = false;
    this.showSavedQueriesPopup = false;
    this.deleteSavedQuery.emit(this.deleteQueryData);
  }
  /**
   * Handler for the event for closing the saved query delete confirmation dialog.
   */
  closeSavedQueryDeleteConfirmationHandler() {
    this.showConfirmationDialog = false;
    this.showSavedQueriesPopup = false;
    this.deleteQueryData = undefined;
  }
  /**
   * Handler for the event for closing the saved queries popup.
   */
  closeSavedQueriesPopupHandler() {
    this.showSavedQueriesPopup = false;
  }
  /**
   * Handler for the event fired when the share saved query button is triggered.
   */
  savedQuerySelectedForShareHandler(event) {
    this.shareSavedQuery.emit(event.detail);
    this.showShareQueryDialog = true;
  }
  /**
   * Handler for the event fired when the share query button in the editor is triggered.
   */
  shareQueryHandler() {
    const query = this.ontotextYasgui.getQuery();
    const queryName = this.ontotextYasgui.getTabName();
    this.shareQuery.emit({
      queryName: queryName,
      query: query,
      owner: ''
    });
    this.showShareQueryDialog = true;
  }
  /**
   * Handler for the internal event fired when a query share link is copied in the clipboard.
   */
  savedQueryShareLinkCopiedHandler() {
    this.showShareQueryDialog = false;
    this.queryShareLinkCopied.emit();
  }
  /**
   * Handler for the event for closing the share query dialog.
   */
  closeShareQueryDialogHandler() {
    this.showShareQueryDialog = false;
  }
  /**
   * Handler for the event fired when the copy link dialog is closed without copying the link to the clipboard.
   */
  resourceLinkDialogClosedHandler() {
    this.showCopyResourceLinkDialog = false;
    this.copiedResourceLink = undefined;
  }
  /**
   * Handler for the event fired when the copy link dialog is closed when the link is copied to the clipboard.
   */
  resourceLinkCopiedHandler() {
    const resourceCopiedMessage = this.translationService.translate('yasqe.share.copy_link.dialog.copy.message.success');
    this.notificationMessageService.success(MessageCode.RESOURCE_LINK_COPIED_SUCCESSFULLY, resourceCopiedMessage);
    this.showCopyResourceLinkDialog = false;
    this.copiedResourceLink = undefined;
  }
  /**
   * Handler for the event fired when the copy link dialog is closed when the link is copied to the clipboard.
   */
  showResourceCopyLinkDialogHandler(event) {
    this.copiedResourceLink = event.detail.payload.copyLink;
    this.showCopyResourceLinkDialog = true;
  }
  onDownloadAsEventHandler(downloadAsEvent) {
    this.output.emit(toOutputEvent(downloadAsEvent));
  }
  notificationMessageHandler(event) {
    this.output.emit(toOutputEvent(event));
  }
  onQuery(event) {
    this.output.emit(toOutputEvent(event));
    this.getOntotextYasgui()
      .then((ontotextYasgui) => {
      ontotextYasgui.leaveFullScreen();
      const editorHeight = this.ontotextYasgui.getEditorHeight(ontotextYasgui.getTabId());
      this.changeRenderMode(this.defaultViewMode, editorHeight);
      this.defaultViewMode = RenderingMode.YASGUI;
      VisualisationUtils.setYasqeFullHeight(this.renderingMode, VisualisationUtils.resolveOrientation(this.isVerticalOrientation), editorHeight);
    });
  }
  onInternalQueryExecuted(event) {
    const queryExecuted = toOutputEvent(event);
    this.output.emit(queryExecuted);
  }
  onCountQuery(event) {
    this.output.emit(toOutputEvent(event));
  }
  onCountQueryResponse(event) {
    this.output.emit(toOutputEvent(event));
  }
  onQueryAborted(event) {
    this.output.emit(toOutputEvent(event));
  }
  resolveOrientationButtonTooltip() {
    return this.isVerticalOrientation ?
      this.translationService.translate('yasgui.toolbar.orientation.btn.tooltip.switch_orientation_horizontal') :
      this.translationService.translate('yasgui.toolbar.orientation.btn.tooltip.switch_orientation_vertical');
  }
  changeOrientation() {
    this.isVerticalOrientation = !this.isVerticalOrientation;
    this.getOntotextYasgui()
      .then((ontotextYasgui) => {
      const editorHeight = this.ontotextYasgui.getEditorHeight(ontotextYasgui.getTabId());
      VisualisationUtils.toggleLayoutOrientation(this.hostElement, this.isVerticalOrientation, this.renderingMode, editorHeight);
      ontotextYasgui.refresh();
    });
  }
  getDefaultSaveQueryData() {
    return {
      queryName: '',
      query: '',
      owner: '',
      isPublic: false,
      messages: []
    };
  }
  initSaveQueryData() {
    var _a;
    return {
      queryName: '',
      query: '',
      owner: '',
      isPublic: false,
      messages: ((_a = this.savedQueryConfig) === null || _a === void 0 ? void 0 : _a.errorMessage) || []
    };
  }
  getSaveQueryData() {
    const data = this.saveQueryData || this.getDefaultSaveQueryData();
    // first take into account if there is a saved query selected for edit
    // then take the data from the currently opened yasgui which means a new unsaved yet query
    if (this.savedQueryData) {
      data.queryName = this.savedQueryData.queryName;
      data.query = this.savedQueryData.query;
      data.isPublic = this.savedQueryData.isPublic;
      data.isNew = false;
    }
    else if (this.ontotextYasgui) {
      data.queryName = this.ontotextYasgui.getTabName();
      data.query = this.ontotextYasgui.getTabQuery();
      data.isPublic = false;
      data.isNew = true;
    }
    data.messages = this.saveQueryData && this.saveQueryData.messages;
    return data;
  }
  getSaveQueriesData() {
    const data = {
      savedQueriesList: [],
      popupTarget: this.showSavedQueriesPopupTarget
    };
    if (this.savedQueryConfig && this.savedQueryConfig.savedQueries) {
      data.savedQueriesList = this.savedQueryConfig.savedQueries.map((savedQuery) => {
        return {
          queryName: savedQuery.queryName,
          query: savedQuery.query,
          isPublic: savedQuery.isPublic,
          owner: savedQuery.owner
        };
      });
    }
    return data;
  }
  getRenderMode() {
    return this.config && this.config.render || defaultOntotextYasguiConfig.render;
  }
  getOrientationMode() {
    return this.config && this.config.orientation || defaultOntotextYasguiConfig.orientation;
  }
  shouldShowSaveQueryDialog() {
    this.showSaveQueryDialog = this.showSaveQueryDialog && !this.isSavedQuerySaved();
  }
  isSavedQuerySaved() {
    return this.savedQueryConfig && this.savedQueryConfig.saveSuccess;
  }
  shouldShowSavedQueriesPopup() {
    var _a, _b;
    this.showSavedQueriesPopup = this.showSavedQueriesPopup && (((_a = this.savedQueryConfig) === null || _a === void 0 ? void 0 : _a.savedQueries) && ((_b = this.savedQueryConfig) === null || _b === void 0 ? void 0 : _b.savedQueries.length) > 0);
  }
  getDeleteQueryConfirmationConfig() {
    return {
      title: this.translationService.translate('yasqe.actions.delete_saved_query.confirm.dialog.label'),
      message: this.translationService.translate('yasqe.actions.delete_saved_query.confirm.dialog.message', [{
          key: 'query',
          value: this.deleteQueryData.queryName
        }])
    };
  }
  getShareLinkDialogConfig() {
    var _a;
    return {
      shareQueryLink: (_a = this.savedQueryConfig) === null || _a === void 0 ? void 0 : _a.shareQueryLink
    };
  }
  getOntotextYasgui() {
    return new Promise((resolve, reject) => {
      if (this.isOntotextYasguiInitialiazed()) {
        return resolve(this.ontotextYasgui);
      }
      let maxIterationsToComplete = 15;
      const timer = setInterval(() => {
        maxIterationsToComplete--;
        if (this.isOntotextYasguiInitialiazed()) {
          clearInterval(timer);
          return resolve(this.ontotextYasgui);
        }
        if (maxIterationsToComplete === 0) {
          clearInterval(timer);
          return reject(`Can't initialize Yasgui!`);
        }
      }, 100);
    });
  }
  changeRenderingMode(mode) {
    this.renderingMode = mode;
    const editorHeight = this.ontotextYasgui.getEditorHeight(this.ontotextYasgui.getTabId());
    VisualisationUtils.changeRenderMode(this.hostElement, mode, this.isVerticalOrientation, editorHeight);
  }
  isOntotextYasguiInitialiazed() {
    return !!this.ontotextYasgui && !!this.ontotextYasgui.getInstance();
  }
  afterInit() {
    // Configure the web component
    this.ontotextYasguiService.postConstruct(this.hostElement, this.ontotextYasgui.getConfig());
    if (this.ontotextYasgui.getConfig().yasguiConfig.clearState) {
      this.ontotextYasgui.reInitYasgui({
        resetSameAs: true,
        resetCurrentTab: true,
        resetYasr: true,
        resetInferResults: true
      });
    }
  }
  registerEventHandlers() {
    const hint = HtmlElementsUtil.createAutocompleteHintElement(this.translationService.translate('yasqe.autocomplete.hint'));
    this.ontotextYasgui.getInstance().on('tabAdd', (_yasgui, _tab) => {
      VisualisationUtils.changeRenderMode(this.hostElement, this.renderingMode, this.isVerticalOrientation);
    });
    this.ontotextYasgui.getInstance().on('tabSelect', (_yasgui, _tab) => {
      const editorHeight = this.ontotextYasgui.getEditorHeight(_tab);
      VisualisationUtils.changeRenderMode(this.hostElement, this.renderingMode, this.isVerticalOrientation, editorHeight);
      this.updateYasrTopMargin();
    });
    this.ontotextYasgui.getInstance().on('autocompletionShown', (_instance, _tab, _widget) => {
      hint.parentNode && hint.parentNode.removeChild(hint);
      const codemirrorHints = document.querySelector('.CodeMirror-hints');
      const elRect = codemirrorHints.getBoundingClientRect();
      // We don't use boundingClientRect because it is not accurate in case yasgui is two column layout and dropdown
      // appears to have incorrect position.
      const hintsTop = codemirrorHints.style.top.substring(0, codemirrorHints.style.top.length - 2) * 1;
      codemirrorHints.style.top = hintsTop + 14 + 'px';
      document.body.appendChild(hint);
      const topPosition = elRect.top - 6;
      let leftPosition = elRect.right - hint.offsetWidth;
      leftPosition = leftPosition < elRect.left ? elRect.left : leftPosition - 12;
      hint.style.top = `${topPosition}px`;
      hint.style.left = `${leftPosition}px`;
    });
    this.ontotextYasgui.getInstance().on('autocompletionClose', (_instance, _tab) => {
      hint.parentNode && hint.parentNode.removeChild(hint);
    });
  }
  updateYasrTopMargin() {
    const yasrElement = this.hostElement.querySelector('.tabPanel.active .yasr');
    if (yasrElement) {
      yasrElement.parentElement.style.marginTop = this.isVerticalOrientation ? '10px' : `-${this.tabsListHeight}px`;
    }
  }
  registerTabsListResizeObserver() {
    this.unregisterTabsListResizeObserver();
    const debouncedResizeHandler = Debounce.createDebouncedFunction((entries) => {
      this.tabsListHeight = entries[0].contentRect.height;
      this.updateYasrTopMargin();
    }, 10);
    this.tabsListResizeObserver = new ResizeObserver((entries) => {
      debouncedResizeHandler(entries);
    });
    const tabsListElement = this.hostElement.querySelector('.yasgui .tabsList');
    if (tabsListElement) {
      this.tabsListResizeObserver.observe(tabsListElement);
    }
  }
  unregisterTabsListResizeObserver() {
    if (this.tabsListResizeObserver) {
      this.tabsListResizeObserver.disconnect();
      this.tabsListResizeObserver = undefined;
    }
  }
  destroy() {
    this.unregisterTabsListResizeObserver();
    if (this.ontotextYasgui) {
      this.ontotextYasgui.destroy();
      const yasgui = HtmlElementsUtil.getOntotextYasgui(this.hostElement);
      while (yasgui.firstChild) {
        yasgui.firstChild.remove();
      }
    }
  }
  initComponentTranslationHandlers() {
    this.subscriptions.push(this.translationService.onTranslate('yasgui.toolbar.mode_yasqe.btn.label', (yasqeBtnLabel) => {
      this.yasqeBtnLabel = yasqeBtnLabel;
    }));
    this.subscriptions.push(this.translationService.onTranslate('yasgui.toolbar.mode_yasgui.btn.label', (yasguiBtnLabel) => {
      this.yasguiBtnLabel = yasguiBtnLabel;
    }));
    this.subscriptions.push(this.translationService.onTranslate('yasgui.toolbar.mode_yasr.btn.label', (yasrBtnLabel) => {
      this.yasrBtnLabel = yasrBtnLabel;
    }));
  }
  init(externalConfiguration) {
    if (!HtmlElementsUtil.getOntotextYasgui(this.hostElement) || !externalConfiguration) {
      return;
    }
    this.destroy();
    // @ts-ignore
    if (window.Yasgui) {
      this.translationService.setLanguage(externalConfiguration.language);
      // * Build the internal yasgui configuration using the provided external configuration
      const yasguiConfiguration = this.yasguiConfigurationBuilder.build(externalConfiguration);
      // * Build a yasgui instance using the configuration
      YasrService.registerPlugin(PivotTablePlugin.PLUGIN_NAME, PivotTablePlugin);
      YasrService.registerPlugin(ChartsPlugin.PLUGIN_NAME, ChartsPlugin);
      this.ontotextYasgui = this.yasguiBuilder.build(this.hostElement, yasguiConfiguration);
      this.registerEventHandlers();
      this.afterInit();
      const editorHeight = this.ontotextYasgui.getEditorHeight(this.ontotextYasgui.getTabId());
      VisualisationUtils.setYasqeFullHeight(this.renderingMode, VisualisationUtils.resolveOrientation(this.isVerticalOrientation), editorHeight);
    }
  }
  componentWillLoad() {
    // @ts-ignore
    if (!window.Yasgui) {
      // Load the yasgui script once.
      YASGUI_MIN_SCRIPT();
    }
    this.translationService.setLanguage(this.language);
  }
  componentDidLoad() {
    // As documentation said "The @Watch() decorator does not fire when a component initially loads."
    // yasgui instance will not be created if we set configuration when component is loaded, which
    // will be most case of the component usage. So we call the method manually when component is
    // loaded. More info https://github.com/TriplyDB/Yasgui/issues/143
    this.init(this.config);
  }
  disconnectedCallback() {
    if (this.subscriptions) {
      this.subscriptions.forEach((subscription) => subscription());
      this.subscriptions = [];
    }
    this.destroy();
  }
  render() {
    const classList = `yasgui-host-element ${this.getOrientationMode()} ${this.getRenderMode()}`;
    return (h(Host, { class: classList }, h("div", { class: "yasgui-toolbar hidden" }, h("button", { class: "yasgui-btn btn-mode-yasqe", onClick: () => this.changeRenderingMode(RenderingMode.YASQE) }, this.yasqeBtnLabel), h("button", { class: "yasgui-btn btn-mode-yasgui", onClick: () => this.changeRenderingMode(RenderingMode.YASGUI) }, this.yasguiBtnLabel), h("button", { class: "yasgui-btn btn-mode-yasr", onClick: () => this.changeRenderingMode(RenderingMode.YASR) }, this.yasrBtnLabel), h("yasgui-tooltip", { "data-tooltip": this.resolveOrientationButtonTooltip(), placement: "left", "show-on-click": true }, h("button", { class: "btn-orientation icon-columns red", onClick: () => this.changeOrientation() }, "\u00A0"))), h("div", { class: "ontotext-yasgui" }), this.showSaveQueryDialog &&
      h("save-query-dialog", { data: this.getSaveQueryData(), serviceFactory: this.serviceFactory }, "\u00A0"), this.showSavedQueriesPopup &&
      h("saved-queries-popup", { config: this.getSaveQueriesData(), serviceFactory: this.serviceFactory }), this.showConfirmationDialog &&
      h("confirmation-dialog", { translationService: this.translationService, config: this.getDeleteQueryConfirmationConfig() }), this.showShareQueryDialog &&
      h("share-query-dialog", { serviceFactory: this.serviceFactory, config: this.getShareLinkDialogConfig() }), this.showCopyResourceLinkDialog &&
      h("copy-resource-link-dialog", { serviceFactory: this.serviceFactory, resourceLink: this.copiedResourceLink })));
  }
  static get is() { return "ontotext-yasgui"; }
  static get originalStyleUrls() {
    return {
      "$": ["ontotext-yasgui-web-component.scss"]
    };
  }
  static get styleUrls() {
    return {
      "$": ["ontotext-yasgui-web-component.css"]
    };
  }
  static get properties() {
    return {
      "config": {
        "type": "unknown",
        "mutable": false,
        "complexType": {
          "original": "ExternalYasguiConfiguration",
          "resolved": "ExternalYasguiConfiguration",
          "references": {
            "ExternalYasguiConfiguration": {
              "location": "import",
              "path": "../../models/external-yasgui-configuration"
            }
          }
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "An input object property containing the yasgui configuration."
        }
      },
      "language": {
        "type": "string",
        "mutable": false,
        "complexType": {
          "original": "string",
          "resolved": "string",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "An input property containing the chosen translation language."
        },
        "attribute": "language",
        "reflect": false
      },
      "savedQueryConfig": {
        "type": "unknown",
        "mutable": false,
        "complexType": {
          "original": "SavedQueryConfig",
          "resolved": "SavedQueryConfig",
          "references": {
            "SavedQueryConfig": {
              "location": "import",
              "path": "../../models/saved-query-configuration"
            }
          }
        },
        "required": false,
        "optional": true,
        "docs": {
          "tags": [],
          "text": "A configuration model related with all the saved queries actions."
        }
      }
    };
  }
  static get states() {
    return {
      "saveQueryData": {},
      "savedQueryData": {},
      "deleteQueryData": {},
      "isVerticalOrientation": {},
      "renderingMode": {},
      "showSaveQueryDialog": {},
      "showSavedQueriesPopup": {},
      "showSavedQueriesPopupTarget": {},
      "showConfirmationDialog": {},
      "showShareQueryDialog": {},
      "showCopyResourceLinkDialog": {},
      "copiedResourceLink": {},
      "loaderMessage": {},
      "additionalLoaderMessage": {},
      "yasqeBtnLabel": {},
      "yasguiBtnLabel": {},
      "yasrBtnLabel": {}
    };
  }
  static get events() {
    return [{
        "method": "createSavedQuery",
        "name": "createSavedQuery",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Event emitted when saved query payload is collected and the query should be saved by the\ncomponent client."
        },
        "complexType": {
          "original": "SaveQueryData",
          "resolved": "SaveQueryData",
          "references": {
            "SaveQueryData": {
              "location": "import",
              "path": "../../models/saved-query-configuration"
            }
          }
        }
      }, {
        "method": "updateSavedQuery",
        "name": "updateSavedQuery",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Event emitted when a query payload is updated and the query name is the same as the one being\nedited. In result the client must perform a query update."
        },
        "complexType": {
          "original": "SaveQueryData",
          "resolved": "SaveQueryData",
          "references": {
            "SaveQueryData": {
              "location": "import",
              "path": "../../models/saved-query-configuration"
            }
          }
        }
      }, {
        "method": "saveQueryOpened",
        "name": "saveQueryOpened",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Event emitted when a saved query is loaded into a tab."
        },
        "complexType": {
          "original": "SavedQueryOpened",
          "resolved": "SavedQueryOpened",
          "references": {
            "SavedQueryOpened": {
              "location": "import",
              "path": "../../models/output-events/saved-query-opened"
            }
          }
        }
      }, {
        "method": "deleteSavedQuery",
        "name": "deleteSavedQuery",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Event emitted when a saved query should be deleted. In result the client must perform a query\ndelete."
        },
        "complexType": {
          "original": "SaveQueryData",
          "resolved": "SaveQueryData",
          "references": {
            "SaveQueryData": {
              "location": "import",
              "path": "../../models/saved-query-configuration"
            }
          }
        }
      }, {
        "method": "loadSavedQueries",
        "name": "loadSavedQueries",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Event emitted when saved queries is expected to be loaded by the component client and provided\nback in order to be displayed."
        },
        "complexType": {
          "original": "boolean",
          "resolved": "boolean",
          "references": {}
        }
      }, {
        "method": "shareSavedQuery",
        "name": "shareSavedQuery",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Event emitted when saved query share link has to be build by the client."
        },
        "complexType": {
          "original": "SaveQueryData",
          "resolved": "SaveQueryData",
          "references": {
            "SaveQueryData": {
              "location": "import",
              "path": "../../models/saved-query-configuration"
            }
          }
        }
      }, {
        "method": "queryShareLinkCopied",
        "name": "queryShareLinkCopied",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Event emitted when query share link gets copied in the clipboard."
        },
        "complexType": {
          "original": "any",
          "resolved": "any",
          "references": {}
        }
      }, {
        "method": "shareQuery",
        "name": "shareQuery",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Event emitted when saved query share link has to be build by the client."
        },
        "complexType": {
          "original": "TabQueryModel",
          "resolved": "TabQueryModel",
          "references": {
            "TabQueryModel": {
              "location": "import",
              "path": "../../models/external-yasgui-configuration"
            }
          }
        }
      }, {
        "method": "output",
        "name": "output",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Event emitter used to send message to the clients of component."
        },
        "complexType": {
          "original": "OutputEvent",
          "resolved": "OutputEvent",
          "references": {
            "OutputEvent": {
              "location": "import",
              "path": "../../models/output-events/output-event"
            }
          }
        }
      }];
  }
  static get methods() {
    return {
      "changeRenderMode": {
        "complexType": {
          "signature": "(newRenderMode: RenderingMode, editorHeight?: number) => Promise<void>",
          "parameters": [{
              "tags": [{
                  "name": "param",
                  "text": "newRenderMode - then new render mode of component."
                }],
              "text": "- then new render mode of component."
            }, {
              "tags": [{
                  "name": "param",
                  "text": "editorHeight - the height for yasqe."
                }],
              "text": "- the height for yasqe."
            }],
          "references": {
            "Promise": {
              "location": "global"
            },
            "RenderingMode": {
              "location": "import",
              "path": "../../models/yasgui-configuration"
            }
          },
          "return": "Promise<void>"
        },
        "docs": {
          "text": "Changes rendering mode of component.",
          "tags": [{
              "name": "param",
              "text": "newRenderMode - then new render mode of component."
            }, {
              "name": "param",
              "text": "editorHeight - the height for yasqe."
            }]
        }
      },
      "setQuery": {
        "complexType": {
          "signature": "(query: string) => Promise<void>",
          "parameters": [{
              "tags": [{
                  "name": "param",
                  "text": "query The query that should be set in the current focused tab."
                }],
              "text": "The query that should be set in the current focused tab."
            }],
          "references": {
            "Promise": {
              "location": "global"
            }
          },
          "return": "Promise<void>"
        },
        "docs": {
          "text": "Allows the client to set a query in the current opened tab.\nThe cursor position is preserved.",
          "tags": [{
              "name": "param",
              "text": "query The query that should be set in the current focused tab."
            }]
        }
      },
      "query": {
        "complexType": {
          "signature": "(renderingMode?: RenderingMode) => Promise<any>",
          "parameters": [{
              "tags": [{
                  "name": "param",
                  "text": "renderingMode - specifies the new view mode of the component when the query is executed."
                }],
              "text": "- specifies the new view mode of the component when the query is executed."
            }],
          "references": {
            "Promise": {
              "location": "global"
            },
            "RenderingMode": {
              "location": "import",
              "path": "../../models/yasgui-configuration"
            }
          },
          "return": "Promise<any>"
        },
        "docs": {
          "text": "Executes the YASQE query from the currently opened tab and switches to the specified <code>renderingMode</code> when the query is executed.",
          "tags": [{
              "name": "param",
              "text": "renderingMode - specifies the new view mode of the component when the query is executed."
            }]
        }
      },
      "getQuery": {
        "complexType": {
          "signature": "() => Promise<string>",
          "parameters": [],
          "references": {
            "Promise": {
              "location": "global"
            }
          },
          "return": "Promise<string>"
        },
        "docs": {
          "text": "Fetches the query from YASQE editor.",
          "tags": []
        }
      },
      "isQueryValid": {
        "complexType": {
          "signature": "() => Promise<boolean>",
          "parameters": [],
          "references": {
            "Promise": {
              "location": "global"
            }
          },
          "return": "Promise<boolean>"
        },
        "docs": {
          "text": "Checks if query is valid.",
          "tags": []
        }
      },
      "openTab": {
        "complexType": {
          "signature": "(queryModel: TabQueryModel) => Promise<Tab>",
          "parameters": [{
              "tags": [{
                  "name": "param",
                  "text": "queryModel The query model."
                }],
              "text": "The query model."
            }],
          "references": {
            "Promise": {
              "location": "global"
            },
            "Tab": {
              "location": "import",
              "path": "../../models/yasgui/tab"
            },
            "TabQueryModel": {
              "location": "import",
              "path": "../../models/external-yasgui-configuration"
            }
          },
          "return": "Promise<Tab>"
        },
        "docs": {
          "text": "Allows the client to init the editor using a query model. When the query and query name are\nfound in any existing opened tab, then it'd be focused. Otherwise a new tab will be created and\ninitialized using the provided query model.",
          "tags": [{
              "name": "param",
              "text": "queryModel The query model."
            }]
        }
      },
      "getQueryMode": {
        "complexType": {
          "signature": "() => Promise<string>",
          "parameters": [],
          "references": {
            "Promise": {
              "location": "global"
            }
          },
          "return": "Promise<string>"
        },
        "docs": {
          "text": "Utility method allowing the client to get the mode of the query which is written in the current\neditor tab.\nThe query mode can be either `query` or `update` regarding the query mode. This method just\nexposes the similar utility method from the yasqe component.",
          "tags": [{
              "name": "return",
              "text": "A promise which resolves with a string representing the query mode."
            }]
        }
      },
      "getQueryType": {
        "complexType": {
          "signature": "() => Promise<string>",
          "parameters": [],
          "references": {
            "Promise": {
              "location": "global"
            }
          },
          "return": "Promise<string>"
        },
        "docs": {
          "text": "Utility method allowing the client to get the type of the query which is written in the current\neditor tab.\nThe query mode can be `INSERT`, `LOAD`, `CLEAR`, `DELETE`, etc. This method just exposes the\nsimilar utility method from the yasqe component.",
          "tags": [{
              "name": "return",
              "text": "A promise which resolves with a string representing the query type."
            }]
        }
      },
      "getEmbeddedResultAsJson": {
        "complexType": {
          "signature": "() => Promise<unknown>",
          "parameters": [],
          "references": {
            "Promise": {
              "location": "global"
            }
          },
          "return": "Promise<unknown>"
        },
        "docs": {
          "text": "Fetches the query result and return it as JSON.",
          "tags": []
        }
      },
      "getEmbeddedResultAsCSV": {
        "complexType": {
          "signature": "() => Promise<unknown>",
          "parameters": [],
          "references": {
            "Promise": {
              "location": "global"
            }
          },
          "return": "Promise<unknown>"
        },
        "docs": {
          "text": "Fetches the query result and return it as CSV.",
          "tags": []
        }
      },
      "hideYasqeActionButton": {
        "complexType": {
          "signature": "(yasqeActionButtonNames: YasqeButtonType | YasqeButtonType[]) => Promise<void>",
          "parameters": [{
              "tags": [{
                  "name": "param",
                  "text": "yasqeActionButtonNames - the name of the action that needs to be hidden."
                }],
              "text": "- the name of the action that needs to be hidden."
            }],
          "references": {
            "Promise": {
              "location": "global"
            },
            "YasqeButtonType": {
              "location": "import",
              "path": "../../models/yasqe-button-name"
            }
          },
          "return": "Promise<void>"
        },
        "docs": {
          "text": "Hides the YASQE action button with the name <code>yasqeActionButtonNames</code>.",
          "tags": [{
              "name": "param",
              "text": "yasqeActionButtonNames - the name of the action that needs to be hidden."
            }]
        }
      },
      "showYasqeActionButton": {
        "complexType": {
          "signature": "(yasqeActionButtonNames: YasqeButtonType | YasqeButtonType[]) => Promise<void>",
          "parameters": [{
              "tags": [{
                  "name": "param",
                  "text": "yasqeActionButtonNames - the name of the action that needs to be displayed."
                }],
              "text": "- the name of the action that needs to be displayed."
            }],
          "references": {
            "Promise": {
              "location": "global"
            },
            "YasqeButtonType": {
              "location": "import",
              "path": "../../models/yasqe-button-name"
            }
          },
          "return": "Promise<void>"
        },
        "docs": {
          "text": "Shows the YASQE action button with the name <code>yasqeActionButtonNames</code>.",
          "tags": [{
              "name": "param",
              "text": "yasqeActionButtonNames - the name of the action that needs to be displayed."
            }]
        }
      },
      "getOngoingRequestsInfo": {
        "complexType": {
          "signature": "() => Promise<OngoingRequestsInfo>",
          "parameters": [],
          "references": {
            "Promise": {
              "location": "global"
            },
            "OngoingRequestsInfo": {
              "location": "import",
              "path": "../../models/ongoing-requests-info"
            }
          },
          "return": "Promise<OngoingRequestsInfo>"
        },
        "docs": {
          "text": "Fetches info about ongoing requests.",
          "tags": []
        }
      },
      "abortQuery": {
        "complexType": {
          "signature": "() => Promise<any>",
          "parameters": [],
          "references": {
            "Promise": {
              "location": "global"
            }
          },
          "return": "Promise<any>"
        },
        "docs": {
          "text": "Aborts the running query if any.",
          "tags": []
        }
      },
      "abortAllRequests": {
        "complexType": {
          "signature": "() => Promise<any>",
          "parameters": [],
          "references": {
            "Promise": {
              "location": "global"
            }
          },
          "return": "Promise<any>"
        },
        "docs": {
          "text": "Aborts all running requests.",
          "tags": []
        }
      },
      "reInitYasgui": {
        "complexType": {
          "signature": "(resetFlags: YasguiResetFlags) => Promise<any>",
          "parameters": [{
              "tags": [{
                  "name": "param",
                  "text": "resetFlags - object defining the flags for what to reset"
                }],
              "text": "- object defining the flags for what to reset"
            }],
          "references": {
            "Promise": {
              "location": "global"
            },
            "YasguiResetFlags": {
              "location": "import",
              "path": "../../models/yasgui/yasgui-reset-flags"
            }
          },
          "return": "Promise<any>"
        },
        "docs": {
          "text": "Reinitializes the Yasgui component by resetting the defined flags to those in the configuration.\nClears the results of the query.",
          "tags": [{
              "name": "param",
              "text": "resetFlags - object defining the flags for what to reset"
            }]
        }
      },
      "isQueryDirty": {
        "complexType": {
          "signature": "() => Promise<boolean>",
          "parameters": [],
          "references": {
            "Promise": {
              "location": "global"
            }
          },
          "return": "Promise<boolean>"
        },
        "docs": {
          "text": "Checks whether the query has been modified after the initialization of the YASQE editor.",
          "tags": []
        }
      }
    };
  }
  static get elementRef() { return "hostElement"; }
  static get watchers() {
    return [{
        "propName": "config",
        "methodName": "configurationChanged"
      }, {
        "propName": "language",
        "methodName": "languageChanged"
      }, {
        "propName": "savedQueryConfig",
        "methodName": "savedQueryConfigChanged"
      }, {
        "propName": "isVerticalOrientation",
        "methodName": "onOrientationChanged"
      }];
  }
  static get listeners() {
    return [{
        "name": "resize",
        "method": "onResize",
        "target": "window",
        "capture": false,
        "passive": true
      }, {
        "name": "beforeunload",
        "method": "onBeforeunloadHandler",
        "target": "window",
        "capture": false,
        "passive": false
      }, {
        "name": "internalCreateSavedQueryEvent",
        "method": "saveQueryHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalSaveQueryEvent",
        "method": "createSavedQueryHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalUpdateQueryEvent",
        "method": "updateSavedQueryHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalSaveQueryDialogClosedEvent",
        "method": "closeSaveDialogHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalShowSavedQueriesEvent",
        "method": "showSavedQueriesHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalSaveQuerySelectedEvent",
        "method": "savedQuerySelectedHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalEditSavedQueryEvent",
        "method": "editSavedQueryHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalSavedQuerySelectedForDeleteEvent",
        "method": "savedQuerySelectedForEditHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalConfirmationApprovedEvent",
        "method": "deleteSavedQueryHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalConfirmationRejectedEvent",
        "method": "closeSavedQueryDeleteConfirmationHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalCloseSavedQueriesPopupEvent",
        "method": "closeSavedQueriesPopupHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalSavedQuerySelectedForShareEvent",
        "method": "savedQuerySelectedForShareHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalShareQueryEvent",
        "method": "shareQueryHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalQueryShareLinkCopiedEvent",
        "method": "savedQueryShareLinkCopiedHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalShareQueryDialogClosedEvent",
        "method": "closeShareQueryDialogHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalResourceLinkDialogClosedEvent",
        "method": "resourceLinkDialogClosedHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalResourceLinkCopiedEvent",
        "method": "resourceLinkCopiedHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalShowResourceCopyLinkDialogEvent",
        "method": "showResourceCopyLinkDialogHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalDownloadAsEvent",
        "method": "onDownloadAsEventHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalNotificationMessageEvent",
        "method": "notificationMessageHandler",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalQueryEvent",
        "method": "onQuery",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalQueryExecuted",
        "method": "onInternalQueryExecuted",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalCountQueryEvent",
        "method": "onCountQuery",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalCountQueryResponseEvent",
        "method": "onCountQueryResponse",
        "target": undefined,
        "capture": false,
        "passive": false
      }, {
        "name": "internalRequestAbortedEvent",
        "method": "onQueryAborted",
        "target": undefined,
        "capture": false,
        "passive": false
      }];
  }
}
