import { h, Host } from '@stencil/core';
import { TranslationService } from "../../services/translation.service";
import { SaveQueryData, UpdateQueryData } from "../../models/saved-query-configuration";
import { HtmlUtil } from '../../services/utils/html-util';
export class SaveQueryDialog {
  constructor() {
    this.serviceFactory = undefined;
    this.data = undefined;
    this.queryName = '';
    this.query = '';
    this.isPublic = false;
    this.isNew = false;
    this.isSaveAllowed = false;
  }
  /**
   * Handles the Escape key keydown event and closes the dialog.
   * @param ev The keyboard event.
   */
  keydownListener(ev) {
    if (ev.key === 'Escape') {
      this.internalSaveQueryDialogClosedEvent.emit();
    }
  }
  componentWillLoad() {
    // TranslationService is injected here because the service factory is not available
    // in the constructor.
    this.translationService = this.serviceFactory.get(TranslationService);
    if (this.data) {
      this.queryName = this.data.queryName || this.queryName;
      this.query = this.data.query || this.query;
      this.isPublic = this.data.isPublic !== undefined ? this.data.isPublic : this.isPublic;
      this.isNew = this.data.isNew !== undefined ? this.data.isNew : this.isNew;
      this.resolveIsSaveAllowed();
    }
  }
  componentDidLoad() {
    this.documentOverflow = HtmlUtil.hideDocumentBodyOverflow();
    this.hostElement.addEventListener('keydown', this.preventLeavingDialog.bind(this));
    this.cancelButton.focus();
  }
  disconnectedCallback() {
    HtmlUtil.setDocumentBodyOverflow(this.documentOverflow);
    this.hostElement.removeEventListener('keydown', this.preventLeavingDialog.bind(this));
  }
  onClose(evt) {
    const target = evt.target;
    evt.stopPropagation();
    const isOverlay = target.classList.contains('dialog-overlay');
    const isCloseButton = target.classList.contains('close-button');
    const isCancelButton = target.classList.contains('cancel-button');
    if (isOverlay || isCloseButton || isCancelButton) {
      this.internalSaveQueryDialogClosedEvent.emit();
    }
  }
  onCreate(evt) {
    evt.stopPropagation();
    const queryName = this.queryName;
    const query = this.query;
    const isPublic = this.isPublic;
    if (this.isNew) {
      this.internalSaveQueryEvent.emit(new SaveQueryData(queryName, query, isPublic));
    }
    else {
      this.internalUpdateQueryEvent.emit(new UpdateQueryData(queryName, query, isPublic));
    }
  }
  handleQueryNameChange(event) {
    this.queryName = event.target.value;
    this.resolveIsSaveAllowed();
  }
  handleQueryChange(event) {
    this.query = event.target.value;
    this.resolveIsSaveAllowed();
  }
  handleIsPublicChange(event) {
    this.isPublic = event.target.checked;
  }
  preventLeavingDialog(ev) {
    HtmlUtil.preventLeavingDialog(this.hostElement, ev);
  }
  resolveIsSaveAllowed() {
    this.isSaveAllowed = !!(this.queryName.trim().length && this.query.trim().length);
  }
  getErrorMessages() {
    const errorMessages = [];
    if (this.hasMissingQueryName()) {
      errorMessages.push(this.translationService.translate('yasqe.actions.save_query.dialog.query_name.empty_error'));
    }
    if (this.hasMissingQuery()) {
      errorMessages.push(this.translationService.translate('yasqe.actions.save_query.dialog.query.empty_error'));
    }
    if (this.data.messages) {
      errorMessages.push(...this.data.messages);
    }
    return errorMessages;
  }
  hasMissingQueryName() {
    return !this.queryName.trim().length;
  }
  hasMissingQuery() {
    return !this.query.trim().length;
  }
  hasMissingFields() {
    return this.hasMissingQuery() || this.hasMissingQueryName();
  }
  showErrorMessage() {
    var _a;
    return this.hasMissingFields() || ((_a = this.data.messages) === null || _a === void 0 ? void 0 : _a.length) > 0;
  }
  render() {
    return (h(Host, { tabindex: '-1' }, h("div", { class: "dialog-overlay", onClick: (evt) => this.onClose(evt) }, h("div", { class: "dialog" }, h("div", { class: "dialog-header" }, h("h3", { class: "dialog-title" }, this.translationService.translate('yasqe.actions.save_query.dialog.title')), h("button", { class: "close-button icon-close", onClick: (evt) => this.onClose(evt) })), h("div", { class: "dialog-body" }, h("div", { class: "save-query-form" }, h("div", { class: "form-field query-name-field" }, h("input", { type: "text", name: "queryName", id: "queryName", required: true, value: this.queryName, class: !this.queryName.trim().length ? 'invalid' : '', placeholder: this.translationService.translate('yasqe.actions.save_query.dialog.query_name.label'), onInput: (evt) => this.handleQueryNameChange(evt) })), h("div", { class: "form-field is-public-field" }, h("yasgui-tooltip", { placement: "top", "data-tooltip": this.translationService.translate('yasqe.actions.save_query.dialog.public_query.tooltip') }, h("label", null, h("input", { type: "checkbox", name: "publicQuery", id: "publicQuery", checked: this.isPublic, onChange: (evt) => this.handleIsPublicChange(evt) }), this.translationService.translate('yasqe.actions.save_query.dialog.public_query.label')))), h("div", { class: "form-field query-field" }, h("textarea", { name: "query", id: "query", required: true, rows: 8, class: !this.query.trim().length ? 'invalid' : '', placeholder: this.translationService.translate('yasqe.actions.save_query.dialog.query.label'), value: this.query, onInput: (evt) => this.handleQueryChange(evt) })), this.showErrorMessage() && h(Alert, { messages: this.getErrorMessages() }, "\u00A0"))), h("div", { class: "dialog-footer" }, h("button", { class: "cancel-button", onClick: (evt) => this.onClose(evt), ref: (el) => (this.cancelButton = el) }, this.translationService.translate('yasqe.actions.save_query.dialog.cancel.button')), h("button", { class: "ok-button", disabled: !this.isSaveAllowed, onClick: (evt) => this.onCreate(evt) }, this.translationService.translate('yasqe.actions.save_query.dialog.create.button')))))));
  }
  static get is() { return "save-query-dialog"; }
  static get originalStyleUrls() {
    return {
      "$": ["save-query-dialog.scss"]
    };
  }
  static get styleUrls() {
    return {
      "$": ["save-query-dialog.css"]
    };
  }
  static get properties() {
    return {
      "serviceFactory": {
        "type": "unknown",
        "mutable": false,
        "complexType": {
          "original": "ServiceFactory",
          "resolved": "ServiceFactory",
          "references": {
            "ServiceFactory": {
              "location": "import",
              "path": "../../services/service-factory"
            }
          }
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": ""
        }
      },
      "data": {
        "type": "unknown",
        "mutable": false,
        "complexType": {
          "original": "SaveQueryData",
          "resolved": "SaveQueryData",
          "references": {
            "SaveQueryData": {
              "location": "import",
              "path": "../../models/saved-query-configuration"
            }
          }
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "Input holding the saved query data if available. This data is used to initialize the form."
        }
      }
    };
  }
  static get states() {
    return {
      "queryName": {},
      "query": {},
      "isPublic": {},
      "isNew": {},
      "isSaveAllowed": {}
    };
  }
  static get events() {
    return [{
        "method": "internalSaveQueryDialogClosedEvent",
        "name": "internalSaveQueryDialogClosedEvent",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Event fired when the dialog is closed by triggering one of the close controls, e.g. close or\ncancel button."
        },
        "complexType": {
          "original": "any",
          "resolved": "any",
          "references": {}
        }
      }, {
        "method": "internalSaveQueryEvent",
        "name": "internalSaveQueryEvent",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Event fired when the create button in the dialog is triggered. The event payload holds the new\nsaved query data."
        },
        "complexType": {
          "original": "SaveQueryData",
          "resolved": "SaveQueryData",
          "references": {
            "SaveQueryData": {
              "location": "import",
              "path": "../../models/saved-query-configuration"
            }
          }
        }
      }, {
        "method": "internalUpdateQueryEvent",
        "name": "internalUpdateQueryEvent",
        "bubbles": true,
        "cancelable": true,
        "composed": true,
        "docs": {
          "tags": [],
          "text": "Event fired when the create button in the dialog is triggered and the query name is the same as\nthe one that was initially provided a.k.a. the query is updated. The event payload holds the\nupdated query data."
        },
        "complexType": {
          "original": "UpdateQueryData",
          "resolved": "UpdateQueryData",
          "references": {
            "UpdateQueryData": {
              "location": "import",
              "path": "../../models/saved-query-configuration"
            }
          }
        }
      }];
  }
  static get elementRef() { return "hostElement"; }
  static get listeners() {
    return [{
        "name": "keydown",
        "method": "keydownListener",
        "target": "window",
        "capture": false,
        "passive": false
      }];
  }
}
const Alert = ({ messages }) => (h("div", { class: "alert alert-danger" }, messages.map((message) => (h("div", { class: "error-message" }, message)))));
