/**
 * An adapter around the actual yasgui instance.
 */
export class OntotextYasgui {
  constructor(yasgui, config) {
    this.yasgui = yasgui;
    this.config = config;
    this.init();
  }
  refresh() {
    this.yasgui.getTab().show();
  }
  /**
   * Initializes the adapter.
   */
  init() {
    if (this.config.yasqeConfig.initialQuery) {
      this.setQuery(this.config.yasqeConfig.initialQuery);
    }
  }
  registerYasqeEventListener(eventName, callback) {
    this.yasgui.getTab().yasqe.on(eventName, (...args) => {
      callback(args);
    });
  }
  getYasqe() {
    var _a;
    return (_a = this.yasgui) === null || _a === void 0 ? void 0 : _a.getTab().getYasqe();
  }
  leaveFullScreen() {
    var _a;
    (_a = this.getYasqe()) === null || _a === void 0 ? void 0 : _a.leaveFullScreen();
  }
  /**
   * Sets a query value in the editor by preserving the cursor position.
   * @param query The query value to be set.
   */
  setQuery(query) {
    const yasqe = this.yasgui.getTab().getYasqe();
    const cursor = yasqe.getDoc().getCursor();
    const lastLine = yasqe.getDoc().lastLine();
    const lastLineLength = yasqe.getDoc().getLine(lastLine).length;
    yasqe.getDoc().replaceRange(query, { line: 0, ch: 0 }, { line: lastLine, ch: lastLineLength });
    yasqe.getDoc().setCursor(cursor);
  }
  query() {
    return this.yasgui.getTab().getYasqe().query();
  }
  getOngoingRequestsInfo() {
    return this.getInstance().getOngoingRequestsInfo();
  }
  abortQuery() {
    this.yasgui.getTab().getYasqe().abortQuery();
  }
  abortAllRequests() {
    Object.values(this.yasgui._tabs).forEach((tab) => {
      const yasqe = tab.getYasqe();
      if (yasqe) {
        yasqe.abortQuery();
      }
    });
  }
  /**
   * Reinitializes the Yasgui component by resetting the defined flags to those in the configuration.
   * Clears the results of the query.
   * @param resetFlags - the flags object used when resetting the Yasgui component
   */
  reInitYasgui(resetFlags) {
    this.yasgui.reInitYasgui(resetFlags, this.config);
  }
  getQuery() {
    return this.yasgui.getTab().getYasqe().getValue();
  }
  saveQuery() {
    this.getInstance().getTab().getYasqe().emit('blur');
  }
  isQueryDirty() {
    return !this.getYasqe().getDoc().isClean();
  }
  isQueryValid() {
    return this.yasgui.getTab().getYasqe().queryValid;
  }
  getQueryMode() {
    return this.yasgui.getTab().getYasqe().getQueryMode();
  }
  getQueryType() {
    return this.yasgui.getTab().getYasqe().getQueryType();
  }
  getEmbeddedResultAsJson() {
    return this.yasgui.getTab().getYasr().results.getAsJson();
  }
  getEmbeddedResultAsCSV() {
    return this.yasgui.getTab().getYasr().results.asCsv();
  }
  getConfig() {
    return this.config;
  }
  setConfig(yasguiConfiguration) {
    this.config = yasguiConfiguration;
  }
  getInstance() {
    return this.yasgui;
  }
  setInstance(yasgui) {
    this.yasgui = yasgui;
  }
  getTab(tabId) {
    return this.getInstance().getTab(tabId);
  }
  getTabId() {
    return this.getInstance().getTab().getId();
  }
  getTabName() {
    return this.getInstance().getTab().getName();
  }
  // TODO: What's the difference between getQuery() and this method?
  getTabQuery() {
    return this.getInstance().getTab().getQuery();
  }
  openTab(queryModel) {
    const existingTab = this.getInstance().getTabByNameAndQuery(queryModel === null || queryModel === void 0 ? void 0 : queryModel.queryName, queryModel === null || queryModel === void 0 ? void 0 : queryModel.query);
    if (existingTab) {
      this.getInstance().selectTabId(existingTab.getId());
      return existingTab;
    }
    else {
      return this.createNewTab(queryModel === null || queryModel === void 0 ? void 0 : queryModel.queryName, queryModel === null || queryModel === void 0 ? void 0 : queryModel.query);
    }
  }
  createNewTab(queryName, query) {
    const tabInstance = this.getInstance().addTab(true, {
      name: queryName
    });
    if (query) {
      tabInstance.setQuery(query);
    }
    return tabInstance;
  }
  /**
   * Searches the local store for the given tab ID and returns the height for the yasqe editor. If no height can be found,
   * the default 300 is returned.
   * @param tabId
   */
  getEditorHeight(tabId) {
    const heightString = this.getInstance().getTab(tabId).persistentJson.yasqe.editorHeight;
    if (heightString) {
      return parseInt(heightString.replace("px", ""), 10);
    }
    else {
      return 300;
    }
  }
  destroy() {
    if (this.yasgui) {
      this.yasgui.destroy();
      this.yasgui = null;
      localStorage.removeItem('yasqe__query');
    }
  }
}
