import { EXPLAIN_PLAN_TYPE, KeyboardShortcutDescription, KeyboardShortcutName } from '../models/keyboard-shortcut-description';
import { YasqeButtonName } from '../models/yasqe-button-name';
import { YasqeService } from './yasqe/yasqe-service';
import { IndentSelection } from '../models/yasgui/indent-selection';
export class KeyboardShortcutService {
  static createTriggerAutocomplete() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.TRIGGER_AUTOCOMPLETION;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-Space');
    keyboardShortcut.keyboardShortcuts.push('Alt-Enter');
    keyboardShortcut.keyboardShortcuts.push('Cmd-Space');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.autocomplete();
    };
    return keyboardShortcut;
  }
  static createDeleteCurrentLine() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.DELETE_CURRENT_LINE;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-D');
    keyboardShortcut.keyboardShortcuts.push('Ctrl-K');
    keyboardShortcut.keyboardShortcuts.push('Cmd-D');
    keyboardShortcut.keyboardShortcuts.push('Cmd-K');
    keyboardShortcut.executeFunction = (yasqe) => {
      const lineNumber = yasqe.getDoc().getCursor().line;
      //delete current line including the linebreak after
      return yasqe.getDoc().replaceRange("", { ch: 0, line: lineNumber }, { ch: 0, line: lineNumber + 1 });
    };
    return keyboardShortcut;
  }
  static createCommentCurrentLine() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.COMMENT_SELECTED_LINE;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-/');
    keyboardShortcut.keyboardShortcuts.push('Cmd-/');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.commentLines();
    };
    return keyboardShortcut;
  }
  static createCopyLineDown() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.COPY_LINE_DOWN;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-Alt-Down');
    keyboardShortcut.keyboardShortcuts.push('Cmd-Alt-Down');
    keyboardShortcut.executeFunction = (yasqe) => {
      let cursor = yasqe.getCursor();
      const cursorLinePosition = cursor.ch;
      yasqe.duplicateLine();
      // Sets cursor in same position.
      cursor = yasqe.getCursor();
      cursor.ch = cursorLinePosition;
      yasqe.setCursor(cursor);
    };
    return keyboardShortcut;
  }
  static createCopyLineUp() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.COPY_LINE_UP;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-Alt-Up');
    keyboardShortcut.keyboardShortcuts.push('Cmd-Alt-Up');
    keyboardShortcut.executeFunction = (yasqe) => {
      let cursor = yasqe.getCursor();
      if (!cursor) {
        return;
      }
      const currentLine = yasqe.getDoc().getLine(cursor.line);
      if (!currentLine) {
        return;
      }
      const cursorLinePosition = cursor.ch;
      cursor.line--;
      yasqe.setCursor(cursor);
      const line = yasqe.getDoc().getLine(cursor.line);
      if (!line) {
        return;
      }
      yasqe.getDoc().replaceRange(line + "\n" + currentLine, { ch: 0, line: cursor.line }, { ch: line.length, line: cursor.line });
      // Sets cursor in same position.
      cursor = yasqe.getCursor();
      cursor.ch = cursorLinePosition;
      yasqe.setCursor(cursor);
    };
    return keyboardShortcut;
  }
  static createAutoFormat() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.AUTO_FORMAT_SELECTED_LINE;
    keyboardShortcut.keyboardShortcuts.push('Shift-Ctrl-F');
    keyboardShortcut.keyboardShortcuts.push('Shift-Cmd-F');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.autoformat();
    };
    return keyboardShortcut;
  }
  static createIndentMore() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.INDENT_CURRENT_LINE_MORE;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-]');
    keyboardShortcut.keyboardShortcuts.push('Cmd-]');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.indentSelection(IndentSelection.ADD);
    };
    return keyboardShortcut;
  }
  static createIndentLess() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.INDENT_CURRENT_LINE_LESS;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-[');
    keyboardShortcut.keyboardShortcuts.push('Cmd-[');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.indentSelection(IndentSelection.SUBTRACT);
    };
    return keyboardShortcut;
  }
  static createExecuteQuery() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.EXECUTE_QUERY_OR_UPDATE;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-Enter');
    keyboardShortcut.keyboardShortcuts.push('Cmd-Enter');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.query().then().catch(() => {
        // catch this to avoid unhandled rejection
      });
    };
    return keyboardShortcut;
  }
  static createExecuteExplainPlanForQuery() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.EXECUTE_EXPLAIN_PLAN_FOR_QUERY;
    keyboardShortcut.keyboardShortcuts.push('Shift-Ctrl-Enter');
    keyboardShortcut.keyboardShortcuts.push('Shift-Cmd-Enter');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.query(undefined, EXPLAIN_PLAN_TYPE.EXPLAIN).catch(() => {
        // catch this to avoid unhandled rejection
      });
    };
    return keyboardShortcut;
  }
  static createExecuteChatGPTExplainPlanForQuery() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.EXECUTE_CHAT_GPT_EXPLAIN_PLAN_FOR_QUERY;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-Alt-Enter');
    keyboardShortcut.keyboardShortcuts.push('Cmd-Alt-Enter');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.query(undefined, EXPLAIN_PLAN_TYPE.CHAT_GPT_EXPLAIN).catch(() => {
        // catch this to avoid unhandled rejection
      });
    };
    return keyboardShortcut;
  }
  static createCreateTab() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.CREATE_TAB;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-Alt-T');
    keyboardShortcut.keyboardShortcuts.push('Cmd-Alt-T');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.emit('openNewTab');
    };
    return keyboardShortcut;
  }
  static createSavedQuery() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.CREATE_SAVE_QUERY;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-S');
    keyboardShortcut.keyboardShortcuts.push('Cmd-S');
    keyboardShortcut.executeFunction = (yasqe) => {
      const wrapperElement = yasqe.getWrapperElement();
      const querySelector = wrapperElement.querySelector(`.${YasqeService.getActionButtonClassName(YasqeButtonName.CREATE_SAVED_QUERY)}`);
      if (querySelector) {
        querySelector.click();
      }
    };
    return keyboardShortcut;
  }
  static createSwitchToNextTab() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.SWITCH_NEXT_TAB;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-Alt-Right');
    keyboardShortcut.keyboardShortcuts.push('Cmd-Alt-Right');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.emit('openNextTab');
    };
    return keyboardShortcut;
  }
  static createSwitchToPreviousTab() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.SWITCH_PREVIOUS_TAB;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-Alt-Left');
    keyboardShortcut.keyboardShortcuts.push('Cmd-Alt-Left');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.emit('openPreviousTab');
    };
    return keyboardShortcut;
  }
  static createCloseOtherTabs() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.CLOSES_OTHER_TABS;
    keyboardShortcut.keyboardShortcuts.push('Shift-Ctrl-F4');
    keyboardShortcut.keyboardShortcuts.push('Cmd-Ctrl-F4');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.emit('closeOtherTabs');
    };
    return keyboardShortcut;
  }
  static createCloseOtherTabsByMouseClick() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.CLOSES_OTHER_TABS_BY_MOUSE_CLICK;
    keyboardShortcut.executeFunction = () => {
      // Nothing to do the description is added just to explain the functionality, that closes all other tabs.
      // This functionality is triggered by clicking on the close tab button while holding down the Shift key.
    };
    return keyboardShortcut;
  }
  static createFullScreen() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.FULL_SCREEN;
    keyboardShortcut.keyboardShortcuts.push('Ctrl-Alt-F');
    keyboardShortcut.keyboardShortcuts.push('Cmd-Alt-F');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.toggleFullScreen();
    };
    return keyboardShortcut;
  }
  static createEscape() {
    const keyboardShortcut = new KeyboardShortcutDescription();
    keyboardShortcut.NAME = KeyboardShortcutName.ESC;
    keyboardShortcut.keyboardShortcuts.push('Esc');
    keyboardShortcut.executeFunction = (yasqe) => {
      yasqe.leaveFullScreen();
    };
    return keyboardShortcut;
  }
  static initAllKeyboardShortcuts() {
    const keyboardShortcuts = new Map();
    keyboardShortcuts.set(KeyboardShortcutName.TRIGGER_AUTOCOMPLETION, KeyboardShortcutService.createTriggerAutocomplete);
    keyboardShortcuts.set(KeyboardShortcutName.DELETE_CURRENT_LINE, KeyboardShortcutService.createDeleteCurrentLine);
    keyboardShortcuts.set(KeyboardShortcutName.COMMENT_SELECTED_LINE, KeyboardShortcutService.createCommentCurrentLine);
    keyboardShortcuts.set(KeyboardShortcutName.COPY_LINE_DOWN, KeyboardShortcutService.createCopyLineDown);
    keyboardShortcuts.set(KeyboardShortcutName.COPY_LINE_UP, KeyboardShortcutService.createCopyLineUp);
    keyboardShortcuts.set(KeyboardShortcutName.AUTO_FORMAT_SELECTED_LINE, KeyboardShortcutService.createAutoFormat);
    keyboardShortcuts.set(KeyboardShortcutName.INDENT_CURRENT_LINE_MORE, KeyboardShortcutService.createIndentMore);
    keyboardShortcuts.set(KeyboardShortcutName.INDENT_CURRENT_LINE_LESS, KeyboardShortcutService.createIndentLess);
    keyboardShortcuts.set(KeyboardShortcutName.EXECUTE_QUERY_OR_UPDATE, KeyboardShortcutService.createExecuteQuery);
    keyboardShortcuts.set(KeyboardShortcutName.EXECUTE_EXPLAIN_PLAN_FOR_QUERY, KeyboardShortcutService.createExecuteExplainPlanForQuery);
    keyboardShortcuts.set(KeyboardShortcutName.EXECUTE_CHAT_GPT_EXPLAIN_PLAN_FOR_QUERY, KeyboardShortcutService.createExecuteChatGPTExplainPlanForQuery);
    keyboardShortcuts.set(KeyboardShortcutName.CREATE_TAB, KeyboardShortcutService.createCreateTab);
    keyboardShortcuts.set(KeyboardShortcutName.CREATE_SAVE_QUERY, KeyboardShortcutService.createSavedQuery);
    keyboardShortcuts.set(KeyboardShortcutName.SWITCH_NEXT_TAB, KeyboardShortcutService.createSwitchToNextTab);
    keyboardShortcuts.set(KeyboardShortcutName.SWITCH_PREVIOUS_TAB, KeyboardShortcutService.createSwitchToPreviousTab);
    keyboardShortcuts.set(KeyboardShortcutName.CLOSES_OTHER_TABS, KeyboardShortcutService.createCloseOtherTabs);
    keyboardShortcuts.set(KeyboardShortcutName.CLOSES_OTHER_TABS_BY_MOUSE_CLICK, KeyboardShortcutService.createCloseOtherTabsByMouseClick);
    keyboardShortcuts.set(KeyboardShortcutName.FULL_SCREEN, KeyboardShortcutService.createFullScreen);
    keyboardShortcuts.set(KeyboardShortcutName.ESC, KeyboardShortcutService.createEscape);
    return keyboardShortcuts;
  }
}
KeyboardShortcutService.keyboardShortcutNameToFactoryFunction = KeyboardShortcutService.initAllKeyboardShortcuts();
KeyboardShortcutService.initKeyboardShortcutMapping = (config) => {
  const keyboardShortcutDescriptions = [];
  const insertAll = !config.keyboardShortcutConfiguration || config.keyboardShortcutConfiguration.length < 1;
  KeyboardShortcutService.keyboardShortcutNameToFactoryFunction
    .forEach((factoryFunction, keyboardShortcutName) => {
    if (insertAll || config.keyboardShortcutConfiguration[keyboardShortcutName] === undefined || config.keyboardShortcutConfiguration[keyboardShortcutName]) {
      keyboardShortcutDescriptions.push(factoryFunction());
    }
  });
  return keyboardShortcutDescriptions;
};
