import en from '../i18n/locale-en.json';
import fr from '../i18n/locale-fr.json';
import { DEFAULT_LANG } from '../configurations/constants';
/**
 * Service responsible for translation operations in the component.
 */
export class TranslationService {
  constructor() {
    this.currentLang = DEFAULT_LANG;
    this.bundle = { en, fr };
    this.languageChangeObservers = {};
    this.translationChangedObservers = {};
  }
  /**
   * Translates the <code>messageLabelKey</code> with <code>translationParameter</code> and call <code>translationCallback</code> with translation of current language.
   * The <code>translationCallback</code> is called upon subscription and whenever the selected language is changed.
   *
   * @param messageLabelKey - The label key for the translation.
   * @param translationCallback - A function to be called when translating the `messageLabelKey`.
   * @param translationParameter - Parameters, if needed, for translation.
   * @returns A function that, when called, unsubscribes the provided callback from further translation updates.
   */
  onTranslate(messageLabelKey, translationCallback = () => { }, translationParameter = []) {
    this.translationChangedObservers[messageLabelKey] = this.translationChangedObservers[messageLabelKey] || [];
    const observer = { parameters: translationParameter, callback: translationCallback };
    this.translationChangedObservers[messageLabelKey].push(observer);
    translationCallback(this.translate(messageLabelKey, translationParameter));
    return () => {
      const index = this.translationChangedObservers[messageLabelKey].indexOf(observer);
      if (index !== -1) {
        this.translationChangedObservers[messageLabelKey].splice(index, 1);
      }
    };
  }
  notifyTranslationsChanged() {
    Object.keys(this.translationChangedObservers).forEach((eventName) => {
      const observers = this.translationChangedObservers[eventName] || [];
      observers.forEach((observer) => observer.callback(this.translate(eventName, observer.parameters)));
    });
  }
  /**
   * Sets the language which should be used for registered labels. If there is no registered bundle
   * for the provided language, then the default language will be set and used.
   * This method also notifies all registered LanguageChangeObserver's.
   *
   * @param lang The language to be set.
   */
  setLanguage(lang = DEFAULT_LANG) {
    if (!this.bundle || !this.bundle[this.currentLang]) {
      console.warn(`Missing locale file for [${this.currentLang}]`);
      this.currentLang = DEFAULT_LANG;
    }
    else {
      this.currentLang = lang;
    }
    this.notifyLanguageChangeObservers(this.currentLang);
    this.notifyTranslationsChanged();
  }
  getLanguage() {
    return this.currentLang;
  }
  /**
   * Subscribes the observer for further language change events.
   *
   * @param observer The observer to be registered for the language change events.
   * @return Returns an unsubscribe function which can be called by the observer to unsubscribe
   * itself.
   */
  subscribeForLanguageChange(observer) {
    const languageChangeObservers = this.languageChangeObservers[observer.name];
    if (languageChangeObservers) {
      languageChangeObservers.push(observer);
    }
    else {
      this.languageChangeObservers[observer.name] = [observer];
    }
    return () => this.unsubscribeFromLanguageChange(observer);
  }
  /**
   * Unsubscribes the observer from the language change events.
   * @param observer The observer to be unsubscribed.
   */
  unsubscribeFromLanguageChange(observer) {
    const languageChangeObservers = this.languageChangeObservers[observer.name];
    if (languageChangeObservers) {
      const index = languageChangeObservers.indexOf(observer);
      if (index !== -1) {
        languageChangeObservers.splice(index, 1);
      }
    }
  }
  notifyLanguageChangeObservers(currentLang) {
    Object.values(this.languageChangeObservers).forEach((observers) => {
      observers.forEach((observer) => observer.notify(currentLang));
    });
  }
  /**
   * Merges external i18e  configuration with bundled one.
   * * @param translations - external i18e configuration.
   * Key -> value translations as JSON. If the language is supported, then not needed to pass all label values.
   * If pass a new language then all label's values have to be present, otherwise they will be translated to the default English language.
   * Example:
   * {
   *   en: {
   *     "yasgui.toolbar.orientation.btn.tooltip.switch_orientation_horizontal": "Switch to horizontal view",
   *     "yasgui.toolbar.orientation.btn.tooltip.switch_orientation_vertical": "Switch to vertical view",
   *     "yasgui.toolbar.mode_yasqe.btn.label": "Editor only",
   *     "yasgui.toolbar.mode_yasgui.btn.label": "Editor and results",
   *     "yasgui.toolbar.mode_yasr.btn.label": "Results only",
   *   }
   *   fr: {
   *     "yasgui.toolbar.orientation.btn.tooltip.switch_orientation_vertical": "Basculer vers verticale voir",
   *     "yasgui.toolbar.mode_yasqe.btn.label": "Éditeur seulement",
   *   }
   * }
   */
  addTranslations(translations) {
    Object.keys(translations).forEach((language) => {
      // If is new language add it all
      if (!this.bundle[language]) {
        this.bundle[language] = translations[language];
      }
      else {
        // merge passed key -> values
        Object.assign(this.bundle[language], translations[language]);
      }
    });
  }
  /**
   * Translates the provided key using the currently selected language by applying the parameters if
   * provided.
   * @param key The key for the label which needs to be translated.
   * @param parameters Optional parameters which to be applied during the translation.
   */
  translate(key, parameters) {
    return this.translateInLocale(key, this.currentLang, parameters);
  }
  translateInLocale(key, locale, parameters) {
    let translation = this.bundle[locale][key];
    if (!translation) {
      // Fallback to the default language
      translation = this.bundle[DEFAULT_LANG][key];
    }
    if (translation) {
      translation = this.applyParameters(translation, parameters);
      return translation;
    }
    console.warn(`Missing translation for [${key}] key in [${this.currentLang}] locale`);
    return key;
  }
  applyParameters(translation, parameters) {
    if (parameters) {
      return parameters.reduce(
      // replace all occurrence of parameter key with parameter value.
      (translation, parameter) => TranslationService.replaceAll(translation, parameter), translation);
    }
    return translation;
  }
  static replaceAll(translation, parameter) {
    return parameter ? translation.split(`{{${parameter.key}}}`).join(parameter.value) : translation;
  }
}
