import { HtmlUtil } from '../utils/html-util';
import { SparqlUtils } from '../utils/sparql-utils';
export class YasrService {
  static registerPlugin(name, plugin, enable = true) {
    // @ts-ignore
    Yasr.registerPlugin(name, plugin, enable);
  }
  static getPluginsConfigurations(externalConfiguration) {
    const pluginsConfigurations = new Map();
    this.addExtendedTableConfiguration(externalConfiguration, pluginsConfigurations);
    this.addTableConfiguration(externalConfiguration, pluginsConfigurations);
    return pluginsConfigurations;
  }
  static addTableConfiguration(externalConfiguration, pluginsConfigurations) {
    const configuration = {
      maxResizableResultsColumns: externalConfiguration.maxResizableResultsColumns ? externalConfiguration.maxResizableResultsColumns : 19,
    };
    pluginsConfigurations.set('table', configuration);
  }
  static addExtendedTableConfiguration(externalConfiguration, pluginsConfigurations) {
    const configuration = {
      getCellContent: externalConfiguration.getCellContent ? externalConfiguration.getCellContent : YasrService.getCellContent().bind(this),
    };
    pluginsConfigurations.set('extended_table', configuration);
  }
  // @ts-ignore
  static getCellContent() {
    const shacl = window.location.href.includes(YasrService.SHACL_GRAPH_URL);
    const ontotextResourceLocation = window.location.origin + '/resource/';
    const context = new CellContentContext(shacl, ontotextResourceLocation);
    //@ts-ignore
    return (binding, prefixes) => {
      context.setPrefixes(prefixes);
      return this.toCellContent(binding, context);
    };
  }
  // @ts-ignore
  static toCellContent(binding, context) {
    if ('uri' === binding.type) {
      return YasrService.getUriCellContent(binding, context);
    }
    else if ('triple' === binding.type) {
      return YasrService.getTripleCellContent(binding, context);
    }
    return YasrService.getLiteralCellContent(binding);
  }
  // @ts-ignore
  static getUriCellContent(binding, context) {
    const uri = binding.value;
    if (!context.hasElement(uri)) {
      const content = `<div class="uri-cell" lang="${this.getLang(binding, 'xx')}">` +
        `<a title="${uri}" class="uri-link" href="${this.getHref(uri, context)}">${YasrService.addWordBreakToIRIs(context.getShortUri(uri))}</a>` +
        `<copy-resource-link-button title="${uri}" class="resource-copy-link" uri="${uri}"></copy-resource-link-button>` +
        '<span class="spacer"></span></div>';
      context.setElement(uri, content);
    }
    return context.getElement(uri);
  }
  static getHref(uri, context) {
    let localHref;
    if (context.isOntotextResource(uri)) {
      // URI is within our URL space, use it as is
      localHref = uri;
    }
    else {
      // URI is not within our URL space, needs to be passed as parameter
      localHref = "resource?uri=" + encodeURIComponent(uri);
    }
    if (context.isShacl()) {
      localHref += YasrService.SHACL_GRAPH_URL_CONTEXT_PARAMETER;
    }
    return this.replaceSingleQuote(localHref);
  }
  // @ts-ignore
  static getTripleCellContent(binding, context) {
    const tripleAsString = this.getValueAsString(binding, false);
    const tripleLinkHref = `resource?triple=${this.replaceSingleQuote(encodeURIComponent(tripleAsString))}`;
    const escapedTriple = HtmlUtil.escapeHTMLEntities(tripleAsString);
    return `<div class="triple-cell">` +
      `<div class="triple-open-link">` +
      `<a title="${escapedTriple}" class="triple-link" href="${tripleLinkHref}">${YasrService.ESCAPED_HTML_DOUBLE_LOWER}</a>` +
      `<copy-resource-link-button title="${escapedTriple}" class="resource-copy-link" uri="${escapedTriple}"></copy-resource-link-button>` +
      `<span class="spacer"></span>` +
      `</div>` +
      `<div class="triple-list">` +
      `<div>${this.toCellContent(binding.value['s'], context)}</div>` +
      `<div>${this.toCellContent(binding.value['p'], context)}</div>` +
      `<div>${this.toCellContent(binding.value['o'], context)}</div>` +
      `</div>` +
      `<div class="triple-close-link">` +
      `<a title="${escapedTriple}" class="triple-link triple-link-end" href="${tripleLinkHref}">${YasrService.ESCAPED_HTML_DOUBLE_GREATER}</a>` +
      `<copy-resource-link-button title="${escapedTriple}" class="resource-copy-link" uri="${escapedTriple}"></copy-resource-link-button>` +
      `<span class="spacer"></span>` +
      `</div>` +
      `</div>`;
  }
  static replaceSingleQuote(text) {
    return text.replace(/'/g, "&#39;");
  }
  // @ts-ignore
  static getValueAsString(binding, forHtml) {
    if (binding.type === "uri") {
      return `<${binding.value}>`;
    }
    if (binding.type === "triple") {
      return `<<${this.getValueAsString(binding.value['s'], forHtml)} ${this.getValueAsString(binding.value['p'], forHtml)} ${this.getValueAsString(binding.value['o'], forHtml)}>>`;
    }
    return this.getLiteralAsString(binding, forHtml);
  }
  // @ts-ignore
  static getLiteralCellContent(binding) {
    return `<div lang="${YasrService.getLang(binding, 'xx')}" class="literal-cell"><p class="nonUri">${this.getLiteralAsString(binding, true)}</p></div>`;
  }
  //@ts-ignore
  static getLiteralAsString(binding, forHtml) {
    if (this.isExplainResponse(binding)) {
      return this.getExplainPlanQueryResponse(binding, forHtml);
    }
    if (binding.type == "bnode") {
      return YasrService.addWordBreakToLiterals(`_:${HtmlUtil.escapeHTMLEntities(binding.value)}`);
    }
    const stringRepresentation = HtmlUtil.escapeHTMLEntities(binding.value);
    if (binding["xml:lang"]) {
      return YasrService.addWordBreakToLiterals(`"${stringRepresentation}"${forHtml ? '<sup>' : ''}@${binding["xml:lang"]}${forHtml ? '</sup>' : ''}`);
    }
    if (binding["lang"]) {
      return YasrService.addWordBreakToLiterals(`"${stringRepresentation}"${forHtml ? '<sup>' : ''}@${binding["lang"]}${forHtml ? '</sup>' : ''}`);
    }
    if (binding.datatype && YasrService.XML_SCHEMA_NS_STRING !== binding.datatype) {
      let dataType = binding.datatype;
      if (forHtml && dataType.indexOf(YasrService.XML_SCHEMA_NS) === 0) {
        dataType = `xsd:${dataType.substring(YasrService.XML_SCHEMA_NS.length)}`;
      }
      else if (forHtml) {
        dataType = `&lt;${dataType}&gt;`;
      }
      else {
        dataType = `<${dataType}>`;
      }
      return YasrService.addWordBreakToLiterals(`"${stringRepresentation}"${forHtml ? '<sup>' : ''}^^${dataType}${forHtml ? '</sup>' : ''}`);
    }
    return YasrService.addWordBreakToLiterals(stringRepresentation.startsWith('"') ? stringRepresentation : `"${stringRepresentation}"`);
  }
  //@ts-ignore
  static isExplainResponse(binding) {
    return "literal" === binding.type && binding.value.includes("# NOTE: Optimization groups");
  }
  //@ts-ignore
  static getExplainPlanQueryResponse(binding, forHtml) {
    const stringRepresentation = HtmlUtil.escapeHTMLEntities(binding.value);
    if (forHtml) {
      return `<div id="explainPlanQuery" class="cm-s-default">${stringRepresentation}</div>`;
    }
    return stringRepresentation.startsWith('"') ? stringRepresentation : `"${stringRepresentation}"`;
  }
  static addWorldBreakTagAfterSpecialCharacters(text) {
    return text.replace(/([_:/-](?![_:/-]|sup>))/g, "$1<wbr>");
  }
  static addWorldBreakTagBeforeSpecialCharacters(text) {
    return text.replace(/(\^\^)/g, "<wbr>$1");
  }
  static addWordBreakToIRIs(text) {
    return YasrService.addWorldBreakTagAfterSpecialCharacters(text);
  }
  static addWordBreakToLiterals(text) {
    const result = this.addWorldBreakTagBeforeSpecialCharacters(text);
    return YasrService.addWorldBreakTagAfterSpecialCharacters(result);
  }
  static getLang(literalBinding, defaultLang) {
    if (literalBinding["xml:lang"]) {
      return literalBinding["xml:lang"];
    }
    if (literalBinding["lang"]) {
      return literalBinding["lang"];
    }
    return defaultLang;
  }
}
YasrService.SHACL_GRAPH_URL = "http:%2F%2Frdf4j.org%2Fschema%2Frdf4j%23SHACLShapeGraph";
YasrService.XML_SCHEMA_NS = "http://www.w3.org/2001/XMLSchema#";
YasrService.XML_SCHEMA_NS_STRING = YasrService.XML_SCHEMA_NS + 'string';
/**
 * Escaped <<
 */
YasrService.ESCAPED_HTML_DOUBLE_LOWER = '&lt;&lt;';
/**
 * Escaped >>
 *
 */
YasrService.ESCAPED_HTML_DOUBLE_GREATER = '&gt;&gt';
class CellContentContext {
  constructor(shacl, ontotextResourceLocation) {
    this.uriToCellElementMapping = new Map();
    this.fullUriToShortUri = new Map();
    this.prefixes = {};
    this.shacl = shacl;
    this.ontotextResourceLocation = ontotextResourceLocation;
  }
  /**
   * Setter for prefixes.
   *
   * @param prefixes - Object with uris and their corresponding prefixes.
   * For example:
   * <pre>
   *    {
   *      "gn": "http://www.geonames.org/ontology#",
   *      "path": "http://www.ontotext.com/path#",
   *      "rdf": "http://www.w3.org/1999/02/22-rdf-syntax-ns#",
   *      "rdfs": "http://www.w3.org/2000/01/rdf-schema#",
   *      "xsd": "http://www.w3.org/2001/XMLSchema#",
   *     }
   * </pre>
   * @param prefixes
   */
  setPrefixes(prefixes = {}) {
    this.prefixes = prefixes;
  }
  getShortUri(uri) {
    if (!this.fullUriToShortUri.has(uri)) {
      this.fullUriToShortUri.set(uri, SparqlUtils.uriToPrefixWithLocalName(uri, this.prefixes));
    }
    return this.fullUriToShortUri.get(uri);
  }
  hasElement(uri) {
    return this.uriToCellElementMapping.has(uri);
  }
  getElement(url) {
    return this.uriToCellElementMapping.get(url);
  }
  setElement(uri, cellHtmlElement) {
    this.uriToCellElementMapping.set(uri, cellHtmlElement);
  }
  isOntotextResource(uri) {
    return uri.indexOf(this.ontotextResourceLocation) === 0 && uri.length > this.ontotextResourceLocation.length;
  }
  isShacl() {
    return this.shacl;
  }
}
