import { Translations } from '../models/yasgui-configuration';
export type LanguageChangeObserver = {
  name: string;
  notify: (language: string) => void;
};
export interface TranslationParameter {
  key: string;
  value: string;
}
type TranslationCallback = (translation: string) => void;
/**
 * Service responsible for translation operations in the component.
 */
export declare class TranslationService {
  private currentLang;
  private bundle;
  private languageChangeObservers;
  private translationChangedObservers;
  /**
   * Translates the <code>messageLabelKey</code> with <code>translationParameter</code> and call <code>translationCallback</code> with translation of current language.
   * The <code>translationCallback</code> is called upon subscription and whenever the selected language is changed.
   *
   * @param messageLabelKey - The label key for the translation.
   * @param translationCallback - A function to be called when translating the `messageLabelKey`.
   * @param translationParameter - Parameters, if needed, for translation.
   * @returns A function that, when called, unsubscribes the provided callback from further translation updates.
   */
  onTranslate(messageLabelKey: string, translationCallback?: TranslationCallback, translationParameter?: TranslationParameter[]): () => void;
  private notifyTranslationsChanged;
  /**
   * Sets the language which should be used for registered labels. If there is no registered bundle
   * for the provided language, then the default language will be set and used.
   * This method also notifies all registered LanguageChangeObserver's.
   *
   * @param lang The language to be set.
   */
  setLanguage(lang?: string): void;
  getLanguage(): string;
  /**
   * Subscribes the observer for further language change events.
   *
   * @param observer The observer to be registered for the language change events.
   * @return Returns an unsubscribe function which can be called by the observer to unsubscribe
   * itself.
   */
  subscribeForLanguageChange(observer: LanguageChangeObserver): () => void;
  /**
   * Unsubscribes the observer from the language change events.
   * @param observer The observer to be unsubscribed.
   */
  unsubscribeFromLanguageChange(observer: LanguageChangeObserver): void;
  private notifyLanguageChangeObservers;
  /**
   * Merges external i18e  configuration with bundled one.
   * * @param translations - external i18e configuration.
   * Key -> value translations as JSON. If the language is supported, then not needed to pass all label values.
   * If pass a new language then all label's values have to be present, otherwise they will be translated to the default English language.
   * Example:
   * {
   *   en: {
   *     "yasgui.toolbar.orientation.btn.tooltip.switch_orientation_horizontal": "Switch to horizontal view",
   *     "yasgui.toolbar.orientation.btn.tooltip.switch_orientation_vertical": "Switch to vertical view",
   *     "yasgui.toolbar.mode_yasqe.btn.label": "Editor only",
   *     "yasgui.toolbar.mode_yasgui.btn.label": "Editor and results",
   *     "yasgui.toolbar.mode_yasr.btn.label": "Results only",
   *   }
   *   fr: {
   *     "yasgui.toolbar.orientation.btn.tooltip.switch_orientation_vertical": "Basculer vers verticale voir",
   *     "yasgui.toolbar.mode_yasqe.btn.label": "Éditeur seulement",
   *   }
   * }
   */
  addTranslations(translations: Translations): void;
  /**
   * Translates the provided key using the currently selected language by applying the parameters if
   * provided.
   * @param key The key for the label which needs to be translated.
   * @param parameters Optional parameters which to be applied during the translation.
   */
  translate(key: string, parameters?: TranslationParameter[]): string;
  translateInLocale(key: string, locale: string, parameters?: TranslationParameter[]): string;
  private applyParameters;
  private static replaceAll;
}
export {};
