"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiValidator = void 0;
const openapi_request_validator_1 = require("./middlewares/openapi.request.validator");
const base_serdes_1 = require("./framework/base.serdes");
const ajvOptions_1 = require("./framework/ajvOptions");
class OpenApiValidator {
    constructor(options) {
        this.validateOptions(options);
        this.normalizeOptions(options);
        if (!options.validateRequests && options.validateRequests !== false) {
            options.validateRequests = true;
        }
        if (options.validateRequests === true) {
            options.validateRequests = {
                allowUnknownQueryParameters: false
            };
        }
        this.options = options;
        this.ajvOpts = new ajvOptions_1.AjvOptions(options);
    }
    createValidator() {
        const specAsync = this.loadSpec(this.options.apiSpec);
        let requestValidator;
        return async (request) => {
            if (!requestValidator) {
                const spec = await specAsync;
                const ajvOpts = this.ajvOpts.preprocessor;
                const { SchemaPreprocessor } = require('./middlewares/parsers/schema.preprocessor');
                new SchemaPreprocessor(spec, ajvOpts).preProcess();
                requestValidator = new openapi_request_validator_1.RequestValidator(spec, this.ajvOpts.request);
            }
            requestValidator.validate(request);
        };
    }
    async compileValidator() {
        const specAsync = this.loadSpec(this.options.apiSpec);
        const spec = await specAsync;
        const ajvOpts = this.ajvOpts.preprocessor;
        const { SchemaPreprocessor } = require('./middlewares/parsers/schema.preprocessor');
        new SchemaPreprocessor(spec, ajvOpts).preProcess();
        const requestValidator = new openapi_request_validator_1.RequestValidator(spec, this.ajvOpts.request);
        await requestValidator.compile(this.options.compiledFilePath);
    }
    async loadSpec(schemaOrPath) {
        if (typeof schemaOrPath === 'string') {
            const origCwd = process.cwd();
            const path = require('path');
            const absolutePath = path.resolve(origCwd, schemaOrPath);
            const { access } = require('fs').promises;
            await access(absolutePath);
            const $RefParser = require('@apidevtools/json-schema-ref-parser');
            return Object.assign($RefParser.dereference(absolutePath));
        }
        // Test the full parser
        // const $RefParser = require('@apidevtools/json-schema-ref-parser');
        // const result = await $RefParser.dereference(await schemaOrPath);
        const cloneDeep = require('lodash.clonedeep');
        const dereference = require('@apidevtools/json-schema-ref-parser/lib/dereference');
        const $Refs = require('@apidevtools/json-schema-ref-parser/lib/refs');
        const handler = { schema: null, $refs: new $Refs() };
        // eslint-disable-next-line no-underscore-dangle
        const $ref = handler.$refs._add('');
        $ref.value = cloneDeep(await schemaOrPath);
        $ref.pathType = 'http';
        handler.schema = $ref.value;
        dereference(handler, { parse: {}, dereference: { excludedPathMatcher: () => false } });
        return Object.assign(handler.schema);
    }
    async loadValidator() {
        const requestValidator = new openapi_request_validator_1.RequestValidator(null, this.ajvOpts.request);
        await requestValidator.loadCompiled(this.options.compiledFilePath);
        return async (request) => {
            await requestValidator.validate(request);
        };
    }
    validateOptions(options) {
        if (!options.apiSpec && !options.compiledFilePath) {
            throw Error('apiSpec required');
        }
    }
    normalizeOptions(options) {
        if (!options.serDes) {
            options.serDes = base_serdes_1.defaultSerDes;
        }
        else {
            base_serdes_1.defaultSerDes.forEach(currentDefaultSerDes => {
                const defaultSerDesOverride = options.serDes.find(currentOptionSerDes => {
                    return currentDefaultSerDes.format === currentOptionSerDes.format;
                });
                if (!defaultSerDesOverride) {
                    options.serDes.push(currentDefaultSerDes);
                }
            });
        }
    }
}
exports.OpenApiValidator = OpenApiValidator;
