"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SchemaPreprocessor = exports.httpMethods = void 0;
const _get = require("lodash.get");
class Node {
    constructor(parent, schema, path) {
        this.path = path;
        this.parent = parent;
        this.schema = schema;
    }
}
class Root extends Node {
    constructor(schema, path) {
        super(null, schema, path);
    }
}
exports.httpMethods = new Set([
    'get',
    'put',
    'post',
    'delete',
    'options',
    'head',
    'patch',
    'trace'
]);
class SchemaPreprocessor {
    constructor(apiDoc, ajvOptions) {
        const { createRequestAjv } = require('../../framework/ajv');
        this.ajv = createRequestAjv(apiDoc, ajvOptions);
        this.apiDoc = apiDoc;
        this.serDesMap = ajvOptions.serDesMap;
    }
    preProcess() {
        const componentSchemas = this.gatherComponentSchemaNodes();
        const r = this.gatherSchemaNodesFromPaths();
        const schemaNodes = {
            schemas: componentSchemas,
            requestBodies: r.requestBodies,
            responses: r.responses
        };
        // Traverse the schemas
        this.traverseSchemas(schemaNodes, (parent, schema) => this.schemaVisitor(parent, schema));
    }
    gatherComponentSchemaNodes() {
        var _a, _b, _c;
        const nodes = [];
        const componentSchemaMap = (_c = (_b = (_a = this.apiDoc) === null || _a === void 0 ? void 0 : _a.components) === null || _b === void 0 ? void 0 : _b.schemas) !== null && _c !== void 0 ? _c : [];
        for (const [id, s] of Object.entries(componentSchemaMap)) {
            const schema = this.resolveSchema(s);
            this.apiDoc.components.schemas[id] = schema;
            const path = ['components', 'schemas', id];
            const node = new Root(schema, path);
            nodes.push(node);
        }
        return nodes;
    }
    gatherSchemaNodesFromPaths() {
        const requestBodySchemas = [];
        const responseSchemas = [];
        for (const [p, pi] of Object.entries(this.apiDoc.paths)) {
            const pathItem = this.resolveSchema(pi);
            for (const method of Object.keys(pathItem)) {
                if (exports.httpMethods.has(method)) {
                    const operation = pathItem[method];
                    // Adds path declared parameters to the schema's parameters list
                    this.preprocessPathLevelParameters(method, pathItem);
                    const path = ['paths', p, method];
                    const node = new Root(operation, path);
                    const requestBodies = this.extractRequestBodySchemaNodes(node);
                    const responseBodies = this.extractResponseSchemaNodes(node);
                    requestBodySchemas.push(...requestBodies);
                    responseSchemas.push(...responseBodies);
                }
            }
        }
        return {
            requestBodies: requestBodySchemas,
            responses: responseSchemas
        };
    }
    /**
     * Traverse the schema starting at each node in nodes
     * @param nodes the nodes to traverse
     * @param visit a function to invoke per node
     */
    traverseSchemas(nodes, visit) {
        const seen = new Set();
        const recurse = (parent, node, opts) => {
            const schema = node.schema;
            if (!schema || seen.has(schema)) {
                return;
            }
            seen.add(schema);
            if (schema.$ref) {
                const resolvedSchema = this.resolveSchema(schema);
                const path = schema.$ref.split('/').slice(1);
                opts.req.originalSchema = schema;
                opts.res.originalSchema = schema;
                visit(parent, node, opts);
                recurse(node, new Node(schema, resolvedSchema, path), opts);
                return;
            }
            // Save the original schema so we can check if it was a $ref
            opts.req.originalSchema = schema;
            opts.res.originalSchema = schema;
            visit(parent, node, opts);
            if (schema.allOf) {
                schema.allOf.forEach((s, i) => {
                    const child = new Node(node, s, [...node.path, 'allOf', `${i}`]);
                    recurse(node, child, opts);
                });
            }
            else if (schema.oneOf) {
                schema.oneOf.forEach((s, i) => {
                    const child = new Node(node, s, [...node.path, 'oneOf', `${i}`]);
                    recurse(node, child, opts);
                });
            }
            else if (schema.anyOf) {
                schema.anyOf.forEach((s, i) => {
                    const child = new Node(node, s, [...node.path, 'anyOf', `${i}`]);
                    recurse(node, child, opts);
                });
            }
            else if (schema.type === 'array' && schema.items) {
                const child = new Node(node, schema.items, [...node.path, 'items']);
                recurse(node, child, opts);
            }
            else if (schema.properties) {
                Object.entries(schema.properties).forEach(([id, cschema]) => {
                    const path = [...node.path, 'properties', id];
                    const child = new Node(node, cschema, path);
                    recurse(node, child, opts);
                });
            }
        };
        const initOpts = () => ({
            req: { kind: 'req', path: [] },
            res: { kind: 'res', path: [] }
        });
        for (const node of nodes.schemas) {
            recurse(null, node, initOpts());
        }
        for (const node of nodes.requestBodies) {
            recurse(null, node, initOpts());
        }
        for (const node of nodes.responses) {
            recurse(null, node, initOpts());
        }
    }
    schemaVisitor(parent, node) {
        this.handleSerDes(parent === null || parent === void 0 ? void 0 : parent.schema, node.schema);
        this.handleReadonly(parent === null || parent === void 0 ? void 0 : parent.schema, node.schema, node.path);
    }
    handleSerDes(parent, schema) {
        if (schema.type === 'string'
            && !!schema.format
            && this.serDesMap[schema.format]) {
            schema.type = ['object', 'string'];
            // schema['x-eov-serdes'] = this.serDesMap[schema.format];
        }
    }
    handleReadonly(parent, schema, path) {
        var _a;
        if (!schema.readOnly) {
            return;
        }
        const prop = path === null || path === void 0 ? void 0 : path[(path === null || path === void 0 ? void 0 : path.length) - 1];
        const required = (_a = parent === null || parent === void 0 ? void 0 : parent.required) !== null && _a !== void 0 ? _a : [];
        parent.required = required.filter(p => p !== prop);
        if (parent.required.length === 0) {
            delete parent.required;
        }
    }
    /**
     * extract all requestBodies' schemas from an operation
     * @param op
     */
    extractRequestBodySchemaNodes(node) {
        const op = node.schema;
        const bodySchema = this.resolveSchema(op.requestBody);
        op.requestBody = bodySchema;
        if (!(bodySchema === null || bodySchema === void 0 ? void 0 : bodySchema.content)) {
            return [];
        }
        const result = [];
        const contentEntries = Object.entries(bodySchema.content);
        for (const [type, mediaTypeObject] of contentEntries) {
            const mediaTypeSchema = this.resolveSchema(mediaTypeObject.schema);
            op.requestBody.content[type].schema = mediaTypeSchema;
            const path = [...node.path, 'requestBody', 'content', type, 'schema'];
            result.push(new Root(mediaTypeSchema, path));
        }
        return result;
    }
    extractResponseSchemaNodes(node) {
        const op = node.schema;
        const responses = op.responses;
        if (!responses) {
            return [];
        }
        const schemas = [];
        for (const [statusCode, response] of Object.entries(responses)) {
            const rschema = this.resolveSchema(response);
            if (!rschema) {
                // issue #553
                // TODO the schema failed to resolve.
                // This can occur with multi-file specs
                // improve resolution, so that rschema resolves (use json ref parser?)
                continue;
            }
            responses[statusCode] = rschema;
            if (rschema.content) {
                for (const [type, mediaType] of Object.entries(rschema.content)) {
                    const schema = this.resolveSchema(mediaType === null || mediaType === void 0 ? void 0 : mediaType.schema);
                    if (schema) {
                        rschema.content[type].schema = schema;
                        const path = [
                            ...node.path,
                            'responses',
                            statusCode,
                            'content',
                            type,
                            'schema'
                        ];
                        schemas.push(new Root(schema, path));
                    }
                }
            }
        }
        return schemas;
    }
    resolveSchema(schema) {
        var _a;
        if (!schema) {
            return null;
        }
        const ref = schema === null || schema === void 0 ? void 0 : schema.$ref;
        let res = (ref ? (_a = this.ajv.getSchema(ref)) === null || _a === void 0 ? void 0 : _a.schema : schema);
        if (ref && !res) {
            const path = ref.split('/').join('.');
            const p = path.substring(path.indexOf('.') + 1);
            res = _get(this.apiDoc, p);
        }
        return res;
    }
    /**
     * add path level parameters to the schema's parameters list
     * @param pathItemKey
     * @param pathItem
     */
    preprocessPathLevelParameters(pathItemKey, pathItem) {
        var _a;
        const parameters = (_a = pathItem.parameters) !== null && _a !== void 0 ? _a : [];
        if (parameters.length === 0) {
            return;
        }
        const v = this.resolveSchema(pathItem[pathItemKey]);
        if (v === parameters) {
            return;
        }
        v.parameters = v.parameters || [];
        for (const param of parameters) {
            v.parameters.push(param);
        }
    }
}
exports.SchemaPreprocessor = SchemaPreprocessor;
