"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RequestValidator = void 0;
const { writeFile, readFile } = require('fs').promises;
const util_1 = require("./util");
const types_1 = require("../framework/types");
function getSchemaGeneral(validationId, parameters, body) {
    // $schema: "http://json-schema.org/draft-04/schema#",
    // eslint-disable-next-line dot-notation
    const isBodyBinary = (body === null || body === void 0 ? void 0 : body['format']) === 'binary';
    const bodyProps = !isBodyBinary && body || {};
    const bodySchema = {
        $id: validationId,
        required: ['query', 'headers', 'path'],
        properties: {
            query: {},
            headers: {},
            path: {},
            cookies: {},
            body: bodyProps,
            ...parameters
        }
    };
    const requireBody = body.required && !isBodyBinary;
    if (requireBody) {
        bodySchema.required.push('body');
    }
    return bodySchema;
}
class RequestValidator {
    constructor(apiDoc, options = {}) {
        this.middlewareCache = {};
        this.requestOpts = {};
        this.validationModule = {};
        this.middlewareCache = {};
        this.apiDoc = apiDoc;
        this.requestOpts.allowUnknownQueryParameters = options.allowUnknownQueryParameters;
        this.options = options;
    }
    validate(req) {
        var _a;
        const route = req.route;
        // cache middleware by combining method, path, and contentType
        const contentType = util_1.ContentType.from(req);
        const contentTypeKey = (_a = contentType.equivalents()[0]) !== null && _a !== void 0 ? _a : 'not_provided';
        const key = `${req.method || req.httpMethod}-${route}-${contentTypeKey}`.toLocaleLowerCase();
        if (!this.middlewareCache[key]) {
            if (this.validationModule[key]) {
                this.middlewareCache[key] = this.buildMiddlewareFromModule(key);
            }
            else {
                this.middlewareCache[key] = this.buildMiddleware(key, route, req.method || req.httpMethod, contentType);
            }
        }
        this.middlewareCache[key](req);
    }
    async compile(filepath) {
        var _a, _b;
        const { createRequestAjv } = require('../framework/ajv');
        const { BodySchemaParser } = require('./parsers/body.parse');
        const { ParametersSchemaParser } = require('./parsers/schema.parse');
        const ajvCompiled = createRequestAjv(this.apiDoc, Object.assign({ code: { source: true } }, this.options));
        const keyMap = {};
        for (const path of Object.keys(this.apiDoc.paths)) {
            for (const method of Object.keys(this.apiDoc.paths[path])) {
                const reqSchema = this.apiDoc.paths[path][method];
                const schemaParser = new ParametersSchemaParser(ajvCompiled, this.apiDoc);
                const parameters = schemaParser.parse(path, reqSchema.parameters);
                // Include no body with `null`
                let defaultKey;
                for (const contentTypeRaw of Object.keys(((_a = reqSchema.requestBody) === null || _a === void 0 ? void 0 : _a.content) || {}).concat(null)) {
                    const contentTypeData = util_1.ContentType.fromString(contentTypeRaw);
                    const contentTypeKey = (_b = (contentTypeRaw && contentTypeData.equivalents()[0])) !== null && _b !== void 0 ? _b : 'not_provided';
                    const key = `${method}-${path}-${contentTypeKey}`.toLocaleLowerCase();
                    if (!contentTypeRaw && defaultKey) {
                        keyMap[key] = defaultKey;
                        continue;
                    }
                    keyMap[key] = defaultKey = key;
                    const body = contentTypeRaw && new BodySchemaParser().parse(path, reqSchema, contentTypeData);
                    // eslint-disable-next-line dot-notation
                    const isBodyBinary = (body === null || body === void 0 ? void 0 : body['format']) === 'binary';
                    const bodyProps = !isBodyBinary && body || {};
                    const bodySchema = {
                        $id: key,
                        required: ['query', 'headers', 'path'],
                        properties: {
                            query: {
                                additionalProperties: !!this.requestOpts.allowUnknownQueryParameters
                            },
                            headers: {},
                            path: {},
                            cookies: {},
                            body: bodyProps,
                            ...parameters
                        }
                    };
                    const requireBody = (body === null || body === void 0 ? void 0 : body.required) && !isBodyBinary;
                    if (requireBody) {
                        bodySchema.required.push('body');
                    }
                    ajvCompiled.addSchema(bodySchema);
                }
            }
        }
        const standaloneCode = require('ajv/dist/standalone').default;
        const moduleCode = standaloneCode(ajvCompiled, keyMap);
        if (filepath) {
            await writeFile(filepath, moduleCode);
        }
    }
    async loadCompiled(filepath) {
        const moduleCode = await readFile(filepath);
        this.validationModule = require('require-from-string')(moduleCode.toString());
    }
    buildMiddlewareFromModule(validationId) {
        return (req) => {
            var _a, _b, _c, _d, _e, _f, _g;
            const data = {
                query: (_b = (_a = req.query) !== null && _a !== void 0 ? _a : req.queryStringParameters) !== null && _b !== void 0 ? _b : {},
                headers: (_c = req.headers) !== null && _c !== void 0 ? _c : {},
                path: (_e = (_d = req.path) !== null && _d !== void 0 ? _d : req.pathParameters) !== null && _e !== void 0 ? _e : {},
                cookies: (_f = req.cookies) !== null && _f !== void 0 ? _f : {},
                body: req.body
            };
            const validator = this.validationModule[validationId];
            const valid = validator(data);
            if (valid) {
                return;
            }
            const errors = (0, util_1.augmentAjvErrors)([].concat((_g = validator.errors) !== null && _g !== void 0 ? _g : []));
            const formattedErrors = (0, util_1.ajvErrorsToValidatorError)(errors);
            let message = 'No errors';
            if (formattedErrors.length) {
                message = formattedErrors.map(m => m.fullMessage).join(', ');
            }
            const error = new types_1.BadRequest({
                path: req.route,
                message: message
            });
            error.errors = formattedErrors;
            throw error;
        };
    }
    buildMiddleware(validationId, path, method, contentType) {
        const reqSchema = this.apiDoc && this.apiDoc.paths[path] && this.apiDoc.paths[path][method.toLowerCase()];
        if (!reqSchema) {
            return (() => { });
        }
        if (!this.ajv) {
            const { createRequestAjv } = require('../framework/ajv');
            this.ajv = createRequestAjv(this.apiDoc, this.options);
        }
        const { BodySchemaParser } = require('./parsers/body.parse');
        const { ParametersSchemaParser } = require('./parsers/schema.parse');
        const schemaParser = new ParametersSchemaParser(this.ajv, this.apiDoc);
        const parameters = schemaParser.parse(path, reqSchema.parameters);
        const body = new BodySchemaParser().parse(path, reqSchema, contentType);
        const schemaGeneral = getSchemaGeneral(validationId, parameters, body);
        const validator = this.ajv.compile(schemaGeneral);
        return (req) => {
            var _a, _b, _c, _d, _e, _f, _g;
            const data = {
                query: (_b = (_a = req.query) !== null && _a !== void 0 ? _a : req.queryStringParameters) !== null && _b !== void 0 ? _b : {},
                headers: (_c = req.headers) !== null && _c !== void 0 ? _c : {},
                path: (_e = (_d = req.path) !== null && _d !== void 0 ? _d : req.pathParameters) !== null && _e !== void 0 ? _e : {},
                cookies: (_f = req.cookies) !== null && _f !== void 0 ? _f : {},
                body: req.body
            };
            const valid = validator(data);
            if (valid) {
                return;
            }
            const errors = (0, util_1.augmentAjvErrors)([].concat((_g = validator.errors) !== null && _g !== void 0 ? _g : []));
            const formattedErrors = (0, util_1.ajvErrorsToValidatorError)(errors);
            // const message = this.ajv.errorsText(errors, { dataVar: 'request' });
            const message = formattedErrors.map(m => m.fullMessage).join(', ');
            const error = new types_1.BadRequest({
                path: req.route,
                message: message
            });
            error.errors = formattedErrors;
            throw error;
        };
    }
}
exports.RequestValidator = RequestValidator;
