import cloneDeep from 'lodash.clonedeep';
import merge from 'lodash.merge';
import { expandN } from 'regex-to-strings';
import xmlFormatter from './xml/xml.js'; // When the type is not known for a property set the displayed type to be this:

const IS_MISSING_TYPE_INFO_TYPE = '';
const EXAMPLE_VALUE_FOR_MISSING_TYPE = '';
/* Generates an schema object containing type and constraint info */

export function getTypeInfo(schema, options = {
  includeNulls: false
}) {
  var _schema$items, _schema$const;

  if (!schema) {
    return undefined;
  }

  let dataType = IS_MISSING_TYPE_INFO_TYPE;

  if (schema.circularReference) {
    dataType = `{recursive: ${schema.circularReference.name}} `;
  } else if (schema.type) {
    const arraySchema = Array.isArray(schema.type) ? schema.type : typeof schema.type === 'string' ? schema.type.split('┃') : schema.type;
    dataType = Array.isArray(arraySchema) ? arraySchema.filter(s => s !== 'null' || options.includeNulls).join('┃') : schema.type;
    dataType = dataType.replace('string', typeof schema.const !== 'undefined' && 'const' || schema.enum && 'enum' || schema.format || 'string');

    if (schema.nullable && options.includeNulls) {
      dataType += '┃null';
    }

    if (dataType.includes('┃null') && schema.format) {
      schema.format += '┃null';
    }
  }

  const info = {
    type: dataType,
    format: schema.format || ((_schema$items = schema.items) === null || _schema$items === void 0 ? void 0 : _schema$items.format) || '',
    cssType: dataType.replace(/┃.*/g, '').replace(/[^a-zA-Z0-9+]/g, '').toLowerCase(),
    pattern: schema.pattern && !schema.enum ? schema.pattern : '',
    readOrWriteOnly: schema.readOnly && '🆁' || schema.writeOnly && '🆆' || '',
    deprecated: !!schema.deprecated,
    example: Array.isArray(schema.example) ? schema.example : typeof schema.example !== 'undefined' ? `${schema.example}` : '',
    default: schema.default || '',
    title: schema.title || '',
    description: schema.description || '',
    constraint: '',
    allowedValues: (_schema$const = schema.const) !== null && _schema$const !== void 0 ? _schema$const : Array.isArray(schema.enum) ? schema.enum.join('┃') : '',
    arrayType: ''
  };

  if (dataType === 'array' && schema.items) {
    var _schema$items$default, _schema$const2;

    const arrayItemType = schema.items.type;
    const arrayItemDefault = (_schema$items$default = schema.items.default) !== null && _schema$items$default !== void 0 ? _schema$items$default : '';
    info.arrayType = `${schema.type} of ${Array.isArray(arrayItemType) ? arrayItemType.join('') : arrayItemType}`;
    info.default = arrayItemDefault;
    info.allowedValues = (_schema$const2 = schema.const) !== null && _schema$const2 !== void 0 ? _schema$const2 : Array.isArray(schema.items.enum) ? schema.items.enum.join('┃') : '';
  }

  if (dataType.match(/integer|number/g)) {
    var _schema$minimum, _schema$maximum;

    const minimum = (_schema$minimum = schema.minimum) !== null && _schema$minimum !== void 0 ? _schema$minimum : schema.exclusiveMinimum;
    const maximum = (_schema$maximum = schema.maximum) !== null && _schema$maximum !== void 0 ? _schema$maximum : schema.exclusiveMaximum;
    const leftBound = schema.minimum !== undefined ? '[' : '(';
    const rightBound = schema.maximum !== undefined ? ']' : ')';

    if (typeof minimum === 'number' || typeof maximum === 'number') {
      info.constraint = `Range: ${leftBound}${minimum !== null && minimum !== void 0 ? minimum : ''},${maximum !== null && maximum !== void 0 ? maximum : ''}${rightBound}`;
    }

    if (schema.multipleOf !== undefined) {
      info.constraint += `${info.constraint ? ', ' : ''}Multiples: ${schema.multipleOf}`;
    }
  }

  if (dataType.match(/string/g)) {
    if (schema.minLength !== undefined && schema.maxLength !== undefined) {
      info.constraint += `Min length: ${schema.minLength}, Max length: ${schema.maxLength}`;
    } else if (schema.minLength !== undefined) {
      info.constraint += `Min length: ${schema.minLength}`;
    } else if (schema.maxLength !== undefined) {
      info.constraint += `Max length: ${schema.maxLength}`;
    }
  }

  info.html = JSON.stringify({
    type: info.type,
    format: info.format,
    cssType: info.cssType,
    readOrWriteOnly: info.readOrWriteOnly,
    constraint: info.constraint,
    defaultValue: info.default,
    example: info.example,
    allowedValues: info.allowedValues,
    pattern: info.pattern,
    schemaDescription: info.description,
    schemaTitle: info.title,
    deprecated: info.deprecated
  });
  return info;
}
export function getSampleValueByType(schemaObj, fallbackPropertyName, skipExampleStrings) {
  var _Object$values$0$valu, _Object$values$, _schemaObj$type;

  const example = Array.isArray(schemaObj.examples) ? schemaObj.examples[0] : (_Object$values$0$valu = (_Object$values$ = Object.values(schemaObj.examples || {})[0]) === null || _Object$values$ === void 0 ? void 0 : _Object$values$.value) !== null && _Object$values$0$valu !== void 0 ? _Object$values$0$valu : schemaObj.example;

  if (skipExampleStrings && typeof example === 'string') {
    return '';
  }

  if (typeof example !== 'undefined') {
    return example;
  }

  if (schemaObj.default) {
    return schemaObj.default;
  }

  if (Object.keys(schemaObj).length === 0) {
    return EXAMPLE_VALUE_FOR_MISSING_TYPE;
  }

  if (schemaObj.$ref) {
    // Indicates a Circular ref
    return schemaObj.$ref;
  }

  const typeValue = Array.isArray(schemaObj.type) ? schemaObj.type.filter(t => t !== 'null')[0] : (_schemaObj$type = schemaObj.type) !== null && _schemaObj$type !== void 0 ? _schemaObj$type : '';

  if (typeof schemaObj.const !== 'undefined') {
    return schemaObj.const;
  }

  if (typeValue.match(/^integer|^number/g)) {
    const multipleOf = Number.isNaN(Number(schemaObj.multipleOf)) ? undefined : Number(schemaObj.multipleOf);
    const maximum = Number.isNaN(Number(schemaObj.maximum)) ? undefined : Number(schemaObj.maximum);
    const minimumPossibleVal = Number.isNaN(Number(schemaObj.minimum)) ? Number.isNaN(Number(schemaObj.exclusiveMinimum)) ? maximum || 0 : Number(schemaObj.exclusiveMinimum) + (typeValue.startsWith('integer') ? 1 : 0.001) : Number(schemaObj.minimum);
    const finalVal = multipleOf ? multipleOf >= minimumPossibleVal ? multipleOf : minimumPossibleVal % multipleOf === 0 ? minimumPossibleVal : Math.ceil(minimumPossibleVal / multipleOf) * multipleOf : minimumPossibleVal;
    return finalVal;
  }

  if (typeValue.match(/^boolean/g)) {
    return false;
  }

  if (typeValue.match(/^null/g)) {
    return null;
  }

  if (skipExampleStrings && typeValue.match(/^string/g)) {
    return '';
  }

  if (typeValue.match(/^string/g)) {
    if (schemaObj.enum) {
      return schemaObj.enum[0];
    }

    if (schemaObj.pattern) {
      const examplePattern = schemaObj.pattern.replace(/[+*](?![^\][]*[\]])/g, '{8}').replace(/\{\d*,(\d+)?\}/g, '{8}');
      return expandN(examplePattern, 1)[0] || fallbackPropertyName || 'string';
    }

    if (schemaObj.format) {
      switch (schemaObj.format.toLowerCase()) {
        case 'url':
          return 'https://example.com';

        case 'uri':
          return 'urn:namespace:type:example/resource';

        case 'date':
          return new Date().toISOString().split('T')[0];

        case 'time':
          return new Date().toISOString().split('T')[1];

        case 'date-time':
          return new Date().toISOString();

        case 'duration':
          return 'P3Y6M4DT12H30M5S';
        // P=Period 3-Years 6-Months 4-Days 12-Hours 30-Minutes 5-Seconds

        case 'email':
        case 'idn-email':
          return 'user@example.com';

        case 'hostname':
        case 'idn-hostname':
          return 'www.example.com';

        case 'ipv4':
          return '192.168.0.1';

        case 'ipv6':
          return '2001:0db8:5b96:0000:0000:426f:8e17:642a';

        case 'uuid':
          return '4e0ba220-9575-11eb-a8b3-0242ac130003';

        default:
          return schemaObj.format;
      }
    } else {
      return fallbackPropertyName || 'string';
    }
  } // If type cannot be determined


  return EXAMPLE_VALUE_FOR_MISSING_TYPE;
}

function duplicateExampleWithNewPropertyValues(objectExamples, propertyName, propertyValues) {
  // Limit max number of property examples to 2 and limit the max number of examples to 10
  return objectExamples.reduce((exampleList, example) => {
    const examplesFromPropertyValues = propertyValues.slice(0, 2).map(value => ({ ...cloneDeep(example),
      [propertyName]: value
    }));
    return exampleList.concat(...examplesFromPropertyValues);
  }, []).slice(0, 10);
}

export function getExampleValuesFromSchema(schema, config = {}) {
  // Wrap the top level so that the recursive object can treat it as a normal property and we'll hit the 'object' below, otherwise we'll never create the top level.
  if (config.xml) {
    const xmlResult = getExampleValuesFromSchemaRecursive(schema.type === 'object' ? {
      properties: {
        _root: schema
      }
    } : schema, config);
    return xmlResult.map(example => example[0]);
  }

  return getExampleValuesFromSchemaRecursive(schema, config);
} // TODO: Support getting the `summary` from the examples object or the `title` from the schema object

function getExampleValuesFromSchemaRecursive(rawSchema, config = {}) {
  var _rawSchema$items, _rawSchema$items$xml, _rawSchema$xml;

  if (!rawSchema) {
    return [];
  } // XML Support


  const xmlAttributes = {};
  const xmlTagProperties = [];
  const {
    prefix,
    namespace
  } = rawSchema.xml || {};

  if (namespace) {
    xmlAttributes[prefix ? `xmlns:${prefix}` : 'xmlns'] = namespace;
  }

  const nodeName = (rawSchema === null || rawSchema === void 0 ? void 0 : (_rawSchema$items = rawSchema.items) === null || _rawSchema$items === void 0 ? void 0 : (_rawSchema$items$xml = _rawSchema$items.xml) === null || _rawSchema$items$xml === void 0 ? void 0 : _rawSchema$items$xml.name) || (rawSchema === null || rawSchema === void 0 ? void 0 : (_rawSchema$xml = rawSchema.xml) === null || _rawSchema$xml === void 0 ? void 0 : _rawSchema$xml.name) || config.propertyName || 'root';
  const overridePropertyName = prefix ? `${prefix}:${nodeName}` : nodeName;
  const {
    allOf,
    oneOf,
    anyOf,
    ...schema
  } = rawSchema;

  if (allOf) {
    const mergedAllOf = merge({}, ...schema.allOf, schema);
    return getExampleValuesFromSchemaRecursive(mergedAllOf, config);
  }

  if (oneOf || anyOf) {
    const examples = (oneOf || anyOf).map(s => getExampleValuesFromSchemaRecursive(merge({}, schema, s), config)).flat(1);

    const hash = value => {
      if (typeof value === 'object') {
        return JSON.stringify(value);
      }

      return value;
    };

    const uniqueExamples = examples.reduce((acc, e) => {
      acc[hash(e)] = e;
      return acc;
    }, {});
    return Object.values(uniqueExamples);
  }

  return getSimpleValueResult(schema, config, namespace, prefix, xmlAttributes, xmlTagProperties, overridePropertyName);
}

function getSimpleValueResult(schema, config, namespace, prefix, xmlAttributes, xmlTagProperties, overridePropertyName) {
  if (schema.type === 'array' || schema.items) {
    if (!config.xml) {
      return [getExampleValuesFromSchemaRecursive(schema.items || {}, config)];
    }

    if (!schema.xml || !schema.xml.wrapped) {
      const arrayExamples = getExampleValuesFromSchemaRecursive(schema.items || {}, config);
      xmlTagProperties.push({
        [overridePropertyName]: arrayExamples[0]
      }, {
        _attr: xmlAttributes
      });
      return [xmlTagProperties];
    }

    const arrayExamples = getExampleValuesFromSchemaRecursive(schema.items || {}, { ...config,
      propertyName: overridePropertyName
    });
    xmlTagProperties.push({
      [overridePropertyName]: arrayExamples[0]
    }, {
      _attr: xmlAttributes
    });
    return [xmlTagProperties];
  }

  if (schema.type === 'object' || schema.properties) {
    let objectExamples = [{}];
    Object.keys(schema.properties || {}).forEach(propertyName => {
      const innerSchema = schema.properties[propertyName];

      if (innerSchema.deprecated) {
        return;
      }

      if (innerSchema.readOnly && !config.includeReadOnly) {
        return;
      }

      if (innerSchema.writeOnly && !config.includeWriteOnly) {
        return;
      }

      const propertyExamples = getExampleValuesFromSchemaRecursive(innerSchema, { ...config,
        propertyName
      });
      objectExamples = duplicateExampleWithNewPropertyValues(objectExamples, propertyName, propertyExamples);

      if (innerSchema.xml && innerSchema.xml.namespace) {
        xmlAttributes[innerSchema.xml.prefix ? `xmlns:${innerSchema.xml.prefix}` : 'xmlns'] = namespace;
      }

      const innerNodeName = innerSchema.xml && innerSchema.xml.name || propertyName || config.propertyName;
      const innerOverridePropertyName = prefix ? `${prefix}:${innerNodeName}` : innerNodeName;

      if (innerSchema.xml && innerSchema.xml.attribute) {
        xmlAttributes[innerOverridePropertyName] = propertyExamples[0];
      } else {
        xmlTagProperties.push({
          [innerOverridePropertyName]: propertyExamples[0]
        });
      }
    });

    if (Object.keys(xmlAttributes).length) {
      xmlTagProperties.push({
        _attr: xmlAttributes
      });
    }

    return config.xml ? [xmlTagProperties] : objectExamples;
  }

  const value = getSampleValueByType(schema, config.propertyName, config.skipExampleStrings);
  return [value];
}
/**
 * For changing OpenAPI-Schema to an Object Notation,
 * This Object would further be an input to UI Components to generate an Object-Tree
 * @param {object} schema - Schema object from OpenAPI spec
 * @param {object} options - recursively pass this object to generate object notation
 * @param {number} level - recursion level
 * @param {string} suffix - used for suffixing property names to avoid duplicate props during object composition
 */


export function schemaInObjectNotation(rawSchema, options, level = 0, suffix = '') {
  if (!rawSchema) {
    return undefined;
  }

  const {
    allOf,
    oneOf,
    anyOf,
    ...schema
  } = rawSchema;

  if (allOf) {
    // If allOf is an array of multiple elements, then all the keys makes a single object
    const objWithAllProps = {};
    allOf.map((v, i) => {
      if (v.type === 'object' || v.properties || v.allOf || v.anyOf || v.oneOf) {
        const propSuffix = (v.anyOf || v.oneOf) && i > 0 ? i : '';
        const partialObj = schemaInObjectNotation(v, options, level + 1, propSuffix);
        Object.assign(objWithAllProps, partialObj);
      } else if (v.type === 'array' || v.items) {
        const partialObj = schemaInObjectNotation(v, options, level + 1);
        Object.assign(objWithAllProps, partialObj);
      } else if (v.type) {
        const prop = `prop${Object.keys(objWithAllProps).length}`;
        const typeObj = getTypeInfo(v, options);
        objWithAllProps[prop] = `${typeObj.html}`;
      }
    });
    const obj = schemaInObjectNotation(schema, options, 0);
    return Object.assign({}, objWithAllProps, typeof obj === 'object' && !Array.isArray(obj) ? obj : {});
  } else if (anyOf || oneOf) {
    const objWithAnyOfProps = {};
    let writeOnly = true;
    let readOnly = true;
    (anyOf || oneOf || []).forEach((v, index) => {
      if (v.type === 'object' || v.properties || v.allOf || v.anyOf || v.oneOf) {
        const partialObj = schemaInObjectNotation(v, options);

        if (partialObj) {
          var _partialObj$Flags, _partialObj$Flags2;

          objWithAnyOfProps[`::OPTION~${index + 1}${v.title ? `~${v.title}` : ''}`] = partialObj;
          objWithAnyOfProps['::type'] = 'xxx-of-option';
          readOnly = readOnly && ((_partialObj$Flags = partialObj['::flags']) === null || _partialObj$Flags === void 0 ? void 0 : _partialObj$Flags['🆁']);
          writeOnly = writeOnly && ((_partialObj$Flags2 = partialObj['::flags']) === null || _partialObj$Flags2 === void 0 ? void 0 : _partialObj$Flags2['🆆']);
        }
      } else if (v.type === 'array' || v.items) {
        var _partialObj$Flags3, _partialObj$Flags4;

        const partialObj = schemaInObjectNotation(v, options);
        objWithAnyOfProps[`::OPTION~${index + 1}${v.title ? `~${v.title}` : ''}`] = partialObj;
        objWithAnyOfProps['::type'] = 'xxx-of-array';
        readOnly = readOnly && ((_partialObj$Flags3 = partialObj['::flags']) === null || _partialObj$Flags3 === void 0 ? void 0 : _partialObj$Flags3['🆁']);
        writeOnly = writeOnly && ((_partialObj$Flags4 = partialObj['::flags']) === null || _partialObj$Flags4 === void 0 ? void 0 : _partialObj$Flags4['🆆']);
      } else {
        const typeInfo = getTypeInfo(v, options);

        if (typeInfo !== null && typeInfo !== void 0 && typeInfo.type) {
          var _objWithAnyOfProps$, _objWithAnyOfProps$2;

          const prop = `::OPTION~${index + 1}${v.title ? `~${v.title}` : ''}`;
          objWithAnyOfProps[prop] = `${typeInfo.html}`;
          objWithAnyOfProps['::type'] = 'xxx-of-option';
          readOnly = readOnly && ((_objWithAnyOfProps$ = objWithAnyOfProps['::flags']) === null || _objWithAnyOfProps$ === void 0 ? void 0 : _objWithAnyOfProps$['🆁']);
          writeOnly = writeOnly && ((_objWithAnyOfProps$2 = objWithAnyOfProps['::flags']) === null || _objWithAnyOfProps$2 === void 0 ? void 0 : _objWithAnyOfProps$2['🆆']);
        }
      }
    });
    const obj = schemaInObjectNotation(schema, options, 0);
    const resultObj = typeof obj === 'object' && !Array.isArray(obj) ? obj : {};

    if (Object.keys(objWithAnyOfProps).length) {
      resultObj[anyOf ? `::ANY~OF ${suffix}` : `::ONE~OF ${suffix}`] = objWithAnyOfProps;
    }

    resultObj['::type'] = 'object';
    resultObj['::flags'] = {
      '🆁': readOnly && '🆁',
      '🆆': writeOnly && '🆆'
    };
    return resultObj;
  } else if (Array.isArray(schema.type)) {
    const obj = {
      '::type': ''
    }; // When a property has multiple types, then check further if any of the types are array or object, if yes then modify the schema using one-of
    // Clone the schema - as it will be modified to replace multi-data-types with one-of;

    const subSchema = JSON.parse(JSON.stringify(schema));
    const primitiveType = [];
    const complexTypes = [];
    subSchema.type.forEach(v => {
      if (v.match(/integer|number|string|null|boolean/g)) {
        primitiveType.push(v);
      } else if (v === 'array' && typeof (subSchema.items && subSchema.items.type) === 'string' && schema.items && subSchema.items.type.match(/integer|number|string|null|boolean/g)) {
        // Array with primitive types should also be treated as primitive type
        if (subSchema.items.type === 'string' && subSchema.items.format) {
          primitiveType.push(`${subSchema.items.format}[]`);
        } else {
          primitiveType.push(`${subSchema.items.type}[]`);
        }
      } else {
        complexTypes.push(v);
      }
    });
    let multiPrimitiveTypes;

    if (primitiveType.length > 0) {
      subSchema.type = primitiveType.join('┃');
      multiPrimitiveTypes = getTypeInfo(subSchema, options);

      if (complexTypes.length === 0) {
        var _multiPrimitiveTypes;

        return `${((_multiPrimitiveTypes = multiPrimitiveTypes) === null || _multiPrimitiveTypes === void 0 ? void 0 : _multiPrimitiveTypes.html) || ''}`;
      }
    }

    if (complexTypes.length > 0) {
      obj['::type'] = 'object';
      const multiTypeOptions = {
        '::type': 'xxx-of-option'
      }; // Generate ONE-OF options for complexTypes

      complexTypes.forEach((v, i) => {
        if (v === 'null') {
          multiTypeOptions[`::OPTION~${i + 1}`] = 'NULL~|~~|~~|~~|~~|~~|~~|~~|~';
        } else if ('integer, number, string, boolean,'.includes(`${v},`)) {
          subSchema.type = Array.isArray(v) ? v.join('┃') : v;
          const primitiveTypeInfo = getTypeInfo(subSchema, options);
          multiTypeOptions[`::OPTION~${i + 1}`] = primitiveTypeInfo.html;
        } else if (v === 'object') {
          // If object type iterate all the properties and create an object-type-option
          const objTypeOption = {
            '::title': schema.title || '',
            '::description': schema.description || '',
            '::flags': {
              '🆁': schema.readOnly && '🆁',
              '🆆': schema.writeOnly && '🆆'
            },
            '::type': 'object',
            '::deprecated': schema.deprecated || false
          };

          for (const key in schema.properties) {
            if (schema.required && schema.required.includes(key)) {
              objTypeOption[`${key}*`] = schemaInObjectNotation(schema.properties[key], options, level + 1);
            } else {
              objTypeOption[key] = schemaInObjectNotation(schema.properties[key], options, level + 1);
            }
          }

          multiTypeOptions[`::OPTION~${i + 1}`] = objTypeOption;
        } else if (v === 'array') {
          multiTypeOptions[`::OPTION~${i + 1}`] = {
            '::title': schema.title || '',
            '::description': schema.description || '',
            '::flags': {
              '🆁': schema.readOnly && '🆁',
              '🆆': schema.writeOnly && '🆆'
            },
            '::type': 'array',
            '::props': schemaInObjectNotation(Object.assign({
              readOnly: schema.readOnly,
              writeOnly: schema.writeOnly
            }, schema.items), options, level + 1)
          };
        }
      });
      multiTypeOptions[`::OPTION~${complexTypes.length + 1}`] = multiPrimitiveTypes && multiPrimitiveTypes.html || '';
      obj['::ONE~OF'] = multiTypeOptions;
    }

    return obj;
  } else if (schema.type === 'object' || schema.properties) {
    const obj = {
      '::type': ''
    };
    obj['::title'] = schema.title || '';
    obj['::description'] = schema.description || '';
    obj['::flags'] = {
      '🆁': schema.readOnly && '🆁',
      '🆆': schema.writeOnly && '🆆'
    };
    obj['::type'] = 'object';
    obj['::deprecated'] = schema.deprecated || false;

    for (const key in schema.properties) {
      if (schema.required && schema.required.includes(key)) {
        obj[`${key}*`] = schemaInObjectNotation(schema.properties[key], options, level + 1);
      } else {
        obj[key] = schemaInObjectNotation(schema.properties[key], options, level + 1);
      }
    }

    if (schema.additionalProperties) {
      obj['<any-key>'] = schemaInObjectNotation(schema.additionalProperties, options);
    }

    return obj;
  } else if (schema.type === 'array' || schema.items) {
    var _schema$items2, _schema$items3;

    // If Array
    const obj = {
      '::type': ''
    };
    obj['::title'] = schema.title || '';
    obj['::description'] = schema.description ? schema.description : (_schema$items2 = schema.items) !== null && _schema$items2 !== void 0 && _schema$items2.description ? `array&lt;${schema.items.description}&gt;` : '';
    obj['::flags'] = {
      '🆁': schema.readOnly && '🆁',
      '🆆': schema.writeOnly && '🆆'
    };
    obj['::type'] = 'array';
    obj['::deprecated'] = schema.deprecated || false;
    obj['::props'] = schemaInObjectNotation(Object.assign({
      deprecated: schema.deprecated,
      readOnly: schema.readOnly,
      writeOnly: schema.writeOnly
    }, schema.items), options, level + 1);

    if ((_schema$items3 = schema.items) !== null && _schema$items3 !== void 0 && _schema$items3.items) {
      obj['::array-type'] = schema.items.items.type;
    }

    return obj;
  }

  const typeObj = getTypeInfo(schema, options);
  return `${(typeObj === null || typeObj === void 0 ? void 0 : typeObj.html) || ''}`;
}
/* Create Example object */

export function generateExample(examples, example, schema, rawMimeType, includeReadOnly = true, includeWriteOnly = true, outputType, skipExampleStrings = false) {
  const mimeType = rawMimeType || 'application/json';
  const finalExamples = []; // First check if examples is provided

  if (examples) {
    for (const eg in examples) {
      let egContent = '';
      let egFormat = 'json';

      if (mimeType.toLowerCase().includes('json')) {
        if (outputType === 'text') {
          egContent = typeof examples[eg].value === 'string' ? examples[eg].value : JSON.stringify(examples[eg].value, undefined, 2);
          egFormat = 'text';
        } else {
          egContent = examples[eg].value;

          if (typeof examples[eg].value === 'string') {
            try {
              const fixedJsonString = examples[eg].value.replace(/([\w]+)(:)/g, '"$1"$2').replace(/'/g, '"');
              egContent = JSON.parse(fixedJsonString);
              egFormat = 'json';
            } catch (err) {
              egFormat = 'text';
              egContent = examples[eg].value;
            }
          }
        }
      } else {
        egContent = examples[eg].value;
        egFormat = 'text';
      }

      finalExamples.push({
        exampleId: eg,
        exampleSummary: examples[eg].summary || eg,
        exampleDescription: examples[eg].description || '',
        exampleType: mimeType,
        exampleValue: egContent,
        exampleFormat: egFormat
      });
    }
  } else if (example) {
    let egContent = '';
    let egFormat = 'json';

    if (mimeType.toLowerCase().includes('json')) {
      if (outputType === 'text') {
        egContent = typeof example === 'string' ? example : JSON.stringify(example, undefined, 2);
        egFormat = 'text';
      } else if (typeof example === 'object') {
        egContent = example;
        egFormat = 'json';
      } else if (typeof example === 'string') {
        try {
          egContent = JSON.parse(example);
          egFormat = 'json';
        } catch (err) {
          egFormat = 'text';
          egContent = example;
        }
      }
    } else {
      egContent = example;
      egFormat = 'text';
    }

    finalExamples.push({
      exampleId: 'Example',
      exampleSummary: '',
      exampleDescription: '',
      exampleType: mimeType,
      exampleValue: egContent,
      exampleFormat: egFormat
    });
  } // If schema-level examples are not provided then generate one based on the schema field types


  if (finalExamples.length) {
    return finalExamples;
  }

  if (schema !== null && schema !== void 0 && schema.example) {
    // Note: schema.examples (plurals) is not allowed as per spec
    return [{
      exampleId: 'Example',
      exampleSummary: '',
      exampleDescription: '',
      exampleType: mimeType,
      exampleValue: schema.example,
      exampleFormat: mimeType.toLowerCase().includes('json') && typeof schema.example === 'object' ? 'json' : 'text'
    }];
  }

  const config = {
    includeReadOnly,
    includeWriteOnly,
    skipExampleStrings,
    xml: mimeType.toLowerCase().includes('xml')
  };
  const samples = getExampleValuesFromSchema(schema, config);

  if (!samples || !mimeType.toLowerCase().includes('json') && !mimeType.toLowerCase().includes('text') && !mimeType.toLowerCase().includes('*/*') && !mimeType.toLowerCase().includes('xml')) {
    return [{
      exampleId: 'Example',
      exampleSummary: '',
      exampleDescription: '',
      exampleType: mimeType,
      exampleValue: '',
      exampleFormat: 'text'
    }];
  }

  return samples.map((sample, sampleCounter) => {
    let exampleValue = '';

    if (mimeType.toLowerCase().includes('xml')) {
      exampleValue = xmlFormatter(sample, {
        declaration: true,
        indent: '    '
      });
    } else {
      exampleValue = outputType === 'text' ? JSON.stringify(sample, null, 8) : sample;
    }

    return {
      exampleId: `Example-${sampleCounter}`,
      exampleSummary: `Example ${sampleCounter + 1}`,
      exampleDescription: '',
      exampleType: mimeType,
      exampleFormat: mimeType.toLowerCase().includes('xml') ? 'text' : outputType,
      exampleValue
    };
  }).filter(s => s);
}