"use strict";

exports.__esModule = true;
exports.default = void 0;

var _lit = require("lit");

class TagInput extends _lit.LitElement {
  render() {
    const tagItemTemplate = (0, _lit.html)`${(this.value || []).filter(v => v.trim()).map(v => (0, _lit.html)`<span class="tag">${v}</span>`)}`;
    return (0, _lit.html)` <div class="tags" tabindex="0"> ${tagItemTemplate} <input type="text" class="editor" @paste="${e => this.afterPaste(e)}" @keydown="${this.afterKeyDown}" placeholder="${this.placeholder || ''}"> </div> `;
  }

  static get properties() {
    return {
      placeholder: {
        type: String
      },
      value: {
        type: Array,
        attribute: 'value'
      }
    };
  }

  connectedCallback() {
    super.connectedCallback();

    if (!Array.isArray(this.value)) {
      this.value = this.value !== '' ? [this.value] : [];
    }
  }

  attributeChangedCallback(name, oldVal, newVal) {
    if (name === 'value') {
      if (newVal && oldVal !== newVal) {
        this.value = newVal.split(',').filter(v => v.trim());
      }
    }

    super.attributeChangedCallback(name, oldVal, newVal);
  }

  afterPaste(e) {
    const clipboardData = e.clipboardData || window.clipboardData;
    const pastedData = clipboardData.getData('Text');
    const pastedArray = pastedData && pastedData.split(',').filter(v => v.trim()) || [];
    this.value = this.value.concat(pastedArray);
    e.preventDefault();
  }

  afterKeyDown(e) {
    if (e.keyCode === 13) {
      e.stopPropagation();
      e.preventDefault();
      this.value = this.value.concat(e.target.value || []);
      e.target.value = '';
    } else if (e.keyCode === 8) {
      if (e.target.value.length === 0) {
        this.value = this.value.slice(0, -1);
      }
    }
  }

  static finalizeStyles() {
    return [(0, _lit.css)`.tags{display:flex;flex-wrap:wrap;outline:0;padding:0;border-radius:var(--border-radius);border:1px solid var(--border-color);cursor:text;overflow:hidden;background:var(--input-bg)}.editor,.tag{padding:3px;margin:2px}.tag{border:1px solid var(--border-color);background-color:var(--bg3);color:var(--fg3);border-radius:var(--border-radius);word-break:break-all;font-size:var(--font-size-small)}.tag:hover~#cursor{display:block}.editor{flex:1;border:1px solid transparent;color:var(--fg);min-width:60px;outline:0;line-height:inherit;font-family:inherit;background:0 0;font-size:calc(var(--font-size-small) + 1px)}.editor::placeholder{color:var(--placeholder-color);opacity:1}`];
  }

} // Register the element with the browser


exports.default = TagInput;

if (!customElements.get('openapi-explorer')) {
  customElements.define('tag-input', TagInput);
}