"use strict";

exports.__esModule = true;
exports.advancedSearch = advancedSearch;
exports.componentIsInSearch = componentIsInSearch;
exports.copyToClipboard = copyToClipboard;
exports.debounce = debounce;
exports.getBaseUrlFromUrl = getBaseUrlFromUrl;
exports.getCurrentElement = getCurrentElement;
exports.invalidCharsRegEx = void 0;
exports.pathIsInSearch = pathIsInSearch;
exports.replaceState = replaceState;
exports.schemaKeys = schemaKeys;
exports.sleep = sleep;

var _languages = require("../languages");

/* For Delayed Event Handler Execution */
function debounce(fn, delay) {
  let timeoutID = null;
  return (...args) => {
    clearTimeout(timeoutID);
    const that = this;
    timeoutID = setTimeout(() => {
      fn.apply(that, args);
    }, delay);
  };
}

const invalidCharsRegEx = new RegExp(/[\s#:?&={}]/, 'g'); // used for generating valid html element ids by replacing the invalid chars with hyphen (-)

exports.invalidCharsRegEx = invalidCharsRegEx;

function sleep(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

function copyToClipboard(data, e) {
  const btnEl = e.currentTarget;
  const textArea = document.createElement('textarea');
  textArea.value = data;
  textArea.style.position = 'fixed'; // avoid scrolling to bottom

  document.body.appendChild(textArea);
  textArea.focus();
  textArea.select();

  try {
    document.execCommand('copy');
    btnEl.innerText = (0, _languages.getI18nText)('operations.copied');
    setTimeout(() => {
      btnEl.innerText = (0, _languages.getI18nText)('operations.copy');
    }, 5000);
  } catch (err) {
    console.error('Unable to copy', err); // eslint-disable-line no-console
  }

  document.body.removeChild(textArea);
}

function getBaseUrlFromUrl(url) {
  const pathArray = url.split('/');
  return `${pathArray[0]}//${pathArray[2]}`;
}

function componentIsInSearch(searchVal, component) {
  return !searchVal || component.name.toLowerCase().includes(searchVal.toLowerCase());
}

function pathIsInSearch(searchVal, path) {
  if (!searchVal) {
    return true;
  }

  const stringToSearch = `${path.method} ${path.path} ${path.summary || path.description || ''} ${path.operationId || ''}`.toLowerCase();
  return stringToSearch.includes(searchVal.toLowerCase());
}

function schemaKeys(schemaProps, result = new Set()) {
  if (!schemaProps) {
    return result;
  }

  Object.keys(schemaProps).forEach(key => {
    result.add(key);

    if (schemaProps[key].properties) {
      schemaKeys(schemaProps[key].properties, result);
    } else if (schemaProps[key].items && schemaProps[key].items.properties) {
      schemaKeys(schemaProps[key].items.properties, result);
    }
  });
  return result;
}

function advancedSearch(searchVal, allSpecTags, searchOptions = []) {
  if (!searchVal.trim() || searchOptions.length === 0) {
    return undefined;
  }

  const pathsMatched = [];
  allSpecTags.forEach(tag => {
    tag.paths.forEach(path => {
      let stringToSearch = '';

      if (searchOptions.includes('search-api-path')) {
        stringToSearch = path.path;
      }

      if (searchOptions.includes('search-api-descr')) {
        stringToSearch = `${stringToSearch} ${path.summary || path.description || ''}`;
      }

      if (searchOptions.includes('search-api-params')) {
        stringToSearch = `${stringToSearch} ${path.parameters && path.parameters.map(v => v.name).join(' ') || ''}`;
      }

      if (searchOptions.includes('search-api-request-body') && path.requestBody) {
        let schemaKeySet = new Set();

        for (const contentType in path.requestBody && path.requestBody.content) {
          if (path.requestBody.content[contentType].schema && path.requestBody.content[contentType].schema.properties) {
            schemaKeySet = schemaKeys(path.requestBody.content[contentType].schema.properties);
          }

          stringToSearch = `${stringToSearch} ${[...schemaKeySet].join(' ')}`;
        }
      }

      if (searchOptions.includes('search-api-resp-descr')) {
        stringToSearch = `${stringToSearch} ${Object.values(path.responses).map(v => v.description || '').join(' ')}`;
      }

      if (stringToSearch.toLowerCase().includes(searchVal.trim().toLowerCase())) {
        pathsMatched.push({
          elementId: path.elementId,
          method: path.method,
          path: path.path,
          summary: path.summary || path.description || '',
          deprecated: path.deprecated
        });
      }
    });
  });
  return pathsMatched;
}

function getCurrentElement() {
  const currentQuery = (window.location.hash || '').split('?')[1];
  const query = new URLSearchParams(currentQuery);
  return decodeURIComponent(query.get('route') || '');
}

function replaceState(rawElementId) {
  const elementId = rawElementId && rawElementId.replace(/^#/, '') || '';
  const currentNavigationHashPart = (window.location.hash || '').split('?')[0].replace(/^#/, '');
  const currentQuery = (window.location.hash || '').split('?')[1];
  const query = new URLSearchParams(currentQuery);
  query.delete('route');
  const newQuery = query.toString().length > 1 ? `${query.toString()}&route=${elementId}` : `route=${elementId}`;
  window.history.replaceState(null, null, `#${currentNavigationHashPart}?${newQuery}`);
}