"use strict";

var _lit = require("lit");
var _unsafeHtml = require("lit/directives/unsafe-html.js");
var _marked = require("marked");
var _prismjs = _interopRequireDefault(require("prismjs"));
require("./json-tree.js");
require("prismjs/components/prism-markup-templating.js");
require("prismjs/components/prism-css.js");
require("prismjs/components/prism-yaml.js");
require("prismjs/components/prism-go.js");
require("prismjs/components/prism-ruby.js");
require("prismjs/components/prism-java.js");
require("prismjs/components/prism-json.js");
require("prismjs/components/prism-bash.js");
require("prismjs/components/prism-python.js");
require("prismjs/components/prism-http.js");
require("prismjs/components/prism-csharp.js");
require("prismjs/components/prism-rust.js");
require("prismjs/components/prism-php.js");
var _prismStyles = _interopRequireDefault(require("../styles/prism-styles.js"));
var _fontStyles = _interopRequireDefault(require("../styles/font-styles.js"));
var _inputStyles = _interopRequireDefault(require("../styles/input-styles.js"));
var _index = require("../languages/index.js");
var _commonUtils = require("../utils/common-utils.js");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/**
 * Mapping mime-type => prism language
 */
const LANGUAGES = [{
  pattern: /json/,
  language: 'json'
}, {
  pattern: /xml/,
  language: 'html'
}];

/**
 * Syntax Highlighter component.
 */
class SyntaxHighlighter extends _lit.LitElement {
  static get properties() {
    return {
      content: {
        type: Object
      },
      language: {
        type: String,
        attribute: 'language'
      },
      mimeType: {
        type: String,
        attribute: 'mime-type'
      }
    };
  }
  static finalizeStyles() {
    return [_prismStyles.default, _fontStyles.default, _inputStyles.default, (0, _lit.css)`:host{font-weight:400}div{position:relative;display:flex;flex-direction:column}.toolbar-copy-btn{position:absolute;top:0;right:0;margin-right:8px}.toolbar-copy-btn+pre{white-space:pre;max-height:400px;overflow:auto;display:flex;padding-right:70px}`];
  }

  /**
   * Returns the prism language to use based on language/content type
   * @returns {string} The language to use
   */
  detectLanguage() {
    if (this.language) {
      var _this$language;
      return (_this$language = this.language) === null || _this$language === void 0 ? void 0 : _this$language.toLowerCase();
    }
    if (this.mimeType) {
      var _this$mimeType, _LANGUAGES$find;
      const lcMimeType = (_this$mimeType = this.mimeType) === null || _this$mimeType === void 0 ? void 0 : _this$mimeType.toLowerCase();
      return (_LANGUAGES$find = LANGUAGES.find(def => def.pattern.test(lcMimeType))) === null || _LANGUAGES$find === void 0 ? void 0 : _LANGUAGES$find.language;
    }
    return null;
  }
  render() {
    return this.renderCopyWrapper(this.renderHighlight());
  }

  /**
   * Render the highlighted content.
   * @returns Highlighter
   */
  renderHighlight() {
    var _this$content;
    const lang = this.detectLanguage();
    const grammar = _prismjs.default.languages[lang];
    if (typeof this.content !== 'string') {
      return (0, _lit.html)`<json-tree .data="${this.content}">`;
    }
    const stringContent = ((_this$content = this.content) === null || _this$content === void 0 ? void 0 : _this$content.toString()) || '';
    const increasedSpaceContent = lang !== 'python' && lang !== 'yaml' && lang !== 'toml' ? stringContent.split('\n').map(line => line.replace(/^\s{2}/g, '    ')).join('\n') : stringContent;
    return grammar ? (0, _lit.html)`<pre><code>${(0, _unsafeHtml.unsafeHTML)(_prismjs.default.highlight(increasedSpaceContent, grammar, lang))}</code></pre>` : (0, _lit.html)`<pre>${increasedSpaceContent}</pre>`;
  }

  /**
   * Render a copy-to-clipboard button.
   * @param {*} content Content
   * @returns Content
   */
  renderCopyWrapper(content) {
    return (0, _lit.html)`<div class="fs-exclude ph-no-capture" data-hj-suppress data-sl="mask" style="min-height:2rem"> <button class="m-btn outline-primary toolbar-copy-btn" @click="${this.copyToClipboard}" part="btn btn-fill btn-copy">${(0, _index.getI18nText)('operations.copy')}</button> ${content} </div>`;
  }

  /**
   * Copy to clipboard.
   * @param {Event} e Event
   */
  copyToClipboard(e) {
    var _this$content2;
    const data = this.detectLanguage() === 'json' && typeof this.content !== 'string' ? JSON.stringify(this.content, null, 2) : (_this$content2 = this.content) === null || _this$content2 === void 0 ? void 0 : _this$content2.toString();
    (0, _commonUtils.copyToClipboard)(data, e);
  }
}

/*
 * Configure marked globally.
 */
_marked.marked.setOptions({
  highlight(code, lang) {
    if (_prismjs.default.languages[lang]) {
      return _prismjs.default.highlight(code, _prismjs.default.languages[lang], lang);
    }
    return code;
  }
});
if (!customElements.get('openapi-explorer')) {
  customElements.define('syntax-highlighter', SyntaxHighlighter);
}