"use strict";
/*
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.App = void 0;
// Represents the test web app that uses the AppAuthJS library.
var authorization_request_1 = require("../authorization_request");
var authorization_request_handler_1 = require("../authorization_request_handler");
var authorization_service_configuration_1 = require("../authorization_service_configuration");
var logger_1 = require("../logger");
var redirect_based_handler_1 = require("../redirect_based_handler");
var token_request_1 = require("../token_request");
var token_request_handler_1 = require("../token_request_handler");
/* an example open id connect provider */
var openIdConnectUrl = 'https://accounts.google.com';
/* example client configuration */
var clientId = '511828570984-7nmej36h9j2tebiqmpqh835naet4vci4.apps.googleusercontent.com';
var redirectUri = 'http://localhost:8000/app/redirect.html';
var scope = 'openid';
/**
 * The Test application.
 */
var App = /** @class */ (function () {
    function App(snackbar) {
        var _this = this;
        this.snackbar = snackbar;
        this.notifier = new authorization_request_handler_1.AuthorizationNotifier();
        this.authorizationHandler = new redirect_based_handler_1.RedirectRequestHandler();
        this.tokenHandler = new token_request_handler_1.BaseTokenRequestHandler();
        // set notifier to deliver responses
        this.authorizationHandler.setAuthorizationNotifier(this.notifier);
        // set a listener to listen for authorization responses
        this.notifier.setAuthorizationListener(function (request, response, error) {
            logger_1.log('Authorization request complete ', request, response, error);
            if (response) {
                _this.request = request;
                _this.response = response;
                _this.code = response.code;
                _this.showMessage("Authorization Code " + response.code);
            }
        });
    }
    App.prototype.showMessage = function (message) {
        var snackbar = this.snackbar['MaterialSnackbar'];
        snackbar.showSnackbar({ message: message });
    };
    App.prototype.fetchServiceConfiguration = function () {
        var _this = this;
        authorization_service_configuration_1.AuthorizationServiceConfiguration.fetchFromIssuer(openIdConnectUrl)
            .then(function (response) {
            logger_1.log('Fetched service configuration', response);
            _this.configuration = response;
            _this.showMessage('Completed fetching configuration');
        })
            .catch(function (error) {
            logger_1.log('Something bad happened', error);
            _this.showMessage("Something bad happened " + error);
        });
    };
    App.prototype.makeAuthorizationRequest = function () {
        // create a request
        var request = new authorization_request_1.AuthorizationRequest({
            client_id: clientId,
            redirect_uri: redirectUri,
            scope: scope,
            response_type: authorization_request_1.AuthorizationRequest.RESPONSE_TYPE_CODE,
            state: undefined,
            extras: { 'prompt': 'consent', 'access_type': 'offline' }
        });
        if (this.configuration) {
            this.authorizationHandler.performAuthorizationRequest(this.configuration, request);
        }
        else {
            this.showMessage('Fetch Authorization Service configuration, before you make the authorization request.');
        }
    };
    App.prototype.makeTokenRequest = function () {
        var _this = this;
        if (!this.configuration) {
            this.showMessage('Please fetch service configuration.');
            return;
        }
        var request = null;
        if (this.code) {
            var extras = undefined;
            if (this.request && this.request.internal) {
                extras = {};
                extras['code_verifier'] = this.request.internal['code_verifier'];
            }
            // use the code to make the token request.
            request = new token_request_1.TokenRequest({
                client_id: clientId,
                redirect_uri: redirectUri,
                grant_type: token_request_1.GRANT_TYPE_AUTHORIZATION_CODE,
                code: this.code,
                refresh_token: undefined,
                extras: extras
            });
        }
        else if (this.tokenResponse) {
            // use the token response to make a request for an access token
            request = new token_request_1.TokenRequest({
                client_id: clientId,
                redirect_uri: redirectUri,
                grant_type: token_request_1.GRANT_TYPE_REFRESH_TOKEN,
                code: undefined,
                refresh_token: this.tokenResponse.refreshToken,
                extras: undefined
            });
        }
        if (request) {
            this.tokenHandler.performTokenRequest(this.configuration, request)
                .then(function (response) {
                var isFirstRequest = false;
                if (_this.tokenResponse) {
                    // copy over new fields
                    _this.tokenResponse.accessToken = response.accessToken;
                    _this.tokenResponse.issuedAt = response.issuedAt;
                    _this.tokenResponse.expiresIn = response.expiresIn;
                    _this.tokenResponse.tokenType = response.tokenType;
                    _this.tokenResponse.scope = response.scope;
                }
                else {
                    isFirstRequest = true;
                    _this.tokenResponse = response;
                }
                // unset code, so we can do refresh token exchanges subsequently
                _this.code = undefined;
                if (isFirstRequest) {
                    _this.showMessage("Obtained a refresh token " + response.refreshToken);
                }
                else {
                    _this.showMessage("Obtained an access token " + response.accessToken + ".");
                }
            })
                .catch(function (error) {
                logger_1.log('Something bad happened', error);
                _this.showMessage("Something bad happened " + error);
            });
        }
    };
    App.prototype.checkForAuthorizationResponse = function () {
        this.authorizationHandler.completeAuthorizationRequestIfPossible();
    };
    return App;
}());
exports.App = App;
// export App
window['App'] = App;
//# sourceMappingURL=data:application/json;base64,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