"use strict";
/*!
 * Copyright 2020, OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const getter_1 = require("../context/propagation/getter");
const NoopHttpTextPropagator_1 = require("../context/propagation/NoopHttpTextPropagator");
const setter_1 = require("../context/propagation/setter");
const context_1 = require("./context");
const global_utils_1 = require("./global-utils");
const contextApi = context_1.ContextAPI.getInstance();
/**
 * Singleton object which represents the entry point to the OpenTelemetry Propagation API
 */
class PropagationAPI {
    /** Empty private constructor prevents end users from constructing a new instance of the API */
    constructor() { }
    /** Get the singleton instance of the Propagator API */
    static getInstance() {
        if (!this._instance) {
            this._instance = new PropagationAPI();
        }
        return this._instance;
    }
    /**
     * Set the current propagator. Returns the initialized propagator
     */
    setGlobalPropagator(propagator) {
        if (global_utils_1._global[global_utils_1.GLOBAL_PROPAGATION_API_KEY]) {
            // global propagator has already been set
            return this._getGlobalPropagator();
        }
        global_utils_1._global[global_utils_1.GLOBAL_PROPAGATION_API_KEY] = global_utils_1.makeGetter(global_utils_1.API_BACKWARDS_COMPATIBILITY_VERSION, propagator, NoopHttpTextPropagator_1.NOOP_HTTP_TEXT_PROPAGATOR);
        return propagator;
    }
    /**
     * Inject context into a carrier to be propagated inter-process
     *
     * @param carrier carrier to inject context into
     * @param setter Function used to set values on the carrier
     * @param context Context carrying tracing data to inject. Defaults to the currently active context.
     */
    inject(carrier, setter = setter_1.defaultSetter, context = contextApi.active()) {
        return this._getGlobalPropagator().inject(context, carrier, setter);
    }
    /**
     * Extract context from a carrier
     *
     * @param carrier Carrier to extract context from
     * @param getter Function used to extract keys from a carrier
     * @param context Context which the newly created context will inherit from. Defaults to the currently active context.
     */
    extract(carrier, getter = getter_1.defaultGetter, context = contextApi.active()) {
        return this._getGlobalPropagator().extract(context, carrier, getter);
    }
    /** Remove the global propagator */
    disable() {
        delete global_utils_1._global[global_utils_1.GLOBAL_PROPAGATION_API_KEY];
    }
    _getGlobalPropagator() {
        var _a, _b, _c;
        return (_c = (_b = (_a = global_utils_1._global)[global_utils_1.GLOBAL_PROPAGATION_API_KEY]) === null || _b === void 0 ? void 0 : _b.call(_a, global_utils_1.API_BACKWARDS_COMPATIBILITY_VERSION), (_c !== null && _c !== void 0 ? _c : NoopHttpTextPropagator_1.NOOP_HTTP_TEXT_PROPAGATOR));
    }
}
exports.PropagationAPI = PropagationAPI;
//# sourceMappingURL=propagation.js.map