/*!
 * Copyright 2019, OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { Meter } from './Meter';
import { MetricOptions, UnboundMetric, Labels, Counter, Measure, Observer } from './Metric';
import { BoundMeasure, BoundCounter } from './BoundInstrument';
import { CorrelationContext } from '../correlation_context/CorrelationContext';
import { SpanContext } from '../trace/span_context';
import { ObserverResult } from './ObserverResult';
/**
 * NoopMeter is a noop implementation of the {@link Meter} interface. It reuses
 * constant NoopMetrics for all of its methods.
 */
export declare class NoopMeter implements Meter {
    constructor();
    /**
     * Returns constant noop measure.
     * @param name the name of the metric.
     * @param [options] the metric options.
     */
    createMeasure(name: string, options?: MetricOptions): Measure;
    /**
     * Returns a constant noop counter.
     * @param name the name of the metric.
     * @param [options] the metric options.
     */
    createCounter(name: string, options?: MetricOptions): Counter;
    /**
     * Returns constant noop observer.
     * @param name the name of the metric.
     * @param [options] the metric options.
     */
    createObserver(name: string, options?: MetricOptions): Observer;
}
export declare class NoopMetric<T> implements UnboundMetric<T> {
    private readonly _instrument;
    constructor(instrument: T);
    /**
     * Returns a Bound Instrument associated with specified Labels.
     * It is recommended to keep a reference to the Bound Instrument instead of
     * always calling this method for every operations.
     * @param labels key-values pairs that are associated with a specific metric
     *     that you want to record.
     */
    bind(labels: Labels): T;
    /**
     * Removes the Binding from the metric, if it is present.
     * @param labels key-values pairs that are associated with a specific metric.
     */
    unbind(labels: Labels): void;
    /**
     * Clears all timeseries from the Metric.
     */
    clear(): void;
}
export declare class NoopCounterMetric extends NoopMetric<BoundCounter> implements Counter {
    add(value: number, labels: Labels): void;
}
export declare class NoopMeasureMetric extends NoopMetric<BoundMeasure> implements Measure {
    record(value: number, labels: Labels, correlationContext?: CorrelationContext, spanContext?: SpanContext): void;
}
export declare class NoopObserverMetric extends NoopMetric<void> implements Observer {
    setCallback(callback: (observerResult: ObserverResult) => void): void;
}
export declare class NoopBoundCounter implements BoundCounter {
    add(value: number): void;
}
export declare class NoopBoundMeasure implements BoundMeasure {
    record(value: number, correlationContext?: CorrelationContext, spanContext?: SpanContext): void;
}
export declare const NOOP_METER: NoopMeter;
export declare const NOOP_BOUND_COUNTER: NoopBoundCounter;
export declare const NOOP_COUNTER_METRIC: NoopCounterMetric;
export declare const NOOP_BOUND_MEASURE: NoopBoundMeasure;
export declare const NOOP_MEASURE_METRIC: NoopMeasureMetric;
export declare const NOOP_OBSERVER_METRIC: NoopObserverMetric;
//# sourceMappingURL=NoopMeter.d.ts.map