"use strict";
/*!
 * Copyright 2019, OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * NoopMeter is a noop implementation of the {@link Meter} interface. It reuses
 * constant NoopMetrics for all of its methods.
 */
class NoopMeter {
    constructor() { }
    /**
     * Returns constant noop measure.
     * @param name the name of the metric.
     * @param [options] the metric options.
     */
    createMeasure(name, options) {
        return exports.NOOP_MEASURE_METRIC;
    }
    /**
     * Returns a constant noop counter.
     * @param name the name of the metric.
     * @param [options] the metric options.
     */
    createCounter(name, options) {
        return exports.NOOP_COUNTER_METRIC;
    }
    /**
     * Returns constant noop observer.
     * @param name the name of the metric.
     * @param [options] the metric options.
     */
    createObserver(name, options) {
        return exports.NOOP_OBSERVER_METRIC;
    }
}
exports.NoopMeter = NoopMeter;
class NoopMetric {
    constructor(instrument) {
        this._instrument = instrument;
    }
    /**
     * Returns a Bound Instrument associated with specified Labels.
     * It is recommended to keep a reference to the Bound Instrument instead of
     * always calling this method for every operations.
     * @param labels key-values pairs that are associated with a specific metric
     *     that you want to record.
     */
    bind(labels) {
        return this._instrument;
    }
    /**
     * Removes the Binding from the metric, if it is present.
     * @param labels key-values pairs that are associated with a specific metric.
     */
    unbind(labels) {
        return;
    }
    /**
     * Clears all timeseries from the Metric.
     */
    clear() {
        return;
    }
}
exports.NoopMetric = NoopMetric;
class NoopCounterMetric extends NoopMetric {
    add(value, labels) {
        this.bind(labels).add(value);
    }
}
exports.NoopCounterMetric = NoopCounterMetric;
class NoopMeasureMetric extends NoopMetric {
    record(value, labels, correlationContext, spanContext) {
        if (typeof correlationContext === 'undefined') {
            this.bind(labels).record(value);
        }
        else if (typeof spanContext === 'undefined') {
            this.bind(labels).record(value, correlationContext);
        }
        else {
            this.bind(labels).record(value, correlationContext, spanContext);
        }
    }
}
exports.NoopMeasureMetric = NoopMeasureMetric;
class NoopObserverMetric extends NoopMetric {
    setCallback(callback) { }
}
exports.NoopObserverMetric = NoopObserverMetric;
class NoopBoundCounter {
    add(value) {
        return;
    }
}
exports.NoopBoundCounter = NoopBoundCounter;
class NoopBoundMeasure {
    record(value, correlationContext, spanContext) {
        return;
    }
}
exports.NoopBoundMeasure = NoopBoundMeasure;
exports.NOOP_METER = new NoopMeter();
exports.NOOP_BOUND_COUNTER = new NoopBoundCounter();
exports.NOOP_COUNTER_METRIC = new NoopCounterMetric(exports.NOOP_BOUND_COUNTER);
exports.NOOP_BOUND_MEASURE = new NoopBoundMeasure();
exports.NOOP_MEASURE_METRIC = new NoopMeasureMetric(exports.NOOP_BOUND_MEASURE);
exports.NOOP_OBSERVER_METRIC = new NoopObserverMetric();
//# sourceMappingURL=NoopMeter.js.map