'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

exports.createPage = createPage;
exports.renderPage = renderPage;

var _PDFJSAnnotate = require('../PDFJSAnnotate');

var _PDFJSAnnotate2 = _interopRequireDefault(_PDFJSAnnotate);

var _renderScreenReaderHints = require('../a11y/renderScreenReaderHints');

var _renderScreenReaderHints2 = _interopRequireDefault(_renderScreenReaderHints);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// Template for creating a new page
var PAGE_TEMPLATE = '\n  <div style="visibility: hidden;" class="page" data-loaded="false">\n    <div class="canvasWrapper">\n      <canvas></canvas>\n    </div>\n    <svg class="annotationLayer"></svg>\n    <div class="textLayer"></div>\n  </div>\n';

/**
 * Create a new page to be appended to the DOM.
 *
 * @param {Number} pageNumber The page number that is being created
 * @return {HTMLElement}
 */
function createPage(pageNumber) {
  var temp = document.createElement('div');
  temp.innerHTML = PAGE_TEMPLATE;

  var page = temp.children[0];
  var canvas = page.querySelector('canvas');

  page.setAttribute('id', 'pageContainer' + pageNumber);
  page.setAttribute('data-page-number', pageNumber);

  canvas.mozOpaque = true;
  canvas.setAttribute('id', 'page' + pageNumber);

  return page;
}

/**
 * Render a page that has already been created.
 *
 * @param {Number} pageNumber The page number to be rendered
 * @param {Object} renderOptions The options for rendering
 * @return {Promise} Settled once rendering has completed
 *  A settled Promise will be either:
 *    - fulfilled: [pdfPage, annotations]
 *    - rejected: Error
 */
function renderPage(pageNumber, renderOptions) {
  var documentId = renderOptions.documentId;
  var pdfDocument = renderOptions.pdfDocument;
  var scale = renderOptions.scale;
  var rotate = renderOptions.rotate;

  // Load the page and annotations

  return Promise.all([pdfDocument.getPage(pageNumber), _PDFJSAnnotate2.default.getAnnotations(documentId, pageNumber)]).then(function (_ref) {
    var _ref2 = _slicedToArray(_ref, 2);

    var pdfPage = _ref2[0];
    var annotations = _ref2[1];

    var page = document.getElementById('pageContainer' + pageNumber);
    var svg = page.querySelector('.annotationLayer');
    var canvas = page.querySelector('.canvasWrapper canvas');
    var canvasContext = canvas.getContext('2d', { alpha: false });
    var viewport = pdfPage.getViewport(scale, rotate);
    var transform = scalePage(pageNumber, viewport, canvasContext);

    // Render the page
    return Promise.all([pdfPage.render({ canvasContext: canvasContext, viewport: viewport, transform: transform }), _PDFJSAnnotate2.default.render(svg, viewport, annotations)]).then(function () {
      // Text content is needed for a11y, but is also necessary for creating
      // highlight and strikeout annotations which require selecting text.
      return pdfPage.getTextContent({ normalizeWhitespace: true }).then(function (textContent) {
        return new Promise(function (resolve, reject) {
          // Render text layer for a11y of text content
          var textLayer = page.querySelector('.textLayer');
          var textLayerFactory = new PDFJS.DefaultTextLayerFactory();
          var textLayerBuilder = textLayerFactory.createTextLayerBuilder(textLayer, pageNumber - 1, viewport);
          textLayerBuilder.setTextContent(textContent);
          textLayerBuilder.render();

          // Enable a11y for annotations
          // Timeout is needed to wait for `textLayerBuilder.render`
          setTimeout(function () {
            try {
              (0, _renderScreenReaderHints2.default)(annotations.annotations);
              resolve();
            } catch (e) {
              reject(e);
            }
          });
        });
      });
    }).then(function () {
      // Indicate that the page was loaded
      page.setAttribute('data-loaded', 'true');

      return [pdfPage, annotations];
    });
  });
}

/**
 * Scale the elements of a page.
 *
 * @param {Number} pageNumber The page number to be scaled
 * @param {Object} viewport The viewport of the PDF page (see pdfPage.getViewport(scale, rotate))
 * @param {Object} context The canvas context that the PDF page is rendered to
 * @return {Array} The transform data for rendering the PDF page
 */
function scalePage(pageNumber, viewport, context) {
  var page = document.getElementById('pageContainer' + pageNumber);
  var canvas = page.querySelector('.canvasWrapper canvas');
  var svg = page.querySelector('.annotationLayer');
  var wrapper = page.querySelector('.canvasWrapper');
  var textLayer = page.querySelector('.textLayer');
  var outputScale = getOutputScale(context);
  var transform = !outputScale.scaled ? null : [outputScale.sx, 0, 0, outputScale.sy, 0, 0];
  var sfx = approximateFraction(outputScale.sx);
  var sfy = approximateFraction(outputScale.sy);

  // Adjust width/height for scale
  page.style.visibility = '';
  canvas.width = roundToDivide(viewport.width * outputScale.sx, sfx[0]);
  canvas.height = roundToDivide(viewport.height * outputScale.sy, sfy[0]);
  canvas.style.width = roundToDivide(viewport.width, sfx[1]) + 'px';
  canvas.style.height = roundToDivide(viewport.height, sfx[1]) + 'px';
  svg.setAttribute('width', viewport.width);
  svg.setAttribute('height', viewport.height);
  svg.style.width = viewport.width + 'px';
  svg.style.height = viewport.height + 'px';
  page.style.width = viewport.width + 'px';
  page.style.height = viewport.height + 'px';
  wrapper.style.width = viewport.width + 'px';
  wrapper.style.height = viewport.height + 'px';
  textLayer.style.width = viewport.width + 'px';
  textLayer.style.height = viewport.height + 'px';

  return transform;
}

/**
 * Approximates a float number as a fraction using Farey sequence (max order of 8).
 *
 * @param {Number} x Positive float number
 * @return {Array} Estimated fraction: the first array item is a numerator,
 *                 the second one is a denominator.
 */
function approximateFraction(x) {
  // Fast path for int numbers or their inversions.
  if (Math.floor(x) === x) {
    return [x, 1];
  }

  var xinv = 1 / x;
  var limit = 8;
  if (xinv > limit) {
    return [1, limit];
  } else if (Math.floor(xinv) === xinv) {
    return [1, xinv];
  }

  var x_ = x > 1 ? xinv : x;

  // a/b and c/d are neighbours in Farey sequence.
  var a = 0,
      b = 1,
      c = 1,
      d = 1;

  // Limit search to order 8.
  while (true) {
    // Generating next term in sequence (order of q).
    var p = a + c,
        q = b + d;
    if (q > limit) {
      break;
    }
    if (x_ <= p / q) {
      c = p;d = q;
    } else {
      a = p;b = q;
    }
  }

  // Select closest of neighbours to x.
  if (x_ - a / b < c / d - x_) {
    return x_ === x ? [a, b] : [b, a];
  } else {
    return x_ === x ? [c, d] : [d, c];
  }
}

function getOutputScale(ctx) {
  var devicePixelRatio = window.devicePixelRatio || 1;
  var backingStoreRatio = ctx.webkitBackingStorePixelRatio || ctx.mozBackingStorePixelRatio || ctx.msBackingStorePixelRatio || ctx.oBackingStorePixelRatio || ctx.backingStorePixelRatio || 1;
  var pixelRatio = devicePixelRatio / backingStoreRatio;
  return {
    sx: pixelRatio,
    sy: pixelRatio,
    scaled: pixelRatio !== 1
  };
}

function roundToDivide(x, div) {
  var r = x % div;
  return r === 0 ? x : Math.round(x - r + div);
}