'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setPen = setPen;
exports.enablePen = enablePen;
exports.disablePen = disablePen;

var _PDFJSAnnotate = require('../PDFJSAnnotate');

var _PDFJSAnnotate2 = _interopRequireDefault(_PDFJSAnnotate);

var _appendChild = require('../render/appendChild');

var _appendChild2 = _interopRequireDefault(_appendChild);

var _utils = require('./utils');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var _enabled = false;
var _penSize = void 0;
var _penColor = void 0;
var path = void 0;
var lines = void 0;

/**
 * Handle document.mousedown event
 */
function handleDocumentMousedown() {
  path = null;
  lines = [];

  document.addEventListener('mousemove', handleDocumentMousemove);
  document.addEventListener('mouseup', handleDocumentMouseup);
}

/**
 * Handle document.mouseup event
 *
 * @param {Event} e The DOM event to be handled
 */
function handleDocumentMouseup(e) {
  var svg = void 0;
  if (lines.length > 1 && (svg = (0, _utils.findSVGAtPoint)(e.clientX, e.clientY))) {
    var _getMetadata = (0, _utils.getMetadata)(svg);

    var documentId = _getMetadata.documentId;
    var pageNumber = _getMetadata.pageNumber;


    _PDFJSAnnotate2.default.getStoreAdapter().addAnnotation(documentId, pageNumber, {
      type: 'drawing',
      width: _penSize,
      color: _penColor,
      lines: lines
    }).then(function (annotation) {
      if (path) {
        svg.removeChild(path);
      }

      (0, _appendChild2.default)(svg, annotation);
    });
  }

  document.removeEventListener('mousemove', handleDocumentMousemove);
  document.removeEventListener('mouseup', handleDocumentMouseup);
}

/**
 * Handle document.mousemove event
 *
 * @param {Event} e The DOM event to be handled
 */
function handleDocumentMousemove(e) {
  savePoint(e.clientX, e.clientY);
}

/**
 * Handle document.keyup event
 *
 * @param {Event} e The DOM event to be handled
 */
function handleDocumentKeyup(e) {
  // Cancel rect if Esc is pressed
  if (e.keyCode === 27) {
    lines = null;
    path.parentNode.removeChild(path);
    document.removeEventListener('mousemove', handleDocumentMousemove);
    document.removeEventListener('mouseup', handleDocumentMouseup);
  }
}

/**
 * Save a point to the line being drawn.
 *
 * @param {Number} x The x coordinate of the point
 * @param {Number} y The y coordinate of the point
 */
function savePoint(x, y) {
  var svg = (0, _utils.findSVGAtPoint)(x, y);
  if (!svg) {
    return;
  }

  var rect = svg.getBoundingClientRect();
  var point = (0, _utils.scaleDown)(svg, {
    x: x - rect.left,
    y: y - rect.top
  });

  lines.push([point.x, point.y]);

  if (lines.length <= 1) {
    return;
  }

  if (path) {
    svg.removeChild(path);
  }

  path = (0, _appendChild2.default)(svg, {
    type: 'drawing',
    color: _penColor,
    width: _penSize,
    lines: lines
  });
}

/**
 * Set the attributes of the pen.
 *
 * @param {Number} penSize The size of the lines drawn by the pen
 * @param {String} penColor The color of the lines drawn by the pen
 */
function setPen() {
  var penSize = arguments.length <= 0 || arguments[0] === undefined ? 1 : arguments[0];
  var penColor = arguments.length <= 1 || arguments[1] === undefined ? '000000' : arguments[1];

  _penSize = parseInt(penSize, 10);
  _penColor = penColor;
}

/**
 * Enable the pen behavior
 */
function enablePen() {
  if (_enabled) {
    return;
  }

  _enabled = true;
  document.addEventListener('mousedown', handleDocumentMousedown);
  document.addEventListener('keyup', handleDocumentKeyup);
  (0, _utils.disableUserSelect)();
}

/**
 * Disable the pen behavior
 */
function disablePen() {
  if (!_enabled) {
    return;
  }

  _enabled = false;
  document.removeEventListener('mousedown', handleDocumentMousedown);
  document.removeEventListener('keyup', handleDocumentKeyup);
  (0, _utils.enableUserSelect)();
}