'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = appendChild;

var _objectAssign = require('object-assign');

var _objectAssign2 = _interopRequireDefault(_objectAssign);

var _renderLine = require('./renderLine');

var _renderLine2 = _interopRequireDefault(_renderLine);

var _renderPath = require('./renderPath');

var _renderPath2 = _interopRequireDefault(_renderPath);

var _renderPoint = require('./renderPoint');

var _renderPoint2 = _interopRequireDefault(_renderPoint);

var _renderRect = require('./renderRect');

var _renderRect2 = _interopRequireDefault(_renderRect);

var _renderText = require('./renderText');

var _renderText2 = _interopRequireDefault(_renderText);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var isFirefox = /firefox/i.test(navigator.userAgent);

/**
 * Get the x/y translation to be used for transforming the annotations
 * based on the rotation of the viewport.
 *
 * @param {Object} viewport The viewport data from the page
 * @return {Object}
 */
function getTranslation(viewport) {
  var x = void 0;
  var y = void 0;

  // Modulus 360 on the rotation so that we only
  // have to worry about four possible values.
  switch (viewport.rotation % 360) {
    case 0:
      x = y = 0;
      break;
    case 90:
      x = 0;
      y = viewport.width / viewport.scale * -1;
      break;
    case 180:
      x = viewport.width / viewport.scale * -1;
      y = viewport.height / viewport.scale * -1;
      break;
    case 270:
      x = viewport.height / viewport.scale * -1;
      y = 0;
      break;
  }

  return { x: x, y: y };
}

/**
 * Transform the rotation and scale of a node using SVG's native transform attribute.
 *
 * @param {Node} node The node to be transformed
 * @param {Object} viewport The page's viewport data
 * @return {Node}
 */
function transform(node, viewport) {
  var trans = getTranslation(viewport);

  // Let SVG natively transform the element
  node.setAttribute('transform', 'scale(' + viewport.scale + ') rotate(' + viewport.rotation + ') translate(' + trans.x + ', ' + trans.y + ')');

  // Manually adjust x/y for nested SVG nodes
  if (!isFirefox && node.nodeName.toLowerCase() === 'svg') {
    node.setAttribute('x', parseInt(node.getAttribute('x'), 10) * viewport.scale);
    node.setAttribute('y', parseInt(node.getAttribute('y'), 10) * viewport.scale);

    var x = parseInt(node.getAttribute('x', 10));
    var y = parseInt(node.getAttribute('y', 10));
    var width = parseInt(node.getAttribute('width'), 10);
    var height = parseInt(node.getAttribute('height'), 10);
    var path = node.querySelector('path');
    var svg = path.parentNode;

    // Scale width/height
    [node, svg, path, node.querySelector('rect')].forEach(function (n) {
      n.setAttribute('width', parseInt(n.getAttribute('width'), 10) * viewport.scale);
      n.setAttribute('height', parseInt(n.getAttribute('height'), 10) * viewport.scale);
    });

    // Transform path but keep scale at 100% since it will be handled natively
    transform(path, (0, _objectAssign2.default)({}, viewport, { scale: 1 }));

    switch (viewport.rotation % 360) {
      case 90:
        node.setAttribute('x', viewport.width - y - width);
        node.setAttribute('y', x);
        svg.setAttribute('x', 1);
        svg.setAttribute('y', 0);
        break;
      case 180:
        node.setAttribute('x', viewport.width - x - width);
        node.setAttribute('y', viewport.height - y - height);
        svg.setAttribute('y', 2);
        break;
      case 270:
        node.setAttribute('x', y);
        node.setAttribute('y', viewport.height - x - height);
        svg.setAttribute('x', -1);
        svg.setAttribute('y', 0);
        break;
    }
  }

  return node;
}

/**
 * Append an annotation as a child of an SVG.
 *
 * @param {SVGElement} svg The SVG element to append the annotation to
 * @param {Object} annotation The annotation definition to render and append
 * @param {Object} viewport The page's viewport data
 * @return {SVGElement} A node that was created and appended by this function
 */
function appendChild(svg, annotation, viewport) {
  if (!viewport) {
    viewport = JSON.parse(svg.getAttribute('data-pdf-annotate-viewport'));
  }

  var child = void 0;
  switch (annotation.type) {
    case 'area':
    case 'highlight':
      child = (0, _renderRect2.default)(annotation);
      break;
    case 'strikeout':
      child = (0, _renderLine2.default)(annotation);
      break;
    case 'point':
      child = (0, _renderPoint2.default)(annotation);
      break;
    case 'textbox':
      child = (0, _renderText2.default)(annotation);
      break;
    case 'drawing':
      child = (0, _renderPath2.default)(annotation);
      break;
  }

  // If no type was provided for an annotation it will result in node being null.
  // Skip appending/transforming if node doesn't exist.
  if (child) {
    // Set attributes
    child.setAttribute('data-pdf-annotate-id', annotation.uuid);
    child.setAttribute('data-pdf-annotate-type', annotation.type);
    child.setAttribute('aria-hidden', true);

    svg.appendChild(transform(child, viewport));
  }

  return child;
}