import initial from 'lodash/initial';
import last from 'lodash/last';
import { PDFCatalog, } from '../pdf-structures';
import { error } from '../../utils';
import { isInstance, validate } from '../../utils/validate';
import PDFObjectIndex from '../pdf-document/PDFObjectIndex';
import parseDocument from './parseDocument';
var PDFParser = /** @class */ (function () {
    function PDFParser() {
        var _this = this;
        this.activelyParsing = false;
        this.maxObjectNumber = -1;
        this.arrays = [];
        this.dictionaries = [];
        this.body = new Map();
        this.updates = [];
        this.parse = function (bytes, index) {
            validate(index, isInstance(PDFObjectIndex), '"index" must be an instance of PDFObjectIndex');
            if (_this.activelyParsing)
                error('Cannot parse documents concurrently');
            _this.activelyParsing = true;
            _this.maxObjectNumber = -1;
            _this.arrays = [];
            _this.dictionaries = [];
            _this.catalog = undefined;
            _this.header = undefined;
            _this.body = new Map();
            _this.xRefTable = undefined;
            _this.trailer = undefined;
            _this.linearization = undefined;
            _this.updates = [];
            try {
                parseDocument(bytes, index, _this.parseHandlers);
                _this.activelyParsing = false;
            }
            catch (e) {
                _this.activelyParsing = false;
                throw e;
            }
            return {
                maxObjectNumber: _this.maxObjectNumber,
                catalog: _this.catalog,
                arrays: _this.arrays,
                dictionaries: _this.dictionaries,
                original: {
                    header: _this.header,
                    linearization: _this.linearization,
                    body: _this.body,
                    xRefTable: _this.xRefTable,
                    trailer: _this.trailer,
                },
                // Drop the last element, because it will always be empty:
                updates: initial(_this.updates),
            };
        };
        this.updateMaxObjectNumber = function (ref) {
            if (ref.objectNumber > _this.maxObjectNumber) {
                _this.maxObjectNumber = ref.objectNumber;
            }
        };
        this.handleArray = function (array) {
            _this.arrays.push(array);
        };
        this.handleDict = function (dict) {
            _this.dictionaries.push(dict);
            if (dict instanceof PDFCatalog)
                _this.catalog = dict;
        };
        this.handleObjectStream = function (_a) {
            var objects = _a.objects;
            objects.forEach(function (indirectObj) {
                if (_this.updates.length > 0) {
                    last(_this.updates).body.set(indirectObj.getReference(), indirectObj);
                }
                else {
                    _this.body.set(indirectObj.getReference(), indirectObj);
                }
                _this.updateMaxObjectNumber(indirectObj.reference);
            });
        };
        this.handleIndirectObj = function (indirectObj) {
            if (_this.updates.length > 0) {
                last(_this.updates).body.set(indirectObj.getReference(), indirectObj);
            }
            else {
                _this.body.set(indirectObj.getReference(), indirectObj);
            }
            _this.updateMaxObjectNumber(indirectObj.reference);
        };
        this.handleHeader = function (header) {
            _this.header = header;
        };
        this.handleXRefTable = function (xRefTable) {
            if (!_this.trailer)
                _this.xRefTable = xRefTable;
            else
                last(_this.updates).xRefTable = xRefTable;
        };
        this.handleTrailer = function (trailer) {
            if (!_this.trailer)
                _this.trailer = trailer;
            else
                last(_this.updates).trailer = trailer;
            _this.updates.push({
                body: new Map(),
                xRefTable: undefined,
                trailer: undefined,
            });
        };
        this.handleLinearization = function (linearization) {
            _this.linearization = linearization;
        };
        this.parseHandlers = {
            onParseArray: this.handleArray,
            onParseDict: this.handleDict,
            onParseObjectStream: this.handleObjectStream,
            onParseIndirectObj: this.handleIndirectObj,
            onParseHeader: this.handleHeader,
            onParseXRefTable: this.handleXRefTable,
            onParseTrailer: this.handleTrailer,
            onParseLinearization: this.handleLinearization,
        };
    }
    return PDFParser;
}());
export default PDFParser;
