import { PDFArray } from '../pdf-objects';
import { arrayCharAt, error, trimArrayAndRemoveComments } from '../../utils';
import { isIdentity, validate } from '../../utils/validate';
import parseBool from './parseBool';
import parseDict from './parseDict';
import parseHexString from './parseHexString';
import parseIndirectRef from './parseIndirectRef';
import parseName from './parseName';
import parseNull from './parseNull';
import parseNumber from './parseNumber';
import parseString from './parseString';
/**
 * Accepts an array of bytes as input. Checks to see if the first characters in the
 * trimmed input make up a PDF Array.
 *
 * If so, returns a tuple containing (1) an object representing the parsed PDFArray
 * and (2) a subarray of the input with the characters making up the parsed array
 * removed. The "onParseArray" parse handler will also be called with the PDFArray
 * object.
 *
 * If not, null is returned.
 *
 * Note that the elements of the PDF Array are recursively parsed, so the
 * appropriate parse handlers will be called when each element of the array is
 * parsed. The returned PDFArray's elements will be composed of PDFObjects.
 */
var parseArray = function (input, index, parseHandlers) {
    if (parseHandlers === void 0) { parseHandlers = {}; }
    // Make sure it is possible for this to be an array.
    var trimmed = trimArrayAndRemoveComments(input);
    if (arrayCharAt(trimmed, 0) !== '[')
        return undefined;
    var pdfArray = PDFArray.fromArray([], index);
    // Recursively parse each element of the array
    var remainder = trimmed.subarray(1); // Remove the '['
    while (arrayCharAt(trimArrayAndRemoveComments(remainder), 0) !== ']') {
        // Parse the value for this element
        var _a = parseName(remainder, parseHandlers) ||
            parseDict(remainder, index, parseHandlers) ||
            parseArray(remainder, index, parseHandlers) ||
            parseString(remainder, parseHandlers) ||
            parseHexString(remainder, parseHandlers) ||
            parseBool(remainder, parseHandlers) ||
            parseNull(remainder, parseHandlers) ||
            parseIndirectRef(remainder, parseHandlers) ||
            parseNumber(remainder, parseHandlers) ||
            error('Failed to parse array element'), pdfObject = _a[0], r = _a[1];
        pdfArray.push(pdfObject);
        remainder = r;
    }
    var remainderTrim = trimArrayAndRemoveComments(remainder);
    // Make sure the brackets are paired
    validate(arrayCharAt(remainderTrim, 0), isIdentity(']'), 'Mismatched brackets!');
    remainder = trimArrayAndRemoveComments(remainderTrim.subarray(1)); // Remove the ']'
    if (parseHandlers.onParseArray)
        parseHandlers.onParseArray(pdfArray);
    return [pdfArray, remainder];
};
export default parseArray;
