// tslint:disable-next-line:no-unused-variable
import { PDFIndirectObject } from '../pdf-objects';
import { arrayIndexOf, arrayToString, error, trimArrayAndRemoveComments, } from '../../utils';
import parseArray from './parseArray';
import parseBool from './parseBool';
import parseDictOrStream from './parseDictOrStream';
import parseHexString from './parseHexString';
import parseIndirectRef from './parseIndirectRef';
import parseName from './parseName';
import parseNull from './parseNull';
import parseNumber from './parseNumber';
import parseString from './parseString';
/**
 * Accepts an array of bytes as input. Checks to see if the first characters in the
 * trimmed input make up a PDF Indirect Object.
 *
 * If so, returns a tuple containing (1) an object representing the parsed PDF
 * Indirect Object and (2) a subarray of the input with the characters making up
 * the parsed indirect object removed. The "onParseIndirectObj" parse handler will
 * also be called with the PDFIndirectObject.
 *
 * If not, null is returned.
 */
var parseIndirectObj = function (input, index, parseHandlers) {
    if (parseHandlers === void 0) { parseHandlers = {}; }
    var trimmed = trimArrayAndRemoveComments(input);
    var indirectObjRegex = /^(\d+)[\0\t\n\f\r ]*(\d+)[\0\t\n\f\r ]*obj/;
    // Check that initial characters make up an indirect object "header"
    var objIdx = arrayIndexOf(trimmed, 'obj');
    var result = arrayToString(trimmed.subarray(0, objIdx + 3)).match(indirectObjRegex);
    if (!result)
        return undefined;
    var _fullMatch = result[0], objNum = result[1], genNum = result[2];
    // Remove the indirect object "header" from the trimmed input
    var afterHeader = trimArrayAndRemoveComments(trimmed.subarray(objIdx + 3));
    // Try to parse the object contents bytes
    var _a = parseDictOrStream(afterHeader, index, parseHandlers) ||
        parseArray(afterHeader, index, parseHandlers) ||
        parseName(afterHeader, parseHandlers) ||
        parseString(afterHeader, parseHandlers) ||
        parseHexString(afterHeader, parseHandlers) ||
        parseBool(afterHeader, parseHandlers) ||
        parseNull(afterHeader, parseHandlers) ||
        parseIndirectRef(afterHeader, parseHandlers) ||
        parseNumber(afterHeader, parseHandlers) ||
        error('Failed to parse object contents'), contentObj = _a[0], r = _a[1];
    var trimmedRemaining = trimArrayAndRemoveComments(r);
    // Make sure we're now at the end of the object
    var endobjIdx = arrayIndexOf(trimmedRemaining, 'endobj');
    if (endobjIdx === undefined)
        throw new Error('Invalid Indirect Object');
    if (endobjIdx !== 0)
        throw new Error('Incorrectly parsed object contents');
    var indirectObj = PDFIndirectObject.of(contentObj).setReferenceNumbers(Number(objNum), Number(genNum));
    if (parseHandlers.onParseIndirectObj) {
        parseHandlers.onParseIndirectObj(indirectObj);
    }
    return [indirectObj, trimmedRemaining.subarray(6)];
};
export default parseIndirectObj;
