import { PDFLinearizationParams, } from '../pdf-structures';
import { error, trimArray } from '../../utils';
import parseIndirectObj from './parseIndirectObj';
import { parseTrailer, parseTrailerWithoutDict } from './parseTrailer';
import parseXRefTable from './parseXRefTable';
/**
 * Accepts an array of bytes as input. Checks to see if the first characters in the
 * trimmed input make up a PDF Linearization Param Dict, followed by an xref table
 * or stream, and finally a trailer.
 *
 * If so, returns a tuple containing (1) an object representing those three objects
 * and (2) a subarray of the input with the characters making up the parsed objects
 * removed. The "onParseDict" parse handler will be called with the linearization
 * param dict. The "onParseLinearization" parse handler will also be called with
 * objects representing the three parsed linearization objects.
 *
 * If not, null is returned.
 */
var parseLinearization = function (input, index, parseHandlers) {
    if (parseHandlers === void 0) { parseHandlers = {}; }
    var trimmed = trimArray(input);
    // Try to parse a dictionary from the input
    var paramDictMatch = parseIndirectObj(trimmed, index, parseHandlers);
    if (!paramDictMatch)
        return undefined;
    // Make sure it is a Linearization Param Dictionary
    var _a = paramDictMatch, paramDict = _a[0], remaining1 = _a[1];
    if (!(paramDict.pdfObject instanceof PDFLinearizationParams)) {
        return undefined;
    }
    // TODO: Do the parseHandlers really need to be passed to parseIndirectObj?
    // Next we should find a cross reference stream or xref table
    var xrefMatch = parseXRefTable(remaining1) ||
        parseIndirectObj(remaining1, index, parseHandlers) ||
        error('Found Linearization param dict but no first page xref table or stream.');
    var _b = xrefMatch, xref = _b[0], remaining2 = _b[1];
    var trailerMatch = parseTrailer(remaining2, index) ||
        parseTrailerWithoutDict(remaining2, index);
    // Per the PDF spec, a trailer should always be present - but some PDFs in the
    // wild are missing them anyways
    if (!trailerMatch) {
        console.warn('Found Linearization param dict and cross reference index, but no associated trailer.');
    }
    var _c = trailerMatch || [undefined, undefined], trailer = _c[0], remaining3 = _c[1];
    var linearization = { paramDict: paramDict, xref: xref, trailer: trailer };
    if (parseHandlers.onParseLinearization) {
        parseHandlers.onParseLinearization(linearization);
    }
    return [linearization, remaining3 || remaining2];
};
export default parseLinearization;
