import get from 'lodash/get';
import isEmpty from 'lodash/isEmpty';
import { appendBezierCurve, closePath, degreesToRadians, fill, fillAndStroke, fillingRgbColor, lineWidth, moveTo, popGraphicsState, pushGraphicsState, rotateRadians, scale, skewRadians, stroke, strokingRgbColor, translate, } from '../../pdf-operators/simple';
// =============================================================================
// Based on: http://spencermortensen.com/articles/bezier-circle/
//   P_0 = (0,1),  P_1 = (c,1),   P_2 = (1,c),   P_3 = (1,0)
//   P_0 = (1,0),  P_1 = (1,-c),  P_2 = (c,-1),  P_3 = (0,-1)
//   P_0 = (0,-1), P_1 = (-c,-1), P_3 = (-1,-c), P_4 = (-1,0)
//   P_0 = (-1,0), P_1 = (-1,c),  P_2 = (-c,1),  P_3 = (0,1)
//     with c = 0.551915024494
/** @hidden */
var C_VAL = 0.551915024494;
/** @hidden */
var drawEllipsePath = function (_a) {
    var _b = _a.x, x = _b === void 0 ? 0 : _b, _c = _a.y, y = _c === void 0 ? 0 : _c, _d = _a.xScale, xScale = _d === void 0 ? 100 : _d, _e = _a.yScale, yScale = _e === void 0 ? 100 : _e, _f = _a.rotationAngle, rotationAngle = _f === void 0 ? 0 : _f, _g = _a.skewAngles, skewAngles = _g === void 0 ? { xAxis: 0, yAxis: 0 } : _g;
    return [
        pushGraphicsState(),
        translate(x, y),
        rotateRadians(rotationAngle),
        scale(xScale, yScale),
        skewRadians(skewAngles.xAxis, skewAngles.yAxis),
        moveTo(0, 1),
        appendBezierCurve(C_VAL, 1, 1, C_VAL, 1, 0),
        appendBezierCurve(1, -C_VAL, C_VAL, -1, 0, -1),
        appendBezierCurve(-C_VAL, -1, -1, -C_VAL, -1, 0),
        appendBezierCurve(-1, C_VAL, -C_VAL, 1, 0, 1),
        popGraphicsState(),
    ];
};
/**
 * Draws an ellipse in a content stream.
 *
 * ```javascript
 * const contentStream = pdfDoc.register(
 *   pdfDoc.createContentStream(
 *     drawEllipse({
 *       x: 25,
 *       y: 50,
 *       xScale: 50,
 *       yScale: 150,
 *       rotateDegrees: 45,
 *       skewDegrees: { xAxis: 30, yAxis: 30 },
 *       borderWidth: 25,
 *       colorRgb: [0.25, 1.0, 0.79],
 *       borderColorRgb: [0.79, 0.25, 1.0],
 *     }),
 *   ),
 * );
 * const page = pdfDoc
 *   .createPage([250, 500])
 *   .addContentStreams(contentStream);
 * ```
 *
 * @param options An options object with named parameters.
 */
export var drawEllipse = function (options) { return [
    pushGraphicsState(),
    fillingRgbColor(get(options, 'colorRgb[0]', 0), get(options, 'colorRgb[1]', 0), get(options, 'colorRgb[2]', 0)),
    strokingRgbColor(get(options, 'borderColorRgb[0]', 0), get(options, 'borderColorRgb[1]', 0), get(options, 'borderColorRgb[2]', 0)),
    lineWidth(options.borderWidth || 15)
].concat(drawEllipsePath({
    x: options.x || 0,
    y: options.y || 0,
    xScale: options.xScale || 100,
    yScale: options.yScale || 100,
    rotationAngle: options.rotateDegrees
        ? degreesToRadians(options.rotateDegrees)
        : options.rotateRadians,
    skewAngles: options.skewDegrees
        ? {
            xAxis: degreesToRadians(options.skewDegrees.xAxis),
            yAxis: degreesToRadians(options.skewDegrees.yAxis),
        }
        : options.skewRadians,
}), [
    // prettier-ignore
    !isEmpty(options.colorRgb) && !isEmpty(options.borderColorRgb) ? fillAndStroke()
        : !isEmpty(options.colorRgb) ? fill()
            : !isEmpty(options.borderColorRgb) ? stroke()
                : closePath(),
    popGraphicsState(),
]); };
/**
 * Draws a circle in a content stream.
 *
 * ```javascript
 * const contentStream = pdfDoc.register(
 *   pdfDoc.createContentStream(
 *     drawCircle({
 *       x: 25,
 *       y: 50,
 *       size: 50,
 *       rotateDegrees: 45,
 *       skewDegrees: { xAxis: 30, yAxis: 30 },
 *       borderWidth: 25,
 *       colorRgb: [0.25, 1.0, 0.79],
 *       borderColorRgb: [0.79, 0.25, 1.0],
 *     }),
 *   ),
 * );
 * const page = pdfDoc
 *   .createPage([250, 500])
 *   .addContentStreams(contentStream);
 * ```
 *
 * @param options An options object with named parameters.
 */
export var drawCircle = function (options) {
    return drawEllipse({
        x: options.x || 0,
        y: options.y || 0,
        xScale: options.size || 100,
        yScale: options.size || 100,
        borderWidth: options.borderWidth || 15,
        colorRgb: options.colorRgb || [],
        borderColorRgb: options.borderColorRgb || [],
        rotateDegrees: options.rotateDegrees,
        rotateRadians: options.rotateRadians,
        skewDegrees: options.skewDegrees,
        skewRadians: options.skewRadians,
    });
};
