import get from 'lodash/get';
import isEmpty from 'lodash/isEmpty';
import { closePath, fill, fillAndStroke, fillingRgbColor, lineTo, lineWidth, moveTo, popGraphicsState, pushGraphicsState, rotateDegrees, rotateRadians, skewDegrees, skewRadians, stroke, strokingRgbColor, translate, } from '../../pdf-operators/simple';
/**
 * Draws a rectangle in a content stream.
 *
 * ```javascript
 * const contentStream = pdfDoc.register(
 *   pdfDoc.createContentStream(
 *     drawRectangle({
 *       x: 25,
 *       y: 50,
 *       width: 1000,
 *       height: 500,
 *       rotateDegrees: 45,
 *       skewDegrees: { xAxis: 30, yAxis: 30 },
 *       borderWidth: 25,
 *       colorRgb: [0.25, 1.0, 0.79],
 *       borderColorRgb: [0.79, 0.25, 1.0],
 *     }),
 *   ),
 * );
 * const page = pdfDoc
 *   .createPage([250, 500])
 *   .addContentStreams(contentStream);
 * ```
 *
 * @param options An options object with named parameters.
 */
export var drawRectangle = function (options) {
    return [
        pushGraphicsState(),
        fillingRgbColor(get(options, 'colorRgb[0]', 0), get(options, 'colorRgb[1]', 0), get(options, 'colorRgb[2]', 0)),
        strokingRgbColor(get(options, 'borderColorRgb[0]', 0), get(options, 'borderColorRgb[1]', 0), get(options, 'borderColorRgb[2]', 0)),
        lineWidth(options.borderWidth || 15),
        translate(options.x || 0, options.y || 0),
        options.rotateDegrees && rotateDegrees(options.rotateDegrees),
        options.rotateRadians && rotateRadians(options.rotateRadians),
        options.skewDegrees &&
            skewDegrees(options.skewDegrees.xAxis, options.skewDegrees.yAxis),
        options.skewRadians &&
            skewRadians(options.skewRadians.xAxis, options.skewRadians.yAxis),
        moveTo(0, 0),
        lineTo(0, options.height || 100),
        lineTo(options.width || 150, options.height || 100),
        lineTo(options.width || 150, 0),
        closePath(),
        // prettier-ignore
        !isEmpty(options.colorRgb) && !isEmpty(options.borderColorRgb) ? fillAndStroke()
            : !isEmpty(options.colorRgb) ? fill()
                : !isEmpty(options.borderColorRgb) ? stroke()
                    : closePath(),
        popGraphicsState(),
    ].filter(Boolean);
};
/**
 * Draws a square in a content stream.
 *
 * ```javascript
 * const contentStream = pdfDoc.register(
 *   pdfDoc.createContentStream(
 *     drawSquare({
 *       x: 25,
 *       y: 50,
 *       size: 500,
 *       rotateDegrees: 45,
 *       skewDegrees: { xAxis: 30, yAxis: 30 },
 *       borderWidth: 25,
 *       colorRgb: [0.25, 1.0, 0.79],
 *       borderColorRgb: [0.79, 0.25, 1.0],
 *     }),
 *   ),
 * );
 * const page = pdfDoc
 *   .createPage([250, 500])
 *   .addContentStreams(contentStream);
 * ```
 *
 * @param options An options object with named parameters.
 */
export var drawSquare = function (options) {
    return drawRectangle({
        x: options.x || 0,
        y: options.y || 0,
        width: options.size || 100,
        height: options.size || 100,
        rotateDegrees: options.rotateDegrees,
        rotateRadians: options.rotateRadians,
        skewDegrees: options.skewDegrees,
        skewRadians: options.skewRadians,
        borderWidth: options.borderWidth || 15,
        colorRgb: options.colorRgb || [],
        borderColorRgb: options.borderColorRgb || [],
    });
};
