import { IFontNames as IStandardFontNames } from '@pdf-lib/standard-fonts';
import PDFObjectIndex from '../pdf-document/PDFObjectIndex';
import { PDFDictionary, PDFIndirectReference, PDFObject, PDFRawStream } from '../pdf-objects';
import PDFOperator from '../pdf-operators/PDFOperator';
import { PDFCatalog, PDFContentStream, PDFHeader, PDFPage } from '../pdf-structures';
import JPEGXObjectFactory from '../pdf-structures/factories/JPEGXObjectFactory';
import PDFEmbeddedFontFactory from '../pdf-structures/factories/PDFEmbeddedFontFactory';
import PDFFontFactory, { IFontFlagOptions } from '../pdf-structures/factories/PDFFontFactory';
import PDFStandardFontFactory from '../pdf-structures/factories/PDFStandardFontFactory';
import PNGXObjectFactory from '../pdf-structures/factories/PNGXObjectFactory';
declare class PDFDocument {
    static from: (catalog: PDFCatalog, index: PDFObjectIndex) => PDFDocument;
    /** @hidden */
    header: PDFHeader;
    /** @hidden */
    catalog: PDFCatalog;
    /** @hidden */
    index: PDFObjectIndex;
    constructor(catalog: PDFCatalog, index: PDFObjectIndex);
    /**
     * Registers a [[PDFObject]] to the [[PDFDocument]]'s `index`. Returns a
     * [[PDFIndirectReference]] that can be used to reference the given `object`
     * in other `pdf-lib` methods.
     *
     * @param   object The [[PDFObject]] to be registered.
     *
     * @returns The [[PDFIndirectReference]] under which the `object` has been
     *          registered.
     */
    register: <T extends PDFObject>(object: T) => PDFIndirectReference<T>;
    /**
     * @returns An array of [[PDFPage]] objects representing the pages of the
     *          [[PDFDocument]]. The order of the [[PDFPage]] documents in the
     *          array mirrors the order in which they will be rendered in the
     *          [[PDFDocument]].
     */
    getPages: () => PDFPage[];
    /**
     * Creates a new [[PDFPage]] of the given `size`. And optionally, with the
     * given `resources` dictionary.
     *
     * Note that the [[PDFPage]] returned by this method is **not** automatically
     * added to the [[PDFDocument]]. You must call the [[addPage]] or [[insertPage]]
     * methods for it to be rendered in the document.
     *
     * @param size      A tuple containing the width and height of the page,
     *                  respectively.
     * @param resources A resources dictionary for the page.
     *
     * @returns The newly created [[PDFPage]].
     */
    createPage: (size: [number, number], resources?: PDFDictionary | undefined) => PDFPage;
    /**
     * Creates a new [[PDFContentStream]] with the given operators.
     *
     * Note that the [[PDFContentStream]] returned by this method is **not**
     * automatically registered to the document or added to any of its pages.
     * You must first call the [[register]] method for it to be registered to the
     * [[PDFDocument]]. Then, you must call [[PDFPage.addContentStreams]] to add
     * the registered [[PDFContentStream]] to the desired page(s).
     *
     * @param operators One or more [[PDFOperator]]s to be added to the
     *                  [[PDFContentStream]].
     *
     * @returns The newly created [[PDFContentStream]].
     */
    createContentStream: (...operators: (PDFOperator | PDFOperator[])[]) => PDFContentStream;
    /**
     * Adds a page to the end of the [[PDFDocument]].
     *
     * @param page The page to be added.
     */
    addPage: (page: PDFPage) => this;
    /**
     * Removes a page from the document.
     *
     * @param index The index of the page to be removed. The index is zero-based,
     *              e.g. the first page in the document is index `0`.
     */
    removePage: (index: number) => this;
    /**
     * Inserts a page into the document at the specified index. The page that is
     * displaced by the insertion will be become the page immediately following
     * the inserted page.
     *
     * @param index The index of the page to be removed. The index is zero-based,
     *              e.g. the first page in the document is index `0`.
     * @param page  The page to be inserted.
     */
    insertPage: (index: number, page: PDFPage) => this;
    /**
     * Embeds one of the Standard 14 Fonts fonts in the document. This method
     * does **not** require a `Uint8Array` containing a font to be passed, because
     * the Standard 14 Fonts are automatically available to all PDF documents.
     *
     * @param fontName Name of the font to be embedded.
     *
     * @returns A tuple containing the [[PDFIndirectReference]] under which the
     *          specified font is registered.
     */
    embedStandardFont: (fontName: IStandardFontNames) => [PDFIndirectReference<PDFDictionary>, PDFStandardFontFactory];
    /**
     * **Deprecated** - please use [[PDFDocument.embedNonStandardFont]] instead.
     *
     * Embeds the font contained in the specified `Uint8Array` in the document.
     *
     * @param fontData A `Uint8Array` containing an OpenType (`.otf`) or TrueType
     *                 (`.ttf`) font.
     *
     * @returns A tuple containing (1) the [[PDFIndirectReference]] under which the
     *          specified font is registered, and (2) a [[PDFFontFactory]] object
     *          containing font metadata properties and methods.
     */
    embedFont: (fontData: Uint8Array, fontFlags?: IFontFlagOptions) => [PDFIndirectReference<PDFDictionary>, PDFFontFactory];
    /**
     * Embeds the font contained in the specified `Uint8Array` in the document.
     *
     * @param fontData A `Uint8Array` containing an OpenType (`.otf`) or TrueType
     *                 (`.ttf`) font.
     *
     * @returns A tuple containing (1) the [[PDFIndirectReference]] under which the
     *          specified font is registered, and (2) a [[PDFEmbeddedFontFactory]]
     *          object containing font metadata properties and methods.
     */
    embedNonstandardFont: (fontData: Uint8Array) => [PDFIndirectReference<PDFDictionary>, PDFEmbeddedFontFactory];
    /**
     * Embeds the PNG image contained in the specified `Uint8Array` in the document.
     *
     * @param pngData A `Uint8Array` containing a PNG (`.png`) image.
     *
     * @returns A tuple containing (1) the [[PDFIndirectReference]] under which the
     *          specified image is registered, and (2) a [[PNGXObjectFactory]]
     *          object containing the image's width and height.
     */
    embedPNG: (pngData: Uint8Array) => [PDFIndirectReference<PDFRawStream>, PNGXObjectFactory];
    /**
     * Embeds the JPG image contained in the specified `Uint8Array` in the document.
     *
     * @param jpgData A `Uint8Array` containing a JPG (`.jpg`) image.
     *
     * @returns A tuple containing (1) the [[PDFIndirectReference]] under which the
     *          specified image is registered, and (2) a [[JPEGXObjectFactory]]
     *          object containing the image's width and height.
     */
    embedJPG: (jpgData: Uint8Array) => [PDFIndirectReference<PDFRawStream>, JPEGXObjectFactory];
}
export default PDFDocument;
