"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
}
Object.defineProperty(exports, "__esModule", { value: true });
var PDFDocument_1 = __importDefault(require("../pdf-document/PDFDocument"));
var PDFObjectIndex_1 = __importDefault(require("../pdf-document/PDFObjectIndex"));
var pdf_objects_1 = require("../pdf-objects");
var QOps_1 = __importDefault(require("../pdf-operators/graphics/graphics-state/QOps"));
var PDFParser_1 = __importDefault(require("../pdf-parser/PDFParser"));
var pdf_structures_1 = require("../pdf-structures");
var validate_1 = require("../../utils/validate");
var PDFDocumentFactory = /** @class */ (function () {
    function PDFDocumentFactory() {
    }
    /**
     * Creates a new [[PDFDocument]] object. Useful for creating new PDF documents.
     *
     * @returns The new [[PDFDocument]] object.
     */
    PDFDocumentFactory.create = function () {
        var index = PDFObjectIndex_1.default.create();
        var refs = {
            catalog: pdf_objects_1.PDFIndirectReference.forNumbers(1, 0),
            pageTree: pdf_objects_1.PDFIndirectReference.forNumbers(2, 0),
        };
        var catalog = pdf_structures_1.PDFCatalog.create(refs.pageTree, index);
        var pageTree = pdf_structures_1.PDFPageTree.createRootNode(pdf_objects_1.PDFArray.fromArray([], index), index);
        index.assign(refs.catalog, catalog);
        index.assign(refs.pageTree, pageTree);
        return PDFDocument_1.default.from(catalog, index);
    };
    /**
     * Loads an existing PDF document contained from the specified `Uint8Array`
     * into a [[PDFDocument]] object. Useful for modifying existing PDF documents.
     *
     * @param data A `Uint8Array` containing the raw bytes of a PDF document.
     *
     * @returns A [[PDFDocument]] object initialized from the provided document.
     */
    PDFDocumentFactory.load = function (data) {
        validate_1.validate(data, validate_1.isInstance(Uint8Array), '"PDFDocumentFactory.load()" must be called with an argument of type Uint8Array.');
        var index = PDFObjectIndex_1.default.create();
        var pdfParser = new PDFParser_1.default();
        var parsedPdf = pdfParser.parse(data, index);
        var indexMap = PDFDocumentFactory.normalize(parsedPdf);
        index.index = indexMap;
        var pushGraphicsStateContentStream = pdf_structures_1.PDFContentStream.of(pdf_objects_1.PDFDictionary.from({}, index), QOps_1.default.q.operator);
        var popGraphicsStateContentStream = pdf_structures_1.PDFContentStream.of(pdf_objects_1.PDFDictionary.from({}, index), QOps_1.default.Q.operator);
        var maxObjectNumber = parsedPdf.maxObjectNumber;
        var ref1 = pdf_objects_1.PDFIndirectReference.forNumbers(maxObjectNumber + 1, 0);
        var ref2 = pdf_objects_1.PDFIndirectReference.forNumbers(maxObjectNumber + 2, 0);
        index.assign(ref1, pushGraphicsStateContentStream);
        index.assign(ref2, popGraphicsStateContentStream);
        index.pushGraphicsStateContentStream = ref1;
        index.popGraphicsStateContentStream = ref2;
        return PDFDocument_1.default.from(parsedPdf.catalog, index);
    };
    // TODO: Need to throw out objects with "free" obj numbers...
    /** @hidden */
    PDFDocumentFactory.normalize = function (_a) {
        var dictionaries = _a.dictionaries, arrays = _a.arrays, body = _a.original.body, updates = _a.updates;
        var index = new Map();
        // Remove Object Streams and Cross Reference Streams, because we've already
        // parsed the Object Streams into PDFIndirectObjects, and will just write
        // them as such and use normal xref tables to reference them.
        var shouldKeep = function (object) {
            return !(object instanceof pdf_structures_1.PDFObjectStream) &&
                !(object instanceof pdf_objects_1.PDFStream &&
                    object.dictionary.getMaybe('Type') === pdf_objects_1.PDFName.from('XRef'));
        };
        // Initialize index with objects in the original body
        body.forEach(function (_a, ref) {
            var pdfObject = _a.pdfObject;
            if (shouldKeep(pdfObject))
                index.set(ref, pdfObject);
        });
        // Update index with most recent version of each object
        // TODO: This could be omitted to recover a previous version of the document...
        updates.forEach(function (_a) {
            var updateBody = _a.body;
            updateBody.forEach(function (_a, ref) {
                var pdfObject = _a.pdfObject;
                if (shouldKeep(pdfObject))
                    index.set(ref, pdfObject);
            });
        });
        return index;
    };
    return PDFDocumentFactory;
}());
exports.default = PDFDocumentFactory;
