"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
}
Object.defineProperty(exports, "__esModule", { value: true });
var initial_1 = __importDefault(require("lodash/initial"));
var last_1 = __importDefault(require("lodash/last"));
var pdf_structures_1 = require("../pdf-structures");
var utils_1 = require("../../utils");
var validate_1 = require("../../utils/validate");
var PDFObjectIndex_1 = __importDefault(require("../pdf-document/PDFObjectIndex"));
var parseDocument_1 = __importDefault(require("./parseDocument"));
var PDFParser = /** @class */ (function () {
    function PDFParser() {
        var _this = this;
        this.activelyParsing = false;
        this.maxObjectNumber = -1;
        this.arrays = [];
        this.dictionaries = [];
        this.body = new Map();
        this.updates = [];
        this.parse = function (bytes, index) {
            validate_1.validate(index, validate_1.isInstance(PDFObjectIndex_1.default), '"index" must be an instance of PDFObjectIndex');
            if (_this.activelyParsing)
                utils_1.error('Cannot parse documents concurrently');
            _this.activelyParsing = true;
            _this.maxObjectNumber = -1;
            _this.arrays = [];
            _this.dictionaries = [];
            _this.catalog = undefined;
            _this.header = undefined;
            _this.body = new Map();
            _this.xRefTable = undefined;
            _this.trailer = undefined;
            _this.linearization = undefined;
            _this.updates = [];
            try {
                parseDocument_1.default(bytes, index, _this.parseHandlers);
                _this.activelyParsing = false;
            }
            catch (e) {
                _this.activelyParsing = false;
                throw e;
            }
            return {
                maxObjectNumber: _this.maxObjectNumber,
                catalog: _this.catalog,
                arrays: _this.arrays,
                dictionaries: _this.dictionaries,
                original: {
                    header: _this.header,
                    linearization: _this.linearization,
                    body: _this.body,
                    xRefTable: _this.xRefTable,
                    trailer: _this.trailer,
                },
                // Drop the last element, because it will always be empty:
                updates: initial_1.default(_this.updates),
            };
        };
        this.updateMaxObjectNumber = function (ref) {
            if (ref.objectNumber > _this.maxObjectNumber) {
                _this.maxObjectNumber = ref.objectNumber;
            }
        };
        this.handleArray = function (array) {
            _this.arrays.push(array);
        };
        this.handleDict = function (dict) {
            _this.dictionaries.push(dict);
            if (dict instanceof pdf_structures_1.PDFCatalog)
                _this.catalog = dict;
        };
        this.handleObjectStream = function (_a) {
            var objects = _a.objects;
            objects.forEach(function (indirectObj) {
                if (_this.updates.length > 0) {
                    last_1.default(_this.updates).body.set(indirectObj.getReference(), indirectObj);
                }
                else {
                    _this.body.set(indirectObj.getReference(), indirectObj);
                }
                _this.updateMaxObjectNumber(indirectObj.reference);
            });
        };
        this.handleIndirectObj = function (indirectObj) {
            if (_this.updates.length > 0) {
                last_1.default(_this.updates).body.set(indirectObj.getReference(), indirectObj);
            }
            else {
                _this.body.set(indirectObj.getReference(), indirectObj);
            }
            _this.updateMaxObjectNumber(indirectObj.reference);
        };
        this.handleHeader = function (header) {
            _this.header = header;
        };
        this.handleXRefTable = function (xRefTable) {
            if (!_this.trailer)
                _this.xRefTable = xRefTable;
            else
                last_1.default(_this.updates).xRefTable = xRefTable;
        };
        this.handleTrailer = function (trailer) {
            if (!_this.trailer)
                _this.trailer = trailer;
            else
                last_1.default(_this.updates).trailer = trailer;
            _this.updates.push({
                body: new Map(),
                xRefTable: undefined,
                trailer: undefined,
            });
        };
        this.handleLinearization = function (linearization) {
            _this.linearization = linearization;
        };
        this.parseHandlers = {
            onParseArray: this.handleArray,
            onParseDict: this.handleDict,
            onParseObjectStream: this.handleObjectStream,
            onParseIndirectObj: this.handleIndirectObj,
            onParseHeader: this.handleHeader,
            onParseXRefTable: this.handleXRefTable,
            onParseTrailer: this.handleTrailer,
            onParseLinearization: this.handleLinearization,
        };
    }
    return PDFParser;
}());
exports.default = PDFParser;
