"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
}
Object.defineProperty(exports, "__esModule", { value: true });
var pdf_objects_1 = require("../pdf-objects");
var utils_1 = require("../../utils");
var validate_1 = require("../../utils/validate");
var parseBool_1 = __importDefault(require("./parseBool"));
var parseDict_1 = __importDefault(require("./parseDict"));
var parseHexString_1 = __importDefault(require("./parseHexString"));
var parseIndirectRef_1 = __importDefault(require("./parseIndirectRef"));
var parseName_1 = __importDefault(require("./parseName"));
var parseNull_1 = __importDefault(require("./parseNull"));
var parseNumber_1 = __importDefault(require("./parseNumber"));
var parseString_1 = __importDefault(require("./parseString"));
/**
 * Accepts an array of bytes as input. Checks to see if the first characters in the
 * trimmed input make up a PDF Array.
 *
 * If so, returns a tuple containing (1) an object representing the parsed PDFArray
 * and (2) a subarray of the input with the characters making up the parsed array
 * removed. The "onParseArray" parse handler will also be called with the PDFArray
 * object.
 *
 * If not, null is returned.
 *
 * Note that the elements of the PDF Array are recursively parsed, so the
 * appropriate parse handlers will be called when each element of the array is
 * parsed. The returned PDFArray's elements will be composed of PDFObjects.
 */
var parseArray = function (input, index, parseHandlers) {
    if (parseHandlers === void 0) { parseHandlers = {}; }
    // Make sure it is possible for this to be an array.
    var trimmed = utils_1.trimArrayAndRemoveComments(input);
    if (utils_1.arrayCharAt(trimmed, 0) !== '[')
        return undefined;
    var pdfArray = pdf_objects_1.PDFArray.fromArray([], index);
    // Recursively parse each element of the array
    var remainder = trimmed.subarray(1); // Remove the '['
    while (utils_1.arrayCharAt(utils_1.trimArrayAndRemoveComments(remainder), 0) !== ']') {
        // Parse the value for this element
        var _a = parseName_1.default(remainder, parseHandlers) ||
            parseDict_1.default(remainder, index, parseHandlers) ||
            parseArray(remainder, index, parseHandlers) ||
            parseString_1.default(remainder, parseHandlers) ||
            parseHexString_1.default(remainder, parseHandlers) ||
            parseBool_1.default(remainder, parseHandlers) ||
            parseNull_1.default(remainder, parseHandlers) ||
            parseIndirectRef_1.default(remainder, parseHandlers) ||
            parseNumber_1.default(remainder, parseHandlers) ||
            utils_1.error('Failed to parse array element'), pdfObject = _a[0], r = _a[1];
        pdfArray.push(pdfObject);
        remainder = r;
    }
    var remainderTrim = utils_1.trimArrayAndRemoveComments(remainder);
    // Make sure the brackets are paired
    validate_1.validate(utils_1.arrayCharAt(remainderTrim, 0), validate_1.isIdentity(']'), 'Mismatched brackets!');
    remainder = utils_1.trimArrayAndRemoveComments(remainderTrim.subarray(1)); // Remove the ']'
    if (parseHandlers.onParseArray)
        parseHandlers.onParseArray(pdfArray);
    return [pdfArray, remainder];
};
exports.default = parseArray;
