"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
}
Object.defineProperty(exports, "__esModule", { value: true });
var pdf_objects_1 = require("../pdf-objects");
var pdf_structures_1 = require("../pdf-structures");
var utils_1 = require("../../utils");
var validate_1 = require("../../utils/validate");
var parseArray_1 = __importDefault(require("./parseArray"));
var parseBool_1 = __importDefault(require("./parseBool"));
var parseHexString_1 = __importDefault(require("./parseHexString"));
var parseIndirectRef_1 = __importDefault(require("./parseIndirectRef"));
var parseName_1 = __importDefault(require("./parseName"));
var parseNull_1 = __importDefault(require("./parseNull"));
var parseNumber_1 = __importDefault(require("./parseNumber"));
var parseString_1 = __importDefault(require("./parseString"));
// prettier-ignore
var typeDict = function (dict) {
    if (dict.getMaybe('Linearized'))
        return pdf_structures_1.PDFLinearizationParams.fromDict(dict);
    switch (dict.getMaybe('Type')) {
        case pdf_objects_1.PDFName.from('Catalog'): return pdf_structures_1.PDFCatalog.fromDict(dict);
        case pdf_objects_1.PDFName.from('Pages'): return pdf_structures_1.PDFPageTree.fromDict(dict);
        case pdf_objects_1.PDFName.from('Page'): return pdf_structures_1.PDFPage.fromDict(dict);
        default: return dict;
    }
};
/**
 * Accepts an array of bytes as input. Checks to see if the first characters in
 * the trimmed input make up a PDF Dictionary.
 *
 * If so, returns a tuple containing (1) an object representing the parsed
 * PDFDictionary and (2) a subarray of the input with the characters making up
 * the parsed dictionary removed. The "onParseDict" parse handler will also be
 * called with the PDFDictionary object.
 *
 * If not, null is returned.
 *
 * Note that the entries of the PDF Dictionary are recursively parsed, so the
 * appropriate parse handlers will be called when each entry of the dictionary
 * is parsed. The returned PDFDictionary's keys will be PDFName objects, and its
 * values will be PDFObjects.
 */
var parseDict = function (input, index, parseHandlers) {
    if (parseHandlers === void 0) { parseHandlers = {}; }
    var trimmed = utils_1.trimArrayAndRemoveComments(input);
    if (utils_1.arrayToString(trimmed, 0, 2) !== '<<')
        return undefined;
    var pdfDict = pdf_objects_1.PDFDictionary.from(new Map(), index);
    // Recursively parse each entry in the dictionary
    var remainder = utils_1.trimArrayAndRemoveComments(trimmed.subarray(2));
    while (utils_1.arrayToString(utils_1.trimArrayAndRemoveComments(remainder), 0, 2) !== '>>') {
        // Parse the key for this entry
        var _a = parseName_1.default(remainder) || utils_1.error('Failed to parse dictionary key'), key = _a[0], r1 = _a[1];
        remainder = r1;
        // Parse the value for this entry
        var _b = parseName_1.default(remainder, parseHandlers) ||
            parseDict(remainder, index, parseHandlers) ||
            parseArray_1.default(remainder, index, parseHandlers) ||
            parseString_1.default(remainder, parseHandlers) ||
            parseHexString_1.default(remainder, parseHandlers) ||
            parseBool_1.default(remainder, parseHandlers) ||
            parseNull_1.default(remainder, parseHandlers) ||
            parseIndirectRef_1.default(remainder, parseHandlers) ||
            parseNumber_1.default(remainder, parseHandlers) ||
            utils_1.error('Failed to parse dictionary value'), pdfObject = _b[0], r2 = _b[1];
        pdfDict.set(key, pdfObject);
        remainder = r2;
    }
    var remainderTrim = utils_1.trimArrayAndRemoveComments(remainder);
    // Make sure the brackets are paired
    validate_1.validate(utils_1.arrayToString(remainderTrim, 0, 2), validate_1.isIdentity('>>'), 'Mismatched brackets!');
    remainder = utils_1.trimArrayAndRemoveComments(remainderTrim.subarray(2)); // Remove ending '>>' pair
    var typedDict = typeDict(pdfDict);
    if (parseHandlers.onParseDict)
        parseHandlers.onParseDict(typedDict);
    return [typedDict, remainder];
};
exports.default = parseDict;
