"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
}
Object.defineProperty(exports, "__esModule", { value: true });
// tslint:disable-next-line:no-unused-variable
var pdf_objects_1 = require("../pdf-objects");
var utils_1 = require("../../utils");
var parseArray_1 = __importDefault(require("./parseArray"));
var parseBool_1 = __importDefault(require("./parseBool"));
var parseDictOrStream_1 = __importDefault(require("./parseDictOrStream"));
var parseHexString_1 = __importDefault(require("./parseHexString"));
var parseIndirectRef_1 = __importDefault(require("./parseIndirectRef"));
var parseName_1 = __importDefault(require("./parseName"));
var parseNull_1 = __importDefault(require("./parseNull"));
var parseNumber_1 = __importDefault(require("./parseNumber"));
var parseString_1 = __importDefault(require("./parseString"));
/**
 * Accepts an array of bytes as input. Checks to see if the first characters in the
 * trimmed input make up a PDF Indirect Object.
 *
 * If so, returns a tuple containing (1) an object representing the parsed PDF
 * Indirect Object and (2) a subarray of the input with the characters making up
 * the parsed indirect object removed. The "onParseIndirectObj" parse handler will
 * also be called with the PDFIndirectObject.
 *
 * If not, null is returned.
 */
var parseIndirectObj = function (input, index, parseHandlers) {
    if (parseHandlers === void 0) { parseHandlers = {}; }
    var trimmed = utils_1.trimArrayAndRemoveComments(input);
    var indirectObjRegex = /^(\d+)[\0\t\n\f\r ]*(\d+)[\0\t\n\f\r ]*obj/;
    // Check that initial characters make up an indirect object "header"
    var objIdx = utils_1.arrayIndexOf(trimmed, 'obj');
    var result = utils_1.arrayToString(trimmed.subarray(0, objIdx + 3)).match(indirectObjRegex);
    if (!result)
        return undefined;
    var _fullMatch = result[0], objNum = result[1], genNum = result[2];
    // Remove the indirect object "header" from the trimmed input
    var afterHeader = utils_1.trimArrayAndRemoveComments(trimmed.subarray(objIdx + 3));
    // Try to parse the object contents bytes
    var _a = parseDictOrStream_1.default(afterHeader, index, parseHandlers) ||
        parseArray_1.default(afterHeader, index, parseHandlers) ||
        parseName_1.default(afterHeader, parseHandlers) ||
        parseString_1.default(afterHeader, parseHandlers) ||
        parseHexString_1.default(afterHeader, parseHandlers) ||
        parseBool_1.default(afterHeader, parseHandlers) ||
        parseNull_1.default(afterHeader, parseHandlers) ||
        parseIndirectRef_1.default(afterHeader, parseHandlers) ||
        parseNumber_1.default(afterHeader, parseHandlers) ||
        utils_1.error('Failed to parse object contents'), contentObj = _a[0], r = _a[1];
    var trimmedRemaining = utils_1.trimArrayAndRemoveComments(r);
    // Make sure we're now at the end of the object
    var endobjIdx = utils_1.arrayIndexOf(trimmedRemaining, 'endobj');
    if (endobjIdx === undefined)
        throw new Error('Invalid Indirect Object');
    if (endobjIdx !== 0)
        throw new Error('Incorrectly parsed object contents');
    var indirectObj = pdf_objects_1.PDFIndirectObject.of(contentObj).setReferenceNumbers(Number(objNum), Number(genNum));
    if (parseHandlers.onParseIndirectObj) {
        parseHandlers.onParseIndirectObj(indirectObj);
    }
    return [indirectObj, trimmedRemaining.subarray(6)];
};
exports.default = parseIndirectObj;
