"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
}
Object.defineProperty(exports, "__esModule", { value: true });
var sum_1 = __importDefault(require("lodash/sum"));
var standard_fonts_1 = require("@pdf-lib/standard-fonts");
var PDFDocument_1 = __importDefault(require("../../pdf-document/PDFDocument"));
var pdf_objects_1 = require("../../pdf-objects");
var values_1 = __importDefault(require("lodash/values"));
var utils_1 = require("../../../utils");
var validate_1 = require("../../../utils/validate");
/**
 * This Factory supports Standard fonts.
 *
 * A note of thanks to the developers of https://github.com/foliojs/pdfkit,
 * as this class borrows from:
 * https://github.com/foliojs/pdfkit/blob/f91bdd61c164a72ea06be1a43dc0a412afc3925f/lib/font/afm.coffee
 */
var PDFStandardFontFactory = /** @class */ (function () {
    function PDFStandardFontFactory(fontName) {
        var _this = this;
        /**
         * Embeds the font into a [[PDFDocument]].
         *
         * @param pdfDoc A `PDFDocument` object into which the font will be embedded.
         *
         * @returns A `PDFIndirectReference` to the font dictionary that was
         *          embedded in the `PDFDocument`.
         */
        this.embedFontIn = function (pdfDoc) {
            validate_1.validate(pdfDoc, validate_1.isInstance(PDFDocument_1.default), 'PDFFontFactory.embedFontIn: "pdfDoc" must be an instance of PDFDocument');
            var fontDict = pdf_objects_1.PDFDictionary.from({
                Type: pdf_objects_1.PDFName.from('Font'),
                Subtype: pdf_objects_1.PDFName.from('Type1'),
                BaseFont: pdf_objects_1.PDFName.from(_this.font.FontName),
            }, pdfDoc.index);
            // ZapfDingbats and Symbol don't have explicit Encoding entries
            if (_this.encoding === standard_fonts_1.Encodings.WinAnsi) {
                fontDict.set('Encoding', pdf_objects_1.PDFName.from('WinAnsiEncoding'));
            }
            return pdfDoc.register(fontDict);
        };
        /**
         * Encode the JavaScript string into this font. JavaScript encodes strings in
         * Unicode, but standard fonts use either WinAnsi, ZapfDingbats, or Symbol
         * encodings. This method should be used to encode text before passing the
         * encoded text to one of the text showing operators, such as [[drawText]] or
         * [[drawLinesOfText]].
         *
         * @param text The string of text to be encoded.
         *
         * @returns A `PDFHexString` of the encoded text.
         */
        this.encodeText = function (text) {
            return pdf_objects_1.PDFHexString.fromString(_this.encodeTextAsGlyphs(text)
                .map(function (glyph) { return glyph.code; })
                .map(utils_1.toHexString)
                .join(''));
        };
        /**
         * Measures the width of the JavaScript string when displayed as glyphs of
         * this font of a particular `size`.
         *
         * @param text The string of text to be measured.
         * @param size The size to be used when calculating the text's width.
         *
         * @returns A `number` representing the width of the text.
         */
        this.widthOfTextAtSize = function (text, size) {
            var charNames = _this.encodeTextAsGlyphs(text).map(function (glyph) { return glyph.name; });
            var widths = charNames.map(function (charName, idx) {
                var left = charName;
                var right = charNames[idx + 1];
                var kernAmount = _this.font.getXAxisKerningForPair(left, right) || 0;
                return _this.widthOfGlyph(left) + kernAmount;
            });
            var scale = size / 1000;
            return sum_1.default(widths) * scale;
        };
        /**
         * Measures the height of this font at a particular size. Note that the height
         * of the font is independent of the particular glyphs being displayed, so
         * this method does not accept a `text` param, like
         * [[PDFStandardFontFactory.widthOfTextAtSize]] does.
         */
        this.heightOfFontAtSize = function (size) {
            var _a = _this.font, Ascender = _a.Ascender, Descender = _a.Descender, FontBBox = _a.FontBBox;
            var yTop = Ascender || FontBBox[3];
            var yBottom = Descender || FontBBox[1];
            return ((yTop - yBottom) / 1000) * size;
        };
        /**
         * Measures the size of this font at a particular height. Note that the size
         * of the font is independent of the particular glyphs being displayed, so
         * this method does not accept a `text` param, like
         * [[PDFStandardFontFactory.widthOfTextAtSize]] does.
         */
        this.sizeOfFontAtHeight = function (height) {
            var _a = _this.font, Ascender = _a.Ascender, Descender = _a.Descender, FontBBox = _a.FontBBox;
            var yTop = Ascender || FontBBox[3];
            var yBottom = Descender || FontBBox[1];
            return (1000 * height) / (yTop - yBottom);
        };
        // We'll default to 250 if our font metrics don't specify a width
        this.widthOfGlyph = function (glyphName) {
            return _this.font.getWidthOfGlyph(glyphName) || 250;
        };
        this.encodeTextAsGlyphs = function (text) {
            return text
                .split('')
                .map(utils_1.toCharCode)
                .map(_this.encoding.encodeUnicodeCodePoint);
        };
        validate_1.validate(fontName, validate_1.oneOf.apply(void 0, values_1.default(standard_fonts_1.FontNames)), 'PDFDocument.embedFont: "fontName" must be one of the Standard 14 Fonts: ' +
            values_1.default(standard_fonts_1.FontNames).join(', '));
        this.fontName = fontName;
        this.font = standard_fonts_1.Font.load(fontName);
        // prettier-ignore
        this.encoding =
            fontName === standard_fonts_1.FontNames.ZapfDingbats ? standard_fonts_1.Encodings.ZapfDingbats
                : fontName === standard_fonts_1.FontNames.Symbol ? standard_fonts_1.Encodings.Symbol
                    : standard_fonts_1.Encodings.WinAnsi;
    }
    PDFStandardFontFactory.for = function (fontName) {
        return new PDFStandardFontFactory(fontName);
    };
    return PDFStandardFontFactory;
}());
exports.default = PDFStandardFontFactory;
