"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
}
Object.defineProperty(exports, "__esModule", { value: true });
var get_1 = __importDefault(require("lodash/get"));
var isEmpty_1 = __importDefault(require("lodash/isEmpty"));
var simple_1 = require("../../pdf-operators/simple");
// =============================================================================
// Based on: http://spencermortensen.com/articles/bezier-circle/
//   P_0 = (0,1),  P_1 = (c,1),   P_2 = (1,c),   P_3 = (1,0)
//   P_0 = (1,0),  P_1 = (1,-c),  P_2 = (c,-1),  P_3 = (0,-1)
//   P_0 = (0,-1), P_1 = (-c,-1), P_3 = (-1,-c), P_4 = (-1,0)
//   P_0 = (-1,0), P_1 = (-1,c),  P_2 = (-c,1),  P_3 = (0,1)
//     with c = 0.551915024494
/** @hidden */
var C_VAL = 0.551915024494;
/** @hidden */
var drawEllipsePath = function (_a) {
    var _b = _a.x, x = _b === void 0 ? 0 : _b, _c = _a.y, y = _c === void 0 ? 0 : _c, _d = _a.xScale, xScale = _d === void 0 ? 100 : _d, _e = _a.yScale, yScale = _e === void 0 ? 100 : _e, _f = _a.rotationAngle, rotationAngle = _f === void 0 ? 0 : _f, _g = _a.skewAngles, skewAngles = _g === void 0 ? { xAxis: 0, yAxis: 0 } : _g;
    return [
        simple_1.pushGraphicsState(),
        simple_1.translate(x, y),
        simple_1.rotateRadians(rotationAngle),
        simple_1.scale(xScale, yScale),
        simple_1.skewRadians(skewAngles.xAxis, skewAngles.yAxis),
        simple_1.moveTo(0, 1),
        simple_1.appendBezierCurve(C_VAL, 1, 1, C_VAL, 1, 0),
        simple_1.appendBezierCurve(1, -C_VAL, C_VAL, -1, 0, -1),
        simple_1.appendBezierCurve(-C_VAL, -1, -1, -C_VAL, -1, 0),
        simple_1.appendBezierCurve(-1, C_VAL, -C_VAL, 1, 0, 1),
        simple_1.popGraphicsState(),
    ];
};
/**
 * Draws an ellipse in a content stream.
 *
 * ```javascript
 * const contentStream = pdfDoc.register(
 *   pdfDoc.createContentStream(
 *     drawEllipse({
 *       x: 25,
 *       y: 50,
 *       xScale: 50,
 *       yScale: 150,
 *       rotateDegrees: 45,
 *       skewDegrees: { xAxis: 30, yAxis: 30 },
 *       borderWidth: 25,
 *       colorRgb: [0.25, 1.0, 0.79],
 *       borderColorRgb: [0.79, 0.25, 1.0],
 *     }),
 *   ),
 * );
 * const page = pdfDoc
 *   .createPage([250, 500])
 *   .addContentStreams(contentStream);
 * ```
 *
 * @param options An options object with named parameters.
 */
exports.drawEllipse = function (options) { return [
    simple_1.pushGraphicsState(),
    simple_1.fillingRgbColor(get_1.default(options, 'colorRgb[0]', 0), get_1.default(options, 'colorRgb[1]', 0), get_1.default(options, 'colorRgb[2]', 0)),
    simple_1.strokingRgbColor(get_1.default(options, 'borderColorRgb[0]', 0), get_1.default(options, 'borderColorRgb[1]', 0), get_1.default(options, 'borderColorRgb[2]', 0)),
    simple_1.lineWidth(options.borderWidth || 15)
].concat(drawEllipsePath({
    x: options.x || 0,
    y: options.y || 0,
    xScale: options.xScale || 100,
    yScale: options.yScale || 100,
    rotationAngle: options.rotateDegrees
        ? simple_1.degreesToRadians(options.rotateDegrees)
        : options.rotateRadians,
    skewAngles: options.skewDegrees
        ? {
            xAxis: simple_1.degreesToRadians(options.skewDegrees.xAxis),
            yAxis: simple_1.degreesToRadians(options.skewDegrees.yAxis),
        }
        : options.skewRadians,
}), [
    // prettier-ignore
    !isEmpty_1.default(options.colorRgb) && !isEmpty_1.default(options.borderColorRgb) ? simple_1.fillAndStroke()
        : !isEmpty_1.default(options.colorRgb) ? simple_1.fill()
            : !isEmpty_1.default(options.borderColorRgb) ? simple_1.stroke()
                : simple_1.closePath(),
    simple_1.popGraphicsState(),
]); };
/**
 * Draws a circle in a content stream.
 *
 * ```javascript
 * const contentStream = pdfDoc.register(
 *   pdfDoc.createContentStream(
 *     drawCircle({
 *       x: 25,
 *       y: 50,
 *       size: 50,
 *       rotateDegrees: 45,
 *       skewDegrees: { xAxis: 30, yAxis: 30 },
 *       borderWidth: 25,
 *       colorRgb: [0.25, 1.0, 0.79],
 *       borderColorRgb: [0.79, 0.25, 1.0],
 *     }),
 *   ),
 * );
 * const page = pdfDoc
 *   .createPage([250, 500])
 *   .addContentStreams(contentStream);
 * ```
 *
 * @param options An options object with named parameters.
 */
exports.drawCircle = function (options) {
    return exports.drawEllipse({
        x: options.x || 0,
        y: options.y || 0,
        xScale: options.size || 100,
        yScale: options.size || 100,
        borderWidth: options.borderWidth || 15,
        colorRgb: options.colorRgb || [],
        borderColorRgb: options.borderColorRgb || [],
        rotateDegrees: options.rotateDegrees,
        rotateRadians: options.rotateRadians,
        skewDegrees: options.skewDegrees,
        skewRadians: options.skewRadians,
    });
};
