import pako from 'pako';

var global$1 = typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : {};

var lookup = [];
var revLookup = [];
var Arr = typeof Uint8Array !== 'undefined' ? Uint8Array : Array;
var inited = false;

function init() {
  inited = true;
  var code = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';

  for (var i = 0, len = code.length; i < len; ++i) {
    lookup[i] = code[i];
    revLookup[code.charCodeAt(i)] = i;
  }

  revLookup['-'.charCodeAt(0)] = 62;
  revLookup['_'.charCodeAt(0)] = 63;
}

function toByteArray(b64) {
  if (!inited) {
    init();
  }

  var i, j, l, tmp, placeHolders, arr;
  var len = b64.length;

  if (len % 4 > 0) {
    throw new Error('Invalid string. Length must be a multiple of 4');
  } // the number of equal signs (place holders)
  // if there are two placeholders, than the two characters before it
  // represent one byte
  // if there is only one, then the three characters before it represent 2 bytes
  // this is just a cheap hack to not do indexOf twice


  placeHolders = b64[len - 2] === '=' ? 2 : b64[len - 1] === '=' ? 1 : 0; // base64 is 4/3 + up to two characters of the original data

  arr = new Arr(len * 3 / 4 - placeHolders); // if there are placeholders, only get up to the last complete 4 chars

  l = placeHolders > 0 ? len - 4 : len;
  var L = 0;

  for (i = 0, j = 0; i < l; i += 4, j += 3) {
    tmp = revLookup[b64.charCodeAt(i)] << 18 | revLookup[b64.charCodeAt(i + 1)] << 12 | revLookup[b64.charCodeAt(i + 2)] << 6 | revLookup[b64.charCodeAt(i + 3)];
    arr[L++] = tmp >> 16 & 0xFF;
    arr[L++] = tmp >> 8 & 0xFF;
    arr[L++] = tmp & 0xFF;
  }

  if (placeHolders === 2) {
    tmp = revLookup[b64.charCodeAt(i)] << 2 | revLookup[b64.charCodeAt(i + 1)] >> 4;
    arr[L++] = tmp & 0xFF;
  } else if (placeHolders === 1) {
    tmp = revLookup[b64.charCodeAt(i)] << 10 | revLookup[b64.charCodeAt(i + 1)] << 4 | revLookup[b64.charCodeAt(i + 2)] >> 2;
    arr[L++] = tmp >> 8 & 0xFF;
    arr[L++] = tmp & 0xFF;
  }

  return arr;
}

function tripletToBase64(num) {
  return lookup[num >> 18 & 0x3F] + lookup[num >> 12 & 0x3F] + lookup[num >> 6 & 0x3F] + lookup[num & 0x3F];
}

function encodeChunk(uint8, start, end) {
  var tmp;
  var output = [];

  for (var i = start; i < end; i += 3) {
    tmp = (uint8[i] << 16) + (uint8[i + 1] << 8) + uint8[i + 2];
    output.push(tripletToBase64(tmp));
  }

  return output.join('');
}

function fromByteArray(uint8) {
  if (!inited) {
    init();
  }

  var tmp;
  var len = uint8.length;
  var extraBytes = len % 3; // if we have 1 byte left, pad 2 bytes

  var output = '';
  var parts = [];
  var maxChunkLength = 16383; // must be multiple of 3
  // go through the array every three bytes, we'll deal with trailing stuff later

  for (var i = 0, len2 = len - extraBytes; i < len2; i += maxChunkLength) {
    parts.push(encodeChunk(uint8, i, i + maxChunkLength > len2 ? len2 : i + maxChunkLength));
  } // pad the end with zeros, but make sure to not forget the extra bytes


  if (extraBytes === 1) {
    tmp = uint8[len - 1];
    output += lookup[tmp >> 2];
    output += lookup[tmp << 4 & 0x3F];
    output += '==';
  } else if (extraBytes === 2) {
    tmp = (uint8[len - 2] << 8) + uint8[len - 1];
    output += lookup[tmp >> 10];
    output += lookup[tmp >> 4 & 0x3F];
    output += lookup[tmp << 2 & 0x3F];
    output += '=';
  }

  parts.push(output);
  return parts.join('');
}

function read(buffer, offset, isLE, mLen, nBytes) {
  var e, m;
  var eLen = nBytes * 8 - mLen - 1;
  var eMax = (1 << eLen) - 1;
  var eBias = eMax >> 1;
  var nBits = -7;
  var i = isLE ? nBytes - 1 : 0;
  var d = isLE ? -1 : 1;
  var s = buffer[offset + i];
  i += d;
  e = s & (1 << -nBits) - 1;
  s >>= -nBits;
  nBits += eLen;

  for (; nBits > 0; e = e * 256 + buffer[offset + i], i += d, nBits -= 8) {}

  m = e & (1 << -nBits) - 1;
  e >>= -nBits;
  nBits += mLen;

  for (; nBits > 0; m = m * 256 + buffer[offset + i], i += d, nBits -= 8) {}

  if (e === 0) {
    e = 1 - eBias;
  } else if (e === eMax) {
    return m ? NaN : (s ? -1 : 1) * Infinity;
  } else {
    m = m + Math.pow(2, mLen);
    e = e - eBias;
  }

  return (s ? -1 : 1) * m * Math.pow(2, e - mLen);
}
function write(buffer, value, offset, isLE, mLen, nBytes) {
  var e, m, c;
  var eLen = nBytes * 8 - mLen - 1;
  var eMax = (1 << eLen) - 1;
  var eBias = eMax >> 1;
  var rt = mLen === 23 ? Math.pow(2, -24) - Math.pow(2, -77) : 0;
  var i = isLE ? 0 : nBytes - 1;
  var d = isLE ? 1 : -1;
  var s = value < 0 || value === 0 && 1 / value < 0 ? 1 : 0;
  value = Math.abs(value);

  if (isNaN(value) || value === Infinity) {
    m = isNaN(value) ? 1 : 0;
    e = eMax;
  } else {
    e = Math.floor(Math.log(value) / Math.LN2);

    if (value * (c = Math.pow(2, -e)) < 1) {
      e--;
      c *= 2;
    }

    if (e + eBias >= 1) {
      value += rt / c;
    } else {
      value += rt * Math.pow(2, 1 - eBias);
    }

    if (value * c >= 2) {
      e++;
      c /= 2;
    }

    if (e + eBias >= eMax) {
      m = 0;
      e = eMax;
    } else if (e + eBias >= 1) {
      m = (value * c - 1) * Math.pow(2, mLen);
      e = e + eBias;
    } else {
      m = value * Math.pow(2, eBias - 1) * Math.pow(2, mLen);
      e = 0;
    }
  }

  for (; mLen >= 8; buffer[offset + i] = m & 0xff, i += d, m /= 256, mLen -= 8) {}

  e = e << mLen | m;
  eLen += mLen;

  for (; eLen > 0; buffer[offset + i] = e & 0xff, i += d, e /= 256, eLen -= 8) {}

  buffer[offset + i - d] |= s * 128;
}

var toString = {}.toString;
var isArray = Array.isArray || function (arr) {
  return toString.call(arr) == '[object Array]';
};

var INSPECT_MAX_BYTES = 50;
/**
 * If `Buffer.TYPED_ARRAY_SUPPORT`:
 *   === true    Use Uint8Array implementation (fastest)
 *   === false   Use Object implementation (most compatible, even IE6)
 *
 * Browsers that support typed arrays are IE 10+, Firefox 4+, Chrome 7+, Safari 5.1+,
 * Opera 11.6+, iOS 4.2+.
 *
 * Due to various browser bugs, sometimes the Object implementation will be used even
 * when the browser supports typed arrays.
 *
 * Note:
 *
 *   - Firefox 4-29 lacks support for adding new properties to `Uint8Array` instances,
 *     See: https://bugzilla.mozilla.org/show_bug.cgi?id=695438.
 *
 *   - Chrome 9-10 is missing the `TypedArray.prototype.subarray` function.
 *
 *   - IE10 has a broken `TypedArray.prototype.subarray` function which returns arrays of
 *     incorrect length in some situations.

 * We detect these buggy browsers and set `Buffer.TYPED_ARRAY_SUPPORT` to `false` so they
 * get the Object implementation, which is slower but behaves correctly.
 */

Buffer.TYPED_ARRAY_SUPPORT = global$1.TYPED_ARRAY_SUPPORT !== undefined ? global$1.TYPED_ARRAY_SUPPORT : true;
/*
 * Export kMaxLength after typed array support is determined.
 */

var _kMaxLength = kMaxLength();

function kMaxLength() {
  return Buffer.TYPED_ARRAY_SUPPORT ? 0x7fffffff : 0x3fffffff;
}

function createBuffer(that, length) {
  if (kMaxLength() < length) {
    throw new RangeError('Invalid typed array length');
  }

  if (Buffer.TYPED_ARRAY_SUPPORT) {
    // Return an augmented `Uint8Array` instance, for best performance
    that = new Uint8Array(length);
    that.__proto__ = Buffer.prototype;
  } else {
    // Fallback: Return an object instance of the Buffer class
    if (that === null) {
      that = new Buffer(length);
    }

    that.length = length;
  }

  return that;
}
/**
 * The Buffer constructor returns instances of `Uint8Array` that have their
 * prototype changed to `Buffer.prototype`. Furthermore, `Buffer` is a subclass of
 * `Uint8Array`, so the returned instances will have all the node `Buffer` methods
 * and the `Uint8Array` methods. Square bracket notation works as expected -- it
 * returns a single octet.
 *
 * The `Uint8Array` prototype remains unmodified.
 */


function Buffer(arg, encodingOrOffset, length) {
  if (!Buffer.TYPED_ARRAY_SUPPORT && !(this instanceof Buffer)) {
    return new Buffer(arg, encodingOrOffset, length);
  } // Common case.


  if (typeof arg === 'number') {
    if (typeof encodingOrOffset === 'string') {
      throw new Error('If encoding is specified then the first argument must be a string');
    }

    return allocUnsafe(this, arg);
  }

  return from(this, arg, encodingOrOffset, length);
}
Buffer.poolSize = 8192; // not used by this implementation
// TODO: Legacy, not needed anymore. Remove in next major version.

Buffer._augment = function (arr) {
  arr.__proto__ = Buffer.prototype;
  return arr;
};

function from(that, value, encodingOrOffset, length) {
  if (typeof value === 'number') {
    throw new TypeError('"value" argument must not be a number');
  }

  if (typeof ArrayBuffer !== 'undefined' && value instanceof ArrayBuffer) {
    return fromArrayBuffer(that, value, encodingOrOffset, length);
  }

  if (typeof value === 'string') {
    return fromString(that, value, encodingOrOffset);
  }

  return fromObject(that, value);
}
/**
 * Functionally equivalent to Buffer(arg, encoding) but throws a TypeError
 * if value is a number.
 * Buffer.from(str[, encoding])
 * Buffer.from(array)
 * Buffer.from(buffer)
 * Buffer.from(arrayBuffer[, byteOffset[, length]])
 **/


Buffer.from = function (value, encodingOrOffset, length) {
  return from(null, value, encodingOrOffset, length);
};

if (Buffer.TYPED_ARRAY_SUPPORT) {
  Buffer.prototype.__proto__ = Uint8Array.prototype;
  Buffer.__proto__ = Uint8Array;
}

function assertSize(size) {
  if (typeof size !== 'number') {
    throw new TypeError('"size" argument must be a number');
  } else if (size < 0) {
    throw new RangeError('"size" argument must not be negative');
  }
}

function alloc(that, size, fill, encoding) {
  assertSize(size);

  if (size <= 0) {
    return createBuffer(that, size);
  }

  if (fill !== undefined) {
    // Only pay attention to encoding if it's a string. This
    // prevents accidentally sending in a number that would
    // be interpretted as a start offset.
    return typeof encoding === 'string' ? createBuffer(that, size).fill(fill, encoding) : createBuffer(that, size).fill(fill);
  }

  return createBuffer(that, size);
}
/**
 * Creates a new filled Buffer instance.
 * alloc(size[, fill[, encoding]])
 **/


Buffer.alloc = function (size, fill, encoding) {
  return alloc(null, size, fill, encoding);
};

function allocUnsafe(that, size) {
  assertSize(size);
  that = createBuffer(that, size < 0 ? 0 : checked(size) | 0);

  if (!Buffer.TYPED_ARRAY_SUPPORT) {
    for (var i = 0; i < size; ++i) {
      that[i] = 0;
    }
  }

  return that;
}
/**
 * Equivalent to Buffer(num), by default creates a non-zero-filled Buffer instance.
 * */


Buffer.allocUnsafe = function (size) {
  return allocUnsafe(null, size);
};
/**
 * Equivalent to SlowBuffer(num), by default creates a non-zero-filled Buffer instance.
 */


Buffer.allocUnsafeSlow = function (size) {
  return allocUnsafe(null, size);
};

function fromString(that, string, encoding) {
  if (typeof encoding !== 'string' || encoding === '') {
    encoding = 'utf8';
  }

  if (!Buffer.isEncoding(encoding)) {
    throw new TypeError('"encoding" must be a valid string encoding');
  }

  var length = byteLength(string, encoding) | 0;
  that = createBuffer(that, length);
  var actual = that.write(string, encoding);

  if (actual !== length) {
    // Writing a hex string, for example, that contains invalid characters will
    // cause everything after the first invalid character to be ignored. (e.g.
    // 'abxxcd' will be treated as 'ab')
    that = that.slice(0, actual);
  }

  return that;
}

function fromArrayLike(that, array) {
  var length = array.length < 0 ? 0 : checked(array.length) | 0;
  that = createBuffer(that, length);

  for (var i = 0; i < length; i += 1) {
    that[i] = array[i] & 255;
  }

  return that;
}

function fromArrayBuffer(that, array, byteOffset, length) {
  array.byteLength; // this throws if `array` is not a valid ArrayBuffer

  if (byteOffset < 0 || array.byteLength < byteOffset) {
    throw new RangeError('\'offset\' is out of bounds');
  }

  if (array.byteLength < byteOffset + (length || 0)) {
    throw new RangeError('\'length\' is out of bounds');
  }

  if (byteOffset === undefined && length === undefined) {
    array = new Uint8Array(array);
  } else if (length === undefined) {
    array = new Uint8Array(array, byteOffset);
  } else {
    array = new Uint8Array(array, byteOffset, length);
  }

  if (Buffer.TYPED_ARRAY_SUPPORT) {
    // Return an augmented `Uint8Array` instance, for best performance
    that = array;
    that.__proto__ = Buffer.prototype;
  } else {
    // Fallback: Return an object instance of the Buffer class
    that = fromArrayLike(that, array);
  }

  return that;
}

function fromObject(that, obj) {
  if (internalIsBuffer(obj)) {
    var len = checked(obj.length) | 0;
    that = createBuffer(that, len);

    if (that.length === 0) {
      return that;
    }

    obj.copy(that, 0, 0, len);
    return that;
  }

  if (obj) {
    if (typeof ArrayBuffer !== 'undefined' && obj.buffer instanceof ArrayBuffer || 'length' in obj) {
      if (typeof obj.length !== 'number' || isnan(obj.length)) {
        return createBuffer(that, 0);
      }

      return fromArrayLike(that, obj);
    }

    if (obj.type === 'Buffer' && isArray(obj.data)) {
      return fromArrayLike(that, obj.data);
    }
  }

  throw new TypeError('First argument must be a string, Buffer, ArrayBuffer, Array, or array-like object.');
}

function checked(length) {
  // Note: cannot use `length < kMaxLength()` here because that fails when
  // length is NaN (which is otherwise coerced to zero.)
  if (length >= kMaxLength()) {
    throw new RangeError('Attempt to allocate Buffer larger than maximum ' + 'size: 0x' + kMaxLength().toString(16) + ' bytes');
  }

  return length | 0;
}

function SlowBuffer(length) {
  if (+length != length) {
    // eslint-disable-line eqeqeq
    length = 0;
  }

  return Buffer.alloc(+length);
}
Buffer.isBuffer = isBuffer;

function internalIsBuffer(b) {
  return !!(b != null && b._isBuffer);
}

Buffer.compare = function compare(a, b) {
  if (!internalIsBuffer(a) || !internalIsBuffer(b)) {
    throw new TypeError('Arguments must be Buffers');
  }

  if (a === b) return 0;
  var x = a.length;
  var y = b.length;

  for (var i = 0, len = Math.min(x, y); i < len; ++i) {
    if (a[i] !== b[i]) {
      x = a[i];
      y = b[i];
      break;
    }
  }

  if (x < y) return -1;
  if (y < x) return 1;
  return 0;
};

Buffer.isEncoding = function isEncoding(encoding) {
  switch (String(encoding).toLowerCase()) {
    case 'hex':
    case 'utf8':
    case 'utf-8':
    case 'ascii':
    case 'latin1':
    case 'binary':
    case 'base64':
    case 'ucs2':
    case 'ucs-2':
    case 'utf16le':
    case 'utf-16le':
      return true;

    default:
      return false;
  }
};

Buffer.concat = function concat(list, length) {
  if (!isArray(list)) {
    throw new TypeError('"list" argument must be an Array of Buffers');
  }

  if (list.length === 0) {
    return Buffer.alloc(0);
  }

  var i;

  if (length === undefined) {
    length = 0;

    for (i = 0; i < list.length; ++i) {
      length += list[i].length;
    }
  }

  var buffer = Buffer.allocUnsafe(length);
  var pos = 0;

  for (i = 0; i < list.length; ++i) {
    var buf = list[i];

    if (!internalIsBuffer(buf)) {
      throw new TypeError('"list" argument must be an Array of Buffers');
    }

    buf.copy(buffer, pos);
    pos += buf.length;
  }

  return buffer;
};

function byteLength(string, encoding) {
  if (internalIsBuffer(string)) {
    return string.length;
  }

  if (typeof ArrayBuffer !== 'undefined' && typeof ArrayBuffer.isView === 'function' && (ArrayBuffer.isView(string) || string instanceof ArrayBuffer)) {
    return string.byteLength;
  }

  if (typeof string !== 'string') {
    string = '' + string;
  }

  var len = string.length;
  if (len === 0) return 0; // Use a for loop to avoid recursion

  var loweredCase = false;

  for (;;) {
    switch (encoding) {
      case 'ascii':
      case 'latin1':
      case 'binary':
        return len;

      case 'utf8':
      case 'utf-8':
      case undefined:
        return utf8ToBytes(string).length;

      case 'ucs2':
      case 'ucs-2':
      case 'utf16le':
      case 'utf-16le':
        return len * 2;

      case 'hex':
        return len >>> 1;

      case 'base64':
        return base64ToBytes(string).length;

      default:
        if (loweredCase) return utf8ToBytes(string).length; // assume utf8

        encoding = ('' + encoding).toLowerCase();
        loweredCase = true;
    }
  }
}

Buffer.byteLength = byteLength;

function slowToString(encoding, start, end) {
  var loweredCase = false; // No need to verify that "this.length <= MAX_UINT32" since it's a read-only
  // property of a typed array.
  // This behaves neither like String nor Uint8Array in that we set start/end
  // to their upper/lower bounds if the value passed is out of range.
  // undefined is handled specially as per ECMA-262 6th Edition,
  // Section 13.3.3.7 Runtime Semantics: KeyedBindingInitialization.

  if (start === undefined || start < 0) {
    start = 0;
  } // Return early if start > this.length. Done here to prevent potential uint32
  // coercion fail below.


  if (start > this.length) {
    return '';
  }

  if (end === undefined || end > this.length) {
    end = this.length;
  }

  if (end <= 0) {
    return '';
  } // Force coersion to uint32. This will also coerce falsey/NaN values to 0.


  end >>>= 0;
  start >>>= 0;

  if (end <= start) {
    return '';
  }

  if (!encoding) encoding = 'utf8';

  while (true) {
    switch (encoding) {
      case 'hex':
        return hexSlice(this, start, end);

      case 'utf8':
      case 'utf-8':
        return utf8Slice(this, start, end);

      case 'ascii':
        return asciiSlice(this, start, end);

      case 'latin1':
      case 'binary':
        return latin1Slice(this, start, end);

      case 'base64':
        return base64Slice(this, start, end);

      case 'ucs2':
      case 'ucs-2':
      case 'utf16le':
      case 'utf-16le':
        return utf16leSlice(this, start, end);

      default:
        if (loweredCase) throw new TypeError('Unknown encoding: ' + encoding);
        encoding = (encoding + '').toLowerCase();
        loweredCase = true;
    }
  }
} // The property is used by `Buffer.isBuffer` and `is-buffer` (in Safari 5-7) to detect
// Buffer instances.


Buffer.prototype._isBuffer = true;

function swap(b, n, m) {
  var i = b[n];
  b[n] = b[m];
  b[m] = i;
}

Buffer.prototype.swap16 = function swap16() {
  var len = this.length;

  if (len % 2 !== 0) {
    throw new RangeError('Buffer size must be a multiple of 16-bits');
  }

  for (var i = 0; i < len; i += 2) {
    swap(this, i, i + 1);
  }

  return this;
};

Buffer.prototype.swap32 = function swap32() {
  var len = this.length;

  if (len % 4 !== 0) {
    throw new RangeError('Buffer size must be a multiple of 32-bits');
  }

  for (var i = 0; i < len; i += 4) {
    swap(this, i, i + 3);
    swap(this, i + 1, i + 2);
  }

  return this;
};

Buffer.prototype.swap64 = function swap64() {
  var len = this.length;

  if (len % 8 !== 0) {
    throw new RangeError('Buffer size must be a multiple of 64-bits');
  }

  for (var i = 0; i < len; i += 8) {
    swap(this, i, i + 7);
    swap(this, i + 1, i + 6);
    swap(this, i + 2, i + 5);
    swap(this, i + 3, i + 4);
  }

  return this;
};

Buffer.prototype.toString = function toString() {
  var length = this.length | 0;
  if (length === 0) return '';
  if (arguments.length === 0) return utf8Slice(this, 0, length);
  return slowToString.apply(this, arguments);
};

Buffer.prototype.equals = function equals(b) {
  if (!internalIsBuffer(b)) throw new TypeError('Argument must be a Buffer');
  if (this === b) return true;
  return Buffer.compare(this, b) === 0;
};

Buffer.prototype.inspect = function inspect() {
  var str = '';
  var max = INSPECT_MAX_BYTES;

  if (this.length > 0) {
    str = this.toString('hex', 0, max).match(/.{2}/g).join(' ');
    if (this.length > max) str += ' ... ';
  }

  return '<Buffer ' + str + '>';
};

Buffer.prototype.compare = function compare(target, start, end, thisStart, thisEnd) {
  if (!internalIsBuffer(target)) {
    throw new TypeError('Argument must be a Buffer');
  }

  if (start === undefined) {
    start = 0;
  }

  if (end === undefined) {
    end = target ? target.length : 0;
  }

  if (thisStart === undefined) {
    thisStart = 0;
  }

  if (thisEnd === undefined) {
    thisEnd = this.length;
  }

  if (start < 0 || end > target.length || thisStart < 0 || thisEnd > this.length) {
    throw new RangeError('out of range index');
  }

  if (thisStart >= thisEnd && start >= end) {
    return 0;
  }

  if (thisStart >= thisEnd) {
    return -1;
  }

  if (start >= end) {
    return 1;
  }

  start >>>= 0;
  end >>>= 0;
  thisStart >>>= 0;
  thisEnd >>>= 0;
  if (this === target) return 0;
  var x = thisEnd - thisStart;
  var y = end - start;
  var len = Math.min(x, y);
  var thisCopy = this.slice(thisStart, thisEnd);
  var targetCopy = target.slice(start, end);

  for (var i = 0; i < len; ++i) {
    if (thisCopy[i] !== targetCopy[i]) {
      x = thisCopy[i];
      y = targetCopy[i];
      break;
    }
  }

  if (x < y) return -1;
  if (y < x) return 1;
  return 0;
}; // Finds either the first index of `val` in `buffer` at offset >= `byteOffset`,
// OR the last index of `val` in `buffer` at offset <= `byteOffset`.
//
// Arguments:
// - buffer - a Buffer to search
// - val - a string, Buffer, or number
// - byteOffset - an index into `buffer`; will be clamped to an int32
// - encoding - an optional encoding, relevant is val is a string
// - dir - true for indexOf, false for lastIndexOf


function bidirectionalIndexOf(buffer, val, byteOffset, encoding, dir) {
  // Empty buffer means no match
  if (buffer.length === 0) return -1; // Normalize byteOffset

  if (typeof byteOffset === 'string') {
    encoding = byteOffset;
    byteOffset = 0;
  } else if (byteOffset > 0x7fffffff) {
    byteOffset = 0x7fffffff;
  } else if (byteOffset < -0x80000000) {
    byteOffset = -0x80000000;
  }

  byteOffset = +byteOffset; // Coerce to Number.

  if (isNaN(byteOffset)) {
    // byteOffset: it it's undefined, null, NaN, "foo", etc, search whole buffer
    byteOffset = dir ? 0 : buffer.length - 1;
  } // Normalize byteOffset: negative offsets start from the end of the buffer


  if (byteOffset < 0) byteOffset = buffer.length + byteOffset;

  if (byteOffset >= buffer.length) {
    if (dir) return -1;else byteOffset = buffer.length - 1;
  } else if (byteOffset < 0) {
    if (dir) byteOffset = 0;else return -1;
  } // Normalize val


  if (typeof val === 'string') {
    val = Buffer.from(val, encoding);
  } // Finally, search either indexOf (if dir is true) or lastIndexOf


  if (internalIsBuffer(val)) {
    // Special case: looking for empty string/buffer always fails
    if (val.length === 0) {
      return -1;
    }

    return arrayIndexOf(buffer, val, byteOffset, encoding, dir);
  } else if (typeof val === 'number') {
    val = val & 0xFF; // Search for a byte value [0-255]

    if (Buffer.TYPED_ARRAY_SUPPORT && typeof Uint8Array.prototype.indexOf === 'function') {
      if (dir) {
        return Uint8Array.prototype.indexOf.call(buffer, val, byteOffset);
      } else {
        return Uint8Array.prototype.lastIndexOf.call(buffer, val, byteOffset);
      }
    }

    return arrayIndexOf(buffer, [val], byteOffset, encoding, dir);
  }

  throw new TypeError('val must be string, number or Buffer');
}

function arrayIndexOf(arr, val, byteOffset, encoding, dir) {
  var indexSize = 1;
  var arrLength = arr.length;
  var valLength = val.length;

  if (encoding !== undefined) {
    encoding = String(encoding).toLowerCase();

    if (encoding === 'ucs2' || encoding === 'ucs-2' || encoding === 'utf16le' || encoding === 'utf-16le') {
      if (arr.length < 2 || val.length < 2) {
        return -1;
      }

      indexSize = 2;
      arrLength /= 2;
      valLength /= 2;
      byteOffset /= 2;
    }
  }

  function read$$1(buf, i) {
    if (indexSize === 1) {
      return buf[i];
    } else {
      return buf.readUInt16BE(i * indexSize);
    }
  }

  var i;

  if (dir) {
    var foundIndex = -1;

    for (i = byteOffset; i < arrLength; i++) {
      if (read$$1(arr, i) === read$$1(val, foundIndex === -1 ? 0 : i - foundIndex)) {
        if (foundIndex === -1) foundIndex = i;
        if (i - foundIndex + 1 === valLength) return foundIndex * indexSize;
      } else {
        if (foundIndex !== -1) i -= i - foundIndex;
        foundIndex = -1;
      }
    }
  } else {
    if (byteOffset + valLength > arrLength) byteOffset = arrLength - valLength;

    for (i = byteOffset; i >= 0; i--) {
      var found = true;

      for (var j = 0; j < valLength; j++) {
        if (read$$1(arr, i + j) !== read$$1(val, j)) {
          found = false;
          break;
        }
      }

      if (found) return i;
    }
  }

  return -1;
}

Buffer.prototype.includes = function includes(val, byteOffset, encoding) {
  return this.indexOf(val, byteOffset, encoding) !== -1;
};

Buffer.prototype.indexOf = function indexOf(val, byteOffset, encoding) {
  return bidirectionalIndexOf(this, val, byteOffset, encoding, true);
};

Buffer.prototype.lastIndexOf = function lastIndexOf(val, byteOffset, encoding) {
  return bidirectionalIndexOf(this, val, byteOffset, encoding, false);
};

function hexWrite(buf, string, offset, length) {
  offset = Number(offset) || 0;
  var remaining = buf.length - offset;

  if (!length) {
    length = remaining;
  } else {
    length = Number(length);

    if (length > remaining) {
      length = remaining;
    }
  } // must be an even number of digits


  var strLen = string.length;
  if (strLen % 2 !== 0) throw new TypeError('Invalid hex string');

  if (length > strLen / 2) {
    length = strLen / 2;
  }

  for (var i = 0; i < length; ++i) {
    var parsed = parseInt(string.substr(i * 2, 2), 16);
    if (isNaN(parsed)) return i;
    buf[offset + i] = parsed;
  }

  return i;
}

function utf8Write(buf, string, offset, length) {
  return blitBuffer(utf8ToBytes(string, buf.length - offset), buf, offset, length);
}

function asciiWrite(buf, string, offset, length) {
  return blitBuffer(asciiToBytes(string), buf, offset, length);
}

function latin1Write(buf, string, offset, length) {
  return asciiWrite(buf, string, offset, length);
}

function base64Write(buf, string, offset, length) {
  return blitBuffer(base64ToBytes(string), buf, offset, length);
}

function ucs2Write(buf, string, offset, length) {
  return blitBuffer(utf16leToBytes(string, buf.length - offset), buf, offset, length);
}

Buffer.prototype.write = function write$$1(string, offset, length, encoding) {
  // Buffer#write(string)
  if (offset === undefined) {
    encoding = 'utf8';
    length = this.length;
    offset = 0; // Buffer#write(string, encoding)
  } else if (length === undefined && typeof offset === 'string') {
    encoding = offset;
    length = this.length;
    offset = 0; // Buffer#write(string, offset[, length][, encoding])
  } else if (isFinite(offset)) {
    offset = offset | 0;

    if (isFinite(length)) {
      length = length | 0;
      if (encoding === undefined) encoding = 'utf8';
    } else {
      encoding = length;
      length = undefined;
    } // legacy write(string, encoding, offset, length) - remove in v0.13

  } else {
    throw new Error('Buffer.write(string, encoding, offset[, length]) is no longer supported');
  }

  var remaining = this.length - offset;
  if (length === undefined || length > remaining) length = remaining;

  if (string.length > 0 && (length < 0 || offset < 0) || offset > this.length) {
    throw new RangeError('Attempt to write outside buffer bounds');
  }

  if (!encoding) encoding = 'utf8';
  var loweredCase = false;

  for (;;) {
    switch (encoding) {
      case 'hex':
        return hexWrite(this, string, offset, length);

      case 'utf8':
      case 'utf-8':
        return utf8Write(this, string, offset, length);

      case 'ascii':
        return asciiWrite(this, string, offset, length);

      case 'latin1':
      case 'binary':
        return latin1Write(this, string, offset, length);

      case 'base64':
        // Warning: maxLength not taken into account in base64Write
        return base64Write(this, string, offset, length);

      case 'ucs2':
      case 'ucs-2':
      case 'utf16le':
      case 'utf-16le':
        return ucs2Write(this, string, offset, length);

      default:
        if (loweredCase) throw new TypeError('Unknown encoding: ' + encoding);
        encoding = ('' + encoding).toLowerCase();
        loweredCase = true;
    }
  }
};

Buffer.prototype.toJSON = function toJSON() {
  return {
    type: 'Buffer',
    data: Array.prototype.slice.call(this._arr || this, 0)
  };
};

function base64Slice(buf, start, end) {
  if (start === 0 && end === buf.length) {
    return fromByteArray(buf);
  } else {
    return fromByteArray(buf.slice(start, end));
  }
}

function utf8Slice(buf, start, end) {
  end = Math.min(buf.length, end);
  var res = [];
  var i = start;

  while (i < end) {
    var firstByte = buf[i];
    var codePoint = null;
    var bytesPerSequence = firstByte > 0xEF ? 4 : firstByte > 0xDF ? 3 : firstByte > 0xBF ? 2 : 1;

    if (i + bytesPerSequence <= end) {
      var secondByte, thirdByte, fourthByte, tempCodePoint;

      switch (bytesPerSequence) {
        case 1:
          if (firstByte < 0x80) {
            codePoint = firstByte;
          }

          break;

        case 2:
          secondByte = buf[i + 1];

          if ((secondByte & 0xC0) === 0x80) {
            tempCodePoint = (firstByte & 0x1F) << 0x6 | secondByte & 0x3F;

            if (tempCodePoint > 0x7F) {
              codePoint = tempCodePoint;
            }
          }

          break;

        case 3:
          secondByte = buf[i + 1];
          thirdByte = buf[i + 2];

          if ((secondByte & 0xC0) === 0x80 && (thirdByte & 0xC0) === 0x80) {
            tempCodePoint = (firstByte & 0xF) << 0xC | (secondByte & 0x3F) << 0x6 | thirdByte & 0x3F;

            if (tempCodePoint > 0x7FF && (tempCodePoint < 0xD800 || tempCodePoint > 0xDFFF)) {
              codePoint = tempCodePoint;
            }
          }

          break;

        case 4:
          secondByte = buf[i + 1];
          thirdByte = buf[i + 2];
          fourthByte = buf[i + 3];

          if ((secondByte & 0xC0) === 0x80 && (thirdByte & 0xC0) === 0x80 && (fourthByte & 0xC0) === 0x80) {
            tempCodePoint = (firstByte & 0xF) << 0x12 | (secondByte & 0x3F) << 0xC | (thirdByte & 0x3F) << 0x6 | fourthByte & 0x3F;

            if (tempCodePoint > 0xFFFF && tempCodePoint < 0x110000) {
              codePoint = tempCodePoint;
            }
          }

      }
    }

    if (codePoint === null) {
      // we did not generate a valid codePoint so insert a
      // replacement char (U+FFFD) and advance only 1 byte
      codePoint = 0xFFFD;
      bytesPerSequence = 1;
    } else if (codePoint > 0xFFFF) {
      // encode to utf16 (surrogate pair dance)
      codePoint -= 0x10000;
      res.push(codePoint >>> 10 & 0x3FF | 0xD800);
      codePoint = 0xDC00 | codePoint & 0x3FF;
    }

    res.push(codePoint);
    i += bytesPerSequence;
  }

  return decodeCodePointsArray(res);
} // Based on http://stackoverflow.com/a/22747272/680742, the browser with
// the lowest limit is Chrome, with 0x10000 args.
// We go 1 magnitude less, for safety


var MAX_ARGUMENTS_LENGTH = 0x1000;

function decodeCodePointsArray(codePoints) {
  var len = codePoints.length;

  if (len <= MAX_ARGUMENTS_LENGTH) {
    return String.fromCharCode.apply(String, codePoints); // avoid extra slice()
  } // Decode in chunks to avoid "call stack size exceeded".


  var res = '';
  var i = 0;

  while (i < len) {
    res += String.fromCharCode.apply(String, codePoints.slice(i, i += MAX_ARGUMENTS_LENGTH));
  }

  return res;
}

function asciiSlice(buf, start, end) {
  var ret = '';
  end = Math.min(buf.length, end);

  for (var i = start; i < end; ++i) {
    ret += String.fromCharCode(buf[i] & 0x7F);
  }

  return ret;
}

function latin1Slice(buf, start, end) {
  var ret = '';
  end = Math.min(buf.length, end);

  for (var i = start; i < end; ++i) {
    ret += String.fromCharCode(buf[i]);
  }

  return ret;
}

function hexSlice(buf, start, end) {
  var len = buf.length;
  if (!start || start < 0) start = 0;
  if (!end || end < 0 || end > len) end = len;
  var out = '';

  for (var i = start; i < end; ++i) {
    out += toHex(buf[i]);
  }

  return out;
}

function utf16leSlice(buf, start, end) {
  var bytes = buf.slice(start, end);
  var res = '';

  for (var i = 0; i < bytes.length; i += 2) {
    res += String.fromCharCode(bytes[i] + bytes[i + 1] * 256);
  }

  return res;
}

Buffer.prototype.slice = function slice(start, end) {
  var len = this.length;
  start = ~~start;
  end = end === undefined ? len : ~~end;

  if (start < 0) {
    start += len;
    if (start < 0) start = 0;
  } else if (start > len) {
    start = len;
  }

  if (end < 0) {
    end += len;
    if (end < 0) end = 0;
  } else if (end > len) {
    end = len;
  }

  if (end < start) end = start;
  var newBuf;

  if (Buffer.TYPED_ARRAY_SUPPORT) {
    newBuf = this.subarray(start, end);
    newBuf.__proto__ = Buffer.prototype;
  } else {
    var sliceLen = end - start;
    newBuf = new Buffer(sliceLen, undefined);

    for (var i = 0; i < sliceLen; ++i) {
      newBuf[i] = this[i + start];
    }
  }

  return newBuf;
};
/*
 * Need to make sure that buffer isn't trying to write out of bounds.
 */


function checkOffset(offset, ext, length) {
  if (offset % 1 !== 0 || offset < 0) throw new RangeError('offset is not uint');
  if (offset + ext > length) throw new RangeError('Trying to access beyond buffer length');
}

Buffer.prototype.readUIntLE = function readUIntLE(offset, byteLength, noAssert) {
  offset = offset | 0;
  byteLength = byteLength | 0;
  if (!noAssert) checkOffset(offset, byteLength, this.length);
  var val = this[offset];
  var mul = 1;
  var i = 0;

  while (++i < byteLength && (mul *= 0x100)) {
    val += this[offset + i] * mul;
  }

  return val;
};

Buffer.prototype.readUIntBE = function readUIntBE(offset, byteLength, noAssert) {
  offset = offset | 0;
  byteLength = byteLength | 0;

  if (!noAssert) {
    checkOffset(offset, byteLength, this.length);
  }

  var val = this[offset + --byteLength];
  var mul = 1;

  while (byteLength > 0 && (mul *= 0x100)) {
    val += this[offset + --byteLength] * mul;
  }

  return val;
};

Buffer.prototype.readUInt8 = function readUInt8(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 1, this.length);
  return this[offset];
};

Buffer.prototype.readUInt16LE = function readUInt16LE(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 2, this.length);
  return this[offset] | this[offset + 1] << 8;
};

Buffer.prototype.readUInt16BE = function readUInt16BE(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 2, this.length);
  return this[offset] << 8 | this[offset + 1];
};

Buffer.prototype.readUInt32LE = function readUInt32LE(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 4, this.length);
  return (this[offset] | this[offset + 1] << 8 | this[offset + 2] << 16) + this[offset + 3] * 0x1000000;
};

Buffer.prototype.readUInt32BE = function readUInt32BE(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 4, this.length);
  return this[offset] * 0x1000000 + (this[offset + 1] << 16 | this[offset + 2] << 8 | this[offset + 3]);
};

Buffer.prototype.readIntLE = function readIntLE(offset, byteLength, noAssert) {
  offset = offset | 0;
  byteLength = byteLength | 0;
  if (!noAssert) checkOffset(offset, byteLength, this.length);
  var val = this[offset];
  var mul = 1;
  var i = 0;

  while (++i < byteLength && (mul *= 0x100)) {
    val += this[offset + i] * mul;
  }

  mul *= 0x80;
  if (val >= mul) val -= Math.pow(2, 8 * byteLength);
  return val;
};

Buffer.prototype.readIntBE = function readIntBE(offset, byteLength, noAssert) {
  offset = offset | 0;
  byteLength = byteLength | 0;
  if (!noAssert) checkOffset(offset, byteLength, this.length);
  var i = byteLength;
  var mul = 1;
  var val = this[offset + --i];

  while (i > 0 && (mul *= 0x100)) {
    val += this[offset + --i] * mul;
  }

  mul *= 0x80;
  if (val >= mul) val -= Math.pow(2, 8 * byteLength);
  return val;
};

Buffer.prototype.readInt8 = function readInt8(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 1, this.length);
  if (!(this[offset] & 0x80)) return this[offset];
  return (0xff - this[offset] + 1) * -1;
};

Buffer.prototype.readInt16LE = function readInt16LE(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 2, this.length);
  var val = this[offset] | this[offset + 1] << 8;
  return val & 0x8000 ? val | 0xFFFF0000 : val;
};

Buffer.prototype.readInt16BE = function readInt16BE(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 2, this.length);
  var val = this[offset + 1] | this[offset] << 8;
  return val & 0x8000 ? val | 0xFFFF0000 : val;
};

Buffer.prototype.readInt32LE = function readInt32LE(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 4, this.length);
  return this[offset] | this[offset + 1] << 8 | this[offset + 2] << 16 | this[offset + 3] << 24;
};

Buffer.prototype.readInt32BE = function readInt32BE(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 4, this.length);
  return this[offset] << 24 | this[offset + 1] << 16 | this[offset + 2] << 8 | this[offset + 3];
};

Buffer.prototype.readFloatLE = function readFloatLE(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 4, this.length);
  return read(this, offset, true, 23, 4);
};

Buffer.prototype.readFloatBE = function readFloatBE(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 4, this.length);
  return read(this, offset, false, 23, 4);
};

Buffer.prototype.readDoubleLE = function readDoubleLE(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 8, this.length);
  return read(this, offset, true, 52, 8);
};

Buffer.prototype.readDoubleBE = function readDoubleBE(offset, noAssert) {
  if (!noAssert) checkOffset(offset, 8, this.length);
  return read(this, offset, false, 52, 8);
};

function checkInt(buf, value, offset, ext, max, min) {
  if (!internalIsBuffer(buf)) throw new TypeError('"buffer" argument must be a Buffer instance');
  if (value > max || value < min) throw new RangeError('"value" argument is out of bounds');
  if (offset + ext > buf.length) throw new RangeError('Index out of range');
}

Buffer.prototype.writeUIntLE = function writeUIntLE(value, offset, byteLength, noAssert) {
  value = +value;
  offset = offset | 0;
  byteLength = byteLength | 0;

  if (!noAssert) {
    var maxBytes = Math.pow(2, 8 * byteLength) - 1;
    checkInt(this, value, offset, byteLength, maxBytes, 0);
  }

  var mul = 1;
  var i = 0;
  this[offset] = value & 0xFF;

  while (++i < byteLength && (mul *= 0x100)) {
    this[offset + i] = value / mul & 0xFF;
  }

  return offset + byteLength;
};

Buffer.prototype.writeUIntBE = function writeUIntBE(value, offset, byteLength, noAssert) {
  value = +value;
  offset = offset | 0;
  byteLength = byteLength | 0;

  if (!noAssert) {
    var maxBytes = Math.pow(2, 8 * byteLength) - 1;
    checkInt(this, value, offset, byteLength, maxBytes, 0);
  }

  var i = byteLength - 1;
  var mul = 1;
  this[offset + i] = value & 0xFF;

  while (--i >= 0 && (mul *= 0x100)) {
    this[offset + i] = value / mul & 0xFF;
  }

  return offset + byteLength;
};

Buffer.prototype.writeUInt8 = function writeUInt8(value, offset, noAssert) {
  value = +value;
  offset = offset | 0;
  if (!noAssert) checkInt(this, value, offset, 1, 0xff, 0);
  if (!Buffer.TYPED_ARRAY_SUPPORT) value = Math.floor(value);
  this[offset] = value & 0xff;
  return offset + 1;
};

function objectWriteUInt16(buf, value, offset, littleEndian) {
  if (value < 0) value = 0xffff + value + 1;

  for (var i = 0, j = Math.min(buf.length - offset, 2); i < j; ++i) {
    buf[offset + i] = (value & 0xff << 8 * (littleEndian ? i : 1 - i)) >>> (littleEndian ? i : 1 - i) * 8;
  }
}

Buffer.prototype.writeUInt16LE = function writeUInt16LE(value, offset, noAssert) {
  value = +value;
  offset = offset | 0;
  if (!noAssert) checkInt(this, value, offset, 2, 0xffff, 0);

  if (Buffer.TYPED_ARRAY_SUPPORT) {
    this[offset] = value & 0xff;
    this[offset + 1] = value >>> 8;
  } else {
    objectWriteUInt16(this, value, offset, true);
  }

  return offset + 2;
};

Buffer.prototype.writeUInt16BE = function writeUInt16BE(value, offset, noAssert) {
  value = +value;
  offset = offset | 0;
  if (!noAssert) checkInt(this, value, offset, 2, 0xffff, 0);

  if (Buffer.TYPED_ARRAY_SUPPORT) {
    this[offset] = value >>> 8;
    this[offset + 1] = value & 0xff;
  } else {
    objectWriteUInt16(this, value, offset, false);
  }

  return offset + 2;
};

function objectWriteUInt32(buf, value, offset, littleEndian) {
  if (value < 0) value = 0xffffffff + value + 1;

  for (var i = 0, j = Math.min(buf.length - offset, 4); i < j; ++i) {
    buf[offset + i] = value >>> (littleEndian ? i : 3 - i) * 8 & 0xff;
  }
}

Buffer.prototype.writeUInt32LE = function writeUInt32LE(value, offset, noAssert) {
  value = +value;
  offset = offset | 0;
  if (!noAssert) checkInt(this, value, offset, 4, 0xffffffff, 0);

  if (Buffer.TYPED_ARRAY_SUPPORT) {
    this[offset + 3] = value >>> 24;
    this[offset + 2] = value >>> 16;
    this[offset + 1] = value >>> 8;
    this[offset] = value & 0xff;
  } else {
    objectWriteUInt32(this, value, offset, true);
  }

  return offset + 4;
};

Buffer.prototype.writeUInt32BE = function writeUInt32BE(value, offset, noAssert) {
  value = +value;
  offset = offset | 0;
  if (!noAssert) checkInt(this, value, offset, 4, 0xffffffff, 0);

  if (Buffer.TYPED_ARRAY_SUPPORT) {
    this[offset] = value >>> 24;
    this[offset + 1] = value >>> 16;
    this[offset + 2] = value >>> 8;
    this[offset + 3] = value & 0xff;
  } else {
    objectWriteUInt32(this, value, offset, false);
  }

  return offset + 4;
};

Buffer.prototype.writeIntLE = function writeIntLE(value, offset, byteLength, noAssert) {
  value = +value;
  offset = offset | 0;

  if (!noAssert) {
    var limit = Math.pow(2, 8 * byteLength - 1);
    checkInt(this, value, offset, byteLength, limit - 1, -limit);
  }

  var i = 0;
  var mul = 1;
  var sub = 0;
  this[offset] = value & 0xFF;

  while (++i < byteLength && (mul *= 0x100)) {
    if (value < 0 && sub === 0 && this[offset + i - 1] !== 0) {
      sub = 1;
    }

    this[offset + i] = (value / mul >> 0) - sub & 0xFF;
  }

  return offset + byteLength;
};

Buffer.prototype.writeIntBE = function writeIntBE(value, offset, byteLength, noAssert) {
  value = +value;
  offset = offset | 0;

  if (!noAssert) {
    var limit = Math.pow(2, 8 * byteLength - 1);
    checkInt(this, value, offset, byteLength, limit - 1, -limit);
  }

  var i = byteLength - 1;
  var mul = 1;
  var sub = 0;
  this[offset + i] = value & 0xFF;

  while (--i >= 0 && (mul *= 0x100)) {
    if (value < 0 && sub === 0 && this[offset + i + 1] !== 0) {
      sub = 1;
    }

    this[offset + i] = (value / mul >> 0) - sub & 0xFF;
  }

  return offset + byteLength;
};

Buffer.prototype.writeInt8 = function writeInt8(value, offset, noAssert) {
  value = +value;
  offset = offset | 0;
  if (!noAssert) checkInt(this, value, offset, 1, 0x7f, -0x80);
  if (!Buffer.TYPED_ARRAY_SUPPORT) value = Math.floor(value);
  if (value < 0) value = 0xff + value + 1;
  this[offset] = value & 0xff;
  return offset + 1;
};

Buffer.prototype.writeInt16LE = function writeInt16LE(value, offset, noAssert) {
  value = +value;
  offset = offset | 0;
  if (!noAssert) checkInt(this, value, offset, 2, 0x7fff, -0x8000);

  if (Buffer.TYPED_ARRAY_SUPPORT) {
    this[offset] = value & 0xff;
    this[offset + 1] = value >>> 8;
  } else {
    objectWriteUInt16(this, value, offset, true);
  }

  return offset + 2;
};

Buffer.prototype.writeInt16BE = function writeInt16BE(value, offset, noAssert) {
  value = +value;
  offset = offset | 0;
  if (!noAssert) checkInt(this, value, offset, 2, 0x7fff, -0x8000);

  if (Buffer.TYPED_ARRAY_SUPPORT) {
    this[offset] = value >>> 8;
    this[offset + 1] = value & 0xff;
  } else {
    objectWriteUInt16(this, value, offset, false);
  }

  return offset + 2;
};

Buffer.prototype.writeInt32LE = function writeInt32LE(value, offset, noAssert) {
  value = +value;
  offset = offset | 0;
  if (!noAssert) checkInt(this, value, offset, 4, 0x7fffffff, -0x80000000);

  if (Buffer.TYPED_ARRAY_SUPPORT) {
    this[offset] = value & 0xff;
    this[offset + 1] = value >>> 8;
    this[offset + 2] = value >>> 16;
    this[offset + 3] = value >>> 24;
  } else {
    objectWriteUInt32(this, value, offset, true);
  }

  return offset + 4;
};

Buffer.prototype.writeInt32BE = function writeInt32BE(value, offset, noAssert) {
  value = +value;
  offset = offset | 0;
  if (!noAssert) checkInt(this, value, offset, 4, 0x7fffffff, -0x80000000);
  if (value < 0) value = 0xffffffff + value + 1;

  if (Buffer.TYPED_ARRAY_SUPPORT) {
    this[offset] = value >>> 24;
    this[offset + 1] = value >>> 16;
    this[offset + 2] = value >>> 8;
    this[offset + 3] = value & 0xff;
  } else {
    objectWriteUInt32(this, value, offset, false);
  }

  return offset + 4;
};

function checkIEEE754(buf, value, offset, ext, max, min) {
  if (offset + ext > buf.length) throw new RangeError('Index out of range');
  if (offset < 0) throw new RangeError('Index out of range');
}

function writeFloat(buf, value, offset, littleEndian, noAssert) {
  if (!noAssert) {
    checkIEEE754(buf, value, offset, 4, 3.4028234663852886e+38, -3.4028234663852886e+38);
  }

  write(buf, value, offset, littleEndian, 23, 4);
  return offset + 4;
}

Buffer.prototype.writeFloatLE = function writeFloatLE(value, offset, noAssert) {
  return writeFloat(this, value, offset, true, noAssert);
};

Buffer.prototype.writeFloatBE = function writeFloatBE(value, offset, noAssert) {
  return writeFloat(this, value, offset, false, noAssert);
};

function writeDouble(buf, value, offset, littleEndian, noAssert) {
  if (!noAssert) {
    checkIEEE754(buf, value, offset, 8, 1.7976931348623157E+308, -1.7976931348623157E+308);
  }

  write(buf, value, offset, littleEndian, 52, 8);
  return offset + 8;
}

Buffer.prototype.writeDoubleLE = function writeDoubleLE(value, offset, noAssert) {
  return writeDouble(this, value, offset, true, noAssert);
};

Buffer.prototype.writeDoubleBE = function writeDoubleBE(value, offset, noAssert) {
  return writeDouble(this, value, offset, false, noAssert);
}; // copy(targetBuffer, targetStart=0, sourceStart=0, sourceEnd=buffer.length)


Buffer.prototype.copy = function copy(target, targetStart, start, end) {
  if (!start) start = 0;
  if (!end && end !== 0) end = this.length;
  if (targetStart >= target.length) targetStart = target.length;
  if (!targetStart) targetStart = 0;
  if (end > 0 && end < start) end = start; // Copy 0 bytes; we're done

  if (end === start) return 0;
  if (target.length === 0 || this.length === 0) return 0; // Fatal error conditions

  if (targetStart < 0) {
    throw new RangeError('targetStart out of bounds');
  }

  if (start < 0 || start >= this.length) throw new RangeError('sourceStart out of bounds');
  if (end < 0) throw new RangeError('sourceEnd out of bounds'); // Are we oob?

  if (end > this.length) end = this.length;

  if (target.length - targetStart < end - start) {
    end = target.length - targetStart + start;
  }

  var len = end - start;
  var i;

  if (this === target && start < targetStart && targetStart < end) {
    // descending copy from end
    for (i = len - 1; i >= 0; --i) {
      target[i + targetStart] = this[i + start];
    }
  } else if (len < 1000 || !Buffer.TYPED_ARRAY_SUPPORT) {
    // ascending copy from start
    for (i = 0; i < len; ++i) {
      target[i + targetStart] = this[i + start];
    }
  } else {
    Uint8Array.prototype.set.call(target, this.subarray(start, start + len), targetStart);
  }

  return len;
}; // Usage:
//    buffer.fill(number[, offset[, end]])
//    buffer.fill(buffer[, offset[, end]])
//    buffer.fill(string[, offset[, end]][, encoding])


Buffer.prototype.fill = function fill(val, start, end, encoding) {
  // Handle string cases:
  if (typeof val === 'string') {
    if (typeof start === 'string') {
      encoding = start;
      start = 0;
      end = this.length;
    } else if (typeof end === 'string') {
      encoding = end;
      end = this.length;
    }

    if (val.length === 1) {
      var code = val.charCodeAt(0);

      if (code < 256) {
        val = code;
      }
    }

    if (encoding !== undefined && typeof encoding !== 'string') {
      throw new TypeError('encoding must be a string');
    }

    if (typeof encoding === 'string' && !Buffer.isEncoding(encoding)) {
      throw new TypeError('Unknown encoding: ' + encoding);
    }
  } else if (typeof val === 'number') {
    val = val & 255;
  } // Invalid ranges are not set to a default, so can range check early.


  if (start < 0 || this.length < start || this.length < end) {
    throw new RangeError('Out of range index');
  }

  if (end <= start) {
    return this;
  }

  start = start >>> 0;
  end = end === undefined ? this.length : end >>> 0;
  if (!val) val = 0;
  var i;

  if (typeof val === 'number') {
    for (i = start; i < end; ++i) {
      this[i] = val;
    }
  } else {
    var bytes = internalIsBuffer(val) ? val : utf8ToBytes(new Buffer(val, encoding).toString());
    var len = bytes.length;

    for (i = 0; i < end - start; ++i) {
      this[i + start] = bytes[i % len];
    }
  }

  return this;
}; // HELPER FUNCTIONS
// ================


var INVALID_BASE64_RE = /[^+\/0-9A-Za-z-_]/g;

function base64clean(str) {
  // Node strips out invalid characters like \n and \t from the string, base64-js does not
  str = stringtrim(str).replace(INVALID_BASE64_RE, ''); // Node converts strings with length < 2 to ''

  if (str.length < 2) return ''; // Node allows for non-padded base64 strings (missing trailing ===), base64-js does not

  while (str.length % 4 !== 0) {
    str = str + '=';
  }

  return str;
}

function stringtrim(str) {
  if (str.trim) return str.trim();
  return str.replace(/^\s+|\s+$/g, '');
}

function toHex(n) {
  if (n < 16) return '0' + n.toString(16);
  return n.toString(16);
}

function utf8ToBytes(string, units) {
  units = units || Infinity;
  var codePoint;
  var length = string.length;
  var leadSurrogate = null;
  var bytes = [];

  for (var i = 0; i < length; ++i) {
    codePoint = string.charCodeAt(i); // is surrogate component

    if (codePoint > 0xD7FF && codePoint < 0xE000) {
      // last char was a lead
      if (!leadSurrogate) {
        // no lead yet
        if (codePoint > 0xDBFF) {
          // unexpected trail
          if ((units -= 3) > -1) bytes.push(0xEF, 0xBF, 0xBD);
          continue;
        } else if (i + 1 === length) {
          // unpaired lead
          if ((units -= 3) > -1) bytes.push(0xEF, 0xBF, 0xBD);
          continue;
        } // valid lead


        leadSurrogate = codePoint;
        continue;
      } // 2 leads in a row


      if (codePoint < 0xDC00) {
        if ((units -= 3) > -1) bytes.push(0xEF, 0xBF, 0xBD);
        leadSurrogate = codePoint;
        continue;
      } // valid surrogate pair


      codePoint = (leadSurrogate - 0xD800 << 10 | codePoint - 0xDC00) + 0x10000;
    } else if (leadSurrogate) {
      // valid bmp char, but last char was a lead
      if ((units -= 3) > -1) bytes.push(0xEF, 0xBF, 0xBD);
    }

    leadSurrogate = null; // encode utf8

    if (codePoint < 0x80) {
      if ((units -= 1) < 0) break;
      bytes.push(codePoint);
    } else if (codePoint < 0x800) {
      if ((units -= 2) < 0) break;
      bytes.push(codePoint >> 0x6 | 0xC0, codePoint & 0x3F | 0x80);
    } else if (codePoint < 0x10000) {
      if ((units -= 3) < 0) break;
      bytes.push(codePoint >> 0xC | 0xE0, codePoint >> 0x6 & 0x3F | 0x80, codePoint & 0x3F | 0x80);
    } else if (codePoint < 0x110000) {
      if ((units -= 4) < 0) break;
      bytes.push(codePoint >> 0x12 | 0xF0, codePoint >> 0xC & 0x3F | 0x80, codePoint >> 0x6 & 0x3F | 0x80, codePoint & 0x3F | 0x80);
    } else {
      throw new Error('Invalid code point');
    }
  }

  return bytes;
}

function asciiToBytes(str) {
  var byteArray = [];

  for (var i = 0; i < str.length; ++i) {
    // Node's code seems to be doing this and not & 0x7F..
    byteArray.push(str.charCodeAt(i) & 0xFF);
  }

  return byteArray;
}

function utf16leToBytes(str, units) {
  var c, hi, lo;
  var byteArray = [];

  for (var i = 0; i < str.length; ++i) {
    if ((units -= 2) < 0) break;
    c = str.charCodeAt(i);
    hi = c >> 8;
    lo = c % 256;
    byteArray.push(lo);
    byteArray.push(hi);
  }

  return byteArray;
}

function base64ToBytes(str) {
  return toByteArray(base64clean(str));
}

function blitBuffer(src, dst, offset, length) {
  for (var i = 0; i < length; ++i) {
    if (i + offset >= dst.length || i >= src.length) break;
    dst[i + offset] = src[i];
  }

  return i;
}

function isnan(val) {
  return val !== val; // eslint-disable-line no-self-compare
} // the following is from is-buffer, also by Feross Aboukhadijeh and with same lisence
// The _isBuffer check is for Safari 5-7 support, because it's missing
// Object.prototype.constructor. Remove this eventually


function isBuffer(obj) {
  return obj != null && (!!obj._isBuffer || isFastBuffer(obj) || isSlowBuffer(obj));
}

function isFastBuffer(obj) {
  return !!obj.constructor && typeof obj.constructor.isBuffer === 'function' && obj.constructor.isBuffer(obj);
} // For Node v0.10 support. Remove this eventually.


function isSlowBuffer(obj) {
  return typeof obj.readFloatLE === 'function' && typeof obj.slice === 'function' && isFastBuffer(obj.slice(0, 0));
}

var bufferEs6 = /*#__PURE__*/Object.freeze({
            INSPECT_MAX_BYTES: INSPECT_MAX_BYTES,
            kMaxLength: _kMaxLength,
            Buffer: Buffer,
            SlowBuffer: SlowBuffer,
            isBuffer: isBuffer
});

var commonjsGlobal = typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

function unwrapExports (x) {
	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x.default : x;
}

function createCommonjsModule(fn, module) {
	return module = { exports: {} }, fn(module, module.exports), module.exports;
}

function getCjsExportFromNamespace (n) {
	return n && n.default || n;
}

var domain; // This constructor is used to store event handlers. Instantiating this is
// faster than explicitly calling `Object.create(null)` to get a "clean" empty
// object (tested with v8 v4.9).

function EventHandlers() {}

EventHandlers.prototype = Object.create(null);

function EventEmitter() {
  EventEmitter.init.call(this);
}
// require('events') === require('events').EventEmitter

EventEmitter.EventEmitter = EventEmitter;
EventEmitter.usingDomains = false;
EventEmitter.prototype.domain = undefined;
EventEmitter.prototype._events = undefined;
EventEmitter.prototype._maxListeners = undefined; // By default EventEmitters will print a warning if more than 10 listeners are
// added to it. This is a useful default which helps finding memory leaks.

EventEmitter.defaultMaxListeners = 10;

EventEmitter.init = function () {
  this.domain = null;

  if (EventEmitter.usingDomains) {
    // if there is an active domain, then attach to it.
    if (domain.active && !(this instanceof domain.Domain)) ;
  }

  if (!this._events || this._events === Object.getPrototypeOf(this)._events) {
    this._events = new EventHandlers();
    this._eventsCount = 0;
  }

  this._maxListeners = this._maxListeners || undefined;
}; // Obviously not all Emitters should be limited to 10. This function allows
// that to be increased. Set to zero for unlimited.


EventEmitter.prototype.setMaxListeners = function setMaxListeners(n) {
  if (typeof n !== 'number' || n < 0 || isNaN(n)) throw new TypeError('"n" argument must be a positive number');
  this._maxListeners = n;
  return this;
};

function $getMaxListeners(that) {
  if (that._maxListeners === undefined) return EventEmitter.defaultMaxListeners;
  return that._maxListeners;
}

EventEmitter.prototype.getMaxListeners = function getMaxListeners() {
  return $getMaxListeners(this);
}; // These standalone emit* functions are used to optimize calling of event
// handlers for fast cases because emit() itself often has a variable number of
// arguments and can be deoptimized because of that. These functions always have
// the same number of arguments and thus do not get deoptimized, so the code
// inside them can execute faster.


function emitNone(handler, isFn, self) {
  if (isFn) handler.call(self);else {
    var len = handler.length;
    var listeners = arrayClone(handler, len);

    for (var i = 0; i < len; ++i) {
      listeners[i].call(self);
    }
  }
}

function emitOne(handler, isFn, self, arg1) {
  if (isFn) handler.call(self, arg1);else {
    var len = handler.length;
    var listeners = arrayClone(handler, len);

    for (var i = 0; i < len; ++i) {
      listeners[i].call(self, arg1);
    }
  }
}

function emitTwo(handler, isFn, self, arg1, arg2) {
  if (isFn) handler.call(self, arg1, arg2);else {
    var len = handler.length;
    var listeners = arrayClone(handler, len);

    for (var i = 0; i < len; ++i) {
      listeners[i].call(self, arg1, arg2);
    }
  }
}

function emitThree(handler, isFn, self, arg1, arg2, arg3) {
  if (isFn) handler.call(self, arg1, arg2, arg3);else {
    var len = handler.length;
    var listeners = arrayClone(handler, len);

    for (var i = 0; i < len; ++i) {
      listeners[i].call(self, arg1, arg2, arg3);
    }
  }
}

function emitMany(handler, isFn, self, args) {
  if (isFn) handler.apply(self, args);else {
    var len = handler.length;
    var listeners = arrayClone(handler, len);

    for (var i = 0; i < len; ++i) {
      listeners[i].apply(self, args);
    }
  }
}

EventEmitter.prototype.emit = function emit(type) {
  var er, handler, len, args, i, events, domain;
  var doError = type === 'error';
  events = this._events;
  if (events) doError = doError && events.error == null;else if (!doError) return false;
  domain = this.domain; // If there is no 'error' event listener then throw.

  if (doError) {
    er = arguments[1];

    if (domain) {
      if (!er) er = new Error('Uncaught, unspecified "error" event');
      er.domainEmitter = this;
      er.domain = domain;
      er.domainThrown = false;
      domain.emit('error', er);
    } else if (er instanceof Error) {
      throw er; // Unhandled 'error' event
    } else {
      // At least give some kind of context to the user
      var err = new Error('Uncaught, unspecified "error" event. (' + er + ')');
      err.context = er;
      throw err;
    }

    return false;
  }

  handler = events[type];
  if (!handler) return false;
  var isFn = typeof handler === 'function';
  len = arguments.length;

  switch (len) {
    // fast cases
    case 1:
      emitNone(handler, isFn, this);
      break;

    case 2:
      emitOne(handler, isFn, this, arguments[1]);
      break;

    case 3:
      emitTwo(handler, isFn, this, arguments[1], arguments[2]);
      break;

    case 4:
      emitThree(handler, isFn, this, arguments[1], arguments[2], arguments[3]);
      break;
    // slower

    default:
      args = new Array(len - 1);

      for (i = 1; i < len; i++) {
        args[i - 1] = arguments[i];
      }

      emitMany(handler, isFn, this, args);
  }
  return true;
};

function _addListener(target, type, listener, prepend) {
  var m;
  var events;
  var existing;
  if (typeof listener !== 'function') throw new TypeError('"listener" argument must be a function');
  events = target._events;

  if (!events) {
    events = target._events = new EventHandlers();
    target._eventsCount = 0;
  } else {
    // To avoid recursion in the case that type === "newListener"! Before
    // adding it to the listeners, first emit "newListener".
    if (events.newListener) {
      target.emit('newListener', type, listener.listener ? listener.listener : listener); // Re-assign `events` because a newListener handler could have caused the
      // this._events to be assigned to a new object

      events = target._events;
    }

    existing = events[type];
  }

  if (!existing) {
    // Optimize the case of one listener. Don't need the extra array object.
    existing = events[type] = listener;
    ++target._eventsCount;
  } else {
    if (typeof existing === 'function') {
      // Adding the second element, need to change to array.
      existing = events[type] = prepend ? [listener, existing] : [existing, listener];
    } else {
      // If we've already got an array, just append.
      if (prepend) {
        existing.unshift(listener);
      } else {
        existing.push(listener);
      }
    } // Check for listener leak


    if (!existing.warned) {
      m = $getMaxListeners(target);

      if (m && m > 0 && existing.length > m) {
        existing.warned = true;
        var w = new Error('Possible EventEmitter memory leak detected. ' + existing.length + ' ' + type + ' listeners added. ' + 'Use emitter.setMaxListeners() to increase limit');
        w.name = 'MaxListenersExceededWarning';
        w.emitter = target;
        w.type = type;
        w.count = existing.length;
        emitWarning(w);
      }
    }
  }

  return target;
}

function emitWarning(e) {
  typeof console.warn === 'function' ? console.warn(e) : console.log(e);
}

EventEmitter.prototype.addListener = function addListener(type, listener) {
  return _addListener(this, type, listener, false);
};

EventEmitter.prototype.on = EventEmitter.prototype.addListener;

EventEmitter.prototype.prependListener = function prependListener(type, listener) {
  return _addListener(this, type, listener, true);
};

function _onceWrap(target, type, listener) {
  var fired = false;

  function g() {
    target.removeListener(type, g);

    if (!fired) {
      fired = true;
      listener.apply(target, arguments);
    }
  }

  g.listener = listener;
  return g;
}

EventEmitter.prototype.once = function once(type, listener) {
  if (typeof listener !== 'function') throw new TypeError('"listener" argument must be a function');
  this.on(type, _onceWrap(this, type, listener));
  return this;
};

EventEmitter.prototype.prependOnceListener = function prependOnceListener(type, listener) {
  if (typeof listener !== 'function') throw new TypeError('"listener" argument must be a function');
  this.prependListener(type, _onceWrap(this, type, listener));
  return this;
}; // emits a 'removeListener' event iff the listener was removed


EventEmitter.prototype.removeListener = function removeListener(type, listener) {
  var list, events, position, i, originalListener;
  if (typeof listener !== 'function') throw new TypeError('"listener" argument must be a function');
  events = this._events;
  if (!events) return this;
  list = events[type];
  if (!list) return this;

  if (list === listener || list.listener && list.listener === listener) {
    if (--this._eventsCount === 0) this._events = new EventHandlers();else {
      delete events[type];
      if (events.removeListener) this.emit('removeListener', type, list.listener || listener);
    }
  } else if (typeof list !== 'function') {
    position = -1;

    for (i = list.length; i-- > 0;) {
      if (list[i] === listener || list[i].listener && list[i].listener === listener) {
        originalListener = list[i].listener;
        position = i;
        break;
      }
    }

    if (position < 0) return this;

    if (list.length === 1) {
      list[0] = undefined;

      if (--this._eventsCount === 0) {
        this._events = new EventHandlers();
        return this;
      } else {
        delete events[type];
      }
    } else {
      spliceOne(list, position);
    }

    if (events.removeListener) this.emit('removeListener', type, originalListener || listener);
  }

  return this;
};

EventEmitter.prototype.removeAllListeners = function removeAllListeners(type) {
  var listeners, events;
  events = this._events;
  if (!events) return this; // not listening for removeListener, no need to emit

  if (!events.removeListener) {
    if (arguments.length === 0) {
      this._events = new EventHandlers();
      this._eventsCount = 0;
    } else if (events[type]) {
      if (--this._eventsCount === 0) this._events = new EventHandlers();else delete events[type];
    }

    return this;
  } // emit removeListener for all listeners on all events


  if (arguments.length === 0) {
    var keys = Object.keys(events);

    for (var i = 0, key; i < keys.length; ++i) {
      key = keys[i];
      if (key === 'removeListener') continue;
      this.removeAllListeners(key);
    }

    this.removeAllListeners('removeListener');
    this._events = new EventHandlers();
    this._eventsCount = 0;
    return this;
  }

  listeners = events[type];

  if (typeof listeners === 'function') {
    this.removeListener(type, listeners);
  } else if (listeners) {
    // LIFO order
    do {
      this.removeListener(type, listeners[listeners.length - 1]);
    } while (listeners[0]);
  }

  return this;
};

EventEmitter.prototype.listeners = function listeners(type) {
  var evlistener;
  var ret;
  var events = this._events;
  if (!events) ret = [];else {
    evlistener = events[type];
    if (!evlistener) ret = [];else if (typeof evlistener === 'function') ret = [evlistener.listener || evlistener];else ret = unwrapListeners(evlistener);
  }
  return ret;
};

EventEmitter.listenerCount = function (emitter, type) {
  if (typeof emitter.listenerCount === 'function') {
    return emitter.listenerCount(type);
  } else {
    return listenerCount.call(emitter, type);
  }
};

EventEmitter.prototype.listenerCount = listenerCount;

function listenerCount(type) {
  var events = this._events;

  if (events) {
    var evlistener = events[type];

    if (typeof evlistener === 'function') {
      return 1;
    } else if (evlistener) {
      return evlistener.length;
    }
  }

  return 0;
}

EventEmitter.prototype.eventNames = function eventNames() {
  return this._eventsCount > 0 ? Reflect.ownKeys(this._events) : [];
}; // About 1.5x faster than the two-arg version of Array#splice().


function spliceOne(list, index) {
  for (var i = index, k = i + 1, n = list.length; k < n; i += 1, k += 1) {
    list[i] = list[k];
  }

  list.pop();
}

function arrayClone(arr, i) {
  var copy = new Array(i);

  while (i--) {
    copy[i] = arr[i];
  }

  return copy;
}

function unwrapListeners(arr) {
  var ret = new Array(arr.length);

  for (var i = 0; i < ret.length; ++i) {
    ret[i] = arr[i].listener || arr[i];
  }

  return ret;
}

// shim for using process in browser
// based off https://github.com/defunctzombie/node-process/blob/master/browser.js
function defaultSetTimout() {
  throw new Error('setTimeout has not been defined');
}

function defaultClearTimeout() {
  throw new Error('clearTimeout has not been defined');
}

var cachedSetTimeout = defaultSetTimout;
var cachedClearTimeout = defaultClearTimeout;

if (typeof global$1.setTimeout === 'function') {
  cachedSetTimeout = setTimeout;
}

if (typeof global$1.clearTimeout === 'function') {
  cachedClearTimeout = clearTimeout;
}

function runTimeout(fun) {
  if (cachedSetTimeout === setTimeout) {
    //normal enviroments in sane situations
    return setTimeout(fun, 0);
  } // if setTimeout wasn't available but was latter defined


  if ((cachedSetTimeout === defaultSetTimout || !cachedSetTimeout) && setTimeout) {
    cachedSetTimeout = setTimeout;
    return setTimeout(fun, 0);
  }

  try {
    // when when somebody has screwed with setTimeout but no I.E. maddness
    return cachedSetTimeout(fun, 0);
  } catch (e) {
    try {
      // When we are in I.E. but the script has been evaled so I.E. doesn't trust the global object when called normally
      return cachedSetTimeout.call(null, fun, 0);
    } catch (e) {
      // same as above but when it's a version of I.E. that must have the global object for 'this', hopfully our context correct otherwise it will throw a global error
      return cachedSetTimeout.call(this, fun, 0);
    }
  }
}

function runClearTimeout(marker) {
  if (cachedClearTimeout === clearTimeout) {
    //normal enviroments in sane situations
    return clearTimeout(marker);
  } // if clearTimeout wasn't available but was latter defined


  if ((cachedClearTimeout === defaultClearTimeout || !cachedClearTimeout) && clearTimeout) {
    cachedClearTimeout = clearTimeout;
    return clearTimeout(marker);
  }

  try {
    // when when somebody has screwed with setTimeout but no I.E. maddness
    return cachedClearTimeout(marker);
  } catch (e) {
    try {
      // When we are in I.E. but the script has been evaled so I.E. doesn't  trust the global object when called normally
      return cachedClearTimeout.call(null, marker);
    } catch (e) {
      // same as above but when it's a version of I.E. that must have the global object for 'this', hopfully our context correct otherwise it will throw a global error.
      // Some versions of I.E. have different rules for clearTimeout vs setTimeout
      return cachedClearTimeout.call(this, marker);
    }
  }
}

var queue = [];
var draining = false;
var currentQueue;
var queueIndex = -1;

function cleanUpNextTick() {
  if (!draining || !currentQueue) {
    return;
  }

  draining = false;

  if (currentQueue.length) {
    queue = currentQueue.concat(queue);
  } else {
    queueIndex = -1;
  }

  if (queue.length) {
    drainQueue();
  }
}

function drainQueue() {
  if (draining) {
    return;
  }

  var timeout = runTimeout(cleanUpNextTick);
  draining = true;
  var len = queue.length;

  while (len) {
    currentQueue = queue;
    queue = [];

    while (++queueIndex < len) {
      if (currentQueue) {
        currentQueue[queueIndex].run();
      }
    }

    queueIndex = -1;
    len = queue.length;
  }

  currentQueue = null;
  draining = false;
  runClearTimeout(timeout);
}

function nextTick(fun) {
  var args = new Array(arguments.length - 1);

  if (arguments.length > 1) {
    for (var i = 1; i < arguments.length; i++) {
      args[i - 1] = arguments[i];
    }
  }

  queue.push(new Item(fun, args));

  if (queue.length === 1 && !draining) {
    runTimeout(drainQueue);
  }
} // v8 likes predictible objects

function Item(fun, array) {
  this.fun = fun;
  this.array = array;
}

Item.prototype.run = function () {
  this.fun.apply(null, this.array);
};

var title = 'browser';
var platform = 'browser';
var browser = true;
var env = {};
var argv = [];
var version = ''; // empty string to avoid regexp issues

var versions = {};
var release = {};
var config = {};

function noop() {}

var on = noop;
var addListener = noop;
var once = noop;
var off = noop;
var removeListener = noop;
var removeAllListeners = noop;
var emit = noop;
function binding(name) {
  throw new Error('process.binding is not supported');
}
function cwd() {
  return '/';
}
function chdir(dir) {
  throw new Error('process.chdir is not supported');
}
function umask() {
  return 0;
} // from https://github.com/kumavis/browser-process-hrtime/blob/master/index.js

var performance = global$1.performance || {};

var performanceNow = performance.now || performance.mozNow || performance.msNow || performance.oNow || performance.webkitNow || function () {
  return new Date().getTime();
}; // generate timestamp or delta
// see http://nodejs.org/api/process.html#process_process_hrtime


function hrtime(previousTimestamp) {
  var clocktime = performanceNow.call(performance) * 1e-3;
  var seconds = Math.floor(clocktime);
  var nanoseconds = Math.floor(clocktime % 1 * 1e9);

  if (previousTimestamp) {
    seconds = seconds - previousTimestamp[0];
    nanoseconds = nanoseconds - previousTimestamp[1];

    if (nanoseconds < 0) {
      seconds--;
      nanoseconds += 1e9;
    }
  }

  return [seconds, nanoseconds];
}
var startTime = new Date();
function uptime() {
  var currentTime = new Date();
  var dif = currentTime - startTime;
  return dif / 1000;
}
var process = {
  nextTick: nextTick,
  title: title,
  browser: browser,
  env: env,
  argv: argv,
  version: version,
  versions: versions,
  on: on,
  addListener: addListener,
  once: once,
  off: off,
  removeListener: removeListener,
  removeAllListeners: removeAllListeners,
  emit: emit,
  binding: binding,
  cwd: cwd,
  chdir: chdir,
  umask: umask,
  hrtime: hrtime,
  platform: platform,
  release: release,
  config: config,
  uptime: uptime
};

var inherits;

if (typeof Object.create === 'function') {
  inherits = function inherits(ctor, superCtor) {
    // implementation from standard node.js 'util' module
    ctor.super_ = superCtor;
    ctor.prototype = Object.create(superCtor.prototype, {
      constructor: {
        value: ctor,
        enumerable: false,
        writable: true,
        configurable: true
      }
    });
  };
} else {
  inherits = function inherits(ctor, superCtor) {
    ctor.super_ = superCtor;

    var TempCtor = function TempCtor() {};

    TempCtor.prototype = superCtor.prototype;
    ctor.prototype = new TempCtor();
    ctor.prototype.constructor = ctor;
  };
}

var inherits$1 = inherits;

var formatRegExp = /%[sdj%]/g;
function format(f) {
  if (!isString(f)) {
    var objects = [];

    for (var i = 0; i < arguments.length; i++) {
      objects.push(inspect(arguments[i]));
    }

    return objects.join(' ');
  }

  var i = 1;
  var args = arguments;
  var len = args.length;
  var str = String(f).replace(formatRegExp, function (x) {
    if (x === '%%') return '%';
    if (i >= len) return x;

    switch (x) {
      case '%s':
        return String(args[i++]);

      case '%d':
        return Number(args[i++]);

      case '%j':
        try {
          return JSON.stringify(args[i++]);
        } catch (_) {
          return '[Circular]';
        }

      default:
        return x;
    }
  });

  for (var x = args[i]; i < len; x = args[++i]) {
    if (isNull(x) || !isObject(x)) {
      str += ' ' + x;
    } else {
      str += ' ' + inspect(x);
    }
  }

  return str;
}
// Returns a modified function which warns once by default.
// If --no-deprecation is set, then it is a no-op.

function deprecate(fn, msg) {
  // Allow for deprecating things in the process of starting up.
  if (isUndefined(global$1.process)) {
    return function () {
      return deprecate(fn, msg).apply(this, arguments);
    };
  }

  var warned = false;

  function deprecated() {
    if (!warned) {
      {
        console.error(msg);
      }

      warned = true;
    }

    return fn.apply(this, arguments);
  }

  return deprecated;
}
var debugs = {};
var debugEnviron;
function debuglog(set) {
  if (isUndefined(debugEnviron)) debugEnviron = '';
  set = set.toUpperCase();

  if (!debugs[set]) {
    if (new RegExp('\\b' + set + '\\b', 'i').test(debugEnviron)) {
      var pid = 0;

      debugs[set] = function () {
        var msg = format.apply(null, arguments);
        console.error('%s %d: %s', set, pid, msg);
      };
    } else {
      debugs[set] = function () {};
    }
  }

  return debugs[set];
}
/**
 * Echos the value of a value. Trys to print the value out
 * in the best way possible given the different types.
 *
 * @param {Object} obj The object to print out.
 * @param {Object} opts Optional options object that alters the output.
 */

/* legacy: obj, showHidden, depth, colors*/

function inspect(obj, opts) {
  // default options
  var ctx = {
    seen: [],
    stylize: stylizeNoColor
  }; // legacy...

  if (arguments.length >= 3) ctx.depth = arguments[2];
  if (arguments.length >= 4) ctx.colors = arguments[3];

  if (isBoolean(opts)) {
    // legacy...
    ctx.showHidden = opts;
  } else if (opts) {
    // got an "options" object
    _extend(ctx, opts);
  } // set default options


  if (isUndefined(ctx.showHidden)) ctx.showHidden = false;
  if (isUndefined(ctx.depth)) ctx.depth = 2;
  if (isUndefined(ctx.colors)) ctx.colors = false;
  if (isUndefined(ctx.customInspect)) ctx.customInspect = true;
  if (ctx.colors) ctx.stylize = stylizeWithColor;
  return formatValue(ctx, obj, ctx.depth);
} // http://en.wikipedia.org/wiki/ANSI_escape_code#graphics

inspect.colors = {
  'bold': [1, 22],
  'italic': [3, 23],
  'underline': [4, 24],
  'inverse': [7, 27],
  'white': [37, 39],
  'grey': [90, 39],
  'black': [30, 39],
  'blue': [34, 39],
  'cyan': [36, 39],
  'green': [32, 39],
  'magenta': [35, 39],
  'red': [31, 39],
  'yellow': [33, 39]
}; // Don't use 'blue' not visible on cmd.exe

inspect.styles = {
  'special': 'cyan',
  'number': 'yellow',
  'boolean': 'yellow',
  'undefined': 'grey',
  'null': 'bold',
  'string': 'green',
  'date': 'magenta',
  // "name": intentionally not styling
  'regexp': 'red'
};

function stylizeWithColor(str, styleType) {
  var style = inspect.styles[styleType];

  if (style) {
    return "\x1B[" + inspect.colors[style][0] + 'm' + str + "\x1B[" + inspect.colors[style][1] + 'm';
  } else {
    return str;
  }
}

function stylizeNoColor(str, styleType) {
  return str;
}

function arrayToHash(array) {
  var hash = {};
  array.forEach(function (val, idx) {
    hash[val] = true;
  });
  return hash;
}

function formatValue(ctx, value, recurseTimes) {
  // Provide a hook for user-specified inspect functions.
  // Check that value is an object with an inspect function on it
  if (ctx.customInspect && value && isFunction(value.inspect) && // Filter out the util module, it's inspect function is special
  value.inspect !== inspect && // Also filter out any prototype objects using the circular check.
  !(value.constructor && value.constructor.prototype === value)) {
    var ret = value.inspect(recurseTimes, ctx);

    if (!isString(ret)) {
      ret = formatValue(ctx, ret, recurseTimes);
    }

    return ret;
  } // Primitive types cannot have properties


  var primitive = formatPrimitive(ctx, value);

  if (primitive) {
    return primitive;
  } // Look up the keys of the object.


  var keys = Object.keys(value);
  var visibleKeys = arrayToHash(keys);

  if (ctx.showHidden) {
    keys = Object.getOwnPropertyNames(value);
  } // IE doesn't make error fields non-enumerable
  // http://msdn.microsoft.com/en-us/library/ie/dww52sbt(v=vs.94).aspx


  if (isError(value) && (keys.indexOf('message') >= 0 || keys.indexOf('description') >= 0)) {
    return formatError(value);
  } // Some type of object without properties can be shortcutted.


  if (keys.length === 0) {
    if (isFunction(value)) {
      var name = value.name ? ': ' + value.name : '';
      return ctx.stylize('[Function' + name + ']', 'special');
    }

    if (isRegExp(value)) {
      return ctx.stylize(RegExp.prototype.toString.call(value), 'regexp');
    }

    if (isDate(value)) {
      return ctx.stylize(Date.prototype.toString.call(value), 'date');
    }

    if (isError(value)) {
      return formatError(value);
    }
  }

  var base = '',
      array = false,
      braces = ['{', '}']; // Make Array say that they are Array

  if (isArray$1(value)) {
    array = true;
    braces = ['[', ']'];
  } // Make functions say that they are functions


  if (isFunction(value)) {
    var n = value.name ? ': ' + value.name : '';
    base = ' [Function' + n + ']';
  } // Make RegExps say that they are RegExps


  if (isRegExp(value)) {
    base = ' ' + RegExp.prototype.toString.call(value);
  } // Make dates with properties first say the date


  if (isDate(value)) {
    base = ' ' + Date.prototype.toUTCString.call(value);
  } // Make error with message first say the error


  if (isError(value)) {
    base = ' ' + formatError(value);
  }

  if (keys.length === 0 && (!array || value.length == 0)) {
    return braces[0] + base + braces[1];
  }

  if (recurseTimes < 0) {
    if (isRegExp(value)) {
      return ctx.stylize(RegExp.prototype.toString.call(value), 'regexp');
    } else {
      return ctx.stylize('[Object]', 'special');
    }
  }

  ctx.seen.push(value);
  var output;

  if (array) {
    output = formatArray(ctx, value, recurseTimes, visibleKeys, keys);
  } else {
    output = keys.map(function (key) {
      return formatProperty(ctx, value, recurseTimes, visibleKeys, key, array);
    });
  }

  ctx.seen.pop();
  return reduceToSingleString(output, base, braces);
}

function formatPrimitive(ctx, value) {
  if (isUndefined(value)) return ctx.stylize('undefined', 'undefined');

  if (isString(value)) {
    var simple = '\'' + JSON.stringify(value).replace(/^"|"$/g, '').replace(/'/g, "\\'").replace(/\\"/g, '"') + '\'';
    return ctx.stylize(simple, 'string');
  }

  if (isNumber(value)) return ctx.stylize('' + value, 'number');
  if (isBoolean(value)) return ctx.stylize('' + value, 'boolean'); // For some reason typeof null is "object", so special case here.

  if (isNull(value)) return ctx.stylize('null', 'null');
}

function formatError(value) {
  return '[' + Error.prototype.toString.call(value) + ']';
}

function formatArray(ctx, value, recurseTimes, visibleKeys, keys) {
  var output = [];

  for (var i = 0, l = value.length; i < l; ++i) {
    if (hasOwnProperty(value, String(i))) {
      output.push(formatProperty(ctx, value, recurseTimes, visibleKeys, String(i), true));
    } else {
      output.push('');
    }
  }

  keys.forEach(function (key) {
    if (!key.match(/^\d+$/)) {
      output.push(formatProperty(ctx, value, recurseTimes, visibleKeys, key, true));
    }
  });
  return output;
}

function formatProperty(ctx, value, recurseTimes, visibleKeys, key, array) {
  var name, str, desc;
  desc = Object.getOwnPropertyDescriptor(value, key) || {
    value: value[key]
  };

  if (desc.get) {
    if (desc.set) {
      str = ctx.stylize('[Getter/Setter]', 'special');
    } else {
      str = ctx.stylize('[Getter]', 'special');
    }
  } else {
    if (desc.set) {
      str = ctx.stylize('[Setter]', 'special');
    }
  }

  if (!hasOwnProperty(visibleKeys, key)) {
    name = '[' + key + ']';
  }

  if (!str) {
    if (ctx.seen.indexOf(desc.value) < 0) {
      if (isNull(recurseTimes)) {
        str = formatValue(ctx, desc.value, null);
      } else {
        str = formatValue(ctx, desc.value, recurseTimes - 1);
      }

      if (str.indexOf('\n') > -1) {
        if (array) {
          str = str.split('\n').map(function (line) {
            return '  ' + line;
          }).join('\n').substr(2);
        } else {
          str = '\n' + str.split('\n').map(function (line) {
            return '   ' + line;
          }).join('\n');
        }
      }
    } else {
      str = ctx.stylize('[Circular]', 'special');
    }
  }

  if (isUndefined(name)) {
    if (array && key.match(/^\d+$/)) {
      return str;
    }

    name = JSON.stringify('' + key);

    if (name.match(/^"([a-zA-Z_][a-zA-Z_0-9]*)"$/)) {
      name = name.substr(1, name.length - 2);
      name = ctx.stylize(name, 'name');
    } else {
      name = name.replace(/'/g, "\\'").replace(/\\"/g, '"').replace(/(^"|"$)/g, "'");
      name = ctx.stylize(name, 'string');
    }
  }

  return name + ': ' + str;
}

function reduceToSingleString(output, base, braces) {
  var length = output.reduce(function (prev, cur) {
    if (cur.indexOf('\n') >= 0) ;
    return prev + cur.replace(/\u001b\[\d\d?m/g, '').length + 1;
  }, 0);

  if (length > 60) {
    return braces[0] + (base === '' ? '' : base + '\n ') + ' ' + output.join(',\n  ') + ' ' + braces[1];
  }

  return braces[0] + base + ' ' + output.join(', ') + ' ' + braces[1];
} // NOTE: These type checking functions intentionally don't use `instanceof`
// because it is fragile and can be easily faked with `Object.create()`.


function isArray$1(ar) {
  return Array.isArray(ar);
}
function isBoolean(arg) {
  return typeof arg === 'boolean';
}
function isNull(arg) {
  return arg === null;
}
function isNullOrUndefined(arg) {
  return arg == null;
}
function isNumber(arg) {
  return typeof arg === 'number';
}
function isString(arg) {
  return typeof arg === 'string';
}
function isSymbol(arg) {
  return typeof arg === 'symbol';
}
function isUndefined(arg) {
  return arg === void 0;
}
function isRegExp(re) {
  return isObject(re) && objectToString(re) === '[object RegExp]';
}
function isObject(arg) {
  return typeof arg === 'object' && arg !== null;
}
function isDate(d) {
  return isObject(d) && objectToString(d) === '[object Date]';
}
function isError(e) {
  return isObject(e) && (objectToString(e) === '[object Error]' || e instanceof Error);
}
function isFunction(arg) {
  return typeof arg === 'function';
}
function isPrimitive(arg) {
  return arg === null || typeof arg === 'boolean' || typeof arg === 'number' || typeof arg === 'string' || typeof arg === 'symbol' || // ES6 symbol
  typeof arg === 'undefined';
}
function isBuffer$1(maybeBuf) {
  return isBuffer(maybeBuf);
}

function objectToString(o) {
  return Object.prototype.toString.call(o);
}

function pad(n) {
  return n < 10 ? '0' + n.toString(10) : n.toString(10);
}

var months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']; // 26 Feb 16:19:34

function timestamp() {
  var d = new Date();
  var time = [pad(d.getHours()), pad(d.getMinutes()), pad(d.getSeconds())].join(':');
  return [d.getDate(), months[d.getMonth()], time].join(' ');
} // log is just a thin wrapper to console.log that prepends a timestamp


function log() {
  console.log('%s - %s', timestamp(), format.apply(null, arguments));
}
function _extend(origin, add) {
  // Don't do anything if add isn't an object
  if (!add || !isObject(add)) return origin;
  var keys = Object.keys(add);
  var i = keys.length;

  while (i--) {
    origin[keys[i]] = add[keys[i]];
  }

  return origin;
}

function hasOwnProperty(obj, prop) {
  return Object.prototype.hasOwnProperty.call(obj, prop);
}

var require$$3 = {
  inherits: inherits$1,
  _extend: _extend,
  log: log,
  isBuffer: isBuffer$1,
  isPrimitive: isPrimitive,
  isFunction: isFunction,
  isError: isError,
  isDate: isDate,
  isObject: isObject,
  isRegExp: isRegExp,
  isUndefined: isUndefined,
  isSymbol: isSymbol,
  isString: isString,
  isNumber: isNumber,
  isNullOrUndefined: isNullOrUndefined,
  isNull: isNull,
  isBoolean: isBoolean,
  isArray: isArray$1,
  inspect: inspect,
  deprecate: deprecate,
  format: format,
  debuglog: debuglog
};

function BufferList() {
  this.head = null;
  this.tail = null;
  this.length = 0;
}

BufferList.prototype.push = function (v) {
  var entry = {
    data: v,
    next: null
  };
  if (this.length > 0) this.tail.next = entry;else this.head = entry;
  this.tail = entry;
  ++this.length;
};

BufferList.prototype.unshift = function (v) {
  var entry = {
    data: v,
    next: this.head
  };
  if (this.length === 0) this.tail = entry;
  this.head = entry;
  ++this.length;
};

BufferList.prototype.shift = function () {
  if (this.length === 0) return;
  var ret = this.head.data;
  if (this.length === 1) this.head = this.tail = null;else this.head = this.head.next;
  --this.length;
  return ret;
};

BufferList.prototype.clear = function () {
  this.head = this.tail = null;
  this.length = 0;
};

BufferList.prototype.join = function (s) {
  if (this.length === 0) return '';
  var p = this.head;
  var ret = '' + p.data;

  while (p = p.next) {
    ret += s + p.data;
  }

  return ret;
};

BufferList.prototype.concat = function (n) {
  if (this.length === 0) return Buffer.alloc(0);
  if (this.length === 1) return this.head.data;
  var ret = Buffer.allocUnsafe(n >>> 0);
  var p = this.head;
  var i = 0;

  while (p) {
    p.data.copy(ret, i);
    i += p.data.length;
    p = p.next;
  }

  return ret;
};

var string_decoder = createCommonjsModule(function (module, exports) {
  // Copyright Joyent, Inc. and other Node contributors.
  //
  // Permission is hereby granted, free of charge, to any person obtaining a
  // copy of this software and associated documentation files (the
  // "Software"), to deal in the Software without restriction, including
  // without limitation the rights to use, copy, modify, merge, publish,
  // distribute, sublicense, and/or sell copies of the Software, and to permit
  // persons to whom the Software is furnished to do so, subject to the
  // following conditions:
  //
  // The above copyright notice and this permission notice shall be included
  // in all copies or substantial portions of the Software.
  //
  // THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  // OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  // MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
  // NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
  // DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
  // OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
  // USE OR OTHER DEALINGS IN THE SOFTWARE.
  var Buffer = bufferEs6.Buffer;

  var isBufferEncoding = Buffer.isEncoding || function (encoding) {
    switch (encoding && encoding.toLowerCase()) {
      case 'hex':
      case 'utf8':
      case 'utf-8':
      case 'ascii':
      case 'binary':
      case 'base64':
      case 'ucs2':
      case 'ucs-2':
      case 'utf16le':
      case 'utf-16le':
      case 'raw':
        return true;

      default:
        return false;
    }
  };

  function assertEncoding(encoding) {
    if (encoding && !isBufferEncoding(encoding)) {
      throw new Error('Unknown encoding: ' + encoding);
    }
  } // StringDecoder provides an interface for efficiently splitting a series of
  // buffers into a series of JS strings without breaking apart multi-byte
  // characters. CESU-8 is handled as part of the UTF-8 encoding.
  //
  // @TODO Handling all encodings inside a single object makes it very difficult
  // to reason about this code, so it should be split up in the future.
  // @TODO There should be a utf8-strict encoding that rejects invalid UTF-8 code
  // points as used by CESU-8.


  var StringDecoder = exports.StringDecoder = function (encoding) {
    this.encoding = (encoding || 'utf8').toLowerCase().replace(/[-_]/, '');
    assertEncoding(encoding);

    switch (this.encoding) {
      case 'utf8':
        // CESU-8 represents each of Surrogate Pair by 3-bytes
        this.surrogateSize = 3;
        break;

      case 'ucs2':
      case 'utf16le':
        // UTF-16 represents each of Surrogate Pair by 2-bytes
        this.surrogateSize = 2;
        this.detectIncompleteChar = utf16DetectIncompleteChar;
        break;

      case 'base64':
        // Base-64 stores 3 bytes in 4 chars, and pads the remainder.
        this.surrogateSize = 3;
        this.detectIncompleteChar = base64DetectIncompleteChar;
        break;

      default:
        this.write = passThroughWrite;
        return;
    } // Enough space to store all bytes of a single character. UTF-8 needs 4
    // bytes, but CESU-8 may require up to 6 (3 bytes per surrogate).


    this.charBuffer = new Buffer(6); // Number of bytes received for the current incomplete multi-byte character.

    this.charReceived = 0; // Number of bytes expected for the current incomplete multi-byte character.

    this.charLength = 0;
  }; // write decodes the given buffer and returns it as JS string that is
  // guaranteed to not contain any partial multi-byte characters. Any partial
  // character found at the end of the buffer is buffered up, and will be
  // returned when calling write again with the remaining bytes.
  //
  // Note: Converting a Buffer containing an orphan surrogate to a String
  // currently works, but converting a String to a Buffer (via `new Buffer`, or
  // Buffer#write) will replace incomplete surrogates with the unicode
  // replacement character. See https://codereview.chromium.org/121173009/ .


  StringDecoder.prototype.write = function (buffer) {
    var charStr = ''; // if our last write ended with an incomplete multibyte character

    while (this.charLength) {
      // determine how many remaining bytes this buffer has to offer for this char
      var available = buffer.length >= this.charLength - this.charReceived ? this.charLength - this.charReceived : buffer.length; // add the new bytes to the char buffer

      buffer.copy(this.charBuffer, this.charReceived, 0, available);
      this.charReceived += available;

      if (this.charReceived < this.charLength) {
        // still not enough chars in this buffer? wait for more ...
        return '';
      } // remove bytes belonging to the current character from the buffer


      buffer = buffer.slice(available, buffer.length); // get the character that was split

      charStr = this.charBuffer.slice(0, this.charLength).toString(this.encoding); // CESU-8: lead surrogate (D800-DBFF) is also the incomplete character

      var charCode = charStr.charCodeAt(charStr.length - 1);

      if (charCode >= 0xD800 && charCode <= 0xDBFF) {
        this.charLength += this.surrogateSize;
        charStr = '';
        continue;
      }

      this.charReceived = this.charLength = 0; // if there are no more bytes in this buffer, just emit our char

      if (buffer.length === 0) {
        return charStr;
      }

      break;
    } // determine and set charLength / charReceived


    this.detectIncompleteChar(buffer);
    var end = buffer.length;

    if (this.charLength) {
      // buffer the incomplete character bytes we got
      buffer.copy(this.charBuffer, 0, buffer.length - this.charReceived, end);
      end -= this.charReceived;
    }

    charStr += buffer.toString(this.encoding, 0, end);
    var end = charStr.length - 1;
    var charCode = charStr.charCodeAt(end); // CESU-8: lead surrogate (D800-DBFF) is also the incomplete character

    if (charCode >= 0xD800 && charCode <= 0xDBFF) {
      var size = this.surrogateSize;
      this.charLength += size;
      this.charReceived += size;
      this.charBuffer.copy(this.charBuffer, size, 0, size);
      buffer.copy(this.charBuffer, 0, 0, size);
      return charStr.substring(0, end);
    } // or just emit the charStr


    return charStr;
  }; // detectIncompleteChar determines if there is an incomplete UTF-8 character at
  // the end of the given buffer. If so, it sets this.charLength to the byte
  // length that character, and sets this.charReceived to the number of bytes
  // that are available for this character.


  StringDecoder.prototype.detectIncompleteChar = function (buffer) {
    // determine how many bytes we have to check at the end of this buffer
    var i = buffer.length >= 3 ? 3 : buffer.length; // Figure out if one of the last i bytes of our buffer announces an
    // incomplete char.

    for (; i > 0; i--) {
      var c = buffer[buffer.length - i]; // See http://en.wikipedia.org/wiki/UTF-8#Description
      // 110XXXXX

      if (i == 1 && c >> 5 == 0x06) {
        this.charLength = 2;
        break;
      } // 1110XXXX


      if (i <= 2 && c >> 4 == 0x0E) {
        this.charLength = 3;
        break;
      } // 11110XXX


      if (i <= 3 && c >> 3 == 0x1E) {
        this.charLength = 4;
        break;
      }
    }

    this.charReceived = i;
  };

  StringDecoder.prototype.end = function (buffer) {
    var res = '';
    if (buffer && buffer.length) res = this.write(buffer);

    if (this.charReceived) {
      var cr = this.charReceived;
      var buf = this.charBuffer;
      var enc = this.encoding;
      res += buf.slice(0, cr).toString(enc);
    }

    return res;
  };

  function passThroughWrite(buffer) {
    return buffer.toString(this.encoding);
  }

  function utf16DetectIncompleteChar(buffer) {
    this.charReceived = buffer.length % 2;
    this.charLength = this.charReceived ? 2 : 0;
  }

  function base64DetectIncompleteChar(buffer) {
    this.charReceived = buffer.length % 3;
    this.charLength = this.charReceived ? 3 : 0;
  }
});
var string_decoder_1 = string_decoder.StringDecoder;

Readable.ReadableState = ReadableState;
var debug = debuglog('stream');
inherits$1(Readable, EventEmitter);

function prependListener(emitter, event, fn) {
  // Sadly this is not cacheable as some libraries bundle their own
  // event emitter implementation with them.
  if (typeof emitter.prependListener === 'function') {
    return emitter.prependListener(event, fn);
  } else {
    // This is a hack to make sure that our error handler is attached before any
    // userland ones.  NEVER DO THIS. This is here only because this code needs
    // to continue to work with older versions of Node.js that do not include
    // the prependListener() method. The goal is to eventually remove this hack.
    if (!emitter._events || !emitter._events[event]) emitter.on(event, fn);else if (Array.isArray(emitter._events[event])) emitter._events[event].unshift(fn);else emitter._events[event] = [fn, emitter._events[event]];
  }
}

function listenerCount$1(emitter, type) {
  return emitter.listeners(type).length;
}

function ReadableState(options, stream) {
  options = options || {}; // object stream flag. Used to make read(n) ignore n and to
  // make all the buffer merging and length checks go away

  this.objectMode = !!options.objectMode;
  if (stream instanceof Duplex) this.objectMode = this.objectMode || !!options.readableObjectMode; // the point at which it stops calling _read() to fill the buffer
  // Note: 0 is a valid value, means "don't call _read preemptively ever"

  var hwm = options.highWaterMark;
  var defaultHwm = this.objectMode ? 16 : 16 * 1024;
  this.highWaterMark = hwm || hwm === 0 ? hwm : defaultHwm; // cast to ints.

  this.highWaterMark = ~~this.highWaterMark; // A linked list is used to store data chunks instead of an array because the
  // linked list can remove elements from the beginning faster than
  // array.shift()

  this.buffer = new BufferList();
  this.length = 0;
  this.pipes = null;
  this.pipesCount = 0;
  this.flowing = null;
  this.ended = false;
  this.endEmitted = false;
  this.reading = false; // a flag to be able to tell if the onwrite cb is called immediately,
  // or on a later tick.  We set this to true at first, because any
  // actions that shouldn't happen until "later" should generally also
  // not happen before the first write call.

  this.sync = true; // whenever we return null, then we set a flag to say
  // that we're awaiting a 'readable' event emission.

  this.needReadable = false;
  this.emittedReadable = false;
  this.readableListening = false;
  this.resumeScheduled = false; // Crypto is kind of old and crusty.  Historically, its default string
  // encoding is 'binary' so we have to make this configurable.
  // Everything else in the universe uses 'utf8', though.

  this.defaultEncoding = options.defaultEncoding || 'utf8'; // when piping, we only care about 'readable' events that happen
  // after read()ing all the bytes and not getting any pushback.

  this.ranOut = false; // the number of writers that are awaiting a drain event in .pipe()s

  this.awaitDrain = 0; // if true, a maybeReadMore has been scheduled

  this.readingMore = false;
  this.decoder = null;
  this.encoding = null;

  if (options.encoding) {
    this.decoder = new string_decoder_1(options.encoding);
    this.encoding = options.encoding;
  }
}
function Readable(options) {
  if (!(this instanceof Readable)) return new Readable(options);
  this._readableState = new ReadableState(options, this); // legacy

  this.readable = true;
  if (options && typeof options.read === 'function') this._read = options.read;
  EventEmitter.call(this);
} // Manually shove something into the read() buffer.
// This returns true if the highWaterMark has not been hit yet,
// similar to how Writable.write() returns true if you should
// write() some more.

Readable.prototype.push = function (chunk, encoding) {
  var state = this._readableState;

  if (!state.objectMode && typeof chunk === 'string') {
    encoding = encoding || state.defaultEncoding;

    if (encoding !== state.encoding) {
      chunk = Buffer.from(chunk, encoding);
      encoding = '';
    }
  }

  return readableAddChunk(this, state, chunk, encoding, false);
}; // Unshift should *always* be something directly out of read()


Readable.prototype.unshift = function (chunk) {
  var state = this._readableState;
  return readableAddChunk(this, state, chunk, '', true);
};

Readable.prototype.isPaused = function () {
  return this._readableState.flowing === false;
};

function readableAddChunk(stream, state, chunk, encoding, addToFront) {
  var er = chunkInvalid(state, chunk);

  if (er) {
    stream.emit('error', er);
  } else if (chunk === null) {
    state.reading = false;
    onEofChunk(stream, state);
  } else if (state.objectMode || chunk && chunk.length > 0) {
    if (state.ended && !addToFront) {
      var e = new Error('stream.push() after EOF');
      stream.emit('error', e);
    } else if (state.endEmitted && addToFront) {
      var _e = new Error('stream.unshift() after end event');

      stream.emit('error', _e);
    } else {
      var skipAdd;

      if (state.decoder && !addToFront && !encoding) {
        chunk = state.decoder.write(chunk);
        skipAdd = !state.objectMode && chunk.length === 0;
      }

      if (!addToFront) state.reading = false; // Don't add to the buffer if we've decoded to an empty string chunk and
      // we're not in object mode

      if (!skipAdd) {
        // if we want the data now, just emit it.
        if (state.flowing && state.length === 0 && !state.sync) {
          stream.emit('data', chunk);
          stream.read(0);
        } else {
          // update the buffer info.
          state.length += state.objectMode ? 1 : chunk.length;
          if (addToFront) state.buffer.unshift(chunk);else state.buffer.push(chunk);
          if (state.needReadable) emitReadable(stream);
        }
      }

      maybeReadMore(stream, state);
    }
  } else if (!addToFront) {
    state.reading = false;
  }

  return needMoreData(state);
} // if it's past the high water mark, we can push in some more.
// Also, if we have no data yet, we can stand some
// more bytes.  This is to work around cases where hwm=0,
// such as the repl.  Also, if the push() triggered a
// readable event, and the user called read(largeNumber) such that
// needReadable was set, then we ought to push more, so that another
// 'readable' event will be triggered.


function needMoreData(state) {
  return !state.ended && (state.needReadable || state.length < state.highWaterMark || state.length === 0);
} // backwards compatibility.


Readable.prototype.setEncoding = function (enc) {
  this._readableState.decoder = new string_decoder_1(enc);
  this._readableState.encoding = enc;
  return this;
}; // Don't raise the hwm > 8MB


var MAX_HWM = 0x800000;

function computeNewHighWaterMark(n) {
  if (n >= MAX_HWM) {
    n = MAX_HWM;
  } else {
    // Get the next highest power of 2 to prevent increasing hwm excessively in
    // tiny amounts
    n--;
    n |= n >>> 1;
    n |= n >>> 2;
    n |= n >>> 4;
    n |= n >>> 8;
    n |= n >>> 16;
    n++;
  }

  return n;
} // This function is designed to be inlinable, so please take care when making
// changes to the function body.


function howMuchToRead(n, state) {
  if (n <= 0 || state.length === 0 && state.ended) return 0;
  if (state.objectMode) return 1;

  if (n !== n) {
    // Only flow one buffer at a time
    if (state.flowing && state.length) return state.buffer.head.data.length;else return state.length;
  } // If we're asking for more than the current hwm, then raise the hwm.


  if (n > state.highWaterMark) state.highWaterMark = computeNewHighWaterMark(n);
  if (n <= state.length) return n; // Don't have enough

  if (!state.ended) {
    state.needReadable = true;
    return 0;
  }

  return state.length;
} // you can override either this method, or the async _read(n) below.


Readable.prototype.read = function (n) {
  debug('read', n);
  n = parseInt(n, 10);
  var state = this._readableState;
  var nOrig = n;
  if (n !== 0) state.emittedReadable = false; // if we're doing read(0) to trigger a readable event, but we
  // already have a bunch of data in the buffer, then just trigger
  // the 'readable' event and move on.

  if (n === 0 && state.needReadable && (state.length >= state.highWaterMark || state.ended)) {
    debug('read: emitReadable', state.length, state.ended);
    if (state.length === 0 && state.ended) endReadable(this);else emitReadable(this);
    return null;
  }

  n = howMuchToRead(n, state); // if we've ended, and we're now clear, then finish it up.

  if (n === 0 && state.ended) {
    if (state.length === 0) endReadable(this);
    return null;
  } // All the actual chunk generation logic needs to be
  // *below* the call to _read.  The reason is that in certain
  // synthetic stream cases, such as passthrough streams, _read
  // may be a completely synchronous operation which may change
  // the state of the read buffer, providing enough data when
  // before there was *not* enough.
  //
  // So, the steps are:
  // 1. Figure out what the state of things will be after we do
  // a read from the buffer.
  //
  // 2. If that resulting state will trigger a _read, then call _read.
  // Note that this may be asynchronous, or synchronous.  Yes, it is
  // deeply ugly to write APIs this way, but that still doesn't mean
  // that the Readable class should behave improperly, as streams are
  // designed to be sync/async agnostic.
  // Take note if the _read call is sync or async (ie, if the read call
  // has returned yet), so that we know whether or not it's safe to emit
  // 'readable' etc.
  //
  // 3. Actually pull the requested chunks out of the buffer and return.
  // if we need a readable event, then we need to do some reading.


  var doRead = state.needReadable;
  debug('need readable', doRead); // if we currently have less than the highWaterMark, then also read some

  if (state.length === 0 || state.length - n < state.highWaterMark) {
    doRead = true;
    debug('length less than watermark', doRead);
  } // however, if we've ended, then there's no point, and if we're already
  // reading, then it's unnecessary.


  if (state.ended || state.reading) {
    doRead = false;
    debug('reading or ended', doRead);
  } else if (doRead) {
    debug('do read');
    state.reading = true;
    state.sync = true; // if the length is currently zero, then we *need* a readable event.

    if (state.length === 0) state.needReadable = true; // call internal read method

    this._read(state.highWaterMark);

    state.sync = false; // If _read pushed data synchronously, then `reading` will be false,
    // and we need to re-evaluate how much data we can return to the user.

    if (!state.reading) n = howMuchToRead(nOrig, state);
  }

  var ret;
  if (n > 0) ret = fromList(n, state);else ret = null;

  if (ret === null) {
    state.needReadable = true;
    n = 0;
  } else {
    state.length -= n;
  }

  if (state.length === 0) {
    // If we have nothing in the buffer, then we want to know
    // as soon as we *do* get something into the buffer.
    if (!state.ended) state.needReadable = true; // If we tried to read() past the EOF, then emit end on the next tick.

    if (nOrig !== n && state.ended) endReadable(this);
  }

  if (ret !== null) this.emit('data', ret);
  return ret;
};

function chunkInvalid(state, chunk) {
  var er = null;

  if (!isBuffer(chunk) && typeof chunk !== 'string' && chunk !== null && chunk !== undefined && !state.objectMode) {
    er = new TypeError('Invalid non-string/buffer chunk');
  }

  return er;
}

function onEofChunk(stream, state) {
  if (state.ended) return;

  if (state.decoder) {
    var chunk = state.decoder.end();

    if (chunk && chunk.length) {
      state.buffer.push(chunk);
      state.length += state.objectMode ? 1 : chunk.length;
    }
  }

  state.ended = true; // emit 'readable' now to make sure it gets picked up.

  emitReadable(stream);
} // Don't emit readable right away in sync mode, because this can trigger
// another read() call => stack overflow.  This way, it might trigger
// a nextTick recursion warning, but that's not so bad.


function emitReadable(stream) {
  var state = stream._readableState;
  state.needReadable = false;

  if (!state.emittedReadable) {
    debug('emitReadable', state.flowing);
    state.emittedReadable = true;
    if (state.sync) nextTick(emitReadable_, stream);else emitReadable_(stream);
  }
}

function emitReadable_(stream) {
  debug('emit readable');
  stream.emit('readable');
  flow(stream);
} // at this point, the user has presumably seen the 'readable' event,
// and called read() to consume some data.  that may have triggered
// in turn another _read(n) call, in which case reading = true if
// it's in progress.
// However, if we're not ended, or reading, and the length < hwm,
// then go ahead and try to read some more preemptively.


function maybeReadMore(stream, state) {
  if (!state.readingMore) {
    state.readingMore = true;
    nextTick(maybeReadMore_, stream, state);
  }
}

function maybeReadMore_(stream, state) {
  var len = state.length;

  while (!state.reading && !state.flowing && !state.ended && state.length < state.highWaterMark) {
    debug('maybeReadMore read 0');
    stream.read(0);
    if (len === state.length) // didn't get any data, stop spinning.
      break;else len = state.length;
  }

  state.readingMore = false;
} // abstract method.  to be overridden in specific implementation classes.
// call cb(er, data) where data is <= n in length.
// for virtual (non-string, non-buffer) streams, "length" is somewhat
// arbitrary, and perhaps not very meaningful.


Readable.prototype._read = function (n) {
  this.emit('error', new Error('not implemented'));
};

Readable.prototype.pipe = function (dest, pipeOpts) {
  var src = this;
  var state = this._readableState;

  switch (state.pipesCount) {
    case 0:
      state.pipes = dest;
      break;

    case 1:
      state.pipes = [state.pipes, dest];
      break;

    default:
      state.pipes.push(dest);
      break;
  }

  state.pipesCount += 1;
  debug('pipe count=%d opts=%j', state.pipesCount, pipeOpts);
  var doEnd = !pipeOpts || pipeOpts.end !== false;
  var endFn = doEnd ? onend : cleanup;
  if (state.endEmitted) nextTick(endFn);else src.once('end', endFn);
  dest.on('unpipe', onunpipe);

  function onunpipe(readable) {
    debug('onunpipe');

    if (readable === src) {
      cleanup();
    }
  }

  function onend() {
    debug('onend');
    dest.end();
  } // when the dest drains, it reduces the awaitDrain counter
  // on the source.  This would be more elegant with a .once()
  // handler in flow(), but adding and removing repeatedly is
  // too slow.


  var ondrain = pipeOnDrain(src);
  dest.on('drain', ondrain);
  var cleanedUp = false;

  function cleanup() {
    debug('cleanup'); // cleanup event handlers once the pipe is broken

    dest.removeListener('close', onclose);
    dest.removeListener('finish', onfinish);
    dest.removeListener('drain', ondrain);
    dest.removeListener('error', onerror);
    dest.removeListener('unpipe', onunpipe);
    src.removeListener('end', onend);
    src.removeListener('end', cleanup);
    src.removeListener('data', ondata);
    cleanedUp = true; // if the reader is waiting for a drain event from this
    // specific writer, then it would cause it to never start
    // flowing again.
    // So, if this is awaiting a drain, then we just call it now.
    // If we don't know, then assume that we are waiting for one.

    if (state.awaitDrain && (!dest._writableState || dest._writableState.needDrain)) ondrain();
  } // If the user pushes more data while we're writing to dest then we'll end up
  // in ondata again. However, we only want to increase awaitDrain once because
  // dest will only emit one 'drain' event for the multiple writes.
  // => Introduce a guard on increasing awaitDrain.


  var increasedAwaitDrain = false;
  src.on('data', ondata);

  function ondata(chunk) {
    debug('ondata');
    increasedAwaitDrain = false;
    var ret = dest.write(chunk);

    if (false === ret && !increasedAwaitDrain) {
      // If the user unpiped during `dest.write()`, it is possible
      // to get stuck in a permanently paused state if that write
      // also returned false.
      // => Check whether `dest` is still a piping destination.
      if ((state.pipesCount === 1 && state.pipes === dest || state.pipesCount > 1 && indexOf(state.pipes, dest) !== -1) && !cleanedUp) {
        debug('false write response, pause', src._readableState.awaitDrain);
        src._readableState.awaitDrain++;
        increasedAwaitDrain = true;
      }

      src.pause();
    }
  } // if the dest has an error, then stop piping into it.
  // however, don't suppress the throwing behavior for this.


  function onerror(er) {
    debug('onerror', er);
    unpipe();
    dest.removeListener('error', onerror);
    if (listenerCount$1(dest, 'error') === 0) dest.emit('error', er);
  } // Make sure our error handler is attached before userland ones.


  prependListener(dest, 'error', onerror); // Both close and finish should trigger unpipe, but only once.

  function onclose() {
    dest.removeListener('finish', onfinish);
    unpipe();
  }

  dest.once('close', onclose);

  function onfinish() {
    debug('onfinish');
    dest.removeListener('close', onclose);
    unpipe();
  }

  dest.once('finish', onfinish);

  function unpipe() {
    debug('unpipe');
    src.unpipe(dest);
  } // tell the dest that it's being piped to


  dest.emit('pipe', src); // start the flow if it hasn't been started already.

  if (!state.flowing) {
    debug('pipe resume');
    src.resume();
  }

  return dest;
};

function pipeOnDrain(src) {
  return function () {
    var state = src._readableState;
    debug('pipeOnDrain', state.awaitDrain);
    if (state.awaitDrain) state.awaitDrain--;

    if (state.awaitDrain === 0 && src.listeners('data').length) {
      state.flowing = true;
      flow(src);
    }
  };
}

Readable.prototype.unpipe = function (dest) {
  var state = this._readableState; // if we're not piping anywhere, then do nothing.

  if (state.pipesCount === 0) return this; // just one destination.  most common case.

  if (state.pipesCount === 1) {
    // passed in one, but it's not the right one.
    if (dest && dest !== state.pipes) return this;
    if (!dest) dest = state.pipes; // got a match.

    state.pipes = null;
    state.pipesCount = 0;
    state.flowing = false;
    if (dest) dest.emit('unpipe', this);
    return this;
  } // slow case. multiple pipe destinations.


  if (!dest) {
    // remove all.
    var dests = state.pipes;
    var len = state.pipesCount;
    state.pipes = null;
    state.pipesCount = 0;
    state.flowing = false;

    for (var _i = 0; _i < len; _i++) {
      dests[_i].emit('unpipe', this);
    }

    return this;
  } // try to find the right one.


  var i = indexOf(state.pipes, dest);
  if (i === -1) return this;
  state.pipes.splice(i, 1);
  state.pipesCount -= 1;
  if (state.pipesCount === 1) state.pipes = state.pipes[0];
  dest.emit('unpipe', this);
  return this;
}; // set up data events if they are asked for
// Ensure readable listeners eventually get something


Readable.prototype.on = function (ev, fn) {
  var res = EventEmitter.prototype.on.call(this, ev, fn);

  if (ev === 'data') {
    // Start flowing on next tick if stream isn't explicitly paused
    if (this._readableState.flowing !== false) this.resume();
  } else if (ev === 'readable') {
    var state = this._readableState;

    if (!state.endEmitted && !state.readableListening) {
      state.readableListening = state.needReadable = true;
      state.emittedReadable = false;

      if (!state.reading) {
        nextTick(nReadingNextTick, this);
      } else if (state.length) {
        emitReadable(this, state);
      }
    }
  }

  return res;
};

Readable.prototype.addListener = Readable.prototype.on;

function nReadingNextTick(self) {
  debug('readable nexttick read 0');
  self.read(0);
} // pause() and resume() are remnants of the legacy readable stream API
// If the user uses them, then switch into old mode.


Readable.prototype.resume = function () {
  var state = this._readableState;

  if (!state.flowing) {
    debug('resume');
    state.flowing = true;
    resume(this, state);
  }

  return this;
};

function resume(stream, state) {
  if (!state.resumeScheduled) {
    state.resumeScheduled = true;
    nextTick(resume_, stream, state);
  }
}

function resume_(stream, state) {
  if (!state.reading) {
    debug('resume read 0');
    stream.read(0);
  }

  state.resumeScheduled = false;
  state.awaitDrain = 0;
  stream.emit('resume');
  flow(stream);
  if (state.flowing && !state.reading) stream.read(0);
}

Readable.prototype.pause = function () {
  debug('call pause flowing=%j', this._readableState.flowing);

  if (false !== this._readableState.flowing) {
    debug('pause');
    this._readableState.flowing = false;
    this.emit('pause');
  }

  return this;
};

function flow(stream) {
  var state = stream._readableState;
  debug('flow', state.flowing);

  while (state.flowing && stream.read() !== null) {}
} // wrap an old-style stream as the async data source.
// This is *not* part of the readable stream interface.
// It is an ugly unfortunate mess of history.


Readable.prototype.wrap = function (stream) {
  var state = this._readableState;
  var paused = false;
  var self = this;
  stream.on('end', function () {
    debug('wrapped end');

    if (state.decoder && !state.ended) {
      var chunk = state.decoder.end();
      if (chunk && chunk.length) self.push(chunk);
    }

    self.push(null);
  });
  stream.on('data', function (chunk) {
    debug('wrapped data');
    if (state.decoder) chunk = state.decoder.write(chunk); // don't skip over falsy values in objectMode

    if (state.objectMode && (chunk === null || chunk === undefined)) return;else if (!state.objectMode && (!chunk || !chunk.length)) return;
    var ret = self.push(chunk);

    if (!ret) {
      paused = true;
      stream.pause();
    }
  }); // proxy all the other methods.
  // important when wrapping filters and duplexes.

  for (var i in stream) {
    if (this[i] === undefined && typeof stream[i] === 'function') {
      this[i] = function (method) {
        return function () {
          return stream[method].apply(stream, arguments);
        };
      }(i);
    }
  } // proxy certain important events.


  var events = ['error', 'close', 'destroy', 'pause', 'resume'];
  forEach(events, function (ev) {
    stream.on(ev, self.emit.bind(self, ev));
  }); // when we try to consume some more bytes, simply unpause the
  // underlying stream.

  self._read = function (n) {
    debug('wrapped _read', n);

    if (paused) {
      paused = false;
      stream.resume();
    }
  };

  return self;
}; // exposed for testing purposes only.


Readable._fromList = fromList; // Pluck off n bytes from an array of buffers.
// Length is the combined lengths of all the buffers in the list.
// This function is designed to be inlinable, so please take care when making
// changes to the function body.

function fromList(n, state) {
  // nothing buffered
  if (state.length === 0) return null;
  var ret;
  if (state.objectMode) ret = state.buffer.shift();else if (!n || n >= state.length) {
    // read it all, truncate the list
    if (state.decoder) ret = state.buffer.join('');else if (state.buffer.length === 1) ret = state.buffer.head.data;else ret = state.buffer.concat(state.length);
    state.buffer.clear();
  } else {
    // read part of list
    ret = fromListPartial(n, state.buffer, state.decoder);
  }
  return ret;
} // Extracts only enough buffered data to satisfy the amount requested.
// This function is designed to be inlinable, so please take care when making
// changes to the function body.


function fromListPartial(n, list, hasStrings) {
  var ret;

  if (n < list.head.data.length) {
    // slice is the same for buffers and strings
    ret = list.head.data.slice(0, n);
    list.head.data = list.head.data.slice(n);
  } else if (n === list.head.data.length) {
    // first chunk is a perfect match
    ret = list.shift();
  } else {
    // result spans more than one buffer
    ret = hasStrings ? copyFromBufferString(n, list) : copyFromBuffer(n, list);
  }

  return ret;
} // Copies a specified amount of characters from the list of buffered data
// chunks.
// This function is designed to be inlinable, so please take care when making
// changes to the function body.


function copyFromBufferString(n, list) {
  var p = list.head;
  var c = 1;
  var ret = p.data;
  n -= ret.length;

  while (p = p.next) {
    var str = p.data;
    var nb = n > str.length ? str.length : n;
    if (nb === str.length) ret += str;else ret += str.slice(0, n);
    n -= nb;

    if (n === 0) {
      if (nb === str.length) {
        ++c;
        if (p.next) list.head = p.next;else list.head = list.tail = null;
      } else {
        list.head = p;
        p.data = str.slice(nb);
      }

      break;
    }

    ++c;
  }

  list.length -= c;
  return ret;
} // Copies a specified amount of bytes from the list of buffered data chunks.
// This function is designed to be inlinable, so please take care when making
// changes to the function body.


function copyFromBuffer(n, list) {
  var ret = Buffer.allocUnsafe(n);
  var p = list.head;
  var c = 1;
  p.data.copy(ret);
  n -= p.data.length;

  while (p = p.next) {
    var buf = p.data;
    var nb = n > buf.length ? buf.length : n;
    buf.copy(ret, ret.length - n, 0, nb);
    n -= nb;

    if (n === 0) {
      if (nb === buf.length) {
        ++c;
        if (p.next) list.head = p.next;else list.head = list.tail = null;
      } else {
        list.head = p;
        p.data = buf.slice(nb);
      }

      break;
    }

    ++c;
  }

  list.length -= c;
  return ret;
}

function endReadable(stream) {
  var state = stream._readableState; // If we get here before consuming all the bytes, then that is a
  // bug in node.  Should never happen.

  if (state.length > 0) throw new Error('"endReadable()" called on non-empty stream');

  if (!state.endEmitted) {
    state.ended = true;
    nextTick(endReadableNT, state, stream);
  }
}

function endReadableNT(state, stream) {
  // Check that we didn't get one last unshift.
  if (!state.endEmitted && state.length === 0) {
    state.endEmitted = true;
    stream.readable = false;
    stream.emit('end');
  }
}

function forEach(xs, f) {
  for (var i = 0, l = xs.length; i < l; i++) {
    f(xs[i], i);
  }
}

function indexOf(xs, x) {
  for (var i = 0, l = xs.length; i < l; i++) {
    if (xs[i] === x) return i;
  }

  return -1;
}

// A bit simpler than readable streams.
Writable.WritableState = WritableState;
inherits$1(Writable, EventEmitter);

function nop() {}

function WriteReq(chunk, encoding, cb) {
  this.chunk = chunk;
  this.encoding = encoding;
  this.callback = cb;
  this.next = null;
}

function WritableState(options, stream) {
  Object.defineProperty(this, 'buffer', {
    get: deprecate(function () {
      return this.getBuffer();
    }, '_writableState.buffer is deprecated. Use _writableState.getBuffer ' + 'instead.')
  });
  options = options || {}; // object stream flag to indicate whether or not this stream
  // contains buffers or objects.

  this.objectMode = !!options.objectMode;
  if (stream instanceof Duplex) this.objectMode = this.objectMode || !!options.writableObjectMode; // the point at which write() starts returning false
  // Note: 0 is a valid value, means that we always return false if
  // the entire buffer is not flushed immediately on write()

  var hwm = options.highWaterMark;
  var defaultHwm = this.objectMode ? 16 : 16 * 1024;
  this.highWaterMark = hwm || hwm === 0 ? hwm : defaultHwm; // cast to ints.

  this.highWaterMark = ~~this.highWaterMark;
  this.needDrain = false; // at the start of calling end()

  this.ending = false; // when end() has been called, and returned

  this.ended = false; // when 'finish' is emitted

  this.finished = false; // should we decode strings into buffers before passing to _write?
  // this is here so that some node-core streams can optimize string
  // handling at a lower level.

  var noDecode = options.decodeStrings === false;
  this.decodeStrings = !noDecode; // Crypto is kind of old and crusty.  Historically, its default string
  // encoding is 'binary' so we have to make this configurable.
  // Everything else in the universe uses 'utf8', though.

  this.defaultEncoding = options.defaultEncoding || 'utf8'; // not an actual buffer we keep track of, but a measurement
  // of how much we're waiting to get pushed to some underlying
  // socket or file.

  this.length = 0; // a flag to see when we're in the middle of a write.

  this.writing = false; // when true all writes will be buffered until .uncork() call

  this.corked = 0; // a flag to be able to tell if the onwrite cb is called immediately,
  // or on a later tick.  We set this to true at first, because any
  // actions that shouldn't happen until "later" should generally also
  // not happen before the first write call.

  this.sync = true; // a flag to know if we're processing previously buffered items, which
  // may call the _write() callback in the same tick, so that we don't
  // end up in an overlapped onwrite situation.

  this.bufferProcessing = false; // the callback that's passed to _write(chunk,cb)

  this.onwrite = function (er) {
    onwrite(stream, er);
  }; // the callback that the user supplies to write(chunk,encoding,cb)


  this.writecb = null; // the amount that is being written when _write is called.

  this.writelen = 0;
  this.bufferedRequest = null;
  this.lastBufferedRequest = null; // number of pending user-supplied write callbacks
  // this must be 0 before 'finish' can be emitted

  this.pendingcb = 0; // emit prefinish if the only thing we're waiting for is _write cbs
  // This is relevant for synchronous Transform streams

  this.prefinished = false; // True if the error was already emitted and should not be thrown again

  this.errorEmitted = false; // count buffered requests

  this.bufferedRequestCount = 0; // allocate the first CorkedRequest, there is always
  // one allocated and free to use, and we maintain at most two

  this.corkedRequestsFree = new CorkedRequest(this);
}

WritableState.prototype.getBuffer = function writableStateGetBuffer() {
  var current = this.bufferedRequest;
  var out = [];

  while (current) {
    out.push(current);
    current = current.next;
  }

  return out;
};
function Writable(options) {
  // Writable ctor is applied to Duplexes, though they're not
  // instanceof Writable, they're instanceof Readable.
  if (!(this instanceof Writable) && !(this instanceof Duplex)) return new Writable(options);
  this._writableState = new WritableState(options, this); // legacy.

  this.writable = true;

  if (options) {
    if (typeof options.write === 'function') this._write = options.write;
    if (typeof options.writev === 'function') this._writev = options.writev;
  }

  EventEmitter.call(this);
} // Otherwise people can pipe Writable streams, which is just wrong.

Writable.prototype.pipe = function () {
  this.emit('error', new Error('Cannot pipe, not readable'));
};

function writeAfterEnd(stream, cb) {
  var er = new Error('write after end'); // TODO: defer error events consistently everywhere, not just the cb

  stream.emit('error', er);
  nextTick(cb, er);
} // If we get something that is not a buffer, string, null, or undefined,
// and we're not in objectMode, then that's an error.
// Otherwise stream chunks are all considered to be of length=1, and the
// watermarks determine how many objects to keep in the buffer, rather than
// how many bytes or characters.


function validChunk(stream, state, chunk, cb) {
  var valid = true;
  var er = false; // Always throw error if a null is written
  // if we are not in object mode then throw
  // if it is not a buffer, string, or undefined.

  if (chunk === null) {
    er = new TypeError('May not write null values to stream');
  } else if (!Buffer.isBuffer(chunk) && typeof chunk !== 'string' && chunk !== undefined && !state.objectMode) {
    er = new TypeError('Invalid non-string/buffer chunk');
  }

  if (er) {
    stream.emit('error', er);
    nextTick(cb, er);
    valid = false;
  }

  return valid;
}

Writable.prototype.write = function (chunk, encoding, cb) {
  var state = this._writableState;
  var ret = false;

  if (typeof encoding === 'function') {
    cb = encoding;
    encoding = null;
  }

  if (Buffer.isBuffer(chunk)) encoding = 'buffer';else if (!encoding) encoding = state.defaultEncoding;
  if (typeof cb !== 'function') cb = nop;
  if (state.ended) writeAfterEnd(this, cb);else if (validChunk(this, state, chunk, cb)) {
    state.pendingcb++;
    ret = writeOrBuffer(this, state, chunk, encoding, cb);
  }
  return ret;
};

Writable.prototype.cork = function () {
  var state = this._writableState;
  state.corked++;
};

Writable.prototype.uncork = function () {
  var state = this._writableState;

  if (state.corked) {
    state.corked--;
    if (!state.writing && !state.corked && !state.finished && !state.bufferProcessing && state.bufferedRequest) clearBuffer(this, state);
  }
};

Writable.prototype.setDefaultEncoding = function setDefaultEncoding(encoding) {
  // node::ParseEncoding() requires lower case.
  if (typeof encoding === 'string') encoding = encoding.toLowerCase();
  if (!(['hex', 'utf8', 'utf-8', 'ascii', 'binary', 'base64', 'ucs2', 'ucs-2', 'utf16le', 'utf-16le', 'raw'].indexOf((encoding + '').toLowerCase()) > -1)) throw new TypeError('Unknown encoding: ' + encoding);
  this._writableState.defaultEncoding = encoding;
  return this;
};

function decodeChunk(state, chunk, encoding) {
  if (!state.objectMode && state.decodeStrings !== false && typeof chunk === 'string') {
    chunk = Buffer.from(chunk, encoding);
  }

  return chunk;
} // if we're already writing something, then just put this
// in the queue, and wait our turn.  Otherwise, call _write
// If we return false, then we need a drain event, so set that flag.


function writeOrBuffer(stream, state, chunk, encoding, cb) {
  chunk = decodeChunk(state, chunk, encoding);
  if (Buffer.isBuffer(chunk)) encoding = 'buffer';
  var len = state.objectMode ? 1 : chunk.length;
  state.length += len;
  var ret = state.length < state.highWaterMark; // we must ensure that previous needDrain will not be reset to false.

  if (!ret) state.needDrain = true;

  if (state.writing || state.corked) {
    var last = state.lastBufferedRequest;
    state.lastBufferedRequest = new WriteReq(chunk, encoding, cb);

    if (last) {
      last.next = state.lastBufferedRequest;
    } else {
      state.bufferedRequest = state.lastBufferedRequest;
    }

    state.bufferedRequestCount += 1;
  } else {
    doWrite(stream, state, false, len, chunk, encoding, cb);
  }

  return ret;
}

function doWrite(stream, state, writev, len, chunk, encoding, cb) {
  state.writelen = len;
  state.writecb = cb;
  state.writing = true;
  state.sync = true;
  if (writev) stream._writev(chunk, state.onwrite);else stream._write(chunk, encoding, state.onwrite);
  state.sync = false;
}

function onwriteError(stream, state, sync, er, cb) {
  --state.pendingcb;
  if (sync) nextTick(cb, er);else cb(er);
  stream._writableState.errorEmitted = true;
  stream.emit('error', er);
}

function onwriteStateUpdate(state) {
  state.writing = false;
  state.writecb = null;
  state.length -= state.writelen;
  state.writelen = 0;
}

function onwrite(stream, er) {
  var state = stream._writableState;
  var sync = state.sync;
  var cb = state.writecb;
  onwriteStateUpdate(state);
  if (er) onwriteError(stream, state, sync, er, cb);else {
    // Check if we're actually ready to finish, but don't emit yet
    var finished = needFinish(state);

    if (!finished && !state.corked && !state.bufferProcessing && state.bufferedRequest) {
      clearBuffer(stream, state);
    }

    if (sync) {
      /*<replacement>*/
      nextTick(afterWrite, stream, state, finished, cb);
      /*</replacement>*/
    } else {
      afterWrite(stream, state, finished, cb);
    }
  }
}

function afterWrite(stream, state, finished, cb) {
  if (!finished) onwriteDrain(stream, state);
  state.pendingcb--;
  cb();
  finishMaybe(stream, state);
} // Must force callback to be called on nextTick, so that we don't
// emit 'drain' before the write() consumer gets the 'false' return
// value, and has a chance to attach a 'drain' listener.


function onwriteDrain(stream, state) {
  if (state.length === 0 && state.needDrain) {
    state.needDrain = false;
    stream.emit('drain');
  }
} // if there's something in the buffer waiting, then process it


function clearBuffer(stream, state) {
  state.bufferProcessing = true;
  var entry = state.bufferedRequest;

  if (stream._writev && entry && entry.next) {
    // Fast case, write everything using _writev()
    var l = state.bufferedRequestCount;
    var buffer = new Array(l);
    var holder = state.corkedRequestsFree;
    holder.entry = entry;
    var count = 0;

    while (entry) {
      buffer[count] = entry;
      entry = entry.next;
      count += 1;
    }

    doWrite(stream, state, true, state.length, buffer, '', holder.finish); // doWrite is almost always async, defer these to save a bit of time
    // as the hot path ends with doWrite

    state.pendingcb++;
    state.lastBufferedRequest = null;

    if (holder.next) {
      state.corkedRequestsFree = holder.next;
      holder.next = null;
    } else {
      state.corkedRequestsFree = new CorkedRequest(state);
    }
  } else {
    // Slow case, write chunks one-by-one
    while (entry) {
      var chunk = entry.chunk;
      var encoding = entry.encoding;
      var cb = entry.callback;
      var len = state.objectMode ? 1 : chunk.length;
      doWrite(stream, state, false, len, chunk, encoding, cb);
      entry = entry.next; // if we didn't call the onwrite immediately, then
      // it means that we need to wait until it does.
      // also, that means that the chunk and cb are currently
      // being processed, so move the buffer counter past them.

      if (state.writing) {
        break;
      }
    }

    if (entry === null) state.lastBufferedRequest = null;
  }

  state.bufferedRequestCount = 0;
  state.bufferedRequest = entry;
  state.bufferProcessing = false;
}

Writable.prototype._write = function (chunk, encoding, cb) {
  cb(new Error('not implemented'));
};

Writable.prototype._writev = null;

Writable.prototype.end = function (chunk, encoding, cb) {
  var state = this._writableState;

  if (typeof chunk === 'function') {
    cb = chunk;
    chunk = null;
    encoding = null;
  } else if (typeof encoding === 'function') {
    cb = encoding;
    encoding = null;
  }

  if (chunk !== null && chunk !== undefined) this.write(chunk, encoding); // .end() fully uncorks

  if (state.corked) {
    state.corked = 1;
    this.uncork();
  } // ignore unnecessary end() calls.


  if (!state.ending && !state.finished) endWritable(this, state, cb);
};

function needFinish(state) {
  return state.ending && state.length === 0 && state.bufferedRequest === null && !state.finished && !state.writing;
}

function prefinish(stream, state) {
  if (!state.prefinished) {
    state.prefinished = true;
    stream.emit('prefinish');
  }
}

function finishMaybe(stream, state) {
  var need = needFinish(state);

  if (need) {
    if (state.pendingcb === 0) {
      prefinish(stream, state);
      state.finished = true;
      stream.emit('finish');
    } else {
      prefinish(stream, state);
    }
  }

  return need;
}

function endWritable(stream, state, cb) {
  state.ending = true;
  finishMaybe(stream, state);

  if (cb) {
    if (state.finished) nextTick(cb);else stream.once('finish', cb);
  }

  state.ended = true;
  stream.writable = false;
} // It seems a linked list but it is not
// there will be only 2 of these for each stream


function CorkedRequest(state) {
  var _this = this;

  this.next = null;
  this.entry = null;

  this.finish = function (err) {
    var entry = _this.entry;
    _this.entry = null;

    while (entry) {
      var cb = entry.callback;
      state.pendingcb--;
      cb(err);
      entry = entry.next;
    }

    if (state.corkedRequestsFree) {
      state.corkedRequestsFree.next = _this;
    } else {
      state.corkedRequestsFree = _this;
    }
  };
}

inherits$1(Duplex, Readable);
var keys = Object.keys(Writable.prototype);

for (var v = 0; v < keys.length; v++) {
  var method = keys[v];
  if (!Duplex.prototype[method]) Duplex.prototype[method] = Writable.prototype[method];
}
function Duplex(options) {
  if (!(this instanceof Duplex)) return new Duplex(options);
  Readable.call(this, options);
  Writable.call(this, options);
  if (options && options.readable === false) this.readable = false;
  if (options && options.writable === false) this.writable = false;
  this.allowHalfOpen = true;
  if (options && options.allowHalfOpen === false) this.allowHalfOpen = false;
  this.once('end', onend);
} // the no-half-open enforcer

function onend() {
  // if we allow half-open state, or if the writable side ended,
  // then we're ok.
  if (this.allowHalfOpen || this._writableState.ended) return; // no more data can be written.
  // But allow more writes to happen in this tick.

  nextTick(onEndNT, this);
}

function onEndNT(self) {
  self.end();
}

// a transform stream is a readable/writable stream where you do
inherits$1(Transform, Duplex);

function TransformState(stream) {
  this.afterTransform = function (er, data) {
    return afterTransform(stream, er, data);
  };

  this.needTransform = false;
  this.transforming = false;
  this.writecb = null;
  this.writechunk = null;
  this.writeencoding = null;
}

function afterTransform(stream, er, data) {
  var ts = stream._transformState;
  ts.transforming = false;
  var cb = ts.writecb;
  if (!cb) return stream.emit('error', new Error('no writecb in Transform class'));
  ts.writechunk = null;
  ts.writecb = null;
  if (data !== null && data !== undefined) stream.push(data);
  cb(er);
  var rs = stream._readableState;
  rs.reading = false;

  if (rs.needReadable || rs.length < rs.highWaterMark) {
    stream._read(rs.highWaterMark);
  }
}
function Transform(options) {
  if (!(this instanceof Transform)) return new Transform(options);
  Duplex.call(this, options);
  this._transformState = new TransformState(this); // when the writable side finishes, then flush out anything remaining.

  var stream = this; // start out asking for a readable event once data is transformed.

  this._readableState.needReadable = true; // we have implemented the _read method, and done the other things
  // that Readable wants before the first _read call, so unset the
  // sync guard flag.

  this._readableState.sync = false;

  if (options) {
    if (typeof options.transform === 'function') this._transform = options.transform;
    if (typeof options.flush === 'function') this._flush = options.flush;
  }

  this.once('prefinish', function () {
    if (typeof this._flush === 'function') this._flush(function (er) {
      done(stream, er);
    });else done(stream);
  });
}

Transform.prototype.push = function (chunk, encoding) {
  this._transformState.needTransform = false;
  return Duplex.prototype.push.call(this, chunk, encoding);
}; // This is the part where you do stuff!
// override this function in implementation classes.
// 'chunk' is an input chunk.
//
// Call `push(newChunk)` to pass along transformed output
// to the readable side.  You may call 'push' zero or more times.
//
// Call `cb(err)` when you are done with this chunk.  If you pass
// an error, then that'll put the hurt on the whole operation.  If you
// never call cb(), then you'll never get another chunk.


Transform.prototype._transform = function (chunk, encoding, cb) {
  throw new Error('Not implemented');
};

Transform.prototype._write = function (chunk, encoding, cb) {
  var ts = this._transformState;
  ts.writecb = cb;
  ts.writechunk = chunk;
  ts.writeencoding = encoding;

  if (!ts.transforming) {
    var rs = this._readableState;
    if (ts.needTransform || rs.needReadable || rs.length < rs.highWaterMark) this._read(rs.highWaterMark);
  }
}; // Doesn't matter what the args are here.
// _transform does all the work.
// That we got here means that the readable side wants more data.


Transform.prototype._read = function (n) {
  var ts = this._transformState;

  if (ts.writechunk !== null && ts.writecb && !ts.transforming) {
    ts.transforming = true;

    this._transform(ts.writechunk, ts.writeencoding, ts.afterTransform);
  } else {
    // mark that we need a transform, so that any data that comes in
    // will get processed, now that we've asked for it.
    ts.needTransform = true;
  }
};

function done(stream, er) {
  if (er) return stream.emit('error', er); // if there's nothing in the write buffer, then that means
  // that nothing more will ever be provided

  var ws = stream._writableState;
  var ts = stream._transformState;
  if (ws.length) throw new Error('Calling transform done when ws.length != 0');
  if (ts.transforming) throw new Error('Calling transform done when still transforming');
  return stream.push(null);
}

inherits$1(PassThrough, Transform);
function PassThrough(options) {
  if (!(this instanceof PassThrough)) return new PassThrough(options);
  Transform.call(this, options);
}

PassThrough.prototype._transform = function (chunk, encoding, cb) {
  cb(null, chunk);
};

inherits$1(Stream, EventEmitter);
Stream.Readable = Readable;
Stream.Writable = Writable;
Stream.Duplex = Duplex;
Stream.Transform = Transform;
Stream.PassThrough = PassThrough; // Backwards-compat with node 0.4.x

Stream.Stream = Stream;
// part of this class) is overridden in the Readable class.

function Stream() {
  EventEmitter.call(this);
}

Stream.prototype.pipe = function (dest, options) {
  var source = this;

  function ondata(chunk) {
    if (dest.writable) {
      if (false === dest.write(chunk) && source.pause) {
        source.pause();
      }
    }
  }

  source.on('data', ondata);

  function ondrain() {
    if (source.readable && source.resume) {
      source.resume();
    }
  }

  dest.on('drain', ondrain); // If the 'end' option is not supplied, dest.end() will be called when
  // source gets the 'end' or 'close' events.  Only dest.end() once.

  if (!dest._isStdio && (!options || options.end !== false)) {
    source.on('end', onend);
    source.on('close', onclose);
  }

  var didOnEnd = false;

  function onend() {
    if (didOnEnd) return;
    didOnEnd = true;
    dest.end();
  }

  function onclose() {
    if (didOnEnd) return;
    didOnEnd = true;
    if (typeof dest.destroy === 'function') dest.destroy();
  } // don't leave dangling pipes when there are errors.


  function onerror(er) {
    cleanup();

    if (EventEmitter.listenerCount(this, 'error') === 0) {
      throw er; // Unhandled stream error in pipe.
    }
  }

  source.on('error', onerror);
  dest.on('error', onerror); // remove all the event listeners that were added.

  function cleanup() {
    source.removeListener('data', ondata);
    dest.removeListener('drain', ondrain);
    source.removeListener('end', onend);
    source.removeListener('close', onclose);
    source.removeListener('error', onerror);
    dest.removeListener('error', onerror);
    source.removeListener('end', cleanup);
    source.removeListener('close', cleanup);
    dest.removeListener('close', cleanup);
  }

  source.on('end', cleanup);
  source.on('close', cleanup);
  dest.on('close', cleanup);
  dest.emit('pipe', source); // Allow for unix-like usage: A.pipe(B).pipe(C)

  return dest;
};

var Buffer$1 = bufferEs6.Buffer;
var safer = {};
var key;

for (key in bufferEs6) {
  if (!bufferEs6.hasOwnProperty(key)) continue;
  if (key === 'SlowBuffer' || key === 'Buffer') continue;
  safer[key] = bufferEs6[key];
}

var Safer = safer.Buffer = {};

for (key in Buffer$1) {
  if (!Buffer$1.hasOwnProperty(key)) continue;
  if (key === 'allocUnsafe' || key === 'allocUnsafeSlow') continue;
  Safer[key] = Buffer$1[key];
}

safer.Buffer.prototype = Buffer$1.prototype;

if (!Safer.from || Safer.from === Uint8Array.from) {
  Safer.from = function (value, encodingOrOffset, length) {
    if (typeof value === 'number') {
      throw new TypeError('The "value" argument must not be of type number. Received type ' + typeof value);
    }

    if (value && typeof value.length === 'undefined') {
      throw new TypeError('The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type ' + typeof value);
    }

    return Buffer$1(value, encodingOrOffset, length);
  };
}

if (!Safer.alloc) {
  Safer.alloc = function (size, fill, encoding) {
    if (typeof size !== 'number') {
      throw new TypeError('The "size" argument must be of type number. Received type ' + typeof size);
    }

    if (size < 0 || size >= 2 * (1 << 30)) {
      throw new RangeError('The value "' + size + '" is invalid for option "size"');
    }

    var buf = Buffer$1(size);

    if (!fill || fill.length === 0) {
      buf.fill(0);
    } else if (typeof encoding === 'string') {
      buf.fill(fill, encoding);
    } else {
      buf.fill(fill);
    }

    return buf;
  };
}

if (!safer.kStringMaxLength) {
  try {
    safer.kStringMaxLength = process.binding('buffer').kStringMaxLength;
  } catch (e) {// we can't determine kStringMaxLength in environments where process.binding
    // is unsupported, so let's not set it
  }
}

if (!safer.constants) {
  safer.constants = {
    MAX_LENGTH: safer.kMaxLength
  };

  if (safer.kStringMaxLength) {
    safer.constants.MAX_STRING_LENGTH = safer.kStringMaxLength;
  }
}

var safer_1 = safer;

var BOMChar = "\uFEFF";
var PrependBOM = PrependBOMWrapper;

function PrependBOMWrapper(encoder, options) {
  this.encoder = encoder;
  this.addBOM = true;
}

PrependBOMWrapper.prototype.write = function (str) {
  if (this.addBOM) {
    str = BOMChar + str;
    this.addBOM = false;
  }

  return this.encoder.write(str);
};

PrependBOMWrapper.prototype.end = function () {
  return this.encoder.end();
}; //------------------------------------------------------------------------------


var StripBOM = StripBOMWrapper;

function StripBOMWrapper(decoder, options) {
  this.decoder = decoder;
  this.pass = false;
  this.options = options || {};
}

StripBOMWrapper.prototype.write = function (buf) {
  var res = this.decoder.write(buf);
  if (this.pass || !res) return res;

  if (res[0] === BOMChar) {
    res = res.slice(1);
    if (typeof this.options.stripBOM === 'function') this.options.stripBOM();
  }

  this.pass = true;
  return res;
};

StripBOMWrapper.prototype.end = function () {
  return this.decoder.end();
};

var bomHandling = {
  PrependBOM: PrependBOM,
  StripBOM: StripBOM
};

var Buffer$2 = safer_1.Buffer; // Export Node.js internal encodings.

var internal = {
  // Encodings
  utf8: {
    type: "_internal",
    bomAware: true
  },
  cesu8: {
    type: "_internal",
    bomAware: true
  },
  unicode11utf8: "utf8",
  ucs2: {
    type: "_internal",
    bomAware: true
  },
  utf16le: "ucs2",
  binary: {
    type: "_internal"
  },
  base64: {
    type: "_internal"
  },
  hex: {
    type: "_internal"
  },
  // Codec.
  _internal: InternalCodec
}; //------------------------------------------------------------------------------

function InternalCodec(codecOptions, iconv) {
  this.enc = codecOptions.encodingName;
  this.bomAware = codecOptions.bomAware;
  if (this.enc === "base64") this.encoder = InternalEncoderBase64;else if (this.enc === "cesu8") {
    this.enc = "utf8"; // Use utf8 for decoding.

    this.encoder = InternalEncoderCesu8; // Add decoder for versions of Node not supporting CESU-8

    if (Buffer$2.from('eda0bdedb2a9', 'hex').toString() !== '💩') {
      this.decoder = InternalDecoderCesu8;
      this.defaultCharUnicode = iconv.defaultCharUnicode;
    }
  }
}

InternalCodec.prototype.encoder = InternalEncoder;
InternalCodec.prototype.decoder = InternalDecoder; //------------------------------------------------------------------------------
// We use node.js internal decoder. Its signature is the same as ours.

var StringDecoder = string_decoder.StringDecoder;
if (!StringDecoder.prototype.end) // Node v0.8 doesn't have this method.
  StringDecoder.prototype.end = function () {};

function InternalDecoder(options, codec) {
  StringDecoder.call(this, codec.enc);
}

InternalDecoder.prototype = StringDecoder.prototype; //------------------------------------------------------------------------------
// Encoder is mostly trivial

function InternalEncoder(options, codec) {
  this.enc = codec.enc;
}

InternalEncoder.prototype.write = function (str) {
  return Buffer$2.from(str, this.enc);
};

InternalEncoder.prototype.end = function () {}; //------------------------------------------------------------------------------
// Except base64 encoder, which must keep its state.


function InternalEncoderBase64(options, codec) {
  this.prevStr = '';
}

InternalEncoderBase64.prototype.write = function (str) {
  str = this.prevStr + str;
  var completeQuads = str.length - str.length % 4;
  this.prevStr = str.slice(completeQuads);
  str = str.slice(0, completeQuads);
  return Buffer$2.from(str, "base64");
};

InternalEncoderBase64.prototype.end = function () {
  return Buffer$2.from(this.prevStr, "base64");
}; //------------------------------------------------------------------------------
// CESU-8 encoder is also special.


function InternalEncoderCesu8(options, codec) {}

InternalEncoderCesu8.prototype.write = function (str) {
  var buf = Buffer$2.alloc(str.length * 3),
      bufIdx = 0;

  for (var i = 0; i < str.length; i++) {
    var charCode = str.charCodeAt(i); // Naive implementation, but it works because CESU-8 is especially easy
    // to convert from UTF-16 (which all JS strings are encoded in).

    if (charCode < 0x80) buf[bufIdx++] = charCode;else if (charCode < 0x800) {
      buf[bufIdx++] = 0xC0 + (charCode >>> 6);
      buf[bufIdx++] = 0x80 + (charCode & 0x3f);
    } else {
      // charCode will always be < 0x10000 in javascript.
      buf[bufIdx++] = 0xE0 + (charCode >>> 12);
      buf[bufIdx++] = 0x80 + (charCode >>> 6 & 0x3f);
      buf[bufIdx++] = 0x80 + (charCode & 0x3f);
    }
  }

  return buf.slice(0, bufIdx);
};

InternalEncoderCesu8.prototype.end = function () {}; //------------------------------------------------------------------------------
// CESU-8 decoder is not implemented in Node v4.0+


function InternalDecoderCesu8(options, codec) {
  this.acc = 0;
  this.contBytes = 0;
  this.accBytes = 0;
  this.defaultCharUnicode = codec.defaultCharUnicode;
}

InternalDecoderCesu8.prototype.write = function (buf) {
  var acc = this.acc,
      contBytes = this.contBytes,
      accBytes = this.accBytes,
      res = '';

  for (var i = 0; i < buf.length; i++) {
    var curByte = buf[i];

    if ((curByte & 0xC0) !== 0x80) {
      // Leading byte
      if (contBytes > 0) {
        // Previous code is invalid
        res += this.defaultCharUnicode;
        contBytes = 0;
      }

      if (curByte < 0x80) {
        // Single-byte code
        res += String.fromCharCode(curByte);
      } else if (curByte < 0xE0) {
        // Two-byte code
        acc = curByte & 0x1F;
        contBytes = 1;
        accBytes = 1;
      } else if (curByte < 0xF0) {
        // Three-byte code
        acc = curByte & 0x0F;
        contBytes = 2;
        accBytes = 1;
      } else {
        // Four or more are not supported for CESU-8.
        res += this.defaultCharUnicode;
      }
    } else {
      // Continuation byte
      if (contBytes > 0) {
        // We're waiting for it.
        acc = acc << 6 | curByte & 0x3f;
        contBytes--;
        accBytes++;

        if (contBytes === 0) {
          // Check for overlong encoding, but support Modified UTF-8 (encoding NULL as C0 80)
          if (accBytes === 2 && acc < 0x80 && acc > 0) res += this.defaultCharUnicode;else if (accBytes === 3 && acc < 0x800) res += this.defaultCharUnicode;else // Actually add character.
            res += String.fromCharCode(acc);
        }
      } else {
        // Unexpected continuation byte
        res += this.defaultCharUnicode;
      }
    }
  }

  this.acc = acc;
  this.contBytes = contBytes;
  this.accBytes = accBytes;
  return res;
};

InternalDecoderCesu8.prototype.end = function () {
  var res = 0;
  if (this.contBytes > 0) res += this.defaultCharUnicode;
  return res;
};

var Buffer$3 = safer_1.Buffer; // Note: UTF16-LE (or UCS2) codec is Node.js native. See encodings/internal.js
// == UTF16-BE codec. ==========================================================

var utf16be = Utf16BECodec;

function Utf16BECodec() {}

Utf16BECodec.prototype.encoder = Utf16BEEncoder;
Utf16BECodec.prototype.decoder = Utf16BEDecoder;
Utf16BECodec.prototype.bomAware = true; // -- Encoding

function Utf16BEEncoder() {}

Utf16BEEncoder.prototype.write = function (str) {
  var buf = Buffer$3.from(str, 'ucs2');

  for (var i = 0; i < buf.length; i += 2) {
    var tmp = buf[i];
    buf[i] = buf[i + 1];
    buf[i + 1] = tmp;
  }

  return buf;
};

Utf16BEEncoder.prototype.end = function () {}; // -- Decoding


function Utf16BEDecoder() {
  this.overflowByte = -1;
}

Utf16BEDecoder.prototype.write = function (buf) {
  if (buf.length == 0) return '';
  var buf2 = Buffer$3.alloc(buf.length + 1),
      i = 0,
      j = 0;

  if (this.overflowByte !== -1) {
    buf2[0] = buf[0];
    buf2[1] = this.overflowByte;
    i = 1;
    j = 2;
  }

  for (; i < buf.length - 1; i += 2, j += 2) {
    buf2[j] = buf[i + 1];
    buf2[j + 1] = buf[i];
  }

  this.overflowByte = i == buf.length - 1 ? buf[buf.length - 1] : -1;
  return buf2.slice(0, j).toString('ucs2');
};

Utf16BEDecoder.prototype.end = function () {}; // == UTF-16 codec =============================================================
// Decoder chooses automatically from UTF-16LE and UTF-16BE using BOM and space-based heuristic.
// Defaults to UTF-16LE, as it's prevalent and default in Node.
// http://en.wikipedia.org/wiki/UTF-16 and http://encoding.spec.whatwg.org/#utf-16le
// Decoder default can be changed: iconv.decode(buf, 'utf16', {defaultEncoding: 'utf-16be'});
// Encoder uses UTF-16LE and prepends BOM (which can be overridden with addBOM: false).


var utf16_1 = Utf16Codec;

function Utf16Codec(codecOptions, iconv) {
  this.iconv = iconv;
}

Utf16Codec.prototype.encoder = Utf16Encoder;
Utf16Codec.prototype.decoder = Utf16Decoder; // -- Encoding (pass-through)

function Utf16Encoder(options, codec) {
  options = options || {};
  if (options.addBOM === undefined) options.addBOM = true;
  this.encoder = codec.iconv.getEncoder('utf-16le', options);
}

Utf16Encoder.prototype.write = function (str) {
  return this.encoder.write(str);
};

Utf16Encoder.prototype.end = function () {
  return this.encoder.end();
}; // -- Decoding


function Utf16Decoder(options, codec) {
  this.decoder = null;
  this.initialBytes = [];
  this.initialBytesLen = 0;
  this.options = options || {};
  this.iconv = codec.iconv;
}

Utf16Decoder.prototype.write = function (buf) {
  if (!this.decoder) {
    // Codec is not chosen yet. Accumulate initial bytes.
    this.initialBytes.push(buf);
    this.initialBytesLen += buf.length;
    if (this.initialBytesLen < 16) // We need more bytes to use space heuristic (see below)
      return ''; // We have enough bytes -> detect endianness.

    var buf = Buffer$3.concat(this.initialBytes),
        encoding = detectEncoding(buf, this.options.defaultEncoding);
    this.decoder = this.iconv.getDecoder(encoding, this.options);
    this.initialBytes.length = this.initialBytesLen = 0;
  }

  return this.decoder.write(buf);
};

Utf16Decoder.prototype.end = function () {
  if (!this.decoder) {
    var buf = Buffer$3.concat(this.initialBytes),
        encoding = detectEncoding(buf, this.options.defaultEncoding);
    this.decoder = this.iconv.getDecoder(encoding, this.options);
    var res = this.decoder.write(buf),
        trail = this.decoder.end();
    return trail ? res + trail : res;
  }

  return this.decoder.end();
};

function detectEncoding(buf, defaultEncoding) {
  var enc = defaultEncoding || 'utf-16le';

  if (buf.length >= 2) {
    // Check BOM.
    if (buf[0] == 0xFE && buf[1] == 0xFF) // UTF-16BE BOM
      enc = 'utf-16be';else if (buf[0] == 0xFF && buf[1] == 0xFE) // UTF-16LE BOM
      enc = 'utf-16le';else {
      // No BOM found. Try to deduce encoding from initial content.
      // Most of the time, the content has ASCII chars (U+00**), but the opposite (U+**00) is uncommon.
      // So, we count ASCII as if it was LE or BE, and decide from that.
      var asciiCharsLE = 0,
          asciiCharsBE = 0,
          // Counts of chars in both positions
      _len = Math.min(buf.length - buf.length % 2, 64); // Len is always even.


      for (var i = 0; i < _len; i += 2) {
        if (buf[i] === 0 && buf[i + 1] !== 0) asciiCharsBE++;
        if (buf[i] !== 0 && buf[i + 1] === 0) asciiCharsLE++;
      }

      if (asciiCharsBE > asciiCharsLE) enc = 'utf-16be';else if (asciiCharsBE < asciiCharsLE) enc = 'utf-16le';
    }
  }

  return enc;
}

var utf16 = {
  utf16be: utf16be,
  utf16: utf16_1
};

var Buffer$4 = safer_1.Buffer; // UTF-7 codec, according to https://tools.ietf.org/html/rfc2152
// See also below a UTF-7-IMAP codec, according to http://tools.ietf.org/html/rfc3501#section-5.1.3

var utf7_1 = Utf7Codec;
var unicode11utf7 = 'utf7'; // Alias UNICODE-1-1-UTF-7

function Utf7Codec(codecOptions, iconv) {
  this.iconv = iconv;
}
Utf7Codec.prototype.encoder = Utf7Encoder;
Utf7Codec.prototype.decoder = Utf7Decoder;
Utf7Codec.prototype.bomAware = true; // -- Encoding

var nonDirectChars = /[^A-Za-z0-9'\(\),-\.\/:\? \n\r\t]+/g;

function Utf7Encoder(options, codec) {
  this.iconv = codec.iconv;
}

Utf7Encoder.prototype.write = function (str) {
  // Naive implementation.
  // Non-direct chars are encoded as "+<base64>-"; single "+" char is encoded as "+-".
  return Buffer$4.from(str.replace(nonDirectChars, function (chunk) {
    return "+" + (chunk === '+' ? '' : this.iconv.encode(chunk, 'utf16-be').toString('base64').replace(/=+$/, '')) + "-";
  }.bind(this)));
};

Utf7Encoder.prototype.end = function () {}; // -- Decoding


function Utf7Decoder(options, codec) {
  this.iconv = codec.iconv;
  this.inBase64 = false;
  this.base64Accum = '';
}

var base64Regex = /[A-Za-z0-9\/+]/;
var base64Chars = [];

for (var i = 0; i < 256; i++) {
  base64Chars[i] = base64Regex.test(String.fromCharCode(i));
}

var plusChar = '+'.charCodeAt(0),
    minusChar = '-'.charCodeAt(0),
    andChar = '&'.charCodeAt(0);

Utf7Decoder.prototype.write = function (buf) {
  var res = "",
      lastI = 0,
      inBase64 = this.inBase64,
      base64Accum = this.base64Accum; // The decoder is more involved as we must handle chunks in stream.

  for (var i = 0; i < buf.length; i++) {
    if (!inBase64) {
      // We're in direct mode.
      // Write direct chars until '+'
      if (buf[i] == plusChar) {
        res += this.iconv.decode(buf.slice(lastI, i), "ascii"); // Write direct chars.

        lastI = i + 1;
        inBase64 = true;
      }
    } else {
      // We decode base64.
      if (!base64Chars[buf[i]]) {
        // Base64 ended.
        if (i == lastI && buf[i] == minusChar) {
          // "+-" -> "+"
          res += "+";
        } else {
          var b64str = base64Accum + buf.slice(lastI, i).toString();
          res += this.iconv.decode(Buffer$4.from(b64str, 'base64'), "utf16-be");
        }

        if (buf[i] != minusChar) // Minus is absorbed after base64.
          i--;
        lastI = i + 1;
        inBase64 = false;
        base64Accum = '';
      }
    }
  }

  if (!inBase64) {
    res += this.iconv.decode(buf.slice(lastI), "ascii"); // Write direct chars.
  } else {
    var b64str = base64Accum + buf.slice(lastI).toString();
    var canBeDecoded = b64str.length - b64str.length % 8; // Minimal chunk: 2 quads -> 2x3 bytes -> 3 chars.

    base64Accum = b64str.slice(canBeDecoded); // The rest will be decoded in future.

    b64str = b64str.slice(0, canBeDecoded);
    res += this.iconv.decode(Buffer$4.from(b64str, 'base64'), "utf16-be");
  }

  this.inBase64 = inBase64;
  this.base64Accum = base64Accum;
  return res;
};

Utf7Decoder.prototype.end = function () {
  var res = "";
  if (this.inBase64 && this.base64Accum.length > 0) res = this.iconv.decode(Buffer$4.from(this.base64Accum, 'base64'), "utf16-be");
  this.inBase64 = false;
  this.base64Accum = '';
  return res;
}; // UTF-7-IMAP codec.
// RFC3501 Sec. 5.1.3 Modified UTF-7 (http://tools.ietf.org/html/rfc3501#section-5.1.3)
// Differences:
//  * Base64 part is started by "&" instead of "+"
//  * Direct characters are 0x20-0x7E, except "&" (0x26)
//  * In Base64, "," is used instead of "/"
//  * Base64 must not be used to represent direct characters.
//  * No implicit shift back from Base64 (should always end with '-')
//  * String must end in non-shifted position.
//  * "-&" while in base64 is not allowed.


var utf7imap = Utf7IMAPCodec;

function Utf7IMAPCodec(codecOptions, iconv) {
  this.iconv = iconv;
}
Utf7IMAPCodec.prototype.encoder = Utf7IMAPEncoder;
Utf7IMAPCodec.prototype.decoder = Utf7IMAPDecoder;
Utf7IMAPCodec.prototype.bomAware = true; // -- Encoding

function Utf7IMAPEncoder(options, codec) {
  this.iconv = codec.iconv;
  this.inBase64 = false;
  this.base64Accum = Buffer$4.alloc(6);
  this.base64AccumIdx = 0;
}

Utf7IMAPEncoder.prototype.write = function (str) {
  var inBase64 = this.inBase64,
      base64Accum = this.base64Accum,
      base64AccumIdx = this.base64AccumIdx,
      buf = Buffer$4.alloc(str.length * 5 + 10),
      bufIdx = 0;

  for (var i = 0; i < str.length; i++) {
    var uChar = str.charCodeAt(i);

    if (0x20 <= uChar && uChar <= 0x7E) {
      // Direct character or '&'.
      if (inBase64) {
        if (base64AccumIdx > 0) {
          bufIdx += buf.write(base64Accum.slice(0, base64AccumIdx).toString('base64').replace(/\//g, ',').replace(/=+$/, ''), bufIdx);
          base64AccumIdx = 0;
        }

        buf[bufIdx++] = minusChar; // Write '-', then go to direct mode.

        inBase64 = false;
      }

      if (!inBase64) {
        buf[bufIdx++] = uChar; // Write direct character

        if (uChar === andChar) // Ampersand -> '&-'
          buf[bufIdx++] = minusChar;
      }
    } else {
      // Non-direct character
      if (!inBase64) {
        buf[bufIdx++] = andChar; // Write '&', then go to base64 mode.

        inBase64 = true;
      }

      if (inBase64) {
        base64Accum[base64AccumIdx++] = uChar >> 8;
        base64Accum[base64AccumIdx++] = uChar & 0xFF;

        if (base64AccumIdx == base64Accum.length) {
          bufIdx += buf.write(base64Accum.toString('base64').replace(/\//g, ','), bufIdx);
          base64AccumIdx = 0;
        }
      }
    }
  }

  this.inBase64 = inBase64;
  this.base64AccumIdx = base64AccumIdx;
  return buf.slice(0, bufIdx);
};

Utf7IMAPEncoder.prototype.end = function () {
  var buf = Buffer$4.alloc(10),
      bufIdx = 0;

  if (this.inBase64) {
    if (this.base64AccumIdx > 0) {
      bufIdx += buf.write(this.base64Accum.slice(0, this.base64AccumIdx).toString('base64').replace(/\//g, ',').replace(/=+$/, ''), bufIdx);
      this.base64AccumIdx = 0;
    }

    buf[bufIdx++] = minusChar; // Write '-', then go to direct mode.

    this.inBase64 = false;
  }

  return buf.slice(0, bufIdx);
}; // -- Decoding


function Utf7IMAPDecoder(options, codec) {
  this.iconv = codec.iconv;
  this.inBase64 = false;
  this.base64Accum = '';
}

var base64IMAPChars = base64Chars.slice();
base64IMAPChars[','.charCodeAt(0)] = true;

Utf7IMAPDecoder.prototype.write = function (buf) {
  var res = "",
      lastI = 0,
      inBase64 = this.inBase64,
      base64Accum = this.base64Accum; // The decoder is more involved as we must handle chunks in stream.
  // It is forgiving, closer to standard UTF-7 (for example, '-' is optional at the end).

  for (var i = 0; i < buf.length; i++) {
    if (!inBase64) {
      // We're in direct mode.
      // Write direct chars until '&'
      if (buf[i] == andChar) {
        res += this.iconv.decode(buf.slice(lastI, i), "ascii"); // Write direct chars.

        lastI = i + 1;
        inBase64 = true;
      }
    } else {
      // We decode base64.
      if (!base64IMAPChars[buf[i]]) {
        // Base64 ended.
        if (i == lastI && buf[i] == minusChar) {
          // "&-" -> "&"
          res += "&";
        } else {
          var b64str = base64Accum + buf.slice(lastI, i).toString().replace(/,/g, '/');
          res += this.iconv.decode(Buffer$4.from(b64str, 'base64'), "utf16-be");
        }

        if (buf[i] != minusChar) // Minus may be absorbed after base64.
          i--;
        lastI = i + 1;
        inBase64 = false;
        base64Accum = '';
      }
    }
  }

  if (!inBase64) {
    res += this.iconv.decode(buf.slice(lastI), "ascii"); // Write direct chars.
  } else {
    var b64str = base64Accum + buf.slice(lastI).toString().replace(/,/g, '/');
    var canBeDecoded = b64str.length - b64str.length % 8; // Minimal chunk: 2 quads -> 2x3 bytes -> 3 chars.

    base64Accum = b64str.slice(canBeDecoded); // The rest will be decoded in future.

    b64str = b64str.slice(0, canBeDecoded);
    res += this.iconv.decode(Buffer$4.from(b64str, 'base64'), "utf16-be");
  }

  this.inBase64 = inBase64;
  this.base64Accum = base64Accum;
  return res;
};

Utf7IMAPDecoder.prototype.end = function () {
  var res = "";
  if (this.inBase64 && this.base64Accum.length > 0) res = this.iconv.decode(Buffer$4.from(this.base64Accum, 'base64'), "utf16-be");
  this.inBase64 = false;
  this.base64Accum = '';
  return res;
};

var utf7 = {
  utf7: utf7_1,
  unicode11utf7: unicode11utf7,
  utf7imap: utf7imap
};

var Buffer$5 = safer_1.Buffer; // Single-byte codec. Needs a 'chars' string parameter that contains 256 or 128 chars that
// correspond to encoded bytes (if 128 - then lower half is ASCII). 

var _sbcs = SBCSCodec;

function SBCSCodec(codecOptions, iconv) {
  if (!codecOptions) throw new Error("SBCS codec is called without the data."); // Prepare char buffer for decoding.

  if (!codecOptions.chars || codecOptions.chars.length !== 128 && codecOptions.chars.length !== 256) throw new Error("Encoding '" + codecOptions.type + "' has incorrect 'chars' (must be of len 128 or 256)");

  if (codecOptions.chars.length === 128) {
    var asciiString = "";

    for (var i = 0; i < 128; i++) {
      asciiString += String.fromCharCode(i);
    }

    codecOptions.chars = asciiString + codecOptions.chars;
  }

  this.decodeBuf = new Buffer$5.from(codecOptions.chars, 'ucs2'); // Encoding buffer.

  var encodeBuf = new Buffer$5.alloc(65536, iconv.defaultCharSingleByte.charCodeAt(0));

  for (var i = 0; i < codecOptions.chars.length; i++) {
    encodeBuf[codecOptions.chars.charCodeAt(i)] = i;
  }

  this.encodeBuf = encodeBuf;
}

SBCSCodec.prototype.encoder = SBCSEncoder;
SBCSCodec.prototype.decoder = SBCSDecoder;

function SBCSEncoder(options, codec) {
  this.encodeBuf = codec.encodeBuf;
}

SBCSEncoder.prototype.write = function (str) {
  var buf = Buffer$5.alloc(str.length);

  for (var i = 0; i < str.length; i++) {
    buf[i] = this.encodeBuf[str.charCodeAt(i)];
  }

  return buf;
};

SBCSEncoder.prototype.end = function () {};

function SBCSDecoder(options, codec) {
  this.decodeBuf = codec.decodeBuf;
}

SBCSDecoder.prototype.write = function (buf) {
  // Strings are immutable in JS -> we use ucs2 buffer to speed up computations.
  var decodeBuf = this.decodeBuf;
  var newBuf = Buffer$5.alloc(buf.length * 2);
  var idx1 = 0,
      idx2 = 0;

  for (var i = 0; i < buf.length; i++) {
    idx1 = buf[i] * 2;
    idx2 = i * 2;
    newBuf[idx2] = decodeBuf[idx1];
    newBuf[idx2 + 1] = decodeBuf[idx1 + 1];
  }

  return newBuf.toString('ucs2');
};

SBCSDecoder.prototype.end = function () {};

var sbcsCodec = {
  _sbcs: _sbcs
};

var sbcsData = {
  // Not supported by iconv, not sure why.
  "10029": "maccenteuro",
  "maccenteuro": {
    "type": "_sbcs",
    "chars": "ÄĀāÉĄÖÜáąČäčĆćéŹźĎíďĒēĖóėôöõúĚěü†°Ę£§•¶ß®©™ę¨≠ģĮįĪ≤≥īĶ∂∑łĻļĽľĹĺŅņŃ¬√ńŇ∆«»… ňŐÕőŌ–—“”‘’÷◊ōŔŕŘ‹›řŖŗŠ‚„šŚśÁŤťÍŽžŪÓÔūŮÚůŰűŲųÝýķŻŁżĢˇ"
  },
  "808": "cp808",
  "ibm808": "cp808",
  "cp808": {
    "type": "_sbcs",
    "chars": "АБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯабвгдежзийклмноп░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀рстуфхцчшщъыьэюяЁёЄєЇїЎў°∙·√№€■ "
  },
  // Aliases of generated encodings.
  "ascii8bit": "ascii",
  "usascii": "ascii",
  "ansix34": "ascii",
  "ansix341968": "ascii",
  "ansix341986": "ascii",
  "csascii": "ascii",
  "cp367": "ascii",
  "ibm367": "ascii",
  "isoir6": "ascii",
  "iso646us": "ascii",
  "iso646irv": "ascii",
  "us": "ascii",
  "latin1": "iso88591",
  "latin2": "iso88592",
  "latin3": "iso88593",
  "latin4": "iso88594",
  "latin5": "iso88599",
  "latin6": "iso885910",
  "latin7": "iso885913",
  "latin8": "iso885914",
  "latin9": "iso885915",
  "latin10": "iso885916",
  "csisolatin1": "iso88591",
  "csisolatin2": "iso88592",
  "csisolatin3": "iso88593",
  "csisolatin4": "iso88594",
  "csisolatincyrillic": "iso88595",
  "csisolatinarabic": "iso88596",
  "csisolatingreek": "iso88597",
  "csisolatinhebrew": "iso88598",
  "csisolatin5": "iso88599",
  "csisolatin6": "iso885910",
  "l1": "iso88591",
  "l2": "iso88592",
  "l3": "iso88593",
  "l4": "iso88594",
  "l5": "iso88599",
  "l6": "iso885910",
  "l7": "iso885913",
  "l8": "iso885914",
  "l9": "iso885915",
  "l10": "iso885916",
  "isoir14": "iso646jp",
  "isoir57": "iso646cn",
  "isoir100": "iso88591",
  "isoir101": "iso88592",
  "isoir109": "iso88593",
  "isoir110": "iso88594",
  "isoir144": "iso88595",
  "isoir127": "iso88596",
  "isoir126": "iso88597",
  "isoir138": "iso88598",
  "isoir148": "iso88599",
  "isoir157": "iso885910",
  "isoir166": "tis620",
  "isoir179": "iso885913",
  "isoir199": "iso885914",
  "isoir203": "iso885915",
  "isoir226": "iso885916",
  "cp819": "iso88591",
  "ibm819": "iso88591",
  "cyrillic": "iso88595",
  "arabic": "iso88596",
  "arabic8": "iso88596",
  "ecma114": "iso88596",
  "asmo708": "iso88596",
  "greek": "iso88597",
  "greek8": "iso88597",
  "ecma118": "iso88597",
  "elot928": "iso88597",
  "hebrew": "iso88598",
  "hebrew8": "iso88598",
  "turkish": "iso88599",
  "turkish8": "iso88599",
  "thai": "iso885911",
  "thai8": "iso885911",
  "celtic": "iso885914",
  "celtic8": "iso885914",
  "isoceltic": "iso885914",
  "tis6200": "tis620",
  "tis62025291": "tis620",
  "tis62025330": "tis620",
  "10000": "macroman",
  "10006": "macgreek",
  "10007": "maccyrillic",
  "10079": "maciceland",
  "10081": "macturkish",
  "cspc8codepage437": "cp437",
  "cspc775baltic": "cp775",
  "cspc850multilingual": "cp850",
  "cspcp852": "cp852",
  "cspc862latinhebrew": "cp862",
  "cpgr": "cp869",
  "msee": "cp1250",
  "mscyrl": "cp1251",
  "msansi": "cp1252",
  "msgreek": "cp1253",
  "msturk": "cp1254",
  "mshebr": "cp1255",
  "msarab": "cp1256",
  "winbaltrim": "cp1257",
  "cp20866": "koi8r",
  "20866": "koi8r",
  "ibm878": "koi8r",
  "cskoi8r": "koi8r",
  "cp21866": "koi8u",
  "21866": "koi8u",
  "ibm1168": "koi8u",
  "strk10482002": "rk1048",
  "tcvn5712": "tcvn",
  "tcvn57121": "tcvn",
  "gb198880": "iso646cn",
  "cn": "iso646cn",
  "csiso14jisc6220ro": "iso646jp",
  "jisc62201969ro": "iso646jp",
  "jp": "iso646jp",
  "cshproman8": "hproman8",
  "r8": "hproman8",
  "roman8": "hproman8",
  "xroman8": "hproman8",
  "ibm1051": "hproman8",
  "mac": "macintosh",
  "csmacintosh": "macintosh"
};

var sbcsDataGenerated = {
  "437": "cp437",
  "737": "cp737",
  "775": "cp775",
  "850": "cp850",
  "852": "cp852",
  "855": "cp855",
  "856": "cp856",
  "857": "cp857",
  "858": "cp858",
  "860": "cp860",
  "861": "cp861",
  "862": "cp862",
  "863": "cp863",
  "864": "cp864",
  "865": "cp865",
  "866": "cp866",
  "869": "cp869",
  "874": "windows874",
  "922": "cp922",
  "1046": "cp1046",
  "1124": "cp1124",
  "1125": "cp1125",
  "1129": "cp1129",
  "1133": "cp1133",
  "1161": "cp1161",
  "1162": "cp1162",
  "1163": "cp1163",
  "1250": "windows1250",
  "1251": "windows1251",
  "1252": "windows1252",
  "1253": "windows1253",
  "1254": "windows1254",
  "1255": "windows1255",
  "1256": "windows1256",
  "1257": "windows1257",
  "1258": "windows1258",
  "28591": "iso88591",
  "28592": "iso88592",
  "28593": "iso88593",
  "28594": "iso88594",
  "28595": "iso88595",
  "28596": "iso88596",
  "28597": "iso88597",
  "28598": "iso88598",
  "28599": "iso88599",
  "28600": "iso885910",
  "28601": "iso885911",
  "28603": "iso885913",
  "28604": "iso885914",
  "28605": "iso885915",
  "28606": "iso885916",
  "windows874": {
    "type": "_sbcs",
    "chars": "€����…�����������‘’“”•–—�������� กขฃคฅฆงจฉชซฌญฎฏฐฑฒณดตถทธนบปผฝพฟภมยรฤลฦวศษสหฬอฮฯะัาำิีึืฺุู����฿เแโใไๅๆ็่้๊๋์ํ๎๏๐๑๒๓๔๕๖๗๘๙๚๛����"
  },
  "win874": "windows874",
  "cp874": "windows874",
  "windows1250": {
    "type": "_sbcs",
    "chars": "€�‚�„…†‡�‰Š‹ŚŤŽŹ�‘’“”•–—�™š›śťžź ˇ˘Ł¤Ą¦§¨©Ş«¬­®Ż°±˛ł´µ¶·¸ąş»Ľ˝ľżŔÁÂĂÄĹĆÇČÉĘËĚÍÎĎĐŃŇÓÔŐÖ×ŘŮÚŰÜÝŢßŕáâăäĺćçčéęëěíîďđńňóôőö÷řůúűüýţ˙"
  },
  "win1250": "windows1250",
  "cp1250": "windows1250",
  "windows1251": {
    "type": "_sbcs",
    "chars": "ЂЃ‚ѓ„…†‡€‰Љ‹ЊЌЋЏђ‘’“”•–—�™љ›њќћџ ЎўЈ¤Ґ¦§Ё©Є«¬­®Ї°±Ііґµ¶·ё№є»јЅѕїАБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯабвгдежзийклмнопрстуфхцчшщъыьэюя"
  },
  "win1251": "windows1251",
  "cp1251": "windows1251",
  "windows1252": {
    "type": "_sbcs",
    "chars": "€�‚ƒ„…†‡ˆ‰Š‹Œ�Ž��‘’“”•–—˜™š›œ�žŸ ¡¢£¤¥¦§¨©ª«¬­®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖ×ØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõö÷øùúûüýþÿ"
  },
  "win1252": "windows1252",
  "cp1252": "windows1252",
  "windows1253": {
    "type": "_sbcs",
    "chars": "€�‚ƒ„…†‡�‰�‹�����‘’“”•–—�™�›���� ΅Ά£¤¥¦§¨©�«¬­®―°±²³΄µ¶·ΈΉΊ»Ό½ΎΏΐΑΒΓΔΕΖΗΘΙΚΛΜΝΞΟΠΡ�ΣΤΥΦΧΨΩΪΫάέήίΰαβγδεζηθικλμνξοπρςστυφχψωϊϋόύώ�"
  },
  "win1253": "windows1253",
  "cp1253": "windows1253",
  "windows1254": {
    "type": "_sbcs",
    "chars": "€�‚ƒ„…†‡ˆ‰Š‹Œ����‘’“”•–—˜™š›œ��Ÿ ¡¢£¤¥¦§¨©ª«¬­®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏĞÑÒÓÔÕÖ×ØÙÚÛÜİŞßàáâãäåæçèéêëìíîïğñòóôõö÷øùúûüışÿ"
  },
  "win1254": "windows1254",
  "cp1254": "windows1254",
  "windows1255": {
    "type": "_sbcs",
    "chars": "€�‚ƒ„…†‡ˆ‰�‹�����‘’“”•–—˜™�›���� ¡¢£₪¥¦§¨©×«¬­®¯°±²³´µ¶·¸¹÷»¼½¾¿ְֱֲֳִֵֶַָֹֺֻּֽ־ֿ׀ׁׂ׃װױײ׳״�������אבגדהוזחטיךכלםמןנסעףפץצקרשת��‎‏�"
  },
  "win1255": "windows1255",
  "cp1255": "windows1255",
  "windows1256": {
    "type": "_sbcs",
    "chars": "€پ‚ƒ„…†‡ˆ‰ٹ‹Œچژڈگ‘’“”•–—ک™ڑ›œ‌‍ں ،¢£¤¥¦§¨©ھ«¬­®¯°±²³´µ¶·¸¹؛»¼½¾؟ہءآأؤإئابةتثجحخدذرزسشصض×طظعغـفقكàلâمنهوçèéêëىيîïًٌٍَôُِ÷ّùْûü‎‏ے"
  },
  "win1256": "windows1256",
  "cp1256": "windows1256",
  "windows1257": {
    "type": "_sbcs",
    "chars": "€�‚�„…†‡�‰�‹�¨ˇ¸�‘’“”•–—�™�›�¯˛� �¢£¤�¦§Ø©Ŗ«¬­®Æ°±²³´µ¶·ø¹ŗ»¼½¾æĄĮĀĆÄÅĘĒČÉŹĖĢĶĪĻŠŃŅÓŌÕÖ×ŲŁŚŪÜŻŽßąįāćäåęēčéźėģķīļšńņóōõö÷ųłśūüżž˙"
  },
  "win1257": "windows1257",
  "cp1257": "windows1257",
  "windows1258": {
    "type": "_sbcs",
    "chars": "€�‚ƒ„…†‡ˆ‰�‹Œ����‘’“”•–—˜™�›œ��Ÿ ¡¢£¤¥¦§¨©ª«¬­®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂĂÄÅÆÇÈÉÊË̀ÍÎÏĐÑ̉ÓÔƠÖ×ØÙÚÛÜỮßàáâăäåæçèéêë́íîïđṇ̃óôơö÷øùúûüư₫ÿ"
  },
  "win1258": "windows1258",
  "cp1258": "windows1258",
  "iso88591": {
    "type": "_sbcs",
    "chars": " ¡¢£¤¥¦§¨©ª«¬­®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖ×ØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõö÷øùúûüýþÿ"
  },
  "cp28591": "iso88591",
  "iso88592": {
    "type": "_sbcs",
    "chars": " Ą˘Ł¤ĽŚ§¨ŠŞŤŹ­ŽŻ°ą˛ł´ľśˇ¸šşťź˝žżŔÁÂĂÄĹĆÇČÉĘËĚÍÎĎĐŃŇÓÔŐÖ×ŘŮÚŰÜÝŢßŕáâăäĺćçčéęëěíîďđńňóôőö÷řůúűüýţ˙"
  },
  "cp28592": "iso88592",
  "iso88593": {
    "type": "_sbcs",
    "chars": " Ħ˘£¤�Ĥ§¨İŞĞĴ­�Ż°ħ²³´µĥ·¸ışğĵ½�żÀÁÂ�ÄĊĈÇÈÉÊËÌÍÎÏ�ÑÒÓÔĠÖ×ĜÙÚÛÜŬŜßàáâ�äċĉçèéêëìíîï�ñòóôġö÷ĝùúûüŭŝ˙"
  },
  "cp28593": "iso88593",
  "iso88594": {
    "type": "_sbcs",
    "chars": " ĄĸŖ¤ĨĻ§¨ŠĒĢŦ­Ž¯°ą˛ŗ´ĩļˇ¸šēģŧŊžŋĀÁÂÃÄÅÆĮČÉĘËĖÍÎĪĐŅŌĶÔÕÖ×ØŲÚÛÜŨŪßāáâãäåæįčéęëėíîīđņōķôõö÷øųúûüũū˙"
  },
  "cp28594": "iso88594",
  "iso88595": {
    "type": "_sbcs",
    "chars": " ЁЂЃЄЅІЇЈЉЊЋЌ­ЎЏАБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯабвгдежзийклмнопрстуфхцчшщъыьэюя№ёђѓєѕіїјљњћќ§ўџ"
  },
  "cp28595": "iso88595",
  "iso88596": {
    "type": "_sbcs",
    "chars": " ���¤�������،­�������������؛���؟�ءآأؤإئابةتثجحخدذرزسشصضطظعغ�����ـفقكلمنهوىيًٌٍَُِّْ�������������"
  },
  "cp28596": "iso88596",
  "iso88597": {
    "type": "_sbcs",
    "chars": " ‘’£€₯¦§¨©ͺ«¬­�―°±²³΄΅Ά·ΈΉΊ»Ό½ΎΏΐΑΒΓΔΕΖΗΘΙΚΛΜΝΞΟΠΡ�ΣΤΥΦΧΨΩΪΫάέήίΰαβγδεζηθικλμνξοπρςστυφχψωϊϋόύώ�"
  },
  "cp28597": "iso88597",
  "iso88598": {
    "type": "_sbcs",
    "chars": " �¢£¤¥¦§¨©×«¬­®¯°±²³´µ¶·¸¹÷»¼½¾��������������������������������‗אבגדהוזחטיךכלםמןנסעףפץצקרשת��‎‏�"
  },
  "cp28598": "iso88598",
  "iso88599": {
    "type": "_sbcs",
    "chars": " ¡¢£¤¥¦§¨©ª«¬­®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏĞÑÒÓÔÕÖ×ØÙÚÛÜİŞßàáâãäåæçèéêëìíîïğñòóôõö÷øùúûüışÿ"
  },
  "cp28599": "iso88599",
  "iso885910": {
    "type": "_sbcs",
    "chars": " ĄĒĢĪĨĶ§ĻĐŠŦŽ­ŪŊ°ąēģīĩķ·ļđšŧž―ūŋĀÁÂÃÄÅÆĮČÉĘËĖÍÎÏÐŅŌÓÔÕÖŨØŲÚÛÜÝÞßāáâãäåæįčéęëėíîïðņōóôõöũøųúûüýþĸ"
  },
  "cp28600": "iso885910",
  "iso885911": {
    "type": "_sbcs",
    "chars": " กขฃคฅฆงจฉชซฌญฎฏฐฑฒณดตถทธนบปผฝพฟภมยรฤลฦวศษสหฬอฮฯะัาำิีึืฺุู����฿เแโใไๅๆ็่้๊๋์ํ๎๏๐๑๒๓๔๕๖๗๘๙๚๛����"
  },
  "cp28601": "iso885911",
  "iso885913": {
    "type": "_sbcs",
    "chars": " ”¢£¤„¦§Ø©Ŗ«¬­®Æ°±²³“µ¶·ø¹ŗ»¼½¾æĄĮĀĆÄÅĘĒČÉŹĖĢĶĪĻŠŃŅÓŌÕÖ×ŲŁŚŪÜŻŽßąįāćäåęēčéźėģķīļšńņóōõö÷ųłśūüżž’"
  },
  "cp28603": "iso885913",
  "iso885914": {
    "type": "_sbcs",
    "chars": " Ḃḃ£ĊċḊ§Ẁ©ẂḋỲ­®ŸḞḟĠġṀṁ¶ṖẁṗẃṠỳẄẅṡÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏŴÑÒÓÔÕÖṪØÙÚÛÜÝŶßàáâãäåæçèéêëìíîïŵñòóôõöṫøùúûüýŷÿ"
  },
  "cp28604": "iso885914",
  "iso885915": {
    "type": "_sbcs",
    "chars": " ¡¢£€¥Š§š©ª«¬­®¯°±²³Žµ¶·ž¹º»ŒœŸ¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖ×ØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõö÷øùúûüýþÿ"
  },
  "cp28605": "iso885915",
  "iso885916": {
    "type": "_sbcs",
    "chars": " ĄąŁ€„Š§š©Ș«Ź­źŻ°±ČłŽ”¶·žčș»ŒœŸżÀÁÂĂÄĆÆÇÈÉÊËÌÍÎÏĐŃÒÓÔŐÖŚŰÙÚÛÜĘȚßàáâăäćæçèéêëìíîïđńòóôőöśűùúûüęțÿ"
  },
  "cp28606": "iso885916",
  "cp437": {
    "type": "_sbcs",
    "chars": "ÇüéâäàåçêëèïîìÄÅÉæÆôöòûùÿÖÜ¢£¥₧ƒáíóúñÑªº¿⌐¬½¼¡«»░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀αßΓπΣσµτΦΘΩδ∞φε∩≡±≥≤⌠⌡÷≈°∙·√ⁿ²■ "
  },
  "ibm437": "cp437",
  "csibm437": "cp437",
  "cp737": {
    "type": "_sbcs",
    "chars": "ΑΒΓΔΕΖΗΘΙΚΛΜΝΞΟΠΡΣΤΥΦΧΨΩαβγδεζηθικλμνξοπρσςτυφχψ░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀ωάέήϊίόύϋώΆΈΉΊΌΎΏ±≥≤ΪΫ÷≈°∙·√ⁿ²■ "
  },
  "ibm737": "cp737",
  "csibm737": "cp737",
  "cp775": {
    "type": "_sbcs",
    "chars": "ĆüéāäģåćłēŖŗīŹÄÅÉæÆōöĢ¢ŚśÖÜø£Ø×¤ĀĪóŻżź”¦©®¬½¼Ł«»░▒▓│┤ĄČĘĖ╣║╗╝ĮŠ┐└┴┬├─┼ŲŪ╚╔╩╦╠═╬Žąčęėįšųūž┘┌█▄▌▐▀ÓßŌŃõÕµńĶķĻļņĒŅ’­±“¾¶§÷„°∙·¹³²■ "
  },
  "ibm775": "cp775",
  "csibm775": "cp775",
  "cp850": {
    "type": "_sbcs",
    "chars": "ÇüéâäàåçêëèïîìÄÅÉæÆôöòûùÿÖÜø£Ø×ƒáíóúñÑªº¿®¬½¼¡«»░▒▓│┤ÁÂÀ©╣║╗╝¢¥┐└┴┬├─┼ãÃ╚╔╩╦╠═╬¤ðÐÊËÈıÍÎÏ┘┌█▄¦Ì▀ÓßÔÒõÕµþÞÚÛÙýÝ¯´­±‗¾¶§÷¸°¨·¹³²■ "
  },
  "ibm850": "cp850",
  "csibm850": "cp850",
  "cp852": {
    "type": "_sbcs",
    "chars": "ÇüéâäůćçłëŐőîŹÄĆÉĹĺôöĽľŚśÖÜŤťŁ×čáíóúĄąŽžĘę¬źČş«»░▒▓│┤ÁÂĚŞ╣║╗╝Żż┐└┴┬├─┼Ăă╚╔╩╦╠═╬¤đĐĎËďŇÍÎě┘┌█▄ŢŮ▀ÓßÔŃńňŠšŔÚŕŰýÝţ´­˝˛ˇ˘§÷¸°¨˙űŘř■ "
  },
  "ibm852": "cp852",
  "csibm852": "cp852",
  "cp855": {
    "type": "_sbcs",
    "chars": "ђЂѓЃёЁєЄѕЅіІїЇјЈљЉњЊћЋќЌўЎџЏюЮъЪаАбБцЦдДеЕфФгГ«»░▒▓│┤хХиИ╣║╗╝йЙ┐└┴┬├─┼кК╚╔╩╦╠═╬¤лЛмМнНоОп┘┌█▄Пя▀ЯрРсСтТуУжЖвВьЬ№­ыЫзЗшШэЭщЩчЧ§■ "
  },
  "ibm855": "cp855",
  "csibm855": "cp855",
  "cp856": {
    "type": "_sbcs",
    "chars": "אבגדהוזחטיךכלםמןנסעףפץצקרשת�£�×����������®¬½¼�«»░▒▓│┤���©╣║╗╝¢¥┐└┴┬├─┼��╚╔╩╦╠═╬¤���������┘┌█▄¦�▀������µ�������¯´­±‗¾¶§÷¸°¨·¹³²■ "
  },
  "ibm856": "cp856",
  "csibm856": "cp856",
  "cp857": {
    "type": "_sbcs",
    "chars": "ÇüéâäàåçêëèïîıÄÅÉæÆôöòûùİÖÜø£ØŞşáíóúñÑĞğ¿®¬½¼¡«»░▒▓│┤ÁÂÀ©╣║╗╝¢¥┐└┴┬├─┼ãÃ╚╔╩╦╠═╬¤ºªÊËÈ�ÍÎÏ┘┌█▄¦Ì▀ÓßÔÒõÕµ�×ÚÛÙìÿ¯´­±�¾¶§÷¸°¨·¹³²■ "
  },
  "ibm857": "cp857",
  "csibm857": "cp857",
  "cp858": {
    "type": "_sbcs",
    "chars": "ÇüéâäàåçêëèïîìÄÅÉæÆôöòûùÿÖÜø£Ø×ƒáíóúñÑªº¿®¬½¼¡«»░▒▓│┤ÁÂÀ©╣║╗╝¢¥┐└┴┬├─┼ãÃ╚╔╩╦╠═╬¤ðÐÊËÈ€ÍÎÏ┘┌█▄¦Ì▀ÓßÔÒõÕµþÞÚÛÙýÝ¯´­±‗¾¶§÷¸°¨·¹³²■ "
  },
  "ibm858": "cp858",
  "csibm858": "cp858",
  "cp860": {
    "type": "_sbcs",
    "chars": "ÇüéâãàÁçêÊèÍÔìÃÂÉÀÈôõòÚùÌÕÜ¢£Ù₧ÓáíóúñÑªº¿Ò¬½¼¡«»░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀αßΓπΣσµτΦΘΩδ∞φε∩≡±≥≤⌠⌡÷≈°∙·√ⁿ²■ "
  },
  "ibm860": "cp860",
  "csibm860": "cp860",
  "cp861": {
    "type": "_sbcs",
    "chars": "ÇüéâäàåçêëèÐðÞÄÅÉæÆôöþûÝýÖÜø£Ø₧ƒáíóúÁÍÓÚ¿⌐¬½¼¡«»░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀αßΓπΣσµτΦΘΩδ∞φε∩≡±≥≤⌠⌡÷≈°∙·√ⁿ²■ "
  },
  "ibm861": "cp861",
  "csibm861": "cp861",
  "cp862": {
    "type": "_sbcs",
    "chars": "אבגדהוזחטיךכלםמןנסעףפץצקרשת¢£¥₧ƒáíóúñÑªº¿⌐¬½¼¡«»░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀αßΓπΣσµτΦΘΩδ∞φε∩≡±≥≤⌠⌡÷≈°∙·√ⁿ²■ "
  },
  "ibm862": "cp862",
  "csibm862": "cp862",
  "cp863": {
    "type": "_sbcs",
    "chars": "ÇüéâÂà¶çêëèïî‗À§ÉÈÊôËÏûù¤ÔÜ¢£ÙÛƒ¦´óú¨¸³¯Î⌐¬½¼¾«»░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀αßΓπΣσµτΦΘΩδ∞φε∩≡±≥≤⌠⌡÷≈°∙·√ⁿ²■ "
  },
  "ibm863": "cp863",
  "csibm863": "cp863",
  "cp864": {
    "type": "_sbcs",
    "chars": "\0\x01\x02\x03\x04\x05\x06\x07\b\t\n\x0B\f\r\x0E\x0F\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1A\x1B\x1C\x1D\x1E\x1F !\"#$\u066A&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\x7F\xB0\xB7\u2219\u221A\u2592\u2500\u2502\u253C\u2524\u252C\u251C\u2534\u2510\u250C\u2514\u2518\u03B2\u221E\u03C6\xB1\xBD\xBC\u2248\xAB\xBB\uFEF7\uFEF8\uFFFD\uFFFD\uFEFB\uFEFC\uFFFD\xA0\xAD\uFE82\xA3\xA4\uFE84\uFFFD\uFFFD\uFE8E\uFE8F\uFE95\uFE99\u060C\uFE9D\uFEA1\uFEA5\u0660\u0661\u0662\u0663\u0664\u0665\u0666\u0667\u0668\u0669\uFED1\u061B\uFEB1\uFEB5\uFEB9\u061F\xA2\uFE80\uFE81\uFE83\uFE85\uFECA\uFE8B\uFE8D\uFE91\uFE93\uFE97\uFE9B\uFE9F\uFEA3\uFEA7\uFEA9\uFEAB\uFEAD\uFEAF\uFEB3\uFEB7\uFEBB\uFEBF\uFEC1\uFEC5\uFECB\uFECF\xA6\xAC\xF7\xD7\uFEC9\u0640\uFED3\uFED7\uFEDB\uFEDF\uFEE3\uFEE7\uFEEB\uFEED\uFEEF\uFEF3\uFEBD\uFECC\uFECE\uFECD\uFEE1\uFE7D\u0651\uFEE5\uFEE9\uFEEC\uFEF0\uFEF2\uFED0\uFED5\uFEF5\uFEF6\uFEDD\uFED9\uFEF1\u25A0\uFFFD"
  },
  "ibm864": "cp864",
  "csibm864": "cp864",
  "cp865": {
    "type": "_sbcs",
    "chars": "ÇüéâäàåçêëèïîìÄÅÉæÆôöòûùÿÖÜø£Ø₧ƒáíóúñÑªº¿⌐¬½¼¡«¤░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀αßΓπΣσµτΦΘΩδ∞φε∩≡±≥≤⌠⌡÷≈°∙·√ⁿ²■ "
  },
  "ibm865": "cp865",
  "csibm865": "cp865",
  "cp866": {
    "type": "_sbcs",
    "chars": "АБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯабвгдежзийклмноп░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀рстуфхцчшщъыьэюяЁёЄєЇїЎў°∙·√№¤■ "
  },
  "ibm866": "cp866",
  "csibm866": "cp866",
  "cp869": {
    "type": "_sbcs",
    "chars": "������Ά�·¬¦‘’Έ―ΉΊΪΌ��ΎΫ©Ώ²³ά£έήίϊΐόύΑΒΓΔΕΖΗ½ΘΙ«»░▒▓│┤ΚΛΜΝ╣║╗╝ΞΟ┐└┴┬├─┼ΠΡ╚╔╩╦╠═╬ΣΤΥΦΧΨΩαβγ┘┌█▄δε▀ζηθικλμνξοπρσςτ΄­±υφχ§ψ΅°¨ωϋΰώ■ "
  },
  "ibm869": "cp869",
  "csibm869": "cp869",
  "cp922": {
    "type": "_sbcs",
    "chars": " ¡¢£¤¥¦§¨©ª«¬­®‾°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏŠÑÒÓÔÕÖ×ØÙÚÛÜÝŽßàáâãäåæçèéêëìíîïšñòóôõö÷øùúûüýžÿ"
  },
  "ibm922": "cp922",
  "csibm922": "cp922",
  "cp1046": {
    "type": "_sbcs",
    "chars": "ﺈ×÷ﹱ■│─┐┌└┘ﹹﹻﹽﹿﹷﺊﻰﻳﻲﻎﻏﻐﻶﻸﻺﻼ ¤ﺋﺑﺗﺛﺟﺣ،­ﺧﺳ٠١٢٣٤٥٦٧٨٩ﺷ؛ﺻﺿﻊ؟ﻋءآأؤإئابةتثجحخدذرزسشصضطﻇعغﻌﺂﺄﺎﻓـفقكلمنهوىيًٌٍَُِّْﻗﻛﻟﻵﻷﻹﻻﻣﻧﻬﻩ�"
  },
  "ibm1046": "cp1046",
  "csibm1046": "cp1046",
  "cp1124": {
    "type": "_sbcs",
    "chars": " ЁЂҐЄЅІЇЈЉЊЋЌ­ЎЏАБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯабвгдежзийклмнопрстуфхцчшщъыьэюя№ёђґєѕіїјљњћќ§ўџ"
  },
  "ibm1124": "cp1124",
  "csibm1124": "cp1124",
  "cp1125": {
    "type": "_sbcs",
    "chars": "АБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯабвгдежзийклмноп░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀рстуфхцчшщъыьэюяЁёҐґЄєІіЇї·√№¤■ "
  },
  "ibm1125": "cp1125",
  "csibm1125": "cp1125",
  "cp1129": {
    "type": "_sbcs",
    "chars": " ¡¢£¤¥¦§œ©ª«¬­®¯°±²³Ÿµ¶·Œ¹º»¼½¾¿ÀÁÂĂÄÅÆÇÈÉÊË̀ÍÎÏĐÑ̉ÓÔƠÖ×ØÙÚÛÜỮßàáâăäåæçèéêë́íîïđṇ̃óôơö÷øùúûüư₫ÿ"
  },
  "ibm1129": "cp1129",
  "csibm1129": "cp1129",
  "cp1133": {
    "type": "_sbcs",
    "chars": " ກຂຄງຈສຊຍດຕຖທນບປຜຝພຟມຢຣລວຫອຮ���ຯະາຳິີຶືຸູຼັົຽ���ເແໂໃໄ່້໊໋໌ໍໆ�ໜໝ₭����������������໐໑໒໓໔໕໖໗໘໙��¢¬¦�"
  },
  "ibm1133": "cp1133",
  "csibm1133": "cp1133",
  "cp1161": {
    "type": "_sbcs",
    "chars": "��������������������������������่กขฃคฅฆงจฉชซฌญฎฏฐฑฒณดตถทธนบปผฝพฟภมยรฤลฦวศษสหฬอฮฯะัาำิีึืฺุู้๊๋€฿เแโใไๅๆ็่้๊๋์ํ๎๏๐๑๒๓๔๕๖๗๘๙๚๛¢¬¦ "
  },
  "ibm1161": "cp1161",
  "csibm1161": "cp1161",
  "cp1162": {
    "type": "_sbcs",
    "chars": "€…‘’“”•–— กขฃคฅฆงจฉชซฌญฎฏฐฑฒณดตถทธนบปผฝพฟภมยรฤลฦวศษสหฬอฮฯะัาำิีึืฺุู����฿เแโใไๅๆ็่้๊๋์ํ๎๏๐๑๒๓๔๕๖๗๘๙๚๛����"
  },
  "ibm1162": "cp1162",
  "csibm1162": "cp1162",
  "cp1163": {
    "type": "_sbcs",
    "chars": " ¡¢£€¥¦§œ©ª«¬­®¯°±²³Ÿµ¶·Œ¹º»¼½¾¿ÀÁÂĂÄÅÆÇÈÉÊË̀ÍÎÏĐÑ̉ÓÔƠÖ×ØÙÚÛÜỮßàáâăäåæçèéêë́íîïđṇ̃óôơö÷øùúûüư₫ÿ"
  },
  "ibm1163": "cp1163",
  "csibm1163": "cp1163",
  "maccroatian": {
    "type": "_sbcs",
    "chars": "ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûü†°¢£§•¶ß®Š™´¨≠ŽØ∞±≤≥∆µ∂∑∏š∫ªºΩžø¿¡¬√ƒ≈Ć«Č… ÀÃÕŒœĐ—“”‘’÷◊�©⁄¤‹›Æ»–·‚„‰ÂćÁčÈÍÎÏÌÓÔđÒÚÛÙıˆ˜¯πË˚¸Êæˇ"
  },
  "maccyrillic": {
    "type": "_sbcs",
    "chars": "АБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ†°¢£§•¶І®©™Ђђ≠Ѓѓ∞±≤≥іµ∂ЈЄєЇїЉљЊњјЅ¬√ƒ≈∆«»… ЋћЌќѕ–—“”‘’÷„ЎўЏџ№Ёёяабвгдежзийклмнопрстуфхцчшщъыьэю¤"
  },
  "macgreek": {
    "type": "_sbcs",
    "chars": "Ä¹²É³ÖÜ΅àâä΄¨çéèêë£™îï•½‰ôö¦­ùûü†ΓΔΘΛΞΠß®©ΣΪ§≠°·Α±≤≥¥ΒΕΖΗΙΚΜΦΫΨΩάΝ¬ΟΡ≈Τ«»… ΥΧΆΈœ–―“”‘’÷ΉΊΌΎέήίόΏύαβψδεφγηιξκλμνοπώρστθωςχυζϊϋΐΰ�"
  },
  "maciceland": {
    "type": "_sbcs",
    "chars": "ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûüÝ°¢£§•¶ß®©™´¨≠ÆØ∞±≤≥¥µ∂∑∏π∫ªºΩæø¿¡¬√ƒ≈∆«»… ÀÃÕŒœ–—“”‘’÷◊ÿŸ⁄¤ÐðÞþý·‚„‰ÂÊÁËÈÍÎÏÌÓÔ�ÒÚÛÙıˆ˜¯˘˙˚¸˝˛ˇ"
  },
  "macroman": {
    "type": "_sbcs",
    "chars": "ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûü†°¢£§•¶ß®©™´¨≠ÆØ∞±≤≥¥µ∂∑∏π∫ªºΩæø¿¡¬√ƒ≈∆«»… ÀÃÕŒœ–—“”‘’÷◊ÿŸ⁄¤‹›ﬁﬂ‡·‚„‰ÂÊÁËÈÍÎÏÌÓÔ�ÒÚÛÙıˆ˜¯˘˙˚¸˝˛ˇ"
  },
  "macromania": {
    "type": "_sbcs",
    "chars": "ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûü†°¢£§•¶ß®©™´¨≠ĂŞ∞±≤≥¥µ∂∑∏π∫ªºΩăş¿¡¬√ƒ≈∆«»… ÀÃÕŒœ–—“”‘’÷◊ÿŸ⁄¤‹›Ţţ‡·‚„‰ÂÊÁËÈÍÎÏÌÓÔ�ÒÚÛÙıˆ˜¯˘˙˚¸˝˛ˇ"
  },
  "macthai": {
    "type": "_sbcs",
    "chars": "«»…“”�•‘’� กขฃคฅฆงจฉชซฌญฎฏฐฑฒณดตถทธนบปผฝพฟภมยรฤลฦวศษสหฬอฮฯะัาำิีึืฺุู﻿​–—฿เแโใไๅๆ็่้๊๋์ํ™๏๐๑๒๓๔๕๖๗๘๙®©����"
  },
  "macturkish": {
    "type": "_sbcs",
    "chars": "ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûü†°¢£§•¶ß®©™´¨≠ÆØ∞±≤≥¥µ∂∑∏π∫ªºΩæø¿¡¬√ƒ≈∆«»… ÀÃÕŒœ–—“”‘’÷◊ÿŸĞğİıŞş‡·‚„‰ÂÊÁËÈÍÎÏÌÓÔ�ÒÚÛÙ�ˆ˜¯˘˙˚¸˝˛ˇ"
  },
  "macukraine": {
    "type": "_sbcs",
    "chars": "АБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ†°Ґ£§•¶І®©™Ђђ≠Ѓѓ∞±≤≥іµґЈЄєЇїЉљЊњјЅ¬√ƒ≈∆«»… ЋћЌќѕ–—“”‘’÷„ЎўЏџ№Ёёяабвгдежзийклмнопрстуфхцчшщъыьэю¤"
  },
  "koi8r": {
    "type": "_sbcs",
    "chars": "─│┌┐└┘├┤┬┴┼▀▄█▌▐░▒▓⌠■∙√≈≤≥ ⌡°²·÷═║╒ё╓╔╕╖╗╘╙╚╛╜╝╞╟╠╡Ё╢╣╤╥╦╧╨╩╪╫╬©юабцдефгхийклмнопярстужвьызшэщчъЮАБЦДЕФГХИЙКЛМНОПЯРСТУЖВЬЫЗШЭЩЧЪ"
  },
  "koi8u": {
    "type": "_sbcs",
    "chars": "─│┌┐└┘├┤┬┴┼▀▄█▌▐░▒▓⌠■∙√≈≤≥ ⌡°²·÷═║╒ёє╔ії╗╘╙╚╛ґ╝╞╟╠╡ЁЄ╣ІЇ╦╧╨╩╪Ґ╬©юабцдефгхийклмнопярстужвьызшэщчъЮАБЦДЕФГХИЙКЛМНОПЯРСТУЖВЬЫЗШЭЩЧЪ"
  },
  "koi8ru": {
    "type": "_sbcs",
    "chars": "─│┌┐└┘├┤┬┴┼▀▄█▌▐░▒▓⌠■∙√≈≤≥ ⌡°²·÷═║╒ёє╔ії╗╘╙╚╛ґў╞╟╠╡ЁЄ╣ІЇ╦╧╨╩╪ҐЎ©юабцдефгхийклмнопярстужвьызшэщчъЮАБЦДЕФГХИЙКЛМНОПЯРСТУЖВЬЫЗШЭЩЧЪ"
  },
  "koi8t": {
    "type": "_sbcs",
    "chars": "қғ‚Ғ„…†‡�‰ҳ‹ҲҷҶ�Қ‘’“”•–—�™�›�����ӯӮё¤ӣ¦§���«¬­®�°±²Ё�Ӣ¶·�№�»���©юабцдефгхийклмнопярстужвьызшэщчъЮАБЦДЕФГХИЙКЛМНОПЯРСТУЖВЬЫЗШЭЩЧЪ"
  },
  "armscii8": {
    "type": "_sbcs",
    "chars": " �և։)(»«—.՝,-֊…՜՛՞ԱաԲբԳգԴդԵեԶզԷէԸըԹթԺժԻիԼլԽխԾծԿկՀհՁձՂղՃճՄմՅյՆնՇշՈոՉչՊպՋջՌռՍսՎվՏտՐրՑցՒւՓփՔքՕօՖֆ՚�"
  },
  "rk1048": {
    "type": "_sbcs",
    "chars": "ЂЃ‚ѓ„…†‡€‰Љ‹ЊҚҺЏђ‘’“”•–—�™љ›њқһџ ҰұӘ¤Ө¦§Ё©Ғ«¬­®Ү°±Ііөµ¶·ё№ғ»әҢңүАБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯабвгдежзийклмнопрстуфхцчшщъыьэюя"
  },
  "tcvn": {
    "type": "_sbcs",
    "chars": "\0\xDA\u1EE4\x03\u1EEA\u1EEC\u1EEE\x07\b\t\n\x0B\f\r\x0E\x0F\x10\u1EE8\u1EF0\u1EF2\u1EF6\u1EF8\xDD\u1EF4\x18\x19\x1A\x1B\x1C\x1D\x1E\x1F !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\x7F\xC0\u1EA2\xC3\xC1\u1EA0\u1EB6\u1EAC\xC8\u1EBA\u1EBC\xC9\u1EB8\u1EC6\xCC\u1EC8\u0128\xCD\u1ECA\xD2\u1ECE\xD5\xD3\u1ECC\u1ED8\u1EDC\u1EDE\u1EE0\u1EDA\u1EE2\xD9\u1EE6\u0168\xA0\u0102\xC2\xCA\xD4\u01A0\u01AF\u0110\u0103\xE2\xEA\xF4\u01A1\u01B0\u0111\u1EB0\u0300\u0309\u0303\u0301\u0323\xE0\u1EA3\xE3\xE1\u1EA1\u1EB2\u1EB1\u1EB3\u1EB5\u1EAF\u1EB4\u1EAE\u1EA6\u1EA8\u1EAA\u1EA4\u1EC0\u1EB7\u1EA7\u1EA9\u1EAB\u1EA5\u1EAD\xE8\u1EC2\u1EBB\u1EBD\xE9\u1EB9\u1EC1\u1EC3\u1EC5\u1EBF\u1EC7\xEC\u1EC9\u1EC4\u1EBE\u1ED2\u0129\xED\u1ECB\xF2\u1ED4\u1ECF\xF5\xF3\u1ECD\u1ED3\u1ED5\u1ED7\u1ED1\u1ED9\u1EDD\u1EDF\u1EE1\u1EDB\u1EE3\xF9\u1ED6\u1EE7\u0169\xFA\u1EE5\u1EEB\u1EED\u1EEF\u1EE9\u1EF1\u1EF3\u1EF7\u1EF9\xFD\u1EF5\u1ED0"
  },
  "georgianacademy": {
    "type": "_sbcs",
    "chars": "‚ƒ„…†‡ˆ‰Š‹Œ‘’“”•–—˜™š›œŸ ¡¢£¤¥¦§¨©ª«¬­®¯°±²³´µ¶·¸¹º»¼½¾¿აბგდევზთიკლმნოპჟრსტუფქღყშჩცძწჭხჯჰჱჲჳჴჵჶçèéêëìíîïðñòóôõö÷øùúûüýþÿ"
  },
  "georgianps": {
    "type": "_sbcs",
    "chars": "‚ƒ„…†‡ˆ‰Š‹Œ‘’“”•–—˜™š›œŸ ¡¢£¤¥¦§¨©ª«¬­®¯°±²³´µ¶·¸¹º»¼½¾¿აბგდევზჱთიკლმნჲოპჟრსტჳუფქღყშჩცძწჭხჴჯჰჵæçèéêëìíîïðñòóôõö÷øùúûüýþÿ"
  },
  "pt154": {
    "type": "_sbcs",
    "chars": "ҖҒӮғ„…ҶҮҲүҠӢҢҚҺҸҗ‘’“”•–—ҳҷҡӣңқһҹ ЎўЈӨҘҰ§Ё©Ә«¬ӯ®Ҝ°ұІіҙө¶·ё№ә»јҪҫҝАБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯабвгдежзийклмнопрстуфхцчшщъыьэюя"
  },
  "viscii": {
    "type": "_sbcs",
    "chars": "\0\x01\u1EB2\x03\x04\u1EB4\u1EAA\x07\b\t\n\x0B\f\r\x0E\x0F\x10\x11\x12\x13\u1EF6\x15\x16\x17\x18\u1EF8\x1A\x1B\x1C\x1D\u1EF4\x1F !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\x7F\u1EA0\u1EAE\u1EB0\u1EB6\u1EA4\u1EA6\u1EA8\u1EAC\u1EBC\u1EB8\u1EBE\u1EC0\u1EC2\u1EC4\u1EC6\u1ED0\u1ED2\u1ED4\u1ED6\u1ED8\u1EE2\u1EDA\u1EDC\u1EDE\u1ECA\u1ECE\u1ECC\u1EC8\u1EE6\u0168\u1EE4\u1EF2\xD5\u1EAF\u1EB1\u1EB7\u1EA5\u1EA7\u1EA9\u1EAD\u1EBD\u1EB9\u1EBF\u1EC1\u1EC3\u1EC5\u1EC7\u1ED1\u1ED3\u1ED5\u1ED7\u1EE0\u01A0\u1ED9\u1EDD\u1EDF\u1ECB\u1EF0\u1EE8\u1EEA\u1EEC\u01A1\u1EDB\u01AF\xC0\xC1\xC2\xC3\u1EA2\u0102\u1EB3\u1EB5\xC8\xC9\xCA\u1EBA\xCC\xCD\u0128\u1EF3\u0110\u1EE9\xD2\xD3\xD4\u1EA1\u1EF7\u1EEB\u1EED\xD9\xDA\u1EF9\u1EF5\xDD\u1EE1\u01B0\xE0\xE1\xE2\xE3\u1EA3\u0103\u1EEF\u1EAB\xE8\xE9\xEA\u1EBB\xEC\xED\u0129\u1EC9\u0111\u1EF1\xF2\xF3\xF4\xF5\u1ECF\u1ECD\u1EE5\xF9\xFA\u0169\u1EE7\xFD\u1EE3\u1EEE"
  },
  "iso646cn": {
    "type": "_sbcs",
    "chars": "\0\x01\x02\x03\x04\x05\x06\x07\b\t\n\x0B\f\r\x0E\x0F\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1A\x1B\x1C\x1D\x1E\x1F !\"#\xA5%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}\u203E\x7F\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD"
  },
  "iso646jp": {
    "type": "_sbcs",
    "chars": "\0\x01\x02\x03\x04\x05\x06\x07\b\t\n\x0B\f\r\x0E\x0F\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1A\x1B\x1C\x1D\x1E\x1F !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\xA5]^_`abcdefghijklmnopqrstuvwxyz{|}\u203E\x7F\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD\uFFFD"
  },
  "hproman8": {
    "type": "_sbcs",
    "chars": " ÀÂÈÊËÎÏ´ˋˆ¨˜ÙÛ₤¯Ýý°ÇçÑñ¡¿¤£¥§ƒ¢âêôûáéóúàèòùäëöüÅîØÆåíøæÄìÖÜÉïßÔÁÃãÐðÍÌÓÒÕõŠšÚŸÿÞþ·µ¶¾—¼½ªº«■»±�"
  },
  "macintosh": {
    "type": "_sbcs",
    "chars": "ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûü†°¢£§•¶ß®©™´¨≠ÆØ∞±≤≥¥µ∂∑∏π∫ªºΩæø¿¡¬√ƒ≈∆«»… ÀÃÕŒœ–—“”‘’÷◊ÿŸ⁄¤‹›ﬁﬂ‡·‚„‰ÂÊÁËÈÍÎÏÌÓÔ�ÒÚÛÙıˆ˜¯˘˙˚¸˝˛ˇ"
  },
  "ascii": {
    "type": "_sbcs",
    "chars": "��������������������������������������������������������������������������������������������������������������������������������"
  },
  "tis620": {
    "type": "_sbcs",
    "chars": "���������������������������������กขฃคฅฆงจฉชซฌญฎฏฐฑฒณดตถทธนบปผฝพฟภมยรฤลฦวศษสหฬอฮฯะัาำิีึืฺุู����฿เแโใไๅๆ็่้๊๋์ํ๎๏๐๑๒๓๔๕๖๗๘๙๚๛����"
  }
};

var Buffer$6 = safer_1.Buffer; // Multibyte codec. In this scheme, a character is represented by 1 or more bytes.
// Our codec supports UTF-16 surrogates, extensions for GB18030 and unicode sequences.
// To save memory and loading time, we read table files only when requested.

var _dbcs = DBCSCodec;
var UNASSIGNED = -1,
    GB18030_CODE = -2,
    SEQ_START = -10,
    NODE_START = -1000,
    UNASSIGNED_NODE = new Array(0x100),
    DEF_CHAR = -1;

for (var i$1 = 0; i$1 < 0x100; i$1++) {
  UNASSIGNED_NODE[i$1] = UNASSIGNED;
} // Class DBCSCodec reads and initializes mapping tables.


function DBCSCodec(codecOptions, iconv) {
  this.encodingName = codecOptions.encodingName;
  if (!codecOptions) throw new Error("DBCS codec is called without the data.");
  if (!codecOptions.table) throw new Error("Encoding '" + this.encodingName + "' has no data."); // Load tables.

  var mappingTable = codecOptions.table(); // Decode tables: MBCS -> Unicode.
  // decodeTables is a trie, encoded as an array of arrays of integers. Internal arrays are trie nodes and all have len = 256.
  // Trie root is decodeTables[0].
  // Values: >=  0 -> unicode character code. can be > 0xFFFF
  //         == UNASSIGNED -> unknown/unassigned sequence.
  //         == GB18030_CODE -> this is the end of a GB18030 4-byte sequence.
  //         <= NODE_START -> index of the next node in our trie to process next byte.
  //         <= SEQ_START  -> index of the start of a character code sequence, in decodeTableSeq.

  this.decodeTables = [];
  this.decodeTables[0] = UNASSIGNED_NODE.slice(0); // Create root node.
  // Sometimes a MBCS char corresponds to a sequence of unicode chars. We store them as arrays of integers here. 

  this.decodeTableSeq = []; // Actual mapping tables consist of chunks. Use them to fill up decode tables.

  for (var i = 0; i < mappingTable.length; i++) {
    this._addDecodeChunk(mappingTable[i]);
  }

  this.defaultCharUnicode = iconv.defaultCharUnicode; // Encode tables: Unicode -> DBCS.
  // `encodeTable` is array mapping from unicode char to encoded char. All its values are integers for performance.
  // Because it can be sparse, it is represented as array of buckets by 256 chars each. Bucket can be null.
  // Values: >=  0 -> it is a normal char. Write the value (if <=256 then 1 byte, if <=65536 then 2 bytes, etc.).
  //         == UNASSIGNED -> no conversion found. Output a default char.
  //         <= SEQ_START  -> it's an index in encodeTableSeq, see below. The character starts a sequence.

  this.encodeTable = []; // `encodeTableSeq` is used when a sequence of unicode characters is encoded as a single code. We use a tree of
  // objects where keys correspond to characters in sequence and leafs are the encoded dbcs values. A special DEF_CHAR key
  // means end of sequence (needed when one sequence is a strict subsequence of another).
  // Objects are kept separately from encodeTable to increase performance.

  this.encodeTableSeq = []; // Some chars can be decoded, but need not be encoded.

  var skipEncodeChars = {};
  if (codecOptions.encodeSkipVals) for (var i = 0; i < codecOptions.encodeSkipVals.length; i++) {
    var val = codecOptions.encodeSkipVals[i];
    if (typeof val === 'number') skipEncodeChars[val] = true;else for (var j = val.from; j <= val.to; j++) {
      skipEncodeChars[j] = true;
    }
  } // Use decode trie to recursively fill out encode tables.

  this._fillEncodeTable(0, 0, skipEncodeChars); // Add more encoding pairs when needed.


  if (codecOptions.encodeAdd) {
    for (var uChar in codecOptions.encodeAdd) {
      if (Object.prototype.hasOwnProperty.call(codecOptions.encodeAdd, uChar)) this._setEncodeChar(uChar.charCodeAt(0), codecOptions.encodeAdd[uChar]);
    }
  }

  this.defCharSB = this.encodeTable[0][iconv.defaultCharSingleByte.charCodeAt(0)];
  if (this.defCharSB === UNASSIGNED) this.defCharSB = this.encodeTable[0]['?'];
  if (this.defCharSB === UNASSIGNED) this.defCharSB = "?".charCodeAt(0); // Load & create GB18030 tables when needed.

  if (typeof codecOptions.gb18030 === 'function') {
    this.gb18030 = codecOptions.gb18030(); // Load GB18030 ranges.
    // Add GB18030 decode tables.

    var thirdByteNodeIdx = this.decodeTables.length;
    var thirdByteNode = this.decodeTables[thirdByteNodeIdx] = UNASSIGNED_NODE.slice(0);
    var fourthByteNodeIdx = this.decodeTables.length;
    var fourthByteNode = this.decodeTables[fourthByteNodeIdx] = UNASSIGNED_NODE.slice(0);

    for (var i = 0x81; i <= 0xFE; i++) {
      var secondByteNodeIdx = NODE_START - this.decodeTables[0][i];
      var secondByteNode = this.decodeTables[secondByteNodeIdx];

      for (var j = 0x30; j <= 0x39; j++) {
        secondByteNode[j] = NODE_START - thirdByteNodeIdx;
      }
    }

    for (var i = 0x81; i <= 0xFE; i++) {
      thirdByteNode[i] = NODE_START - fourthByteNodeIdx;
    }

    for (var i = 0x30; i <= 0x39; i++) {
      fourthByteNode[i] = GB18030_CODE;
    }
  }
}

DBCSCodec.prototype.encoder = DBCSEncoder;
DBCSCodec.prototype.decoder = DBCSDecoder; // Decoder helpers

DBCSCodec.prototype._getDecodeTrieNode = function (addr) {
  var bytes = [];

  for (; addr > 0; addr >>= 8) {
    bytes.push(addr & 0xFF);
  }

  if (bytes.length == 0) bytes.push(0);
  var node = this.decodeTables[0];

  for (var i = bytes.length - 1; i > 0; i--) {
    // Traverse nodes deeper into the trie.
    var val = node[bytes[i]];

    if (val == UNASSIGNED) {
      // Create new node.
      node[bytes[i]] = NODE_START - this.decodeTables.length;
      this.decodeTables.push(node = UNASSIGNED_NODE.slice(0));
    } else if (val <= NODE_START) {
      // Existing node.
      node = this.decodeTables[NODE_START - val];
    } else throw new Error("Overwrite byte in " + this.encodingName + ", addr: " + addr.toString(16));
  }

  return node;
};

DBCSCodec.prototype._addDecodeChunk = function (chunk) {
  // First element of chunk is the hex mbcs code where we start.
  var curAddr = parseInt(chunk[0], 16); // Choose the decoding node where we'll write our chars.

  var writeTable = this._getDecodeTrieNode(curAddr);

  curAddr = curAddr & 0xFF; // Write all other elements of the chunk to the table.

  for (var k = 1; k < chunk.length; k++) {
    var part = chunk[k];

    if (typeof part === "string") {
      // String, write as-is.
      for (var l = 0; l < part.length;) {
        var code = part.charCodeAt(l++);

        if (0xD800 <= code && code < 0xDC00) {
          // Decode surrogate
          var codeTrail = part.charCodeAt(l++);
          if (0xDC00 <= codeTrail && codeTrail < 0xE000) writeTable[curAddr++] = 0x10000 + (code - 0xD800) * 0x400 + (codeTrail - 0xDC00);else throw new Error("Incorrect surrogate pair in " + this.encodingName + " at chunk " + chunk[0]);
        } else if (0x0FF0 < code && code <= 0x0FFF) {
          // Character sequence (our own encoding used)
          var len = 0xFFF - code + 2;
          var seq = [];

          for (var m = 0; m < len; m++) {
            seq.push(part.charCodeAt(l++));
          } // Simple variation: don't support surrogates or subsequences in seq.


          writeTable[curAddr++] = SEQ_START - this.decodeTableSeq.length;
          this.decodeTableSeq.push(seq);
        } else writeTable[curAddr++] = code; // Basic char

      }
    } else if (typeof part === "number") {
      // Integer, meaning increasing sequence starting with prev character.
      var charCode = writeTable[curAddr - 1] + 1;

      for (var l = 0; l < part; l++) {
        writeTable[curAddr++] = charCode++;
      }
    } else throw new Error("Incorrect type '" + typeof part + "' given in " + this.encodingName + " at chunk " + chunk[0]);
  }

  if (curAddr > 0xFF) throw new Error("Incorrect chunk in " + this.encodingName + " at addr " + chunk[0] + ": too long" + curAddr);
}; // Encoder helpers


DBCSCodec.prototype._getEncodeBucket = function (uCode) {
  var high = uCode >> 8; // This could be > 0xFF because of astral characters.

  if (this.encodeTable[high] === undefined) this.encodeTable[high] = UNASSIGNED_NODE.slice(0); // Create bucket on demand.

  return this.encodeTable[high];
};

DBCSCodec.prototype._setEncodeChar = function (uCode, dbcsCode) {
  var bucket = this._getEncodeBucket(uCode);

  var low = uCode & 0xFF;
  if (bucket[low] <= SEQ_START) this.encodeTableSeq[SEQ_START - bucket[low]][DEF_CHAR] = dbcsCode; // There's already a sequence, set a single-char subsequence of it.
  else if (bucket[low] == UNASSIGNED) bucket[low] = dbcsCode;
};

DBCSCodec.prototype._setEncodeSequence = function (seq, dbcsCode) {
  // Get the root of character tree according to first character of the sequence.
  var uCode = seq[0];

  var bucket = this._getEncodeBucket(uCode);

  var low = uCode & 0xFF;
  var node;

  if (bucket[low] <= SEQ_START) {
    // There's already a sequence with  - use it.
    node = this.encodeTableSeq[SEQ_START - bucket[low]];
  } else {
    // There was no sequence object - allocate a new one.
    node = {};
    if (bucket[low] !== UNASSIGNED) node[DEF_CHAR] = bucket[low]; // If a char was set before - make it a single-char subsequence.

    bucket[low] = SEQ_START - this.encodeTableSeq.length;
    this.encodeTableSeq.push(node);
  } // Traverse the character tree, allocating new nodes as needed.


  for (var j = 1; j < seq.length - 1; j++) {
    var oldVal = node[uCode];
    if (typeof oldVal === 'object') node = oldVal;else {
      node = node[uCode] = {};
      if (oldVal !== undefined) node[DEF_CHAR] = oldVal;
    }
  } // Set the leaf to given dbcsCode.


  uCode = seq[seq.length - 1];
  node[uCode] = dbcsCode;
};

DBCSCodec.prototype._fillEncodeTable = function (nodeIdx, prefix, skipEncodeChars) {
  var node = this.decodeTables[nodeIdx];

  for (var i = 0; i < 0x100; i++) {
    var uCode = node[i];
    var mbCode = prefix + i;
    if (skipEncodeChars[mbCode]) continue;
    if (uCode >= 0) this._setEncodeChar(uCode, mbCode);else if (uCode <= NODE_START) this._fillEncodeTable(NODE_START - uCode, mbCode << 8, skipEncodeChars);else if (uCode <= SEQ_START) this._setEncodeSequence(this.decodeTableSeq[SEQ_START - uCode], mbCode);
  }
}; // == Encoder ==================================================================


function DBCSEncoder(options, codec) {
  // Encoder state
  this.leadSurrogate = -1;
  this.seqObj = undefined; // Static data

  this.encodeTable = codec.encodeTable;
  this.encodeTableSeq = codec.encodeTableSeq;
  this.defaultCharSingleByte = codec.defCharSB;
  this.gb18030 = codec.gb18030;
}

DBCSEncoder.prototype.write = function (str) {
  var newBuf = Buffer$6.alloc(str.length * (this.gb18030 ? 4 : 3)),
      leadSurrogate = this.leadSurrogate,
      seqObj = this.seqObj,
      nextChar = -1,
      i = 0,
      j = 0;

  while (true) {
    // 0. Get next character.
    if (nextChar === -1) {
      if (i == str.length) break;
      var uCode = str.charCodeAt(i++);
    } else {
      var uCode = nextChar;
      nextChar = -1;
    } // 1. Handle surrogates.


    if (0xD800 <= uCode && uCode < 0xE000) {
      // Char is one of surrogates.
      if (uCode < 0xDC00) {
        // We've got lead surrogate.
        if (leadSurrogate === -1) {
          leadSurrogate = uCode;
          continue;
        } else {
          leadSurrogate = uCode; // Double lead surrogate found.

          uCode = UNASSIGNED;
        }
      } else {
        // We've got trail surrogate.
        if (leadSurrogate !== -1) {
          uCode = 0x10000 + (leadSurrogate - 0xD800) * 0x400 + (uCode - 0xDC00);
          leadSurrogate = -1;
        } else {
          // Incomplete surrogate pair - only trail surrogate found.
          uCode = UNASSIGNED;
        }
      }
    } else if (leadSurrogate !== -1) {
      // Incomplete surrogate pair - only lead surrogate found.
      nextChar = uCode;
      uCode = UNASSIGNED; // Write an error, then current char.

      leadSurrogate = -1;
    } // 2. Convert uCode character.


    var dbcsCode = UNASSIGNED;

    if (seqObj !== undefined && uCode != UNASSIGNED) {
      // We are in the middle of the sequence
      var resCode = seqObj[uCode];

      if (typeof resCode === 'object') {
        // Sequence continues.
        seqObj = resCode;
        continue;
      } else if (typeof resCode == 'number') {
        // Sequence finished. Write it.
        dbcsCode = resCode;
      } else if (resCode == undefined) {
        // Current character is not part of the sequence.
        // Try default character for this sequence
        resCode = seqObj[DEF_CHAR];

        if (resCode !== undefined) {
          dbcsCode = resCode; // Found. Write it.

          nextChar = uCode; // Current character will be written too in the next iteration.
        }
      }

      seqObj = undefined;
    } else if (uCode >= 0) {
      // Regular character
      var subtable = this.encodeTable[uCode >> 8];
      if (subtable !== undefined) dbcsCode = subtable[uCode & 0xFF];

      if (dbcsCode <= SEQ_START) {
        // Sequence start
        seqObj = this.encodeTableSeq[SEQ_START - dbcsCode];
        continue;
      }

      if (dbcsCode == UNASSIGNED && this.gb18030) {
        // Use GB18030 algorithm to find character(s) to write.
        var idx = findIdx(this.gb18030.uChars, uCode);

        if (idx != -1) {
          var dbcsCode = this.gb18030.gbChars[idx] + (uCode - this.gb18030.uChars[idx]);
          newBuf[j++] = 0x81 + Math.floor(dbcsCode / 12600);
          dbcsCode = dbcsCode % 12600;
          newBuf[j++] = 0x30 + Math.floor(dbcsCode / 1260);
          dbcsCode = dbcsCode % 1260;
          newBuf[j++] = 0x81 + Math.floor(dbcsCode / 10);
          dbcsCode = dbcsCode % 10;
          newBuf[j++] = 0x30 + dbcsCode;
          continue;
        }
      }
    } // 3. Write dbcsCode character.


    if (dbcsCode === UNASSIGNED) dbcsCode = this.defaultCharSingleByte;

    if (dbcsCode < 0x100) {
      newBuf[j++] = dbcsCode;
    } else if (dbcsCode < 0x10000) {
      newBuf[j++] = dbcsCode >> 8; // high byte

      newBuf[j++] = dbcsCode & 0xFF; // low byte
    } else {
      newBuf[j++] = dbcsCode >> 16;
      newBuf[j++] = dbcsCode >> 8 & 0xFF;
      newBuf[j++] = dbcsCode & 0xFF;
    }
  }

  this.seqObj = seqObj;
  this.leadSurrogate = leadSurrogate;
  return newBuf.slice(0, j);
};

DBCSEncoder.prototype.end = function () {
  if (this.leadSurrogate === -1 && this.seqObj === undefined) return; // All clean. Most often case.

  var newBuf = Buffer$6.alloc(10),
      j = 0;

  if (this.seqObj) {
    // We're in the sequence.
    var dbcsCode = this.seqObj[DEF_CHAR];

    if (dbcsCode !== undefined) {
      // Write beginning of the sequence.
      if (dbcsCode < 0x100) {
        newBuf[j++] = dbcsCode;
      } else {
        newBuf[j++] = dbcsCode >> 8; // high byte

        newBuf[j++] = dbcsCode & 0xFF; // low byte
      }
    }

    this.seqObj = undefined;
  }

  if (this.leadSurrogate !== -1) {
    // Incomplete surrogate pair - only lead surrogate found.
    newBuf[j++] = this.defaultCharSingleByte;
    this.leadSurrogate = -1;
  }

  return newBuf.slice(0, j);
}; // Export for testing


DBCSEncoder.prototype.findIdx = findIdx; // == Decoder ==================================================================

function DBCSDecoder(options, codec) {
  // Decoder state
  this.nodeIdx = 0;
  this.prevBuf = Buffer$6.alloc(0); // Static data

  this.decodeTables = codec.decodeTables;
  this.decodeTableSeq = codec.decodeTableSeq;
  this.defaultCharUnicode = codec.defaultCharUnicode;
  this.gb18030 = codec.gb18030;
}

DBCSDecoder.prototype.write = function (buf) {
  var newBuf = Buffer$6.alloc(buf.length * 2),
      nodeIdx = this.nodeIdx,
      prevBuf = this.prevBuf,
      prevBufOffset = this.prevBuf.length,
      seqStart = -this.prevBuf.length,
      // idx of the start of current parsed sequence.
  uCode;
  if (prevBufOffset > 0) // Make prev buf overlap a little to make it easier to slice later.
    prevBuf = Buffer$6.concat([prevBuf, buf.slice(0, 10)]);

  for (var i = 0, j = 0; i < buf.length; i++) {
    var curByte = i >= 0 ? buf[i] : prevBuf[i + prevBufOffset]; // Lookup in current trie node.

    var uCode = this.decodeTables[nodeIdx][curByte];

    if (uCode >= 0) ; else if (uCode === UNASSIGNED) {
      // Unknown char.
      // TODO: Callback with seq.
      //var curSeq = (seqStart >= 0) ? buf.slice(seqStart, i+1) : prevBuf.slice(seqStart + prevBufOffset, i+1 + prevBufOffset);
      i = seqStart; // Try to parse again, after skipping first byte of the sequence ('i' will be incremented by 'for' cycle).

      uCode = this.defaultCharUnicode.charCodeAt(0);
    } else if (uCode === GB18030_CODE) {
      var curSeq = seqStart >= 0 ? buf.slice(seqStart, i + 1) : prevBuf.slice(seqStart + prevBufOffset, i + 1 + prevBufOffset);
      var ptr = (curSeq[0] - 0x81) * 12600 + (curSeq[1] - 0x30) * 1260 + (curSeq[2] - 0x81) * 10 + (curSeq[3] - 0x30);
      var idx = findIdx(this.gb18030.gbChars, ptr);
      uCode = this.gb18030.uChars[idx] + ptr - this.gb18030.gbChars[idx];
    } else if (uCode <= NODE_START) {
      // Go to next trie node.
      nodeIdx = NODE_START - uCode;
      continue;
    } else if (uCode <= SEQ_START) {
      // Output a sequence of chars.
      var seq = this.decodeTableSeq[SEQ_START - uCode];

      for (var k = 0; k < seq.length - 1; k++) {
        uCode = seq[k];
        newBuf[j++] = uCode & 0xFF;
        newBuf[j++] = uCode >> 8;
      }

      uCode = seq[seq.length - 1];
    } else throw new Error("iconv-lite internal error: invalid decoding table value " + uCode + " at " + nodeIdx + "/" + curByte); // Write the character to buffer, handling higher planes using surrogate pair.


    if (uCode > 0xFFFF) {
      uCode -= 0x10000;
      var uCodeLead = 0xD800 + Math.floor(uCode / 0x400);
      newBuf[j++] = uCodeLead & 0xFF;
      newBuf[j++] = uCodeLead >> 8;
      uCode = 0xDC00 + uCode % 0x400;
    }

    newBuf[j++] = uCode & 0xFF;
    newBuf[j++] = uCode >> 8; // Reset trie node.

    nodeIdx = 0;
    seqStart = i + 1;
  }

  this.nodeIdx = nodeIdx;
  this.prevBuf = seqStart >= 0 ? buf.slice(seqStart) : prevBuf.slice(seqStart + prevBufOffset);
  return newBuf.slice(0, j).toString('ucs2');
};

DBCSDecoder.prototype.end = function () {
  var ret = ''; // Try to parse all remaining chars.

  while (this.prevBuf.length > 0) {
    // Skip 1 character in the buffer.
    ret += this.defaultCharUnicode;
    var buf = this.prevBuf.slice(1); // Parse remaining as usual.

    this.prevBuf = Buffer$6.alloc(0);
    this.nodeIdx = 0;
    if (buf.length > 0) ret += this.write(buf);
  }

  this.nodeIdx = 0;
  return ret;
}; // Binary search for GB18030. Returns largest i such that table[i] <= val.


function findIdx(table, val) {
  if (table[0] > val) return -1;
  var l = 0,
      r = table.length;

  while (l < r - 1) {
    // always table[l] <= val < table[r]
    var mid = l + Math.floor((r - l + 1) / 2);
    if (table[mid] <= val) l = mid;else r = mid;
  }

  return l;
}

var dbcsCodec = {
  _dbcs: _dbcs
};

var shiftjis = [
["0","\u0000",128],
["a1","｡",62],
["8140","　、。，．・：；？！゛゜´｀¨＾￣＿ヽヾゝゞ〃仝々〆〇ー―‐／＼～∥｜…‥‘’“”（）〔〕［］｛｝〈",9,"＋－±×"],
["8180","÷＝≠＜＞≦≧∞∴♂♀°′″℃￥＄￠￡％＃＆＊＠§☆★○●◎◇◆□■△▲▽▼※〒→←↑↓〓"],
["81b8","∈∋⊆⊇⊂⊃∪∩"],
["81c8","∧∨￢⇒⇔∀∃"],
["81da","∠⊥⌒∂∇≡≒≪≫√∽∝∵∫∬"],
["81f0","Å‰♯♭♪†‡¶"],
["81fc","◯"],
["824f","０",9],
["8260","Ａ",25],
["8281","ａ",25],
["829f","ぁ",82],
["8340","ァ",62],
["8380","ム",22],
["839f","Α",16,"Σ",6],
["83bf","α",16,"σ",6],
["8440","А",5,"ЁЖ",25],
["8470","а",5,"ёж",7],
["8480","о",17],
["849f","─│┌┐┘└├┬┤┴┼━┃┏┓┛┗┣┳┫┻╋┠┯┨┷┿┝┰┥┸╂"],
["8740","①",19,"Ⅰ",9],
["875f","㍉㌔㌢㍍㌘㌧㌃㌶㍑㍗㌍㌦㌣㌫㍊㌻㎜㎝㎞㎎㎏㏄㎡"],
["877e","㍻"],
["8780","〝〟№㏍℡㊤",4,"㈱㈲㈹㍾㍽㍼≒≡∫∮∑√⊥∠∟⊿∵∩∪"],
["889f","亜唖娃阿哀愛挨姶逢葵茜穐悪握渥旭葦芦鯵梓圧斡扱宛姐虻飴絢綾鮎或粟袷安庵按暗案闇鞍杏以伊位依偉囲夷委威尉惟意慰易椅為畏異移維緯胃萎衣謂違遺医井亥域育郁磯一壱溢逸稲茨芋鰯允印咽員因姻引飲淫胤蔭"],
["8940","院陰隠韻吋右宇烏羽迂雨卯鵜窺丑碓臼渦嘘唄欝蔚鰻姥厩浦瓜閏噂云運雲荏餌叡営嬰影映曳栄永泳洩瑛盈穎頴英衛詠鋭液疫益駅悦謁越閲榎厭円"],
["8980","園堰奄宴延怨掩援沿演炎焔煙燕猿縁艶苑薗遠鉛鴛塩於汚甥凹央奥往応押旺横欧殴王翁襖鴬鴎黄岡沖荻億屋憶臆桶牡乙俺卸恩温穏音下化仮何伽価佳加可嘉夏嫁家寡科暇果架歌河火珂禍禾稼箇花苛茄荷華菓蝦課嘩貨迦過霞蚊俄峨我牙画臥芽蛾賀雅餓駕介会解回塊壊廻快怪悔恢懐戒拐改"],
["8a40","魁晦械海灰界皆絵芥蟹開階貝凱劾外咳害崖慨概涯碍蓋街該鎧骸浬馨蛙垣柿蛎鈎劃嚇各廓拡撹格核殻獲確穫覚角赫較郭閣隔革学岳楽額顎掛笠樫"],
["8a80","橿梶鰍潟割喝恰括活渇滑葛褐轄且鰹叶椛樺鞄株兜竃蒲釜鎌噛鴨栢茅萱粥刈苅瓦乾侃冠寒刊勘勧巻喚堪姦完官寛干幹患感慣憾換敢柑桓棺款歓汗漢澗潅環甘監看竿管簡緩缶翰肝艦莞観諌貫還鑑間閑関陥韓館舘丸含岸巌玩癌眼岩翫贋雁頑顔願企伎危喜器基奇嬉寄岐希幾忌揮机旗既期棋棄"],
["8b40","機帰毅気汽畿祈季稀紀徽規記貴起軌輝飢騎鬼亀偽儀妓宜戯技擬欺犠疑祇義蟻誼議掬菊鞠吉吃喫桔橘詰砧杵黍却客脚虐逆丘久仇休及吸宮弓急救"],
["8b80","朽求汲泣灸球究窮笈級糾給旧牛去居巨拒拠挙渠虚許距鋸漁禦魚亨享京供侠僑兇競共凶協匡卿叫喬境峡強彊怯恐恭挟教橋況狂狭矯胸脅興蕎郷鏡響饗驚仰凝尭暁業局曲極玉桐粁僅勤均巾錦斤欣欽琴禁禽筋緊芹菌衿襟謹近金吟銀九倶句区狗玖矩苦躯駆駈駒具愚虞喰空偶寓遇隅串櫛釧屑屈"],
["8c40","掘窟沓靴轡窪熊隈粂栗繰桑鍬勲君薫訓群軍郡卦袈祁係傾刑兄啓圭珪型契形径恵慶慧憩掲携敬景桂渓畦稽系経継繋罫茎荊蛍計詣警軽頚鶏芸迎鯨"],
["8c80","劇戟撃激隙桁傑欠決潔穴結血訣月件倹倦健兼券剣喧圏堅嫌建憲懸拳捲検権牽犬献研硯絹県肩見謙賢軒遣鍵険顕験鹸元原厳幻弦減源玄現絃舷言諺限乎個古呼固姑孤己庫弧戸故枯湖狐糊袴股胡菰虎誇跨鈷雇顧鼓五互伍午呉吾娯後御悟梧檎瑚碁語誤護醐乞鯉交佼侯候倖光公功効勾厚口向"],
["8d40","后喉坑垢好孔孝宏工巧巷幸広庚康弘恒慌抗拘控攻昂晃更杭校梗構江洪浩港溝甲皇硬稿糠紅紘絞綱耕考肯肱腔膏航荒行衡講貢購郊酵鉱砿鋼閤降"],
["8d80","項香高鴻剛劫号合壕拷濠豪轟麹克刻告国穀酷鵠黒獄漉腰甑忽惚骨狛込此頃今困坤墾婚恨懇昏昆根梱混痕紺艮魂些佐叉唆嵯左差査沙瑳砂詐鎖裟坐座挫債催再最哉塞妻宰彩才採栽歳済災采犀砕砦祭斎細菜裁載際剤在材罪財冴坂阪堺榊肴咲崎埼碕鷺作削咋搾昨朔柵窄策索錯桜鮭笹匙冊刷"],
["8e40","察拶撮擦札殺薩雑皐鯖捌錆鮫皿晒三傘参山惨撒散桟燦珊産算纂蚕讃賛酸餐斬暫残仕仔伺使刺司史嗣四士始姉姿子屍市師志思指支孜斯施旨枝止"],
["8e80","死氏獅祉私糸紙紫肢脂至視詞詩試誌諮資賜雌飼歯事似侍児字寺慈持時次滋治爾璽痔磁示而耳自蒔辞汐鹿式識鴫竺軸宍雫七叱執失嫉室悉湿漆疾質実蔀篠偲柴芝屡蕊縞舎写射捨赦斜煮社紗者謝車遮蛇邪借勺尺杓灼爵酌釈錫若寂弱惹主取守手朱殊狩珠種腫趣酒首儒受呪寿授樹綬需囚収周"],
["8f40","宗就州修愁拾洲秀秋終繍習臭舟蒐衆襲讐蹴輯週酋酬集醜什住充十従戎柔汁渋獣縦重銃叔夙宿淑祝縮粛塾熟出術述俊峻春瞬竣舜駿准循旬楯殉淳"],
["8f80","準潤盾純巡遵醇順処初所暑曙渚庶緒署書薯藷諸助叙女序徐恕鋤除傷償勝匠升召哨商唱嘗奨妾娼宵将小少尚庄床廠彰承抄招掌捷昇昌昭晶松梢樟樵沼消渉湘焼焦照症省硝礁祥称章笑粧紹肖菖蒋蕉衝裳訟証詔詳象賞醤鉦鍾鐘障鞘上丈丞乗冗剰城場壌嬢常情擾条杖浄状畳穣蒸譲醸錠嘱埴飾"],
["9040","拭植殖燭織職色触食蝕辱尻伸信侵唇娠寝審心慎振新晋森榛浸深申疹真神秦紳臣芯薪親診身辛進針震人仁刃塵壬尋甚尽腎訊迅陣靭笥諏須酢図厨"],
["9080","逗吹垂帥推水炊睡粋翠衰遂酔錐錘随瑞髄崇嵩数枢趨雛据杉椙菅頗雀裾澄摺寸世瀬畝是凄制勢姓征性成政整星晴棲栖正清牲生盛精聖声製西誠誓請逝醒青静斉税脆隻席惜戚斥昔析石積籍績脊責赤跡蹟碩切拙接摂折設窃節説雪絶舌蝉仙先千占宣専尖川戦扇撰栓栴泉浅洗染潜煎煽旋穿箭線"],
["9140","繊羨腺舛船薦詮賎践選遷銭銑閃鮮前善漸然全禅繕膳糎噌塑岨措曾曽楚狙疏疎礎祖租粗素組蘇訴阻遡鼠僧創双叢倉喪壮奏爽宋層匝惣想捜掃挿掻"],
["9180","操早曹巣槍槽漕燥争痩相窓糟総綜聡草荘葬蒼藻装走送遭鎗霜騒像増憎臓蔵贈造促側則即息捉束測足速俗属賊族続卒袖其揃存孫尊損村遜他多太汰詑唾堕妥惰打柁舵楕陀駄騨体堆対耐岱帯待怠態戴替泰滞胎腿苔袋貸退逮隊黛鯛代台大第醍題鷹滝瀧卓啄宅托択拓沢濯琢託鐸濁諾茸凧蛸只"],
["9240","叩但達辰奪脱巽竪辿棚谷狸鱈樽誰丹単嘆坦担探旦歎淡湛炭短端箪綻耽胆蛋誕鍛団壇弾断暖檀段男談値知地弛恥智池痴稚置致蜘遅馳築畜竹筑蓄"],
["9280","逐秩窒茶嫡着中仲宙忠抽昼柱注虫衷註酎鋳駐樗瀦猪苧著貯丁兆凋喋寵帖帳庁弔張彫徴懲挑暢朝潮牒町眺聴脹腸蝶調諜超跳銚長頂鳥勅捗直朕沈珍賃鎮陳津墜椎槌追鎚痛通塚栂掴槻佃漬柘辻蔦綴鍔椿潰坪壷嬬紬爪吊釣鶴亭低停偵剃貞呈堤定帝底庭廷弟悌抵挺提梯汀碇禎程締艇訂諦蹄逓"],
["9340","邸鄭釘鼎泥摘擢敵滴的笛適鏑溺哲徹撤轍迭鉄典填天展店添纏甜貼転顛点伝殿澱田電兎吐堵塗妬屠徒斗杜渡登菟賭途都鍍砥砺努度土奴怒倒党冬"],
["9380","凍刀唐塔塘套宕島嶋悼投搭東桃梼棟盗淘湯涛灯燈当痘祷等答筒糖統到董蕩藤討謄豆踏逃透鐙陶頭騰闘働動同堂導憧撞洞瞳童胴萄道銅峠鴇匿得徳涜特督禿篤毒独読栃橡凸突椴届鳶苫寅酉瀞噸屯惇敦沌豚遁頓呑曇鈍奈那内乍凪薙謎灘捺鍋楢馴縄畷南楠軟難汝二尼弐迩匂賑肉虹廿日乳入"],
["9440","如尿韮任妊忍認濡禰祢寧葱猫熱年念捻撚燃粘乃廼之埜嚢悩濃納能脳膿農覗蚤巴把播覇杷波派琶破婆罵芭馬俳廃拝排敗杯盃牌背肺輩配倍培媒梅"],
["9480","楳煤狽買売賠陪這蝿秤矧萩伯剥博拍柏泊白箔粕舶薄迫曝漠爆縛莫駁麦函箱硲箸肇筈櫨幡肌畑畠八鉢溌発醗髪伐罰抜筏閥鳩噺塙蛤隼伴判半反叛帆搬斑板氾汎版犯班畔繁般藩販範釆煩頒飯挽晩番盤磐蕃蛮匪卑否妃庇彼悲扉批披斐比泌疲皮碑秘緋罷肥被誹費避非飛樋簸備尾微枇毘琵眉美"],
["9540","鼻柊稗匹疋髭彦膝菱肘弼必畢筆逼桧姫媛紐百謬俵彪標氷漂瓢票表評豹廟描病秒苗錨鋲蒜蛭鰭品彬斌浜瀕貧賓頻敏瓶不付埠夫婦富冨布府怖扶敷"],
["9580","斧普浮父符腐膚芙譜負賦赴阜附侮撫武舞葡蕪部封楓風葺蕗伏副復幅服福腹複覆淵弗払沸仏物鮒分吻噴墳憤扮焚奮粉糞紛雰文聞丙併兵塀幣平弊柄並蔽閉陛米頁僻壁癖碧別瞥蔑箆偏変片篇編辺返遍便勉娩弁鞭保舗鋪圃捕歩甫補輔穂募墓慕戊暮母簿菩倣俸包呆報奉宝峰峯崩庖抱捧放方朋"],
["9640","法泡烹砲縫胞芳萌蓬蜂褒訪豊邦鋒飽鳳鵬乏亡傍剖坊妨帽忘忙房暴望某棒冒紡肪膨謀貌貿鉾防吠頬北僕卜墨撲朴牧睦穆釦勃没殆堀幌奔本翻凡盆"],
["9680","摩磨魔麻埋妹昧枚毎哩槙幕膜枕鮪柾鱒桝亦俣又抹末沫迄侭繭麿万慢満漫蔓味未魅巳箕岬密蜜湊蓑稔脈妙粍民眠務夢無牟矛霧鵡椋婿娘冥名命明盟迷銘鳴姪牝滅免棉綿緬面麺摸模茂妄孟毛猛盲網耗蒙儲木黙目杢勿餅尤戻籾貰問悶紋門匁也冶夜爺耶野弥矢厄役約薬訳躍靖柳薮鑓愉愈油癒"],
["9740","諭輸唯佑優勇友宥幽悠憂揖有柚湧涌猶猷由祐裕誘遊邑郵雄融夕予余与誉輿預傭幼妖容庸揚揺擁曜楊様洋溶熔用窯羊耀葉蓉要謡踊遥陽養慾抑欲"],
["9780","沃浴翌翼淀羅螺裸来莱頼雷洛絡落酪乱卵嵐欄濫藍蘭覧利吏履李梨理璃痢裏裡里離陸律率立葎掠略劉流溜琉留硫粒隆竜龍侶慮旅虜了亮僚両凌寮料梁涼猟療瞭稜糧良諒遼量陵領力緑倫厘林淋燐琳臨輪隣鱗麟瑠塁涙累類令伶例冷励嶺怜玲礼苓鈴隷零霊麗齢暦歴列劣烈裂廉恋憐漣煉簾練聯"],
["9840","蓮連錬呂魯櫓炉賂路露労婁廊弄朗楼榔浪漏牢狼篭老聾蝋郎六麓禄肋録論倭和話歪賄脇惑枠鷲亙亘鰐詫藁蕨椀湾碗腕"],
["989f","弌丐丕个丱丶丼丿乂乖乘亂亅豫亊舒弍于亞亟亠亢亰亳亶从仍仄仆仂仗仞仭仟价伉佚估佛佝佗佇佶侈侏侘佻佩佰侑佯來侖儘俔俟俎俘俛俑俚俐俤俥倚倨倔倪倥倅伜俶倡倩倬俾俯們倆偃假會偕偐偈做偖偬偸傀傚傅傴傲"],
["9940","僉僊傳僂僖僞僥僭僣僮價僵儉儁儂儖儕儔儚儡儺儷儼儻儿兀兒兌兔兢竸兩兪兮冀冂囘册冉冏冑冓冕冖冤冦冢冩冪冫决冱冲冰况冽凅凉凛几處凩凭"],
["9980","凰凵凾刄刋刔刎刧刪刮刳刹剏剄剋剌剞剔剪剴剩剳剿剽劍劔劒剱劈劑辨辧劬劭劼劵勁勍勗勞勣勦飭勠勳勵勸勹匆匈甸匍匐匏匕匚匣匯匱匳匸區卆卅丗卉卍凖卞卩卮夘卻卷厂厖厠厦厥厮厰厶參簒雙叟曼燮叮叨叭叺吁吽呀听吭吼吮吶吩吝呎咏呵咎呟呱呷呰咒呻咀呶咄咐咆哇咢咸咥咬哄哈咨"],
["9a40","咫哂咤咾咼哘哥哦唏唔哽哮哭哺哢唹啀啣啌售啜啅啖啗唸唳啝喙喀咯喊喟啻啾喘喞單啼喃喩喇喨嗚嗅嗟嗄嗜嗤嗔嘔嗷嘖嗾嗽嘛嗹噎噐營嘴嘶嘲嘸"],
["9a80","噫噤嘯噬噪嚆嚀嚊嚠嚔嚏嚥嚮嚶嚴囂嚼囁囃囀囈囎囑囓囗囮囹圀囿圄圉圈國圍圓團圖嗇圜圦圷圸坎圻址坏坩埀垈坡坿垉垓垠垳垤垪垰埃埆埔埒埓堊埖埣堋堙堝塲堡塢塋塰毀塒堽塹墅墹墟墫墺壞墻墸墮壅壓壑壗壙壘壥壜壤壟壯壺壹壻壼壽夂夊夐夛梦夥夬夭夲夸夾竒奕奐奎奚奘奢奠奧奬奩"],
["9b40","奸妁妝佞侫妣妲姆姨姜妍姙姚娥娟娑娜娉娚婀婬婉娵娶婢婪媚媼媾嫋嫂媽嫣嫗嫦嫩嫖嫺嫻嬌嬋嬖嬲嫐嬪嬶嬾孃孅孀孑孕孚孛孥孩孰孳孵學斈孺宀"],
["9b80","它宦宸寃寇寉寔寐寤實寢寞寥寫寰寶寳尅將專對尓尠尢尨尸尹屁屆屎屓屐屏孱屬屮乢屶屹岌岑岔妛岫岻岶岼岷峅岾峇峙峩峽峺峭嶌峪崋崕崗嵜崟崛崑崔崢崚崙崘嵌嵒嵎嵋嵬嵳嵶嶇嶄嶂嶢嶝嶬嶮嶽嶐嶷嶼巉巍巓巒巖巛巫已巵帋帚帙帑帛帶帷幄幃幀幎幗幔幟幢幤幇幵并幺麼广庠廁廂廈廐廏"],
["9c40","廖廣廝廚廛廢廡廨廩廬廱廳廰廴廸廾弃弉彝彜弋弑弖弩弭弸彁彈彌彎弯彑彖彗彙彡彭彳彷徃徂彿徊很徑徇從徙徘徠徨徭徼忖忻忤忸忱忝悳忿怡恠"],
["9c80","怙怐怩怎怱怛怕怫怦怏怺恚恁恪恷恟恊恆恍恣恃恤恂恬恫恙悁悍惧悃悚悄悛悖悗悒悧悋惡悸惠惓悴忰悽惆悵惘慍愕愆惶惷愀惴惺愃愡惻惱愍愎慇愾愨愧慊愿愼愬愴愽慂慄慳慷慘慙慚慫慴慯慥慱慟慝慓慵憙憖憇憬憔憚憊憑憫憮懌懊應懷懈懃懆憺懋罹懍懦懣懶懺懴懿懽懼懾戀戈戉戍戌戔戛"],
["9d40","戞戡截戮戰戲戳扁扎扞扣扛扠扨扼抂抉找抒抓抖拔抃抔拗拑抻拏拿拆擔拈拜拌拊拂拇抛拉挌拮拱挧挂挈拯拵捐挾捍搜捏掖掎掀掫捶掣掏掉掟掵捫"],
["9d80","捩掾揩揀揆揣揉插揶揄搖搴搆搓搦搶攝搗搨搏摧摯摶摎攪撕撓撥撩撈撼據擒擅擇撻擘擂擱擧舉擠擡抬擣擯攬擶擴擲擺攀擽攘攜攅攤攣攫攴攵攷收攸畋效敖敕敍敘敞敝敲數斂斃變斛斟斫斷旃旆旁旄旌旒旛旙无旡旱杲昊昃旻杳昵昶昴昜晏晄晉晁晞晝晤晧晨晟晢晰暃暈暎暉暄暘暝曁暹曉暾暼"],
["9e40","曄暸曖曚曠昿曦曩曰曵曷朏朖朞朦朧霸朮朿朶杁朸朷杆杞杠杙杣杤枉杰枩杼杪枌枋枦枡枅枷柯枴柬枳柩枸柤柞柝柢柮枹柎柆柧檜栞框栩桀桍栲桎"],
["9e80","梳栫桙档桷桿梟梏梭梔條梛梃檮梹桴梵梠梺椏梍桾椁棊椈棘椢椦棡椌棍棔棧棕椶椒椄棗棣椥棹棠棯椨椪椚椣椡棆楹楷楜楸楫楔楾楮椹楴椽楙椰楡楞楝榁楪榲榮槐榿槁槓榾槎寨槊槝榻槃榧樮榑榠榜榕榴槞槨樂樛槿權槹槲槧樅榱樞槭樔槫樊樒櫁樣樓橄樌橲樶橸橇橢橙橦橈樸樢檐檍檠檄檢檣"],
["9f40","檗蘗檻櫃櫂檸檳檬櫞櫑櫟檪櫚櫪櫻欅蘖櫺欒欖鬱欟欸欷盜欹飮歇歃歉歐歙歔歛歟歡歸歹歿殀殄殃殍殘殕殞殤殪殫殯殲殱殳殷殼毆毋毓毟毬毫毳毯"],
["9f80","麾氈氓气氛氤氣汞汕汢汪沂沍沚沁沛汾汨汳沒沐泄泱泓沽泗泅泝沮沱沾沺泛泯泙泪洟衍洶洫洽洸洙洵洳洒洌浣涓浤浚浹浙涎涕濤涅淹渕渊涵淇淦涸淆淬淞淌淨淒淅淺淙淤淕淪淮渭湮渮渙湲湟渾渣湫渫湶湍渟湃渺湎渤滿渝游溂溪溘滉溷滓溽溯滄溲滔滕溏溥滂溟潁漑灌滬滸滾漿滲漱滯漲滌"],
["e040","漾漓滷澆潺潸澁澀潯潛濳潭澂潼潘澎澑濂潦澳澣澡澤澹濆澪濟濕濬濔濘濱濮濛瀉瀋濺瀑瀁瀏濾瀛瀚潴瀝瀘瀟瀰瀾瀲灑灣炙炒炯烱炬炸炳炮烟烋烝"],
["e080","烙焉烽焜焙煥煕熈煦煢煌煖煬熏燻熄熕熨熬燗熹熾燒燉燔燎燠燬燧燵燼燹燿爍爐爛爨爭爬爰爲爻爼爿牀牆牋牘牴牾犂犁犇犒犖犢犧犹犲狃狆狄狎狒狢狠狡狹狷倏猗猊猜猖猝猴猯猩猥猾獎獏默獗獪獨獰獸獵獻獺珈玳珎玻珀珥珮珞璢琅瑯琥珸琲琺瑕琿瑟瑙瑁瑜瑩瑰瑣瑪瑶瑾璋璞璧瓊瓏瓔珱"],
["e140","瓠瓣瓧瓩瓮瓲瓰瓱瓸瓷甄甃甅甌甎甍甕甓甞甦甬甼畄畍畊畉畛畆畚畩畤畧畫畭畸當疆疇畴疊疉疂疔疚疝疥疣痂疳痃疵疽疸疼疱痍痊痒痙痣痞痾痿"],
["e180","痼瘁痰痺痲痳瘋瘍瘉瘟瘧瘠瘡瘢瘤瘴瘰瘻癇癈癆癜癘癡癢癨癩癪癧癬癰癲癶癸發皀皃皈皋皎皖皓皙皚皰皴皸皹皺盂盍盖盒盞盡盥盧盪蘯盻眈眇眄眩眤眞眥眦眛眷眸睇睚睨睫睛睥睿睾睹瞎瞋瞑瞠瞞瞰瞶瞹瞿瞼瞽瞻矇矍矗矚矜矣矮矼砌砒礦砠礪硅碎硴碆硼碚碌碣碵碪碯磑磆磋磔碾碼磅磊磬"],
["e240","磧磚磽磴礇礒礑礙礬礫祀祠祗祟祚祕祓祺祿禊禝禧齋禪禮禳禹禺秉秕秧秬秡秣稈稍稘稙稠稟禀稱稻稾稷穃穗穉穡穢穩龝穰穹穽窈窗窕窘窖窩竈窰"],
["e280","窶竅竄窿邃竇竊竍竏竕竓站竚竝竡竢竦竭竰笂笏笊笆笳笘笙笞笵笨笶筐筺笄筍笋筌筅筵筥筴筧筰筱筬筮箝箘箟箍箜箚箋箒箏筝箙篋篁篌篏箴篆篝篩簑簔篦篥籠簀簇簓篳篷簗簍篶簣簧簪簟簷簫簽籌籃籔籏籀籐籘籟籤籖籥籬籵粃粐粤粭粢粫粡粨粳粲粱粮粹粽糀糅糂糘糒糜糢鬻糯糲糴糶糺紆"],
["e340","紂紜紕紊絅絋紮紲紿紵絆絳絖絎絲絨絮絏絣經綉絛綏絽綛綺綮綣綵緇綽綫總綢綯緜綸綟綰緘緝緤緞緻緲緡縅縊縣縡縒縱縟縉縋縢繆繦縻縵縹繃縷"],
["e380","縲縺繧繝繖繞繙繚繹繪繩繼繻纃緕繽辮繿纈纉續纒纐纓纔纖纎纛纜缸缺罅罌罍罎罐网罕罔罘罟罠罨罩罧罸羂羆羃羈羇羌羔羞羝羚羣羯羲羹羮羶羸譱翅翆翊翕翔翡翦翩翳翹飜耆耄耋耒耘耙耜耡耨耿耻聊聆聒聘聚聟聢聨聳聲聰聶聹聽聿肄肆肅肛肓肚肭冐肬胛胥胙胝胄胚胖脉胯胱脛脩脣脯腋"],
["e440","隋腆脾腓腑胼腱腮腥腦腴膃膈膊膀膂膠膕膤膣腟膓膩膰膵膾膸膽臀臂膺臉臍臑臙臘臈臚臟臠臧臺臻臾舁舂舅與舊舍舐舖舩舫舸舳艀艙艘艝艚艟艤"],
["e480","艢艨艪艫舮艱艷艸艾芍芒芫芟芻芬苡苣苟苒苴苳苺莓范苻苹苞茆苜茉苙茵茴茖茲茱荀茹荐荅茯茫茗茘莅莚莪莟莢莖茣莎莇莊荼莵荳荵莠莉莨菴萓菫菎菽萃菘萋菁菷萇菠菲萍萢萠莽萸蔆菻葭萪萼蕚蒄葷葫蒭葮蒂葩葆萬葯葹萵蓊葢蒹蒿蒟蓙蓍蒻蓚蓐蓁蓆蓖蒡蔡蓿蓴蔗蔘蔬蔟蔕蔔蓼蕀蕣蕘蕈"],
["e540","蕁蘂蕋蕕薀薤薈薑薊薨蕭薔薛藪薇薜蕷蕾薐藉薺藏薹藐藕藝藥藜藹蘊蘓蘋藾藺蘆蘢蘚蘰蘿虍乕虔號虧虱蚓蚣蚩蚪蚋蚌蚶蚯蛄蛆蚰蛉蠣蚫蛔蛞蛩蛬"],
["e580","蛟蛛蛯蜒蜆蜈蜀蜃蛻蜑蜉蜍蛹蜊蜴蜿蜷蜻蜥蜩蜚蝠蝟蝸蝌蝎蝴蝗蝨蝮蝙蝓蝣蝪蠅螢螟螂螯蟋螽蟀蟐雖螫蟄螳蟇蟆螻蟯蟲蟠蠏蠍蟾蟶蟷蠎蟒蠑蠖蠕蠢蠡蠱蠶蠹蠧蠻衄衂衒衙衞衢衫袁衾袞衵衽袵衲袂袗袒袮袙袢袍袤袰袿袱裃裄裔裘裙裝裹褂裼裴裨裲褄褌褊褓襃褞褥褪褫襁襄褻褶褸襌褝襠襞"],
["e640","襦襤襭襪襯襴襷襾覃覈覊覓覘覡覩覦覬覯覲覺覽覿觀觚觜觝觧觴觸訃訖訐訌訛訝訥訶詁詛詒詆詈詼詭詬詢誅誂誄誨誡誑誥誦誚誣諄諍諂諚諫諳諧"],
["e680","諤諱謔諠諢諷諞諛謌謇謚諡謖謐謗謠謳鞫謦謫謾謨譁譌譏譎證譖譛譚譫譟譬譯譴譽讀讌讎讒讓讖讙讚谺豁谿豈豌豎豐豕豢豬豸豺貂貉貅貊貍貎貔豼貘戝貭貪貽貲貳貮貶賈賁賤賣賚賽賺賻贄贅贊贇贏贍贐齎贓賍贔贖赧赭赱赳趁趙跂趾趺跏跚跖跌跛跋跪跫跟跣跼踈踉跿踝踞踐踟蹂踵踰踴蹊"],
["e740","蹇蹉蹌蹐蹈蹙蹤蹠踪蹣蹕蹶蹲蹼躁躇躅躄躋躊躓躑躔躙躪躡躬躰軆躱躾軅軈軋軛軣軼軻軫軾輊輅輕輒輙輓輜輟輛輌輦輳輻輹轅轂輾轌轉轆轎轗轜"],
["e780","轢轣轤辜辟辣辭辯辷迚迥迢迪迯邇迴逅迹迺逑逕逡逍逞逖逋逧逶逵逹迸遏遐遑遒逎遉逾遖遘遞遨遯遶隨遲邂遽邁邀邊邉邏邨邯邱邵郢郤扈郛鄂鄒鄙鄲鄰酊酖酘酣酥酩酳酲醋醉醂醢醫醯醪醵醴醺釀釁釉釋釐釖釟釡釛釼釵釶鈞釿鈔鈬鈕鈑鉞鉗鉅鉉鉤鉈銕鈿鉋鉐銜銖銓銛鉚鋏銹銷鋩錏鋺鍄錮"],
["e840","錙錢錚錣錺錵錻鍜鍠鍼鍮鍖鎰鎬鎭鎔鎹鏖鏗鏨鏥鏘鏃鏝鏐鏈鏤鐚鐔鐓鐃鐇鐐鐶鐫鐵鐡鐺鑁鑒鑄鑛鑠鑢鑞鑪鈩鑰鑵鑷鑽鑚鑼鑾钁鑿閂閇閊閔閖閘閙"],
["e880","閠閨閧閭閼閻閹閾闊濶闃闍闌闕闔闖關闡闥闢阡阨阮阯陂陌陏陋陷陜陞陝陟陦陲陬隍隘隕隗險隧隱隲隰隴隶隸隹雎雋雉雍襍雜霍雕雹霄霆霈霓霎霑霏霖霙霤霪霰霹霽霾靄靆靈靂靉靜靠靤靦靨勒靫靱靹鞅靼鞁靺鞆鞋鞏鞐鞜鞨鞦鞣鞳鞴韃韆韈韋韜韭齏韲竟韶韵頏頌頸頤頡頷頽顆顏顋顫顯顰"],
["e940","顱顴顳颪颯颱颶飄飃飆飩飫餃餉餒餔餘餡餝餞餤餠餬餮餽餾饂饉饅饐饋饑饒饌饕馗馘馥馭馮馼駟駛駝駘駑駭駮駱駲駻駸騁騏騅駢騙騫騷驅驂驀驃"],
["e980","騾驕驍驛驗驟驢驥驤驩驫驪骭骰骼髀髏髑髓體髞髟髢髣髦髯髫髮髴髱髷髻鬆鬘鬚鬟鬢鬣鬥鬧鬨鬩鬪鬮鬯鬲魄魃魏魍魎魑魘魴鮓鮃鮑鮖鮗鮟鮠鮨鮴鯀鯊鮹鯆鯏鯑鯒鯣鯢鯤鯔鯡鰺鯲鯱鯰鰕鰔鰉鰓鰌鰆鰈鰒鰊鰄鰮鰛鰥鰤鰡鰰鱇鰲鱆鰾鱚鱠鱧鱶鱸鳧鳬鳰鴉鴈鳫鴃鴆鴪鴦鶯鴣鴟鵄鴕鴒鵁鴿鴾鵆鵈"],
["ea40","鵝鵞鵤鵑鵐鵙鵲鶉鶇鶫鵯鵺鶚鶤鶩鶲鷄鷁鶻鶸鶺鷆鷏鷂鷙鷓鷸鷦鷭鷯鷽鸚鸛鸞鹵鹹鹽麁麈麋麌麒麕麑麝麥麩麸麪麭靡黌黎黏黐黔黜點黝黠黥黨黯"],
["ea80","黴黶黷黹黻黼黽鼇鼈皷鼕鼡鼬鼾齊齒齔齣齟齠齡齦齧齬齪齷齲齶龕龜龠堯槇遙瑤凜熙"],
["ed40","纊褜鍈銈蓜俉炻昱棈鋹曻彅丨仡仼伀伃伹佖侒侊侚侔俍偀倢俿倞偆偰偂傔僴僘兊兤冝冾凬刕劜劦勀勛匀匇匤卲厓厲叝﨎咜咊咩哿喆坙坥垬埈埇﨏"],
["ed80","塚增墲夋奓奛奝奣妤妺孖寀甯寘寬尞岦岺峵崧嵓﨑嵂嵭嶸嶹巐弡弴彧德忞恝悅悊惞惕愠惲愑愷愰憘戓抦揵摠撝擎敎昀昕昻昉昮昞昤晥晗晙晴晳暙暠暲暿曺朎朗杦枻桒柀栁桄棏﨓楨﨔榘槢樰橫橆橳橾櫢櫤毖氿汜沆汯泚洄涇浯涖涬淏淸淲淼渹湜渧渼溿澈澵濵瀅瀇瀨炅炫焏焄煜煆煇凞燁燾犱"],
["ee40","犾猤猪獷玽珉珖珣珒琇珵琦琪琩琮瑢璉璟甁畯皂皜皞皛皦益睆劯砡硎硤硺礰礼神祥禔福禛竑竧靖竫箞精絈絜綷綠緖繒罇羡羽茁荢荿菇菶葈蒴蕓蕙"],
["ee80","蕫﨟薰蘒﨡蠇裵訒訷詹誧誾諟諸諶譓譿賰賴贒赶﨣軏﨤逸遧郞都鄕鄧釚釗釞釭釮釤釥鈆鈐鈊鈺鉀鈼鉎鉙鉑鈹鉧銧鉷鉸鋧鋗鋙鋐﨧鋕鋠鋓錥錡鋻﨨錞鋿錝錂鍰鍗鎤鏆鏞鏸鐱鑅鑈閒隆﨩隝隯霳霻靃靍靏靑靕顗顥飯飼餧館馞驎髙髜魵魲鮏鮱鮻鰀鵰鵫鶴鸙黑"],
["eeef","ⅰ",9,"￢￤＇＂"],
["f040","",62],
["f080","",124],
["f140","",62],
["f180","",124],
["f240","",62],
["f280","",124],
["f340","",62],
["f380","",124],
["f440","",62],
["f480","",124],
["f540","",62],
["f580","",124],
["f640","",62],
["f680","",124],
["f740","",62],
["f780","",124],
["f840","",62],
["f880","",124],
["f940",""],
["fa40","ⅰ",9,"Ⅰ",9,"￢￤＇＂㈱№℡∵纊褜鍈銈蓜俉炻昱棈鋹曻彅丨仡仼伀伃伹佖侒侊侚侔俍偀倢俿倞偆偰偂傔僴僘兊"],
["fa80","兤冝冾凬刕劜劦勀勛匀匇匤卲厓厲叝﨎咜咊咩哿喆坙坥垬埈埇﨏塚增墲夋奓奛奝奣妤妺孖寀甯寘寬尞岦岺峵崧嵓﨑嵂嵭嶸嶹巐弡弴彧德忞恝悅悊惞惕愠惲愑愷愰憘戓抦揵摠撝擎敎昀昕昻昉昮昞昤晥晗晙晴晳暙暠暲暿曺朎朗杦枻桒柀栁桄棏﨓楨﨔榘槢樰橫橆橳橾櫢櫤毖氿汜沆汯泚洄涇浯"],
["fb40","涖涬淏淸淲淼渹湜渧渼溿澈澵濵瀅瀇瀨炅炫焏焄煜煆煇凞燁燾犱犾猤猪獷玽珉珖珣珒琇珵琦琪琩琮瑢璉璟甁畯皂皜皞皛皦益睆劯砡硎硤硺礰礼神"],
["fb80","祥禔福禛竑竧靖竫箞精絈絜綷綠緖繒罇羡羽茁荢荿菇菶葈蒴蕓蕙蕫﨟薰蘒﨡蠇裵訒訷詹誧誾諟諸諶譓譿賰賴贒赶﨣軏﨤逸遧郞都鄕鄧釚釗釞釭釮釤釥鈆鈐鈊鈺鉀鈼鉎鉙鉑鈹鉧銧鉷鉸鋧鋗鋙鋐﨧鋕鋠鋓錥錡鋻﨨錞鋿錝錂鍰鍗鎤鏆鏞鏸鐱鑅鑈閒隆﨩隝隯霳霻靃靍靏靑靕顗顥飯飼餧館馞驎髙"],
["fc40","髜魵魲鮏鮱鮻鰀鵰鵫鶴鸙黑"]
]
;

var shiftjis$1 = /*#__PURE__*/Object.freeze({
            default: shiftjis
});

var eucjp = [
["0","\u0000",127],
["8ea1","｡",62],
["a1a1","　、。，．・：；？！゛゜´｀¨＾￣＿ヽヾゝゞ〃仝々〆〇ー―‐／＼～∥｜…‥‘’“”（）〔〕［］｛｝〈",9,"＋－±×÷＝≠＜＞≦≧∞∴♂♀°′″℃￥＄￠￡％＃＆＊＠§☆★○●◎◇"],
["a2a1","◆□■△▲▽▼※〒→←↑↓〓"],
["a2ba","∈∋⊆⊇⊂⊃∪∩"],
["a2ca","∧∨￢⇒⇔∀∃"],
["a2dc","∠⊥⌒∂∇≡≒≪≫√∽∝∵∫∬"],
["a2f2","Å‰♯♭♪†‡¶"],
["a2fe","◯"],
["a3b0","０",9],
["a3c1","Ａ",25],
["a3e1","ａ",25],
["a4a1","ぁ",82],
["a5a1","ァ",85],
["a6a1","Α",16,"Σ",6],
["a6c1","α",16,"σ",6],
["a7a1","А",5,"ЁЖ",25],
["a7d1","а",5,"ёж",25],
["a8a1","─│┌┐┘└├┬┤┴┼━┃┏┓┛┗┣┳┫┻╋┠┯┨┷┿┝┰┥┸╂"],
["ada1","①",19,"Ⅰ",9],
["adc0","㍉㌔㌢㍍㌘㌧㌃㌶㍑㍗㌍㌦㌣㌫㍊㌻㎜㎝㎞㎎㎏㏄㎡"],
["addf","㍻〝〟№㏍℡㊤",4,"㈱㈲㈹㍾㍽㍼≒≡∫∮∑√⊥∠∟⊿∵∩∪"],
["b0a1","亜唖娃阿哀愛挨姶逢葵茜穐悪握渥旭葦芦鯵梓圧斡扱宛姐虻飴絢綾鮎或粟袷安庵按暗案闇鞍杏以伊位依偉囲夷委威尉惟意慰易椅為畏異移維緯胃萎衣謂違遺医井亥域育郁磯一壱溢逸稲茨芋鰯允印咽員因姻引飲淫胤蔭"],
["b1a1","院陰隠韻吋右宇烏羽迂雨卯鵜窺丑碓臼渦嘘唄欝蔚鰻姥厩浦瓜閏噂云運雲荏餌叡営嬰影映曳栄永泳洩瑛盈穎頴英衛詠鋭液疫益駅悦謁越閲榎厭円園堰奄宴延怨掩援沿演炎焔煙燕猿縁艶苑薗遠鉛鴛塩於汚甥凹央奥往応"],
["b2a1","押旺横欧殴王翁襖鴬鴎黄岡沖荻億屋憶臆桶牡乙俺卸恩温穏音下化仮何伽価佳加可嘉夏嫁家寡科暇果架歌河火珂禍禾稼箇花苛茄荷華菓蝦課嘩貨迦過霞蚊俄峨我牙画臥芽蛾賀雅餓駕介会解回塊壊廻快怪悔恢懐戒拐改"],
["b3a1","魁晦械海灰界皆絵芥蟹開階貝凱劾外咳害崖慨概涯碍蓋街該鎧骸浬馨蛙垣柿蛎鈎劃嚇各廓拡撹格核殻獲確穫覚角赫較郭閣隔革学岳楽額顎掛笠樫橿梶鰍潟割喝恰括活渇滑葛褐轄且鰹叶椛樺鞄株兜竃蒲釜鎌噛鴨栢茅萱"],
["b4a1","粥刈苅瓦乾侃冠寒刊勘勧巻喚堪姦完官寛干幹患感慣憾換敢柑桓棺款歓汗漢澗潅環甘監看竿管簡緩缶翰肝艦莞観諌貫還鑑間閑関陥韓館舘丸含岸巌玩癌眼岩翫贋雁頑顔願企伎危喜器基奇嬉寄岐希幾忌揮机旗既期棋棄"],
["b5a1","機帰毅気汽畿祈季稀紀徽規記貴起軌輝飢騎鬼亀偽儀妓宜戯技擬欺犠疑祇義蟻誼議掬菊鞠吉吃喫桔橘詰砧杵黍却客脚虐逆丘久仇休及吸宮弓急救朽求汲泣灸球究窮笈級糾給旧牛去居巨拒拠挙渠虚許距鋸漁禦魚亨享京"],
["b6a1","供侠僑兇競共凶協匡卿叫喬境峡強彊怯恐恭挟教橋況狂狭矯胸脅興蕎郷鏡響饗驚仰凝尭暁業局曲極玉桐粁僅勤均巾錦斤欣欽琴禁禽筋緊芹菌衿襟謹近金吟銀九倶句区狗玖矩苦躯駆駈駒具愚虞喰空偶寓遇隅串櫛釧屑屈"],
["b7a1","掘窟沓靴轡窪熊隈粂栗繰桑鍬勲君薫訓群軍郡卦袈祁係傾刑兄啓圭珪型契形径恵慶慧憩掲携敬景桂渓畦稽系経継繋罫茎荊蛍計詣警軽頚鶏芸迎鯨劇戟撃激隙桁傑欠決潔穴結血訣月件倹倦健兼券剣喧圏堅嫌建憲懸拳捲"],
["b8a1","検権牽犬献研硯絹県肩見謙賢軒遣鍵険顕験鹸元原厳幻弦減源玄現絃舷言諺限乎個古呼固姑孤己庫弧戸故枯湖狐糊袴股胡菰虎誇跨鈷雇顧鼓五互伍午呉吾娯後御悟梧檎瑚碁語誤護醐乞鯉交佼侯候倖光公功効勾厚口向"],
["b9a1","后喉坑垢好孔孝宏工巧巷幸広庚康弘恒慌抗拘控攻昂晃更杭校梗構江洪浩港溝甲皇硬稿糠紅紘絞綱耕考肯肱腔膏航荒行衡講貢購郊酵鉱砿鋼閤降項香高鴻剛劫号合壕拷濠豪轟麹克刻告国穀酷鵠黒獄漉腰甑忽惚骨狛込"],
["baa1","此頃今困坤墾婚恨懇昏昆根梱混痕紺艮魂些佐叉唆嵯左差査沙瑳砂詐鎖裟坐座挫債催再最哉塞妻宰彩才採栽歳済災采犀砕砦祭斎細菜裁載際剤在材罪財冴坂阪堺榊肴咲崎埼碕鷺作削咋搾昨朔柵窄策索錯桜鮭笹匙冊刷"],
["bba1","察拶撮擦札殺薩雑皐鯖捌錆鮫皿晒三傘参山惨撒散桟燦珊産算纂蚕讃賛酸餐斬暫残仕仔伺使刺司史嗣四士始姉姿子屍市師志思指支孜斯施旨枝止死氏獅祉私糸紙紫肢脂至視詞詩試誌諮資賜雌飼歯事似侍児字寺慈持時"],
["bca1","次滋治爾璽痔磁示而耳自蒔辞汐鹿式識鴫竺軸宍雫七叱執失嫉室悉湿漆疾質実蔀篠偲柴芝屡蕊縞舎写射捨赦斜煮社紗者謝車遮蛇邪借勺尺杓灼爵酌釈錫若寂弱惹主取守手朱殊狩珠種腫趣酒首儒受呪寿授樹綬需囚収周"],
["bda1","宗就州修愁拾洲秀秋終繍習臭舟蒐衆襲讐蹴輯週酋酬集醜什住充十従戎柔汁渋獣縦重銃叔夙宿淑祝縮粛塾熟出術述俊峻春瞬竣舜駿准循旬楯殉淳準潤盾純巡遵醇順処初所暑曙渚庶緒署書薯藷諸助叙女序徐恕鋤除傷償"],
["bea1","勝匠升召哨商唱嘗奨妾娼宵将小少尚庄床廠彰承抄招掌捷昇昌昭晶松梢樟樵沼消渉湘焼焦照症省硝礁祥称章笑粧紹肖菖蒋蕉衝裳訟証詔詳象賞醤鉦鍾鐘障鞘上丈丞乗冗剰城場壌嬢常情擾条杖浄状畳穣蒸譲醸錠嘱埴飾"],
["bfa1","拭植殖燭織職色触食蝕辱尻伸信侵唇娠寝審心慎振新晋森榛浸深申疹真神秦紳臣芯薪親診身辛進針震人仁刃塵壬尋甚尽腎訊迅陣靭笥諏須酢図厨逗吹垂帥推水炊睡粋翠衰遂酔錐錘随瑞髄崇嵩数枢趨雛据杉椙菅頗雀裾"],
["c0a1","澄摺寸世瀬畝是凄制勢姓征性成政整星晴棲栖正清牲生盛精聖声製西誠誓請逝醒青静斉税脆隻席惜戚斥昔析石積籍績脊責赤跡蹟碩切拙接摂折設窃節説雪絶舌蝉仙先千占宣専尖川戦扇撰栓栴泉浅洗染潜煎煽旋穿箭線"],
["c1a1","繊羨腺舛船薦詮賎践選遷銭銑閃鮮前善漸然全禅繕膳糎噌塑岨措曾曽楚狙疏疎礎祖租粗素組蘇訴阻遡鼠僧創双叢倉喪壮奏爽宋層匝惣想捜掃挿掻操早曹巣槍槽漕燥争痩相窓糟総綜聡草荘葬蒼藻装走送遭鎗霜騒像増憎"],
["c2a1","臓蔵贈造促側則即息捉束測足速俗属賊族続卒袖其揃存孫尊損村遜他多太汰詑唾堕妥惰打柁舵楕陀駄騨体堆対耐岱帯待怠態戴替泰滞胎腿苔袋貸退逮隊黛鯛代台大第醍題鷹滝瀧卓啄宅托択拓沢濯琢託鐸濁諾茸凧蛸只"],
["c3a1","叩但達辰奪脱巽竪辿棚谷狸鱈樽誰丹単嘆坦担探旦歎淡湛炭短端箪綻耽胆蛋誕鍛団壇弾断暖檀段男談値知地弛恥智池痴稚置致蜘遅馳築畜竹筑蓄逐秩窒茶嫡着中仲宙忠抽昼柱注虫衷註酎鋳駐樗瀦猪苧著貯丁兆凋喋寵"],
["c4a1","帖帳庁弔張彫徴懲挑暢朝潮牒町眺聴脹腸蝶調諜超跳銚長頂鳥勅捗直朕沈珍賃鎮陳津墜椎槌追鎚痛通塚栂掴槻佃漬柘辻蔦綴鍔椿潰坪壷嬬紬爪吊釣鶴亭低停偵剃貞呈堤定帝底庭廷弟悌抵挺提梯汀碇禎程締艇訂諦蹄逓"],
["c5a1","邸鄭釘鼎泥摘擢敵滴的笛適鏑溺哲徹撤轍迭鉄典填天展店添纏甜貼転顛点伝殿澱田電兎吐堵塗妬屠徒斗杜渡登菟賭途都鍍砥砺努度土奴怒倒党冬凍刀唐塔塘套宕島嶋悼投搭東桃梼棟盗淘湯涛灯燈当痘祷等答筒糖統到"],
["c6a1","董蕩藤討謄豆踏逃透鐙陶頭騰闘働動同堂導憧撞洞瞳童胴萄道銅峠鴇匿得徳涜特督禿篤毒独読栃橡凸突椴届鳶苫寅酉瀞噸屯惇敦沌豚遁頓呑曇鈍奈那内乍凪薙謎灘捺鍋楢馴縄畷南楠軟難汝二尼弐迩匂賑肉虹廿日乳入"],
["c7a1","如尿韮任妊忍認濡禰祢寧葱猫熱年念捻撚燃粘乃廼之埜嚢悩濃納能脳膿農覗蚤巴把播覇杷波派琶破婆罵芭馬俳廃拝排敗杯盃牌背肺輩配倍培媒梅楳煤狽買売賠陪這蝿秤矧萩伯剥博拍柏泊白箔粕舶薄迫曝漠爆縛莫駁麦"],
["c8a1","函箱硲箸肇筈櫨幡肌畑畠八鉢溌発醗髪伐罰抜筏閥鳩噺塙蛤隼伴判半反叛帆搬斑板氾汎版犯班畔繁般藩販範釆煩頒飯挽晩番盤磐蕃蛮匪卑否妃庇彼悲扉批披斐比泌疲皮碑秘緋罷肥被誹費避非飛樋簸備尾微枇毘琵眉美"],
["c9a1","鼻柊稗匹疋髭彦膝菱肘弼必畢筆逼桧姫媛紐百謬俵彪標氷漂瓢票表評豹廟描病秒苗錨鋲蒜蛭鰭品彬斌浜瀕貧賓頻敏瓶不付埠夫婦富冨布府怖扶敷斧普浮父符腐膚芙譜負賦赴阜附侮撫武舞葡蕪部封楓風葺蕗伏副復幅服"],
["caa1","福腹複覆淵弗払沸仏物鮒分吻噴墳憤扮焚奮粉糞紛雰文聞丙併兵塀幣平弊柄並蔽閉陛米頁僻壁癖碧別瞥蔑箆偏変片篇編辺返遍便勉娩弁鞭保舗鋪圃捕歩甫補輔穂募墓慕戊暮母簿菩倣俸包呆報奉宝峰峯崩庖抱捧放方朋"],
["cba1","法泡烹砲縫胞芳萌蓬蜂褒訪豊邦鋒飽鳳鵬乏亡傍剖坊妨帽忘忙房暴望某棒冒紡肪膨謀貌貿鉾防吠頬北僕卜墨撲朴牧睦穆釦勃没殆堀幌奔本翻凡盆摩磨魔麻埋妹昧枚毎哩槙幕膜枕鮪柾鱒桝亦俣又抹末沫迄侭繭麿万慢満"],
["cca1","漫蔓味未魅巳箕岬密蜜湊蓑稔脈妙粍民眠務夢無牟矛霧鵡椋婿娘冥名命明盟迷銘鳴姪牝滅免棉綿緬面麺摸模茂妄孟毛猛盲網耗蒙儲木黙目杢勿餅尤戻籾貰問悶紋門匁也冶夜爺耶野弥矢厄役約薬訳躍靖柳薮鑓愉愈油癒"],
["cda1","諭輸唯佑優勇友宥幽悠憂揖有柚湧涌猶猷由祐裕誘遊邑郵雄融夕予余与誉輿預傭幼妖容庸揚揺擁曜楊様洋溶熔用窯羊耀葉蓉要謡踊遥陽養慾抑欲沃浴翌翼淀羅螺裸来莱頼雷洛絡落酪乱卵嵐欄濫藍蘭覧利吏履李梨理璃"],
["cea1","痢裏裡里離陸律率立葎掠略劉流溜琉留硫粒隆竜龍侶慮旅虜了亮僚両凌寮料梁涼猟療瞭稜糧良諒遼量陵領力緑倫厘林淋燐琳臨輪隣鱗麟瑠塁涙累類令伶例冷励嶺怜玲礼苓鈴隷零霊麗齢暦歴列劣烈裂廉恋憐漣煉簾練聯"],
["cfa1","蓮連錬呂魯櫓炉賂路露労婁廊弄朗楼榔浪漏牢狼篭老聾蝋郎六麓禄肋録論倭和話歪賄脇惑枠鷲亙亘鰐詫藁蕨椀湾碗腕"],
["d0a1","弌丐丕个丱丶丼丿乂乖乘亂亅豫亊舒弍于亞亟亠亢亰亳亶从仍仄仆仂仗仞仭仟价伉佚估佛佝佗佇佶侈侏侘佻佩佰侑佯來侖儘俔俟俎俘俛俑俚俐俤俥倚倨倔倪倥倅伜俶倡倩倬俾俯們倆偃假會偕偐偈做偖偬偸傀傚傅傴傲"],
["d1a1","僉僊傳僂僖僞僥僭僣僮價僵儉儁儂儖儕儔儚儡儺儷儼儻儿兀兒兌兔兢竸兩兪兮冀冂囘册冉冏冑冓冕冖冤冦冢冩冪冫决冱冲冰况冽凅凉凛几處凩凭凰凵凾刄刋刔刎刧刪刮刳刹剏剄剋剌剞剔剪剴剩剳剿剽劍劔劒剱劈劑辨"],
["d2a1","辧劬劭劼劵勁勍勗勞勣勦飭勠勳勵勸勹匆匈甸匍匐匏匕匚匣匯匱匳匸區卆卅丗卉卍凖卞卩卮夘卻卷厂厖厠厦厥厮厰厶參簒雙叟曼燮叮叨叭叺吁吽呀听吭吼吮吶吩吝呎咏呵咎呟呱呷呰咒呻咀呶咄咐咆哇咢咸咥咬哄哈咨"],
["d3a1","咫哂咤咾咼哘哥哦唏唔哽哮哭哺哢唹啀啣啌售啜啅啖啗唸唳啝喙喀咯喊喟啻啾喘喞單啼喃喩喇喨嗚嗅嗟嗄嗜嗤嗔嘔嗷嘖嗾嗽嘛嗹噎噐營嘴嘶嘲嘸噫噤嘯噬噪嚆嚀嚊嚠嚔嚏嚥嚮嚶嚴囂嚼囁囃囀囈囎囑囓囗囮囹圀囿圄圉"],
["d4a1","圈國圍圓團圖嗇圜圦圷圸坎圻址坏坩埀垈坡坿垉垓垠垳垤垪垰埃埆埔埒埓堊埖埣堋堙堝塲堡塢塋塰毀塒堽塹墅墹墟墫墺壞墻墸墮壅壓壑壗壙壘壥壜壤壟壯壺壹壻壼壽夂夊夐夛梦夥夬夭夲夸夾竒奕奐奎奚奘奢奠奧奬奩"],
["d5a1","奸妁妝佞侫妣妲姆姨姜妍姙姚娥娟娑娜娉娚婀婬婉娵娶婢婪媚媼媾嫋嫂媽嫣嫗嫦嫩嫖嫺嫻嬌嬋嬖嬲嫐嬪嬶嬾孃孅孀孑孕孚孛孥孩孰孳孵學斈孺宀它宦宸寃寇寉寔寐寤實寢寞寥寫寰寶寳尅將專對尓尠尢尨尸尹屁屆屎屓"],
["d6a1","屐屏孱屬屮乢屶屹岌岑岔妛岫岻岶岼岷峅岾峇峙峩峽峺峭嶌峪崋崕崗嵜崟崛崑崔崢崚崙崘嵌嵒嵎嵋嵬嵳嵶嶇嶄嶂嶢嶝嶬嶮嶽嶐嶷嶼巉巍巓巒巖巛巫已巵帋帚帙帑帛帶帷幄幃幀幎幗幔幟幢幤幇幵并幺麼广庠廁廂廈廐廏"],
["d7a1","廖廣廝廚廛廢廡廨廩廬廱廳廰廴廸廾弃弉彝彜弋弑弖弩弭弸彁彈彌彎弯彑彖彗彙彡彭彳彷徃徂彿徊很徑徇從徙徘徠徨徭徼忖忻忤忸忱忝悳忿怡恠怙怐怩怎怱怛怕怫怦怏怺恚恁恪恷恟恊恆恍恣恃恤恂恬恫恙悁悍惧悃悚"],
["d8a1","悄悛悖悗悒悧悋惡悸惠惓悴忰悽惆悵惘慍愕愆惶惷愀惴惺愃愡惻惱愍愎慇愾愨愧慊愿愼愬愴愽慂慄慳慷慘慙慚慫慴慯慥慱慟慝慓慵憙憖憇憬憔憚憊憑憫憮懌懊應懷懈懃懆憺懋罹懍懦懣懶懺懴懿懽懼懾戀戈戉戍戌戔戛"],
["d9a1","戞戡截戮戰戲戳扁扎扞扣扛扠扨扼抂抉找抒抓抖拔抃抔拗拑抻拏拿拆擔拈拜拌拊拂拇抛拉挌拮拱挧挂挈拯拵捐挾捍搜捏掖掎掀掫捶掣掏掉掟掵捫捩掾揩揀揆揣揉插揶揄搖搴搆搓搦搶攝搗搨搏摧摯摶摎攪撕撓撥撩撈撼"],
["daa1","據擒擅擇撻擘擂擱擧舉擠擡抬擣擯攬擶擴擲擺攀擽攘攜攅攤攣攫攴攵攷收攸畋效敖敕敍敘敞敝敲數斂斃變斛斟斫斷旃旆旁旄旌旒旛旙无旡旱杲昊昃旻杳昵昶昴昜晏晄晉晁晞晝晤晧晨晟晢晰暃暈暎暉暄暘暝曁暹曉暾暼"],
["dba1","曄暸曖曚曠昿曦曩曰曵曷朏朖朞朦朧霸朮朿朶杁朸朷杆杞杠杙杣杤枉杰枩杼杪枌枋枦枡枅枷柯枴柬枳柩枸柤柞柝柢柮枹柎柆柧檜栞框栩桀桍栲桎梳栫桙档桷桿梟梏梭梔條梛梃檮梹桴梵梠梺椏梍桾椁棊椈棘椢椦棡椌棍"],
["dca1","棔棧棕椶椒椄棗棣椥棹棠棯椨椪椚椣椡棆楹楷楜楸楫楔楾楮椹楴椽楙椰楡楞楝榁楪榲榮槐榿槁槓榾槎寨槊槝榻槃榧樮榑榠榜榕榴槞槨樂樛槿權槹槲槧樅榱樞槭樔槫樊樒櫁樣樓橄樌橲樶橸橇橢橙橦橈樸樢檐檍檠檄檢檣"],
["dda1","檗蘗檻櫃櫂檸檳檬櫞櫑櫟檪櫚櫪櫻欅蘖櫺欒欖鬱欟欸欷盜欹飮歇歃歉歐歙歔歛歟歡歸歹歿殀殄殃殍殘殕殞殤殪殫殯殲殱殳殷殼毆毋毓毟毬毫毳毯麾氈氓气氛氤氣汞汕汢汪沂沍沚沁沛汾汨汳沒沐泄泱泓沽泗泅泝沮沱沾"],
["dea1","沺泛泯泙泪洟衍洶洫洽洸洙洵洳洒洌浣涓浤浚浹浙涎涕濤涅淹渕渊涵淇淦涸淆淬淞淌淨淒淅淺淙淤淕淪淮渭湮渮渙湲湟渾渣湫渫湶湍渟湃渺湎渤滿渝游溂溪溘滉溷滓溽溯滄溲滔滕溏溥滂溟潁漑灌滬滸滾漿滲漱滯漲滌"],
["dfa1","漾漓滷澆潺潸澁澀潯潛濳潭澂潼潘澎澑濂潦澳澣澡澤澹濆澪濟濕濬濔濘濱濮濛瀉瀋濺瀑瀁瀏濾瀛瀚潴瀝瀘瀟瀰瀾瀲灑灣炙炒炯烱炬炸炳炮烟烋烝烙焉烽焜焙煥煕熈煦煢煌煖煬熏燻熄熕熨熬燗熹熾燒燉燔燎燠燬燧燵燼"],
["e0a1","燹燿爍爐爛爨爭爬爰爲爻爼爿牀牆牋牘牴牾犂犁犇犒犖犢犧犹犲狃狆狄狎狒狢狠狡狹狷倏猗猊猜猖猝猴猯猩猥猾獎獏默獗獪獨獰獸獵獻獺珈玳珎玻珀珥珮珞璢琅瑯琥珸琲琺瑕琿瑟瑙瑁瑜瑩瑰瑣瑪瑶瑾璋璞璧瓊瓏瓔珱"],
["e1a1","瓠瓣瓧瓩瓮瓲瓰瓱瓸瓷甄甃甅甌甎甍甕甓甞甦甬甼畄畍畊畉畛畆畚畩畤畧畫畭畸當疆疇畴疊疉疂疔疚疝疥疣痂疳痃疵疽疸疼疱痍痊痒痙痣痞痾痿痼瘁痰痺痲痳瘋瘍瘉瘟瘧瘠瘡瘢瘤瘴瘰瘻癇癈癆癜癘癡癢癨癩癪癧癬癰"],
["e2a1","癲癶癸發皀皃皈皋皎皖皓皙皚皰皴皸皹皺盂盍盖盒盞盡盥盧盪蘯盻眈眇眄眩眤眞眥眦眛眷眸睇睚睨睫睛睥睿睾睹瞎瞋瞑瞠瞞瞰瞶瞹瞿瞼瞽瞻矇矍矗矚矜矣矮矼砌砒礦砠礪硅碎硴碆硼碚碌碣碵碪碯磑磆磋磔碾碼磅磊磬"],
["e3a1","磧磚磽磴礇礒礑礙礬礫祀祠祗祟祚祕祓祺祿禊禝禧齋禪禮禳禹禺秉秕秧秬秡秣稈稍稘稙稠稟禀稱稻稾稷穃穗穉穡穢穩龝穰穹穽窈窗窕窘窖窩竈窰窶竅竄窿邃竇竊竍竏竕竓站竚竝竡竢竦竭竰笂笏笊笆笳笘笙笞笵笨笶筐"],
["e4a1","筺笄筍笋筌筅筵筥筴筧筰筱筬筮箝箘箟箍箜箚箋箒箏筝箙篋篁篌篏箴篆篝篩簑簔篦篥籠簀簇簓篳篷簗簍篶簣簧簪簟簷簫簽籌籃籔籏籀籐籘籟籤籖籥籬籵粃粐粤粭粢粫粡粨粳粲粱粮粹粽糀糅糂糘糒糜糢鬻糯糲糴糶糺紆"],
["e5a1","紂紜紕紊絅絋紮紲紿紵絆絳絖絎絲絨絮絏絣經綉絛綏絽綛綺綮綣綵緇綽綫總綢綯緜綸綟綰緘緝緤緞緻緲緡縅縊縣縡縒縱縟縉縋縢繆繦縻縵縹繃縷縲縺繧繝繖繞繙繚繹繪繩繼繻纃緕繽辮繿纈纉續纒纐纓纔纖纎纛纜缸缺"],
["e6a1","罅罌罍罎罐网罕罔罘罟罠罨罩罧罸羂羆羃羈羇羌羔羞羝羚羣羯羲羹羮羶羸譱翅翆翊翕翔翡翦翩翳翹飜耆耄耋耒耘耙耜耡耨耿耻聊聆聒聘聚聟聢聨聳聲聰聶聹聽聿肄肆肅肛肓肚肭冐肬胛胥胙胝胄胚胖脉胯胱脛脩脣脯腋"],
["e7a1","隋腆脾腓腑胼腱腮腥腦腴膃膈膊膀膂膠膕膤膣腟膓膩膰膵膾膸膽臀臂膺臉臍臑臙臘臈臚臟臠臧臺臻臾舁舂舅與舊舍舐舖舩舫舸舳艀艙艘艝艚艟艤艢艨艪艫舮艱艷艸艾芍芒芫芟芻芬苡苣苟苒苴苳苺莓范苻苹苞茆苜茉苙"],
["e8a1","茵茴茖茲茱荀茹荐荅茯茫茗茘莅莚莪莟莢莖茣莎莇莊荼莵荳荵莠莉莨菴萓菫菎菽萃菘萋菁菷萇菠菲萍萢萠莽萸蔆菻葭萪萼蕚蒄葷葫蒭葮蒂葩葆萬葯葹萵蓊葢蒹蒿蒟蓙蓍蒻蓚蓐蓁蓆蓖蒡蔡蓿蓴蔗蔘蔬蔟蔕蔔蓼蕀蕣蕘蕈"],
["e9a1","蕁蘂蕋蕕薀薤薈薑薊薨蕭薔薛藪薇薜蕷蕾薐藉薺藏薹藐藕藝藥藜藹蘊蘓蘋藾藺蘆蘢蘚蘰蘿虍乕虔號虧虱蚓蚣蚩蚪蚋蚌蚶蚯蛄蛆蚰蛉蠣蚫蛔蛞蛩蛬蛟蛛蛯蜒蜆蜈蜀蜃蛻蜑蜉蜍蛹蜊蜴蜿蜷蜻蜥蜩蜚蝠蝟蝸蝌蝎蝴蝗蝨蝮蝙"],
["eaa1","蝓蝣蝪蠅螢螟螂螯蟋螽蟀蟐雖螫蟄螳蟇蟆螻蟯蟲蟠蠏蠍蟾蟶蟷蠎蟒蠑蠖蠕蠢蠡蠱蠶蠹蠧蠻衄衂衒衙衞衢衫袁衾袞衵衽袵衲袂袗袒袮袙袢袍袤袰袿袱裃裄裔裘裙裝裹褂裼裴裨裲褄褌褊褓襃褞褥褪褫襁襄褻褶褸襌褝襠襞"],
["eba1","襦襤襭襪襯襴襷襾覃覈覊覓覘覡覩覦覬覯覲覺覽覿觀觚觜觝觧觴觸訃訖訐訌訛訝訥訶詁詛詒詆詈詼詭詬詢誅誂誄誨誡誑誥誦誚誣諄諍諂諚諫諳諧諤諱謔諠諢諷諞諛謌謇謚諡謖謐謗謠謳鞫謦謫謾謨譁譌譏譎證譖譛譚譫"],
["eca1","譟譬譯譴譽讀讌讎讒讓讖讙讚谺豁谿豈豌豎豐豕豢豬豸豺貂貉貅貊貍貎貔豼貘戝貭貪貽貲貳貮貶賈賁賤賣賚賽賺賻贄贅贊贇贏贍贐齎贓賍贔贖赧赭赱赳趁趙跂趾趺跏跚跖跌跛跋跪跫跟跣跼踈踉跿踝踞踐踟蹂踵踰踴蹊"],
["eda1","蹇蹉蹌蹐蹈蹙蹤蹠踪蹣蹕蹶蹲蹼躁躇躅躄躋躊躓躑躔躙躪躡躬躰軆躱躾軅軈軋軛軣軼軻軫軾輊輅輕輒輙輓輜輟輛輌輦輳輻輹轅轂輾轌轉轆轎轗轜轢轣轤辜辟辣辭辯辷迚迥迢迪迯邇迴逅迹迺逑逕逡逍逞逖逋逧逶逵逹迸"],
["eea1","遏遐遑遒逎遉逾遖遘遞遨遯遶隨遲邂遽邁邀邊邉邏邨邯邱邵郢郤扈郛鄂鄒鄙鄲鄰酊酖酘酣酥酩酳酲醋醉醂醢醫醯醪醵醴醺釀釁釉釋釐釖釟釡釛釼釵釶鈞釿鈔鈬鈕鈑鉞鉗鉅鉉鉤鉈銕鈿鉋鉐銜銖銓銛鉚鋏銹銷鋩錏鋺鍄錮"],
["efa1","錙錢錚錣錺錵錻鍜鍠鍼鍮鍖鎰鎬鎭鎔鎹鏖鏗鏨鏥鏘鏃鏝鏐鏈鏤鐚鐔鐓鐃鐇鐐鐶鐫鐵鐡鐺鑁鑒鑄鑛鑠鑢鑞鑪鈩鑰鑵鑷鑽鑚鑼鑾钁鑿閂閇閊閔閖閘閙閠閨閧閭閼閻閹閾闊濶闃闍闌闕闔闖關闡闥闢阡阨阮阯陂陌陏陋陷陜陞"],
["f0a1","陝陟陦陲陬隍隘隕隗險隧隱隲隰隴隶隸隹雎雋雉雍襍雜霍雕雹霄霆霈霓霎霑霏霖霙霤霪霰霹霽霾靄靆靈靂靉靜靠靤靦靨勒靫靱靹鞅靼鞁靺鞆鞋鞏鞐鞜鞨鞦鞣鞳鞴韃韆韈韋韜韭齏韲竟韶韵頏頌頸頤頡頷頽顆顏顋顫顯顰"],
["f1a1","顱顴顳颪颯颱颶飄飃飆飩飫餃餉餒餔餘餡餝餞餤餠餬餮餽餾饂饉饅饐饋饑饒饌饕馗馘馥馭馮馼駟駛駝駘駑駭駮駱駲駻駸騁騏騅駢騙騫騷驅驂驀驃騾驕驍驛驗驟驢驥驤驩驫驪骭骰骼髀髏髑髓體髞髟髢髣髦髯髫髮髴髱髷"],
["f2a1","髻鬆鬘鬚鬟鬢鬣鬥鬧鬨鬩鬪鬮鬯鬲魄魃魏魍魎魑魘魴鮓鮃鮑鮖鮗鮟鮠鮨鮴鯀鯊鮹鯆鯏鯑鯒鯣鯢鯤鯔鯡鰺鯲鯱鯰鰕鰔鰉鰓鰌鰆鰈鰒鰊鰄鰮鰛鰥鰤鰡鰰鱇鰲鱆鰾鱚鱠鱧鱶鱸鳧鳬鳰鴉鴈鳫鴃鴆鴪鴦鶯鴣鴟鵄鴕鴒鵁鴿鴾鵆鵈"],
["f3a1","鵝鵞鵤鵑鵐鵙鵲鶉鶇鶫鵯鵺鶚鶤鶩鶲鷄鷁鶻鶸鶺鷆鷏鷂鷙鷓鷸鷦鷭鷯鷽鸚鸛鸞鹵鹹鹽麁麈麋麌麒麕麑麝麥麩麸麪麭靡黌黎黏黐黔黜點黝黠黥黨黯黴黶黷黹黻黼黽鼇鼈皷鼕鼡鼬鼾齊齒齔齣齟齠齡齦齧齬齪齷齲齶龕龜龠"],
["f4a1","堯槇遙瑤凜熙"],
["f9a1","纊褜鍈銈蓜俉炻昱棈鋹曻彅丨仡仼伀伃伹佖侒侊侚侔俍偀倢俿倞偆偰偂傔僴僘兊兤冝冾凬刕劜劦勀勛匀匇匤卲厓厲叝﨎咜咊咩哿喆坙坥垬埈埇﨏塚增墲夋奓奛奝奣妤妺孖寀甯寘寬尞岦岺峵崧嵓﨑嵂嵭嶸嶹巐弡弴彧德"],
["faa1","忞恝悅悊惞惕愠惲愑愷愰憘戓抦揵摠撝擎敎昀昕昻昉昮昞昤晥晗晙晴晳暙暠暲暿曺朎朗杦枻桒柀栁桄棏﨓楨﨔榘槢樰橫橆橳橾櫢櫤毖氿汜沆汯泚洄涇浯涖涬淏淸淲淼渹湜渧渼溿澈澵濵瀅瀇瀨炅炫焏焄煜煆煇凞燁燾犱"],
["fba1","犾猤猪獷玽珉珖珣珒琇珵琦琪琩琮瑢璉璟甁畯皂皜皞皛皦益睆劯砡硎硤硺礰礼神祥禔福禛竑竧靖竫箞精絈絜綷綠緖繒罇羡羽茁荢荿菇菶葈蒴蕓蕙蕫﨟薰蘒﨡蠇裵訒訷詹誧誾諟諸諶譓譿賰賴贒赶﨣軏﨤逸遧郞都鄕鄧釚"],
["fca1","釗釞釭釮釤釥鈆鈐鈊鈺鉀鈼鉎鉙鉑鈹鉧銧鉷鉸鋧鋗鋙鋐﨧鋕鋠鋓錥錡鋻﨨錞鋿錝錂鍰鍗鎤鏆鏞鏸鐱鑅鑈閒隆﨩隝隯霳霻靃靍靏靑靕顗顥飯飼餧館馞驎髙髜魵魲鮏鮱鮻鰀鵰鵫鶴鸙黑"],
["fcf1","ⅰ",9,"￢￤＇＂"],
["8fa2af","˘ˇ¸˙˝¯˛˚～΄΅"],
["8fa2c2","¡¦¿"],
["8fa2eb","ºª©®™¤№"],
["8fa6e1","ΆΈΉΊΪ"],
["8fa6e7","Ό"],
["8fa6e9","ΎΫ"],
["8fa6ec","Ώ"],
["8fa6f1","άέήίϊΐόςύϋΰώ"],
["8fa7c2","Ђ",10,"ЎЏ"],
["8fa7f2","ђ",10,"ўџ"],
["8fa9a1","ÆĐ"],
["8fa9a4","Ħ"],
["8fa9a6","Ĳ"],
["8fa9a8","ŁĿ"],
["8fa9ab","ŊØŒ"],
["8fa9af","ŦÞ"],
["8fa9c1","æđðħıĳĸłŀŉŋøœßŧþ"],
["8faaa1","ÁÀÄÂĂǍĀĄÅÃĆĈČÇĊĎÉÈËÊĚĖĒĘ"],
["8faaba","ĜĞĢĠĤÍÌÏÎǏİĪĮĨĴĶĹĽĻŃŇŅÑÓÒÖÔǑŐŌÕŔŘŖŚŜŠŞŤŢÚÙÜÛŬǓŰŪŲŮŨǗǛǙǕŴÝŸŶŹŽŻ"],
["8faba1","áàäâăǎāąåãćĉčçċďéèëêěėēęǵĝğ"],
["8fabbd","ġĥíìïîǐ"],
["8fabc5","īįĩĵķĺľļńňņñóòöôǒőōõŕřŗśŝšşťţúùüûŭǔűūųůũǘǜǚǖŵýÿŷźžż"],
["8fb0a1","丂丄丅丌丒丟丣两丨丫丮丯丰丵乀乁乄乇乑乚乜乣乨乩乴乵乹乿亍亖亗亝亯亹仃仐仚仛仠仡仢仨仯仱仳仵份仾仿伀伂伃伈伋伌伒伕伖众伙伮伱你伳伵伷伹伻伾佀佂佈佉佋佌佒佔佖佘佟佣佪佬佮佱佷佸佹佺佽佾侁侂侄"],
["8fb1a1","侅侉侊侌侎侐侒侓侔侗侙侚侞侟侲侷侹侻侼侽侾俀俁俅俆俈俉俋俌俍俏俒俜俠俢俰俲俼俽俿倀倁倄倇倊倌倎倐倓倗倘倛倜倝倞倢倧倮倰倲倳倵偀偁偂偅偆偊偌偎偑偒偓偗偙偟偠偢偣偦偧偪偭偰偱倻傁傃傄傆傊傎傏傐"],
["8fb2a1","傒傓傔傖傛傜傞",4,"傪傯傰傹傺傽僀僃僄僇僌僎僐僓僔僘僜僝僟僢僤僦僨僩僯僱僶僺僾儃儆儇儈儋儌儍儎僲儐儗儙儛儜儝儞儣儧儨儬儭儯儱儳儴儵儸儹兂兊兏兓兕兗兘兟兤兦兾冃冄冋冎冘冝冡冣冭冸冺冼冾冿凂"],
["8fb3a1","凈减凑凒凓凕凘凞凢凥凮凲凳凴凷刁刂刅划刓刕刖刘刢刨刱刲刵刼剅剉剕剗剘剚剜剟剠剡剦剮剷剸剹劀劂劅劊劌劓劕劖劗劘劚劜劤劥劦劧劯劰劶劷劸劺劻劽勀勄勆勈勌勏勑勔勖勛勜勡勥勨勩勪勬勰勱勴勶勷匀匃匊匋"],
["8fb4a1","匌匑匓匘匛匜匞匟匥匧匨匩匫匬匭匰匲匵匼匽匾卂卌卋卙卛卡卣卥卬卭卲卹卾厃厇厈厎厓厔厙厝厡厤厪厫厯厲厴厵厷厸厺厽叀叅叏叒叓叕叚叝叞叠另叧叵吂吓吚吡吧吨吪启吱吴吵呃呄呇呍呏呞呢呤呦呧呩呫呭呮呴呿"],
["8fb5a1","咁咃咅咈咉咍咑咕咖咜咟咡咦咧咩咪咭咮咱咷咹咺咻咿哆哊响哎哠哪哬哯哶哼哾哿唀唁唅唈唉唌唍唎唕唪唫唲唵唶唻唼唽啁啇啉啊啍啐啑啘啚啛啞啠啡啤啦啿喁喂喆喈喎喏喑喒喓喔喗喣喤喭喲喿嗁嗃嗆嗉嗋嗌嗎嗑嗒"],
["8fb6a1","嗓嗗嗘嗛嗞嗢嗩嗶嗿嘅嘈嘊嘍",5,"嘙嘬嘰嘳嘵嘷嘹嘻嘼嘽嘿噀噁噃噄噆噉噋噍噏噔噞噠噡噢噣噦噩噭噯噱噲噵嚄嚅嚈嚋嚌嚕嚙嚚嚝嚞嚟嚦嚧嚨嚩嚫嚬嚭嚱嚳嚷嚾囅囉囊囋囏囐囌囍囙囜囝囟囡囤",4,"囱囫园"],
["8fb7a1","囶囷圁圂圇圊圌圑圕圚圛圝圠圢圣圤圥圩圪圬圮圯圳圴圽圾圿坅坆坌坍坒坢坥坧坨坫坭",4,"坳坴坵坷坹坺坻坼坾垁垃垌垔垗垙垚垜垝垞垟垡垕垧垨垩垬垸垽埇埈埌埏埕埝埞埤埦埧埩埭埰埵埶埸埽埾埿堃堄堈堉埡"],
["8fb8a1","堌堍堛堞堟堠堦堧堭堲堹堿塉塌塍塏塐塕塟塡塤塧塨塸塼塿墀墁墇墈墉墊墌墍墏墐墔墖墝墠墡墢墦墩墱墲壄墼壂壈壍壎壐壒壔壖壚壝壡壢壩壳夅夆夋夌夒夓夔虁夝夡夣夤夨夯夰夳夵夶夿奃奆奒奓奙奛奝奞奟奡奣奫奭"],
["8fb9a1","奯奲奵奶她奻奼妋妌妎妒妕妗妟妤妧妭妮妯妰妳妷妺妼姁姃姄姈姊姍姒姝姞姟姣姤姧姮姯姱姲姴姷娀娄娌娍娎娒娓娞娣娤娧娨娪娭娰婄婅婇婈婌婐婕婞婣婥婧婭婷婺婻婾媋媐媓媖媙媜媞媟媠媢媧媬媱媲媳媵媸媺媻媿"],
["8fbaa1","嫄嫆嫈嫏嫚嫜嫠嫥嫪嫮嫵嫶嫽嬀嬁嬈嬗嬴嬙嬛嬝嬡嬥嬭嬸孁孋孌孒孖孞孨孮孯孼孽孾孿宁宄宆宊宎宐宑宓宔宖宨宩宬宭宯宱宲宷宺宼寀寁寍寏寖",4,"寠寯寱寴寽尌尗尞尟尣尦尩尫尬尮尰尲尵尶屙屚屜屢屣屧屨屩"],
["8fbba1","屭屰屴屵屺屻屼屽岇岈岊岏岒岝岟岠岢岣岦岪岲岴岵岺峉峋峒峝峗峮峱峲峴崁崆崍崒崫崣崤崦崧崱崴崹崽崿嵂嵃嵆嵈嵕嵑嵙嵊嵟嵠嵡嵢嵤嵪嵭嵰嵹嵺嵾嵿嶁嶃嶈嶊嶒嶓嶔嶕嶙嶛嶟嶠嶧嶫嶰嶴嶸嶹巃巇巋巐巎巘巙巠巤"],
["8fbca1","巩巸巹帀帇帍帒帔帕帘帟帠帮帨帲帵帾幋幐幉幑幖幘幛幜幞幨幪",4,"幰庀庋庎庢庤庥庨庪庬庱庳庽庾庿廆廌廋廎廑廒廔廕廜廞廥廫异弆弇弈弎弙弜弝弡弢弣弤弨弫弬弮弰弴弶弻弽弿彀彄彅彇彍彐彔彘彛彠彣彤彧"],
["8fbda1","彯彲彴彵彸彺彽彾徉徍徏徖徜徝徢徧徫徤徬徯徰徱徸忄忇忈忉忋忐",4,"忞忡忢忨忩忪忬忭忮忯忲忳忶忺忼怇怊怍怓怔怗怘怚怟怤怭怳怵恀恇恈恉恌恑恔恖恗恝恡恧恱恾恿悂悆悈悊悎悑悓悕悘悝悞悢悤悥您悰悱悷"],
["8fbea1","悻悾惂惄惈惉惊惋惎惏惔惕惙惛惝惞惢惥惲惵惸惼惽愂愇愊愌愐",4,"愖愗愙愜愞愢愪愫愰愱愵愶愷愹慁慅慆慉慞慠慬慲慸慻慼慿憀憁憃憄憋憍憒憓憗憘憜憝憟憠憥憨憪憭憸憹憼懀懁懂懎懏懕懜懝懞懟懡懢懧懩懥"],
["8fbfa1","懬懭懯戁戃戄戇戓戕戜戠戢戣戧戩戫戹戽扂扃扄扆扌扐扑扒扔扖扚扜扤扭扯扳扺扽抍抎抏抐抦抨抳抶抷抺抾抿拄拎拕拖拚拪拲拴拼拽挃挄挊挋挍挐挓挖挘挩挪挭挵挶挹挼捁捂捃捄捆捊捋捎捒捓捔捘捛捥捦捬捭捱捴捵"],
["8fc0a1","捸捼捽捿掂掄掇掊掐掔掕掙掚掞掤掦掭掮掯掽揁揅揈揎揑揓揔揕揜揠揥揪揬揲揳揵揸揹搉搊搐搒搔搘搞搠搢搤搥搩搪搯搰搵搽搿摋摏摑摒摓摔摚摛摜摝摟摠摡摣摭摳摴摻摽撅撇撏撐撑撘撙撛撝撟撡撣撦撨撬撳撽撾撿"],
["8fc1a1","擄擉擊擋擌擎擐擑擕擗擤擥擩擪擭擰擵擷擻擿攁攄攈攉攊攏攓攔攖攙攛攞攟攢攦攩攮攱攺攼攽敃敇敉敐敒敔敟敠敧敫敺敽斁斅斊斒斕斘斝斠斣斦斮斲斳斴斿旂旈旉旎旐旔旖旘旟旰旲旴旵旹旾旿昀昄昈昉昍昑昒昕昖昝"],
["8fc2a1","昞昡昢昣昤昦昩昪昫昬昮昰昱昳昹昷晀晅晆晊晌晑晎晗晘晙晛晜晠晡曻晪晫晬晾晳晵晿晷晸晹晻暀晼暋暌暍暐暒暙暚暛暜暟暠暤暭暱暲暵暻暿曀曂曃曈曌曎曏曔曛曟曨曫曬曮曺朅朇朎朓朙朜朠朢朳朾杅杇杈杌杔杕杝"],
["8fc3a1","杦杬杮杴杶杻极构枎枏枑枓枖枘枙枛枰枱枲枵枻枼枽柹柀柂柃柅柈柉柒柗柙柜柡柦柰柲柶柷桒栔栙栝栟栨栧栬栭栯栰栱栳栻栿桄桅桊桌桕桗桘桛桫桮",4,"桵桹桺桻桼梂梄梆梈梖梘梚梜梡梣梥梩梪梮梲梻棅棈棌棏"],
["8fc4a1","棐棑棓棖棙棜棝棥棨棪棫棬棭棰棱棵棶棻棼棽椆椉椊椐椑椓椖椗椱椳椵椸椻楂楅楉楎楗楛楣楤楥楦楨楩楬楰楱楲楺楻楿榀榍榒榖榘榡榥榦榨榫榭榯榷榸榺榼槅槈槑槖槗槢槥槮槯槱槳槵槾樀樁樃樏樑樕樚樝樠樤樨樰樲"],
["8fc5a1","樴樷樻樾樿橅橆橉橊橎橐橑橒橕橖橛橤橧橪橱橳橾檁檃檆檇檉檋檑檛檝檞檟檥檫檯檰檱檴檽檾檿櫆櫉櫈櫌櫐櫔櫕櫖櫜櫝櫤櫧櫬櫰櫱櫲櫼櫽欂欃欆欇欉欏欐欑欗欛欞欤欨欫欬欯欵欶欻欿歆歊歍歒歖歘歝歠歧歫歮歰歵歽"],
["8fc6a1","歾殂殅殗殛殟殠殢殣殨殩殬殭殮殰殸殹殽殾毃毄毉毌毖毚毡毣毦毧毮毱毷毹毿氂氄氅氉氍氎氐氒氙氟氦氧氨氬氮氳氵氶氺氻氿汊汋汍汏汒汔汙汛汜汫汭汯汴汶汸汹汻沅沆沇沉沔沕沗沘沜沟沰沲沴泂泆泍泏泐泑泒泔泖"],
["8fc7a1","泚泜泠泧泩泫泬泮泲泴洄洇洊洎洏洑洓洚洦洧洨汧洮洯洱洹洼洿浗浞浟浡浥浧浯浰浼涂涇涑涒涔涖涗涘涪涬涴涷涹涽涿淄淈淊淎淏淖淛淝淟淠淢淥淩淯淰淴淶淼渀渄渞渢渧渲渶渹渻渼湄湅湈湉湋湏湑湒湓湔湗湜湝湞"],
["8fc8a1","湢湣湨湳湻湽溍溓溙溠溧溭溮溱溳溻溿滀滁滃滇滈滊滍滎滏滫滭滮滹滻滽漄漈漊漌漍漖漘漚漛漦漩漪漯漰漳漶漻漼漭潏潑潒潓潗潙潚潝潞潡潢潨潬潽潾澃澇澈澋澌澍澐澒澓澔澖澚澟澠澥澦澧澨澮澯澰澵澶澼濅濇濈濊"],
["8fc9a1","濚濞濨濩濰濵濹濼濽瀀瀅瀆瀇瀍瀗瀠瀣瀯瀴瀷瀹瀼灃灄灈灉灊灋灔灕灝灞灎灤灥灬灮灵灶灾炁炅炆炔",4,"炛炤炫炰炱炴炷烊烑烓烔烕烖烘烜烤烺焃",4,"焋焌焏焞焠焫焭焯焰焱焸煁煅煆煇煊煋煐煒煗煚煜煞煠"],
["8fcaa1","煨煹熀熅熇熌熒熚熛熠熢熯熰熲熳熺熿燀燁燄燋燌燓燖燙燚燜燸燾爀爇爈爉爓爗爚爝爟爤爫爯爴爸爹牁牂牃牅牎牏牐牓牕牖牚牜牞牠牣牨牫牮牯牱牷牸牻牼牿犄犉犍犎犓犛犨犭犮犱犴犾狁狇狉狌狕狖狘狟狥狳狴狺狻"],
["8fcba1","狾猂猄猅猇猋猍猒猓猘猙猞猢猤猧猨猬猱猲猵猺猻猽獃獍獐獒獖獘獝獞獟獠獦獧獩獫獬獮獯獱獷獹獼玀玁玃玅玆玎玐玓玕玗玘玜玞玟玠玢玥玦玪玫玭玵玷玹玼玽玿珅珆珉珋珌珏珒珓珖珙珝珡珣珦珧珩珴珵珷珹珺珻珽"],
["8fcca1","珿琀琁琄琇琊琑琚琛琤琦琨",9,"琹瑀瑃瑄瑆瑇瑋瑍瑑瑒瑗瑝瑢瑦瑧瑨瑫瑭瑮瑱瑲璀璁璅璆璇璉璏璐璑璒璘璙璚璜璟璠璡璣璦璨璩璪璫璮璯璱璲璵璹璻璿瓈瓉瓌瓐瓓瓘瓚瓛瓞瓟瓤瓨瓪瓫瓯瓴瓺瓻瓼瓿甆"],
["8fcda1","甒甖甗甠甡甤甧甩甪甯甶甹甽甾甿畀畃畇畈畎畐畒畗畞畟畡畯畱畹",5,"疁疅疐疒疓疕疙疜疢疤疴疺疿痀痁痄痆痌痎痏痗痜痟痠痡痤痧痬痮痯痱痹瘀瘂瘃瘄瘇瘈瘊瘌瘏瘒瘓瘕瘖瘙瘛瘜瘝瘞瘣瘥瘦瘩瘭瘲瘳瘵瘸瘹"],
["8fcea1","瘺瘼癊癀癁癃癄癅癉癋癕癙癟癤癥癭癮癯癱癴皁皅皌皍皕皛皜皝皟皠皢",6,"皪皭皽盁盅盉盋盌盎盔盙盠盦盨盬盰盱盶盹盼眀眆眊眎眒眔眕眗眙眚眜眢眨眭眮眯眴眵眶眹眽眾睂睅睆睊睍睎睏睒睖睗睜睞睟睠睢"],
["8fcfa1","睤睧睪睬睰睲睳睴睺睽瞀瞄瞌瞍瞔瞕瞖瞚瞟瞢瞧瞪瞮瞯瞱瞵瞾矃矉矑矒矕矙矞矟矠矤矦矪矬矰矱矴矸矻砅砆砉砍砎砑砝砡砢砣砭砮砰砵砷硃硄硇硈硌硎硒硜硞硠硡硣硤硨硪确硺硾碊碏碔碘碡碝碞碟碤碨碬碭碰碱碲碳"],
["8fd0a1","碻碽碿磇磈磉磌磎磒磓磕磖磤磛磟磠磡磦磪磲磳礀磶磷磺磻磿礆礌礐礚礜礞礟礠礥礧礩礭礱礴礵礻礽礿祄祅祆祊祋祏祑祔祘祛祜祧祩祫祲祹祻祼祾禋禌禑禓禔禕禖禘禛禜禡禨禩禫禯禱禴禸离秂秄秇秈秊秏秔秖秚秝秞"],
["8fd1a1","秠秢秥秪秫秭秱秸秼稂稃稇稉稊稌稑稕稛稞稡稧稫稭稯稰稴稵稸稹稺穄穅穇穈穌穕穖穙穜穝穟穠穥穧穪穭穵穸穾窀窂窅窆窊窋窐窑窔窞窠窣窬窳窵窹窻窼竆竉竌竎竑竛竨竩竫竬竱竴竻竽竾笇笔笟笣笧笩笪笫笭笮笯笰"],
["8fd2a1","笱笴笽笿筀筁筇筎筕筠筤筦筩筪筭筯筲筳筷箄箉箎箐箑箖箛箞箠箥箬箯箰箲箵箶箺箻箼箽篂篅篈篊篔篖篗篙篚篛篨篪篲篴篵篸篹篺篼篾簁簂簃簄簆簉簋簌簎簏簙簛簠簥簦簨簬簱簳簴簶簹簺籆籊籕籑籒籓籙",5],
["8fd3a1","籡籣籧籩籭籮籰籲籹籼籽粆粇粏粔粞粠粦粰粶粷粺粻粼粿糄糇糈糉糍糏糓糔糕糗糙糚糝糦糩糫糵紃紇紈紉紏紑紒紓紖紝紞紣紦紪紭紱紼紽紾絀絁絇絈絍絑絓絗絙絚絜絝絥絧絪絰絸絺絻絿綁綂綃綅綆綈綋綌綍綑綖綗綝"],
["8fd4a1","綞綦綧綪綳綶綷綹緂",4,"緌緍緎緗緙縀緢緥緦緪緫緭緱緵緶緹緺縈縐縑縕縗縜縝縠縧縨縬縭縯縳縶縿繄繅繇繎繐繒繘繟繡繢繥繫繮繯繳繸繾纁纆纇纊纍纑纕纘纚纝纞缼缻缽缾缿罃罄罇罏罒罓罛罜罝罡罣罤罥罦罭"],
["8fd5a1","罱罽罾罿羀羋羍羏羐羑羖羗羜羡羢羦羪羭羴羼羿翀翃翈翎翏翛翟翣翥翨翬翮翯翲翺翽翾翿耇耈耊耍耎耏耑耓耔耖耝耞耟耠耤耦耬耮耰耴耵耷耹耺耼耾聀聄聠聤聦聭聱聵肁肈肎肜肞肦肧肫肸肹胈胍胏胒胔胕胗胘胠胭胮"],
["8fd6a1","胰胲胳胶胹胺胾脃脋脖脗脘脜脞脠脤脧脬脰脵脺脼腅腇腊腌腒腗腠腡腧腨腩腭腯腷膁膐膄膅膆膋膎膖膘膛膞膢膮膲膴膻臋臃臅臊臎臏臕臗臛臝臞臡臤臫臬臰臱臲臵臶臸臹臽臿舀舃舏舓舔舙舚舝舡舢舨舲舴舺艃艄艅艆"],
["8fd7a1","艋艎艏艑艖艜艠艣艧艭艴艻艽艿芀芁芃芄芇芉芊芎芑芔芖芘芚芛芠芡芣芤芧芨芩芪芮芰芲芴芷芺芼芾芿苆苐苕苚苠苢苤苨苪苭苯苶苷苽苾茀茁茇茈茊茋荔茛茝茞茟茡茢茬茭茮茰茳茷茺茼茽荂荃荄荇荍荎荑荕荖荗荰荸"],
["8fd8a1","荽荿莀莂莄莆莍莒莔莕莘莙莛莜莝莦莧莩莬莾莿菀菇菉菏菐菑菔菝荓菨菪菶菸菹菼萁萆萊萏萑萕萙莭萯萹葅葇葈葊葍葏葑葒葖葘葙葚葜葠葤葥葧葪葰葳葴葶葸葼葽蒁蒅蒒蒓蒕蒞蒦蒨蒩蒪蒯蒱蒴蒺蒽蒾蓀蓂蓇蓈蓌蓏蓓"],
["8fd9a1","蓜蓧蓪蓯蓰蓱蓲蓷蔲蓺蓻蓽蔂蔃蔇蔌蔎蔐蔜蔞蔢蔣蔤蔥蔧蔪蔫蔯蔳蔴蔶蔿蕆蕏",4,"蕖蕙蕜",6,"蕤蕫蕯蕹蕺蕻蕽蕿薁薅薆薉薋薌薏薓薘薝薟薠薢薥薧薴薶薷薸薼薽薾薿藂藇藊藋藎薭藘藚藟藠藦藨藭藳藶藼"],
["8fdaa1","藿蘀蘄蘅蘍蘎蘐蘑蘒蘘蘙蘛蘞蘡蘧蘩蘶蘸蘺蘼蘽虀虂虆虒虓虖虗虘虙虝虠",4,"虩虬虯虵虶虷虺蚍蚑蚖蚘蚚蚜蚡蚦蚧蚨蚭蚱蚳蚴蚵蚷蚸蚹蚿蛀蛁蛃蛅蛑蛒蛕蛗蛚蛜蛠蛣蛥蛧蚈蛺蛼蛽蜄蜅蜇蜋蜎蜏蜐蜓蜔蜙蜞蜟蜡蜣"],
["8fdba1","蜨蜮蜯蜱蜲蜹蜺蜼蜽蜾蝀蝃蝅蝍蝘蝝蝡蝤蝥蝯蝱蝲蝻螃",6,"螋螌螐螓螕螗螘螙螞螠螣螧螬螭螮螱螵螾螿蟁蟈蟉蟊蟎蟕蟖蟙蟚蟜蟟蟢蟣蟤蟪蟫蟭蟱蟳蟸蟺蟿蠁蠃蠆蠉蠊蠋蠐蠙蠒蠓蠔蠘蠚蠛蠜蠞蠟蠨蠭蠮蠰蠲蠵"],
["8fdca1","蠺蠼衁衃衅衈衉衊衋衎衑衕衖衘衚衜衟衠衤衩衱衹衻袀袘袚袛袜袟袠袨袪袺袽袾裀裊",4,"裑裒裓裛裞裧裯裰裱裵裷褁褆褍褎褏褕褖褘褙褚褜褠褦褧褨褰褱褲褵褹褺褾襀襂襅襆襉襏襒襗襚襛襜襡襢襣襫襮襰襳襵襺"],
["8fdda1","襻襼襽覉覍覐覔覕覛覜覟覠覥覰覴覵覶覷覼觔",4,"觥觩觫觭觱觳觶觹觽觿訄訅訇訏訑訒訔訕訞訠訢訤訦訫訬訯訵訷訽訾詀詃詅詇詉詍詎詓詖詗詘詜詝詡詥詧詵詶詷詹詺詻詾詿誀誃誆誋誏誐誒誖誗誙誟誧誩誮誯誳"],
["8fdea1","誶誷誻誾諃諆諈諉諊諑諓諔諕諗諝諟諬諰諴諵諶諼諿謅謆謋謑謜謞謟謊謭謰謷謼譂",4,"譈譒譓譔譙譍譞譣譭譶譸譹譼譾讁讄讅讋讍讏讔讕讜讞讟谸谹谽谾豅豇豉豋豏豑豓豔豗豘豛豝豙豣豤豦豨豩豭豳豵豶豻豾貆"],
["8fdfa1","貇貋貐貒貓貙貛貜貤貹貺賅賆賉賋賏賖賕賙賝賡賨賬賯賰賲賵賷賸賾賿贁贃贉贒贗贛赥赩赬赮赿趂趄趈趍趐趑趕趞趟趠趦趫趬趯趲趵趷趹趻跀跅跆跇跈跊跎跑跔跕跗跙跤跥跧跬跰趼跱跲跴跽踁踄踅踆踋踑踔踖踠踡踢"],
["8fe0a1","踣踦踧踱踳踶踷踸踹踽蹀蹁蹋蹍蹎蹏蹔蹛蹜蹝蹞蹡蹢蹩蹬蹭蹯蹰蹱蹹蹺蹻躂躃躉躐躒躕躚躛躝躞躢躧躩躭躮躳躵躺躻軀軁軃軄軇軏軑軔軜軨軮軰軱軷軹軺軭輀輂輇輈輏輐輖輗輘輞輠輡輣輥輧輨輬輭輮輴輵輶輷輺轀轁"],
["8fe1a1","轃轇轏轑",4,"轘轝轞轥辝辠辡辤辥辦辵辶辸达迀迁迆迊迋迍运迒迓迕迠迣迤迨迮迱迵迶迻迾适逄逈逌逘逛逨逩逯逪逬逭逳逴逷逿遃遄遌遛遝遢遦遧遬遰遴遹邅邈邋邌邎邐邕邗邘邙邛邠邡邢邥邰邲邳邴邶邽郌邾郃"],
["8fe2a1","郄郅郇郈郕郗郘郙郜郝郟郥郒郶郫郯郰郴郾郿鄀鄄鄅鄆鄈鄍鄐鄔鄖鄗鄘鄚鄜鄞鄠鄥鄢鄣鄧鄩鄮鄯鄱鄴鄶鄷鄹鄺鄼鄽酃酇酈酏酓酗酙酚酛酡酤酧酭酴酹酺酻醁醃醅醆醊醎醑醓醔醕醘醞醡醦醨醬醭醮醰醱醲醳醶醻醼醽醿"],
["8fe3a1","釂釃釅釓釔釗釙釚釞釤釥釩釪釬",5,"釷釹釻釽鈀鈁鈄鈅鈆鈇鈉鈊鈌鈐鈒鈓鈖鈘鈜鈝鈣鈤鈥鈦鈨鈮鈯鈰鈳鈵鈶鈸鈹鈺鈼鈾鉀鉂鉃鉆鉇鉊鉍鉎鉏鉑鉘鉙鉜鉝鉠鉡鉥鉧鉨鉩鉮鉯鉰鉵",4,"鉻鉼鉽鉿銈銉銊銍銎銒銗"],
["8fe4a1","銙銟銠銤銥銧銨銫銯銲銶銸銺銻銼銽銿",4,"鋅鋆鋇鋈鋋鋌鋍鋎鋐鋓鋕鋗鋘鋙鋜鋝鋟鋠鋡鋣鋥鋧鋨鋬鋮鋰鋹鋻鋿錀錂錈錍錑錔錕錜錝錞錟錡錤錥錧錩錪錳錴錶錷鍇鍈鍉鍐鍑鍒鍕鍗鍘鍚鍞鍤鍥鍧鍩鍪鍭鍯鍰鍱鍳鍴鍶"],
["8fe5a1","鍺鍽鍿鎀鎁鎂鎈鎊鎋鎍鎏鎒鎕鎘鎛鎞鎡鎣鎤鎦鎨鎫鎴鎵鎶鎺鎩鏁鏄鏅鏆鏇鏉",4,"鏓鏙鏜鏞鏟鏢鏦鏧鏹鏷鏸鏺鏻鏽鐁鐂鐄鐈鐉鐍鐎鐏鐕鐖鐗鐟鐮鐯鐱鐲鐳鐴鐻鐿鐽鑃鑅鑈鑊鑌鑕鑙鑜鑟鑡鑣鑨鑫鑭鑮鑯鑱鑲钄钃镸镹"],
["8fe6a1","镾閄閈閌閍閎閝閞閟閡閦閩閫閬閴閶閺閽閿闆闈闉闋闐闑闒闓闙闚闝闞闟闠闤闦阝阞阢阤阥阦阬阱阳阷阸阹阺阼阽陁陒陔陖陗陘陡陮陴陻陼陾陿隁隂隃隄隉隑隖隚隝隟隤隥隦隩隮隯隳隺雊雒嶲雘雚雝雞雟雩雯雱雺霂"],
["8fe7a1","霃霅霉霚霛霝霡霢霣霨霱霳靁靃靊靎靏靕靗靘靚靛靣靧靪靮靳靶靷靸靻靽靿鞀鞉鞕鞖鞗鞙鞚鞞鞟鞢鞬鞮鞱鞲鞵鞶鞸鞹鞺鞼鞾鞿韁韄韅韇韉韊韌韍韎韐韑韔韗韘韙韝韞韠韛韡韤韯韱韴韷韸韺頇頊頙頍頎頔頖頜頞頠頣頦"],
["8fe8a1","頫頮頯頰頲頳頵頥頾顄顇顊顑顒顓顖顗顙顚顢顣顥顦顪顬颫颭颮颰颴颷颸颺颻颿飂飅飈飌飡飣飥飦飧飪飳飶餂餇餈餑餕餖餗餚餛餜餟餢餦餧餫餱",4,"餹餺餻餼饀饁饆饇饈饍饎饔饘饙饛饜饞饟饠馛馝馟馦馰馱馲馵"],
["8fe9a1","馹馺馽馿駃駉駓駔駙駚駜駞駧駪駫駬駰駴駵駹駽駾騂騃騄騋騌騐騑騖騞騠騢騣騤騧騭騮騳騵騶騸驇驁驄驊驋驌驎驑驔驖驝骪骬骮骯骲骴骵骶骹骻骾骿髁髃髆髈髎髐髒髕髖髗髛髜髠髤髥髧髩髬髲髳髵髹髺髽髿",4],
["8feaa1","鬄鬅鬈鬉鬋鬌鬍鬎鬐鬒鬖鬙鬛鬜鬠鬦鬫鬭鬳鬴鬵鬷鬹鬺鬽魈魋魌魕魖魗魛魞魡魣魥魦魨魪",4,"魳魵魷魸魹魿鮀鮄鮅鮆鮇鮉鮊鮋鮍鮏鮐鮔鮚鮝鮞鮦鮧鮩鮬鮰鮱鮲鮷鮸鮻鮼鮾鮿鯁鯇鯈鯎鯐鯗鯘鯝鯟鯥鯧鯪鯫鯯鯳鯷鯸"],
["8feba1","鯹鯺鯽鯿鰀鰂鰋鰏鰑鰖鰘鰙鰚鰜鰞鰢鰣鰦",4,"鰱鰵鰶鰷鰽鱁鱃鱄鱅鱉鱊鱎鱏鱐鱓鱔鱖鱘鱛鱝鱞鱟鱣鱩鱪鱜鱫鱨鱮鱰鱲鱵鱷鱻鳦鳲鳷鳹鴋鴂鴑鴗鴘鴜鴝鴞鴯鴰鴲鴳鴴鴺鴼鵅鴽鵂鵃鵇鵊鵓鵔鵟鵣鵢鵥鵩鵪鵫鵰鵶鵷鵻"],
["8feca1","鵼鵾鶃鶄鶆鶊鶍鶎鶒鶓鶕鶖鶗鶘鶡鶪鶬鶮鶱鶵鶹鶼鶿鷃鷇鷉鷊鷔鷕鷖鷗鷚鷞鷟鷠鷥鷧鷩鷫鷮鷰鷳鷴鷾鸊鸂鸇鸎鸐鸑鸒鸕鸖鸙鸜鸝鹺鹻鹼麀麂麃麄麅麇麎麏麖麘麛麞麤麨麬麮麯麰麳麴麵黆黈黋黕黟黤黧黬黭黮黰黱黲黵"],
["8feda1","黸黿鼂鼃鼉鼏鼐鼑鼒鼔鼖鼗鼙鼚鼛鼟鼢鼦鼪鼫鼯鼱鼲鼴鼷鼹鼺鼼鼽鼿齁齃",4,"齓齕齖齗齘齚齝齞齨齩齭",4,"齳齵齺齽龏龐龑龒龔龖龗龞龡龢龣龥"]
]
;

var eucjp$1 = /*#__PURE__*/Object.freeze({
            default: eucjp
});

var cp936 = [
["0","\u0000",127,"€"],
["8140","丂丄丅丆丏丒丗丟丠両丣並丩丮丯丱丳丵丷丼乀乁乂乄乆乊乑乕乗乚乛乢乣乤乥乧乨乪",5,"乲乴",9,"乿",6,"亇亊"],
["8180","亐亖亗亙亜亝亞亣亪亯亰亱亴亶亷亸亹亼亽亾仈仌仏仐仒仚仛仜仠仢仦仧仩仭仮仯仱仴仸仹仺仼仾伀伂",6,"伋伌伒",4,"伜伝伡伣伨伩伬伭伮伱伳伵伷伹伻伾",4,"佄佅佇",5,"佒佔佖佡佢佦佨佪佫佭佮佱佲併佷佸佹佺佽侀侁侂侅來侇侊侌侎侐侒侓侕侖侘侙侚侜侞侟価侢"],
["8240","侤侫侭侰",4,"侶",8,"俀俁係俆俇俈俉俋俌俍俒",4,"俙俛俠俢俤俥俧俫俬俰俲俴俵俶俷俹俻俼俽俿",11],
["8280","個倎倐們倓倕倖倗倛倝倞倠倢倣値倧倫倯",10,"倻倽倿偀偁偂偄偅偆偉偊偋偍偐",4,"偖偗偘偙偛偝",7,"偦",5,"偭",8,"偸偹偺偼偽傁傂傃傄傆傇傉傊傋傌傎",20,"傤傦傪傫傭",4,"傳",6,"傼"],
["8340","傽",17,"僐",5,"僗僘僙僛",10,"僨僩僪僫僯僰僱僲僴僶",4,"僼",9,"儈"],
["8380","儉儊儌",5,"儓",13,"儢",28,"兂兇兊兌兎兏児兒兓兗兘兙兛兝",4,"兣兤兦內兩兪兯兲兺兾兿冃冄円冇冊冋冎冏冐冑冓冔冘冚冝冞冟冡冣冦",4,"冭冮冴冸冹冺冾冿凁凂凃凅凈凊凍凎凐凒",5],
["8440","凘凙凚凜凞凟凢凣凥",5,"凬凮凱凲凴凷凾刄刅刉刋刌刏刐刓刔刕刜刞刟刡刢刣別刦刧刪刬刯刱刲刴刵刼刾剄",5,"剋剎剏剒剓剕剗剘"],
["8480","剙剚剛剝剟剠剢剣剤剦剨剫剬剭剮剰剱剳",9,"剾劀劃",4,"劉",6,"劑劒劔",6,"劜劤劥劦劧劮劯劰労",9,"勀勁勂勄勅勆勈勊勌勍勎勏勑勓勔動勗務",5,"勠勡勢勣勥",10,"勱",7,"勻勼勽匁匂匃匄匇匉匊匋匌匎"],
["8540","匑匒匓匔匘匛匜匞匟匢匤匥匧匨匩匫匬匭匯",9,"匼匽區卂卄卆卋卌卍卐協単卙卛卝卥卨卪卬卭卲卶卹卻卼卽卾厀厁厃厇厈厊厎厏"],
["8580","厐",4,"厖厗厙厛厜厞厠厡厤厧厪厫厬厭厯",6,"厷厸厹厺厼厽厾叀參",4,"収叏叐叒叓叕叚叜叝叞叡叢叧叴叺叾叿吀吂吅吇吋吔吘吙吚吜吢吤吥吪吰吳吶吷吺吽吿呁呂呄呅呇呉呌呍呎呏呑呚呝",4,"呣呥呧呩",7,"呴呹呺呾呿咁咃咅咇咈咉咊咍咑咓咗咘咜咞咟咠咡"],
["8640","咢咥咮咰咲咵咶咷咹咺咼咾哃哅哊哋哖哘哛哠",4,"哫哬哯哰哱哴",5,"哻哾唀唂唃唄唅唈唊",4,"唒唓唕",5,"唜唝唞唟唡唥唦"],
["8680","唨唩唫唭唲唴唵唶唸唹唺唻唽啀啂啅啇啈啋",4,"啑啒啓啔啗",4,"啝啞啟啠啢啣啨啩啫啯",5,"啹啺啽啿喅喆喌喍喎喐喒喓喕喖喗喚喛喞喠",6,"喨",8,"喲喴営喸喺喼喿",4,"嗆嗇嗈嗊嗋嗎嗏嗐嗕嗗",4,"嗞嗠嗢嗧嗩嗭嗮嗰嗱嗴嗶嗸",4,"嗿嘂嘃嘄嘅"],
["8740","嘆嘇嘊嘋嘍嘐",7,"嘙嘚嘜嘝嘠嘡嘢嘥嘦嘨嘩嘪嘫嘮嘯嘰嘳嘵嘷嘸嘺嘼嘽嘾噀",11,"噏",4,"噕噖噚噛噝",4],
["8780","噣噥噦噧噭噮噯噰噲噳噴噵噷噸噹噺噽",7,"嚇",6,"嚐嚑嚒嚔",14,"嚤",10,"嚰",6,"嚸嚹嚺嚻嚽",12,"囋",8,"囕囖囘囙囜団囥",5,"囬囮囯囲図囶囷囸囻囼圀圁圂圅圇國",6],
["8840","園",9,"圝圞圠圡圢圤圥圦圧圫圱圲圴",4,"圼圽圿坁坃坄坅坆坈坉坋坒",4,"坘坙坢坣坥坧坬坮坰坱坲坴坵坸坹坺坽坾坿垀"],
["8880","垁垇垈垉垊垍",4,"垔",6,"垜垝垞垟垥垨垪垬垯垰垱垳垵垶垷垹",8,"埄",6,"埌埍埐埑埓埖埗埛埜埞埡埢埣埥",7,"埮埰埱埲埳埵埶執埻埼埾埿堁堃堄堅堈堉堊堌堎堏堐堒堓堔堖堗堘堚堛堜堝堟堢堣堥",4,"堫",4,"報堲堳場堶",7],
["8940","堾",5,"塅",6,"塎塏塐塒塓塕塖塗塙",4,"塟",5,"塦",4,"塭",16,"塿墂墄墆墇墈墊墋墌"],
["8980","墍",4,"墔",4,"墛墜墝墠",7,"墪",17,"墽墾墿壀壂壃壄壆",10,"壒壓壔壖",13,"壥",5,"壭壯壱売壴壵壷壸壺",7,"夃夅夆夈",4,"夎夐夑夒夓夗夘夛夝夞夠夡夢夣夦夨夬夰夲夳夵夶夻"],
["8a40","夽夾夿奀奃奅奆奊奌奍奐奒奓奙奛",4,"奡奣奤奦",12,"奵奷奺奻奼奾奿妀妅妉妋妌妎妏妐妑妔妕妘妚妛妜妝妟妠妡妢妦"],
["8a80","妧妬妭妰妱妳",5,"妺妼妽妿",6,"姇姈姉姌姍姎姏姕姖姙姛姞",4,"姤姦姧姩姪姫姭",11,"姺姼姽姾娀娂娊娋娍娎娏娐娒娔娕娖娗娙娚娛娝娞娡娢娤娦娧娨娪",6,"娳娵娷",4,"娽娾娿婁",4,"婇婈婋",9,"婖婗婘婙婛",5],
["8b40","婡婣婤婥婦婨婩婫",8,"婸婹婻婼婽婾媀",17,"媓",6,"媜",13,"媫媬"],
["8b80","媭",4,"媴媶媷媹",4,"媿嫀嫃",5,"嫊嫋嫍",4,"嫓嫕嫗嫙嫚嫛嫝嫞嫟嫢嫤嫥嫧嫨嫪嫬",4,"嫲",22,"嬊",11,"嬘",25,"嬳嬵嬶嬸",7,"孁",6],
["8c40","孈",7,"孒孖孞孠孡孧孨孫孭孮孯孲孴孶孷學孹孻孼孾孿宂宆宊宍宎宐宑宒宔宖実宧宨宩宬宭宮宯宱宲宷宺宻宼寀寁寃寈寉寊寋寍寎寏"],
["8c80","寑寔",8,"寠寢寣實寧審",4,"寯寱",6,"寽対尀専尃尅將專尋尌對導尐尒尓尗尙尛尞尟尠尡尣尦尨尩尪尫尭尮尯尰尲尳尵尶尷屃屄屆屇屌屍屒屓屔屖屗屘屚屛屜屝屟屢層屧",6,"屰屲",6,"屻屼屽屾岀岃",4,"岉岊岋岎岏岒岓岕岝",4,"岤",4],
["8d40","岪岮岯岰岲岴岶岹岺岻岼岾峀峂峃峅",5,"峌",5,"峓",5,"峚",6,"峢峣峧峩峫峬峮峯峱",9,"峼",4],
["8d80","崁崄崅崈",5,"崏",4,"崕崗崘崙崚崜崝崟",4,"崥崨崪崫崬崯",4,"崵",7,"崿",7,"嵈嵉嵍",10,"嵙嵚嵜嵞",10,"嵪嵭嵮嵰嵱嵲嵳嵵",12,"嶃",21,"嶚嶛嶜嶞嶟嶠"],
["8e40","嶡",21,"嶸",12,"巆",6,"巎",12,"巜巟巠巣巤巪巬巭"],
["8e80","巰巵巶巸",4,"巿帀帄帇帉帊帋帍帎帒帓帗帞",7,"帨",4,"帯帰帲",4,"帹帺帾帿幀幁幃幆",5,"幍",6,"幖",4,"幜幝幟幠幣",14,"幵幷幹幾庁庂広庅庈庉庌庍庎庒庘庛庝庡庢庣庤庨",4,"庮",4,"庴庺庻庼庽庿",6],
["8f40","廆廇廈廋",5,"廔廕廗廘廙廚廜",11,"廩廫",8,"廵廸廹廻廼廽弅弆弇弉弌弍弎弐弒弔弖弙弚弜弝弞弡弢弣弤"],
["8f80","弨弫弬弮弰弲",6,"弻弽弾弿彁",14,"彑彔彙彚彛彜彞彟彠彣彥彧彨彫彮彯彲彴彵彶彸彺彽彾彿徃徆徍徎徏徑従徔徖徚徛徝從徟徠徢",5,"復徫徬徯",5,"徶徸徹徺徻徾",4,"忇忈忊忋忎忓忔忕忚忛応忞忟忢忣忥忦忨忩忬忯忰忲忳忴忶忷忹忺忼怇"],
["9040","怈怉怋怌怐怑怓怗怘怚怞怟怢怣怤怬怭怮怰",4,"怶",4,"怽怾恀恄",6,"恌恎恏恑恓恔恖恗恘恛恜恞恟恠恡恥恦恮恱恲恴恵恷恾悀"],
["9080","悁悂悅悆悇悈悊悋悎悏悐悑悓悕悗悘悙悜悞悡悢悤悥悧悩悪悮悰悳悵悶悷悹悺悽",7,"惇惈惉惌",4,"惒惓惔惖惗惙惛惞惡",4,"惪惱惲惵惷惸惻",4,"愂愃愄愅愇愊愋愌愐",4,"愖愗愘愙愛愜愝愞愡愢愥愨愩愪愬",18,"慀",6],
["9140","慇慉態慍慏慐慒慓慔慖",6,"慞慟慠慡慣慤慥慦慩",6,"慱慲慳慴慶慸",18,"憌憍憏",4,"憕"],
["9180","憖",6,"憞",8,"憪憫憭",9,"憸",5,"憿懀懁懃",4,"應懌",4,"懓懕",16,"懧",13,"懶",8,"戀",5,"戇戉戓戔戙戜戝戞戠戣戦戧戨戩戫戭戯戰戱戲戵戶戸",4,"扂扄扅扆扊"],
["9240","扏扐払扖扗扙扚扜",6,"扤扥扨扱扲扴扵扷扸扺扻扽抁抂抃抅抆抇抈抋",5,"抔抙抜抝択抣抦抧抩抪抭抮抯抰抲抳抴抶抷抸抺抾拀拁"],
["9280","拃拋拏拑拕拝拞拠拡拤拪拫拰拲拵拸拹拺拻挀挃挄挅挆挊挋挌挍挏挐挒挓挔挕挗挘挙挜挦挧挩挬挭挮挰挱挳",5,"挻挼挾挿捀捁捄捇捈捊捑捒捓捔捖",7,"捠捤捥捦捨捪捫捬捯捰捲捳捴捵捸捹捼捽捾捿掁掃掄掅掆掋掍掑掓掔掕掗掙",6,"採掤掦掫掯掱掲掵掶掹掻掽掿揀"],
["9340","揁揂揃揅揇揈揊揋揌揑揓揔揕揗",6,"揟揢揤",4,"揫揬揮揯揰揱揳揵揷揹揺揻揼揾搃搄搆",4,"損搎搑搒搕",5,"搝搟搢搣搤"],
["9380","搥搧搨搩搫搮",5,"搵",4,"搻搼搾摀摂摃摉摋",6,"摓摕摖摗摙",4,"摟",7,"摨摪摫摬摮",9,"摻",6,"撃撆撈",8,"撓撔撗撘撚撛撜撝撟",4,"撥撦撧撨撪撫撯撱撲撳撴撶撹撻撽撾撿擁擃擄擆",6,"擏擑擓擔擕擖擙據"],
["9440","擛擜擝擟擠擡擣擥擧",24,"攁",7,"攊",7,"攓",4,"攙",8],
["9480","攢攣攤攦",4,"攬攭攰攱攲攳攷攺攼攽敀",4,"敆敇敊敋敍敎敐敒敓敔敗敘敚敜敟敠敡敤敥敧敨敩敪敭敮敯敱敳敵敶數",14,"斈斉斊斍斎斏斒斔斕斖斘斚斝斞斠斢斣斦斨斪斬斮斱",7,"斺斻斾斿旀旂旇旈旉旊旍旐旑旓旔旕旘",7,"旡旣旤旪旫"],
["9540","旲旳旴旵旸旹旻",4,"昁昄昅昇昈昉昋昍昐昑昒昖昗昘昚昛昜昞昡昢昣昤昦昩昪昫昬昮昰昲昳昷",4,"昽昿晀時晄",6,"晍晎晐晑晘"],
["9580","晙晛晜晝晞晠晢晣晥晧晩",4,"晱晲晳晵晸晹晻晼晽晿暀暁暃暅暆暈暉暊暋暍暎暏暐暒暓暔暕暘",4,"暞",8,"暩",4,"暯",4,"暵暶暷暸暺暻暼暽暿",25,"曚曞",7,"曧曨曪",5,"曱曵曶書曺曻曽朁朂會"],
["9640","朄朅朆朇朌朎朏朑朒朓朖朘朙朚朜朞朠",5,"朧朩朮朰朲朳朶朷朸朹朻朼朾朿杁杄杅杇杊杋杍杒杔杕杗",4,"杝杢杣杤杦杧杫杬杮東杴杶"],
["9680","杸杹杺杻杽枀枂枃枅枆枈枊枌枍枎枏枑枒枓枔枖枙枛枟枠枡枤枦枩枬枮枱枲枴枹",7,"柂柅",9,"柕柖柗柛柟柡柣柤柦柧柨柪柫柭柮柲柵",7,"柾栁栂栃栄栆栍栐栒栔栕栘",4,"栞栟栠栢",6,"栫",6,"栴栵栶栺栻栿桇桋桍桏桒桖",5],
["9740","桜桝桞桟桪桬",7,"桵桸",8,"梂梄梇",7,"梐梑梒梔梕梖梘",9,"梣梤梥梩梪梫梬梮梱梲梴梶梷梸"],
["9780","梹",6,"棁棃",5,"棊棌棎棏棐棑棓棔棖棗棙棛",4,"棡棢棤",9,"棯棲棳棴棶棷棸棻棽棾棿椀椂椃椄椆",4,"椌椏椑椓",11,"椡椢椣椥",7,"椮椯椱椲椳椵椶椷椸椺椻椼椾楀楁楃",16,"楕楖楘楙楛楜楟"],
["9840","楡楢楤楥楧楨楩楪楬業楯楰楲",4,"楺楻楽楾楿榁榃榅榊榋榌榎",5,"榖榗榙榚榝",9,"榩榪榬榮榯榰榲榳榵榶榸榹榺榼榽"],
["9880","榾榿槀槂",7,"構槍槏槑槒槓槕",5,"槜槝槞槡",11,"槮槯槰槱槳",9,"槾樀",9,"樋",11,"標",5,"樠樢",5,"権樫樬樭樮樰樲樳樴樶",6,"樿",4,"橅橆橈",7,"橑",6,"橚"],
["9940","橜",4,"橢橣橤橦",10,"橲",6,"橺橻橽橾橿檁檂檃檅",8,"檏檒",4,"檘",7,"檡",5],
["9980","檧檨檪檭",114,"欥欦欨",6],
["9a40","欯欰欱欳欴欵欶欸欻欼欽欿歀歁歂歄歅歈歊歋歍",11,"歚",7,"歨歩歫",13,"歺歽歾歿殀殅殈"],
["9a80","殌殎殏殐殑殔殕殗殘殙殜",4,"殢",7,"殫",7,"殶殸",6,"毀毃毄毆",4,"毌毎毐毑毘毚毜",4,"毢",7,"毬毭毮毰毱毲毴毶毷毸毺毻毼毾",6,"氈",4,"氎氒気氜氝氞氠氣氥氫氬氭氱氳氶氷氹氺氻氼氾氿汃汄汅汈汋",4,"汑汒汓汖汘"],
["9b40","汙汚汢汣汥汦汧汫",4,"汱汳汵汷汸決汻汼汿沀沄沇沊沋沍沎沑沒沕沖沗沘沚沜沝沞沠沢沨沬沯沰沴沵沶沷沺泀況泂泃泆泇泈泋泍泎泏泑泒泘"],
["9b80","泙泚泜泝泟泤泦泧泩泬泭泲泴泹泿洀洂洃洅洆洈洉洊洍洏洐洑洓洔洕洖洘洜洝洟",5,"洦洨洩洬洭洯洰洴洶洷洸洺洿浀浂浄浉浌浐浕浖浗浘浛浝浟浡浢浤浥浧浨浫浬浭浰浱浲浳浵浶浹浺浻浽",4,"涃涄涆涇涊涋涍涏涐涒涖",4,"涜涢涥涬涭涰涱涳涴涶涷涹",5,"淁淂淃淈淉淊"],
["9c40","淍淎淏淐淒淓淔淕淗淚淛淜淟淢淣淥淧淨淩淪淭淯淰淲淴淵淶淸淺淽",7,"渆渇済渉渋渏渒渓渕渘渙減渜渞渟渢渦渧渨渪測渮渰渱渳渵"],
["9c80","渶渷渹渻",7,"湅",7,"湏湐湑湒湕湗湙湚湜湝湞湠",10,"湬湭湯",14,"満溁溂溄溇溈溊",4,"溑",6,"溙溚溛溝溞溠溡溣溤溦溨溩溫溬溭溮溰溳溵溸溹溼溾溿滀滃滄滅滆滈滉滊滌滍滎滐滒滖滘滙滛滜滝滣滧滪",5],
["9d40","滰滱滲滳滵滶滷滸滺",7,"漃漄漅漇漈漊",4,"漐漑漒漖",9,"漡漢漣漥漦漧漨漬漮漰漲漴漵漷",6,"漿潀潁潂"],
["9d80","潃潄潅潈潉潊潌潎",9,"潙潚潛潝潟潠潡潣潤潥潧",5,"潯潰潱潳潵潶潷潹潻潽",6,"澅澆澇澊澋澏",12,"澝澞澟澠澢",4,"澨",10,"澴澵澷澸澺",5,"濁濃",5,"濊",6,"濓",10,"濟濢濣濤濥"],
["9e40","濦",7,"濰",32,"瀒",7,"瀜",6,"瀤",6],
["9e80","瀫",9,"瀶瀷瀸瀺",17,"灍灎灐",13,"灟",11,"灮灱灲灳灴灷灹灺灻災炁炂炃炄炆炇炈炋炌炍炏炐炑炓炗炘炚炛炞",12,"炰炲炴炵炶為炾炿烄烅烆烇烉烋",12,"烚"],
["9f40","烜烝烞烠烡烢烣烥烪烮烰",6,"烸烺烻烼烾",10,"焋",4,"焑焒焔焗焛",10,"焧",7,"焲焳焴"],
["9f80","焵焷",13,"煆煇煈煉煋煍煏",12,"煝煟",4,"煥煩",4,"煯煰煱煴煵煶煷煹煻煼煾",5,"熅",4,"熋熌熍熎熐熑熒熓熕熖熗熚",4,"熡",6,"熩熪熫熭",5,"熴熶熷熸熺",8,"燄",9,"燏",4],
["a040","燖",9,"燡燢燣燤燦燨",5,"燯",9,"燺",11,"爇",19],
["a080","爛爜爞",9,"爩爫爭爮爯爲爳爴爺爼爾牀",6,"牉牊牋牎牏牐牑牓牔牕牗牘牚牜牞牠牣牤牥牨牪牫牬牭牰牱牳牴牶牷牸牻牼牽犂犃犅",4,"犌犎犐犑犓",11,"犠",11,"犮犱犲犳犵犺",6,"狅狆狇狉狊狋狌狏狑狓狔狕狖狘狚狛"],
["a1a1","　、。·ˉˇ¨〃々—～‖…‘’“”〔〕〈",7,"〖〗【】±×÷∶∧∨∑∏∪∩∈∷√⊥∥∠⌒⊙∫∮≡≌≈∽∝≠≮≯≤≥∞∵∴♂♀°′″℃＄¤￠￡‰§№☆★○●◎◇◆□■△▲※→←↑↓〓"],
["a2a1","ⅰ",9],
["a2b1","⒈",19,"⑴",19,"①",9],
["a2e5","㈠",9],
["a2f1","Ⅰ",11],
["a3a1","！＂＃￥％",88,"￣"],
["a4a1","ぁ",82],
["a5a1","ァ",85],
["a6a1","Α",16,"Σ",6],
["a6c1","α",16,"σ",6],
["a6e0","︵︶︹︺︿﹀︽︾﹁﹂﹃﹄"],
["a6ee","︻︼︷︸︱"],
["a6f4","︳︴"],
["a7a1","А",5,"ЁЖ",25],
["a7d1","а",5,"ёж",25],
["a840","ˊˋ˙–―‥‵℅℉↖↗↘↙∕∟∣≒≦≧⊿═",35,"▁",6],
["a880","█",7,"▓▔▕▼▽◢◣◤◥☉⊕〒〝〞"],
["a8a1","āáǎàēéěèīíǐìōóǒòūúǔùǖǘǚǜüêɑ"],
["a8bd","ńň"],
["a8c0","ɡ"],
["a8c5","ㄅ",36],
["a940","〡",8,"㊣㎎㎏㎜㎝㎞㎡㏄㏎㏑㏒㏕︰￢￤"],
["a959","℡㈱"],
["a95c","‐"],
["a960","ー゛゜ヽヾ〆ゝゞ﹉",9,"﹔﹕﹖﹗﹙",8],
["a980","﹢",4,"﹨﹩﹪﹫"],
["a996","〇"],
["a9a4","─",75],
["aa40","狜狝狟狢",5,"狪狫狵狶狹狽狾狿猀猂猄",5,"猋猌猍猏猐猑猒猔猘猙猚猟猠猣猤猦猧猨猭猯猰猲猳猵猶猺猻猼猽獀",8],
["aa80","獉獊獋獌獎獏獑獓獔獕獖獘",7,"獡",10,"獮獰獱"],
["ab40","獲",11,"獿",4,"玅玆玈玊玌玍玏玐玒玓玔玕玗玘玙玚玜玝玞玠玡玣",5,"玪玬玭玱玴玵玶玸玹玼玽玾玿珁珃",4],
["ab80","珋珌珎珒",6,"珚珛珜珝珟珡珢珣珤珦珨珪珫珬珮珯珰珱珳",4],
["ac40","珸",10,"琄琇琈琋琌琍琎琑",8,"琜",5,"琣琤琧琩琫琭琯琱琲琷",4,"琽琾琿瑀瑂",11],
["ac80","瑎",6,"瑖瑘瑝瑠",12,"瑮瑯瑱",4,"瑸瑹瑺"],
["ad40","瑻瑼瑽瑿璂璄璅璆璈璉璊璌璍璏璑",10,"璝璟",7,"璪",15,"璻",12],
["ad80","瓈",9,"瓓",8,"瓝瓟瓡瓥瓧",6,"瓰瓱瓲"],
["ae40","瓳瓵瓸",6,"甀甁甂甃甅",7,"甎甐甒甔甕甖甗甛甝甞甠",4,"甦甧甪甮甴甶甹甼甽甿畁畂畃畄畆畇畉畊畍畐畑畒畓畕畖畗畘"],
["ae80","畝",7,"畧畨畩畫",6,"畳畵當畷畺",4,"疀疁疂疄疅疇"],
["af40","疈疉疊疌疍疎疐疓疕疘疛疜疞疢疦",4,"疭疶疷疺疻疿痀痁痆痋痌痎痏痐痑痓痗痙痚痜痝痟痠痡痥痩痬痭痮痯痲痳痵痶痷痸痺痻痽痾瘂瘄瘆瘇"],
["af80","瘈瘉瘋瘍瘎瘏瘑瘒瘓瘔瘖瘚瘜瘝瘞瘡瘣瘧瘨瘬瘮瘯瘱瘲瘶瘷瘹瘺瘻瘽癁療癄"],
["b040","癅",6,"癎",5,"癕癗",4,"癝癟癠癡癢癤",6,"癬癭癮癰",7,"癹発發癿皀皁皃皅皉皊皌皍皏皐皒皔皕皗皘皚皛"],
["b080","皜",7,"皥",8,"皯皰皳皵",9,"盀盁盃啊阿埃挨哎唉哀皑癌蔼矮艾碍爱隘鞍氨安俺按暗岸胺案肮昂盎凹敖熬翱袄傲奥懊澳芭捌扒叭吧笆八疤巴拔跋靶把耙坝霸罢爸白柏百摆佰败拜稗斑班搬扳般颁板版扮拌伴瓣半办绊邦帮梆榜膀绑棒磅蚌镑傍谤苞胞包褒剥"],
["b140","盄盇盉盋盌盓盕盙盚盜盝盞盠",4,"盦",7,"盰盳盵盶盷盺盻盽盿眀眂眃眅眆眊県眎",10,"眛眜眝眞眡眣眤眥眧眪眫"],
["b180","眬眮眰",4,"眹眻眽眾眿睂睄睅睆睈",7,"睒",7,"睜薄雹保堡饱宝抱报暴豹鲍爆杯碑悲卑北辈背贝钡倍狈备惫焙被奔苯本笨崩绷甭泵蹦迸逼鼻比鄙笔彼碧蓖蔽毕毙毖币庇痹闭敝弊必辟壁臂避陛鞭边编贬扁便变卞辨辩辫遍标彪膘表鳖憋别瘪彬斌濒滨宾摈兵冰柄丙秉饼炳"],
["b240","睝睞睟睠睤睧睩睪睭",11,"睺睻睼瞁瞂瞃瞆",5,"瞏瞐瞓",11,"瞡瞣瞤瞦瞨瞫瞭瞮瞯瞱瞲瞴瞶",4],
["b280","瞼瞾矀",12,"矎",8,"矘矙矚矝",4,"矤病并玻菠播拨钵波博勃搏铂箔伯帛舶脖膊渤泊驳捕卜哺补埠不布步簿部怖擦猜裁材才财睬踩采彩菜蔡餐参蚕残惭惨灿苍舱仓沧藏操糙槽曹草厕策侧册测层蹭插叉茬茶查碴搽察岔差诧拆柴豺搀掺蝉馋谗缠铲产阐颤昌猖"],
["b340","矦矨矪矯矰矱矲矴矵矷矹矺矻矼砃",5,"砊砋砎砏砐砓砕砙砛砞砠砡砢砤砨砪砫砮砯砱砲砳砵砶砽砿硁硂硃硄硆硈硉硊硋硍硏硑硓硔硘硙硚"],
["b380","硛硜硞",11,"硯",7,"硸硹硺硻硽",6,"场尝常长偿肠厂敞畅唱倡超抄钞朝嘲潮巢吵炒车扯撤掣彻澈郴臣辰尘晨忱沉陈趁衬撑称城橙成呈乘程惩澄诚承逞骋秤吃痴持匙池迟弛驰耻齿侈尺赤翅斥炽充冲虫崇宠抽酬畴踌稠愁筹仇绸瞅丑臭初出橱厨躇锄雏滁除楚"],
["b440","碄碅碆碈碊碋碏碐碒碔碕碖碙碝碞碠碢碤碦碨",7,"碵碶碷碸確碻碼碽碿磀磂磃磄磆磇磈磌磍磎磏磑磒磓磖磗磘磚",9],
["b480","磤磥磦磧磩磪磫磭",4,"磳磵磶磸磹磻",5,"礂礃礄礆",6,"础储矗搐触处揣川穿椽传船喘串疮窗幢床闯创吹炊捶锤垂春椿醇唇淳纯蠢戳绰疵茨磁雌辞慈瓷词此刺赐次聪葱囱匆从丛凑粗醋簇促蹿篡窜摧崔催脆瘁粹淬翠村存寸磋撮搓措挫错搭达答瘩打大呆歹傣戴带殆代贷袋待逮"],
["b540","礍",5,"礔",9,"礟",4,"礥",14,"礵",4,"礽礿祂祃祄祅祇祊",8,"祔祕祘祙祡祣"],
["b580","祤祦祩祪祫祬祮祰",6,"祹祻",4,"禂禃禆禇禈禉禋禌禍禎禐禑禒怠耽担丹单郸掸胆旦氮但惮淡诞弹蛋当挡党荡档刀捣蹈倒岛祷导到稻悼道盗德得的蹬灯登等瞪凳邓堤低滴迪敌笛狄涤翟嫡抵底地蒂第帝弟递缔颠掂滇碘点典靛垫电佃甸店惦奠淀殿碉叼雕凋刁掉吊钓调跌爹碟蝶迭谍叠"],
["b640","禓",6,"禛",11,"禨",10,"禴",4,"禼禿秂秄秅秇秈秊秌秎秏秐秓秔秖秗秙",5,"秠秡秢秥秨秪"],
["b680","秬秮秱",6,"秹秺秼秾秿稁稄稅稇稈稉稊稌稏",4,"稕稖稘稙稛稜丁盯叮钉顶鼎锭定订丢东冬董懂动栋侗恫冻洞兜抖斗陡豆逗痘都督毒犊独读堵睹赌杜镀肚度渡妒端短锻段断缎堆兑队对墩吨蹲敦顿囤钝盾遁掇哆多夺垛躲朵跺舵剁惰堕蛾峨鹅俄额讹娥恶厄扼遏鄂饿恩而儿耳尔饵洱二"],
["b740","稝稟稡稢稤",14,"稴稵稶稸稺稾穀",5,"穇",9,"穒",4,"穘",16],
["b780","穩",6,"穱穲穳穵穻穼穽穾窂窅窇窉窊窋窌窎窏窐窓窔窙窚窛窞窡窢贰发罚筏伐乏阀法珐藩帆番翻樊矾钒繁凡烦反返范贩犯饭泛坊芳方肪房防妨仿访纺放菲非啡飞肥匪诽吠肺废沸费芬酚吩氛分纷坟焚汾粉奋份忿愤粪丰封枫蜂峰锋风疯烽逢冯缝讽奉凤佛否夫敷肤孵扶拂辐幅氟符伏俘服"],
["b840","窣窤窧窩窪窫窮",4,"窴",10,"竀",10,"竌",9,"竗竘竚竛竜竝竡竢竤竧",5,"竮竰竱竲竳"],
["b880","竴",4,"竻竼竾笀笁笂笅笇笉笌笍笎笐笒笓笖笗笘笚笜笝笟笡笢笣笧笩笭浮涪福袱弗甫抚辅俯釜斧脯腑府腐赴副覆赋复傅付阜父腹负富讣附妇缚咐噶嘎该改概钙盖溉干甘杆柑竿肝赶感秆敢赣冈刚钢缸肛纲岗港杠篙皋高膏羔糕搞镐稿告哥歌搁戈鸽胳疙割革葛格蛤阁隔铬个各给根跟耕更庚羹"],
["b940","笯笰笲笴笵笶笷笹笻笽笿",5,"筆筈筊筍筎筓筕筗筙筜筞筟筡筣",10,"筯筰筳筴筶筸筺筼筽筿箁箂箃箄箆",6,"箎箏"],
["b980","箑箒箓箖箘箙箚箛箞箟箠箣箤箥箮箯箰箲箳箵箶箷箹",7,"篂篃範埂耿梗工攻功恭龚供躬公宫弓巩汞拱贡共钩勾沟苟狗垢构购够辜菇咕箍估沽孤姑鼓古蛊骨谷股故顾固雇刮瓜剐寡挂褂乖拐怪棺关官冠观管馆罐惯灌贯光广逛瑰规圭硅归龟闺轨鬼诡癸桂柜跪贵刽辊滚棍锅郭国果裹过哈"],
["ba40","篅篈築篊篋篍篎篏篐篒篔",4,"篛篜篞篟篠篢篣篤篧篨篩篫篬篭篯篰篲",4,"篸篹篺篻篽篿",7,"簈簉簊簍簎簐",5,"簗簘簙"],
["ba80","簚",4,"簠",5,"簨簩簫",12,"簹",5,"籂骸孩海氦亥害骇酣憨邯韩含涵寒函喊罕翰撼捍旱憾悍焊汗汉夯杭航壕嚎豪毫郝好耗号浩呵喝荷菏核禾和何合盒貉阂河涸赫褐鹤贺嘿黑痕很狠恨哼亨横衡恒轰哄烘虹鸿洪宏弘红喉侯猴吼厚候后呼乎忽瑚壶葫胡蝴狐糊湖"],
["bb40","籃",9,"籎",36,"籵",5,"籾",9],
["bb80","粈粊",6,"粓粔粖粙粚粛粠粡粣粦粧粨粩粫粬粭粯粰粴",4,"粺粻弧虎唬护互沪户花哗华猾滑画划化话槐徊怀淮坏欢环桓还缓换患唤痪豢焕涣宦幻荒慌黄磺蝗簧皇凰惶煌晃幌恍谎灰挥辉徽恢蛔回毁悔慧卉惠晦贿秽会烩汇讳诲绘荤昏婚魂浑混豁活伙火获或惑霍货祸击圾基机畸稽积箕"],
["bc40","粿糀糂糃糄糆糉糋糎",6,"糘糚糛糝糞糡",6,"糩",5,"糰",7,"糹糺糼",13,"紋",5],
["bc80","紑",14,"紡紣紤紥紦紨紩紪紬紭紮細",6,"肌饥迹激讥鸡姬绩缉吉极棘辑籍集及急疾汲即嫉级挤几脊己蓟技冀季伎祭剂悸济寄寂计记既忌际妓继纪嘉枷夹佳家加荚颊贾甲钾假稼价架驾嫁歼监坚尖笺间煎兼肩艰奸缄茧检柬碱硷拣捡简俭剪减荐槛鉴践贱见键箭件"],
["bd40","紷",54,"絯",7],
["bd80","絸",32,"健舰剑饯渐溅涧建僵姜将浆江疆蒋桨奖讲匠酱降蕉椒礁焦胶交郊浇骄娇嚼搅铰矫侥脚狡角饺缴绞剿教酵轿较叫窖揭接皆秸街阶截劫节桔杰捷睫竭洁结解姐戒藉芥界借介疥诫届巾筋斤金今津襟紧锦仅谨进靳晋禁近烬浸"],
["be40","継",12,"綧",6,"綯",42],
["be80","線",32,"尽劲荆兢茎睛晶鲸京惊精粳经井警景颈静境敬镜径痉靖竟竞净炯窘揪究纠玖韭久灸九酒厩救旧臼舅咎就疚鞠拘狙疽居驹菊局咀矩举沮聚拒据巨具距踞锯俱句惧炬剧捐鹃娟倦眷卷绢撅攫抉掘倔爵觉决诀绝均菌钧军君峻"],
["bf40","緻",62],
["bf80","縺縼",4,"繂",4,"繈",21,"俊竣浚郡骏喀咖卡咯开揩楷凯慨刊堪勘坎砍看康慷糠扛抗亢炕考拷烤靠坷苛柯棵磕颗科壳咳可渴克刻客课肯啃垦恳坑吭空恐孔控抠口扣寇枯哭窟苦酷库裤夸垮挎跨胯块筷侩快宽款匡筐狂框矿眶旷况亏盔岿窥葵奎魁傀"],
["c040","繞",35,"纃",23,"纜纝纞"],
["c080","纮纴纻纼绖绤绬绹缊缐缞缷缹缻",6,"罃罆",9,"罒罓馈愧溃坤昆捆困括扩廓阔垃拉喇蜡腊辣啦莱来赖蓝婪栏拦篮阑兰澜谰揽览懒缆烂滥琅榔狼廊郎朗浪捞劳牢老佬姥酪烙涝勒乐雷镭蕾磊累儡垒擂肋类泪棱楞冷厘梨犁黎篱狸离漓理李里鲤礼莉荔吏栗丽厉励砾历利傈例俐"],
["c140","罖罙罛罜罝罞罠罣",4,"罫罬罭罯罰罳罵罶罷罸罺罻罼罽罿羀羂",7,"羋羍羏",4,"羕",4,"羛羜羠羢羣羥羦羨",6,"羱"],
["c180","羳",4,"羺羻羾翀翂翃翄翆翇翈翉翋翍翏",4,"翖翗翙",5,"翢翣痢立粒沥隶力璃哩俩联莲连镰廉怜涟帘敛脸链恋炼练粮凉梁粱良两辆量晾亮谅撩聊僚疗燎寥辽潦了撂镣廖料列裂烈劣猎琳林磷霖临邻鳞淋凛赁吝拎玲菱零龄铃伶羚凌灵陵岭领另令溜琉榴硫馏留刘瘤流柳六龙聋咙笼窿"],
["c240","翤翧翨翪翫翬翭翯翲翴",6,"翽翾翿耂耇耈耉耊耎耏耑耓耚耛耝耞耟耡耣耤耫",5,"耲耴耹耺耼耾聀聁聄聅聇聈聉聎聏聐聑聓聕聖聗"],
["c280","聙聛",13,"聫",5,"聲",11,"隆垄拢陇楼娄搂篓漏陋芦卢颅庐炉掳卤虏鲁麓碌露路赂鹿潞禄录陆戮驴吕铝侣旅履屡缕虑氯律率滤绿峦挛孪滦卵乱掠略抡轮伦仑沦纶论萝螺罗逻锣箩骡裸落洛骆络妈麻玛码蚂马骂嘛吗埋买麦卖迈脉瞒馒蛮满蔓曼慢漫"],
["c340","聾肁肂肅肈肊肍",5,"肔肕肗肙肞肣肦肧肨肬肰肳肵肶肸肹肻胅胇",4,"胏",6,"胘胟胠胢胣胦胮胵胷胹胻胾胿脀脁脃脄脅脇脈脋"],
["c380","脌脕脗脙脛脜脝脟",12,"脭脮脰脳脴脵脷脹",4,"脿谩芒茫盲氓忙莽猫茅锚毛矛铆卯茂冒帽貌贸么玫枚梅酶霉煤没眉媒镁每美昧寐妹媚门闷们萌蒙檬盟锰猛梦孟眯醚靡糜迷谜弥米秘觅泌蜜密幂棉眠绵冕免勉娩缅面苗描瞄藐秒渺庙妙蔑灭民抿皿敏悯闽明螟鸣铭名命谬摸"],
["c440","腀",5,"腇腉腍腎腏腒腖腗腘腛",4,"腡腢腣腤腦腨腪腫腬腯腲腳腵腶腷腸膁膃",4,"膉膋膌膍膎膐膒",5,"膙膚膞",4,"膤膥"],
["c480","膧膩膫",7,"膴",5,"膼膽膾膿臄臅臇臈臉臋臍",6,"摹蘑模膜磨摩魔抹末莫墨默沫漠寞陌谋牟某拇牡亩姆母墓暮幕募慕木目睦牧穆拿哪呐钠那娜纳氖乃奶耐奈南男难囊挠脑恼闹淖呢馁内嫩能妮霓倪泥尼拟你匿腻逆溺蔫拈年碾撵捻念娘酿鸟尿捏聂孽啮镊镍涅您柠狞凝宁"],
["c540","臔",14,"臤臥臦臨臩臫臮",4,"臵",5,"臽臿舃與",4,"舎舏舑舓舕",5,"舝舠舤舥舦舧舩舮舲舺舼舽舿"],
["c580","艀艁艂艃艅艆艈艊艌艍艎艐",7,"艙艛艜艝艞艠",7,"艩拧泞牛扭钮纽脓浓农弄奴努怒女暖虐疟挪懦糯诺哦欧鸥殴藕呕偶沤啪趴爬帕怕琶拍排牌徘湃派攀潘盘磐盼畔判叛乓庞旁耪胖抛咆刨炮袍跑泡呸胚培裴赔陪配佩沛喷盆砰抨烹澎彭蓬棚硼篷膨朋鹏捧碰坯砒霹批披劈琵毗"],
["c640","艪艫艬艭艱艵艶艷艸艻艼芀芁芃芅芆芇芉芌芐芓芔芕芖芚芛芞芠芢芣芧芲芵芶芺芻芼芿苀苂苃苅苆苉苐苖苙苚苝苢苧苨苩苪苬苭苮苰苲苳苵苶苸"],
["c680","苺苼",4,"茊茋茍茐茒茓茖茘茙茝",9,"茩茪茮茰茲茷茻茽啤脾疲皮匹痞僻屁譬篇偏片骗飘漂瓢票撇瞥拼频贫品聘乒坪苹萍平凭瓶评屏坡泼颇婆破魄迫粕剖扑铺仆莆葡菩蒲埔朴圃普浦谱曝瀑期欺栖戚妻七凄漆柒沏其棋奇歧畦崎脐齐旗祈祁骑起岂乞企启契砌器气迄弃汽泣讫掐"],
["c740","茾茿荁荂荄荅荈荊",4,"荓荕",4,"荝荢荰",6,"荹荺荾",6,"莇莈莊莋莌莍莏莐莑莔莕莖莗莙莚莝莟莡",6,"莬莭莮"],
["c780","莯莵莻莾莿菂菃菄菆菈菉菋菍菎菐菑菒菓菕菗菙菚菛菞菢菣菤菦菧菨菫菬菭恰洽牵扦钎铅千迁签仟谦乾黔钱钳前潜遣浅谴堑嵌欠歉枪呛腔羌墙蔷强抢橇锹敲悄桥瞧乔侨巧鞘撬翘峭俏窍切茄且怯窃钦侵亲秦琴勤芹擒禽寝沁青轻氢倾卿清擎晴氰情顷请庆琼穷秋丘邱球求囚酋泅趋区蛆曲躯屈驱渠"],
["c840","菮華菳",4,"菺菻菼菾菿萀萂萅萇萈萉萊萐萒",5,"萙萚萛萞",5,"萩",7,"萲",5,"萹萺萻萾",7,"葇葈葉"],
["c880","葊",6,"葒",4,"葘葝葞葟葠葢葤",4,"葪葮葯葰葲葴葷葹葻葼取娶龋趣去圈颧权醛泉全痊拳犬券劝缺炔瘸却鹊榷确雀裙群然燃冉染瓤壤攘嚷让饶扰绕惹热壬仁人忍韧任认刃妊纫扔仍日戎茸蓉荣融熔溶容绒冗揉柔肉茹蠕儒孺如辱乳汝入褥软阮蕊瑞锐闰润若弱撒洒萨腮鳃塞赛三叁"],
["c940","葽",4,"蒃蒄蒅蒆蒊蒍蒏",7,"蒘蒚蒛蒝蒞蒟蒠蒢",12,"蒰蒱蒳蒵蒶蒷蒻蒼蒾蓀蓂蓃蓅蓆蓇蓈蓋蓌蓎蓏蓒蓔蓕蓗"],
["c980","蓘",4,"蓞蓡蓢蓤蓧",4,"蓭蓮蓯蓱",10,"蓽蓾蔀蔁蔂伞散桑嗓丧搔骚扫嫂瑟色涩森僧莎砂杀刹沙纱傻啥煞筛晒珊苫杉山删煽衫闪陕擅赡膳善汕扇缮墒伤商赏晌上尚裳梢捎稍烧芍勺韶少哨邵绍奢赊蛇舌舍赦摄射慑涉社设砷申呻伸身深娠绅神沈审婶甚肾慎渗声生甥牲升绳"],
["ca40","蔃",8,"蔍蔎蔏蔐蔒蔔蔕蔖蔘蔙蔛蔜蔝蔞蔠蔢",8,"蔭",9,"蔾",4,"蕄蕅蕆蕇蕋",10],
["ca80","蕗蕘蕚蕛蕜蕝蕟",4,"蕥蕦蕧蕩",8,"蕳蕵蕶蕷蕸蕼蕽蕿薀薁省盛剩胜圣师失狮施湿诗尸虱十石拾时什食蚀实识史矢使屎驶始式示士世柿事拭誓逝势是嗜噬适仕侍释饰氏市恃室视试收手首守寿授售受瘦兽蔬枢梳殊抒输叔舒淑疏书赎孰熟薯暑曙署蜀黍鼠属术述树束戍竖墅庶数漱"],
["cb40","薂薃薆薈",6,"薐",10,"薝",6,"薥薦薧薩薫薬薭薱",5,"薸薺",6,"藂",6,"藊",4,"藑藒"],
["cb80","藔藖",5,"藝",6,"藥藦藧藨藪",14,"恕刷耍摔衰甩帅栓拴霜双爽谁水睡税吮瞬顺舜说硕朔烁斯撕嘶思私司丝死肆寺嗣四伺似饲巳松耸怂颂送宋讼诵搜艘擞嗽苏酥俗素速粟僳塑溯宿诉肃酸蒜算虽隋随绥髓碎岁穗遂隧祟孙损笋蓑梭唆缩琐索锁所塌他它她塔"],
["cc40","藹藺藼藽藾蘀",4,"蘆",10,"蘒蘓蘔蘕蘗",15,"蘨蘪",13,"蘹蘺蘻蘽蘾蘿虀"],
["cc80","虁",11,"虒虓處",4,"虛虜虝號虠虡虣",7,"獭挞蹋踏胎苔抬台泰酞太态汰坍摊贪瘫滩坛檀痰潭谭谈坦毯袒碳探叹炭汤塘搪堂棠膛唐糖倘躺淌趟烫掏涛滔绦萄桃逃淘陶讨套特藤腾疼誊梯剔踢锑提题蹄啼体替嚏惕涕剃屉天添填田甜恬舔腆挑条迢眺跳贴铁帖厅听烃"],
["cd40","虭虯虰虲",6,"蚃",6,"蚎",4,"蚔蚖",5,"蚞",4,"蚥蚦蚫蚭蚮蚲蚳蚷蚸蚹蚻",4,"蛁蛂蛃蛅蛈蛌蛍蛒蛓蛕蛖蛗蛚蛜"],
["cd80","蛝蛠蛡蛢蛣蛥蛦蛧蛨蛪蛫蛬蛯蛵蛶蛷蛺蛻蛼蛽蛿蜁蜄蜅蜆蜋蜌蜎蜏蜐蜑蜔蜖汀廷停亭庭挺艇通桐酮瞳同铜彤童桶捅筒统痛偷投头透凸秃突图徒途涂屠土吐兔湍团推颓腿蜕褪退吞屯臀拖托脱鸵陀驮驼椭妥拓唾挖哇蛙洼娃瓦袜歪外豌弯湾玩顽丸烷完碗挽晚皖惋宛婉万腕汪王亡枉网往旺望忘妄威"],
["ce40","蜙蜛蜝蜟蜠蜤蜦蜧蜨蜪蜫蜬蜭蜯蜰蜲蜳蜵蜶蜸蜹蜺蜼蜽蝀",6,"蝊蝋蝍蝏蝐蝑蝒蝔蝕蝖蝘蝚",5,"蝡蝢蝦",7,"蝯蝱蝲蝳蝵"],
["ce80","蝷蝸蝹蝺蝿螀螁螄螆螇螉螊螌螎",4,"螔螕螖螘",6,"螠",4,"巍微危韦违桅围唯惟为潍维苇萎委伟伪尾纬未蔚味畏胃喂魏位渭谓尉慰卫瘟温蚊文闻纹吻稳紊问嗡翁瓮挝蜗涡窝我斡卧握沃巫呜钨乌污诬屋无芜梧吾吴毋武五捂午舞伍侮坞戊雾晤物勿务悟误昔熙析西硒矽晰嘻吸锡牺"],
["cf40","螥螦螧螩螪螮螰螱螲螴螶螷螸螹螻螼螾螿蟁",4,"蟇蟈蟉蟌",4,"蟔",6,"蟜蟝蟞蟟蟡蟢蟣蟤蟦蟧蟨蟩蟫蟬蟭蟯",9],
["cf80","蟺蟻蟼蟽蟿蠀蠁蠂蠄",5,"蠋",7,"蠔蠗蠘蠙蠚蠜",4,"蠣稀息希悉膝夕惜熄烯溪汐犀檄袭席习媳喜铣洗系隙戏细瞎虾匣霞辖暇峡侠狭下厦夏吓掀锨先仙鲜纤咸贤衔舷闲涎弦嫌显险现献县腺馅羡宪陷限线相厢镶香箱襄湘乡翔祥详想响享项巷橡像向象萧硝霄削哮嚣销消宵淆晓"],
["d040","蠤",13,"蠳",5,"蠺蠻蠽蠾蠿衁衂衃衆",5,"衎",5,"衕衖衘衚",6,"衦衧衪衭衯衱衳衴衵衶衸衹衺"],
["d080","衻衼袀袃袆袇袉袊袌袎袏袐袑袓袔袕袗",4,"袝",4,"袣袥",5,"小孝校肖啸笑效楔些歇蝎鞋协挟携邪斜胁谐写械卸蟹懈泄泻谢屑薪芯锌欣辛新忻心信衅星腥猩惺兴刑型形邢行醒幸杏性姓兄凶胸匈汹雄熊休修羞朽嗅锈秀袖绣墟戌需虚嘘须徐许蓄酗叙旭序畜恤絮婿绪续轩喧宣悬旋玄"],
["d140","袬袮袯袰袲",4,"袸袹袺袻袽袾袿裀裃裄裇裈裊裋裌裍裏裐裑裓裖裗裚",4,"裠裡裦裧裩",6,"裲裵裶裷裺裻製裿褀褁褃",5],
["d180","褉褋",4,"褑褔",4,"褜",4,"褢褣褤褦褧褨褩褬褭褮褯褱褲褳褵褷选癣眩绚靴薛学穴雪血勋熏循旬询寻驯巡殉汛训讯逊迅压押鸦鸭呀丫芽牙蚜崖衙涯雅哑亚讶焉咽阉烟淹盐严研蜒岩延言颜阎炎沿奄掩眼衍演艳堰燕厌砚雁唁彦焰宴谚验殃央鸯秧杨扬佯疡羊洋阳氧仰痒养样漾邀腰妖瑶"],
["d240","褸",8,"襂襃襅",24,"襠",5,"襧",19,"襼"],
["d280","襽襾覀覂覄覅覇",26,"摇尧遥窑谣姚咬舀药要耀椰噎耶爷野冶也页掖业叶曳腋夜液一壹医揖铱依伊衣颐夷遗移仪胰疑沂宜姨彝椅蚁倚已乙矣以艺抑易邑屹亿役臆逸肄疫亦裔意毅忆义益溢诣议谊译异翼翌绎茵荫因殷音阴姻吟银淫寅饮尹引隐"],
["d340","覢",30,"觃觍觓觔觕觗觘觙觛觝觟觠觡觢觤觧觨觩觪觬觭觮觰觱觲觴",6],
["d380","觻",4,"訁",5,"計",21,"印英樱婴鹰应缨莹萤营荧蝇迎赢盈影颖硬映哟拥佣臃痈庸雍踊蛹咏泳涌永恿勇用幽优悠忧尤由邮铀犹油游酉有友右佑釉诱又幼迂淤于盂榆虞愚舆余俞逾鱼愉渝渔隅予娱雨与屿禹宇语羽玉域芋郁吁遇喻峪御愈欲狱育誉"],
["d440","訞",31,"訿",8,"詉",21],
["d480","詟",25,"詺",6,"浴寓裕预豫驭鸳渊冤元垣袁原援辕园员圆猿源缘远苑愿怨院曰约越跃钥岳粤月悦阅耘云郧匀陨允运蕴酝晕韵孕匝砸杂栽哉灾宰载再在咱攒暂赞赃脏葬遭糟凿藻枣早澡蚤躁噪造皂灶燥责择则泽贼怎增憎曾赠扎喳渣札轧"],
["d540","誁",7,"誋",7,"誔",46],
["d580","諃",32,"铡闸眨栅榨咋乍炸诈摘斋宅窄债寨瞻毡詹粘沾盏斩辗崭展蘸栈占战站湛绽樟章彰漳张掌涨杖丈帐账仗胀瘴障招昭找沼赵照罩兆肇召遮折哲蛰辙者锗蔗这浙珍斟真甄砧臻贞针侦枕疹诊震振镇阵蒸挣睁征狰争怔整拯正政"],
["d640","諤",34,"謈",27],
["d680","謤謥謧",30,"帧症郑证芝枝支吱蜘知肢脂汁之织职直植殖执值侄址指止趾只旨纸志挚掷至致置帜峙制智秩稚质炙痔滞治窒中盅忠钟衷终种肿重仲众舟周州洲诌粥轴肘帚咒皱宙昼骤珠株蛛朱猪诸诛逐竹烛煮拄瞩嘱主著柱助蛀贮铸筑"],
["d740","譆",31,"譧",4,"譭",25],
["d780","讇",24,"讬讱讻诇诐诪谉谞住注祝驻抓爪拽专砖转撰赚篆桩庄装妆撞壮状椎锥追赘坠缀谆准捉拙卓桌琢茁酌啄着灼浊兹咨资姿滋淄孜紫仔籽滓子自渍字鬃棕踪宗综总纵邹走奏揍租足卒族祖诅阻组钻纂嘴醉最罪尊遵昨左佐柞做作坐座"],
["d840","谸",8,"豂豃豄豅豈豊豋豍",7,"豖豗豘豙豛",5,"豣",6,"豬",6,"豴豵豶豷豻",6,"貃貄貆貇"],
["d880","貈貋貍",6,"貕貖貗貙",20,"亍丌兀丐廿卅丕亘丞鬲孬噩丨禺丿匕乇夭爻卮氐囟胤馗毓睾鼗丶亟鼐乜乩亓芈孛啬嘏仄厍厝厣厥厮靥赝匚叵匦匮匾赜卦卣刂刈刎刭刳刿剀剌剞剡剜蒯剽劂劁劐劓冂罔亻仃仉仂仨仡仫仞伛仳伢佤仵伥伧伉伫佞佧攸佚佝"],
["d940","貮",62],
["d980","賭",32,"佟佗伲伽佶佴侑侉侃侏佾佻侪佼侬侔俦俨俪俅俚俣俜俑俟俸倩偌俳倬倏倮倭俾倜倌倥倨偾偃偕偈偎偬偻傥傧傩傺僖儆僭僬僦僮儇儋仝氽佘佥俎龠汆籴兮巽黉馘冁夔勹匍訇匐凫夙兕亠兖亳衮袤亵脔裒禀嬴蠃羸冫冱冽冼"],
["da40","贎",14,"贠赑赒赗赟赥赨赩赪赬赮赯赱赲赸",8,"趂趃趆趇趈趉趌",4,"趒趓趕",9,"趠趡"],
["da80","趢趤",12,"趲趶趷趹趻趽跀跁跂跅跇跈跉跊跍跐跒跓跔凇冖冢冥讠讦讧讪讴讵讷诂诃诋诏诎诒诓诔诖诘诙诜诟诠诤诨诩诮诰诳诶诹诼诿谀谂谄谇谌谏谑谒谔谕谖谙谛谘谝谟谠谡谥谧谪谫谮谯谲谳谵谶卩卺阝阢阡阱阪阽阼陂陉陔陟陧陬陲陴隈隍隗隰邗邛邝邙邬邡邴邳邶邺"],
["db40","跕跘跙跜跠跡跢跥跦跧跩跭跮跰跱跲跴跶跼跾",6,"踆踇踈踋踍踎踐踑踒踓踕",7,"踠踡踤",4,"踫踭踰踲踳踴踶踷踸踻踼踾"],
["db80","踿蹃蹅蹆蹌",4,"蹓",5,"蹚",11,"蹧蹨蹪蹫蹮蹱邸邰郏郅邾郐郄郇郓郦郢郜郗郛郫郯郾鄄鄢鄞鄣鄱鄯鄹酃酆刍奂劢劬劭劾哿勐勖勰叟燮矍廴凵凼鬯厶弁畚巯坌垩垡塾墼壅壑圩圬圪圳圹圮圯坜圻坂坩垅坫垆坼坻坨坭坶坳垭垤垌垲埏垧垴垓垠埕埘埚埙埒垸埴埯埸埤埝"],
["dc40","蹳蹵蹷",4,"蹽蹾躀躂躃躄躆躈",6,"躑躒躓躕",6,"躝躟",11,"躭躮躰躱躳",6,"躻",7],
["dc80","軃",10,"軏",21,"堋堍埽埭堀堞堙塄堠塥塬墁墉墚墀馨鼙懿艹艽艿芏芊芨芄芎芑芗芙芫芸芾芰苈苊苣芘芷芮苋苌苁芩芴芡芪芟苄苎芤苡茉苷苤茏茇苜苴苒苘茌苻苓茑茚茆茔茕苠苕茜荑荛荜茈莒茼茴茱莛荞茯荏荇荃荟荀茗荠茭茺茳荦荥"],
["dd40","軥",62],
["dd80","輤",32,"荨茛荩荬荪荭荮莰荸莳莴莠莪莓莜莅荼莶莩荽莸荻莘莞莨莺莼菁萁菥菘堇萘萋菝菽菖萜萸萑萆菔菟萏萃菸菹菪菅菀萦菰菡葜葑葚葙葳蒇蒈葺蒉葸萼葆葩葶蒌蒎萱葭蓁蓍蓐蓦蒽蓓蓊蒿蒺蓠蒡蒹蒴蒗蓥蓣蔌甍蔸蓰蔹蔟蔺"],
["de40","轅",32,"轪辀辌辒辝辠辡辢辤辥辦辧辪辬辭辮辯農辳辴辵辷辸辺辻込辿迀迃迆"],
["de80","迉",4,"迏迒迖迗迚迠迡迣迧迬迯迱迲迴迵迶迺迻迼迾迿逇逈逌逎逓逕逘蕖蔻蓿蓼蕙蕈蕨蕤蕞蕺瞢蕃蕲蕻薤薨薇薏蕹薮薜薅薹薷薰藓藁藜藿蘧蘅蘩蘖蘼廾弈夼奁耷奕奚奘匏尢尥尬尴扌扪抟抻拊拚拗拮挢拶挹捋捃掭揶捱捺掎掴捭掬掊捩掮掼揲揸揠揿揄揞揎摒揆掾摅摁搋搛搠搌搦搡摞撄摭撖"],
["df40","這逜連逤逥逧",5,"逰",4,"逷逹逺逽逿遀遃遅遆遈",4,"過達違遖遙遚遜",5,"遤遦遧適遪遫遬遯",4,"遶",6,"遾邁"],
["df80","還邅邆邇邉邊邌",4,"邒邔邖邘邚邜邞邟邠邤邥邧邨邩邫邭邲邷邼邽邿郀摺撷撸撙撺擀擐擗擤擢攉攥攮弋忒甙弑卟叱叽叩叨叻吒吖吆呋呒呓呔呖呃吡呗呙吣吲咂咔呷呱呤咚咛咄呶呦咝哐咭哂咴哒咧咦哓哔呲咣哕咻咿哌哙哚哜咩咪咤哝哏哞唛哧唠哽唔哳唢唣唏唑唧唪啧喏喵啉啭啁啕唿啐唼"],
["e040","郂郃郆郈郉郋郌郍郒郔郕郖郘郙郚郞郟郠郣郤郥郩郪郬郮郰郱郲郳郵郶郷郹郺郻郼郿鄀鄁鄃鄅",19,"鄚鄛鄜"],
["e080","鄝鄟鄠鄡鄤",10,"鄰鄲",6,"鄺",8,"酄唷啖啵啶啷唳唰啜喋嗒喃喱喹喈喁喟啾嗖喑啻嗟喽喾喔喙嗪嗷嗉嘟嗑嗫嗬嗔嗦嗝嗄嗯嗥嗲嗳嗌嗍嗨嗵嗤辔嘞嘈嘌嘁嘤嘣嗾嘀嘧嘭噘嘹噗嘬噍噢噙噜噌噔嚆噤噱噫噻噼嚅嚓嚯囔囗囝囡囵囫囹囿圄圊圉圜帏帙帔帑帱帻帼"],
["e140","酅酇酈酑酓酔酕酖酘酙酛酜酟酠酦酧酨酫酭酳酺酻酼醀",4,"醆醈醊醎醏醓",6,"醜",5,"醤",5,"醫醬醰醱醲醳醶醷醸醹醻"],
["e180","醼",10,"釈釋釐釒",9,"針",8,"帷幄幔幛幞幡岌屺岍岐岖岈岘岙岑岚岜岵岢岽岬岫岱岣峁岷峄峒峤峋峥崂崃崧崦崮崤崞崆崛嵘崾崴崽嵬嵛嵯嵝嵫嵋嵊嵩嵴嶂嶙嶝豳嶷巅彳彷徂徇徉後徕徙徜徨徭徵徼衢彡犭犰犴犷犸狃狁狎狍狒狨狯狩狲狴狷猁狳猃狺"],
["e240","釦",62],
["e280","鈥",32,"狻猗猓猡猊猞猝猕猢猹猥猬猸猱獐獍獗獠獬獯獾舛夥飧夤夂饣饧",5,"饴饷饽馀馄馇馊馍馐馑馓馔馕庀庑庋庖庥庠庹庵庾庳赓廒廑廛廨廪膺忄忉忖忏怃忮怄忡忤忾怅怆忪忭忸怙怵怦怛怏怍怩怫怊怿怡恸恹恻恺恂"],
["e340","鉆",45,"鉵",16],
["e380","銆",7,"銏",24,"恪恽悖悚悭悝悃悒悌悛惬悻悱惝惘惆惚悴愠愦愕愣惴愀愎愫慊慵憬憔憧憷懔懵忝隳闩闫闱闳闵闶闼闾阃阄阆阈阊阋阌阍阏阒阕阖阗阙阚丬爿戕氵汔汜汊沣沅沐沔沌汨汩汴汶沆沩泐泔沭泷泸泱泗沲泠泖泺泫泮沱泓泯泾"],
["e440","銨",5,"銯",24,"鋉",31],
["e480","鋩",32,"洹洧洌浃浈洇洄洙洎洫浍洮洵洚浏浒浔洳涑浯涞涠浞涓涔浜浠浼浣渚淇淅淞渎涿淠渑淦淝淙渖涫渌涮渫湮湎湫溲湟溆湓湔渲渥湄滟溱溘滠漭滢溥溧溽溻溷滗溴滏溏滂溟潢潆潇漤漕滹漯漶潋潴漪漉漩澉澍澌潸潲潼潺濑"],
["e540","錊",51,"錿",10],
["e580","鍊",31,"鍫濉澧澹澶濂濡濮濞濠濯瀚瀣瀛瀹瀵灏灞宀宄宕宓宥宸甯骞搴寤寮褰寰蹇謇辶迓迕迥迮迤迩迦迳迨逅逄逋逦逑逍逖逡逵逶逭逯遄遑遒遐遨遘遢遛暹遴遽邂邈邃邋彐彗彖彘尻咫屐屙孱屣屦羼弪弩弭艴弼鬻屮妁妃妍妩妪妣"],
["e640","鍬",34,"鎐",27],
["e680","鎬",29,"鏋鏌鏍妗姊妫妞妤姒妲妯姗妾娅娆姝娈姣姘姹娌娉娲娴娑娣娓婀婧婊婕娼婢婵胬媪媛婷婺媾嫫媲嫒嫔媸嫠嫣嫱嫖嫦嫘嫜嬉嬗嬖嬲嬷孀尕尜孚孥孳孑孓孢驵驷驸驺驿驽骀骁骅骈骊骐骒骓骖骘骛骜骝骟骠骢骣骥骧纟纡纣纥纨纩"],
["e740","鏎",7,"鏗",54],
["e780","鐎",32,"纭纰纾绀绁绂绉绋绌绐绔绗绛绠绡绨绫绮绯绱绲缍绶绺绻绾缁缂缃缇缈缋缌缏缑缒缗缙缜缛缟缡",6,"缪缫缬缭缯",4,"缵幺畿巛甾邕玎玑玮玢玟珏珂珑玷玳珀珉珈珥珙顼琊珩珧珞玺珲琏琪瑛琦琥琨琰琮琬"],
["e840","鐯",14,"鐿",43,"鑬鑭鑮鑯"],
["e880","鑰",20,"钑钖钘铇铏铓铔铚铦铻锜锠琛琚瑁瑜瑗瑕瑙瑷瑭瑾璜璎璀璁璇璋璞璨璩璐璧瓒璺韪韫韬杌杓杞杈杩枥枇杪杳枘枧杵枨枞枭枋杷杼柰栉柘栊柩枰栌柙枵柚枳柝栀柃枸柢栎柁柽栲栳桠桡桎桢桄桤梃栝桕桦桁桧桀栾桊桉栩梵梏桴桷梓桫棂楮棼椟椠棹"],
["e940","锧锳锽镃镈镋镕镚镠镮镴镵長",7,"門",42],
["e980","閫",32,"椤棰椋椁楗棣椐楱椹楠楂楝榄楫榀榘楸椴槌榇榈槎榉楦楣楹榛榧榻榫榭槔榱槁槊槟榕槠榍槿樯槭樗樘橥槲橄樾檠橐橛樵檎橹樽樨橘橼檑檐檩檗檫猷獒殁殂殇殄殒殓殍殚殛殡殪轫轭轱轲轳轵轶轸轷轹轺轼轾辁辂辄辇辋"],
["ea40","闌",27,"闬闿阇阓阘阛阞阠阣",6,"阫阬阭阯阰阷阸阹阺阾陁陃陊陎陏陑陒陓陖陗"],
["ea80","陘陙陚陜陝陞陠陣陥陦陫陭",4,"陳陸",12,"隇隉隊辍辎辏辘辚軎戋戗戛戟戢戡戥戤戬臧瓯瓴瓿甏甑甓攴旮旯旰昊昙杲昃昕昀炅曷昝昴昱昶昵耆晟晔晁晏晖晡晗晷暄暌暧暝暾曛曜曦曩贲贳贶贻贽赀赅赆赈赉赇赍赕赙觇觊觋觌觎觏觐觑牮犟牝牦牯牾牿犄犋犍犏犒挈挲掰"],
["eb40","隌階隑隒隓隕隖隚際隝",9,"隨",7,"隱隲隴隵隷隸隺隻隿雂雃雈雊雋雐雑雓雔雖",9,"雡",6,"雫"],
["eb80","雬雭雮雰雱雲雴雵雸雺電雼雽雿霂霃霅霊霋霌霐霑霒霔霕霗",4,"霝霟霠搿擘耄毪毳毽毵毹氅氇氆氍氕氘氙氚氡氩氤氪氲攵敕敫牍牒牖爰虢刖肟肜肓肼朊肽肱肫肭肴肷胧胨胩胪胛胂胄胙胍胗朐胝胫胱胴胭脍脎胲胼朕脒豚脶脞脬脘脲腈腌腓腴腙腚腱腠腩腼腽腭腧塍媵膈膂膑滕膣膪臌朦臊膻"],
["ec40","霡",8,"霫霬霮霯霱霳",4,"霺霻霼霽霿",18,"靔靕靗靘靚靜靝靟靣靤靦靧靨靪",7],
["ec80","靲靵靷",4,"靽",7,"鞆",4,"鞌鞎鞏鞐鞓鞕鞖鞗鞙",4,"臁膦欤欷欹歃歆歙飑飒飓飕飙飚殳彀毂觳斐齑斓於旆旄旃旌旎旒旖炀炜炖炝炻烀炷炫炱烨烊焐焓焖焯焱煳煜煨煅煲煊煸煺熘熳熵熨熠燠燔燧燹爝爨灬焘煦熹戾戽扃扈扉礻祀祆祉祛祜祓祚祢祗祠祯祧祺禅禊禚禧禳忑忐"],
["ed40","鞞鞟鞡鞢鞤",6,"鞬鞮鞰鞱鞳鞵",46],
["ed80","韤韥韨韮",4,"韴韷",23,"怼恝恚恧恁恙恣悫愆愍慝憩憝懋懑戆肀聿沓泶淼矶矸砀砉砗砘砑斫砭砜砝砹砺砻砟砼砥砬砣砩硎硭硖硗砦硐硇硌硪碛碓碚碇碜碡碣碲碹碥磔磙磉磬磲礅磴礓礤礞礴龛黹黻黼盱眄眍盹眇眈眚眢眙眭眦眵眸睐睑睇睃睚睨"],
["ee40","頏",62],
["ee80","顎",32,"睢睥睿瞍睽瞀瞌瞑瞟瞠瞰瞵瞽町畀畎畋畈畛畲畹疃罘罡罟詈罨罴罱罹羁罾盍盥蠲钅钆钇钋钊钌钍钏钐钔钗钕钚钛钜钣钤钫钪钭钬钯钰钲钴钶",4,"钼钽钿铄铈",6,"铐铑铒铕铖铗铙铘铛铞铟铠铢铤铥铧铨铪"],
["ef40","顯",5,"颋颎颒颕颙颣風",37,"飏飐飔飖飗飛飜飝飠",4],
["ef80","飥飦飩",30,"铩铫铮铯铳铴铵铷铹铼铽铿锃锂锆锇锉锊锍锎锏锒",4,"锘锛锝锞锟锢锪锫锩锬锱锲锴锶锷锸锼锾锿镂锵镄镅镆镉镌镎镏镒镓镔镖镗镘镙镛镞镟镝镡镢镤",8,"镯镱镲镳锺矧矬雉秕秭秣秫稆嵇稃稂稞稔"],
["f040","餈",4,"餎餏餑",28,"餯",26],
["f080","饊",9,"饖",12,"饤饦饳饸饹饻饾馂馃馉稹稷穑黏馥穰皈皎皓皙皤瓞瓠甬鸠鸢鸨",4,"鸲鸱鸶鸸鸷鸹鸺鸾鹁鹂鹄鹆鹇鹈鹉鹋鹌鹎鹑鹕鹗鹚鹛鹜鹞鹣鹦",6,"鹱鹭鹳疒疔疖疠疝疬疣疳疴疸痄疱疰痃痂痖痍痣痨痦痤痫痧瘃痱痼痿瘐瘀瘅瘌瘗瘊瘥瘘瘕瘙"],
["f140","馌馎馚",10,"馦馧馩",47],
["f180","駙",32,"瘛瘼瘢瘠癀瘭瘰瘿瘵癃瘾瘳癍癞癔癜癖癫癯翊竦穸穹窀窆窈窕窦窠窬窨窭窳衤衩衲衽衿袂袢裆袷袼裉裢裎裣裥裱褚裼裨裾裰褡褙褓褛褊褴褫褶襁襦襻疋胥皲皴矜耒耔耖耜耠耢耥耦耧耩耨耱耋耵聃聆聍聒聩聱覃顸颀颃"],
["f240","駺",62],
["f280","騹",32,"颉颌颍颏颔颚颛颞颟颡颢颥颦虍虔虬虮虿虺虼虻蚨蚍蚋蚬蚝蚧蚣蚪蚓蚩蚶蛄蚵蛎蚰蚺蚱蚯蛉蛏蚴蛩蛱蛲蛭蛳蛐蜓蛞蛴蛟蛘蛑蜃蜇蛸蜈蜊蜍蜉蜣蜻蜞蜥蜮蜚蜾蝈蜴蜱蜩蜷蜿螂蜢蝽蝾蝻蝠蝰蝌蝮螋蝓蝣蝼蝤蝙蝥螓螯螨蟒"],
["f340","驚",17,"驲骃骉骍骎骔骕骙骦骩",6,"骲骳骴骵骹骻骽骾骿髃髄髆",4,"髍髎髏髐髒體髕髖髗髙髚髛髜"],
["f380","髝髞髠髢髣髤髥髧髨髩髪髬髮髰",8,"髺髼",6,"鬄鬅鬆蟆螈螅螭螗螃螫蟥螬螵螳蟋蟓螽蟑蟀蟊蟛蟪蟠蟮蠖蠓蟾蠊蠛蠡蠹蠼缶罂罄罅舐竺竽笈笃笄笕笊笫笏筇笸笪笙笮笱笠笥笤笳笾笞筘筚筅筵筌筝筠筮筻筢筲筱箐箦箧箸箬箝箨箅箪箜箢箫箴篑篁篌篝篚篥篦篪簌篾篼簏簖簋"],
["f440","鬇鬉",5,"鬐鬑鬒鬔",10,"鬠鬡鬢鬤",10,"鬰鬱鬳",7,"鬽鬾鬿魀魆魊魋魌魎魐魒魓魕",5],
["f480","魛",32,"簟簪簦簸籁籀臾舁舂舄臬衄舡舢舣舭舯舨舫舸舻舳舴舾艄艉艋艏艚艟艨衾袅袈裘裟襞羝羟羧羯羰羲籼敉粑粝粜粞粢粲粼粽糁糇糌糍糈糅糗糨艮暨羿翎翕翥翡翦翩翮翳糸絷綦綮繇纛麸麴赳趄趔趑趱赧赭豇豉酊酐酎酏酤"],
["f540","魼",62],
["f580","鮻",32,"酢酡酰酩酯酽酾酲酴酹醌醅醐醍醑醢醣醪醭醮醯醵醴醺豕鹾趸跫踅蹙蹩趵趿趼趺跄跖跗跚跞跎跏跛跆跬跷跸跣跹跻跤踉跽踔踝踟踬踮踣踯踺蹀踹踵踽踱蹉蹁蹂蹑蹒蹊蹰蹶蹼蹯蹴躅躏躔躐躜躞豸貂貊貅貘貔斛觖觞觚觜"],
["f640","鯜",62],
["f680","鰛",32,"觥觫觯訾謦靓雩雳雯霆霁霈霏霎霪霭霰霾龀龃龅",5,"龌黾鼋鼍隹隼隽雎雒瞿雠銎銮鋈錾鍪鏊鎏鐾鑫鱿鲂鲅鲆鲇鲈稣鲋鲎鲐鲑鲒鲔鲕鲚鲛鲞",5,"鲥",4,"鲫鲭鲮鲰",7,"鲺鲻鲼鲽鳄鳅鳆鳇鳊鳋"],
["f740","鰼",62],
["f780","鱻鱽鱾鲀鲃鲄鲉鲊鲌鲏鲓鲖鲗鲘鲙鲝鲪鲬鲯鲹鲾",4,"鳈鳉鳑鳒鳚鳛鳠鳡鳌",4,"鳓鳔鳕鳗鳘鳙鳜鳝鳟鳢靼鞅鞑鞒鞔鞯鞫鞣鞲鞴骱骰骷鹘骶骺骼髁髀髅髂髋髌髑魅魃魇魉魈魍魑飨餍餮饕饔髟髡髦髯髫髻髭髹鬈鬏鬓鬟鬣麽麾縻麂麇麈麋麒鏖麝麟黛黜黝黠黟黢黩黧黥黪黯鼢鼬鼯鼹鼷鼽鼾齄"],
["f840","鳣",62],
["f880","鴢",32],
["f940","鵃",62],
["f980","鶂",32],
["fa40","鶣",62],
["fa80","鷢",32],
["fb40","鸃",27,"鸤鸧鸮鸰鸴鸻鸼鹀鹍鹐鹒鹓鹔鹖鹙鹝鹟鹠鹡鹢鹥鹮鹯鹲鹴",9,"麀"],
["fb80","麁麃麄麅麆麉麊麌",5,"麔",8,"麞麠",5,"麧麨麩麪"],
["fc40","麫",8,"麵麶麷麹麺麼麿",4,"黅黆黇黈黊黋黌黐黒黓黕黖黗黙黚點黡黣黤黦黨黫黬黭黮黰",8,"黺黽黿",6],
["fc80","鼆",4,"鼌鼏鼑鼒鼔鼕鼖鼘鼚",5,"鼡鼣",8,"鼭鼮鼰鼱"],
["fd40","鼲",4,"鼸鼺鼼鼿",4,"齅",10,"齒",38],
["fd80","齹",5,"龁龂龍",11,"龜龝龞龡",4,"郎凉秊裏隣"],
["fe40","兀嗀﨎﨏﨑﨓﨔礼﨟蘒﨡﨣﨤﨧﨨﨩"]
]
;

var cp936$1 = /*#__PURE__*/Object.freeze({
            default: cp936
});

var gbkAdded = [
["a140","",62],
["a180","",32],
["a240","",62],
["a280","",32],
["a2ab","",5],
["a2e3","€"],
["a2ef",""],
["a2fd",""],
["a340","",62],
["a380","",31,"　"],
["a440","",62],
["a480","",32],
["a4f4","",10],
["a540","",62],
["a580","",32],
["a5f7","",7],
["a640","",62],
["a680","",32],
["a6b9","",7],
["a6d9","",6],
["a6ec",""],
["a6f3",""],
["a6f6","",8],
["a740","",62],
["a780","",32],
["a7c2","",14],
["a7f2","",12],
["a896","",10],
["a8bc",""],
["a8bf","ǹ"],
["a8c1",""],
["a8ea","",20],
["a958",""],
["a95b",""],
["a95d",""],
["a989","〾⿰",11],
["a997","",12],
["a9f0","",14],
["aaa1","",93],
["aba1","",93],
["aca1","",93],
["ada1","",93],
["aea1","",93],
["afa1","",93],
["d7fa","",4],
["f8a1","",93],
["f9a1","",93],
["faa1","",93],
["fba1","",93],
["fca1","",93],
["fda1","",93],
["fe50","⺁⺄㑳㑇⺈⺋㖞㘚㘎⺌⺗㥮㤘㧏㧟㩳㧐㭎㱮㳠⺧⺪䁖䅟⺮䌷⺳⺶⺷䎱䎬⺻䏝䓖䙡䙌"],
["fe80","䜣䜩䝼䞍⻊䥇䥺䥽䦂䦃䦅䦆䦟䦛䦷䦶䲣䲟䲠䲡䱷䲢䴓",6,"䶮",93]
]
;

var gbkAdded$1 = /*#__PURE__*/Object.freeze({
            default: gbkAdded
});

var uChars = [
	128,
	165,
	169,
	178,
	184,
	216,
	226,
	235,
	238,
	244,
	248,
	251,
	253,
	258,
	276,
	284,
	300,
	325,
	329,
	334,
	364,
	463,
	465,
	467,
	469,
	471,
	473,
	475,
	477,
	506,
	594,
	610,
	712,
	716,
	730,
	930,
	938,
	962,
	970,
	1026,
	1104,
	1106,
	8209,
	8215,
	8218,
	8222,
	8231,
	8241,
	8244,
	8246,
	8252,
	8365,
	8452,
	8454,
	8458,
	8471,
	8482,
	8556,
	8570,
	8596,
	8602,
	8713,
	8720,
	8722,
	8726,
	8731,
	8737,
	8740,
	8742,
	8748,
	8751,
	8760,
	8766,
	8777,
	8781,
	8787,
	8802,
	8808,
	8816,
	8854,
	8858,
	8870,
	8896,
	8979,
	9322,
	9372,
	9548,
	9588,
	9616,
	9622,
	9634,
	9652,
	9662,
	9672,
	9676,
	9680,
	9702,
	9735,
	9738,
	9793,
	9795,
	11906,
	11909,
	11913,
	11917,
	11928,
	11944,
	11947,
	11951,
	11956,
	11960,
	11964,
	11979,
	12284,
	12292,
	12312,
	12319,
	12330,
	12351,
	12436,
	12447,
	12535,
	12543,
	12586,
	12842,
	12850,
	12964,
	13200,
	13215,
	13218,
	13253,
	13263,
	13267,
	13270,
	13384,
	13428,
	13727,
	13839,
	13851,
	14617,
	14703,
	14801,
	14816,
	14964,
	15183,
	15471,
	15585,
	16471,
	16736,
	17208,
	17325,
	17330,
	17374,
	17623,
	17997,
	18018,
	18212,
	18218,
	18301,
	18318,
	18760,
	18811,
	18814,
	18820,
	18823,
	18844,
	18848,
	18872,
	19576,
	19620,
	19738,
	19887,
	40870,
	59244,
	59336,
	59367,
	59413,
	59417,
	59423,
	59431,
	59437,
	59443,
	59452,
	59460,
	59478,
	59493,
	63789,
	63866,
	63894,
	63976,
	63986,
	64016,
	64018,
	64021,
	64025,
	64034,
	64037,
	64042,
	65074,
	65093,
	65107,
	65112,
	65127,
	65132,
	65375,
	65510,
	65536
];
var gbChars = [
	0,
	36,
	38,
	45,
	50,
	81,
	89,
	95,
	96,
	100,
	103,
	104,
	105,
	109,
	126,
	133,
	148,
	172,
	175,
	179,
	208,
	306,
	307,
	308,
	309,
	310,
	311,
	312,
	313,
	341,
	428,
	443,
	544,
	545,
	558,
	741,
	742,
	749,
	750,
	805,
	819,
	820,
	7922,
	7924,
	7925,
	7927,
	7934,
	7943,
	7944,
	7945,
	7950,
	8062,
	8148,
	8149,
	8152,
	8164,
	8174,
	8236,
	8240,
	8262,
	8264,
	8374,
	8380,
	8381,
	8384,
	8388,
	8390,
	8392,
	8393,
	8394,
	8396,
	8401,
	8406,
	8416,
	8419,
	8424,
	8437,
	8439,
	8445,
	8482,
	8485,
	8496,
	8521,
	8603,
	8936,
	8946,
	9046,
	9050,
	9063,
	9066,
	9076,
	9092,
	9100,
	9108,
	9111,
	9113,
	9131,
	9162,
	9164,
	9218,
	9219,
	11329,
	11331,
	11334,
	11336,
	11346,
	11361,
	11363,
	11366,
	11370,
	11372,
	11375,
	11389,
	11682,
	11686,
	11687,
	11692,
	11694,
	11714,
	11716,
	11723,
	11725,
	11730,
	11736,
	11982,
	11989,
	12102,
	12336,
	12348,
	12350,
	12384,
	12393,
	12395,
	12397,
	12510,
	12553,
	12851,
	12962,
	12973,
	13738,
	13823,
	13919,
	13933,
	14080,
	14298,
	14585,
	14698,
	15583,
	15847,
	16318,
	16434,
	16438,
	16481,
	16729,
	17102,
	17122,
	17315,
	17320,
	17402,
	17418,
	17859,
	17909,
	17911,
	17915,
	17916,
	17936,
	17939,
	17961,
	18664,
	18703,
	18814,
	18962,
	19043,
	33469,
	33470,
	33471,
	33484,
	33485,
	33490,
	33497,
	33501,
	33505,
	33513,
	33520,
	33536,
	33550,
	37845,
	37921,
	37948,
	38029,
	38038,
	38064,
	38065,
	38066,
	38069,
	38075,
	38076,
	38078,
	39108,
	39109,
	39113,
	39114,
	39115,
	39116,
	39265,
	39394,
	189000
];
var gb18030Ranges = {
	uChars: uChars,
	gbChars: gbChars
};

var gb18030Ranges$1 = /*#__PURE__*/Object.freeze({
            uChars: uChars,
            gbChars: gbChars,
            default: gb18030Ranges
});

var cp949 = [
["0","\u0000",127],
["8141","갂갃갅갆갋",4,"갘갞갟갡갢갣갥",6,"갮갲갳갴"],
["8161","갵갶갷갺갻갽갾갿걁",9,"걌걎",5,"걕"],
["8181","걖걗걙걚걛걝",18,"걲걳걵걶걹걻",4,"겂겇겈겍겎겏겑겒겓겕",6,"겞겢",5,"겫겭겮겱",6,"겺겾겿곀곂곃곅곆곇곉곊곋곍",7,"곖곘",7,"곢곣곥곦곩곫곭곮곲곴곷",4,"곾곿괁괂괃괅괇",4,"괎괐괒괓"],
["8241","괔괕괖괗괙괚괛괝괞괟괡",7,"괪괫괮",5],
["8261","괶괷괹괺괻괽",6,"굆굈굊",5,"굑굒굓굕굖굗"],
["8281","굙",7,"굢굤",7,"굮굯굱굲굷굸굹굺굾궀궃",4,"궊궋궍궎궏궑",10,"궞",5,"궥",17,"궸",7,"귂귃귅귆귇귉",6,"귒귔",7,"귝귞귟귡귢귣귥",18],
["8341","귺귻귽귾긂",5,"긊긌긎",5,"긕",7],
["8361","긝",18,"긲긳긵긶긹긻긼"],
["8381","긽긾긿깂깄깇깈깉깋깏깑깒깓깕깗",4,"깞깢깣깤깦깧깪깫깭깮깯깱",6,"깺깾",5,"꺆",5,"꺍",46,"꺿껁껂껃껅",6,"껎껒",5,"껚껛껝",8],
["8441","껦껧껩껪껬껮",5,"껵껶껷껹껺껻껽",8],
["8461","꼆꼉꼊꼋꼌꼎꼏꼑",18],
["8481","꼤",7,"꼮꼯꼱꼳꼵",6,"꼾꽀꽄꽅꽆꽇꽊",5,"꽑",10,"꽞",5,"꽦",18,"꽺",5,"꾁꾂꾃꾅꾆꾇꾉",6,"꾒꾓꾔꾖",5,"꾝",26,"꾺꾻꾽꾾"],
["8541","꾿꿁",5,"꿊꿌꿏",4,"꿕",6,"꿝",4],
["8561","꿢",5,"꿪",5,"꿲꿳꿵꿶꿷꿹",6,"뀂뀃"],
["8581","뀅",6,"뀍뀎뀏뀑뀒뀓뀕",6,"뀞",9,"뀩",26,"끆끇끉끋끍끏끐끑끒끖끘끚끛끜끞",29,"끾끿낁낂낃낅",6,"낎낐낒",5,"낛낝낞낣낤"],
["8641","낥낦낧낪낰낲낶낷낹낺낻낽",6,"냆냊",5,"냒"],
["8661","냓냕냖냗냙",6,"냡냢냣냤냦",10],
["8681","냱",22,"넊넍넎넏넑넔넕넖넗넚넞",4,"넦넧넩넪넫넭",6,"넶넺",5,"녂녃녅녆녇녉",6,"녒녓녖녗녙녚녛녝녞녟녡",22,"녺녻녽녾녿놁놃",4,"놊놌놎놏놐놑놕놖놗놙놚놛놝"],
["8741","놞",9,"놩",15],
["8761","놹",18,"뇍뇎뇏뇑뇒뇓뇕"],
["8781","뇖",5,"뇞뇠",7,"뇪뇫뇭뇮뇯뇱",7,"뇺뇼뇾",5,"눆눇눉눊눍",6,"눖눘눚",5,"눡",18,"눵",6,"눽",26,"뉙뉚뉛뉝뉞뉟뉡",6,"뉪",4],
["8841","뉯",4,"뉶",5,"뉽",6,"늆늇늈늊",4],
["8861","늏늒늓늕늖늗늛",4,"늢늤늧늨늩늫늭늮늯늱늲늳늵늶늷"],
["8881","늸",15,"닊닋닍닎닏닑닓",4,"닚닜닞닟닠닡닣닧닩닪닰닱닲닶닼닽닾댂댃댅댆댇댉",6,"댒댖",5,"댝",54,"덗덙덚덝덠덡덢덣"],
["8941","덦덨덪덬덭덯덲덳덵덶덷덹",6,"뎂뎆",5,"뎍"],
["8961","뎎뎏뎑뎒뎓뎕",10,"뎢",5,"뎩뎪뎫뎭"],
["8981","뎮",21,"돆돇돉돊돍돏돑돒돓돖돘돚돜돞돟돡돢돣돥돦돧돩",18,"돽",18,"됑",6,"됙됚됛됝됞됟됡",6,"됪됬",7,"됵",15],
["8a41","둅",10,"둒둓둕둖둗둙",6,"둢둤둦"],
["8a61","둧",4,"둭",18,"뒁뒂"],
["8a81","뒃",4,"뒉",19,"뒞",5,"뒥뒦뒧뒩뒪뒫뒭",7,"뒶뒸뒺",5,"듁듂듃듅듆듇듉",6,"듑듒듓듔듖",5,"듞듟듡듢듥듧",4,"듮듰듲",5,"듹",26,"딖딗딙딚딝"],
["8b41","딞",5,"딦딫",4,"딲딳딵딶딷딹",6,"땂땆"],
["8b61","땇땈땉땊땎땏땑땒땓땕",6,"땞땢",8],
["8b81","땫",52,"떢떣떥떦떧떩떬떭떮떯떲떶",4,"떾떿뗁뗂뗃뗅",6,"뗎뗒",5,"뗙",18,"뗭",18],
["8c41","똀",15,"똒똓똕똖똗똙",4],
["8c61","똞",6,"똦",5,"똭",6,"똵",5],
["8c81","똻",12,"뙉",26,"뙥뙦뙧뙩",50,"뚞뚟뚡뚢뚣뚥",5,"뚭뚮뚯뚰뚲",16],
["8d41","뛃",16,"뛕",8],
["8d61","뛞",17,"뛱뛲뛳뛵뛶뛷뛹뛺"],
["8d81","뛻",4,"뜂뜃뜄뜆",33,"뜪뜫뜭뜮뜱",6,"뜺뜼",7,"띅띆띇띉띊띋띍",6,"띖",9,"띡띢띣띥띦띧띩",6,"띲띴띶",5,"띾띿랁랂랃랅",6,"랎랓랔랕랚랛랝랞"],
["8e41","랟랡",6,"랪랮",5,"랶랷랹",8],
["8e61","럂",4,"럈럊",19],
["8e81","럞",13,"럮럯럱럲럳럵",6,"럾렂",4,"렊렋렍렎렏렑",6,"렚렜렞",5,"렦렧렩렪렫렭",6,"렶렺",5,"롁롂롃롅",11,"롒롔",7,"롞롟롡롢롣롥",6,"롮롰롲",5,"롹롺롻롽",7],
["8f41","뢅",7,"뢎",17],
["8f61","뢠",7,"뢩",6,"뢱뢲뢳뢵뢶뢷뢹",4],
["8f81","뢾뢿룂룄룆",5,"룍룎룏룑룒룓룕",7,"룞룠룢",5,"룪룫룭룮룯룱",6,"룺룼룾",5,"뤅",18,"뤙",6,"뤡",26,"뤾뤿륁륂륃륅",6,"륍륎륐륒",5],
["9041","륚륛륝륞륟륡",6,"륪륬륮",5,"륶륷륹륺륻륽"],
["9061","륾",5,"릆릈릋릌릏",15],
["9081","릟",12,"릮릯릱릲릳릵",6,"릾맀맂",5,"맊맋맍맓",4,"맚맜맟맠맢맦맧맩맪맫맭",6,"맶맻",4,"먂",5,"먉",11,"먖",33,"먺먻먽먾먿멁멃멄멅멆"],
["9141","멇멊멌멏멐멑멒멖멗멙멚멛멝",6,"멦멪",5],
["9161","멲멳멵멶멷멹",9,"몆몈몉몊몋몍",5],
["9181","몓",20,"몪몭몮몯몱몳",4,"몺몼몾",5,"뫅뫆뫇뫉",14,"뫚",33,"뫽뫾뫿묁묂묃묅",7,"묎묐묒",5,"묙묚묛묝묞묟묡",6],
["9241","묨묪묬",7,"묷묹묺묿",4,"뭆뭈뭊뭋뭌뭎뭑뭒"],
["9261","뭓뭕뭖뭗뭙",7,"뭢뭤",7,"뭭",4],
["9281","뭲",21,"뮉뮊뮋뮍뮎뮏뮑",18,"뮥뮦뮧뮩뮪뮫뮭",6,"뮵뮶뮸",7,"믁믂믃믅믆믇믉",6,"믑믒믔",35,"믺믻믽믾밁"],
["9341","밃",4,"밊밎밐밒밓밙밚밠밡밢밣밦밨밪밫밬밮밯밲밳밵"],
["9361","밶밷밹",6,"뱂뱆뱇뱈뱊뱋뱎뱏뱑",8],
["9381","뱚뱛뱜뱞",37,"벆벇벉벊벍벏",4,"벖벘벛",4,"벢벣벥벦벩",6,"벲벶",5,"벾벿볁볂볃볅",7,"볎볒볓볔볖볗볙볚볛볝",22,"볷볹볺볻볽"],
["9441","볾",5,"봆봈봊",5,"봑봒봓봕",8],
["9461","봞",5,"봥",6,"봭",12],
["9481","봺",5,"뵁",6,"뵊뵋뵍뵎뵏뵑",6,"뵚",9,"뵥뵦뵧뵩",22,"붂붃붅붆붋",4,"붒붔붖붗붘붛붝",6,"붥",10,"붱",6,"붹",24],
["9541","뷒뷓뷖뷗뷙뷚뷛뷝",11,"뷪",5,"뷱"],
["9561","뷲뷳뷵뷶뷷뷹",6,"븁븂븄븆",5,"븎븏븑븒븓"],
["9581","븕",6,"븞븠",35,"빆빇빉빊빋빍빏",4,"빖빘빜빝빞빟빢빣빥빦빧빩빫",4,"빲빶",4,"빾빿뺁뺂뺃뺅",6,"뺎뺒",5,"뺚",13,"뺩",14],
["9641","뺸",23,"뻒뻓"],
["9661","뻕뻖뻙",6,"뻡뻢뻦",5,"뻭",8],
["9681","뻶",10,"뼂",5,"뼊",13,"뼚뼞",33,"뽂뽃뽅뽆뽇뽉",6,"뽒뽓뽔뽖",44],
["9741","뾃",16,"뾕",8],
["9761","뾞",17,"뾱",7],
["9781","뾹",11,"뿆",5,"뿎뿏뿑뿒뿓뿕",6,"뿝뿞뿠뿢",89,"쀽쀾쀿"],
["9841","쁀",16,"쁒",5,"쁙쁚쁛"],
["9861","쁝쁞쁟쁡",6,"쁪",15],
["9881","쁺",21,"삒삓삕삖삗삙",6,"삢삤삦",5,"삮삱삲삷",4,"삾샂샃샄샆샇샊샋샍샎샏샑",6,"샚샞",5,"샦샧샩샪샫샭",6,"샶샸샺",5,"섁섂섃섅섆섇섉",6,"섑섒섓섔섖",5,"섡섢섥섨섩섪섫섮"],
["9941","섲섳섴섵섷섺섻섽섾섿셁",6,"셊셎",5,"셖셗"],
["9961","셙셚셛셝",6,"셦셪",5,"셱셲셳셵셶셷셹셺셻"],
["9981","셼",8,"솆",5,"솏솑솒솓솕솗",4,"솞솠솢솣솤솦솧솪솫솭솮솯솱",11,"솾",5,"쇅쇆쇇쇉쇊쇋쇍",6,"쇕쇖쇙",6,"쇡쇢쇣쇥쇦쇧쇩",6,"쇲쇴",7,"쇾쇿숁숂숃숅",6,"숎숐숒",5,"숚숛숝숞숡숢숣"],
["9a41","숤숥숦숧숪숬숮숰숳숵",16],
["9a61","쉆쉇쉉",6,"쉒쉓쉕쉖쉗쉙",6,"쉡쉢쉣쉤쉦"],
["9a81","쉧",4,"쉮쉯쉱쉲쉳쉵",6,"쉾슀슂",5,"슊",5,"슑",6,"슙슚슜슞",5,"슦슧슩슪슫슮",5,"슶슸슺",33,"싞싟싡싢싥",5,"싮싰싲싳싴싵싷싺싽싾싿쌁",6,"쌊쌋쌎쌏"],
["9b41","쌐쌑쌒쌖쌗쌙쌚쌛쌝",6,"쌦쌧쌪",8],
["9b61","쌳",17,"썆",7],
["9b81","썎",25,"썪썫썭썮썯썱썳",4,"썺썻썾",5,"쎅쎆쎇쎉쎊쎋쎍",50,"쏁",22,"쏚"],
["9c41","쏛쏝쏞쏡쏣",4,"쏪쏫쏬쏮",5,"쏶쏷쏹",5],
["9c61","쏿",8,"쐉",6,"쐑",9],
["9c81","쐛",8,"쐥",6,"쐭쐮쐯쐱쐲쐳쐵",6,"쐾",9,"쑉",26,"쑦쑧쑩쑪쑫쑭",6,"쑶쑷쑸쑺",5,"쒁",18,"쒕",6,"쒝",12],
["9d41","쒪",13,"쒹쒺쒻쒽",8],
["9d61","쓆",25],
["9d81","쓠",8,"쓪",5,"쓲쓳쓵쓶쓷쓹쓻쓼쓽쓾씂",9,"씍씎씏씑씒씓씕",6,"씝",10,"씪씫씭씮씯씱",6,"씺씼씾",5,"앆앇앋앏앐앑앒앖앚앛앜앟앢앣앥앦앧앩",6,"앲앶",5,"앾앿얁얂얃얅얆얈얉얊얋얎얐얒얓얔"],
["9e41","얖얙얚얛얝얞얟얡",7,"얪",9,"얶"],
["9e61","얷얺얿",4,"엋엍엏엒엓엕엖엗엙",6,"엢엤엦엧"],
["9e81","엨엩엪엫엯엱엲엳엵엸엹엺엻옂옃옄옉옊옋옍옎옏옑",6,"옚옝",6,"옦옧옩옪옫옯옱옲옶옸옺옼옽옾옿왂왃왅왆왇왉",6,"왒왖",5,"왞왟왡",10,"왭왮왰왲",5,"왺왻왽왾왿욁",6,"욊욌욎",5,"욖욗욙욚욛욝",6,"욦"],
["9f41","욨욪",5,"욲욳욵욶욷욻",4,"웂웄웆",5,"웎"],
["9f61","웏웑웒웓웕",6,"웞웟웢",5,"웪웫웭웮웯웱웲"],
["9f81","웳",4,"웺웻웼웾",5,"윆윇윉윊윋윍",6,"윖윘윚",5,"윢윣윥윦윧윩",6,"윲윴윶윸윹윺윻윾윿읁읂읃읅",4,"읋읎읐읙읚읛읝읞읟읡",6,"읩읪읬",7,"읶읷읹읺읻읿잀잁잂잆잋잌잍잏잒잓잕잙잛",4,"잢잧",4,"잮잯잱잲잳잵잶잷"],
["a041","잸잹잺잻잾쟂",5,"쟊쟋쟍쟏쟑",6,"쟙쟚쟛쟜"],
["a061","쟞",5,"쟥쟦쟧쟩쟪쟫쟭",13],
["a081","쟻",4,"젂젃젅젆젇젉젋",4,"젒젔젗",4,"젞젟젡젢젣젥",6,"젮젰젲",5,"젹젺젻젽젾젿졁",6,"졊졋졎",5,"졕",26,"졲졳졵졶졷졹졻",4,"좂좄좈좉좊좎",5,"좕",7,"좞좠좢좣좤"],
["a141","좥좦좧좩",18,"좾좿죀죁"],
["a161","죂죃죅죆죇죉죊죋죍",6,"죖죘죚",5,"죢죣죥"],
["a181","죦",14,"죶",5,"죾죿줁줂줃줇",4,"줎　、。·‥…¨〃­―∥＼∼‘’“”〔〕〈",9,"±×÷≠≤≥∞∴°′″℃Å￠￡￥♂♀∠⊥⌒∂∇≡≒§※☆★○●◎◇◆□■△▲▽▼→←↑↓↔〓≪≫√∽∝∵∫∬∈∋⊆⊇⊂⊃∪∩∧∨￢"],
["a241","줐줒",5,"줙",18],
["a261","줭",6,"줵",18],
["a281","쥈",7,"쥒쥓쥕쥖쥗쥙",6,"쥢쥤",7,"쥭쥮쥯⇒⇔∀∃´～ˇ˘˝˚˙¸˛¡¿ː∮∑∏¤℉‰◁◀▷▶♤♠♡♥♧♣⊙◈▣◐◑▒▤▥▨▧▦▩♨☏☎☜☞¶†‡↕↗↙↖↘♭♩♪♬㉿㈜№㏇™㏂㏘℡€®"],
["a341","쥱쥲쥳쥵",6,"쥽",10,"즊즋즍즎즏"],
["a361","즑",6,"즚즜즞",16],
["a381","즯",16,"짂짃짅짆짉짋",4,"짒짔짗짘짛！",58,"￦］",32,"￣"],
["a441","짞짟짡짣짥짦짨짩짪짫짮짲",5,"짺짻짽짾짿쨁쨂쨃쨄"],
["a461","쨅쨆쨇쨊쨎",5,"쨕쨖쨗쨙",12],
["a481","쨦쨧쨨쨪",28,"ㄱ",93],
["a541","쩇",4,"쩎쩏쩑쩒쩓쩕",6,"쩞쩢",5,"쩩쩪"],
["a561","쩫",17,"쩾",5,"쪅쪆"],
["a581","쪇",16,"쪙",14,"ⅰ",9],
["a5b0","Ⅰ",9],
["a5c1","Α",16,"Σ",6],
["a5e1","α",16,"σ",6],
["a641","쪨",19,"쪾쪿쫁쫂쫃쫅"],
["a661","쫆",5,"쫎쫐쫒쫔쫕쫖쫗쫚",5,"쫡",6],
["a681","쫨쫩쫪쫫쫭",6,"쫵",18,"쬉쬊─│┌┐┘└├┬┤┴┼━┃┏┓┛┗┣┳┫┻╋┠┯┨┷┿┝┰┥┸╂┒┑┚┙┖┕┎┍┞┟┡┢┦┧┩┪┭┮┱┲┵┶┹┺┽┾╀╁╃",7],
["a741","쬋",4,"쬑쬒쬓쬕쬖쬗쬙",6,"쬢",7],
["a761","쬪",22,"쭂쭃쭄"],
["a781","쭅쭆쭇쭊쭋쭍쭎쭏쭑",6,"쭚쭛쭜쭞",5,"쭥",7,"㎕㎖㎗ℓ㎘㏄㎣㎤㎥㎦㎙",9,"㏊㎍㎎㎏㏏㎈㎉㏈㎧㎨㎰",9,"㎀",4,"㎺",5,"㎐",4,"Ω㏀㏁㎊㎋㎌㏖㏅㎭㎮㎯㏛㎩㎪㎫㎬㏝㏐㏓㏃㏉㏜㏆"],
["a841","쭭",10,"쭺",14],
["a861","쮉",18,"쮝",6],
["a881","쮤",19,"쮹",11,"ÆÐªĦ"],
["a8a6","Ĳ"],
["a8a8","ĿŁØŒºÞŦŊ"],
["a8b1","㉠",27,"ⓐ",25,"①",14,"½⅓⅔¼¾⅛⅜⅝⅞"],
["a941","쯅",14,"쯕",10],
["a961","쯠쯡쯢쯣쯥쯦쯨쯪",18],
["a981","쯽",14,"찎찏찑찒찓찕",6,"찞찟찠찣찤æđðħıĳĸŀłøœßþŧŋŉ㈀",27,"⒜",25,"⑴",14,"¹²³⁴ⁿ₁₂₃₄"],
["aa41","찥찦찪찫찭찯찱",6,"찺찿",4,"챆챇챉챊챋챍챎"],
["aa61","챏",4,"챖챚",5,"챡챢챣챥챧챩",6,"챱챲"],
["aa81","챳챴챶",29,"ぁ",82],
["ab41","첔첕첖첗첚첛첝첞첟첡",6,"첪첮",5,"첶첷첹"],
["ab61","첺첻첽",6,"쳆쳈쳊",5,"쳑쳒쳓쳕",5],
["ab81","쳛",8,"쳥",6,"쳭쳮쳯쳱",12,"ァ",85],
["ac41","쳾쳿촀촂",5,"촊촋촍촎촏촑",6,"촚촜촞촟촠"],
["ac61","촡촢촣촥촦촧촩촪촫촭",11,"촺",4],
["ac81","촿",28,"쵝쵞쵟А",5,"ЁЖ",25],
["acd1","а",5,"ёж",25],
["ad41","쵡쵢쵣쵥",6,"쵮쵰쵲",5,"쵹",7],
["ad61","춁",6,"춉",10,"춖춗춙춚춛춝춞춟"],
["ad81","춠춡춢춣춦춨춪",5,"춱",18,"췅"],
["ae41","췆",5,"췍췎췏췑",16],
["ae61","췢",5,"췩췪췫췭췮췯췱",6,"췺췼췾",4],
["ae81","츃츅츆츇츉츊츋츍",6,"츕츖츗츘츚",5,"츢츣츥츦츧츩츪츫"],
["af41","츬츭츮츯츲츴츶",19],
["af61","칊",13,"칚칛칝칞칢",5,"칪칬"],
["af81","칮",5,"칶칷칹칺칻칽",6,"캆캈캊",5,"캒캓캕캖캗캙"],
["b041","캚",5,"캢캦",5,"캮",12],
["b061","캻",5,"컂",19],
["b081","컖",13,"컦컧컩컪컭",6,"컶컺",5,"가각간갇갈갉갊감",7,"같",4,"갠갤갬갭갯갰갱갸갹갼걀걋걍걔걘걜거걱건걷걸걺검겁것겄겅겆겉겊겋게겐겔겜겝겟겠겡겨격겪견겯결겸겹겻겼경곁계곈곌곕곗고곡곤곧골곪곬곯곰곱곳공곶과곽관괄괆"],
["b141","켂켃켅켆켇켉",6,"켒켔켖",5,"켝켞켟켡켢켣"],
["b161","켥",6,"켮켲",5,"켹",11],
["b181","콅",14,"콖콗콙콚콛콝",6,"콦콨콪콫콬괌괍괏광괘괜괠괩괬괭괴괵괸괼굄굅굇굉교굔굘굡굣구국군굳굴굵굶굻굼굽굿궁궂궈궉권궐궜궝궤궷귀귁귄귈귐귑귓규균귤그극근귿글긁금급긋긍긔기긱긴긷길긺김깁깃깅깆깊까깍깎깐깔깖깜깝깟깠깡깥깨깩깬깰깸"],
["b241","콭콮콯콲콳콵콶콷콹",6,"쾁쾂쾃쾄쾆",5,"쾍"],
["b261","쾎",18,"쾢",5,"쾩"],
["b281","쾪",5,"쾱",18,"쿅",6,"깹깻깼깽꺄꺅꺌꺼꺽꺾껀껄껌껍껏껐껑께껙껜껨껫껭껴껸껼꼇꼈꼍꼐꼬꼭꼰꼲꼴꼼꼽꼿꽁꽂꽃꽈꽉꽐꽜꽝꽤꽥꽹꾀꾄꾈꾐꾑꾕꾜꾸꾹꾼꿀꿇꿈꿉꿋꿍꿎꿔꿜꿨꿩꿰꿱꿴꿸뀀뀁뀄뀌뀐뀔뀜뀝뀨끄끅끈끊끌끎끓끔끕끗끙"],
["b341","쿌",19,"쿢쿣쿥쿦쿧쿩"],
["b361","쿪",5,"쿲쿴쿶",5,"쿽쿾쿿퀁퀂퀃퀅",5],
["b381","퀋",5,"퀒",5,"퀙",19,"끝끼끽낀낄낌낍낏낑나낙낚난낟날낡낢남납낫",4,"낱낳내낵낸낼냄냅냇냈냉냐냑냔냘냠냥너넉넋넌널넒넓넘넙넛넜넝넣네넥넨넬넴넵넷넸넹녀녁년녈념녑녔녕녘녜녠노녹논놀놂놈놉놋농높놓놔놘놜놨뇌뇐뇔뇜뇝"],
["b441","퀮",5,"퀶퀷퀹퀺퀻퀽",6,"큆큈큊",5],
["b461","큑큒큓큕큖큗큙",6,"큡",10,"큮큯"],
["b481","큱큲큳큵",6,"큾큿킀킂",18,"뇟뇨뇩뇬뇰뇹뇻뇽누눅눈눋눌눔눕눗눙눠눴눼뉘뉜뉠뉨뉩뉴뉵뉼늄늅늉느늑는늘늙늚늠늡늣능늦늪늬늰늴니닉닌닐닒님닙닛닝닢다닥닦단닫",4,"닳담답닷",4,"닿대댁댄댈댐댑댓댔댕댜더덕덖던덛덜덞덟덤덥"],
["b541","킕",14,"킦킧킩킪킫킭",5],
["b561","킳킶킸킺",5,"탂탃탅탆탇탊",5,"탒탖",4],
["b581","탛탞탟탡탢탣탥",6,"탮탲",5,"탹",11,"덧덩덫덮데덱덴델뎀뎁뎃뎄뎅뎌뎐뎔뎠뎡뎨뎬도독돈돋돌돎돐돔돕돗동돛돝돠돤돨돼됐되된될됨됩됫됴두둑둔둘둠둡둣둥둬뒀뒈뒝뒤뒨뒬뒵뒷뒹듀듄듈듐듕드득든듣들듦듬듭듯등듸디딕딘딛딜딤딥딧딨딩딪따딱딴딸"],
["b641","턅",7,"턎",17],
["b661","턠",15,"턲턳턵턶턷턹턻턼턽턾"],
["b681","턿텂텆",5,"텎텏텑텒텓텕",6,"텞텠텢",5,"텩텪텫텭땀땁땃땄땅땋때땍땐땔땜땝땟땠땡떠떡떤떨떪떫떰떱떳떴떵떻떼떽뗀뗄뗌뗍뗏뗐뗑뗘뗬또똑똔똘똥똬똴뙈뙤뙨뚜뚝뚠뚤뚫뚬뚱뛔뛰뛴뛸뜀뜁뜅뜨뜩뜬뜯뜰뜸뜹뜻띄띈띌띔띕띠띤띨띰띱띳띵라락란랄람랍랏랐랑랒랖랗"],
["b741","텮",13,"텽",6,"톅톆톇톉톊"],
["b761","톋",20,"톢톣톥톦톧"],
["b781","톩",6,"톲톴톶톷톸톹톻톽톾톿퇁",14,"래랙랜랠램랩랫랬랭랴략랸럇량러럭런럴럼럽럿렀렁렇레렉렌렐렘렙렛렝려력련렬렴렵렷렸령례롄롑롓로록론롤롬롭롯롱롸롼뢍뢨뢰뢴뢸룀룁룃룅료룐룔룝룟룡루룩룬룰룸룹룻룽뤄뤘뤠뤼뤽륀륄륌륏륑류륙륜률륨륩"],
["b841","퇐",7,"퇙",17],
["b861","퇫",8,"퇵퇶퇷퇹",13],
["b881","툈툊",5,"툑",24,"륫륭르륵른를름릅릇릉릊릍릎리릭린릴림립릿링마막만많",4,"맘맙맛망맞맡맣매맥맨맬맴맵맷맸맹맺먀먁먈먕머먹먼멀멂멈멉멋멍멎멓메멕멘멜멤멥멧멨멩며멱면멸몃몄명몇몌모목몫몬몰몲몸몹못몽뫄뫈뫘뫙뫼"],
["b941","툪툫툮툯툱툲툳툵",6,"툾퉀퉂",5,"퉉퉊퉋퉌"],
["b961","퉍",14,"퉝",6,"퉥퉦퉧퉨"],
["b981","퉩",22,"튂튃튅튆튇튉튊튋튌묀묄묍묏묑묘묜묠묩묫무묵묶문묻물묽묾뭄뭅뭇뭉뭍뭏뭐뭔뭘뭡뭣뭬뮈뮌뮐뮤뮨뮬뮴뮷므믄믈믐믓미믹민믿밀밂밈밉밋밌밍및밑바",4,"받",4,"밤밥밧방밭배백밴밸뱀뱁뱃뱄뱅뱉뱌뱍뱐뱝버벅번벋벌벎범법벗"],
["ba41","튍튎튏튒튓튔튖",5,"튝튞튟튡튢튣튥",6,"튭"],
["ba61","튮튯튰튲",5,"튺튻튽튾틁틃",4,"틊틌",5],
["ba81","틒틓틕틖틗틙틚틛틝",6,"틦",9,"틲틳틵틶틷틹틺벙벚베벡벤벧벨벰벱벳벴벵벼벽변별볍볏볐병볕볘볜보복볶본볼봄봅봇봉봐봔봤봬뵀뵈뵉뵌뵐뵘뵙뵤뵨부북분붇불붉붊붐붑붓붕붙붚붜붤붰붸뷔뷕뷘뷜뷩뷰뷴뷸븀븃븅브븍븐블븜븝븟비빅빈빌빎빔빕빗빙빚빛빠빡빤"],
["bb41","틻",4,"팂팄팆",5,"팏팑팒팓팕팗",4,"팞팢팣"],
["bb61","팤팦팧팪팫팭팮팯팱",6,"팺팾",5,"퍆퍇퍈퍉"],
["bb81","퍊",31,"빨빪빰빱빳빴빵빻빼빽뺀뺄뺌뺍뺏뺐뺑뺘뺙뺨뻐뻑뻔뻗뻘뻠뻣뻤뻥뻬뼁뼈뼉뼘뼙뼛뼜뼝뽀뽁뽄뽈뽐뽑뽕뾔뾰뿅뿌뿍뿐뿔뿜뿟뿡쀼쁑쁘쁜쁠쁨쁩삐삑삔삘삠삡삣삥사삭삯산삳살삵삶삼삽삿샀상샅새색샌샐샘샙샛샜생샤"],
["bc41","퍪",17,"퍾퍿펁펂펃펅펆펇"],
["bc61","펈펉펊펋펎펒",5,"펚펛펝펞펟펡",6,"펪펬펮"],
["bc81","펯",4,"펵펶펷펹펺펻펽",6,"폆폇폊",5,"폑",5,"샥샨샬샴샵샷샹섀섄섈섐섕서",4,"섣설섦섧섬섭섯섰성섶세섹센셀셈셉셋셌셍셔셕션셜셤셥셧셨셩셰셴셸솅소속솎손솔솖솜솝솟송솥솨솩솬솰솽쇄쇈쇌쇔쇗쇘쇠쇤쇨쇰쇱쇳쇼쇽숀숄숌숍숏숑수숙순숟술숨숩숫숭"],
["bd41","폗폙",7,"폢폤",7,"폮폯폱폲폳폵폶폷"],
["bd61","폸폹폺폻폾퐀퐂",5,"퐉",13],
["bd81","퐗",5,"퐞",25,"숯숱숲숴쉈쉐쉑쉔쉘쉠쉥쉬쉭쉰쉴쉼쉽쉿슁슈슉슐슘슛슝스슥슨슬슭슴습슷승시식신싣실싫심십싯싱싶싸싹싻싼쌀쌈쌉쌌쌍쌓쌔쌕쌘쌜쌤쌥쌨쌩썅써썩썬썰썲썸썹썼썽쎄쎈쎌쏀쏘쏙쏜쏟쏠쏢쏨쏩쏭쏴쏵쏸쐈쐐쐤쐬쐰"],
["be41","퐸",7,"푁푂푃푅",14],
["be61","푔",7,"푝푞푟푡푢푣푥",7,"푮푰푱푲"],
["be81","푳",4,"푺푻푽푾풁풃",4,"풊풌풎",5,"풕",8,"쐴쐼쐽쑈쑤쑥쑨쑬쑴쑵쑹쒀쒔쒜쒸쒼쓩쓰쓱쓴쓸쓺쓿씀씁씌씐씔씜씨씩씬씰씸씹씻씽아악안앉않알앍앎앓암압앗았앙앝앞애액앤앨앰앱앳앴앵야약얀얄얇얌얍얏양얕얗얘얜얠얩어억언얹얻얼얽얾엄",6,"엌엎"],
["bf41","풞",10,"풪",14],
["bf61","풹",18,"퓍퓎퓏퓑퓒퓓퓕"],
["bf81","퓖",5,"퓝퓞퓠",7,"퓩퓪퓫퓭퓮퓯퓱",6,"퓹퓺퓼에엑엔엘엠엡엣엥여역엮연열엶엷염",5,"옅옆옇예옌옐옘옙옛옜오옥온올옭옮옰옳옴옵옷옹옻와왁완왈왐왑왓왔왕왜왝왠왬왯왱외왹왼욀욈욉욋욍요욕욘욜욤욥욧용우욱운울욹욺움웁웃웅워웍원월웜웝웠웡웨"],
["c041","퓾",5,"픅픆픇픉픊픋픍",6,"픖픘",5],
["c061","픞",25],
["c081","픸픹픺픻픾픿핁핂핃핅",6,"핎핐핒",5,"핚핛핝핞핟핡핢핣웩웬웰웸웹웽위윅윈윌윔윕윗윙유육윤율윰윱윳융윷으윽은을읊음읍읏응",7,"읜읠읨읫이익인일읽읾잃임입잇있잉잊잎자작잔잖잗잘잚잠잡잣잤장잦재잭잰잴잼잽잿쟀쟁쟈쟉쟌쟎쟐쟘쟝쟤쟨쟬저적전절젊"],
["c141","핤핦핧핪핬핮",5,"핶핷핹핺핻핽",6,"햆햊햋"],
["c161","햌햍햎햏햑",19,"햦햧"],
["c181","햨",31,"점접젓정젖제젝젠젤젬젭젯젱져젼졀졈졉졌졍졔조족존졸졺좀좁좃종좆좇좋좌좍좔좝좟좡좨좼좽죄죈죌죔죕죗죙죠죡죤죵주죽준줄줅줆줌줍줏중줘줬줴쥐쥑쥔쥘쥠쥡쥣쥬쥰쥴쥼즈즉즌즐즘즙즛증지직진짇질짊짐집짓"],
["c241","헊헋헍헎헏헑헓",4,"헚헜헞",5,"헦헧헩헪헫헭헮"],
["c261","헯",4,"헶헸헺",5,"혂혃혅혆혇혉",6,"혒"],
["c281","혖",5,"혝혞혟혡혢혣혥",7,"혮",9,"혺혻징짖짙짚짜짝짠짢짤짧짬짭짯짰짱째짹짼쨀쨈쨉쨋쨌쨍쨔쨘쨩쩌쩍쩐쩔쩜쩝쩟쩠쩡쩨쩽쪄쪘쪼쪽쫀쫄쫌쫍쫏쫑쫓쫘쫙쫠쫬쫴쬈쬐쬔쬘쬠쬡쭁쭈쭉쭌쭐쭘쭙쭝쭤쭸쭹쮜쮸쯔쯤쯧쯩찌찍찐찔찜찝찡찢찧차착찬찮찰참찹찻"],
["c341","혽혾혿홁홂홃홄홆홇홊홌홎홏홐홒홓홖홗홙홚홛홝",4],
["c361","홢",4,"홨홪",5,"홲홳홵",11],
["c381","횁횂횄횆",5,"횎횏횑횒횓횕",7,"횞횠횢",5,"횩횪찼창찾채책챈챌챔챕챗챘챙챠챤챦챨챰챵처척천철첨첩첫첬청체첵첸첼쳄쳅쳇쳉쳐쳔쳤쳬쳰촁초촉촌촐촘촙촛총촤촨촬촹최쵠쵤쵬쵭쵯쵱쵸춈추축춘출춤춥춧충춰췄췌췐취췬췰췸췹췻췽츄츈츌츔츙츠측츤츨츰츱츳층"],
["c441","횫횭횮횯횱",7,"횺횼",7,"훆훇훉훊훋"],
["c461","훍훎훏훐훒훓훕훖훘훚",5,"훡훢훣훥훦훧훩",4],
["c481","훮훯훱훲훳훴훶",5,"훾훿휁휂휃휅",11,"휒휓휔치칙친칟칠칡침칩칫칭카칵칸칼캄캅캇캉캐캑캔캘캠캡캣캤캥캬캭컁커컥컨컫컬컴컵컷컸컹케켁켄켈켐켑켓켕켜켠켤켬켭켯켰켱켸코콕콘콜콤콥콧콩콰콱콴콸쾀쾅쾌쾡쾨쾰쿄쿠쿡쿤쿨쿰쿱쿳쿵쿼퀀퀄퀑퀘퀭퀴퀵퀸퀼"],
["c541","휕휖휗휚휛휝휞휟휡",6,"휪휬휮",5,"휶휷휹"],
["c561","휺휻휽",6,"흅흆흈흊",5,"흒흓흕흚",4],
["c581","흟흢흤흦흧흨흪흫흭흮흯흱흲흳흵",6,"흾흿힀힂",5,"힊힋큄큅큇큉큐큔큘큠크큭큰클큼큽킁키킥킨킬킴킵킷킹타탁탄탈탉탐탑탓탔탕태택탠탤탬탭탯탰탱탸턍터턱턴털턺텀텁텃텄텅테텍텐텔템텝텟텡텨텬텼톄톈토톡톤톨톰톱톳통톺톼퇀퇘퇴퇸툇툉툐투툭툰툴툼툽툿퉁퉈퉜"],
["c641","힍힎힏힑",6,"힚힜힞",5],
["c6a1","퉤튀튁튄튈튐튑튕튜튠튤튬튱트특튼튿틀틂틈틉틋틔틘틜틤틥티틱틴틸팀팁팃팅파팍팎판팔팖팜팝팟팠팡팥패팩팬팰팸팹팻팼팽퍄퍅퍼퍽펀펄펌펍펏펐펑페펙펜펠펨펩펫펭펴편펼폄폅폈평폐폘폡폣포폭폰폴폼폽폿퐁"],
["c7a1","퐈퐝푀푄표푠푤푭푯푸푹푼푿풀풂품풉풋풍풔풩퓌퓐퓔퓜퓟퓨퓬퓰퓸퓻퓽프픈플픔픕픗피픽핀필핌핍핏핑하학한할핥함합핫항해핵핸핼햄햅햇했행햐향허헉헌헐헒험헙헛헝헤헥헨헬헴헵헷헹혀혁현혈혐협혓혔형혜혠"],
["c8a1","혤혭호혹혼홀홅홈홉홋홍홑화확환활홧황홰홱홴횃횅회획횐횔횝횟횡효횬횰횹횻후훅훈훌훑훔훗훙훠훤훨훰훵훼훽휀휄휑휘휙휜휠휨휩휫휭휴휵휸휼흄흇흉흐흑흔흖흗흘흙흠흡흣흥흩희흰흴흼흽힁히힉힌힐힘힙힛힝"],
["caa1","伽佳假價加可呵哥嘉嫁家暇架枷柯歌珂痂稼苛茄街袈訶賈跏軻迦駕刻却各恪慤殼珏脚覺角閣侃刊墾奸姦干幹懇揀杆柬桿澗癎看磵稈竿簡肝艮艱諫間乫喝曷渴碣竭葛褐蝎鞨勘坎堪嵌感憾戡敢柑橄減甘疳監瞰紺邯鑑鑒龕"],
["cba1","匣岬甲胛鉀閘剛堈姜岡崗康强彊慷江畺疆糠絳綱羌腔舡薑襁講鋼降鱇介价個凱塏愷愾慨改槪漑疥皆盖箇芥蓋豈鎧開喀客坑更粳羹醵倨去居巨拒据據擧渠炬祛距踞車遽鉅鋸乾件健巾建愆楗腱虔蹇鍵騫乞傑杰桀儉劍劒檢"],
["cca1","瞼鈐黔劫怯迲偈憩揭擊格檄激膈覡隔堅牽犬甄絹繭肩見譴遣鵑抉決潔結缺訣兼慊箝謙鉗鎌京俓倞傾儆勁勍卿坰境庚徑慶憬擎敬景暻更梗涇炅烱璟璥瓊痙硬磬竟競絅經耕耿脛莖警輕逕鏡頃頸驚鯨係啓堺契季屆悸戒桂械"],
["cda1","棨溪界癸磎稽系繫繼計誡谿階鷄古叩告呱固姑孤尻庫拷攷故敲暠枯槁沽痼皐睾稿羔考股膏苦苽菰藁蠱袴誥賈辜錮雇顧高鼓哭斛曲梏穀谷鵠困坤崑昆梱棍滾琨袞鯤汨滑骨供公共功孔工恐恭拱控攻珙空蚣貢鞏串寡戈果瓜"],
["cea1","科菓誇課跨過鍋顆廓槨藿郭串冠官寬慣棺款灌琯瓘管罐菅觀貫關館刮恝括适侊光匡壙廣曠洸炚狂珖筐胱鑛卦掛罫乖傀塊壞怪愧拐槐魁宏紘肱轟交僑咬喬嬌嶠巧攪敎校橋狡皎矯絞翹膠蕎蛟較轎郊餃驕鮫丘久九仇俱具勾"],
["cfa1","區口句咎嘔坵垢寇嶇廐懼拘救枸柩構歐毆毬求溝灸狗玖球瞿矩究絿耉臼舅舊苟衢謳購軀逑邱鉤銶駒驅鳩鷗龜國局菊鞠鞫麴君窘群裙軍郡堀屈掘窟宮弓穹窮芎躬倦券勸卷圈拳捲權淃眷厥獗蕨蹶闕机櫃潰詭軌饋句晷歸貴"],
["d0a1","鬼龜叫圭奎揆槻珪硅窺竅糾葵規赳逵閨勻均畇筠菌鈞龜橘克剋劇戟棘極隙僅劤勤懃斤根槿瑾筋芹菫覲謹近饉契今妗擒昑檎琴禁禽芩衾衿襟金錦伋及急扱汲級給亘兢矜肯企伎其冀嗜器圻基埼夔奇妓寄岐崎己幾忌技旗旣"],
["d1a1","朞期杞棋棄機欺氣汽沂淇玘琦琪璂璣畸畿碁磯祁祇祈祺箕紀綺羈耆耭肌記譏豈起錡錤飢饑騎騏驥麒緊佶吉拮桔金喫儺喇奈娜懦懶拏拿癩",5,"那樂",4,"諾酪駱亂卵暖欄煖爛蘭難鸞捏捺南嵐枏楠湳濫男藍襤拉"],
["d2a1","納臘蠟衲囊娘廊",4,"乃來內奈柰耐冷女年撚秊念恬拈捻寧寗努勞奴弩怒擄櫓爐瑙盧",5,"駑魯",10,"濃籠聾膿農惱牢磊腦賂雷尿壘",7,"嫩訥杻紐勒",5,"能菱陵尼泥匿溺多茶"],
["d3a1","丹亶但單團壇彖斷旦檀段湍短端簞緞蛋袒鄲鍛撻澾獺疸達啖坍憺擔曇淡湛潭澹痰聃膽蕁覃談譚錟沓畓答踏遝唐堂塘幢戇撞棠當糖螳黨代垈坮大對岱帶待戴擡玳臺袋貸隊黛宅德悳倒刀到圖堵塗導屠島嶋度徒悼挑掉搗桃"],
["d4a1","棹櫂淘渡滔濤燾盜睹禱稻萄覩賭跳蹈逃途道都鍍陶韜毒瀆牘犢獨督禿篤纛讀墩惇敦旽暾沌焞燉豚頓乭突仝冬凍動同憧東桐棟洞潼疼瞳童胴董銅兜斗杜枓痘竇荳讀豆逗頭屯臀芚遁遯鈍得嶝橙燈登等藤謄鄧騰喇懶拏癩羅"],
["d5a1","蘿螺裸邏樂洛烙珞絡落諾酪駱丹亂卵欄欒瀾爛蘭鸞剌辣嵐擥攬欖濫籃纜藍襤覽拉臘蠟廊朗浪狼琅瑯螂郞來崍徠萊冷掠略亮倆兩凉梁樑粮粱糧良諒輛量侶儷勵呂廬慮戾旅櫚濾礪藜蠣閭驢驪麗黎力曆歷瀝礫轢靂憐戀攣漣"],
["d6a1","煉璉練聯蓮輦連鍊冽列劣洌烈裂廉斂殮濂簾獵令伶囹寧岺嶺怜玲笭羚翎聆逞鈴零靈領齡例澧禮醴隷勞怒撈擄櫓潞瀘爐盧老蘆虜路輅露魯鷺鹵碌祿綠菉錄鹿麓論壟弄朧瀧瓏籠聾儡瀨牢磊賂賚賴雷了僚寮廖料燎療瞭聊蓼"],
["d7a1","遼鬧龍壘婁屢樓淚漏瘻累縷蔞褸鏤陋劉旒柳榴流溜瀏琉瑠留瘤硫謬類六戮陸侖倫崙淪綸輪律慄栗率隆勒肋凜凌楞稜綾菱陵俚利厘吏唎履悧李梨浬犁狸理璃異痢籬罹羸莉裏裡里釐離鯉吝潾燐璘藺躪隣鱗麟林淋琳臨霖砬"],
["d8a1","立笠粒摩瑪痲碼磨馬魔麻寞幕漠膜莫邈万卍娩巒彎慢挽晩曼滿漫灣瞞萬蔓蠻輓饅鰻唜抹末沫茉襪靺亡妄忘忙望網罔芒茫莽輞邙埋妹媒寐昧枚梅每煤罵買賣邁魅脈貊陌驀麥孟氓猛盲盟萌冪覓免冕勉棉沔眄眠綿緬面麵滅"],
["d9a1","蔑冥名命明暝椧溟皿瞑茗蓂螟酩銘鳴袂侮冒募姆帽慕摸摹暮某模母毛牟牡瑁眸矛耗芼茅謀謨貌木沐牧目睦穆鶩歿沒夢朦蒙卯墓妙廟描昴杳渺猫竗苗錨務巫憮懋戊拇撫无楙武毋無珷畝繆舞茂蕪誣貿霧鵡墨默們刎吻問文"],
["daa1","汶紊紋聞蚊門雯勿沕物味媚尾嵋彌微未梶楣渼湄眉米美薇謎迷靡黴岷悶愍憫敏旻旼民泯玟珉緡閔密蜜謐剝博拍搏撲朴樸泊珀璞箔粕縛膊舶薄迫雹駁伴半反叛拌搬攀斑槃泮潘班畔瘢盤盼磐磻礬絆般蟠返頒飯勃拔撥渤潑"],
["dba1","發跋醱鉢髮魃倣傍坊妨尨幇彷房放方旁昉枋榜滂磅紡肪膀舫芳蒡蚌訪謗邦防龐倍俳北培徘拜排杯湃焙盃背胚裴裵褙賠輩配陪伯佰帛柏栢白百魄幡樊煩燔番磻繁蕃藩飜伐筏罰閥凡帆梵氾汎泛犯範范法琺僻劈壁擘檗璧癖"],
["dca1","碧蘗闢霹便卞弁變辨辯邊別瞥鱉鼈丙倂兵屛幷昞昺柄棅炳甁病秉竝輧餠騈保堡報寶普步洑湺潽珤甫菩補褓譜輔伏僕匐卜宓復服福腹茯蔔複覆輹輻馥鰒本乶俸奉封峯峰捧棒烽熢琫縫蓬蜂逢鋒鳳不付俯傅剖副否咐埠夫婦"],
["dda1","孚孵富府復扶敷斧浮溥父符簿缶腐腑膚艀芙莩訃負賦賻赴趺部釜阜附駙鳧北分吩噴墳奔奮忿憤扮昐汾焚盆粉糞紛芬賁雰不佛弗彿拂崩朋棚硼繃鵬丕備匕匪卑妃婢庇悲憊扉批斐枇榧比毖毗毘沸泌琵痺砒碑秕秘粃緋翡肥"],
["dea1","脾臂菲蜚裨誹譬費鄙非飛鼻嚬嬪彬斌檳殯浜濱瀕牝玭貧賓頻憑氷聘騁乍事些仕伺似使俟僿史司唆嗣四士奢娑寫寺射巳師徙思捨斜斯柶査梭死沙泗渣瀉獅砂社祀祠私篩紗絲肆舍莎蓑蛇裟詐詞謝賜赦辭邪飼駟麝削數朔索"],
["dfa1","傘刪山散汕珊産疝算蒜酸霰乷撒殺煞薩三參杉森渗芟蔘衫揷澁鈒颯上傷像償商喪嘗孀尙峠常床庠廂想桑橡湘爽牀狀相祥箱翔裳觴詳象賞霜塞璽賽嗇塞穡索色牲生甥省笙墅壻嶼序庶徐恕抒捿敍暑曙書栖棲犀瑞筮絮緖署"],
["e0a1","胥舒薯西誓逝鋤黍鼠夕奭席惜昔晳析汐淅潟石碩蓆釋錫仙僊先善嬋宣扇敾旋渲煽琁瑄璇璿癬禪線繕羨腺膳船蘚蟬詵跣選銑鐥饍鮮卨屑楔泄洩渫舌薛褻設說雪齧剡暹殲纖蟾贍閃陝攝涉燮葉城姓宬性惺成星晟猩珹盛省筬"],
["e1a1","聖聲腥誠醒世勢歲洗稅笹細說貰召嘯塑宵小少巢所掃搔昭梳沼消溯瀟炤燒甦疏疎瘙笑篠簫素紹蔬蕭蘇訴逍遡邵銷韶騷俗屬束涑粟續謖贖速孫巽損蓀遜飡率宋悚松淞訟誦送頌刷殺灑碎鎖衰釗修受嗽囚垂壽嫂守岫峀帥愁"],
["e2a1","戍手授搜收數樹殊水洙漱燧狩獸琇璲瘦睡秀穗竪粹綏綬繡羞脩茱蒐蓚藪袖誰讐輸遂邃酬銖銹隋隧隨雖需須首髓鬚叔塾夙孰宿淑潚熟琡璹肅菽巡徇循恂旬栒楯橓殉洵淳珣盾瞬筍純脣舜荀蓴蕣詢諄醇錞順馴戌術述鉥崇崧"],
["e3a1","嵩瑟膝蝨濕拾習褶襲丞乘僧勝升承昇繩蠅陞侍匙嘶始媤尸屎屍市弑恃施是時枾柴猜矢示翅蒔蓍視試詩諡豕豺埴寔式息拭植殖湜熄篒蝕識軾食飾伸侁信呻娠宸愼新晨燼申神紳腎臣莘薪藎蜃訊身辛辰迅失室實悉審尋心沁"],
["e4a1","沈深瀋甚芯諶什十拾雙氏亞俄兒啞娥峨我牙芽莪蛾衙訝阿雅餓鴉鵝堊岳嶽幄惡愕握樂渥鄂鍔顎鰐齷安岸按晏案眼雁鞍顔鮟斡謁軋閼唵岩巖庵暗癌菴闇壓押狎鴨仰央怏昻殃秧鴦厓哀埃崖愛曖涯碍艾隘靄厄扼掖液縊腋額"],
["e5a1","櫻罌鶯鸚也倻冶夜惹揶椰爺耶若野弱掠略約若葯蒻藥躍亮佯兩凉壤孃恙揚攘敭暘梁楊樣洋瀁煬痒瘍禳穰糧羊良襄諒讓釀陽量養圄御於漁瘀禦語馭魚齬億憶抑檍臆偃堰彦焉言諺孼蘖俺儼嚴奄掩淹嶪業円予余勵呂女如廬"],
["e6a1","旅歟汝濾璵礖礪與艅茹輿轝閭餘驪麗黎亦力域役易曆歷疫繹譯轢逆驛嚥堧姸娟宴年延憐戀捐挻撚椽沇沿涎涓淵演漣烟然煙煉燃燕璉硏硯秊筵緣練縯聯衍軟輦蓮連鉛鍊鳶列劣咽悅涅烈熱裂說閱厭廉念捻染殮炎焰琰艶苒"],
["e7a1","簾閻髥鹽曄獵燁葉令囹塋寧嶺嶸影怜映暎楹榮永泳渶潁濚瀛瀯煐營獰玲瑛瑩瓔盈穎纓羚聆英詠迎鈴鍈零霙靈領乂倪例刈叡曳汭濊猊睿穢芮藝蘂禮裔詣譽豫醴銳隸霓預五伍俉傲午吾吳嗚塢墺奧娛寤悟惡懊敖旿晤梧汚澳"],
["e8a1","烏熬獒筽蜈誤鰲鼇屋沃獄玉鈺溫瑥瘟穩縕蘊兀壅擁瓮甕癰翁邕雍饔渦瓦窩窪臥蛙蝸訛婉完宛梡椀浣玩琓琬碗緩翫脘腕莞豌阮頑曰往旺枉汪王倭娃歪矮外嵬巍猥畏了僚僥凹堯夭妖姚寥寮尿嶢拗搖撓擾料曜樂橈燎燿瑤療"],
["e9a1","窈窯繇繞耀腰蓼蟯要謠遙遼邀饒慾欲浴縟褥辱俑傭冗勇埇墉容庸慂榕涌湧溶熔瑢用甬聳茸蓉踊鎔鏞龍于佑偶優又友右宇寓尤愚憂旴牛玗瑀盂祐禑禹紆羽芋藕虞迂遇郵釪隅雨雩勖彧旭昱栯煜稶郁頊云暈橒殞澐熉耘芸蕓"],
["eaa1","運隕雲韻蔚鬱亐熊雄元原員圓園垣媛嫄寃怨愿援沅洹湲源爰猿瑗苑袁轅遠阮院願鴛月越鉞位偉僞危圍委威尉慰暐渭爲瑋緯胃萎葦蔿蝟衛褘謂違韋魏乳侑儒兪劉唯喩孺宥幼幽庾悠惟愈愉揄攸有杻柔柚柳楡楢油洧流游溜"],
["eba1","濡猶猷琉瑜由留癒硫紐維臾萸裕誘諛諭踰蹂遊逾遺酉釉鍮類六堉戮毓肉育陸倫允奫尹崙淪潤玧胤贇輪鈗閏律慄栗率聿戎瀜絨融隆垠恩慇殷誾銀隱乙吟淫蔭陰音飮揖泣邑凝應膺鷹依倚儀宜意懿擬椅毅疑矣義艤薏蟻衣誼"],
["eca1","議醫二以伊利吏夷姨履已弛彛怡易李梨泥爾珥理異痍痢移罹而耳肄苡荑裏裡貽貳邇里離飴餌匿溺瀷益翊翌翼謚人仁刃印吝咽因姻寅引忍湮燐璘絪茵藺蚓認隣靭靷鱗麟一佚佾壹日溢逸鎰馹任壬妊姙恁林淋稔臨荏賃入卄"],
["eda1","立笠粒仍剩孕芿仔刺咨姉姿子字孜恣慈滋炙煮玆瓷疵磁紫者自茨蔗藉諮資雌作勺嚼斫昨灼炸爵綽芍酌雀鵲孱棧殘潺盞岑暫潛箴簪蠶雜丈仗匠場墻壯奬將帳庄張掌暲杖樟檣欌漿牆狀獐璋章粧腸臟臧莊葬蔣薔藏裝贓醬長"],
["eea1","障再哉在宰才材栽梓渽滓災縡裁財載齋齎爭箏諍錚佇低儲咀姐底抵杵楮樗沮渚狙猪疽箸紵苧菹著藷詛貯躇這邸雎齟勣吊嫡寂摘敵滴狄炙的積笛籍績翟荻謫賊赤跡蹟迪迹適鏑佃佺傳全典前剪塡塼奠專展廛悛戰栓殿氈澱"],
["efa1","煎琠田甸畑癲筌箋箭篆纏詮輾轉鈿銓錢鐫電顚顫餞切截折浙癤竊節絶占岾店漸点粘霑鮎點接摺蝶丁井亭停偵呈姃定幀庭廷征情挺政整旌晶晸柾楨檉正汀淀淨渟湞瀞炡玎珽町睛碇禎程穽精綎艇訂諪貞鄭酊釘鉦鋌錠霆靖"],
["f0a1","靜頂鼎制劑啼堤帝弟悌提梯濟祭第臍薺製諸蹄醍除際霽題齊俎兆凋助嘲弔彫措操早晁曺曹朝條棗槽漕潮照燥爪璪眺祖祚租稠窕粗糟組繰肇藻蚤詔調趙躁造遭釣阻雕鳥族簇足鏃存尊卒拙猝倧宗從悰慫棕淙琮種終綜縱腫"],
["f1a1","踪踵鍾鐘佐坐左座挫罪主住侏做姝胄呪周嗾奏宙州廚晝朱柱株注洲湊澍炷珠疇籌紂紬綢舟蛛註誅走躊輳週酎酒鑄駐竹粥俊儁准埈寯峻晙樽浚準濬焌畯竣蠢逡遵雋駿茁中仲衆重卽櫛楫汁葺增憎曾拯烝甑症繒蒸證贈之只"],
["f2a1","咫地址志持指摯支旨智枝枳止池沚漬知砥祉祗紙肢脂至芝芷蜘誌識贄趾遲直稙稷織職唇嗔塵振搢晉晋桭榛殄津溱珍瑨璡畛疹盡眞瞋秦縉縝臻蔯袗診賑軫辰進鎭陣陳震侄叱姪嫉帙桎瓆疾秩窒膣蛭質跌迭斟朕什執潗緝輯"],
["f3a1","鏶集徵懲澄且侘借叉嗟嵯差次此磋箚茶蹉車遮捉搾着窄錯鑿齪撰澯燦璨瓚竄簒纂粲纘讚贊鑽餐饌刹察擦札紮僭參塹慘慙懺斬站讒讖倉倡創唱娼廠彰愴敞昌昶暢槍滄漲猖瘡窓脹艙菖蒼債埰寀寨彩採砦綵菜蔡采釵冊柵策"],
["f4a1","責凄妻悽處倜刺剔尺慽戚拓擲斥滌瘠脊蹠陟隻仟千喘天川擅泉淺玔穿舛薦賤踐遷釧闡阡韆凸哲喆徹撤澈綴輟轍鐵僉尖沾添甛瞻簽籤詹諂堞妾帖捷牒疊睫諜貼輒廳晴淸聽菁請靑鯖切剃替涕滯締諦逮遞體初剿哨憔抄招梢"],
["f5a1","椒楚樵炒焦硝礁礎秒稍肖艸苕草蕉貂超酢醋醮促囑燭矗蜀觸寸忖村邨叢塚寵悤憁摠總聰蔥銃撮催崔最墜抽推椎楸樞湫皺秋芻萩諏趨追鄒酋醜錐錘鎚雛騶鰍丑畜祝竺筑築縮蓄蹙蹴軸逐春椿瑃出朮黜充忠沖蟲衝衷悴膵萃"],
["f6a1","贅取吹嘴娶就炊翠聚脆臭趣醉驟鷲側仄厠惻測層侈値嗤峙幟恥梔治淄熾痔痴癡稚穉緇緻置致蚩輜雉馳齒則勅飭親七柒漆侵寢枕沈浸琛砧針鍼蟄秤稱快他咤唾墮妥惰打拖朶楕舵陀馱駝倬卓啄坼度托拓擢晫柝濁濯琢琸託"],
["f7a1","鐸呑嘆坦彈憚歎灘炭綻誕奪脫探眈耽貪塔搭榻宕帑湯糖蕩兌台太怠態殆汰泰笞胎苔跆邰颱宅擇澤撑攄兎吐土討慟桶洞痛筒統通堆槌腿褪退頹偸套妬投透鬪慝特闖坡婆巴把播擺杷波派爬琶破罷芭跛頗判坂板版瓣販辦鈑"],
["f8a1","阪八叭捌佩唄悖敗沛浿牌狽稗覇貝彭澎烹膨愎便偏扁片篇編翩遍鞭騙貶坪平枰萍評吠嬖幣廢弊斃肺蔽閉陛佈包匍匏咆哺圃布怖抛抱捕暴泡浦疱砲胞脯苞葡蒲袍褒逋鋪飽鮑幅暴曝瀑爆輻俵剽彪慓杓標漂瓢票表豹飇飄驃"],
["f9a1","品稟楓諷豊風馮彼披疲皮被避陂匹弼必泌珌畢疋筆苾馝乏逼下何厦夏廈昰河瑕荷蝦賀遐霞鰕壑學虐謔鶴寒恨悍旱汗漢澣瀚罕翰閑閒限韓割轄函含咸啣喊檻涵緘艦銜陷鹹合哈盒蛤閤闔陜亢伉姮嫦巷恒抗杭桁沆港缸肛航"],
["faa1","行降項亥偕咳垓奚孩害懈楷海瀣蟹解該諧邂駭骸劾核倖幸杏荇行享向嚮珦鄕響餉饗香噓墟虛許憲櫶獻軒歇險驗奕爀赫革俔峴弦懸晛泫炫玄玹現眩睍絃絢縣舷衒見賢鉉顯孑穴血頁嫌俠協夾峽挾浹狹脅脇莢鋏頰亨兄刑型"],
["fba1","形泂滎瀅灐炯熒珩瑩荊螢衡逈邢鎣馨兮彗惠慧暳蕙蹊醯鞋乎互呼壕壺好岵弧戶扈昊晧毫浩淏湖滸澔濠濩灝狐琥瑚瓠皓祜糊縞胡芦葫蒿虎號蝴護豪鎬頀顥惑或酷婚昏混渾琿魂忽惚笏哄弘汞泓洪烘紅虹訌鴻化和嬅樺火畵"],
["fca1","禍禾花華話譁貨靴廓擴攫確碻穫丸喚奐宦幻患換歡晥桓渙煥環紈還驩鰥活滑猾豁闊凰幌徨恍惶愰慌晃晄榥況湟滉潢煌璜皇篁簧荒蝗遑隍黃匯回廻徊恢悔懷晦會檜淮澮灰獪繪膾茴蛔誨賄劃獲宖橫鐄哮嚆孝效斅曉梟涍淆"],
["fda1","爻肴酵驍侯候厚后吼喉嗅帿後朽煦珝逅勛勳塤壎焄熏燻薰訓暈薨喧暄煊萱卉喙毁彙徽揮暉煇諱輝麾休携烋畦虧恤譎鷸兇凶匈洶胸黑昕欣炘痕吃屹紇訖欠欽歆吸恰洽翕興僖凞喜噫囍姬嬉希憙憘戱晞曦熙熹熺犧禧稀羲詰"]
]
;

var cp949$1 = /*#__PURE__*/Object.freeze({
            default: cp949
});

var cp950 = [
["0","\u0000",127],
["a140","　，、。．‧；：？！︰…‥﹐﹑﹒·﹔﹕﹖﹗｜–︱—︳╴︴﹏（）︵︶｛｝︷︸〔〕︹︺【】︻︼《》︽︾〈〉︿﹀「」﹁﹂『』﹃﹄﹙﹚"],
["a1a1","﹛﹜﹝﹞‘’“”〝〞‵′＃＆＊※§〃○●△▲◎☆★◇◆□■▽▼㊣℅¯￣＿ˍ﹉﹊﹍﹎﹋﹌﹟﹠﹡＋－×÷±√＜＞＝≦≧≠∞≒≡﹢",4,"～∩∪⊥∠∟⊿㏒㏑∫∮∵∴♀♂⊕⊙↑↓←→↖↗↙↘∥∣／"],
["a240","＼∕﹨＄￥〒￠￡％＠℃℉﹩﹪﹫㏕㎜㎝㎞㏎㎡㎎㎏㏄°兙兛兞兝兡兣嗧瓩糎▁",7,"▏▎▍▌▋▊▉┼┴┬┤├▔─│▕┌┐└┘╭"],
["a2a1","╮╰╯═╞╪╡◢◣◥◤╱╲╳０",9,"Ⅰ",9,"〡",8,"十卄卅Ａ",25,"ａ",21],
["a340","ｗｘｙｚΑ",16,"Σ",6,"α",16,"σ",6,"ㄅ",10],
["a3a1","ㄐ",25,"˙ˉˊˇˋ"],
["a3e1","€"],
["a440","一乙丁七乃九了二人儿入八几刀刁力匕十卜又三下丈上丫丸凡久么也乞于亡兀刃勺千叉口土士夕大女子孑孓寸小尢尸山川工己已巳巾干廾弋弓才"],
["a4a1","丑丐不中丰丹之尹予云井互五亢仁什仃仆仇仍今介仄元允內六兮公冗凶分切刈勻勾勿化匹午升卅卞厄友及反壬天夫太夭孔少尤尺屯巴幻廿弔引心戈戶手扎支文斗斤方日曰月木欠止歹毋比毛氏水火爪父爻片牙牛犬王丙"],
["a540","世丕且丘主乍乏乎以付仔仕他仗代令仙仞充兄冉冊冬凹出凸刊加功包匆北匝仟半卉卡占卯卮去可古右召叮叩叨叼司叵叫另只史叱台句叭叻四囚外"],
["a5a1","央失奴奶孕它尼巨巧左市布平幼弁弘弗必戊打扔扒扑斥旦朮本未末札正母民氐永汁汀氾犯玄玉瓜瓦甘生用甩田由甲申疋白皮皿目矛矢石示禾穴立丞丟乒乓乩亙交亦亥仿伉伙伊伕伍伐休伏仲件任仰仳份企伋光兇兆先全"],
["a640","共再冰列刑划刎刖劣匈匡匠印危吉吏同吊吐吁吋各向名合吃后吆吒因回囝圳地在圭圬圯圩夙多夷夸妄奸妃好她如妁字存宇守宅安寺尖屹州帆并年"],
["a6a1","式弛忙忖戎戌戍成扣扛托收早旨旬旭曲曳有朽朴朱朵次此死氖汝汗汙江池汐汕污汛汍汎灰牟牝百竹米糸缶羊羽老考而耒耳聿肉肋肌臣自至臼舌舛舟艮色艾虫血行衣西阡串亨位住佇佗佞伴佛何估佐佑伽伺伸佃佔似但佣"],
["a740","作你伯低伶余佝佈佚兌克免兵冶冷別判利刪刨劫助努劬匣即卵吝吭吞吾否呎吧呆呃吳呈呂君吩告吹吻吸吮吵吶吠吼呀吱含吟听囪困囤囫坊坑址坍"],
["a7a1","均坎圾坐坏圻壯夾妝妒妨妞妣妙妖妍妤妓妊妥孝孜孚孛完宋宏尬局屁尿尾岐岑岔岌巫希序庇床廷弄弟彤形彷役忘忌志忍忱快忸忪戒我抄抗抖技扶抉扭把扼找批扳抒扯折扮投抓抑抆改攻攸旱更束李杏材村杜杖杞杉杆杠"],
["a840","杓杗步每求汞沙沁沈沉沅沛汪決沐汰沌汨沖沒汽沃汲汾汴沆汶沍沔沘沂灶灼災灸牢牡牠狄狂玖甬甫男甸皂盯矣私秀禿究系罕肖肓肝肘肛肚育良芒"],
["a8a1","芋芍見角言谷豆豕貝赤走足身車辛辰迂迆迅迄巡邑邢邪邦那酉釆里防阮阱阪阬並乖乳事些亞享京佯依侍佳使佬供例來侃佰併侈佩佻侖佾侏侑佺兔兒兕兩具其典冽函刻券刷刺到刮制剁劾劻卒協卓卑卦卷卸卹取叔受味呵"],
["a940","咖呸咕咀呻呷咄咒咆呼咐呱呶和咚呢周咋命咎固垃坷坪坩坡坦坤坼夜奉奇奈奄奔妾妻委妹妮姑姆姐姍始姓姊妯妳姒姅孟孤季宗定官宜宙宛尚屈居"],
["a9a1","屆岷岡岸岩岫岱岳帘帚帖帕帛帑幸庚店府底庖延弦弧弩往征彿彼忝忠忽念忿怏怔怯怵怖怪怕怡性怩怫怛或戕房戾所承拉拌拄抿拂抹拒招披拓拔拋拈抨抽押拐拙拇拍抵拚抱拘拖拗拆抬拎放斧於旺昔易昌昆昂明昀昏昕昊"],
["aa40","昇服朋杭枋枕東果杳杷枇枝林杯杰板枉松析杵枚枓杼杪杲欣武歧歿氓氛泣注泳沱泌泥河沽沾沼波沫法泓沸泄油況沮泗泅泱沿治泡泛泊沬泯泜泖泠"],
["aaa1","炕炎炒炊炙爬爭爸版牧物狀狎狙狗狐玩玨玟玫玥甽疝疙疚的盂盲直知矽社祀祁秉秈空穹竺糾罔羌羋者肺肥肢肱股肫肩肴肪肯臥臾舍芳芝芙芭芽芟芹花芬芥芯芸芣芰芾芷虎虱初表軋迎返近邵邸邱邶采金長門阜陀阿阻附"],
["ab40","陂隹雨青非亟亭亮信侵侯便俠俑俏保促侶俘俟俊俗侮俐俄係俚俎俞侷兗冒冑冠剎剃削前剌剋則勇勉勃勁匍南卻厚叛咬哀咨哎哉咸咦咳哇哂咽咪品"],
["aba1","哄哈咯咫咱咻咩咧咿囿垂型垠垣垢城垮垓奕契奏奎奐姜姘姿姣姨娃姥姪姚姦威姻孩宣宦室客宥封屎屏屍屋峙峒巷帝帥帟幽庠度建弈弭彥很待徊律徇後徉怒思怠急怎怨恍恰恨恢恆恃恬恫恪恤扁拜挖按拼拭持拮拽指拱拷"],
["ac40","拯括拾拴挑挂政故斫施既春昭映昧是星昨昱昤曷柿染柱柔某柬架枯柵柩柯柄柑枴柚查枸柏柞柳枰柙柢柝柒歪殃殆段毒毗氟泉洋洲洪流津洌洱洞洗"],
["aca1","活洽派洶洛泵洹洧洸洩洮洵洎洫炫為炳炬炯炭炸炮炤爰牲牯牴狩狠狡玷珊玻玲珍珀玳甚甭畏界畎畋疫疤疥疢疣癸皆皇皈盈盆盃盅省盹相眉看盾盼眇矜砂研砌砍祆祉祈祇禹禺科秒秋穿突竿竽籽紂紅紀紉紇約紆缸美羿耄"],
["ad40","耐耍耑耶胖胥胚胃胄背胡胛胎胞胤胝致舢苧范茅苣苛苦茄若茂茉苒苗英茁苜苔苑苞苓苟苯茆虐虹虻虺衍衫要觔計訂訃貞負赴赳趴軍軌述迦迢迪迥"],
["ada1","迭迫迤迨郊郎郁郃酋酊重閂限陋陌降面革韋韭音頁風飛食首香乘亳倌倍倣俯倦倥俸倩倖倆值借倚倒們俺倀倔倨俱倡個候倘俳修倭倪俾倫倉兼冤冥冢凍凌准凋剖剜剔剛剝匪卿原厝叟哨唐唁唷哼哥哲唆哺唔哩哭員唉哮哪"],
["ae40","哦唧唇哽唏圃圄埂埔埋埃堉夏套奘奚娑娘娜娟娛娓姬娠娣娩娥娌娉孫屘宰害家宴宮宵容宸射屑展屐峭峽峻峪峨峰島崁峴差席師庫庭座弱徒徑徐恙"],
["aea1","恣恥恐恕恭恩息悄悟悚悍悔悌悅悖扇拳挈拿捎挾振捕捂捆捏捉挺捐挽挪挫挨捍捌效敉料旁旅時晉晏晃晒晌晅晁書朔朕朗校核案框桓根桂桔栩梳栗桌桑栽柴桐桀格桃株桅栓栘桁殊殉殷氣氧氨氦氤泰浪涕消涇浦浸海浙涓"],
["af40","浬涉浮浚浴浩涌涊浹涅浥涔烊烘烤烙烈烏爹特狼狹狽狸狷玆班琉珮珠珪珞畔畝畜畚留疾病症疲疳疽疼疹痂疸皋皰益盍盎眩真眠眨矩砰砧砸砝破砷"],
["afa1","砥砭砠砟砲祕祐祠祟祖神祝祗祚秤秣秧租秦秩秘窄窈站笆笑粉紡紗紋紊素索純紐紕級紜納紙紛缺罟羔翅翁耆耘耕耙耗耽耿胱脂胰脅胭胴脆胸胳脈能脊胼胯臭臬舀舐航舫舨般芻茫荒荔荊茸荐草茵茴荏茲茹茶茗荀茱茨荃"],
["b040","虔蚊蚪蚓蚤蚩蚌蚣蚜衰衷袁袂衽衹記訐討訌訕訊託訓訖訏訑豈豺豹財貢起躬軒軔軏辱送逆迷退迺迴逃追逅迸邕郡郝郢酒配酌釘針釗釜釙閃院陣陡"],
["b0a1","陛陝除陘陞隻飢馬骨高鬥鬲鬼乾偺偽停假偃偌做偉健偶偎偕偵側偷偏倏偯偭兜冕凰剪副勒務勘動匐匏匙匿區匾參曼商啪啦啄啞啡啃啊唱啖問啕唯啤唸售啜唬啣唳啁啗圈國圉域堅堊堆埠埤基堂堵執培夠奢娶婁婉婦婪婀"],
["b140","娼婢婚婆婊孰寇寅寄寂宿密尉專將屠屜屝崇崆崎崛崖崢崑崩崔崙崤崧崗巢常帶帳帷康庸庶庵庾張強彗彬彩彫得徙從徘御徠徜恿患悉悠您惋悴惦悽"],
["b1a1","情悻悵惜悼惘惕惆惟悸惚惇戚戛扈掠控捲掖探接捷捧掘措捱掩掉掃掛捫推掄授掙採掬排掏掀捻捩捨捺敝敖救教敗啟敏敘敕敔斜斛斬族旋旌旎晝晚晤晨晦晞曹勗望梁梯梢梓梵桿桶梱梧梗械梃棄梭梆梅梔條梨梟梡梂欲殺"],
["b240","毫毬氫涎涼淳淙液淡淌淤添淺清淇淋涯淑涮淞淹涸混淵淅淒渚涵淚淫淘淪深淮淨淆淄涪淬涿淦烹焉焊烽烯爽牽犁猜猛猖猓猙率琅琊球理現琍瓠瓶"],
["b2a1","瓷甜產略畦畢異疏痔痕疵痊痍皎盔盒盛眷眾眼眶眸眺硫硃硎祥票祭移窒窕笠笨笛第符笙笞笮粒粗粕絆絃統紮紹紼絀細紳組累終紲紱缽羞羚翌翎習耜聊聆脯脖脣脫脩脰脤舂舵舷舶船莎莞莘荸莢莖莽莫莒莊莓莉莠荷荻荼"],
["b340","莆莧處彪蛇蛀蚶蛄蚵蛆蛋蚱蚯蛉術袞袈被袒袖袍袋覓規訪訝訣訥許設訟訛訢豉豚販責貫貨貪貧赧赦趾趺軛軟這逍通逗連速逝逐逕逞造透逢逖逛途"],
["b3a1","部郭都酗野釵釦釣釧釭釩閉陪陵陳陸陰陴陶陷陬雀雪雩章竟頂頃魚鳥鹵鹿麥麻傢傍傅備傑傀傖傘傚最凱割剴創剩勞勝勛博厥啻喀喧啼喊喝喘喂喜喪喔喇喋喃喳單喟唾喲喚喻喬喱啾喉喫喙圍堯堪場堤堰報堡堝堠壹壺奠"],
["b440","婷媚婿媒媛媧孳孱寒富寓寐尊尋就嵌嵐崴嵇巽幅帽幀幃幾廊廁廂廄弼彭復循徨惑惡悲悶惠愜愣惺愕惰惻惴慨惱愎惶愉愀愒戟扉掣掌描揀揩揉揆揍"],
["b4a1","插揣提握揖揭揮捶援揪換摒揚揹敞敦敢散斑斐斯普晰晴晶景暑智晾晷曾替期朝棺棕棠棘棗椅棟棵森棧棹棒棲棣棋棍植椒椎棉棚楮棻款欺欽殘殖殼毯氮氯氬港游湔渡渲湧湊渠渥渣減湛湘渤湖湮渭渦湯渴湍渺測湃渝渾滋"],
["b540","溉渙湎湣湄湲湩湟焙焚焦焰無然煮焜牌犄犀猶猥猴猩琺琪琳琢琥琵琶琴琯琛琦琨甥甦畫番痢痛痣痙痘痞痠登發皖皓皴盜睏短硝硬硯稍稈程稅稀窘"],
["b5a1","窗窖童竣等策筆筐筒答筍筋筏筑粟粥絞結絨絕紫絮絲絡給絢絰絳善翔翕耋聒肅腕腔腋腑腎脹腆脾腌腓腴舒舜菩萃菸萍菠菅萋菁華菱菴著萊菰萌菌菽菲菊萸萎萄菜萇菔菟虛蛟蛙蛭蛔蛛蛤蛐蛞街裁裂袱覃視註詠評詞証詁"],
["b640","詔詛詐詆訴診訶詖象貂貯貼貳貽賁費賀貴買貶貿貸越超趁跎距跋跚跑跌跛跆軻軸軼辜逮逵週逸進逶鄂郵鄉郾酣酥量鈔鈕鈣鈉鈞鈍鈐鈇鈑閔閏開閑"],
["b6a1","間閒閎隊階隋陽隅隆隍陲隄雁雅雄集雇雯雲韌項順須飧飪飯飩飲飭馮馭黃黍黑亂傭債傲傳僅傾催傷傻傯僇剿剷剽募勦勤勢勣匯嗟嗨嗓嗦嗎嗜嗇嗑嗣嗤嗯嗚嗡嗅嗆嗥嗉園圓塞塑塘塗塚塔填塌塭塊塢塒塋奧嫁嫉嫌媾媽媼"],
["b740","媳嫂媲嵩嵯幌幹廉廈弒彙徬微愚意慈感想愛惹愁愈慎慌慄慍愾愴愧愍愆愷戡戢搓搾搞搪搭搽搬搏搜搔損搶搖搗搆敬斟新暗暉暇暈暖暄暘暍會榔業"],
["b7a1","楚楷楠楔極椰概楊楨楫楞楓楹榆楝楣楛歇歲毀殿毓毽溢溯滓溶滂源溝滇滅溥溘溼溺溫滑準溜滄滔溪溧溴煎煙煩煤煉照煜煬煦煌煥煞煆煨煖爺牒猷獅猿猾瑯瑚瑕瑟瑞瑁琿瑙瑛瑜當畸瘀痰瘁痲痱痺痿痴痳盞盟睛睫睦睞督"],
["b840","睹睪睬睜睥睨睢矮碎碰碗碘碌碉硼碑碓硿祺祿禁萬禽稜稚稠稔稟稞窟窠筷節筠筮筧粱粳粵經絹綑綁綏絛置罩罪署義羨群聖聘肆肄腱腰腸腥腮腳腫"],
["b8a1","腹腺腦舅艇蒂葷落萱葵葦葫葉葬葛萼萵葡董葩葭葆虞虜號蛹蜓蜈蜇蜀蛾蛻蜂蜃蜆蜊衙裟裔裙補裘裝裡裊裕裒覜解詫該詳試詩詰誇詼詣誠話誅詭詢詮詬詹詻訾詨豢貊貉賊資賈賄貲賃賂賅跡跟跨路跳跺跪跤跦躲較載軾輊"],
["b940","辟農運遊道遂達逼違遐遇遏過遍遑逾遁鄒鄗酬酪酩釉鈷鉗鈸鈽鉀鈾鉛鉋鉤鉑鈴鉉鉍鉅鈹鈿鉚閘隘隔隕雍雋雉雊雷電雹零靖靴靶預頑頓頊頒頌飼飴"],
["b9a1","飽飾馳馱馴髡鳩麂鼎鼓鼠僧僮僥僖僭僚僕像僑僱僎僩兢凳劃劂匱厭嗾嘀嘛嘗嗽嘔嘆嘉嘍嘎嗷嘖嘟嘈嘐嗶團圖塵塾境墓墊塹墅塽壽夥夢夤奪奩嫡嫦嫩嫗嫖嫘嫣孵寞寧寡寥實寨寢寤察對屢嶄嶇幛幣幕幗幔廓廖弊彆彰徹慇"],
["ba40","愿態慷慢慣慟慚慘慵截撇摘摔撤摸摟摺摑摧搴摭摻敲斡旗旖暢暨暝榜榨榕槁榮槓構榛榷榻榫榴槐槍榭槌榦槃榣歉歌氳漳演滾漓滴漩漾漠漬漏漂漢"],
["baa1","滿滯漆漱漸漲漣漕漫漯澈漪滬漁滲滌滷熔熙煽熊熄熒爾犒犖獄獐瑤瑣瑪瑰瑭甄疑瘧瘍瘋瘉瘓盡監瞄睽睿睡磁碟碧碳碩碣禎福禍種稱窪窩竭端管箕箋筵算箝箔箏箸箇箄粹粽精綻綰綜綽綾綠緊綴網綱綺綢綿綵綸維緒緇綬"],
["bb40","罰翠翡翟聞聚肇腐膀膏膈膊腿膂臧臺與舔舞艋蓉蒿蓆蓄蒙蒞蒲蒜蓋蒸蓀蓓蒐蒼蓑蓊蜿蜜蜻蜢蜥蜴蜘蝕蜷蜩裳褂裴裹裸製裨褚裯誦誌語誣認誡誓誤"],
["bba1","說誥誨誘誑誚誧豪貍貌賓賑賒赫趙趕跼輔輒輕輓辣遠遘遜遣遙遞遢遝遛鄙鄘鄞酵酸酷酴鉸銀銅銘銖鉻銓銜銨鉼銑閡閨閩閣閥閤隙障際雌雒需靼鞅韶頗領颯颱餃餅餌餉駁骯骰髦魁魂鳴鳶鳳麼鼻齊億儀僻僵價儂儈儉儅凜"],
["bc40","劇劈劉劍劊勰厲嘮嘻嘹嘲嘿嘴嘩噓噎噗噴嘶嘯嘰墀墟增墳墜墮墩墦奭嬉嫻嬋嫵嬌嬈寮寬審寫層履嶝嶔幢幟幡廢廚廟廝廣廠彈影德徵慶慧慮慝慕憂"],
["bca1","慼慰慫慾憧憐憫憎憬憚憤憔憮戮摩摯摹撞撲撈撐撰撥撓撕撩撒撮播撫撚撬撙撢撳敵敷數暮暫暴暱樣樟槨樁樞標槽模樓樊槳樂樅槭樑歐歎殤毅毆漿潼澄潑潦潔澆潭潛潸潮澎潺潰潤澗潘滕潯潠潟熟熬熱熨牖犛獎獗瑩璋璃"],
["bd40","瑾璀畿瘠瘩瘟瘤瘦瘡瘢皚皺盤瞎瞇瞌瞑瞋磋磅確磊碾磕碼磐稿稼穀稽稷稻窯窮箭箱範箴篆篇篁箠篌糊締練緯緻緘緬緝編緣線緞緩綞緙緲緹罵罷羯"],
["bda1","翩耦膛膜膝膠膚膘蔗蔽蔚蓮蔬蔭蔓蔑蔣蔡蔔蓬蔥蓿蔆螂蝴蝶蝠蝦蝸蝨蝙蝗蝌蝓衛衝褐複褒褓褕褊誼諒談諄誕請諸課諉諂調誰論諍誶誹諛豌豎豬賠賞賦賤賬賭賢賣賜質賡赭趟趣踫踐踝踢踏踩踟踡踞躺輝輛輟輩輦輪輜輞"],
["be40","輥適遮遨遭遷鄰鄭鄧鄱醇醉醋醃鋅銻銷鋪銬鋤鋁銳銼鋒鋇鋰銲閭閱霄霆震霉靠鞍鞋鞏頡頫頜颳養餓餒餘駝駐駟駛駑駕駒駙骷髮髯鬧魅魄魷魯鴆鴉"],
["bea1","鴃麩麾黎墨齒儒儘儔儐儕冀冪凝劑劓勳噙噫噹噩噤噸噪器噥噱噯噬噢噶壁墾壇壅奮嬝嬴學寰導彊憲憑憩憊懍憶憾懊懈戰擅擁擋撻撼據擄擇擂操撿擒擔撾整曆曉暹曄曇暸樽樸樺橙橫橘樹橄橢橡橋橇樵機橈歙歷氅濂澱澡"],
["bf40","濃澤濁澧澳激澹澶澦澠澴熾燉燐燒燈燕熹燎燙燜燃燄獨璜璣璘璟璞瓢甌甍瘴瘸瘺盧盥瞠瞞瞟瞥磨磚磬磧禦積穎穆穌穋窺篙簑築篤篛篡篩篦糕糖縊"],
["bfa1","縑縈縛縣縞縝縉縐罹羲翰翱翮耨膳膩膨臻興艘艙蕊蕙蕈蕨蕩蕃蕉蕭蕪蕞螃螟螞螢融衡褪褲褥褫褡親覦諦諺諫諱謀諜諧諮諾謁謂諷諭諳諶諼豫豭貓賴蹄踱踴蹂踹踵輻輯輸輳辨辦遵遴選遲遼遺鄴醒錠錶鋸錳錯錢鋼錫錄錚"],
["c040","錐錦錡錕錮錙閻隧隨險雕霎霑霖霍霓霏靛靜靦鞘頰頸頻頷頭頹頤餐館餞餛餡餚駭駢駱骸骼髻髭鬨鮑鴕鴣鴦鴨鴒鴛默黔龍龜優償儡儲勵嚎嚀嚐嚅嚇"],
["c0a1","嚏壕壓壑壎嬰嬪嬤孺尷屨嶼嶺嶽嶸幫彌徽應懂懇懦懋戲戴擎擊擘擠擰擦擬擱擢擭斂斃曙曖檀檔檄檢檜櫛檣橾檗檐檠歜殮毚氈濘濱濟濠濛濤濫濯澀濬濡濩濕濮濰燧營燮燦燥燭燬燴燠爵牆獰獲璩環璦璨癆療癌盪瞳瞪瞰瞬"],
["c140","瞧瞭矯磷磺磴磯礁禧禪穗窿簇簍篾篷簌篠糠糜糞糢糟糙糝縮績繆縷縲繃縫總縱繅繁縴縹繈縵縿縯罄翳翼聱聲聰聯聳臆臃膺臂臀膿膽臉膾臨舉艱薪"],
["c1a1","薄蕾薜薑薔薯薛薇薨薊虧蟀蟑螳蟒蟆螫螻螺蟈蟋褻褶襄褸褽覬謎謗謙講謊謠謝謄謐豁谿豳賺賽購賸賻趨蹉蹋蹈蹊轄輾轂轅輿避遽還邁邂邀鄹醣醞醜鍍鎂錨鍵鍊鍥鍋錘鍾鍬鍛鍰鍚鍔闊闋闌闈闆隱隸雖霜霞鞠韓顆颶餵騁"],
["c240","駿鮮鮫鮪鮭鴻鴿麋黏點黜黝黛鼾齋叢嚕嚮壙壘嬸彝懣戳擴擲擾攆擺擻擷斷曜朦檳檬櫃檻檸櫂檮檯歟歸殯瀉瀋濾瀆濺瀑瀏燻燼燾燸獷獵璧璿甕癖癘"],
["c2a1","癒瞽瞿瞻瞼礎禮穡穢穠竄竅簫簧簪簞簣簡糧織繕繞繚繡繒繙罈翹翻職聶臍臏舊藏薩藍藐藉薰薺薹薦蟯蟬蟲蟠覆覲觴謨謹謬謫豐贅蹙蹣蹦蹤蹟蹕軀轉轍邇邃邈醫醬釐鎔鎊鎖鎢鎳鎮鎬鎰鎘鎚鎗闔闖闐闕離雜雙雛雞霤鞣鞦"],
["c340","鞭韹額顏題顎顓颺餾餿餽餮馥騎髁鬃鬆魏魎魍鯊鯉鯽鯈鯀鵑鵝鵠黠鼕鼬儳嚥壞壟壢寵龐廬懲懷懶懵攀攏曠曝櫥櫝櫚櫓瀛瀟瀨瀚瀝瀕瀘爆爍牘犢獸"],
["c3a1","獺璽瓊瓣疇疆癟癡矇礙禱穫穩簾簿簸簽簷籀繫繭繹繩繪羅繳羶羹羸臘藩藝藪藕藤藥藷蟻蠅蠍蟹蟾襠襟襖襞譁譜識證譚譎譏譆譙贈贊蹼蹲躇蹶蹬蹺蹴轔轎辭邊邋醱醮鏡鏑鏟鏃鏈鏜鏝鏖鏢鏍鏘鏤鏗鏨關隴難霪霧靡韜韻類"],
["c440","願顛颼饅饉騖騙鬍鯨鯧鯖鯛鶉鵡鵲鵪鵬麒麗麓麴勸嚨嚷嚶嚴嚼壤孀孃孽寶巉懸懺攘攔攙曦朧櫬瀾瀰瀲爐獻瓏癢癥礦礪礬礫竇競籌籃籍糯糰辮繽繼"],
["c4a1","纂罌耀臚艦藻藹蘑藺蘆蘋蘇蘊蠔蠕襤覺觸議譬警譯譟譫贏贍躉躁躅躂醴釋鐘鐃鏽闡霰飄饒饑馨騫騰騷騵鰓鰍鹹麵黨鼯齟齣齡儷儸囁囀囂夔屬巍懼懾攝攜斕曩櫻欄櫺殲灌爛犧瓖瓔癩矓籐纏續羼蘗蘭蘚蠣蠢蠡蠟襪襬覽譴"],
["c540","護譽贓躊躍躋轟辯醺鐮鐳鐵鐺鐸鐲鐫闢霸霹露響顧顥饗驅驃驀騾髏魔魑鰭鰥鶯鶴鷂鶸麝黯鼙齜齦齧儼儻囈囊囉孿巔巒彎懿攤權歡灑灘玀瓤疊癮癬"],
["c5a1","禳籠籟聾聽臟襲襯觼讀贖贗躑躓轡酈鑄鑑鑒霽霾韃韁顫饕驕驍髒鬚鱉鰱鰾鰻鷓鷗鼴齬齪龔囌巖戀攣攫攪曬欐瓚竊籤籣籥纓纖纔臢蘸蘿蠱變邐邏鑣鑠鑤靨顯饜驚驛驗髓體髑鱔鱗鱖鷥麟黴囑壩攬灞癱癲矗罐羈蠶蠹衢讓讒"],
["c640","讖艷贛釀鑪靂靈靄韆顰驟鬢魘鱟鷹鷺鹼鹽鼇齷齲廳欖灣籬籮蠻觀躡釁鑲鑰顱饞髖鬣黌灤矚讚鑷韉驢驥纜讜躪釅鑽鑾鑼鱷鱸黷豔鑿鸚爨驪鬱鸛鸞籲"],
["c940","乂乜凵匚厂万丌乇亍囗兀屮彳丏冇与丮亓仂仉仈冘勼卬厹圠夃夬尐巿旡殳毌气爿丱丼仨仜仩仡仝仚刌匜卌圢圣夗夯宁宄尒尻屴屳帄庀庂忉戉扐氕"],
["c9a1","氶汃氿氻犮犰玊禸肊阞伎优伬仵伔仱伀价伈伝伂伅伢伓伄仴伒冱刓刉刐劦匢匟卍厊吇囡囟圮圪圴夼妀奼妅奻奾奷奿孖尕尥屼屺屻屾巟幵庄异弚彴忕忔忏扜扞扤扡扦扢扙扠扚扥旯旮朾朹朸朻机朿朼朳氘汆汒汜汏汊汔汋"],
["ca40","汌灱牞犴犵玎甪癿穵网艸艼芀艽艿虍襾邙邗邘邛邔阢阤阠阣佖伻佢佉体佤伾佧佒佟佁佘伭伳伿佡冏冹刜刞刡劭劮匉卣卲厎厏吰吷吪呔呅吙吜吥吘"],
["caa1","吽呏呁吨吤呇囮囧囥坁坅坌坉坋坒夆奀妦妘妠妗妎妢妐妏妧妡宎宒尨尪岍岏岈岋岉岒岊岆岓岕巠帊帎庋庉庌庈庍弅弝彸彶忒忑忐忭忨忮忳忡忤忣忺忯忷忻怀忴戺抃抌抎抏抔抇扱扻扺扰抁抈扷扽扲扴攷旰旴旳旲旵杅杇"],
["cb40","杙杕杌杈杝杍杚杋毐氙氚汸汧汫沄沋沏汱汯汩沚汭沇沕沜汦汳汥汻沎灴灺牣犿犽狃狆狁犺狅玕玗玓玔玒町甹疔疕皁礽耴肕肙肐肒肜芐芏芅芎芑芓"],
["cba1","芊芃芄豸迉辿邟邡邥邞邧邠阰阨阯阭丳侘佼侅佽侀侇佶佴侉侄佷佌侗佪侚佹侁佸侐侜侔侞侒侂侕佫佮冞冼冾刵刲刳剆刱劼匊匋匼厒厔咇呿咁咑咂咈呫呺呾呥呬呴呦咍呯呡呠咘呣呧呤囷囹坯坲坭坫坱坰坶垀坵坻坳坴坢"],
["cc40","坨坽夌奅妵妺姏姎妲姌姁妶妼姃姖妱妽姀姈妴姇孢孥宓宕屄屇岮岤岠岵岯岨岬岟岣岭岢岪岧岝岥岶岰岦帗帔帙弨弢弣弤彔徂彾彽忞忥怭怦怙怲怋"],
["cca1","怴怊怗怳怚怞怬怢怍怐怮怓怑怌怉怜戔戽抭抴拑抾抪抶拊抮抳抯抻抩抰抸攽斨斻昉旼昄昒昈旻昃昋昍昅旽昑昐曶朊枅杬枎枒杶杻枘枆构杴枍枌杺枟枑枙枃杽极杸杹枔欥殀歾毞氝沓泬泫泮泙沶泔沭泧沷泐泂沺泃泆泭泲"],
["cd40","泒泝沴沊沝沀泞泀洰泍泇沰泹泏泩泑炔炘炅炓炆炄炑炖炂炚炃牪狖狋狘狉狜狒狔狚狌狑玤玡玭玦玢玠玬玝瓝瓨甿畀甾疌疘皯盳盱盰盵矸矼矹矻矺"],
["cda1","矷祂礿秅穸穻竻籵糽耵肏肮肣肸肵肭舠芠苀芫芚芘芛芵芧芮芼芞芺芴芨芡芩苂芤苃芶芢虰虯虭虮豖迒迋迓迍迖迕迗邲邴邯邳邰阹阽阼阺陃俍俅俓侲俉俋俁俔俜俙侻侳俛俇俖侺俀侹俬剄剉勀勂匽卼厗厖厙厘咺咡咭咥哏"],
["ce40","哃茍咷咮哖咶哅哆咠呰咼咢咾呲哞咰垵垞垟垤垌垗垝垛垔垘垏垙垥垚垕壴复奓姡姞姮娀姱姝姺姽姼姶姤姲姷姛姩姳姵姠姾姴姭宨屌峐峘峌峗峋峛"],
["cea1","峞峚峉峇峊峖峓峔峏峈峆峎峟峸巹帡帢帣帠帤庰庤庢庛庣庥弇弮彖徆怷怹恔恲恞恅恓恇恉恛恌恀恂恟怤恄恘恦恮扂扃拏挍挋拵挎挃拫拹挏挌拸拶挀挓挔拺挕拻拰敁敃斪斿昶昡昲昵昜昦昢昳昫昺昝昴昹昮朏朐柁柲柈枺"],
["cf40","柜枻柸柘柀枷柅柫柤柟枵柍枳柷柶柮柣柂枹柎柧柰枲柼柆柭柌枮柦柛柺柉柊柃柪柋欨殂殄殶毖毘毠氠氡洨洴洭洟洼洿洒洊泚洳洄洙洺洚洑洀洝浂"],
["cfa1","洁洘洷洃洏浀洇洠洬洈洢洉洐炷炟炾炱炰炡炴炵炩牁牉牊牬牰牳牮狊狤狨狫狟狪狦狣玅珌珂珈珅玹玶玵玴珫玿珇玾珃珆玸珋瓬瓮甮畇畈疧疪癹盄眈眃眄眅眊盷盻盺矧矨砆砑砒砅砐砏砎砉砃砓祊祌祋祅祄秕种秏秖秎窀"],
["d040","穾竑笀笁籺籸籹籿粀粁紃紈紁罘羑羍羾耇耎耏耔耷胘胇胠胑胈胂胐胅胣胙胜胊胕胉胏胗胦胍臿舡芔苙苾苹茇苨茀苕茺苫苖苴苬苡苲苵茌苻苶苰苪"],
["d0a1","苤苠苺苳苭虷虴虼虳衁衎衧衪衩觓訄訇赲迣迡迮迠郱邽邿郕郅邾郇郋郈釔釓陔陏陑陓陊陎倞倅倇倓倢倰倛俵俴倳倷倬俶俷倗倜倠倧倵倯倱倎党冔冓凊凄凅凈凎剡剚剒剞剟剕剢勍匎厞唦哢唗唒哧哳哤唚哿唄唈哫唑唅哱"],
["d140","唊哻哷哸哠唎唃唋圁圂埌堲埕埒垺埆垽垼垸垶垿埇埐垹埁夎奊娙娖娭娮娕娏娗娊娞娳孬宧宭宬尃屖屔峬峿峮峱峷崀峹帩帨庨庮庪庬弳弰彧恝恚恧"],
["d1a1","恁悢悈悀悒悁悝悃悕悛悗悇悜悎戙扆拲挐捖挬捄捅挶捃揤挹捋捊挼挩捁挴捘捔捙挭捇挳捚捑挸捗捀捈敊敆旆旃旄旂晊晟晇晑朒朓栟栚桉栲栳栻桋桏栖栱栜栵栫栭栯桎桄栴栝栒栔栦栨栮桍栺栥栠欬欯欭欱欴歭肂殈毦毤"],
["d240","毨毣毢毧氥浺浣浤浶洍浡涒浘浢浭浯涑涍淯浿涆浞浧浠涗浰浼浟涂涘洯浨涋浾涀涄洖涃浻浽浵涐烜烓烑烝烋缹烢烗烒烞烠烔烍烅烆烇烚烎烡牂牸"],
["d2a1","牷牶猀狺狴狾狶狳狻猁珓珙珥珖玼珧珣珩珜珒珛珔珝珚珗珘珨瓞瓟瓴瓵甡畛畟疰痁疻痄痀疿疶疺皊盉眝眛眐眓眒眣眑眕眙眚眢眧砣砬砢砵砯砨砮砫砡砩砳砪砱祔祛祏祜祓祒祑秫秬秠秮秭秪秜秞秝窆窉窅窋窌窊窇竘笐"],
["d340","笄笓笅笏笈笊笎笉笒粄粑粊粌粈粍粅紞紝紑紎紘紖紓紟紒紏紌罜罡罞罠罝罛羖羒翃翂翀耖耾耹胺胲胹胵脁胻脀舁舯舥茳茭荄茙荑茥荖茿荁茦茜茢"],
["d3a1","荂荎茛茪茈茼荍茖茤茠茷茯茩荇荅荌荓茞茬荋茧荈虓虒蚢蚨蚖蚍蚑蚞蚇蚗蚆蚋蚚蚅蚥蚙蚡蚧蚕蚘蚎蚝蚐蚔衃衄衭衵衶衲袀衱衿衯袃衾衴衼訒豇豗豻貤貣赶赸趵趷趶軑軓迾迵适迿迻逄迼迶郖郠郙郚郣郟郥郘郛郗郜郤酐"],
["d440","酎酏釕釢釚陜陟隼飣髟鬯乿偰偪偡偞偠偓偋偝偲偈偍偁偛偊偢倕偅偟偩偫偣偤偆偀偮偳偗偑凐剫剭剬剮勖勓匭厜啵啶唼啍啐唴唪啑啢唶唵唰啒啅"],
["d4a1","唌唲啥啎唹啈唭唻啀啋圊圇埻堔埢埶埜埴堀埭埽堈埸堋埳埏堇埮埣埲埥埬埡堎埼堐埧堁堌埱埩埰堍堄奜婠婘婕婧婞娸娵婭婐婟婥婬婓婤婗婃婝婒婄婛婈媎娾婍娹婌婰婩婇婑婖婂婜孲孮寁寀屙崞崋崝崚崠崌崨崍崦崥崏"],
["d540","崰崒崣崟崮帾帴庱庴庹庲庳弶弸徛徖徟悊悐悆悾悰悺惓惔惏惤惙惝惈悱惛悷惊悿惃惍惀挲捥掊掂捽掽掞掭掝掗掫掎捯掇掐据掯捵掜捭掮捼掤挻掟"],
["d5a1","捸掅掁掑掍捰敓旍晥晡晛晙晜晢朘桹梇梐梜桭桮梮梫楖桯梣梬梩桵桴梲梏桷梒桼桫桲梪梀桱桾梛梖梋梠梉梤桸桻梑梌梊桽欶欳欷欸殑殏殍殎殌氪淀涫涴涳湴涬淩淢涷淶淔渀淈淠淟淖涾淥淜淝淛淴淊涽淭淰涺淕淂淏淉"],
["d640","淐淲淓淽淗淍淣涻烺焍烷焗烴焌烰焄烳焐烼烿焆焓焀烸烶焋焂焎牾牻牼牿猝猗猇猑猘猊猈狿猏猞玈珶珸珵琄琁珽琇琀珺珼珿琌琋珴琈畤畣痎痒痏"],
["d6a1","痋痌痑痐皏皉盓眹眯眭眱眲眴眳眽眥眻眵硈硒硉硍硊硌砦硅硐祤祧祩祪祣祫祡离秺秸秶秷窏窔窐笵筇笴笥笰笢笤笳笘笪笝笱笫笭笯笲笸笚笣粔粘粖粣紵紽紸紶紺絅紬紩絁絇紾紿絊紻紨罣羕羜羝羛翊翋翍翐翑翇翏翉耟"],
["d740","耞耛聇聃聈脘脥脙脛脭脟脬脞脡脕脧脝脢舑舸舳舺舴舲艴莐莣莨莍荺荳莤荴莏莁莕莙荵莔莩荽莃莌莝莛莪莋荾莥莯莈莗莰荿莦莇莮荶莚虙虖蚿蚷"],
["d7a1","蛂蛁蛅蚺蚰蛈蚹蚳蚸蛌蚴蚻蚼蛃蚽蚾衒袉袕袨袢袪袚袑袡袟袘袧袙袛袗袤袬袌袓袎覂觖觙觕訰訧訬訞谹谻豜豝豽貥赽赻赹趼跂趹趿跁軘軞軝軜軗軠軡逤逋逑逜逌逡郯郪郰郴郲郳郔郫郬郩酖酘酚酓酕釬釴釱釳釸釤釹釪"],
["d840","釫釷釨釮镺閆閈陼陭陫陱陯隿靪頄飥馗傛傕傔傞傋傣傃傌傎傝偨傜傒傂傇兟凔匒匑厤厧喑喨喥喭啷噅喢喓喈喏喵喁喣喒喤啽喌喦啿喕喡喎圌堩堷"],
["d8a1","堙堞堧堣堨埵塈堥堜堛堳堿堶堮堹堸堭堬堻奡媯媔媟婺媢媞婸媦婼媥媬媕媮娷媄媊媗媃媋媩婻婽媌媜媏媓媝寪寍寋寔寑寊寎尌尰崷嵃嵫嵁嵋崿崵嵑嵎嵕崳崺嵒崽崱嵙嵂崹嵉崸崼崲崶嵀嵅幄幁彘徦徥徫惉悹惌惢惎惄愔"],
["d940","惲愊愖愅惵愓惸惼惾惁愃愘愝愐惿愄愋扊掔掱掰揎揥揨揯揃撝揳揊揠揶揕揲揵摡揟掾揝揜揄揘揓揂揇揌揋揈揰揗揙攲敧敪敤敜敨敥斌斝斞斮旐旒"],
["d9a1","晼晬晻暀晱晹晪晲朁椌棓椄棜椪棬棪棱椏棖棷棫棤棶椓椐棳棡椇棌椈楰梴椑棯棆椔棸棐棽棼棨椋椊椗棎棈棝棞棦棴棑椆棔棩椕椥棇欹欻欿欼殔殗殙殕殽毰毲毳氰淼湆湇渟湉溈渼渽湅湢渫渿湁湝湳渜渳湋湀湑渻渃渮湞"],
["da40","湨湜湡渱渨湠湱湫渹渢渰湓湥渧湸湤湷湕湹湒湦渵渶湚焠焞焯烻焮焱焣焥焢焲焟焨焺焛牋牚犈犉犆犅犋猒猋猰猢猱猳猧猲猭猦猣猵猌琮琬琰琫琖"],
["daa1","琚琡琭琱琤琣琝琩琠琲瓻甯畯畬痧痚痡痦痝痟痤痗皕皒盚睆睇睄睍睅睊睎睋睌矞矬硠硤硥硜硭硱硪确硰硩硨硞硢祴祳祲祰稂稊稃稌稄窙竦竤筊笻筄筈筌筎筀筘筅粢粞粨粡絘絯絣絓絖絧絪絏絭絜絫絒絔絩絑絟絎缾缿罥"],
["db40","罦羢羠羡翗聑聏聐胾胔腃腊腒腏腇脽腍脺臦臮臷臸臹舄舼舽舿艵茻菏菹萣菀菨萒菧菤菼菶萐菆菈菫菣莿萁菝菥菘菿菡菋菎菖菵菉萉萏菞萑萆菂菳"],
["dba1","菕菺菇菑菪萓菃菬菮菄菻菗菢萛菛菾蛘蛢蛦蛓蛣蛚蛪蛝蛫蛜蛬蛩蛗蛨蛑衈衖衕袺裗袹袸裀袾袶袼袷袽袲褁裉覕覘覗觝觚觛詎詍訹詙詀詗詘詄詅詒詈詑詊詌詏豟貁貀貺貾貰貹貵趄趀趉跘跓跍跇跖跜跏跕跙跈跗跅軯軷軺"],
["dc40","軹軦軮軥軵軧軨軶軫軱軬軴軩逭逴逯鄆鄬鄄郿郼鄈郹郻鄁鄀鄇鄅鄃酡酤酟酢酠鈁鈊鈥鈃鈚鈦鈏鈌鈀鈒釿釽鈆鈄鈧鈂鈜鈤鈙鈗鈅鈖镻閍閌閐隇陾隈"],
["dca1","隉隃隀雂雈雃雱雰靬靰靮頇颩飫鳦黹亃亄亶傽傿僆傮僄僊傴僈僂傰僁傺傱僋僉傶傸凗剺剸剻剼嗃嗛嗌嗐嗋嗊嗝嗀嗔嗄嗩喿嗒喍嗏嗕嗢嗖嗈嗲嗍嗙嗂圔塓塨塤塏塍塉塯塕塎塝塙塥塛堽塣塱壼嫇嫄嫋媺媸媱媵媰媿嫈媻嫆"],
["dd40","媷嫀嫊媴媶嫍媹媐寖寘寙尟尳嵱嵣嵊嵥嵲嵬嵞嵨嵧嵢巰幏幎幊幍幋廅廌廆廋廇彀徯徭惷慉慊愫慅愶愲愮慆愯慏愩慀戠酨戣戥戤揅揱揫搐搒搉搠搤"],
["dda1","搳摃搟搕搘搹搷搢搣搌搦搰搨摁搵搯搊搚摀搥搧搋揧搛搮搡搎敯斒旓暆暌暕暐暋暊暙暔晸朠楦楟椸楎楢楱椿楅楪椹楂楗楙楺楈楉椵楬椳椽楥棰楸椴楩楀楯楄楶楘楁楴楌椻楋椷楜楏楑椲楒椯楻椼歆歅歃歂歈歁殛嗀毻毼"],
["de40","毹毷毸溛滖滈溏滀溟溓溔溠溱溹滆滒溽滁溞滉溷溰滍溦滏溲溾滃滜滘溙溒溎溍溤溡溿溳滐滊溗溮溣煇煔煒煣煠煁煝煢煲煸煪煡煂煘煃煋煰煟煐煓"],
["dea1","煄煍煚牏犍犌犑犐犎猼獂猻猺獀獊獉瑄瑊瑋瑒瑑瑗瑀瑏瑐瑎瑂瑆瑍瑔瓡瓿瓾瓽甝畹畷榃痯瘏瘃痷痾痼痹痸瘐痻痶痭痵痽皙皵盝睕睟睠睒睖睚睩睧睔睙睭矠碇碚碔碏碄碕碅碆碡碃硹碙碀碖硻祼禂祽祹稑稘稙稒稗稕稢稓"],
["df40","稛稐窣窢窞竫筦筤筭筴筩筲筥筳筱筰筡筸筶筣粲粴粯綈綆綀綍絿綅絺綎絻綃絼綌綔綄絽綒罭罫罧罨罬羦羥羧翛翜耡腤腠腷腜腩腛腢腲朡腞腶腧腯"],
["dfa1","腄腡舝艉艄艀艂艅蓱萿葖葶葹蒏蒍葥葑葀蒆葧萰葍葽葚葙葴葳葝蔇葞萷萺萴葺葃葸萲葅萩菙葋萯葂萭葟葰萹葎葌葒葯蓅蒎萻葇萶萳葨葾葄萫葠葔葮葐蜋蜄蛷蜌蛺蛖蛵蝍蛸蜎蜉蜁蛶蜍蜅裖裋裍裎裞裛裚裌裐覅覛觟觥觤"],
["e040","觡觠觢觜触詶誆詿詡訿詷誂誄詵誃誁詴詺谼豋豊豥豤豦貆貄貅賌赨赩趑趌趎趏趍趓趔趐趒跰跠跬跱跮跐跩跣跢跧跲跫跴輆軿輁輀輅輇輈輂輋遒逿"],
["e0a1","遄遉逽鄐鄍鄏鄑鄖鄔鄋鄎酮酯鉈鉒鈰鈺鉦鈳鉥鉞銃鈮鉊鉆鉭鉬鉏鉠鉧鉯鈶鉡鉰鈱鉔鉣鉐鉲鉎鉓鉌鉖鈲閟閜閞閛隒隓隑隗雎雺雽雸雵靳靷靸靲頏頍頎颬飶飹馯馲馰馵骭骫魛鳪鳭鳧麀黽僦僔僗僨僳僛僪僝僤僓僬僰僯僣僠"],
["e140","凘劀劁勩勫匰厬嘧嘕嘌嘒嗼嘏嘜嘁嘓嘂嗺嘝嘄嗿嗹墉塼墐墘墆墁塿塴墋塺墇墑墎塶墂墈塻墔墏壾奫嫜嫮嫥嫕嫪嫚嫭嫫嫳嫢嫠嫛嫬嫞嫝嫙嫨嫟孷寠"],
["e1a1","寣屣嶂嶀嵽嶆嵺嶁嵷嶊嶉嶈嵾嵼嶍嵹嵿幘幙幓廘廑廗廎廜廕廙廒廔彄彃彯徶愬愨慁慞慱慳慒慓慲慬憀慴慔慺慛慥愻慪慡慖戩戧戫搫摍摛摝摴摶摲摳摽摵摦撦摎撂摞摜摋摓摠摐摿搿摬摫摙摥摷敳斠暡暠暟朅朄朢榱榶槉"],
["e240","榠槎榖榰榬榼榑榙榎榧榍榩榾榯榿槄榽榤槔榹槊榚槏榳榓榪榡榞槙榗榐槂榵榥槆歊歍歋殞殟殠毃毄毾滎滵滱漃漥滸漷滻漮漉潎漙漚漧漘漻漒滭漊"],
["e2a1","漶潳滹滮漭潀漰漼漵滫漇漎潃漅滽滶漹漜滼漺漟漍漞漈漡熇熐熉熀熅熂熏煻熆熁熗牄牓犗犕犓獃獍獑獌瑢瑳瑱瑵瑲瑧瑮甀甂甃畽疐瘖瘈瘌瘕瘑瘊瘔皸瞁睼瞅瞂睮瞀睯睾瞃碲碪碴碭碨硾碫碞碥碠碬碢碤禘禊禋禖禕禔禓"],
["e340","禗禈禒禐稫穊稰稯稨稦窨窫窬竮箈箜箊箑箐箖箍箌箛箎箅箘劄箙箤箂粻粿粼粺綧綷緂綣綪緁緀緅綝緎緄緆緋緌綯綹綖綼綟綦綮綩綡緉罳翢翣翥翞"],
["e3a1","耤聝聜膉膆膃膇膍膌膋舕蒗蒤蒡蒟蒺蓎蓂蒬蒮蒫蒹蒴蓁蓍蒪蒚蒱蓐蒝蒧蒻蒢蒔蓇蓌蒛蒩蒯蒨蓖蒘蒶蓏蒠蓗蓔蓒蓛蒰蒑虡蜳蜣蜨蝫蝀蜮蜞蜡蜙蜛蝃蜬蝁蜾蝆蜠蜲蜪蜭蜼蜒蜺蜱蜵蝂蜦蜧蜸蜤蜚蜰蜑裷裧裱裲裺裾裮裼裶裻"],
["e440","裰裬裫覝覡覟覞觩觫觨誫誙誋誒誏誖谽豨豩賕賏賗趖踉踂跿踍跽踊踃踇踆踅跾踀踄輐輑輎輍鄣鄜鄠鄢鄟鄝鄚鄤鄡鄛酺酲酹酳銥銤鉶銛鉺銠銔銪銍"],
["e4a1","銦銚銫鉹銗鉿銣鋮銎銂銕銢鉽銈銡銊銆銌銙銧鉾銇銩銝銋鈭隞隡雿靘靽靺靾鞃鞀鞂靻鞄鞁靿韎韍頖颭颮餂餀餇馝馜駃馹馻馺駂馽駇骱髣髧鬾鬿魠魡魟鳱鳲鳵麧僿儃儰僸儆儇僶僾儋儌僽儊劋劌勱勯噈噂噌嘵噁噊噉噆噘"],
["e540","噚噀嘳嘽嘬嘾嘸嘪嘺圚墫墝墱墠墣墯墬墥墡壿嫿嫴嫽嫷嫶嬃嫸嬂嫹嬁嬇嬅嬏屧嶙嶗嶟嶒嶢嶓嶕嶠嶜嶡嶚嶞幩幝幠幜緳廛廞廡彉徲憋憃慹憱憰憢憉"],
["e5a1","憛憓憯憭憟憒憪憡憍慦憳戭摮摰撖撠撅撗撜撏撋撊撌撣撟摨撱撘敶敺敹敻斲斳暵暰暩暲暷暪暯樀樆樗槥槸樕槱槤樠槿槬槢樛樝槾樧槲槮樔槷槧橀樈槦槻樍槼槫樉樄樘樥樏槶樦樇槴樖歑殥殣殢殦氁氀毿氂潁漦潾澇濆澒"],
["e640","澍澉澌潢潏澅潚澖潶潬澂潕潲潒潐潗澔澓潝漀潡潫潽潧澐潓澋潩潿澕潣潷潪潻熲熯熛熰熠熚熩熵熝熥熞熤熡熪熜熧熳犘犚獘獒獞獟獠獝獛獡獚獙"],
["e6a1","獢璇璉璊璆璁瑽璅璈瑼瑹甈甇畾瘥瘞瘙瘝瘜瘣瘚瘨瘛皜皝皞皛瞍瞏瞉瞈磍碻磏磌磑磎磔磈磃磄磉禚禡禠禜禢禛歶稹窲窴窳箷篋箾箬篎箯箹篊箵糅糈糌糋緷緛緪緧緗緡縃緺緦緶緱緰緮緟罶羬羰羭翭翫翪翬翦翨聤聧膣膟"],
["e740","膞膕膢膙膗舖艏艓艒艐艎艑蔤蔻蔏蔀蔩蔎蔉蔍蔟蔊蔧蔜蓻蔫蓺蔈蔌蓴蔪蓲蔕蓷蓫蓳蓼蔒蓪蓩蔖蓾蔨蔝蔮蔂蓽蔞蓶蔱蔦蓧蓨蓰蓯蓹蔘蔠蔰蔋蔙蔯虢"],
["e7a1","蝖蝣蝤蝷蟡蝳蝘蝔蝛蝒蝡蝚蝑蝞蝭蝪蝐蝎蝟蝝蝯蝬蝺蝮蝜蝥蝏蝻蝵蝢蝧蝩衚褅褌褔褋褗褘褙褆褖褑褎褉覢覤覣觭觰觬諏諆誸諓諑諔諕誻諗誾諀諅諘諃誺誽諙谾豍貏賥賟賙賨賚賝賧趠趜趡趛踠踣踥踤踮踕踛踖踑踙踦踧"],
["e840","踔踒踘踓踜踗踚輬輤輘輚輠輣輖輗遳遰遯遧遫鄯鄫鄩鄪鄲鄦鄮醅醆醊醁醂醄醀鋐鋃鋄鋀鋙銶鋏鋱鋟鋘鋩鋗鋝鋌鋯鋂鋨鋊鋈鋎鋦鋍鋕鋉鋠鋞鋧鋑鋓"],
["e8a1","銵鋡鋆銴镼閬閫閮閰隤隢雓霅霈霂靚鞊鞎鞈韐韏頞頝頦頩頨頠頛頧颲餈飺餑餔餖餗餕駜駍駏駓駔駎駉駖駘駋駗駌骳髬髫髳髲髱魆魃魧魴魱魦魶魵魰魨魤魬鳼鳺鳽鳿鳷鴇鴀鳹鳻鴈鴅鴄麃黓鼏鼐儜儓儗儚儑凞匴叡噰噠噮"],
["e940","噳噦噣噭噲噞噷圜圛壈墽壉墿墺壂墼壆嬗嬙嬛嬡嬔嬓嬐嬖嬨嬚嬠嬞寯嶬嶱嶩嶧嶵嶰嶮嶪嶨嶲嶭嶯嶴幧幨幦幯廩廧廦廨廥彋徼憝憨憖懅憴懆懁懌憺"],
["e9a1","憿憸憌擗擖擐擏擉撽撉擃擛擳擙攳敿敼斢曈暾曀曊曋曏暽暻暺曌朣樴橦橉橧樲橨樾橝橭橶橛橑樨橚樻樿橁橪橤橐橏橔橯橩橠樼橞橖橕橍橎橆歕歔歖殧殪殫毈毇氄氃氆澭濋澣濇澼濎濈潞濄澽澞濊澨瀄澥澮澺澬澪濏澿澸"],
["ea40","澢濉澫濍澯澲澰燅燂熿熸燖燀燁燋燔燊燇燏熽燘熼燆燚燛犝犞獩獦獧獬獥獫獪瑿璚璠璔璒璕璡甋疀瘯瘭瘱瘽瘳瘼瘵瘲瘰皻盦瞚瞝瞡瞜瞛瞢瞣瞕瞙"],
["eaa1","瞗磝磩磥磪磞磣磛磡磢磭磟磠禤穄穈穇窶窸窵窱窷篞篣篧篝篕篥篚篨篹篔篪篢篜篫篘篟糒糔糗糐糑縒縡縗縌縟縠縓縎縜縕縚縢縋縏縖縍縔縥縤罃罻罼罺羱翯耪耩聬膱膦膮膹膵膫膰膬膴膲膷膧臲艕艖艗蕖蕅蕫蕍蕓蕡蕘"],
["eb40","蕀蕆蕤蕁蕢蕄蕑蕇蕣蔾蕛蕱蕎蕮蕵蕕蕧蕠薌蕦蕝蕔蕥蕬虣虥虤螛螏螗螓螒螈螁螖螘蝹螇螣螅螐螑螝螄螔螜螚螉褞褦褰褭褮褧褱褢褩褣褯褬褟觱諠"],
["eba1","諢諲諴諵諝謔諤諟諰諈諞諡諨諿諯諻貑貒貐賵賮賱賰賳赬赮趥趧踳踾踸蹀蹅踶踼踽蹁踰踿躽輶輮輵輲輹輷輴遶遹遻邆郺鄳鄵鄶醓醐醑醍醏錧錞錈錟錆錏鍺錸錼錛錣錒錁鍆錭錎錍鋋錝鋺錥錓鋹鋷錴錂錤鋿錩錹錵錪錔錌"],
["ec40","錋鋾錉錀鋻錖閼闍閾閹閺閶閿閵閽隩雔霋霒霐鞙鞗鞔韰韸頵頯頲餤餟餧餩馞駮駬駥駤駰駣駪駩駧骹骿骴骻髶髺髹髷鬳鮀鮅鮇魼魾魻鮂鮓鮒鮐魺鮕"],
["eca1","魽鮈鴥鴗鴠鴞鴔鴩鴝鴘鴢鴐鴙鴟麈麆麇麮麭黕黖黺鼒鼽儦儥儢儤儠儩勴嚓嚌嚍嚆嚄嚃噾嚂噿嚁壖壔壏壒嬭嬥嬲嬣嬬嬧嬦嬯嬮孻寱寲嶷幬幪徾徻懃憵憼懧懠懥懤懨懞擯擩擣擫擤擨斁斀斶旚曒檍檖檁檥檉檟檛檡檞檇檓檎"],
["ed40","檕檃檨檤檑橿檦檚檅檌檒歛殭氉濌澩濴濔濣濜濭濧濦濞濲濝濢濨燡燱燨燲燤燰燢獳獮獯璗璲璫璐璪璭璱璥璯甐甑甒甏疄癃癈癉癇皤盩瞵瞫瞲瞷瞶"],
["eda1","瞴瞱瞨矰磳磽礂磻磼磲礅磹磾礄禫禨穜穛穖穘穔穚窾竀竁簅簏篲簀篿篻簎篴簋篳簂簉簃簁篸篽簆篰篱簐簊糨縭縼繂縳顈縸縪繉繀繇縩繌縰縻縶繄縺罅罿罾罽翴翲耬膻臄臌臊臅臇膼臩艛艚艜薃薀薏薧薕薠薋薣蕻薤薚薞"],
["ee40","蕷蕼薉薡蕺蕸蕗薎薖薆薍薙薝薁薢薂薈薅蕹蕶薘薐薟虨螾螪螭蟅螰螬螹螵螼螮蟉蟃蟂蟌螷螯蟄蟊螴螶螿螸螽蟞螲褵褳褼褾襁襒褷襂覭覯覮觲觳謞"],
["eea1","謘謖謑謅謋謢謏謒謕謇謍謈謆謜謓謚豏豰豲豱豯貕貔賹赯蹎蹍蹓蹐蹌蹇轃轀邅遾鄸醚醢醛醙醟醡醝醠鎡鎃鎯鍤鍖鍇鍼鍘鍜鍶鍉鍐鍑鍠鍭鎏鍌鍪鍹鍗鍕鍒鍏鍱鍷鍻鍡鍞鍣鍧鎀鍎鍙闇闀闉闃闅閷隮隰隬霠霟霘霝霙鞚鞡鞜"],
["ef40","鞞鞝韕韔韱顁顄顊顉顅顃餥餫餬餪餳餲餯餭餱餰馘馣馡騂駺駴駷駹駸駶駻駽駾駼騃骾髾髽鬁髼魈鮚鮨鮞鮛鮦鮡鮥鮤鮆鮢鮠鮯鴳鵁鵧鴶鴮鴯鴱鴸鴰"],
["efa1","鵅鵂鵃鴾鴷鵀鴽翵鴭麊麉麍麰黈黚黻黿鼤鼣鼢齔龠儱儭儮嚘嚜嚗嚚嚝嚙奰嬼屩屪巀幭幮懘懟懭懮懱懪懰懫懖懩擿攄擽擸攁攃擼斔旛曚曛曘櫅檹檽櫡櫆檺檶檷櫇檴檭歞毉氋瀇瀌瀍瀁瀅瀔瀎濿瀀濻瀦濼濷瀊爁燿燹爃燽獶"],
["f040","璸瓀璵瓁璾璶璻瓂甔甓癜癤癙癐癓癗癚皦皽盬矂瞺磿礌礓礔礉礐礒礑禭禬穟簜簩簙簠簟簭簝簦簨簢簥簰繜繐繖繣繘繢繟繑繠繗繓羵羳翷翸聵臑臒"],
["f0a1","臐艟艞薴藆藀藃藂薳薵薽藇藄薿藋藎藈藅薱薶藒蘤薸薷薾虩蟧蟦蟢蟛蟫蟪蟥蟟蟳蟤蟔蟜蟓蟭蟘蟣螤蟗蟙蠁蟴蟨蟝襓襋襏襌襆襐襑襉謪謧謣謳謰謵譇謯謼謾謱謥謷謦謶謮謤謻謽謺豂豵貙貘貗賾贄贂贀蹜蹢蹠蹗蹖蹞蹥蹧"],
["f140","蹛蹚蹡蹝蹩蹔轆轇轈轋鄨鄺鄻鄾醨醥醧醯醪鎵鎌鎒鎷鎛鎝鎉鎧鎎鎪鎞鎦鎕鎈鎙鎟鎍鎱鎑鎲鎤鎨鎴鎣鎥闒闓闑隳雗雚巂雟雘雝霣霢霥鞬鞮鞨鞫鞤鞪"],
["f1a1","鞢鞥韗韙韖韘韺顐顑顒颸饁餼餺騏騋騉騍騄騑騊騅騇騆髀髜鬈鬄鬅鬩鬵魊魌魋鯇鯆鯃鮿鯁鮵鮸鯓鮶鯄鮹鮽鵜鵓鵏鵊鵛鵋鵙鵖鵌鵗鵒鵔鵟鵘鵚麎麌黟鼁鼀鼖鼥鼫鼪鼩鼨齌齕儴儵劖勷厴嚫嚭嚦嚧嚪嚬壚壝壛夒嬽嬾嬿巃幰"],
["f240","徿懻攇攐攍攉攌攎斄旞旝曞櫧櫠櫌櫑櫙櫋櫟櫜櫐櫫櫏櫍櫞歠殰氌瀙瀧瀠瀖瀫瀡瀢瀣瀩瀗瀤瀜瀪爌爊爇爂爅犥犦犤犣犡瓋瓅璷瓃甖癠矉矊矄矱礝礛"],
["f2a1","礡礜礗礞禰穧穨簳簼簹簬簻糬糪繶繵繸繰繷繯繺繲繴繨罋罊羃羆羷翽翾聸臗臕艤艡艣藫藱藭藙藡藨藚藗藬藲藸藘藟藣藜藑藰藦藯藞藢蠀蟺蠃蟶蟷蠉蠌蠋蠆蟼蠈蟿蠊蠂襢襚襛襗襡襜襘襝襙覈覷覶觶譐譈譊譀譓譖譔譋譕"],
["f340","譑譂譒譗豃豷豶貚贆贇贉趬趪趭趫蹭蹸蹳蹪蹯蹻軂轒轑轏轐轓辴酀鄿醰醭鏞鏇鏏鏂鏚鏐鏹鏬鏌鏙鎩鏦鏊鏔鏮鏣鏕鏄鏎鏀鏒鏧镽闚闛雡霩霫霬霨霦"],
["f3a1","鞳鞷鞶韝韞韟顜顙顝顗颿颽颻颾饈饇饃馦馧騚騕騥騝騤騛騢騠騧騣騞騜騔髂鬋鬊鬎鬌鬷鯪鯫鯠鯞鯤鯦鯢鯰鯔鯗鯬鯜鯙鯥鯕鯡鯚鵷鶁鶊鶄鶈鵱鶀鵸鶆鶋鶌鵽鵫鵴鵵鵰鵩鶅鵳鵻鶂鵯鵹鵿鶇鵨麔麑黀黼鼭齀齁齍齖齗齘匷嚲"],
["f440","嚵嚳壣孅巆巇廮廯忀忁懹攗攖攕攓旟曨曣曤櫳櫰櫪櫨櫹櫱櫮櫯瀼瀵瀯瀷瀴瀱灂瀸瀿瀺瀹灀瀻瀳灁爓爔犨獽獼璺皫皪皾盭矌矎矏矍矲礥礣礧礨礤礩"],
["f4a1","禲穮穬穭竷籉籈籊籇籅糮繻繾纁纀羺翿聹臛臙舋艨艩蘢藿蘁藾蘛蘀藶蘄蘉蘅蘌藽蠙蠐蠑蠗蠓蠖襣襦覹觷譠譪譝譨譣譥譧譭趮躆躈躄轙轖轗轕轘轚邍酃酁醷醵醲醳鐋鐓鏻鐠鐏鐔鏾鐕鐐鐨鐙鐍鏵鐀鏷鐇鐎鐖鐒鏺鐉鏸鐊鏿"],
["f540","鏼鐌鏶鐑鐆闞闠闟霮霯鞹鞻韽韾顠顢顣顟飁飂饐饎饙饌饋饓騲騴騱騬騪騶騩騮騸騭髇髊髆鬐鬒鬑鰋鰈鯷鰅鰒鯸鱀鰇鰎鰆鰗鰔鰉鶟鶙鶤鶝鶒鶘鶐鶛"],
["f5a1","鶠鶔鶜鶪鶗鶡鶚鶢鶨鶞鶣鶿鶩鶖鶦鶧麙麛麚黥黤黧黦鼰鼮齛齠齞齝齙龑儺儹劘劗囃嚽嚾孈孇巋巏廱懽攛欂櫼欃櫸欀灃灄灊灈灉灅灆爝爚爙獾甗癪矐礭礱礯籔籓糲纊纇纈纋纆纍罍羻耰臝蘘蘪蘦蘟蘣蘜蘙蘧蘮蘡蘠蘩蘞蘥"],
["f640","蠩蠝蠛蠠蠤蠜蠫衊襭襩襮襫觺譹譸譅譺譻贐贔趯躎躌轞轛轝酆酄酅醹鐿鐻鐶鐩鐽鐼鐰鐹鐪鐷鐬鑀鐱闥闤闣霵霺鞿韡顤飉飆飀饘饖騹騽驆驄驂驁騺"],
["f6a1","騿髍鬕鬗鬘鬖鬺魒鰫鰝鰜鰬鰣鰨鰩鰤鰡鶷鶶鶼鷁鷇鷊鷏鶾鷅鷃鶻鶵鷎鶹鶺鶬鷈鶱鶭鷌鶳鷍鶲鹺麜黫黮黭鼛鼘鼚鼱齎齥齤龒亹囆囅囋奱孋孌巕巑廲攡攠攦攢欋欈欉氍灕灖灗灒爞爟犩獿瓘瓕瓙瓗癭皭礵禴穰穱籗籜籙籛籚"],
["f740","糴糱纑罏羇臞艫蘴蘵蘳蘬蘲蘶蠬蠨蠦蠪蠥襱覿覾觻譾讄讂讆讅譿贕躕躔躚躒躐躖躗轠轢酇鑌鑐鑊鑋鑏鑇鑅鑈鑉鑆霿韣顪顩飋饔饛驎驓驔驌驏驈驊"],
["f7a1","驉驒驐髐鬙鬫鬻魖魕鱆鱈鰿鱄鰹鰳鱁鰼鰷鰴鰲鰽鰶鷛鷒鷞鷚鷋鷐鷜鷑鷟鷩鷙鷘鷖鷵鷕鷝麶黰鼵鼳鼲齂齫龕龢儽劙壨壧奲孍巘蠯彏戁戃戄攩攥斖曫欑欒欏毊灛灚爢玂玁玃癰矔籧籦纕艬蘺虀蘹蘼蘱蘻蘾蠰蠲蠮蠳襶襴襳觾"],
["f840","讌讎讋讈豅贙躘轤轣醼鑢鑕鑝鑗鑞韄韅頀驖驙鬞鬟鬠鱒鱘鱐鱊鱍鱋鱕鱙鱌鱎鷻鷷鷯鷣鷫鷸鷤鷶鷡鷮鷦鷲鷰鷢鷬鷴鷳鷨鷭黂黐黲黳鼆鼜鼸鼷鼶齃齏"],
["f8a1","齱齰齮齯囓囍孎屭攭曭曮欓灟灡灝灠爣瓛瓥矕礸禷禶籪纗羉艭虃蠸蠷蠵衋讔讕躞躟躠躝醾醽釂鑫鑨鑩雥靆靃靇韇韥驞髕魙鱣鱧鱦鱢鱞鱠鸂鷾鸇鸃鸆鸅鸀鸁鸉鷿鷽鸄麠鼞齆齴齵齶囔攮斸欘欙欗欚灢爦犪矘矙礹籩籫糶纚"],
["f940","纘纛纙臠臡虆虇虈襹襺襼襻觿讘讙躥躤躣鑮鑭鑯鑱鑳靉顲饟鱨鱮鱭鸋鸍鸐鸏鸒鸑麡黵鼉齇齸齻齺齹圞灦籯蠼趲躦釃鑴鑸鑶鑵驠鱴鱳鱱鱵鸔鸓黶鼊"],
["f9a1","龤灨灥糷虪蠾蠽蠿讞貜躩軉靋顳顴飌饡馫驤驦驧鬤鸕鸗齈戇欞爧虌躨钂钀钁驩驨鬮鸙爩虋讟钃鱹麷癵驫鱺鸝灩灪麤齾齉龘碁銹裏墻恒粧嫺╔╦╗╠╬╣╚╩╝╒╤╕╞╪╡╘╧╛╓╥╖╟╫╢╙╨╜║═╭╮╰╯▓"]
]
;

var cp950$1 = /*#__PURE__*/Object.freeze({
            default: cp950
});

var big5Added = [
["8740","䏰䰲䘃䖦䕸𧉧䵷䖳𧲱䳢𧳅㮕䜶䝄䱇䱀𤊿𣘗𧍒𦺋𧃒䱗𪍑䝏䗚䲅𧱬䴇䪤䚡𦬣爥𥩔𡩣𣸆𣽡晍囻"],
["8767","綕夝𨮹㷴霴𧯯寛𡵞媤㘥𩺰嫑宷峼杮薓𩥅瑡璝㡵𡵓𣚞𦀡㻬"],
["87a1","𥣞㫵竼龗𤅡𨤍𣇪𠪊𣉞䌊蒄龖鐯䤰蘓墖靊鈘秐稲晠権袝瑌篅枂稬剏遆㓦珄𥶹瓆鿇垳䤯呌䄱𣚎堘穲𧭥讏䚮𦺈䆁𥶙箮𢒼鿈𢓁𢓉𢓌鿉蔄𣖻䂴鿊䓡𪷿拁灮鿋"],
["8840","㇀",4,"𠄌㇅𠃑𠃍㇆㇇𠃋𡿨㇈𠃊㇉㇊㇋㇌𠄎㇍㇎ĀÁǍÀĒÉĚÈŌÓǑÒ࿿Ê̄Ế࿿Ê̌ỀÊāáǎàɑēéěèīíǐìōóǒòūúǔùǖǘǚ"],
["88a1","ǜü࿿ê̄ế࿿ê̌ềêɡ⏚⏛"],
["8940","𪎩𡅅"],
["8943","攊"],
["8946","丽滝鵎釟"],
["894c","𧜵撑会伨侨兖兴农凤务动医华发变团声处备夲头学实実岚庆总斉柾栄桥济炼电纤纬纺织经统缆缷艺苏药视设询车轧轮"],
["89a1","琑糼緍楆竉刧"],
["89ab","醌碸酞肼"],
["89b0","贋胶𠧧"],
["89b5","肟黇䳍鷉鸌䰾𩷶𧀎鸊𪄳㗁"],
["89c1","溚舾甙"],
["89c5","䤑马骏龙禇𨑬𡷊𠗐𢫦两亁亀亇亿仫伷㑌侽㹈倃傈㑽㒓㒥円夅凛凼刅争剹劐匧㗇厩㕑厰㕓参吣㕭㕲㚁咓咣咴咹哐哯唘唣唨㖘唿㖥㖿嗗㗅"],
["8a40","𧶄唥"],
["8a43","𠱂𠴕𥄫喐𢳆㧬𠍁蹆𤶸𩓥䁓𨂾睺𢰸㨴䟕𨅝𦧲𤷪擝𠵼𠾴𠳕𡃴撍蹾𠺖𠰋𠽤𢲩𨉖𤓓"],
["8a64","𠵆𩩍𨃩䟴𤺧𢳂骲㩧𩗴㿭㔆𥋇𩟔𧣈𢵄鵮頕"],
["8a76","䏙𦂥撴哣𢵌𢯊𡁷㧻𡁯"],
["8aa1","𦛚𦜖𧦠擪𥁒𠱃蹨𢆡𨭌𠜱"],
["8aac","䠋𠆩㿺塳𢶍"],
["8ab2","𤗈𠓼𦂗𠽌𠶖啹䂻䎺"],
["8abb","䪴𢩦𡂝膪飵𠶜捹㧾𢝵跀嚡摼㹃"],
["8ac9","𪘁𠸉𢫏𢳉"],
["8ace","𡃈𣧂㦒㨆𨊛㕸𥹉𢃇噒𠼱𢲲𩜠㒼氽𤸻"],
["8adf","𧕴𢺋𢈈𪙛𨳍𠹺𠰴𦠜羓𡃏𢠃𢤹㗻𥇣𠺌𠾍𠺪㾓𠼰𠵇𡅏𠹌"],
["8af6","𠺫𠮩𠵈𡃀𡄽㿹𢚖搲𠾭"],
["8b40","𣏴𧘹𢯎𠵾𠵿𢱑𢱕㨘𠺘𡃇𠼮𪘲𦭐𨳒𨶙𨳊閪哌苄喹"],
["8b55","𩻃鰦骶𧝞𢷮煀腭胬尜𦕲脴㞗卟𨂽醶𠻺𠸏𠹷𠻻㗝𤷫㘉𠳖嚯𢞵𡃉𠸐𠹸𡁸𡅈𨈇𡑕𠹹𤹐𢶤婔𡀝𡀞𡃵𡃶垜𠸑"],
["8ba1","𧚔𨋍𠾵𠹻𥅾㜃𠾶𡆀𥋘𪊽𤧚𡠺𤅷𨉼墙剨㘚𥜽箲孨䠀䬬鼧䧧鰟鮍𥭴𣄽嗻㗲嚉丨夂𡯁屮靑𠂆乛亻㔾尣彑忄㣺扌攵歺氵氺灬爫丬犭𤣩罒礻糹罓𦉪㓁"],
["8bde","𦍋耂肀𦘒𦥑卝衤见𧢲讠贝钅镸长门𨸏韦页风飞饣𩠐鱼鸟黄歯龜丷𠂇阝户钢"],
["8c40","倻淾𩱳龦㷉袏𤅎灷峵䬠𥇍㕙𥴰愢𨨲辧釶熑朙玺𣊁𪄇㲋𡦀䬐磤琂冮𨜏䀉橣𪊺䈣蘏𠩯稪𩥇𨫪靕灍匤𢁾鏴盙𨧣龧矝亣俰傼丯众龨吴綋墒壐𡶶庒庙忂𢜒斋"],
["8ca1","𣏹椙橃𣱣泿"],
["8ca7","爀𤔅玌㻛𤨓嬕璹讃𥲤𥚕窓篬糃繬苸薗龩袐龪躹龫迏蕟駠鈡龬𨶹𡐿䁱䊢娚"],
["8cc9","顨杫䉶圽"],
["8cce","藖𤥻芿𧄍䲁𦵴嵻𦬕𦾾龭龮宖龯曧繛湗秊㶈䓃𣉖𢞖䎚䔶"],
["8ce6","峕𣬚諹屸㴒𣕑嵸龲煗䕘𤃬𡸣䱷㥸㑊𠆤𦱁諌侴𠈹妿腬顖𩣺弻"],
["8d40","𠮟"],
["8d42","𢇁𨥭䄂䚻𩁹㼇龳𪆵䃸㟖䛷𦱆䅼𨚲𧏿䕭㣔𥒚䕡䔛䶉䱻䵶䗪㿈𤬏㙡䓞䒽䇭崾嵈嵖㷼㠏嶤嶹㠠㠸幂庽弥徃㤈㤔㤿㥍惗愽峥㦉憷憹懏㦸戬抐拥挘㧸嚱"],
["8da1","㨃揢揻搇摚㩋擀崕嘡龟㪗斆㪽旿晓㫲暒㬢朖㭂枤栀㭘桊梄㭲㭱㭻椉楃牜楤榟榅㮼槖㯝橥橴橱檂㯬檙㯲檫檵櫔櫶殁毁毪汵沪㳋洂洆洦涁㳯涤涱渕渘温溆𨧀溻滢滚齿滨滩漤漴㵆𣽁澁澾㵪㵵熷岙㶊瀬㶑灐灔灯灿炉𠌥䏁㗱𠻘"],
["8e40","𣻗垾𦻓焾𥟠㙎榢𨯩孴穉𥣡𩓙穥穽𥦬窻窰竂竃燑𦒍䇊竚竝竪䇯咲𥰁笋筕笩𥌎𥳾箢筯莜𥮴𦱿篐萡箒箸𥴠㶭𥱥蒒篺簆簵𥳁籄粃𤢂粦晽𤕸糉糇糦籴糳糵糎"],
["8ea1","繧䔝𦹄絝𦻖璍綉綫焵綳緒𤁗𦀩緤㴓緵𡟹緥𨍭縝𦄡𦅚繮纒䌫鑬縧罀罁罇礶𦋐駡羗𦍑羣𡙡𠁨䕜𣝦䔃𨌺翺𦒉者耈耝耨耯𪂇𦳃耻耼聡𢜔䦉𦘦𣷣𦛨朥肧𨩈脇脚墰𢛶汿𦒘𤾸擧𡒊舘𡡞橓𤩥𤪕䑺舩𠬍𦩒𣵾俹𡓽蓢荢𦬊𤦧𣔰𡝳𣷸芪椛芳䇛"],
["8f40","蕋苐茚𠸖𡞴㛁𣅽𣕚艻苢茘𣺋𦶣𦬅𦮗𣗎㶿茝嗬莅䔋𦶥莬菁菓㑾𦻔橗蕚㒖𦹂𢻯葘𥯤葱㷓䓤檧葊𣲵祘蒨𦮖𦹷𦹃蓞萏莑䒠蒓蓤𥲑䉀𥳀䕃蔴嫲𦺙䔧蕳䔖枿蘖"],
["8fa1","𨘥𨘻藁𧂈蘂𡖂𧃍䕫䕪蘨㙈𡢢号𧎚虾蝱𪃸蟮𢰧螱蟚蠏噡虬桖䘏衅衆𧗠𣶹𧗤衞袜䙛袴袵揁装睷𧜏覇覊覦覩覧覼𨨥觧𧤤𧪽誜瞓釾誐𧩙竩𧬺𣾏䜓𧬸煼謌謟𥐰𥕥謿譌譍誩𤩺讐讛誯𡛟䘕衏貛𧵔𧶏貫㜥𧵓賖𧶘𧶽贒贃𡤐賛灜贑𤳉㻐起"],
["9040","趩𨀂𡀔𤦊㭼𨆼𧄌竧躭躶軃鋔輙輭𨍥𨐒辥錃𪊟𠩐辳䤪𨧞𨔽𣶻廸𣉢迹𪀔𨚼𨔁𢌥㦀𦻗逷𨔼𧪾遡𨕬𨘋邨𨜓郄𨛦邮都酧㫰醩釄粬𨤳𡺉鈎沟鉁鉢𥖹銹𨫆𣲛𨬌𥗛"],
["90a1","𠴱錬鍫𨫡𨯫炏嫃𨫢𨫥䥥鉄𨯬𨰹𨯿鍳鑛躼閅閦鐦閠濶䊹𢙺𨛘𡉼𣸮䧟氜陻隖䅬隣𦻕懚隶磵𨫠隽双䦡𦲸𠉴𦐐𩂯𩃥𤫑𡤕𣌊霱虂霶䨏䔽䖅𤫩灵孁霛靜𩇕靗孊𩇫靟鐥僐𣂷𣂼鞉鞟鞱鞾韀韒韠𥑬韮琜𩐳響韵𩐝𧥺䫑頴頳顋顦㬎𧅵㵑𠘰𤅜"],
["9140","𥜆飊颷飈飇䫿𦴧𡛓喰飡飦飬鍸餹𤨩䭲𩡗𩤅駵騌騻騐驘𥜥㛄𩂱𩯕髠髢𩬅髴䰎鬔鬭𨘀倴鬴𦦨㣃𣁽魐魀𩴾婅𡡣鮎𤉋鰂鯿鰌𩹨鷔𩾷𪆒𪆫𪃡𪄣𪇟鵾鶃𪄴鸎梈"],
["91a1","鷄𢅛𪆓𪈠𡤻𪈳鴹𪂹𪊴麐麕麞麢䴴麪麯𤍤黁㭠㧥㴝伲㞾𨰫鼂鼈䮖鐤𦶢鼗鼖鼹嚟嚊齅馸𩂋韲葿齢齩竜龎爖䮾𤥵𤦻煷𤧸𤍈𤩑玞𨯚𡣺禟𨥾𨸶鍩鏳𨩄鋬鎁鏋𨥬𤒹爗㻫睲穃烐𤑳𤏸煾𡟯炣𡢾𣖙㻇𡢅𥐯𡟸㜢𡛻𡠹㛡𡝴𡣑𥽋㜣𡛀坛𤨥𡏾𡊨"],
["9240","𡏆𡒶蔃𣚦蔃葕𤦔𧅥𣸱𥕜𣻻𧁒䓴𣛮𩦝𦼦柹㜳㰕㷧塬𡤢栐䁗𣜿𤃡𤂋𤄏𦰡哋嚞𦚱嚒𠿟𠮨𠸍鏆𨬓鎜仸儫㠙𤐶亼𠑥𠍿佋侊𥙑婨𠆫𠏋㦙𠌊𠐔㐵伩𠋀𨺳𠉵諚𠈌亘"],
["92a1","働儍侢伃𤨎𣺊佂倮偬傁俌俥偘僼兙兛兝兞湶𣖕𣸹𣺿浲𡢄𣺉冨凃𠗠䓝𠒣𠒒𠒑赺𨪜𠜎剙劤𠡳勡鍮䙺熌𤎌𠰠𤦬𡃤槑𠸝瑹㻞璙琔瑖玘䮎𤪼𤂍叐㖄爏𤃉喴𠍅响𠯆圝鉝雴鍦埝垍坿㘾壋媙𨩆𡛺𡝯𡜐娬妸銏婾嫏娒𥥆𡧳𡡡𤊕㛵洅瑃娡𥺃"],
["9340","媁𨯗𠐓鏠璌𡌃焅䥲鐈𨧻鎽㞠尞岞幞幈𡦖𡥼𣫮廍孏𡤃𡤄㜁𡢠㛝𡛾㛓脪𨩇𡶺𣑲𨦨弌弎𡤧𡞫婫𡜻孄蘔𧗽衠恾𢡠𢘫忛㺸𢖯𢖾𩂈𦽳懀𠀾𠁆𢘛憙憘恵𢲛𢴇𤛔𩅍"],
["93a1","摱𤙥𢭪㨩𢬢𣑐𩣪𢹸挷𪑛撶挱揑𤧣𢵧护𢲡搻敫楲㯴𣂎𣊭𤦉𣊫唍𣋠𡣙𩐿曎𣊉𣆳㫠䆐𥖄𨬢𥖏𡛼𥕛𥐥磮𣄃𡠪𣈴㑤𣈏𣆂𤋉暎𦴤晫䮓昰𧡰𡷫晣𣋒𣋡昞𥡲㣑𣠺𣞼㮙𣞢𣏾瓐㮖枏𤘪梶栞㯄檾㡣𣟕𤒇樳橒櫉欅𡤒攑梘橌㯗橺歗𣿀𣲚鎠鋲𨯪𨫋"],
["9440","銉𨀞𨧜鑧涥漋𤧬浧𣽿㶏渄𤀼娽渊塇洤硂焻𤌚𤉶烱牐犇犔𤞏𤜥兹𤪤𠗫瑺𣻸𣙟𤩊𤤗𥿡㼆㺱𤫟𨰣𣼵悧㻳瓌琼鎇琷䒟𦷪䕑疃㽣𤳙𤴆㽘畕癳𪗆㬙瑨𨫌𤦫𤦎㫻"],
["94a1","㷍𤩎㻿𤧅𤣳釺圲鍂𨫣𡡤僟𥈡𥇧睸𣈲眎眏睻𤚗𣞁㩞𤣰琸璛㺿𤪺𤫇䃈𤪖𦆮錇𥖁砞碍碈磒珐祙𧝁𥛣䄎禛蒖禥樭𣻺稺秴䅮𡛦䄲鈵秱𠵌𤦌𠊙𣶺𡝮㖗啫㕰㚪𠇔𠰍竢婙𢛵𥪯𥪜娍𠉛磰娪𥯆竾䇹籝籭䈑𥮳𥺼𥺦糍𤧹𡞰粎籼粮檲緜縇緓罎𦉡"],
["9540","𦅜𧭈綗𥺂䉪𦭵𠤖柖𠁎𣗏埄𦐒𦏸𤥢翝笧𠠬𥫩𥵃笌𥸎駦虅驣樜𣐿㧢𤧷𦖭騟𦖠蒀𧄧𦳑䓪脷䐂胆脉腂𦞴飃𦩂艢艥𦩑葓𦶧蘐𧈛媆䅿𡡀嬫𡢡嫤𡣘蚠蜨𣶏蠭𧐢娂"],
["95a1","衮佅袇袿裦襥襍𥚃襔𧞅𧞄𨯵𨯙𨮜𨧹㺭蒣䛵䛏㟲訽訜𩑈彍鈫𤊄旔焩烄𡡅鵭貟賩𧷜妚矃姰䍮㛔踪躧𤰉輰轊䋴汘澻𢌡䢛潹溋𡟚鯩㚵𤤯邻邗啱䤆醻鐄𨩋䁢𨫼鐧𨰝𨰻蓥訫閙閧閗閖𨴴瑅㻂𤣿𤩂𤏪㻧𣈥随𨻧𨹦𨹥㻌𤧭𤩸𣿮琒瑫㻼靁𩂰"],
["9640","桇䨝𩂓𥟟靝鍨𨦉𨰦𨬯𦎾銺嬑譩䤼珹𤈛鞛靱餸𠼦巁𨯅𤪲頟𩓚鋶𩗗釥䓀𨭐𤩧𨭤飜𨩅㼀鈪䤥萔餻饍𧬆㷽馛䭯馪驜𨭥𥣈檏騡嫾騯𩣱䮐𩥈馼䮽䮗鍽塲𡌂堢𤦸"],
["96a1","𡓨硄𢜟𣶸棅㵽鑘㤧慐𢞁𢥫愇鱏鱓鱻鰵鰐魿鯏𩸭鮟𪇵𪃾鴡䲮𤄄鸘䲰鴌𪆴𪃭𪃳𩤯鶥蒽𦸒𦿟𦮂藼䔳𦶤𦺄𦷰萠藮𦸀𣟗𦁤秢𣖜𣙀䤭𤧞㵢鏛銾鍈𠊿碹鉷鑍俤㑀遤𥕝砽硔碶硋𡝗𣇉𤥁㚚佲濚濙瀞瀞吔𤆵垻壳垊鴖埗焴㒯𤆬燫𦱀𤾗嬨𡞵𨩉"],
["9740","愌嫎娋䊼𤒈㜬䭻𨧼鎻鎸𡣖𠼝葲𦳀𡐓𤋺𢰦𤏁妔𣶷𦝁綨𦅛𦂤𤦹𤦋𨧺鋥珢㻩璴𨭣𡢟㻡𤪳櫘珳珻㻖𤨾𤪔𡟙𤩦𠎧𡐤𤧥瑈𤤖炥𤥶銄珦鍟𠓾錱𨫎𨨖鎆𨯧𥗕䤵𨪂煫"],
["97a1","𤥃𠳿嚤𠘚𠯫𠲸唂秄𡟺緾𡛂𤩐𡡒䔮鐁㜊𨫀𤦭妰𡢿𡢃𧒄媡㛢𣵛㚰鉟婹𨪁𡡢鍴㳍𠪴䪖㦊僴㵩㵌𡎜煵䋻𨈘渏𩃤䓫浗𧹏灧沯㳖𣿭𣸭渂漌㵯𠏵畑㚼㓈䚀㻚䡱姄鉮䤾轁𨰜𦯀堒埈㛖𡑒烾𤍢𤩱𢿣𡊰𢎽梹楧𡎘𣓥𧯴𣛟𨪃𣟖𣏺𤲟樚𣚭𦲷萾䓟䓎"],
["9840","𦴦𦵑𦲂𦿞漗𧄉茽𡜺菭𦲀𧁓𡟛妉媂𡞳婡婱𡤅𤇼㜭姯𡜼㛇熎鎐暚𤊥婮娫𤊓樫𣻹𧜶𤑛𤋊焝𤉙𨧡侰𦴨峂𤓎𧹍𤎽樌𤉖𡌄炦焳𤏩㶥泟勇𤩏繥姫崯㷳彜𤩝𡟟綤萦"],
["98a1","咅𣫺𣌀𠈔坾𠣕𠘙㿥𡾞𪊶瀃𩅛嵰玏糓𨩙𩐠俈翧狍猐𧫴猸猹𥛶獁獈㺩𧬘遬燵𤣲珡臶㻊県㻑沢国琙琞琟㻢㻰㻴㻺瓓㼎㽓畂畭畲疍㽼痈痜㿀癍㿗癴㿜発𤽜熈嘣覀塩䀝睃䀹条䁅㗛瞘䁪䁯属瞾矋売砘点砜䂨砹硇硑硦葈𥔵礳栃礲䄃"],
["9940","䄉禑禙辻稆込䅧窑䆲窼艹䇄竏竛䇏両筢筬筻簒簛䉠䉺类粜䊌粸䊔糭输烀𠳏総緔緐緽羮羴犟䎗耠耥笹耮耱联㷌垴炠肷胩䏭脌猪脎脒畠脔䐁㬹腖腙腚"],
["99a1","䐓堺腼膄䐥膓䐭膥埯臁臤艔䒏芦艶苊苘苿䒰荗险榊萅烵葤惣蒈䔄蒾蓡蓸蔐蔸蕒䔻蕯蕰藠䕷虲蚒蚲蛯际螋䘆䘗袮裿褤襇覑𧥧訩訸誔誴豑賔賲贜䞘塟跃䟭仮踺嗘坔蹱嗵躰䠷軎転軤軭軲辷迁迊迌逳駄䢭飠鈓䤞鈨鉘鉫銱銮銿"],
["9a40","鋣鋫鋳鋴鋽鍃鎄鎭䥅䥑麿鐗匁鐝鐭鐾䥪鑔鑹锭関䦧间阳䧥枠䨤靀䨵鞲韂噔䫤惨颹䬙飱塄餎餙冴餜餷饂饝饢䭰駅䮝騼鬏窃魩鮁鯝鯱鯴䱭鰠㝯𡯂鵉鰺"],
["9aa1","黾噐鶓鶽鷀鷼银辶鹻麬麱麽黆铜黢黱黸竈齄𠂔𠊷𠎠椚铃妬𠓗塀铁㞹𠗕𠘕𠙶𡚺块煳𠫂𠫍𠮿呪吆𠯋咞𠯻𠰻𠱓𠱥𠱼惧𠲍噺𠲵𠳝𠳭𠵯𠶲𠷈楕鰯螥𠸄𠸎𠻗𠾐𠼭𠹳尠𠾼帋𡁜𡁏𡁶朞𡁻𡂈𡂖㙇𡂿𡃓𡄯𡄻卤蒭𡋣𡍵𡌶讁𡕷𡘙𡟃𡟇乸炻𡠭𡥪"],
["9b40","𡨭𡩅𡰪𡱰𡲬𡻈拃𡻕𡼕熘桕𢁅槩㛈𢉼𢏗𢏺𢜪𢡱𢥏苽𢥧𢦓𢫕覥𢫨辠𢬎鞸𢬿顇骽𢱌"],
["9b62","𢲈𢲷𥯨𢴈𢴒𢶷𢶕𢹂𢽴𢿌𣀳𣁦𣌟𣏞徱晈暿𧩹𣕧𣗳爁𤦺矗𣘚𣜖纇𠍆墵朎"],
["9ba1","椘𣪧𧙗𥿢𣸑𣺹𧗾𢂚䣐䪸𤄙𨪚𤋮𤌍𤀻𤌴𤎖𤩅𠗊凒𠘑妟𡺨㮾𣳿𤐄𤓖垈𤙴㦛𤜯𨗨𩧉㝢𢇃譞𨭎駖𤠒𤣻𤨕爉𤫀𠱸奥𤺥𤾆𠝹軚𥀬劏圿煱𥊙𥐙𣽊𤪧喼𥑆𥑮𦭒釔㑳𥔿𧘲𥕞䜘𥕢𥕦𥟇𤤿𥡝偦㓻𣏌惞𥤃䝼𨥈𥪮𥮉𥰆𡶐垡煑澶𦄂𧰒遖𦆲𤾚譢𦐂𦑊"],
["9c40","嵛𦯷輶𦒄𡤜諪𤧶𦒈𣿯𦔒䯀𦖿𦚵𢜛鑥𥟡憕娧晉侻嚹𤔡𦛼乪𤤴陖涏𦲽㘘襷𦞙𦡮𦐑𦡞營𦣇筂𩃀𠨑𦤦鄄𦤹穅鷰𦧺騦𦨭㙟𦑩𠀡禃𦨴𦭛崬𣔙菏𦮝䛐𦲤画补𦶮墶"],
["9ca1","㜜𢖍𧁋𧇍㱔𧊀𧊅銁𢅺𧊋錰𧋦𤧐氹钟𧑐𠻸蠧裵𢤦𨑳𡞱溸𤨪𡠠㦤㚹尐秣䔿暶𩲭𩢤襃𧟌𧡘囖䃟𡘊㦡𣜯𨃨𡏅熭荦𧧝𩆨婧䲷𧂯𨦫𧧽𧨊𧬋𧵦𤅺筃祾𨀉澵𪋟樃𨌘厢𦸇鎿栶靝𨅯𨀣𦦵𡏭𣈯𨁈嶅𨰰𨂃圕頣𨥉嶫𤦈斾槕叒𤪥𣾁㰑朶𨂐𨃴𨄮𡾡𨅏"],
["9d40","𨆉𨆯𨈚𨌆𨌯𨎊㗊𨑨𨚪䣺揦𨥖砈鉕𨦸䏲𨧧䏟𨧨𨭆𨯔姸𨰉輋𨿅𩃬筑𩄐𩄼㷷𩅞𤫊运犏嚋𩓧𩗩𩖰𩖸𩜲𩣑𩥉𩥪𩧃𩨨𩬎𩵚𩶛纟𩻸𩼣䲤镇𪊓熢𪋿䶑递𪗋䶜𠲜达嗁"],
["9da1","辺𢒰边𤪓䔉繿潖檱仪㓤𨬬𧢝㜺躀𡟵𨀤𨭬𨮙𧨾𦚯㷫𧙕𣲷𥘵𥥖亚𥺁𦉘嚿𠹭踎孭𣺈𤲞揞拐𡟶𡡻攰嘭𥱊吚𥌑㷆𩶘䱽嘢嘞罉𥻘奵𣵀蝰东𠿪𠵉𣚺脗鵞贘瘻鱅癎瞹鍅吲腈苷嘥脲萘肽嗪祢噃吖𠺝㗎嘅嗱曱𨋢㘭甴嗰喺咗啲𠱁𠲖廐𥅈𠹶𢱢"],
["9e40","𠺢麫絚嗞𡁵抝靭咔賍燶酶揼掹揾啩𢭃鱲𢺳冚㓟𠶧冧呍唞唓癦踭𦢊疱肶蠄螆裇膶萜𡃁䓬猄𤜆宐茋𦢓噻𢛴𧴯𤆣𧵳𦻐𧊶酰𡇙鈈𣳼𪚩𠺬𠻹牦𡲢䝎𤿂𧿹𠿫䃺"],
["9ea1","鱝攟𢶠䣳𤟠𩵼𠿬𠸊恢𧖣𠿭"],
["9ead","𦁈𡆇熣纎鵐业丄㕷嬍沲卧㚬㧜卽㚥𤘘墚𤭮舭呋垪𥪕𠥹"],
["9ec5","㩒𢑥獴𩺬䴉鯭𣳾𩼰䱛𤾩𩖞𩿞葜𣶶𧊲𦞳𣜠挮紥𣻷𣸬㨪逈勌㹴㙺䗩𠒎癀嫰𠺶硺𧼮墧䂿噼鮋嵴癔𪐴麅䳡痹㟻愙𣃚𤏲"],
["9ef5","噝𡊩垧𤥣𩸆刴𧂮㖭汊鵼"],
["9f40","籖鬹埞𡝬屓擓𩓐𦌵𧅤蚭𠴨𦴢𤫢𠵱"],
["9f4f","凾𡼏嶎霃𡷑麁遌笟鬂峑箣扨挵髿篏鬪籾鬮籂粆鰕篼鬉鼗鰛𤤾齚啳寃俽麘俲剠㸆勑坧偖妷帒韈鶫轜呩鞴饀鞺匬愰"],
["9fa1","椬叚鰊鴂䰻陁榀傦畆𡝭駚剳"],
["9fae","酙隁酜"],
["9fb2","酑𨺗捿𦴣櫊嘑醎畺抅𠏼獏籰𥰡𣳽"],
["9fc1","𤤙盖鮝个𠳔莾衂"],
["9fc9","届槀僭坺刟巵从氱𠇲伹咜哚劚趂㗾弌㗳"],
["9fdb","歒酼龥鮗頮颴骺麨麄煺笔"],
["9fe7","毺蠘罸"],
["9feb","嘠𪙊蹷齓"],
["9ff0","跔蹏鸜踁抂𨍽踨蹵竓𤩷稾磘泪詧瘇"],
["a040","𨩚鼦泎蟖痃𪊲硓咢贌狢獱謭猂瓱賫𤪻蘯徺袠䒷"],
["a055","𡠻𦸅"],
["a058","詾𢔛"],
["a05b","惽癧髗鵄鍮鮏蟵"],
["a063","蠏賷猬霡鮰㗖犲䰇籑饊𦅙慙䰄麖慽"],
["a073","坟慯抦戹拎㩜懢厪𣏵捤栂㗒"],
["a0a1","嵗𨯂迚𨸹"],
["a0a6","僙𡵆礆匲阸𠼻䁥"],
["a0ae","矾"],
["a0b0","糂𥼚糚稭聦聣絍甅瓲覔舚朌聢𧒆聛瓰脃眤覉𦟌畓𦻑螩蟎臈螌詉貭譃眫瓸蓚㘵榲趦"],
["a0d4","覩瑨涹蟁𤀑瓧㷛煶悤憜㳑煢恷"],
["a0e2","罱𨬭牐惩䭾删㰘𣳇𥻗𧙖𥔱𡥄𡋾𩤃𦷜𧂭峁𦆭𨨏𣙷𠃮𦡆𤼎䕢嬟𦍌齐麦𦉫"],
["a3c0","␀",31,"␡"],
["c6a1","①",9,"⑴",9,"ⅰ",9,"丶丿亅亠冂冖冫勹匸卩厶夊宀巛⼳广廴彐彡攴无疒癶辵隶¨ˆヽヾゝゞ〃仝々〆〇ー［］✽ぁ",23],
["c740","す",58,"ァアィイ"],
["c7a1","ゥ",81,"А",5,"ЁЖ",4],
["c840","Л",26,"ёж",25,"⇧↸↹㇏𠃌乚𠂊刂䒑"],
["c8a1","龰冈龱𧘇"],
["c8cd","￢￤＇＂㈱№℡゛゜⺀⺄⺆⺇⺈⺊⺌⺍⺕⺜⺝⺥⺧⺪⺬⺮⺶⺼⺾⻆⻊⻌⻍⻏⻖⻗⻞⻣"],
["c8f5","ʃɐɛɔɵœøŋʊɪ"],
["f9fe","￭"],
["fa40","𠕇鋛𠗟𣿅蕌䊵珯况㙉𤥂𨧤鍄𡧛苮𣳈砼杄拟𤤳𨦪𠊠𦮳𡌅侫𢓭倈𦴩𧪄𣘀𤪱𢔓倩𠍾徤𠎀𠍇滛𠐟偽儁㑺儎顬㝃萖𤦤𠒇兠𣎴兪𠯿𢃼𠋥𢔰𠖎𣈳𡦃宂蝽𠖳𣲙冲冸"],
["faa1","鴴凉减凑㳜凓𤪦决凢卂凭菍椾𣜭彻刋刦刼劵剗劔効勅簕蕂勠蘍𦬓包𨫞啉滙𣾀𠥔𣿬匳卄𠯢泋𡜦栛珕恊㺪㣌𡛨燝䒢卭却𨚫卾卿𡖖𡘓矦厓𨪛厠厫厮玧𥝲㽙玜叁叅汉义埾叙㪫𠮏叠𣿫𢶣叶𠱷吓灹唫晗浛呭𦭓𠵴啝咏咤䞦𡜍𠻝㶴𠵍"],
["fb40","𨦼𢚘啇䳭启琗喆喩嘅𡣗𤀺䕒𤐵暳𡂴嘷曍𣊊暤暭噍噏磱囱鞇叾圀囯园𨭦㘣𡉏坆𤆥汮炋坂㚱𦱾埦𡐖堃𡑔𤍣堦𤯵塜墪㕡壠壜𡈼壻寿坃𪅐𤉸鏓㖡够梦㛃湙"],
["fba1","𡘾娤啓𡚒蔅姉𠵎𦲁𦴪𡟜姙𡟻𡞲𦶦浱𡠨𡛕姹𦹅媫婣㛦𤦩婷㜈媖瑥嫓𦾡𢕔㶅𡤑㜲𡚸広勐孶斈孼𧨎䀄䡝𠈄寕慠𡨴𥧌𠖥寳宝䴐尅𡭄尓珎尔𡲥𦬨屉䣝岅峩峯嶋𡷹𡸷崐崘嵆𡺤岺巗苼㠭𤤁𢁉𢅳芇㠶㯂帮檊幵幺𤒼𠳓厦亷廐厨𡝱帉廴𨒂"],
["fc40","廹廻㢠廼栾鐛弍𠇁弢㫞䢮𡌺强𦢈𢏐彘𢑱彣鞽𦹮彲鍀𨨶徧嶶㵟𥉐𡽪𧃸𢙨釖𠊞𨨩怱暅𡡷㥣㷇㘹垐𢞴祱㹀悞悤悳𤦂𤦏𧩓璤僡媠慤萤慂慈𦻒憁凴𠙖憇宪𣾷"],
["fca1","𢡟懓𨮝𩥝懐㤲𢦀𢣁怣慜攞掋𠄘担𡝰拕𢸍捬𤧟㨗搸揸𡎎𡟼撐澊𢸶頔𤂌𥜝擡擥鑻㩦携㩗敍漖𤨨𤨣斅敭敟𣁾斵𤥀䬷旑䃘𡠩无旣忟𣐀昘𣇷𣇸晄𣆤𣆥晋𠹵晧𥇦晳晴𡸽𣈱𨗴𣇈𥌓矅𢣷馤朂𤎜𤨡㬫槺𣟂杞杧杢𤇍𩃭柗䓩栢湐鈼栁𣏦𦶠桝"],
["fd40","𣑯槡樋𨫟楳棃𣗍椁椀㴲㨁𣘼㮀枬楡𨩊䋼椶榘㮡𠏉荣傐槹𣙙𢄪橅𣜃檝㯳枱櫈𩆜㰍欝𠤣惞欵歴𢟍溵𣫛𠎵𡥘㝀吡𣭚毡𣻼毜氷𢒋𤣱𦭑汚舦汹𣶼䓅𣶽𤆤𤤌𤤀"],
["fda1","𣳉㛥㳫𠴲鮃𣇹𢒑羏样𦴥𦶡𦷫涖浜湼漄𤥿𤂅𦹲蔳𦽴凇沜渝萮𨬡港𣸯瑓𣾂秌湏媑𣁋濸㜍澝𣸰滺𡒗𤀽䕕鏰潄潜㵎潴𩅰㴻澟𤅄濓𤂑𤅕𤀹𣿰𣾴𤄿凟𤅖𤅗𤅀𦇝灋灾炧炁烌烕烖烟䄄㷨熴熖𤉷焫煅媈煊煮岜𤍥煏鍢𤋁焬𤑚𤨧𤨢熺𨯨炽爎"],
["fe40","鑂爕夑鑃爤鍁𥘅爮牀𤥴梽牕牗㹕𣁄栍漽犂猪猫𤠣𨠫䣭𨠄猨献珏玪𠰺𦨮珉瑉𤇢𡛧𤨤昣㛅𤦷𤦍𤧻珷琕椃𤨦琹𠗃㻗瑜𢢭瑠𨺲瑇珤瑶莹瑬㜰瑴鏱樬璂䥓𤪌"],
["fea1","𤅟𤩹𨮏孆𨰃𡢞瓈𡦈甎瓩甞𨻙𡩋寗𨺬鎅畍畊畧畮𤾂㼄𤴓疎瑝疞疴瘂瘬癑癏癯癶𦏵皐臯㟸𦤑𦤎皡皥皷盌𦾟葢𥂝𥅽𡸜眞眦着撯𥈠睘𣊬瞯𨥤𨥨𡛁矴砉𡍶𤨒棊碯磇磓隥礮𥗠磗礴碱𧘌辸袄𨬫𦂃𢘜禆褀椂禀𥡗禝𧬹礼禩渪𧄦㺨秆𩄍秔"]
]
;

var big5Added$1 = /*#__PURE__*/Object.freeze({
            default: big5Added
});

var require$$0 = getCjsExportFromNamespace(shiftjis$1);

var require$$1 = getCjsExportFromNamespace(eucjp$1);

var require$$2 = getCjsExportFromNamespace(cp936$1);

var require$$3$1 = getCjsExportFromNamespace(gbkAdded$1);

var require$$4 = getCjsExportFromNamespace(gb18030Ranges$1);

var require$$5 = getCjsExportFromNamespace(cp949$1);

var require$$6 = getCjsExportFromNamespace(cp950$1);

var require$$7 = getCjsExportFromNamespace(big5Added$1);

// Tables are not require()-d until they are needed to speed up library load.
// require()-s are direct to support Browserify.


var dbcsData = {
  // == Japanese/ShiftJIS ====================================================
  // All japanese encodings are based on JIS X set of standards:
  // JIS X 0201 - Single-byte encoding of ASCII + ¥ + Kana chars at 0xA1-0xDF.
  // JIS X 0208 - Main set of 6879 characters, placed in 94x94 plane, to be encoded by 2 bytes. 
  //              Has several variations in 1978, 1983, 1990 and 1997.
  // JIS X 0212 - Supplementary plane of 6067 chars in 94x94 plane. 1990. Effectively dead.
  // JIS X 0213 - Extension and modern replacement of 0208 and 0212. Total chars: 11233.
  //              2 planes, first is superset of 0208, second - revised 0212.
  //              Introduced in 2000, revised 2004. Some characters are in Unicode Plane 2 (0x2xxxx)
  // Byte encodings are:
  //  * Shift_JIS: Compatible with 0201, uses not defined chars in top half as lead bytes for double-byte
  //               encoding of 0208. Lead byte ranges: 0x81-0x9F, 0xE0-0xEF; Trail byte ranges: 0x40-0x7E, 0x80-0x9E, 0x9F-0xFC.
  //               Windows CP932 is a superset of Shift_JIS. Some companies added more chars, notably KDDI.
  //  * EUC-JP:    Up to 3 bytes per character. Used mostly on *nixes.
  //               0x00-0x7F       - lower part of 0201
  //               0x8E, 0xA1-0xDF - upper part of 0201
  //               (0xA1-0xFE)x2   - 0208 plane (94x94).
  //               0x8F, (0xA1-0xFE)x2 - 0212 plane (94x94).
  //  * JIS X 208: 7-bit, direct encoding of 0208. Byte ranges: 0x21-0x7E (94 values). Uncommon.
  //               Used as-is in ISO2022 family.
  //  * ISO2022-JP: Stateful encoding, with escape sequences to switch between ASCII, 
  //                0201-1976 Roman, 0208-1978, 0208-1983.
  //  * ISO2022-JP-1: Adds esc seq for 0212-1990.
  //  * ISO2022-JP-2: Adds esc seq for GB2313-1980, KSX1001-1992, ISO8859-1, ISO8859-7.
  //  * ISO2022-JP-3: Adds esc seq for 0201-1976 Kana set, 0213-2000 Planes 1, 2.
  //  * ISO2022-JP-2004: Adds 0213-2004 Plane 1.
  //
  // After JIS X 0213 appeared, Shift_JIS-2004, EUC-JISX0213 and ISO2022-JP-2004 followed, with just changing the planes.
  //
  // Overall, it seems that it's a mess :( http://www8.plala.or.jp/tkubota1/unicode-symbols-map2.html
  'shiftjis': {
    type: '_dbcs',
    table: function table() {
      return require$$0;
    },
    encodeAdd: {
      "\xA5": 0x5C,
      "\u203E": 0x7E
    },
    encodeSkipVals: [{
      from: 0xED40,
      to: 0xF940
    }]
  },
  'csshiftjis': 'shiftjis',
  'mskanji': 'shiftjis',
  'sjis': 'shiftjis',
  'windows31j': 'shiftjis',
  'ms31j': 'shiftjis',
  'xsjis': 'shiftjis',
  'windows932': 'shiftjis',
  'ms932': 'shiftjis',
  '932': 'shiftjis',
  'cp932': 'shiftjis',
  'eucjp': {
    type: '_dbcs',
    table: function table() {
      return require$$1;
    },
    encodeAdd: {
      "\xA5": 0x5C,
      "\u203E": 0x7E
    }
  },
  // TODO: KDDI extension to Shift_JIS
  // TODO: IBM CCSID 942 = CP932, but F0-F9 custom chars and other char changes.
  // TODO: IBM CCSID 943 = Shift_JIS = CP932 with original Shift_JIS lower 128 chars.
  // == Chinese/GBK ==========================================================
  // http://en.wikipedia.org/wiki/GBK
  // We mostly implement W3C recommendation: https://www.w3.org/TR/encoding/#gbk-encoder
  // Oldest GB2312 (1981, ~7600 chars) is a subset of CP936
  'gb2312': 'cp936',
  'gb231280': 'cp936',
  'gb23121980': 'cp936',
  'csgb2312': 'cp936',
  'csiso58gb231280': 'cp936',
  'euccn': 'cp936',
  // Microsoft's CP936 is a subset and approximation of GBK.
  'windows936': 'cp936',
  'ms936': 'cp936',
  '936': 'cp936',
  'cp936': {
    type: '_dbcs',
    table: function table() {
      return require$$2;
    }
  },
  // GBK (~22000 chars) is an extension of CP936 that added user-mapped chars and some other.
  'gbk': {
    type: '_dbcs',
    table: function table() {
      return require$$2.concat(require$$3$1);
    }
  },
  'xgbk': 'gbk',
  'isoir58': 'gbk',
  // GB18030 is an algorithmic extension of GBK.
  // Main source: https://www.w3.org/TR/encoding/#gbk-encoder
  // http://icu-project.org/docs/papers/gb18030.html
  // http://source.icu-project.org/repos/icu/data/trunk/charset/data/xml/gb-18030-2000.xml
  // http://www.khngai.com/chinese/charmap/tblgbk.php?page=0
  'gb18030': {
    type: '_dbcs',
    table: function table() {
      return require$$2.concat(require$$3$1);
    },
    gb18030: function gb18030() {
      return require$$4;
    },
    encodeSkipVals: [0x80],
    encodeAdd: {
      '€': 0xA2E3
    }
  },
  'chinese': 'gb18030',
  // == Korean ===============================================================
  // EUC-KR, KS_C_5601 and KS X 1001 are exactly the same.
  'windows949': 'cp949',
  'ms949': 'cp949',
  '949': 'cp949',
  'cp949': {
    type: '_dbcs',
    table: function table() {
      return require$$5;
    }
  },
  'cseuckr': 'cp949',
  'csksc56011987': 'cp949',
  'euckr': 'cp949',
  'isoir149': 'cp949',
  'korean': 'cp949',
  'ksc56011987': 'cp949',
  'ksc56011989': 'cp949',
  'ksc5601': 'cp949',
  // == Big5/Taiwan/Hong Kong ================================================
  // There are lots of tables for Big5 and cp950. Please see the following links for history:
  // http://moztw.org/docs/big5/  http://www.haible.de/bruno/charsets/conversion-tables/Big5.html
  // Variations, in roughly number of defined chars:
  //  * Windows CP 950: Microsoft variant of Big5. Canonical: http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP950.TXT
  //  * Windows CP 951: Microsoft variant of Big5-HKSCS-2001. Seems to be never public. http://me.abelcheung.org/articles/research/what-is-cp951/
  //  * Big5-2003 (Taiwan standard) almost superset of cp950.
  //  * Unicode-at-on (UAO) / Mozilla 1.8. Falling out of use on the Web. Not supported by other browsers.
  //  * Big5-HKSCS (-2001, -2004, -2008). Hong Kong standard. 
  //    many unicode code points moved from PUA to Supplementary plane (U+2XXXX) over the years.
  //    Plus, it has 4 combining sequences.
  //    Seems that Mozilla refused to support it for 10 yrs. https://bugzilla.mozilla.org/show_bug.cgi?id=162431 https://bugzilla.mozilla.org/show_bug.cgi?id=310299
  //    because big5-hkscs is the only encoding to include astral characters in non-algorithmic way.
  //    Implementations are not consistent within browsers; sometimes labeled as just big5.
  //    MS Internet Explorer switches from big5 to big5-hkscs when a patch applied.
  //    Great discussion & recap of what's going on https://bugzilla.mozilla.org/show_bug.cgi?id=912470#c31
  //    In the encoder, it might make sense to support encoding old PUA mappings to Big5 bytes seq-s.
  //    Official spec: http://www.ogcio.gov.hk/en/business/tech_promotion/ccli/terms/doc/2003cmp_2008.txt
  //                   http://www.ogcio.gov.hk/tc/business/tech_promotion/ccli/terms/doc/hkscs-2008-big5-iso.txt
  // 
  // Current understanding of how to deal with Big5(-HKSCS) is in the Encoding Standard, http://encoding.spec.whatwg.org/#big5-encoder
  // Unicode mapping (http://www.unicode.org/Public/MAPPINGS/OBSOLETE/EASTASIA/OTHER/BIG5.TXT) is said to be wrong.
  'windows950': 'cp950',
  'ms950': 'cp950',
  '950': 'cp950',
  'cp950': {
    type: '_dbcs',
    table: function table() {
      return require$$6;
    }
  },
  // Big5 has many variations and is an extension of cp950. We use Encoding Standard's as a consensus.
  'big5': 'big5hkscs',
  'big5hkscs': {
    type: '_dbcs',
    table: function table() {
      return require$$6.concat(require$$7);
    },
    encodeSkipVals: [0xa2cc]
  },
  'cnbig5': 'big5hkscs',
  'csbig5': 'big5hkscs',
  'xxbig5': 'big5hkscs'
};

var encodings = createCommonjsModule(function (module, exports) {
  // We support Browserify by skipping automatic module discovery and requiring modules directly.

  var modules = [internal, utf16, utf7, sbcsCodec, sbcsData, sbcsDataGenerated, dbcsCodec, dbcsData]; // Put all encoding/alias/codec definitions to single object and export it. 

  for (var i = 0; i < modules.length; i++) {
    var module = modules[i];

    for (var enc in module) {
      if (Object.prototype.hasOwnProperty.call(module, enc)) exports[enc] = module[enc];
    }
  }
});

var Buffer$7 = bufferEs6.Buffer,
    Transform$1 = Stream.Transform; // == Exports ==================================================================

var streams = function streams(iconv) {
  // Additional Public API.
  iconv.encodeStream = function encodeStream(encoding, options) {
    return new IconvLiteEncoderStream(iconv.getEncoder(encoding, options), options);
  };

  iconv.decodeStream = function decodeStream(encoding, options) {
    return new IconvLiteDecoderStream(iconv.getDecoder(encoding, options), options);
  };

  iconv.supportsStreams = true; // Not published yet.

  iconv.IconvLiteEncoderStream = IconvLiteEncoderStream;
  iconv.IconvLiteDecoderStream = IconvLiteDecoderStream;
  iconv._collect = IconvLiteDecoderStream.prototype.collect;
}; // == Encoder stream =======================================================


function IconvLiteEncoderStream(conv, options) {
  this.conv = conv;
  options = options || {};
  options.decodeStrings = false; // We accept only strings, so we don't need to decode them.

  Transform$1.call(this, options);
}

IconvLiteEncoderStream.prototype = Object.create(Transform$1.prototype, {
  constructor: {
    value: IconvLiteEncoderStream
  }
});

IconvLiteEncoderStream.prototype._transform = function (chunk, encoding, done) {
  if (typeof chunk != 'string') return done(new Error("Iconv encoding stream needs strings as its input."));

  try {
    var res = this.conv.write(chunk);
    if (res && res.length) this.push(res);
    done();
  } catch (e) {
    done(e);
  }
};

IconvLiteEncoderStream.prototype._flush = function (done) {
  try {
    var res = this.conv.end();
    if (res && res.length) this.push(res);
    done();
  } catch (e) {
    done(e);
  }
};

IconvLiteEncoderStream.prototype.collect = function (cb) {
  var chunks = [];
  this.on('error', cb);
  this.on('data', function (chunk) {
    chunks.push(chunk);
  });
  this.on('end', function () {
    cb(null, Buffer$7.concat(chunks));
  });
  return this;
}; // == Decoder stream =======================================================


function IconvLiteDecoderStream(conv, options) {
  this.conv = conv;
  options = options || {};
  options.encoding = this.encoding = 'utf8'; // We output strings.

  Transform$1.call(this, options);
}

IconvLiteDecoderStream.prototype = Object.create(Transform$1.prototype, {
  constructor: {
    value: IconvLiteDecoderStream
  }
});

IconvLiteDecoderStream.prototype._transform = function (chunk, encoding, done) {
  if (!Buffer$7.isBuffer(chunk)) return done(new Error("Iconv decoding stream needs buffers as its input."));

  try {
    var res = this.conv.write(chunk);
    if (res && res.length) this.push(res, this.encoding);
    done();
  } catch (e) {
    done(e);
  }
};

IconvLiteDecoderStream.prototype._flush = function (done) {
  try {
    var res = this.conv.end();
    if (res && res.length) this.push(res, this.encoding);
    done();
  } catch (e) {
    done(e);
  }
};

IconvLiteDecoderStream.prototype.collect = function (cb) {
  var res = '';
  this.on('error', cb);
  this.on('data', function (chunk) {
    res += chunk;
  });
  this.on('end', function () {
    cb(null, res);
  });
  return this;
};

var Buffer$8 = bufferEs6.Buffer; // Note: not polyfilled with safer-buffer on a purpose, as overrides Buffer
// == Extend Node primitives to use iconv-lite =================================

var extendNode = function extendNode(iconv) {
  var original = undefined; // Place to keep original methods.
  // Node authors rewrote Buffer internals to make it compatible with
  // Uint8Array and we cannot patch key functions since then.
  // Note: this does use older Buffer API on a purpose

  iconv.supportsNodeEncodingsExtension = !(Buffer$8.from || new Buffer$8(0) instanceof Uint8Array);

  iconv.extendNodeEncodings = function extendNodeEncodings() {
    if (original) return;
    original = {};

    if (!iconv.supportsNodeEncodingsExtension) {
      console.error("ACTION NEEDED: require('iconv-lite').extendNodeEncodings() is not supported in your version of Node");
      console.error("See more info at https://github.com/ashtuchkin/iconv-lite/wiki/Node-v4-compatibility");
      return;
    }

    var nodeNativeEncodings = {
      'hex': true,
      'utf8': true,
      'utf-8': true,
      'ascii': true,
      'binary': true,
      'base64': true,
      'ucs2': true,
      'ucs-2': true,
      'utf16le': true,
      'utf-16le': true
    };

    Buffer$8.isNativeEncoding = function (enc) {
      return enc && nodeNativeEncodings[enc.toLowerCase()];
    }; // -- SlowBuffer -----------------------------------------------------------


    var SlowBuffer = bufferEs6.SlowBuffer;
    original.SlowBufferToString = SlowBuffer.prototype.toString;

    SlowBuffer.prototype.toString = function (encoding, start, end) {
      encoding = String(encoding || 'utf8').toLowerCase(); // Use native conversion when possible

      if (Buffer$8.isNativeEncoding(encoding)) return original.SlowBufferToString.call(this, encoding, start, end); // Otherwise, use our decoding method.

      if (typeof start == 'undefined') start = 0;
      if (typeof end == 'undefined') end = this.length;
      return iconv.decode(this.slice(start, end), encoding);
    };

    original.SlowBufferWrite = SlowBuffer.prototype.write;

    SlowBuffer.prototype.write = function (string, offset, length, encoding) {
      // Support both (string, offset, length, encoding)
      // and the legacy (string, encoding, offset, length)
      if (isFinite(offset)) {
        if (!isFinite(length)) {
          encoding = length;
          length = undefined;
        }
      } else {
        // legacy
        var swap = encoding;
        encoding = offset;
        offset = length;
        length = swap;
      }

      offset = +offset || 0;
      var remaining = this.length - offset;

      if (!length) {
        length = remaining;
      } else {
        length = +length;

        if (length > remaining) {
          length = remaining;
        }
      }

      encoding = String(encoding || 'utf8').toLowerCase(); // Use native conversion when possible

      if (Buffer$8.isNativeEncoding(encoding)) return original.SlowBufferWrite.call(this, string, offset, length, encoding);
      if (string.length > 0 && (length < 0 || offset < 0)) throw new RangeError('attempt to write beyond buffer bounds'); // Otherwise, use our encoding method.

      var buf = iconv.encode(string, encoding);
      if (buf.length < length) length = buf.length;
      buf.copy(this, offset, 0, length);
      return length;
    }; // -- Buffer ---------------------------------------------------------------


    original.BufferIsEncoding = Buffer$8.isEncoding;

    Buffer$8.isEncoding = function (encoding) {
      return Buffer$8.isNativeEncoding(encoding) || iconv.encodingExists(encoding);
    };

    original.BufferByteLength = Buffer$8.byteLength;

    Buffer$8.byteLength = SlowBuffer.byteLength = function (str, encoding) {
      encoding = String(encoding || 'utf8').toLowerCase(); // Use native conversion when possible

      if (Buffer$8.isNativeEncoding(encoding)) return original.BufferByteLength.call(this, str, encoding); // Slow, I know, but we don't have a better way yet.

      return iconv.encode(str, encoding).length;
    };

    original.BufferToString = Buffer$8.prototype.toString;

    Buffer$8.prototype.toString = function (encoding, start, end) {
      encoding = String(encoding || 'utf8').toLowerCase(); // Use native conversion when possible

      if (Buffer$8.isNativeEncoding(encoding)) return original.BufferToString.call(this, encoding, start, end); // Otherwise, use our decoding method.

      if (typeof start == 'undefined') start = 0;
      if (typeof end == 'undefined') end = this.length;
      return iconv.decode(this.slice(start, end), encoding);
    };

    original.BufferWrite = Buffer$8.prototype.write;

    Buffer$8.prototype.write = function (string, offset, length, encoding) {
      var _offset = offset,
          _length = length,
          _encoding = encoding; // Support both (string, offset, length, encoding)
      // and the legacy (string, encoding, offset, length)

      if (isFinite(offset)) {
        if (!isFinite(length)) {
          encoding = length;
          length = undefined;
        }
      } else {
        // legacy
        var swap = encoding;
        encoding = offset;
        offset = length;
        length = swap;
      }

      encoding = String(encoding || 'utf8').toLowerCase(); // Use native conversion when possible

      if (Buffer$8.isNativeEncoding(encoding)) return original.BufferWrite.call(this, string, _offset, _length, _encoding);
      offset = +offset || 0;
      var remaining = this.length - offset;

      if (!length) {
        length = remaining;
      } else {
        length = +length;

        if (length > remaining) {
          length = remaining;
        }
      }

      if (string.length > 0 && (length < 0 || offset < 0)) throw new RangeError('attempt to write beyond buffer bounds'); // Otherwise, use our encoding method.

      var buf = iconv.encode(string, encoding);
      if (buf.length < length) length = buf.length;
      buf.copy(this, offset, 0, length);
      return length; // TODO: Set _charsWritten.
    }; // -- Readable -------------------------------------------------------------


    if (iconv.supportsStreams) {
      var Readable = Stream.Readable;
      original.ReadableSetEncoding = Readable.prototype.setEncoding;

      Readable.prototype.setEncoding = function setEncoding(enc, options) {
        // Use our own decoder, it has the same interface.
        // We cannot use original function as it doesn't handle BOM-s.
        this._readableState.decoder = iconv.getDecoder(enc, options);
        this._readableState.encoding = enc;
      };

      Readable.prototype.collect = iconv._collect;
    }
  }; // Remove iconv-lite Node primitive extensions.


  iconv.undoExtendNodeEncodings = function undoExtendNodeEncodings() {
    if (!iconv.supportsNodeEncodingsExtension) return;
    if (!original) throw new Error("require('iconv-lite').undoExtendNodeEncodings(): Nothing to undo; extendNodeEncodings() is not called.");
    delete Buffer$8.isNativeEncoding;
    var SlowBuffer = bufferEs6.SlowBuffer;
    SlowBuffer.prototype.toString = original.SlowBufferToString;
    SlowBuffer.prototype.write = original.SlowBufferWrite;
    Buffer$8.isEncoding = original.BufferIsEncoding;
    Buffer$8.byteLength = original.BufferByteLength;
    Buffer$8.prototype.toString = original.BufferToString;
    Buffer$8.prototype.write = original.BufferWrite;

    if (iconv.supportsStreams) {
      var Readable = Stream.Readable;
      Readable.prototype.setEncoding = original.ReadableSetEncoding;
      delete Readable.prototype.collect;
    }

    original = undefined;
  };
};

var lib = createCommonjsModule(function (module) {
  // Solution would be installing npm modules "buffer" and "stream" explicitly.

  var Buffer = safer_1.Buffer;
  var iconv = module.exports; // All codecs and aliases are kept here, keyed by encoding name/alias.
  // They are lazy loaded in `iconv.getCodec` from `encodings/index.js`.

  iconv.encodings = null; // Characters emitted in case of error.

  iconv.defaultCharUnicode = '�';
  iconv.defaultCharSingleByte = '?'; // Public API.

  iconv.encode = function encode(str, encoding, options) {
    str = "" + (str || ""); // Ensure string.

    var encoder = iconv.getEncoder(encoding, options);
    var res = encoder.write(str);
    var trail = encoder.end();
    return trail && trail.length > 0 ? Buffer.concat([res, trail]) : res;
  };

  iconv.decode = function decode(buf, encoding, options) {
    if (typeof buf === 'string') {
      if (!iconv.skipDecodeWarning) {
        console.error('Iconv-lite warning: decode()-ing strings is deprecated. Refer to https://github.com/ashtuchkin/iconv-lite/wiki/Use-Buffers-when-decoding');
        iconv.skipDecodeWarning = true;
      }

      buf = Buffer.from("" + (buf || ""), "binary"); // Ensure buffer.
    }

    var decoder = iconv.getDecoder(encoding, options);
    var res = decoder.write(buf);
    var trail = decoder.end();
    return trail ? res + trail : res;
  };

  iconv.encodingExists = function encodingExists(enc) {
    try {
      iconv.getCodec(enc);
      return true;
    } catch (e) {
      return false;
    }
  }; // Legacy aliases to convert functions


  iconv.toEncoding = iconv.encode;
  iconv.fromEncoding = iconv.decode; // Search for a codec in iconv.encodings. Cache codec data in iconv._codecDataCache.

  iconv._codecDataCache = {};

  iconv.getCodec = function getCodec(encoding) {
    if (!iconv.encodings) iconv.encodings = encodings; // Lazy load all encoding definitions.
    // Canonicalize encoding name: strip all non-alphanumeric chars and appended year.

    var enc = iconv._canonicalizeEncoding(encoding); // Traverse iconv.encodings to find actual codec.


    var codecOptions = {};

    while (true) {
      var codec = iconv._codecDataCache[enc];
      if (codec) return codec;
      var codecDef = iconv.encodings[enc];

      switch (typeof codecDef) {
        case "string":
          // Direct alias to other encoding.
          enc = codecDef;
          break;

        case "object":
          // Alias with options. Can be layered.
          for (var key in codecDef) {
            codecOptions[key] = codecDef[key];
          }

          if (!codecOptions.encodingName) codecOptions.encodingName = enc;
          enc = codecDef.type;
          break;

        case "function":
          // Codec itself.
          if (!codecOptions.encodingName) codecOptions.encodingName = enc; // The codec function must load all tables and return object with .encoder and .decoder methods.
          // It'll be called only once (for each different options object).

          codec = new codecDef(codecOptions, iconv);
          iconv._codecDataCache[codecOptions.encodingName] = codec; // Save it to be reused later.

          return codec;

        default:
          throw new Error("Encoding not recognized: '" + encoding + "' (searched as: '" + enc + "')");
      }
    }
  };

  iconv._canonicalizeEncoding = function (encoding) {
    // Canonicalize encoding name: strip all non-alphanumeric chars and appended year.
    return ('' + encoding).toLowerCase().replace(/:\d{4}$|[^0-9a-z]/g, "");
  };

  iconv.getEncoder = function getEncoder(encoding, options) {
    var codec = iconv.getCodec(encoding),
        encoder = new codec.encoder(options, codec);
    if (codec.bomAware && options && options.addBOM) encoder = new bomHandling.PrependBOM(encoder, options);
    return encoder;
  };

  iconv.getDecoder = function getDecoder(encoding, options) {
    var codec = iconv.getCodec(encoding),
        decoder = new codec.decoder(options, codec);
    if (codec.bomAware && !(options && options.stripBOM === false)) decoder = new bomHandling.StripBOM(decoder, options);
    return decoder;
  }; // Load extensions in Node. All of them are omitted in Browserify build via 'browser' field in package.json.


  var nodeVer = typeof process !== 'undefined' && process.versions && process.versions.node;

  if (nodeVer) {
    // Load streaming support in Node v0.10+
    var nodeVerArr = nodeVer.split(".").map(Number);

    if (nodeVerArr[0] > 0 || nodeVerArr[1] >= 10) {
      streams(iconv);
    } // Load Node primitive extensions.


    extendNode(iconv);
  }
});

var DecodeStream = createCommonjsModule(function (module) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var DecodeStream, iconv;

    try {
      iconv = lib;
    } catch (_error) {}

    DecodeStream = function () {
      var key;

      function DecodeStream(buffer) {
        this.buffer = buffer;
        this.pos = 0;
        this.length = this.buffer.length;
      }

      DecodeStream.TYPES = {
        UInt8: 1,
        UInt16: 2,
        UInt24: 3,
        UInt32: 4,
        Int8: 1,
        Int16: 2,
        Int24: 3,
        Int32: 4,
        Float: 4,
        Double: 8
      };

      for (key in Buffer.prototype) {
        if (key.slice(0, 4) === 'read') {
          (function (key) {
            var bytes;
            bytes = DecodeStream.TYPES[key.replace(/read|[BL]E/g, '')];
            return DecodeStream.prototype[key] = function () {
              var ret;
              ret = this.buffer[key](this.pos);
              this.pos += bytes;
              return ret;
            };
          })(key);
        }
      }

      DecodeStream.prototype.readString = function (length, encoding) {
        var buf, byte, i, _i, _ref;

        if (encoding == null) {
          encoding = 'ascii';
        }

        switch (encoding) {
          case 'utf16le':
          case 'ucs2':
          case 'utf8':
          case 'ascii':
            return this.buffer.toString(encoding, this.pos, this.pos += length);

          case 'utf16be':
            buf = new Buffer(this.readBuffer(length));

            for (i = _i = 0, _ref = buf.length - 1; _i < _ref; i = _i += 2) {
              byte = buf[i];
              buf[i] = buf[i + 1];
              buf[i + 1] = byte;
            }

            return buf.toString('utf16le');

          default:
            buf = this.readBuffer(length);

            if (iconv) {
              try {
                return iconv.decode(buf, encoding);
              } catch (_error) {}
            }

            return buf;
        }
      };

      DecodeStream.prototype.readBuffer = function (length) {
        return this.buffer.slice(this.pos, this.pos += length);
      };

      DecodeStream.prototype.readUInt24BE = function () {
        return (this.readUInt16BE() << 8) + this.readUInt8();
      };

      DecodeStream.prototype.readUInt24LE = function () {
        return this.readUInt16LE() + (this.readUInt8() << 16);
      };

      DecodeStream.prototype.readInt24BE = function () {
        return (this.readInt16BE() << 8) + this.readUInt8();
      };

      DecodeStream.prototype.readInt24LE = function () {
        return this.readUInt16LE() + (this.readInt8() << 16);
      };

      return DecodeStream;
    }();

    module.exports = DecodeStream;
  }).call(commonjsGlobal);
});

var EncodeStream = createCommonjsModule(function (module) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var DecodeStream$$1,
        EncodeStream,
        iconv,
        stream,
        __hasProp = {}.hasOwnProperty,
        __extends = function __extends(child, parent) {
      for (var key in parent) {
        if (__hasProp.call(parent, key)) child[key] = parent[key];
      }

      function ctor() {
        this.constructor = child;
      }

      ctor.prototype = parent.prototype;
      child.prototype = new ctor();
      child.__super__ = parent.prototype;
      return child;
    };

    stream = Stream;
    DecodeStream$$1 = DecodeStream;

    try {
      iconv = lib;
    } catch (_error) {}

    EncodeStream = function (_super) {
      var key;

      __extends(EncodeStream, _super);

      function EncodeStream(bufferSize) {
        if (bufferSize == null) {
          bufferSize = 65536;
        }

        EncodeStream.__super__.constructor.apply(this, arguments);

        this.buffer = new Buffer(bufferSize);
        this.bufferOffset = 0;
        this.pos = 0;
      }

      for (key in Buffer.prototype) {
        if (key.slice(0, 5) === 'write') {
          (function (key) {
            var bytes;
            bytes = +DecodeStream$$1.TYPES[key.replace(/write|[BL]E/g, '')];
            return EncodeStream.prototype[key] = function (value) {
              this.ensure(bytes);
              this.buffer[key](value, this.bufferOffset);
              this.bufferOffset += bytes;
              return this.pos += bytes;
            };
          })(key);
        }
      }

      EncodeStream.prototype._read = function () {};

      EncodeStream.prototype.ensure = function (bytes) {
        if (this.bufferOffset + bytes > this.buffer.length) {
          return this.flush();
        }
      };

      EncodeStream.prototype.flush = function () {
        if (this.bufferOffset > 0) {
          this.push(new Buffer(this.buffer.slice(0, this.bufferOffset)));
          return this.bufferOffset = 0;
        }
      };

      EncodeStream.prototype.writeBuffer = function (buffer) {
        this.flush();
        this.push(buffer);
        return this.pos += buffer.length;
      };

      EncodeStream.prototype.writeString = function (string, encoding) {
        var buf, byte, i, _i, _ref;

        if (encoding == null) {
          encoding = 'ascii';
        }

        switch (encoding) {
          case 'utf16le':
          case 'ucs2':
          case 'utf8':
          case 'ascii':
            return this.writeBuffer(new Buffer(string, encoding));

          case 'utf16be':
            buf = new Buffer(string, 'utf16le');

            for (i = _i = 0, _ref = buf.length - 1; _i < _ref; i = _i += 2) {
              byte = buf[i];
              buf[i] = buf[i + 1];
              buf[i + 1] = byte;
            }

            return this.writeBuffer(buf);

          default:
            if (iconv) {
              return this.writeBuffer(iconv.encode(string, encoding));
            } else {
              throw new Error('Install iconv-lite to enable additional string encodings.');
            }

        }
      };

      EncodeStream.prototype.writeUInt24BE = function (val) {
        this.ensure(3);
        this.buffer[this.bufferOffset++] = val >>> 16 & 0xff;
        this.buffer[this.bufferOffset++] = val >>> 8 & 0xff;
        this.buffer[this.bufferOffset++] = val & 0xff;
        return this.pos += 3;
      };

      EncodeStream.prototype.writeUInt24LE = function (val) {
        this.ensure(3);
        this.buffer[this.bufferOffset++] = val & 0xff;
        this.buffer[this.bufferOffset++] = val >>> 8 & 0xff;
        this.buffer[this.bufferOffset++] = val >>> 16 & 0xff;
        return this.pos += 3;
      };

      EncodeStream.prototype.writeInt24BE = function (val) {
        if (val >= 0) {
          return this.writeUInt24BE(val);
        } else {
          return this.writeUInt24BE(val + 0xffffff + 1);
        }
      };

      EncodeStream.prototype.writeInt24LE = function (val) {
        if (val >= 0) {
          return this.writeUInt24LE(val);
        } else {
          return this.writeUInt24LE(val + 0xffffff + 1);
        }
      };

      EncodeStream.prototype.fill = function (val, length) {
        var buf;

        if (length < this.buffer.length) {
          this.ensure(length);
          this.buffer.fill(val, this.bufferOffset, this.bufferOffset + length);
          this.bufferOffset += length;
          return this.pos += length;
        } else {
          buf = new Buffer(length);
          buf.fill(val);
          return this.writeBuffer(buf);
        }
      };

      EncodeStream.prototype.end = function () {
        this.flush();
        return this.push(null);
      };

      return EncodeStream;
    }(stream.Readable);

    module.exports = EncodeStream;
  }).call(commonjsGlobal);
});

var _Number = createCommonjsModule(function (module, exports) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var DecodeStream$$1,
        Fixed,
        NumberT,
        __hasProp = {}.hasOwnProperty,
        __extends = function __extends(child, parent) {
      for (var key in parent) {
        if (__hasProp.call(parent, key)) child[key] = parent[key];
      }

      function ctor() {
        this.constructor = child;
      }

      ctor.prototype = parent.prototype;
      child.prototype = new ctor();
      child.__super__ = parent.prototype;
      return child;
    };

    DecodeStream$$1 = DecodeStream;

    NumberT = function () {
      function NumberT(type, endian) {
        this.type = type;
        this.endian = endian != null ? endian : 'BE';
        this.fn = this.type;

        if (this.type[this.type.length - 1] !== '8') {
          this.fn += this.endian;
        }
      }

      NumberT.prototype.size = function () {
        return DecodeStream$$1.TYPES[this.type];
      };

      NumberT.prototype.decode = function (stream) {
        return stream['read' + this.fn]();
      };

      NumberT.prototype.encode = function (stream, val) {
        return stream['write' + this.fn](val);
      };

      return NumberT;
    }();

    exports.Number = NumberT;
    exports.uint8 = new NumberT('UInt8');
    exports.uint16be = exports.uint16 = new NumberT('UInt16', 'BE');
    exports.uint16le = new NumberT('UInt16', 'LE');
    exports.uint24be = exports.uint24 = new NumberT('UInt24', 'BE');
    exports.uint24le = new NumberT('UInt24', 'LE');
    exports.uint32be = exports.uint32 = new NumberT('UInt32', 'BE');
    exports.uint32le = new NumberT('UInt32', 'LE');
    exports.int8 = new NumberT('Int8');
    exports.int16be = exports.int16 = new NumberT('Int16', 'BE');
    exports.int16le = new NumberT('Int16', 'LE');
    exports.int24be = exports.int24 = new NumberT('Int24', 'BE');
    exports.int24le = new NumberT('Int24', 'LE');
    exports.int32be = exports.int32 = new NumberT('Int32', 'BE');
    exports.int32le = new NumberT('Int32', 'LE');
    exports.floatbe = exports.float = new NumberT('Float', 'BE');
    exports.floatle = new NumberT('Float', 'LE');
    exports.doublebe = exports.double = new NumberT('Double', 'BE');
    exports.doublele = new NumberT('Double', 'LE');

    Fixed = function (_super) {
      __extends(Fixed, _super);

      function Fixed(size, endian, fracBits) {
        if (fracBits == null) {
          fracBits = size >> 1;
        }

        Fixed.__super__.constructor.call(this, "Int" + size, endian);

        this._point = 1 << fracBits;
      }

      Fixed.prototype.decode = function (stream) {
        return Fixed.__super__.decode.call(this, stream) / this._point;
      };

      Fixed.prototype.encode = function (stream, val) {
        return Fixed.__super__.encode.call(this, stream, val * this._point | 0);
      };

      return Fixed;
    }(NumberT);

    exports.Fixed = Fixed;
    exports.fixed16be = exports.fixed16 = new Fixed(16, 'BE');
    exports.fixed16le = new Fixed(16, 'LE');
    exports.fixed32be = exports.fixed32 = new Fixed(32, 'BE');
    exports.fixed32le = new Fixed(32, 'LE');
  }).call(commonjsGlobal);
});
var _Number_1 = _Number.Number;
var _Number_2 = _Number.uint8;
var _Number_3 = _Number.uint16be;
var _Number_4 = _Number.uint16;
var _Number_5 = _Number.uint16le;
var _Number_6 = _Number.uint24be;
var _Number_7 = _Number.uint24;
var _Number_8 = _Number.uint24le;
var _Number_9 = _Number.uint32be;
var _Number_10 = _Number.uint32;
var _Number_11 = _Number.uint32le;
var _Number_12 = _Number.int8;
var _Number_13 = _Number.int16be;
var _Number_14 = _Number.int16;
var _Number_15 = _Number.int16le;
var _Number_16 = _Number.int24be;
var _Number_17 = _Number.int24;
var _Number_18 = _Number.int24le;
var _Number_19 = _Number.int32be;
var _Number_20 = _Number.int32;
var _Number_21 = _Number.int32le;
var _Number_22 = _Number.floatbe;
var _Number_23 = _Number.floatle;
var _Number_24 = _Number.doublebe;
var _Number_25 = _Number.doublele;
var _Number_26 = _Number.Fixed;
var _Number_27 = _Number.fixed16be;
var _Number_28 = _Number.fixed16;
var _Number_29 = _Number.fixed16le;
var _Number_30 = _Number.fixed32be;
var _Number_31 = _Number.fixed32;
var _Number_32 = _Number.fixed32le;

var utils = createCommonjsModule(function (module, exports) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var NumberT, PropertyDescriptor;
    NumberT = _Number.Number;

    exports.resolveLength = function (length, stream, parent) {
      var res;

      if (typeof length === 'number') {
        res = length;
      } else if (typeof length === 'function') {
        res = length.call(parent, parent);
      } else if (parent && typeof length === 'string') {
        res = parent[length];
      } else if (stream && length instanceof NumberT) {
        res = length.decode(stream);
      }

      if (isNaN(res)) {
        throw new Error('Not a fixed size');
      }

      return res;
    };

    PropertyDescriptor = function () {
      function PropertyDescriptor(opts) {
        var key, val;

        if (opts == null) {
          opts = {};
        }

        this.enumerable = true;
        this.configurable = true;

        for (key in opts) {
          val = opts[key];
          this[key] = val;
        }
      }

      return PropertyDescriptor;
    }();

    exports.PropertyDescriptor = PropertyDescriptor;
  }).call(commonjsGlobal);
});
var utils_1 = utils.resolveLength;
var utils_2 = utils.PropertyDescriptor;

var _Array = createCommonjsModule(function (module) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var ArrayT, NumberT, utils$$1;
    NumberT = _Number.Number;
    utils$$1 = utils;

    ArrayT = function () {
      function ArrayT(type, length, lengthType) {
        this.type = type;
        this.length = length;
        this.lengthType = lengthType != null ? lengthType : 'count';
      }

      ArrayT.prototype.decode = function (stream, parent) {
        var ctx, i, length, pos, res, target, _i;

        pos = stream.pos;
        res = [];
        ctx = parent;

        if (this.length != null) {
          length = utils$$1.resolveLength(this.length, stream, parent);
        }

        if (this.length instanceof NumberT) {
          Object.defineProperties(res, {
            parent: {
              value: parent
            },
            _startOffset: {
              value: pos
            },
            _currentOffset: {
              value: 0,
              writable: true
            },
            _length: {
              value: length
            }
          });
          ctx = res;
        }

        if (length == null || this.lengthType === 'bytes') {
          target = length != null ? stream.pos + length : (parent != null ? parent._length : void 0) ? parent._startOffset + parent._length : stream.length;

          while (stream.pos < target) {
            res.push(this.type.decode(stream, ctx));
          }
        } else {
          for (i = _i = 0; _i < length; i = _i += 1) {
            res.push(this.type.decode(stream, ctx));
          }
        }

        return res;
      };

      ArrayT.prototype.size = function (array, ctx) {
        var item, size, _i, _len;

        if (!array) {
          return this.type.size(null, ctx) * utils$$1.resolveLength(this.length, null, ctx);
        }

        size = 0;

        if (this.length instanceof NumberT) {
          size += this.length.size();
          ctx = {
            parent: ctx
          };
        }

        for (_i = 0, _len = array.length; _i < _len; _i++) {
          item = array[_i];
          size += this.type.size(item, ctx);
        }

        return size;
      };

      ArrayT.prototype.encode = function (stream, array, parent) {
        var ctx, i, item, ptr, _i, _len;

        ctx = parent;

        if (this.length instanceof NumberT) {
          ctx = {
            pointers: [],
            startOffset: stream.pos,
            parent: parent
          };
          ctx.pointerOffset = stream.pos + this.size(array, ctx);
          this.length.encode(stream, array.length);
        }

        for (_i = 0, _len = array.length; _i < _len; _i++) {
          item = array[_i];
          this.type.encode(stream, item, ctx);
        }

        if (this.length instanceof NumberT) {
          i = 0;

          while (i < ctx.pointers.length) {
            ptr = ctx.pointers[i++];
            ptr.type.encode(stream, ptr.val);
          }
        }
      };

      return ArrayT;
    }();

    module.exports = ArrayT;
  }).call(commonjsGlobal);
});

var LazyArray = createCommonjsModule(function (module) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var ArrayT,
        LazyArray,
        LazyArrayT,
        NumberT,
        inspect,
        utils$$1,
        __hasProp = {}.hasOwnProperty,
        __extends = function __extends(child, parent) {
      for (var key in parent) {
        if (__hasProp.call(parent, key)) child[key] = parent[key];
      }

      function ctor() {
        this.constructor = child;
      }

      ctor.prototype = parent.prototype;
      child.prototype = new ctor();
      child.__super__ = parent.prototype;
      return child;
    };

    ArrayT = _Array;
    NumberT = _Number.Number;
    utils$$1 = utils;
    inspect = require$$3.inspect;

    LazyArrayT = function (_super) {
      __extends(LazyArrayT, _super);

      function LazyArrayT() {
        return LazyArrayT.__super__.constructor.apply(this, arguments);
      }

      LazyArrayT.prototype.decode = function (stream, parent) {
        var length, pos, res;
        pos = stream.pos;
        length = utils$$1.resolveLength(this.length, stream, parent);

        if (this.length instanceof NumberT) {
          parent = {
            parent: parent,
            _startOffset: pos,
            _currentOffset: 0,
            _length: length
          };
        }

        res = new LazyArray(this.type, length, stream, parent);
        stream.pos += length * this.type.size(null, parent);
        return res;
      };

      LazyArrayT.prototype.size = function (val, ctx) {
        if (val instanceof LazyArray) {
          val = val.toArray();
        }

        return LazyArrayT.__super__.size.call(this, val, ctx);
      };

      LazyArrayT.prototype.encode = function (stream, val, ctx) {
        if (val instanceof LazyArray) {
          val = val.toArray();
        }

        return LazyArrayT.__super__.encode.call(this, stream, val, ctx);
      };

      return LazyArrayT;
    }(ArrayT);

    LazyArray = function () {
      function LazyArray(type, length, stream, ctx) {
        this.type = type;
        this.length = length;
        this.stream = stream;
        this.ctx = ctx;
        this.base = this.stream.pos;
        this.items = [];
      }

      LazyArray.prototype.get = function (index) {
        var pos;

        if (index < 0 || index >= this.length) {
          return void 0;
        }

        if (this.items[index] == null) {
          pos = this.stream.pos;
          this.stream.pos = this.base + this.type.size(null, this.ctx) * index;
          this.items[index] = this.type.decode(this.stream, this.ctx);
          this.stream.pos = pos;
        }

        return this.items[index];
      };

      LazyArray.prototype.toArray = function () {
        var i, _i, _ref, _results;

        _results = [];

        for (i = _i = 0, _ref = this.length; _i < _ref; i = _i += 1) {
          _results.push(this.get(i));
        }

        return _results;
      };

      LazyArray.prototype.inspect = function () {
        return inspect(this.toArray());
      };

      return LazyArray;
    }();

    module.exports = LazyArrayT;
  }).call(commonjsGlobal);
});

var Bitfield = createCommonjsModule(function (module) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var Bitfield;

    Bitfield = function () {
      function Bitfield(type, flags) {
        this.type = type;
        this.flags = flags != null ? flags : [];
      }

      Bitfield.prototype.decode = function (stream) {
        var flag, i, res, val, _i, _len, _ref;

        val = this.type.decode(stream);
        res = {};
        _ref = this.flags;

        for (i = _i = 0, _len = _ref.length; _i < _len; i = ++_i) {
          flag = _ref[i];

          if (flag != null) {
            res[flag] = !!(val & 1 << i);
          }
        }

        return res;
      };

      Bitfield.prototype.size = function () {
        return this.type.size();
      };

      Bitfield.prototype.encode = function (stream, keys) {
        var flag, i, val, _i, _len, _ref;

        val = 0;
        _ref = this.flags;

        for (i = _i = 0, _len = _ref.length; _i < _len; i = ++_i) {
          flag = _ref[i];

          if (flag != null) {
            if (keys[flag]) {
              val |= 1 << i;
            }
          }
        }

        return this.type.encode(stream, val);
      };

      return Bitfield;
    }();

    module.exports = Bitfield;
  }).call(commonjsGlobal);
});

var _Boolean = createCommonjsModule(function (module) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var BooleanT;

    BooleanT = function () {
      function BooleanT(type) {
        this.type = type;
      }

      BooleanT.prototype.decode = function (stream, parent) {
        return !!this.type.decode(stream, parent);
      };

      BooleanT.prototype.size = function (val, parent) {
        return this.type.size(val, parent);
      };

      BooleanT.prototype.encode = function (stream, val, parent) {
        return this.type.encode(stream, +val, parent);
      };

      return BooleanT;
    }();

    module.exports = BooleanT;
  }).call(commonjsGlobal);
});

var Buffer$9 = createCommonjsModule(function (module) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var BufferT, NumberT, utils$$1;
    utils$$1 = utils;
    NumberT = _Number.Number;

    BufferT = function () {
      function BufferT(length) {
        this.length = length;
      }

      BufferT.prototype.decode = function (stream, parent) {
        var length;
        length = utils$$1.resolveLength(this.length, stream, parent);
        return stream.readBuffer(length);
      };

      BufferT.prototype.size = function (val, parent) {
        if (!val) {
          return utils$$1.resolveLength(this.length, null, parent);
        }

        return val.length;
      };

      BufferT.prototype.encode = function (stream, buf, parent) {
        if (this.length instanceof NumberT) {
          this.length.encode(stream, buf.length);
        }

        return stream.writeBuffer(buf);
      };

      return BufferT;
    }();

    module.exports = BufferT;
  }).call(commonjsGlobal);
});

var Enum = createCommonjsModule(function (module) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var Enum;

    Enum = function () {
      function Enum(type, options) {
        this.type = type;
        this.options = options != null ? options : [];
      }

      Enum.prototype.decode = function (stream) {
        var index;
        index = this.type.decode(stream);
        return this.options[index] || index;
      };

      Enum.prototype.size = function () {
        return this.type.size();
      };

      Enum.prototype.encode = function (stream, val) {
        var index;
        index = this.options.indexOf(val);

        if (index === -1) {
          throw new Error("Unknown option in enum: " + val);
        }

        return this.type.encode(stream, index);
      };

      return Enum;
    }();

    module.exports = Enum;
  }).call(commonjsGlobal);
});

var Optional = createCommonjsModule(function (module) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var Optional;

    Optional = function () {
      function Optional(type, condition) {
        this.type = type;
        this.condition = condition != null ? condition : true;
      }

      Optional.prototype.decode = function (stream, parent) {
        var condition;
        condition = this.condition;

        if (typeof condition === 'function') {
          condition = condition.call(parent, parent);
        }

        if (condition) {
          return this.type.decode(stream, parent);
        }
      };

      Optional.prototype.size = function (val, parent) {
        var condition;
        condition = this.condition;

        if (typeof condition === 'function') {
          condition = condition.call(parent, parent);
        }

        if (condition) {
          return this.type.size(val, parent);
        } else {
          return 0;
        }
      };

      Optional.prototype.encode = function (stream, val, parent) {
        var condition;
        condition = this.condition;

        if (typeof condition === 'function') {
          condition = condition.call(parent, parent);
        }

        if (condition) {
          return this.type.encode(stream, val, parent);
        }
      };

      return Optional;
    }();

    module.exports = Optional;
  }).call(commonjsGlobal);
});

var Reserved = createCommonjsModule(function (module) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var Reserved, utils$$1;
    utils$$1 = utils;

    Reserved = function () {
      function Reserved(type, count) {
        this.type = type;
        this.count = count != null ? count : 1;
      }

      Reserved.prototype.decode = function (stream, parent) {
        stream.pos += this.size(null, parent);
        return void 0;
      };

      Reserved.prototype.size = function (data, parent) {
        var count;
        count = utils$$1.resolveLength(this.count, null, parent);
        return this.type.size() * count;
      };

      Reserved.prototype.encode = function (stream, val, parent) {
        return stream.fill(0, this.size(val, parent));
      };

      return Reserved;
    }();

    module.exports = Reserved;
  }).call(commonjsGlobal);
});

var _String = createCommonjsModule(function (module) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var NumberT, StringT, utils$$1;
    NumberT = _Number.Number;
    utils$$1 = utils;

    StringT = function () {
      function StringT(length, encoding) {
        this.length = length;
        this.encoding = encoding != null ? encoding : 'ascii';
      }

      StringT.prototype.decode = function (stream, parent) {
        var buffer, encoding, length, pos, string;

        length = function () {
          if (this.length != null) {
            return utils$$1.resolveLength(this.length, stream, parent);
          } else {
            buffer = stream.buffer, length = stream.length, pos = stream.pos;

            while (pos < length && buffer[pos] !== 0x00) {
              ++pos;
            }

            return pos - stream.pos;
          }
        }.call(this);

        encoding = this.encoding;

        if (typeof encoding === 'function') {
          encoding = encoding.call(parent, parent) || 'ascii';
        }

        string = stream.readString(length, encoding);

        if (this.length == null && stream.pos < stream.length) {
          stream.pos++;
        }

        return string;
      };

      StringT.prototype.size = function (val, parent) {
        var encoding, size;

        if (!val) {
          return utils$$1.resolveLength(this.length, null, parent);
        }

        encoding = this.encoding;

        if (typeof encoding === 'function') {
          encoding = encoding.call(parent != null ? parent.val : void 0, parent != null ? parent.val : void 0) || 'ascii';
        }

        if (encoding === 'utf16be') {
          encoding = 'utf16le';
        }

        size = Buffer.byteLength(val, encoding);

        if (this.length instanceof NumberT) {
          size += this.length.size();
        }

        if (this.length == null) {
          size++;
        }

        return size;
      };

      StringT.prototype.encode = function (stream, val, parent) {
        var encoding;
        encoding = this.encoding;

        if (typeof encoding === 'function') {
          encoding = encoding.call(parent != null ? parent.val : void 0, parent != null ? parent.val : void 0) || 'ascii';
        }

        if (this.length instanceof NumberT) {
          this.length.encode(stream, Buffer.byteLength(val, encoding));
        }

        stream.writeString(val, encoding);

        if (this.length == null) {
          return stream.writeUInt8(0x00);
        }
      };

      return StringT;
    }();

    module.exports = StringT;
  }).call(commonjsGlobal);
});

var Struct = createCommonjsModule(function (module) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var Struct, utils$$1;
    utils$$1 = utils;

    Struct = function () {
      function Struct(fields) {
        this.fields = fields != null ? fields : {};
      }

      Struct.prototype.decode = function (stream, parent, length) {
        var res, _ref;

        if (length == null) {
          length = 0;
        }

        res = this._setup(stream, parent, length);

        this._parseFields(stream, res, this.fields);

        if ((_ref = this.process) != null) {
          _ref.call(res, stream);
        }

        return res;
      };

      Struct.prototype._setup = function (stream, parent, length) {
        var res;
        res = {};
        Object.defineProperties(res, {
          parent: {
            value: parent
          },
          _startOffset: {
            value: stream.pos
          },
          _currentOffset: {
            value: 0,
            writable: true
          },
          _length: {
            value: length
          }
        });
        return res;
      };

      Struct.prototype._parseFields = function (stream, res, fields) {
        var key, type, val;

        for (key in fields) {
          type = fields[key];

          if (typeof type === 'function') {
            val = type.call(res, res);
          } else {
            val = type.decode(stream, res);
          }

          if (val !== void 0) {
            if (val instanceof utils$$1.PropertyDescriptor) {
              Object.defineProperty(res, key, val);
            } else {
              res[key] = val;
            }
          }

          res._currentOffset = stream.pos - res._startOffset;
        }
      };

      Struct.prototype.size = function (val, parent, includePointers) {
        var ctx, key, size, type, _ref;

        if (val == null) {
          val = {};
        }

        if (includePointers == null) {
          includePointers = true;
        }

        ctx = {
          parent: parent,
          val: val,
          pointerSize: 0
        };
        size = 0;
        _ref = this.fields;

        for (key in _ref) {
          type = _ref[key];

          if (type.size != null) {
            size += type.size(val[key], ctx);
          }
        }

        if (includePointers) {
          size += ctx.pointerSize;
        }

        return size;
      };

      Struct.prototype.encode = function (stream, val, parent) {
        var ctx, i, key, ptr, type, _ref, _ref1;

        if ((_ref = this.preEncode) != null) {
          _ref.call(val, stream);
        }

        ctx = {
          pointers: [],
          startOffset: stream.pos,
          parent: parent,
          val: val,
          pointerSize: 0
        };
        ctx.pointerOffset = stream.pos + this.size(val, ctx, false);
        _ref1 = this.fields;

        for (key in _ref1) {
          type = _ref1[key];

          if (type.encode != null) {
            type.encode(stream, val[key], ctx);
          }
        }

        i = 0;

        while (i < ctx.pointers.length) {
          ptr = ctx.pointers[i++];
          ptr.type.encode(stream, ptr.val, ptr.parent);
        }
      };

      return Struct;
    }();

    module.exports = Struct;
  }).call(commonjsGlobal);
});

var VersionedStruct = createCommonjsModule(function (module) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var Struct$$1,
        VersionedStruct,
        __hasProp = {}.hasOwnProperty,
        __extends = function __extends(child, parent) {
      for (var key in parent) {
        if (__hasProp.call(parent, key)) child[key] = parent[key];
      }

      function ctor() {
        this.constructor = child;
      }

      ctor.prototype = parent.prototype;
      child.prototype = new ctor();
      child.__super__ = parent.prototype;
      return child;
    };

    Struct$$1 = Struct;

    VersionedStruct = function (_super) {
      __extends(VersionedStruct, _super);

      function VersionedStruct(type, versions) {
        this.type = type;
        this.versions = versions != null ? versions : {};
      }

      VersionedStruct.prototype.versionGetter = function (parent) {
        if (typeof this.type === 'string') {
          return this.type.split('.').reduce(function (obj, prop) {
            if (typeof obj !== 'undefined') {
              return obj[prop];
            } else {
              return void 0;
            }
          }, parent);
        }
      };

      VersionedStruct.prototype.versionSetter = function (parent, version) {
        if (typeof this.type === 'string') {
          return this.type.split('.').reduce(function (obj, prop) {
            return obj[prop] = version;
          }, parent);
        }
      };

      VersionedStruct.prototype.decode = function (stream, parent, length) {
        var fields, res, _ref;

        if (length == null) {
          length = 0;
        }

        res = this._setup(stream, parent, length);

        if (typeof this.type === 'string') {
          res.version = this.versionGetter(parent);
        } else {
          res.version = this.type.decode(stream);
        }

        if (this.versions.header) {
          this._parseFields(stream, res, this.versions.header);
        }

        fields = this.versions[res.version];

        if (fields == null) {
          throw new Error("Unknown version " + res.version);
        }

        if (fields instanceof VersionedStruct) {
          return fields.decode(stream, parent);
        }

        this._parseFields(stream, res, fields);

        if ((_ref = this.process) != null) {
          _ref.call(res, stream);
        }

        return res;
      };

      VersionedStruct.prototype.size = function (val, parent, includePointers) {
        var ctx, fields, key, size, type, _ref;

        if (includePointers == null) {
          includePointers = true;
        }

        if (!val) {
          throw new Error('Not a fixed size');
        }

        ctx = {
          parent: parent,
          val: val,
          pointerSize: 0
        };
        size = 0;

        if (typeof this.type !== 'string') {
          size += this.type.size(val.version, ctx);
        }

        if (this.versions.header) {
          _ref = this.versions.header;

          for (key in _ref) {
            type = _ref[key];

            if (type.size != null) {
              size += type.size(val[key], ctx);
            }
          }
        }

        fields = this.versions[val.version];

        if (fields == null) {
          throw new Error("Unknown version " + val.version);
        }

        for (key in fields) {
          type = fields[key];

          if (type.size != null) {
            size += type.size(val[key], ctx);
          }
        }

        if (includePointers) {
          size += ctx.pointerSize;
        }

        return size;
      };

      VersionedStruct.prototype.encode = function (stream, val, parent) {
        var ctx, fields, i, key, ptr, type, _ref, _ref1;

        if ((_ref = this.preEncode) != null) {
          _ref.call(val, stream);
        }

        ctx = {
          pointers: [],
          startOffset: stream.pos,
          parent: parent,
          val: val,
          pointerSize: 0
        };
        ctx.pointerOffset = stream.pos + this.size(val, ctx, false);

        if (typeof this.type !== 'string') {
          this.type.encode(stream, val.version);
        }

        if (this.versions.header) {
          _ref1 = this.versions.header;

          for (key in _ref1) {
            type = _ref1[key];

            if (type.encode != null) {
              type.encode(stream, val[key], ctx);
            }
          }
        }

        fields = this.versions[val.version];

        for (key in fields) {
          type = fields[key];

          if (type.encode != null) {
            type.encode(stream, val[key], ctx);
          }
        }

        i = 0;

        while (i < ctx.pointers.length) {
          ptr = ctx.pointers[i++];
          ptr.type.encode(stream, ptr.val, ptr.parent);
        }
      };

      return VersionedStruct;
    }(Struct$$1);

    module.exports = VersionedStruct;
  }).call(commonjsGlobal);
});

var Pointer = createCommonjsModule(function (module, exports) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var Pointer, VoidPointer, utils$$1;
    utils$$1 = utils;

    Pointer = function () {
      function Pointer(offsetType, type, options) {
        var _base, _base1, _base2, _base3;

        this.offsetType = offsetType;
        this.type = type;
        this.options = options != null ? options : {};

        if (this.type === 'void') {
          this.type = null;
        }

        if ((_base = this.options).type == null) {
          _base.type = 'local';
        }

        if ((_base1 = this.options).allowNull == null) {
          _base1.allowNull = true;
        }

        if ((_base2 = this.options).nullValue == null) {
          _base2.nullValue = 0;
        }

        if ((_base3 = this.options).lazy == null) {
          _base3.lazy = false;
        }
      }

      Pointer.prototype.relativeToGetter = function (ctx) {
        return this.options.relativeTo.split('.').reduce(function (obj, prop) {
          return obj[prop];
        }, ctx);
      };

      Pointer.prototype.decode = function (stream, ctx) {
        var c, decodeValue, offset, ptr, relative, val;
        offset = this.offsetType.decode(stream, ctx);

        if (offset === this.options.nullValue && this.options.allowNull) {
          return null;
        }

        relative = function () {
          switch (this.options.type) {
            case 'local':
              return ctx._startOffset;

            case 'immediate':
              return stream.pos - this.offsetType.size();

            case 'parent':
              return ctx.parent._startOffset;

            default:
              c = ctx;

              while (c.parent) {
                c = c.parent;
              }

              return c._startOffset || 0;
          }
        }.call(this);

        if (this.options.relativeTo) {
          relative += this.relativeToGetter(ctx);
        }

        ptr = offset + relative;

        if (this.type != null) {
          val = null;

          decodeValue = function (_this) {
            return function () {
              var pos;

              if (val != null) {
                return val;
              }

              pos = stream.pos;
              stream.pos = ptr;
              val = _this.type.decode(stream, ctx);
              stream.pos = pos;
              return val;
            };
          }(this);

          if (this.options.lazy) {
            return new utils$$1.PropertyDescriptor({
              get: decodeValue
            });
          }

          return decodeValue();
        } else {
          return ptr;
        }
      };

      Pointer.prototype.size = function (val, ctx) {
        var parent, type;
        parent = ctx;

        switch (this.options.type) {
          case 'local':
          case 'immediate':
            break;

          case 'parent':
            ctx = ctx.parent;
            break;

          default:
            while (ctx.parent) {
              ctx = ctx.parent;
            }

        }

        type = this.type;

        if (type == null) {
          if (!(val instanceof VoidPointer)) {
            throw new Error("Must be a VoidPointer");
          }

          type = val.type;
          val = val.value;
        }

        if (val && ctx) {
          ctx.pointerSize += type.size(val, parent);
        }

        return this.offsetType.size();
      };

      Pointer.prototype.encode = function (stream, val, ctx) {
        var parent, relative, type;
        parent = ctx;

        if (val == null) {
          this.offsetType.encode(stream, this.options.nullValue);
          return;
        }

        switch (this.options.type) {
          case 'local':
            relative = ctx.startOffset;
            break;

          case 'immediate':
            relative = stream.pos + this.offsetType.size(val, parent);
            break;

          case 'parent':
            ctx = ctx.parent;
            relative = ctx.startOffset;
            break;

          default:
            relative = 0;

            while (ctx.parent) {
              ctx = ctx.parent;
            }

        }

        if (this.options.relativeTo) {
          relative += this.relativeToGetter(parent.val);
        }

        this.offsetType.encode(stream, ctx.pointerOffset - relative);
        type = this.type;

        if (type == null) {
          if (!(val instanceof VoidPointer)) {
            throw new Error("Must be a VoidPointer");
          }

          type = val.type;
          val = val.value;
        }

        ctx.pointers.push({
          type: type,
          val: val,
          parent: parent
        });
        return ctx.pointerOffset += type.size(val, parent);
      };

      return Pointer;
    }();

    VoidPointer = function () {
      function VoidPointer(type, value) {
        this.type = type;
        this.value = value;
      }

      return VoidPointer;
    }();

    exports.Pointer = Pointer;
    exports.VoidPointer = VoidPointer;
  }).call(commonjsGlobal);
});
var Pointer_1 = Pointer.Pointer;
var Pointer_2 = Pointer.VoidPointer;

var restructure = createCommonjsModule(function (module, exports) {
  // Generated by CoffeeScript 1.7.1
  (function () {
    var key, val, _ref, _ref1;

    exports.EncodeStream = EncodeStream;
    exports.DecodeStream = DecodeStream;
    exports.Array = _Array;
    exports.LazyArray = LazyArray;
    exports.Bitfield = Bitfield;
    exports.Boolean = _Boolean;
    exports.Buffer = Buffer$9;
    exports.Enum = Enum;
    exports.Optional = Optional;
    exports.Reserved = Reserved;
    exports.String = _String;
    exports.Struct = Struct;
    exports.VersionedStruct = VersionedStruct;
    _ref = _Number;

    for (key in _ref) {
      val = _ref[key];
      exports[key] = val;
    }

    _ref1 = Pointer;

    for (key in _ref1) {
      val = _ref1[key];
      exports[key] = val;
    }
  }).call(commonjsGlobal);
});
var restructure_1 = restructure.EncodeStream;
var restructure_2 = restructure.DecodeStream;
var restructure_3 = restructure.Array;
var restructure_4 = restructure.LazyArray;
var restructure_5 = restructure.Bitfield;
var restructure_6 = restructure.Boolean;
var restructure_7 = restructure.Buffer;
var restructure_8 = restructure.Enum;
var restructure_9 = restructure.Optional;
var restructure_10 = restructure.Reserved;
var restructure_11 = restructure.String;
var restructure_12 = restructure.Struct;
var restructure_13 = restructure.VersionedStruct;

var formats = [];
var fontkit = {
  logErrors: false,
  registerFormat: function registerFormat(format) {
    formats.push(format);
  },
  create: function create(uint8ArrayFontData, postscriptName) {
    var buffer = new Buffer.from(uint8ArrayFontData);

    for (var i = 0; i < formats.length; i++) {
      var format = formats[i];

      if (format.probe(buffer)) {
        var font = new format(new restructure.DecodeStream(buffer));

        if (postscriptName) {
          return font.getFont(postscriptName);
        }

        return font;
      }
    }

    throw new Error('Unknown font format');
  }
};

function _defineProperties(target, props) {
  for (var i = 0; i < props.length; i++) {
    var descriptor = props[i];
    descriptor.enumerable = descriptor.enumerable || false;
    descriptor.configurable = true;
    if ("value" in descriptor) descriptor.writable = true;
    Object.defineProperty(target, descriptor.key, descriptor);
  }
}

function _createClass(Constructor, protoProps, staticProps) {
  if (protoProps) _defineProperties(Constructor.prototype, protoProps);
  if (staticProps) _defineProperties(Constructor, staticProps);
  return Constructor;
}

function _defineProperty(obj, key, value) {
  if (key in obj) {
    Object.defineProperty(obj, key, {
      value: value,
      enumerable: true,
      configurable: true,
      writable: true
    });
  } else {
    obj[key] = value;
  }

  return obj;
}

function _inheritsLoose(subClass, superClass) {
  subClass.prototype = Object.create(superClass.prototype);
  subClass.prototype.constructor = subClass;
  subClass.__proto__ = superClass;
}

function _applyDecoratedDescriptor(target, property, decorators, descriptor, context) {
  var desc = {};
  Object['ke' + 'ys'](descriptor).forEach(function (key) {
    desc[key] = descriptor[key];
  });
  desc.enumerable = !!desc.enumerable;
  desc.configurable = !!desc.configurable;

  if ('value' in desc || desc.initializer) {
    desc.writable = true;
  }

  desc = decorators.slice().reverse().reduce(function (desc, decorator) {
    return decorator(target, property, desc) || desc;
  }, desc);

  if (context && desc.initializer !== void 0) {
    desc.value = desc.initializer ? desc.initializer.call(context) : void 0;
    desc.initializer = undefined;
  }

  if (desc.initializer === void 0) {
    Object['define' + 'Property'](target, property, desc);
    desc = null;
  }

  return desc;
}

/**
 * This decorator caches the results of a getter or method such that
 * the results are lazily computed once, and then cached.
 * @private
 */
function cache(target, key, descriptor) {
  if (descriptor.get) {
    var get = descriptor.get;

    descriptor.get = function () {
      var value = get.call(this);
      Object.defineProperty(this, key, {
        value: value
      });
      return value;
    };
  } else if (typeof descriptor.value === 'function') {
    var fn = descriptor.value;
    return {
      get: function get() {
        var cache = new Map();

        function memoized() {
          for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
            args[_key] = arguments[_key];
          }

          var key = args.length > 0 ? args[0] : 'value';

          if (cache.has(key)) {
            return cache.get(key);
          }

          var result = fn.apply(this, args);
          cache.set(key, result);
          return result;
        }
        Object.defineProperty(this, key, {
          value: memoized
        });
        return memoized;
      }
    };
  }
}

var SubHeader = new restructure.Struct({
  firstCode: restructure.uint16,
  entryCount: restructure.uint16,
  idDelta: restructure.int16,
  idRangeOffset: restructure.uint16
});
var CmapGroup = new restructure.Struct({
  startCharCode: restructure.uint32,
  endCharCode: restructure.uint32,
  glyphID: restructure.uint32
});
var UnicodeValueRange = new restructure.Struct({
  startUnicodeValue: restructure.uint24,
  additionalCount: restructure.uint8
});
var UVSMapping = new restructure.Struct({
  unicodeValue: restructure.uint24,
  glyphID: restructure.uint16
});
var DefaultUVS = new restructure.Array(UnicodeValueRange, restructure.uint32);
var NonDefaultUVS = new restructure.Array(UVSMapping, restructure.uint32);
var VarSelectorRecord = new restructure.Struct({
  varSelector: restructure.uint24,
  defaultUVS: new restructure.Pointer(restructure.uint32, DefaultUVS, {
    type: 'parent'
  }),
  nonDefaultUVS: new restructure.Pointer(restructure.uint32, NonDefaultUVS, {
    type: 'parent'
  })
});
var CmapSubtable = new restructure.VersionedStruct(restructure.uint16, {
  0: {
    // Byte encoding
    length: restructure.uint16,
    // Total table length in bytes (set to 262 for format 0)
    language: restructure.uint16,
    // Language code for this encoding subtable, or zero if language-independent
    codeMap: new restructure.LazyArray(restructure.uint8, 256)
  },
  2: {
    // High-byte mapping (CJK)
    length: restructure.uint16,
    language: restructure.uint16,
    subHeaderKeys: new restructure.Array(restructure.uint16, 256),
    subHeaderCount: function subHeaderCount(t) {
      return Math.max.apply(Math, t.subHeaderKeys);
    },
    subHeaders: new restructure.LazyArray(SubHeader, 'subHeaderCount'),
    glyphIndexArray: new restructure.LazyArray(restructure.uint16, 'subHeaderCount')
  },
  4: {
    // Segment mapping to delta values
    length: restructure.uint16,
    // Total table length in bytes
    language: restructure.uint16,
    // Language code
    segCountX2: restructure.uint16,
    segCount: function segCount(t) {
      return t.segCountX2 >> 1;
    },
    searchRange: restructure.uint16,
    entrySelector: restructure.uint16,
    rangeShift: restructure.uint16,
    endCode: new restructure.LazyArray(restructure.uint16, 'segCount'),
    reservedPad: new restructure.Reserved(restructure.uint16),
    // This value should be zero
    startCode: new restructure.LazyArray(restructure.uint16, 'segCount'),
    idDelta: new restructure.LazyArray(restructure.int16, 'segCount'),
    idRangeOffset: new restructure.LazyArray(restructure.uint16, 'segCount'),
    glyphIndexArray: new restructure.LazyArray(restructure.uint16, function (t) {
      return (t.length - t._currentOffset) / 2;
    })
  },
  6: {
    // Trimmed table
    length: restructure.uint16,
    language: restructure.uint16,
    firstCode: restructure.uint16,
    entryCount: restructure.uint16,
    glyphIndices: new restructure.LazyArray(restructure.uint16, 'entryCount')
  },
  8: {
    // mixed 16-bit and 32-bit coverage
    reserved: new restructure.Reserved(restructure.uint16),
    length: restructure.uint32,
    language: restructure.uint16,
    is32: new restructure.LazyArray(restructure.uint8, 8192),
    nGroups: restructure.uint32,
    groups: new restructure.LazyArray(CmapGroup, 'nGroups')
  },
  10: {
    // Trimmed Array
    reserved: new restructure.Reserved(restructure.uint16),
    length: restructure.uint32,
    language: restructure.uint32,
    firstCode: restructure.uint32,
    entryCount: restructure.uint32,
    glyphIndices: new restructure.LazyArray(restructure.uint16, 'numChars')
  },
  12: {
    // Segmented coverage
    reserved: new restructure.Reserved(restructure.uint16),
    length: restructure.uint32,
    language: restructure.uint32,
    nGroups: restructure.uint32,
    groups: new restructure.LazyArray(CmapGroup, 'nGroups')
  },
  13: {
    // Many-to-one range mappings (same as 12 except for group.startGlyphID)
    reserved: new restructure.Reserved(restructure.uint16),
    length: restructure.uint32,
    language: restructure.uint32,
    nGroups: restructure.uint32,
    groups: new restructure.LazyArray(CmapGroup, 'nGroups')
  },
  14: {
    // Unicode Variation Sequences
    length: restructure.uint32,
    numRecords: restructure.uint32,
    varSelectors: new restructure.LazyArray(VarSelectorRecord, 'numRecords')
  }
});
var CmapEntry = new restructure.Struct({
  platformID: restructure.uint16,
  // Platform identifier
  encodingID: restructure.uint16,
  // Platform-specific encoding identifier
  table: new restructure.Pointer(restructure.uint32, CmapSubtable, {
    type: 'parent',
    lazy: true
  })
}); // character to glyph mapping

var cmap = new restructure.Struct({
  version: restructure.uint16,
  numSubtables: restructure.uint16,
  tables: new restructure.Array(CmapEntry, 'numSubtables')
});

var head = new restructure.Struct({
  version: restructure.int32,
  // 0x00010000 (version 1.0)
  revision: restructure.int32,
  // set by font manufacturer
  checkSumAdjustment: restructure.uint32,
  magicNumber: restructure.uint32,
  // set to 0x5F0F3CF5
  flags: restructure.uint16,
  unitsPerEm: restructure.uint16,
  // range from 64 to 16384
  created: new restructure.Array(restructure.int32, 2),
  modified: new restructure.Array(restructure.int32, 2),
  xMin: restructure.int16,
  // for all glyph bounding boxes
  yMin: restructure.int16,
  // for all glyph bounding boxes
  xMax: restructure.int16,
  // for all glyph bounding boxes
  yMax: restructure.int16,
  // for all glyph bounding boxes
  macStyle: new restructure.Bitfield(restructure.uint16, ['bold', 'italic', 'underline', 'outline', 'shadow', 'condensed', 'extended']),
  lowestRecPPEM: restructure.uint16,
  // smallest readable size in pixels
  fontDirectionHint: restructure.int16,
  indexToLocFormat: restructure.int16,
  // 0 for short offsets, 1 for long
  glyphDataFormat: restructure.int16 // 0 for current format

});

var hhea = new restructure.Struct({
  version: restructure.int32,
  ascent: restructure.int16,
  // Distance from baseline of highest ascender
  descent: restructure.int16,
  // Distance from baseline of lowest descender
  lineGap: restructure.int16,
  // Typographic line gap
  advanceWidthMax: restructure.uint16,
  // Maximum advance width value in 'hmtx' table
  minLeftSideBearing: restructure.int16,
  // Maximum advance width value in 'hmtx' table
  minRightSideBearing: restructure.int16,
  // Minimum right sidebearing value
  xMaxExtent: restructure.int16,
  caretSlopeRise: restructure.int16,
  // Used to calculate the slope of the cursor (rise/run); 1 for vertical
  caretSlopeRun: restructure.int16,
  // 0 for vertical
  caretOffset: restructure.int16,
  // Set to 0 for non-slanted fonts
  reserved: new restructure.Reserved(restructure.int16, 4),
  metricDataFormat: restructure.int16,
  // 0 for current format
  numberOfMetrics: restructure.uint16 // Number of advance widths in 'hmtx' table

});

var HmtxEntry = new restructure.Struct({
  advance: restructure.uint16,
  bearing: restructure.int16
});
var hmtx = new restructure.Struct({
  metrics: new restructure.LazyArray(HmtxEntry, function (t) {
    return t.parent.hhea.numberOfMetrics;
  }),
  bearings: new restructure.LazyArray(restructure.int16, function (t) {
    return t.parent.maxp.numGlyphs - t.parent.hhea.numberOfMetrics;
  })
});

var maxp = new restructure.Struct({
  version: restructure.int32,
  numGlyphs: restructure.uint16,
  // The number of glyphs in the font
  maxPoints: restructure.uint16,
  // Maximum points in a non-composite glyph
  maxContours: restructure.uint16,
  // Maximum contours in a non-composite glyph
  maxComponentPoints: restructure.uint16,
  // Maximum points in a composite glyph
  maxComponentContours: restructure.uint16,
  // Maximum contours in a composite glyph
  maxZones: restructure.uint16,
  // 1 if instructions do not use the twilight zone, 2 otherwise
  maxTwilightPoints: restructure.uint16,
  // Maximum points used in Z0
  maxStorage: restructure.uint16,
  // Number of Storage Area locations
  maxFunctionDefs: restructure.uint16,
  // Number of FDEFs
  maxInstructionDefs: restructure.uint16,
  // Number of IDEFs
  maxStackElements: restructure.uint16,
  // Maximum stack depth
  maxSizeOfInstructions: restructure.uint16,
  // Maximum byte count for glyph instructions
  maxComponentElements: restructure.uint16,
  // Maximum number of components referenced at “top level” for any composite glyph
  maxComponentDepth: restructure.uint16 // Maximum levels of recursion; 1 for simple components

});

/**
 * Gets an encoding name from platform, encoding, and language ids.
 * Returned encoding names can be used in iconv-lite to decode text.
 */
function getEncoding(platformID, encodingID, languageID) {
  if (languageID === void 0) {
    languageID = 0;
  }

  if (platformID === 1 && MAC_LANGUAGE_ENCODINGS[languageID]) {
    return MAC_LANGUAGE_ENCODINGS[languageID];
  }

  return ENCODINGS[platformID][encodingID];
} // Map of platform ids to encoding ids.

var ENCODINGS = [// unicode
['utf16be', 'utf16be', 'utf16be', 'utf16be', 'utf16be', 'utf16be'], // macintosh
// Mappings available at http://unicode.org/Public/MAPPINGS/VENDORS/APPLE/
// 0	Roman                 17	Malayalam
// 1	Japanese	            18	Sinhalese
// 2	Traditional Chinese	  19	Burmese
// 3	Korean	              20	Khmer
// 4	Arabic	              21	Thai
// 5	Hebrew	              22	Laotian
// 6	Greek	                23	Georgian
// 7	Russian	              24	Armenian
// 8	RSymbol	              25	Simplified Chinese
// 9	Devanagari	          26	Tibetan
// 10	Gurmukhi	            27	Mongolian
// 11	Gujarati	            28	Geez
// 12	Oriya	                29	Slavic
// 13	Bengali	              30	Vietnamese
// 14	Tamil	                31	Sindhi
// 15	Telugu	              32	(Uninterpreted)
// 16	Kannada
['macroman', 'shift-jis', 'big5', 'euc-kr', 'iso-8859-6', 'iso-8859-8', 'macgreek', 'maccyrillic', 'symbol', 'Devanagari', 'Gurmukhi', 'Gujarati', 'Oriya', 'Bengali', 'Tamil', 'Telugu', 'Kannada', 'Malayalam', 'Sinhalese', 'Burmese', 'Khmer', 'macthai', 'Laotian', 'Georgian', 'Armenian', 'gb-2312-80', 'Tibetan', 'Mongolian', 'Geez', 'maccenteuro', 'Vietnamese', 'Sindhi'], // ISO (deprecated)
['ascii'], // windows
// Docs here: http://msdn.microsoft.com/en-us/library/system.text.encoding(v=vs.110).aspx
['symbol', 'utf16be', 'shift-jis', 'gb18030', 'big5', 'wansung', 'johab', null, null, null, 'utf16be']]; // Overrides for Mac scripts by language id.
// See http://unicode.org/Public/MAPPINGS/VENDORS/APPLE/Readme.txt

var MAC_LANGUAGE_ENCODINGS = {
  15: 'maciceland',
  17: 'macturkish',
  18: 'maccroatian',
  24: 'maccenteuro',
  25: 'maccenteuro',
  26: 'maccenteuro',
  27: 'maccenteuro',
  28: 'maccenteuro',
  30: 'maciceland',
  37: 'macromania',
  38: 'maccenteuro',
  39: 'maccenteuro',
  40: 'maccenteuro',
  143: 'macinuit',
  // Unsupported by iconv-lite
  146: 'macgaelic' // Unsupported by iconv-lite

}; // Map of platform ids to BCP-47 language codes.

var LANGUAGES = [// unicode
[], {
  // macintosh
  0: 'en',
  30: 'fo',
  60: 'ks',
  90: 'rw',
  1: 'fr',
  31: 'fa',
  61: 'ku',
  91: 'rn',
  2: 'de',
  32: 'ru',
  62: 'sd',
  92: 'ny',
  3: 'it',
  33: 'zh',
  63: 'bo',
  93: 'mg',
  4: 'nl',
  34: 'nl-BE',
  64: 'ne',
  94: 'eo',
  5: 'sv',
  35: 'ga',
  65: 'sa',
  128: 'cy',
  6: 'es',
  36: 'sq',
  66: 'mr',
  129: 'eu',
  7: 'da',
  37: 'ro',
  67: 'bn',
  130: 'ca',
  8: 'pt',
  38: 'cz',
  68: 'as',
  131: 'la',
  9: 'no',
  39: 'sk',
  69: 'gu',
  132: 'qu',
  10: 'he',
  40: 'si',
  70: 'pa',
  133: 'gn',
  11: 'ja',
  41: 'yi',
  71: 'or',
  134: 'ay',
  12: 'ar',
  42: 'sr',
  72: 'ml',
  135: 'tt',
  13: 'fi',
  43: 'mk',
  73: 'kn',
  136: 'ug',
  14: 'el',
  44: 'bg',
  74: 'ta',
  137: 'dz',
  15: 'is',
  45: 'uk',
  75: 'te',
  138: 'jv',
  16: 'mt',
  46: 'be',
  76: 'si',
  139: 'su',
  17: 'tr',
  47: 'uz',
  77: 'my',
  140: 'gl',
  18: 'hr',
  48: 'kk',
  78: 'km',
  141: 'af',
  19: 'zh-Hant',
  49: 'az-Cyrl',
  79: 'lo',
  142: 'br',
  20: 'ur',
  50: 'az-Arab',
  80: 'vi',
  143: 'iu',
  21: 'hi',
  51: 'hy',
  81: 'id',
  144: 'gd',
  22: 'th',
  52: 'ka',
  82: 'tl',
  145: 'gv',
  23: 'ko',
  53: 'mo',
  83: 'ms',
  146: 'ga',
  24: 'lt',
  54: 'ky',
  84: 'ms-Arab',
  147: 'to',
  25: 'pl',
  55: 'tg',
  85: 'am',
  148: 'el-polyton',
  26: 'hu',
  56: 'tk',
  86: 'ti',
  149: 'kl',
  27: 'es',
  57: 'mn-CN',
  87: 'om',
  150: 'az',
  28: 'lv',
  58: 'mn',
  88: 'so',
  151: 'nn',
  29: 'se',
  59: 'ps',
  89: 'sw'
}, // ISO (deprecated)
[], {
  // windows                                        
  0x0436: 'af',
  0x4009: 'en-IN',
  0x0487: 'rw',
  0x0432: 'tn',
  0x041C: 'sq',
  0x1809: 'en-IE',
  0x0441: 'sw',
  0x045B: 'si',
  0x0484: 'gsw',
  0x2009: 'en-JM',
  0x0457: 'kok',
  0x041B: 'sk',
  0x045E: 'am',
  0x4409: 'en-MY',
  0x0412: 'ko',
  0x0424: 'sl',
  0x1401: 'ar-DZ',
  0x1409: 'en-NZ',
  0x0440: 'ky',
  0x2C0A: 'es-AR',
  0x3C01: 'ar-BH',
  0x3409: 'en-PH',
  0x0454: 'lo',
  0x400A: 'es-BO',
  0x0C01: 'ar',
  0x4809: 'en-SG',
  0x0426: 'lv',
  0x340A: 'es-CL',
  0x0801: 'ar-IQ',
  0x1C09: 'en-ZA',
  0x0427: 'lt',
  0x240A: 'es-CO',
  0x2C01: 'ar-JO',
  0x2C09: 'en-TT',
  0x082E: 'dsb',
  0x140A: 'es-CR',
  0x3401: 'ar-KW',
  0x0809: 'en-GB',
  0x046E: 'lb',
  0x1C0A: 'es-DO',
  0x3001: 'ar-LB',
  0x0409: 'en',
  0x042F: 'mk',
  0x300A: 'es-EC',
  0x1001: 'ar-LY',
  0x3009: 'en-ZW',
  0x083E: 'ms-BN',
  0x440A: 'es-SV',
  0x1801: 'ary',
  0x0425: 'et',
  0x043E: 'ms',
  0x100A: 'es-GT',
  0x2001: 'ar-OM',
  0x0438: 'fo',
  0x044C: 'ml',
  0x480A: 'es-HN',
  0x4001: 'ar-QA',
  0x0464: 'fil',
  0x043A: 'mt',
  0x080A: 'es-MX',
  0x0401: 'ar-SA',
  0x040B: 'fi',
  0x0481: 'mi',
  0x4C0A: 'es-NI',
  0x2801: 'ar-SY',
  0x080C: 'fr-BE',
  0x047A: 'arn',
  0x180A: 'es-PA',
  0x1C01: 'aeb',
  0x0C0C: 'fr-CA',
  0x044E: 'mr',
  0x3C0A: 'es-PY',
  0x3801: 'ar-AE',
  0x040C: 'fr',
  0x047C: 'moh',
  0x280A: 'es-PE',
  0x2401: 'ar-YE',
  0x140C: 'fr-LU',
  0x0450: 'mn',
  0x500A: 'es-PR',
  0x042B: 'hy',
  0x180C: 'fr-MC',
  0x0850: 'mn-CN',
  0x0C0A: 'es',
  0x044D: 'as',
  0x100C: 'fr-CH',
  0x0461: 'ne',
  0x040A: 'es',
  0x082C: 'az-Cyrl',
  0x0462: 'fy',
  0x0414: 'nb',
  0x540A: 'es-US',
  0x042C: 'az',
  0x0456: 'gl',
  0x0814: 'nn',
  0x380A: 'es-UY',
  0x046D: 'ba',
  0x0437: 'ka',
  0x0482: 'oc',
  0x200A: 'es-VE',
  0x042D: 'eu',
  0x0C07: 'de-AT',
  0x0448: 'or',
  0x081D: 'sv-FI',
  0x0423: 'be',
  0x0407: 'de',
  0x0463: 'ps',
  0x041D: 'sv',
  0x0845: 'bn',
  0x1407: 'de-LI',
  0x0415: 'pl',
  0x045A: 'syr',
  0x0445: 'bn-IN',
  0x1007: 'de-LU',
  0x0416: 'pt',
  0x0428: 'tg',
  0x201A: 'bs-Cyrl',
  0x0807: 'de-CH',
  0x0816: 'pt-PT',
  0x085F: 'tzm',
  0x141A: 'bs',
  0x0408: 'el',
  0x0446: 'pa',
  0x0449: 'ta',
  0x047E: 'br',
  0x046F: 'kl',
  0x046B: 'qu-BO',
  0x0444: 'tt',
  0x0402: 'bg',
  0x0447: 'gu',
  0x086B: 'qu-EC',
  0x044A: 'te',
  0x0403: 'ca',
  0x0468: 'ha',
  0x0C6B: 'qu',
  0x041E: 'th',
  0x0C04: 'zh-HK',
  0x040D: 'he',
  0x0418: 'ro',
  0x0451: 'bo',
  0x1404: 'zh-MO',
  0x0439: 'hi',
  0x0417: 'rm',
  0x041F: 'tr',
  0x0804: 'zh',
  0x040E: 'hu',
  0x0419: 'ru',
  0x0442: 'tk',
  0x1004: 'zh-SG',
  0x040F: 'is',
  0x243B: 'smn',
  0x0480: 'ug',
  0x0404: 'zh-TW',
  0x0470: 'ig',
  0x103B: 'smj-NO',
  0x0422: 'uk',
  0x0483: 'co',
  0x0421: 'id',
  0x143B: 'smj',
  0x042E: 'hsb',
  0x041A: 'hr',
  0x045D: 'iu',
  0x0C3B: 'se-FI',
  0x0420: 'ur',
  0x101A: 'hr-BA',
  0x085D: 'iu-Latn',
  0x043B: 'se',
  0x0843: 'uz-Cyrl',
  0x0405: 'cs',
  0x083C: 'ga',
  0x083B: 'se-SE',
  0x0443: 'uz',
  0x0406: 'da',
  0x0434: 'xh',
  0x203B: 'sms',
  0x042A: 'vi',
  0x048C: 'prs',
  0x0435: 'zu',
  0x183B: 'sma-NO',
  0x0452: 'cy',
  0x0465: 'dv',
  0x0410: 'it',
  0x1C3B: 'sms',
  0x0488: 'wo',
  0x0813: 'nl-BE',
  0x0810: 'it-CH',
  0x044F: 'sa',
  0x0485: 'sah',
  0x0413: 'nl',
  0x0411: 'ja',
  0x1C1A: 'sr-Cyrl-BA',
  0x0478: 'ii',
  0x0C09: 'en-AU',
  0x044B: 'kn',
  0x0C1A: 'sr',
  0x046A: 'yo',
  0x2809: 'en-BZ',
  0x043F: 'kk',
  0x181A: 'sr-Latn-BA',
  0x1009: 'en-CA',
  0x0453: 'km',
  0x081A: 'sr-Latn',
  0x2409: 'en-029',
  0x0486: 'quc',
  0x046C: 'nso'
}];

var NameRecord = new restructure.Struct({
  platformID: restructure.uint16,
  encodingID: restructure.uint16,
  languageID: restructure.uint16,
  nameID: restructure.uint16,
  length: restructure.uint16,
  string: new restructure.Pointer(restructure.uint16, new restructure.String('length', function (t) {
    return getEncoding(t.platformID, t.encodingID, t.languageID);
  }), {
    type: 'parent',
    relativeTo: 'parent.stringOffset',
    allowNull: false
  })
});
var LangTagRecord = new restructure.Struct({
  length: restructure.uint16,
  tag: new restructure.Pointer(restructure.uint16, new restructure.String('length', 'utf16be'), {
    type: 'parent',
    relativeTo: 'stringOffset'
  })
});
var NameTable = new restructure.VersionedStruct(restructure.uint16, {
  0: {
    count: restructure.uint16,
    stringOffset: restructure.uint16,
    records: new restructure.Array(NameRecord, 'count')
  },
  1: {
    count: restructure.uint16,
    stringOffset: restructure.uint16,
    records: new restructure.Array(NameRecord, 'count'),
    langTagCount: restructure.uint16,
    langTags: new restructure.Array(LangTagRecord, 'langTagCount')
  }
});
var NAMES = ['copyright', 'fontFamily', 'fontSubfamily', 'uniqueSubfamily', 'fullName', 'version', 'postscriptName', // Note: A font may have only one PostScript name and that name must be ASCII.
'trademark', 'manufacturer', 'designer', 'description', 'vendorURL', 'designerURL', 'license', 'licenseURL', null, // reserved
'preferredFamily', 'preferredSubfamily', 'compatibleFull', 'sampleText', 'postscriptCIDFontName', 'wwsFamilyName', 'wwsSubfamilyName'];

NameTable.process = function (stream) {
  var records = {};

  for (var _iterator = this.records, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
    var _ref;

    if (_isArray) {
      if (_i >= _iterator.length) break;
      _ref = _iterator[_i++];
    } else {
      _i = _iterator.next();
      if (_i.done) break;
      _ref = _i.value;
    }

    var record = _ref;
    // find out what language this is for
    var language = LANGUAGES[record.platformID][record.languageID];

    if (language == null && this.langTags != null && record.languageID >= 0x8000) {
      language = this.langTags[record.languageID - 0x8000].tag;
    }

    if (language == null) {
      language = record.platformID + '-' + record.languageID;
    } // if the nameID is >= 256, it is a font feature record (AAT)


    var key = record.nameID >= 256 ? 'fontFeatures' : NAMES[record.nameID] || record.nameID;

    if (records[key] == null) {
      records[key] = {};
    }

    var obj = records[key];

    if (record.nameID >= 256) {
      obj = obj[record.nameID] || (obj[record.nameID] = {});
    }

    if (typeof record.string === 'string' || typeof obj[language] !== 'string') {
      obj[language] = record.string;
    }
  }

  this.records = records;
};

NameTable.preEncode = function () {
  if (Array.isArray(this.records)) return;
  this.version = 0;
  var records = [];

  for (var key in this.records) {
    var val = this.records[key];
    if (key === 'fontFeatures') continue;
    records.push({
      platformID: 3,
      encodingID: 1,
      languageID: 0x409,
      nameID: NAMES.indexOf(key),
      length: Buffer.byteLength(val.en, 'utf16le'),
      string: val.en
    });

    if (key === 'postscriptName') {
      records.push({
        platformID: 1,
        encodingID: 0,
        languageID: 0,
        nameID: NAMES.indexOf(key),
        length: val.en.length,
        string: val.en
      });
    }
  }

  this.records = records;
  this.count = records.length;
  this.stringOffset = NameTable.size(this, null, false);
};

var OS2 = new restructure.VersionedStruct(restructure.uint16, {
  header: {
    xAvgCharWidth: restructure.int16,
    // average weighted advance width of lower case letters and space
    usWeightClass: restructure.uint16,
    // visual weight of stroke in glyphs
    usWidthClass: restructure.uint16,
    // relative change from the normal aspect ratio (width to height ratio)
    fsType: new restructure.Bitfield(restructure.uint16, [// Indicates font embedding licensing rights
    null, 'noEmbedding', 'viewOnly', 'editable', null, null, null, null, 'noSubsetting', 'bitmapOnly']),
    ySubscriptXSize: restructure.int16,
    // recommended horizontal size in pixels for subscripts
    ySubscriptYSize: restructure.int16,
    // recommended vertical size in pixels for subscripts
    ySubscriptXOffset: restructure.int16,
    // recommended horizontal offset for subscripts
    ySubscriptYOffset: restructure.int16,
    // recommended vertical offset form the baseline for subscripts
    ySuperscriptXSize: restructure.int16,
    // recommended horizontal size in pixels for superscripts
    ySuperscriptYSize: restructure.int16,
    // recommended vertical size in pixels for superscripts
    ySuperscriptXOffset: restructure.int16,
    // recommended horizontal offset for superscripts
    ySuperscriptYOffset: restructure.int16,
    // recommended vertical offset from the baseline for superscripts
    yStrikeoutSize: restructure.int16,
    // width of the strikeout stroke
    yStrikeoutPosition: restructure.int16,
    // position of the strikeout stroke relative to the baseline
    sFamilyClass: restructure.int16,
    // classification of font-family design
    panose: new restructure.Array(restructure.uint8, 10),
    // describe the visual characteristics of a given typeface
    ulCharRange: new restructure.Array(restructure.uint32, 4),
    vendorID: new restructure.String(4),
    // four character identifier for the font vendor
    fsSelection: new restructure.Bitfield(restructure.uint16, [// bit field containing information about the font
    'italic', 'underscore', 'negative', 'outlined', 'strikeout', 'bold', 'regular', 'useTypoMetrics', 'wws', 'oblique']),
    usFirstCharIndex: restructure.uint16,
    // The minimum Unicode index in this font
    usLastCharIndex: restructure.uint16 // The maximum Unicode index in this font

  },
  // The Apple version of this table ends here, but the Microsoft one continues on...
  0: {},
  1: {
    typoAscender: restructure.int16,
    typoDescender: restructure.int16,
    typoLineGap: restructure.int16,
    winAscent: restructure.uint16,
    winDescent: restructure.uint16,
    codePageRange: new restructure.Array(restructure.uint32, 2)
  },
  2: {
    // these should be common with version 1 somehow
    typoAscender: restructure.int16,
    typoDescender: restructure.int16,
    typoLineGap: restructure.int16,
    winAscent: restructure.uint16,
    winDescent: restructure.uint16,
    codePageRange: new restructure.Array(restructure.uint32, 2),
    xHeight: restructure.int16,
    capHeight: restructure.int16,
    defaultChar: restructure.uint16,
    breakChar: restructure.uint16,
    maxContent: restructure.uint16
  },
  5: {
    typoAscender: restructure.int16,
    typoDescender: restructure.int16,
    typoLineGap: restructure.int16,
    winAscent: restructure.uint16,
    winDescent: restructure.uint16,
    codePageRange: new restructure.Array(restructure.uint32, 2),
    xHeight: restructure.int16,
    capHeight: restructure.int16,
    defaultChar: restructure.uint16,
    breakChar: restructure.uint16,
    maxContent: restructure.uint16,
    usLowerOpticalPointSize: restructure.uint16,
    usUpperOpticalPointSize: restructure.uint16
  }
});
var versions$1 = OS2.versions;
versions$1[3] = versions$1[4] = versions$1[2];

var post = new restructure.VersionedStruct(restructure.fixed32, {
  header: {
    // these fields exist at the top of all versions
    italicAngle: restructure.fixed32,
    // Italic angle in counter-clockwise degrees from the vertical.
    underlinePosition: restructure.int16,
    // Suggested distance of the top of the underline from the baseline
    underlineThickness: restructure.int16,
    // Suggested values for the underline thickness
    isFixedPitch: restructure.uint32,
    // Whether the font is monospaced
    minMemType42: restructure.uint32,
    // Minimum memory usage when a TrueType font is downloaded as a Type 42 font
    maxMemType42: restructure.uint32,
    // Maximum memory usage when a TrueType font is downloaded as a Type 42 font
    minMemType1: restructure.uint32,
    // Minimum memory usage when a TrueType font is downloaded as a Type 1 font
    maxMemType1: restructure.uint32 // Maximum memory usage when a TrueType font is downloaded as a Type 1 font

  },
  1: {},
  // version 1 has no additional fields
  2: {
    numberOfGlyphs: restructure.uint16,
    glyphNameIndex: new restructure.Array(restructure.uint16, 'numberOfGlyphs'),
    names: new restructure.Array(new restructure.String(restructure.uint8))
  },
  2.5: {
    numberOfGlyphs: restructure.uint16,
    offsets: new restructure.Array(restructure.uint8, 'numberOfGlyphs')
  },
  3: {},
  // version 3 has no additional fields
  4: {
    map: new restructure.Array(restructure.uint32, function (t) {
      return t.parent.maxp.numGlyphs;
    })
  }
});

var cvt = new restructure.Struct({
  controlValues: new restructure.Array(restructure.int16)
});

// These instructions are known as the font program. The main use of this table
// is for the definition of functions that are used in many different glyph programs.

var fpgm = new restructure.Struct({
  instructions: new restructure.Array(restructure.uint8)
});

var loca = new restructure.VersionedStruct('head.indexToLocFormat', {
  0: {
    offsets: new restructure.Array(restructure.uint16)
  },
  1: {
    offsets: new restructure.Array(restructure.uint32)
  }
});

loca.process = function () {
  if (this.version === 0) {
    for (var i = 0; i < this.offsets.length; i++) {
      this.offsets[i] <<= 1;
    }
  }
};

loca.preEncode = function () {
  if (this.version != null) return; // assume this.offsets is a sorted array

  this.version = this.offsets[this.offsets.length - 1] > 0xffff ? 1 : 0;

  if (this.version === 0) {
    for (var i = 0; i < this.offsets.length; i++) {
      this.offsets[i] >>>= 1;
    }
  }
};

var prep = new restructure.Struct({
  controlValueProgram: new restructure.Array(restructure.uint8)
});

var glyf = new restructure.Array(new restructure.Buffer());

var CFFIndex =
/*#__PURE__*/
function () {
  function CFFIndex(type) {
    this.type = type;
  }

  var _proto = CFFIndex.prototype;

  _proto.getCFFVersion = function getCFFVersion(ctx) {
    while (ctx && !ctx.hdrSize) {
      ctx = ctx.parent;
    }

    return ctx ? ctx.version : -1;
  };

  _proto.decode = function decode(stream, parent) {
    var version = this.getCFFVersion(parent);
    var count = version >= 2 ? stream.readUInt32BE() : stream.readUInt16BE();

    if (count === 0) {
      return [];
    }

    var offSize = stream.readUInt8();
    var offsetType;

    if (offSize === 1) {
      offsetType = restructure.uint8;
    } else if (offSize === 2) {
      offsetType = restructure.uint16;
    } else if (offSize === 3) {
      offsetType = restructure.uint24;
    } else if (offSize === 4) {
      offsetType = restructure.uint32;
    } else {
      throw new Error("Bad offset size in CFFIndex: " + offSize + " " + stream.pos);
    }

    var ret = [];
    var startPos = stream.pos + (count + 1) * offSize - 1;
    var start = offsetType.decode(stream);

    for (var i = 0; i < count; i++) {
      var end = offsetType.decode(stream);

      if (this.type != null) {
        var pos = stream.pos;
        stream.pos = startPos + start;
        parent.length = end - start;
        ret.push(this.type.decode(stream, parent));
        stream.pos = pos;
      } else {
        ret.push({
          offset: startPos + start,
          length: end - start
        });
      }

      start = end;
    }

    stream.pos = startPos + start;
    return ret;
  };

  _proto.size = function size(arr, parent) {
    var size = 2;

    if (arr.length === 0) {
      return size;
    }

    var type = this.type || new restructure.Buffer(); // find maximum offset to detminine offset type

    var offset = 1;

    for (var i = 0; i < arr.length; i++) {
      var item = arr[i];
      offset += type.size(item, parent);
    }

    var offsetType;

    if (offset <= 0xff) {
      offsetType = restructure.uint8;
    } else if (offset <= 0xffff) {
      offsetType = restructure.uint16;
    } else if (offset <= 0xffffff) {
      offsetType = restructure.uint24;
    } else if (offset <= 0xffffffff) {
      offsetType = restructure.uint32;
    } else {
      throw new Error("Bad offset in CFFIndex");
    }

    size += 1 + offsetType.size() * (arr.length + 1);
    size += offset - 1;
    return size;
  };

  _proto.encode = function encode(stream, arr, parent) {
    stream.writeUInt16BE(arr.length);

    if (arr.length === 0) {
      return;
    }

    var type = this.type || new restructure.Buffer(); // find maximum offset to detminine offset type

    var sizes = [];
    var offset = 1;

    for (var _iterator = arr, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
      var _ref;

      if (_isArray) {
        if (_i >= _iterator.length) break;
        _ref = _iterator[_i++];
      } else {
        _i = _iterator.next();
        if (_i.done) break;
        _ref = _i.value;
      }

      var item = _ref;
      var s = type.size(item, parent);
      sizes.push(s);
      offset += s;
    }

    var offsetType;

    if (offset <= 0xff) {
      offsetType = restructure.uint8;
    } else if (offset <= 0xffff) {
      offsetType = restructure.uint16;
    } else if (offset <= 0xffffff) {
      offsetType = restructure.uint24;
    } else if (offset <= 0xffffffff) {
      offsetType = restructure.uint32;
    } else {
      throw new Error("Bad offset in CFFIndex");
    } // write offset size


    stream.writeUInt8(offsetType.size()); // write elements

    offset = 1;
    offsetType.encode(stream, offset);

    for (var _i2 = 0; _i2 < sizes.length; _i2++) {
      var size = sizes[_i2];
      offset += size;
      offsetType.encode(stream, offset);
    }

    for (var _iterator2 = arr, _isArray2 = Array.isArray(_iterator2), _i3 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
      var _ref2;

      if (_isArray2) {
        if (_i3 >= _iterator2.length) break;
        _ref2 = _iterator2[_i3++];
      } else {
        _i3 = _iterator2.next();
        if (_i3.done) break;
        _ref2 = _i3.value;
      }

      var _item = _ref2;
      type.encode(stream, _item, parent);
    }

    return;
  };

  return CFFIndex;
}();

var keys$1 = createCommonjsModule(function (module, exports) {
  exports = module.exports = typeof Object.keys === 'function' ? Object.keys : shim;
  exports.shim = shim;

  function shim(obj) {
    var keys = [];

    for (var key in obj) {
      keys.push(key);
    }

    return keys;
  }
});
var keys_1 = keys$1.shim;

var is_arguments = createCommonjsModule(function (module, exports) {
  var supportsArgumentsClass = function () {
    return Object.prototype.toString.call(arguments);
  }() == '[object Arguments]';

  exports = module.exports = supportsArgumentsClass ? supported : unsupported;
  exports.supported = supported;

  function supported(object) {
    return Object.prototype.toString.call(object) == '[object Arguments]';
  }
  exports.unsupported = unsupported;

  function unsupported(object) {
    return object && typeof object == 'object' && typeof object.length == 'number' && Object.prototype.hasOwnProperty.call(object, 'callee') && !Object.prototype.propertyIsEnumerable.call(object, 'callee') || false;
  }
});
var is_arguments_1 = is_arguments.supported;
var is_arguments_2 = is_arguments.unsupported;

var deepEqual_1 = createCommonjsModule(function (module) {
  var pSlice = Array.prototype.slice;

  var deepEqual = module.exports = function (actual, expected, opts) {
    if (!opts) opts = {}; // 7.1. All identical values are equivalent, as determined by ===.

    if (actual === expected) {
      return true;
    } else if (actual instanceof Date && expected instanceof Date) {
      return actual.getTime() === expected.getTime(); // 7.3. Other pairs that do not both pass typeof value == 'object',
      // equivalence is determined by ==.
    } else if (!actual || !expected || typeof actual != 'object' && typeof expected != 'object') {
      return opts.strict ? actual === expected : actual == expected; // 7.4. For all other Object pairs, including Array objects, equivalence is
      // determined by having the same number of owned properties (as verified
      // with Object.prototype.hasOwnProperty.call), the same set of keys
      // (although not necessarily the same order), equivalent values for every
      // corresponding key, and an identical 'prototype' property. Note: this
      // accounts for both named and indexed properties on Arrays.
    } else {
      return objEquiv(actual, expected, opts);
    }
  };

  function isUndefinedOrNull(value) {
    return value === null || value === undefined;
  }

  function isBuffer(x) {
    if (!x || typeof x !== 'object' || typeof x.length !== 'number') return false;

    if (typeof x.copy !== 'function' || typeof x.slice !== 'function') {
      return false;
    }

    if (x.length > 0 && typeof x[0] !== 'number') return false;
    return true;
  }

  function objEquiv(a, b, opts) {
    var i, key;
    if (isUndefinedOrNull(a) || isUndefinedOrNull(b)) return false; // an identical 'prototype' property.

    if (a.prototype !== b.prototype) return false; //~~~I've managed to break Object.keys through screwy arguments passing.
    //   Converting to array solves the problem.

    if (is_arguments(a)) {
      if (!is_arguments(b)) {
        return false;
      }

      a = pSlice.call(a);
      b = pSlice.call(b);
      return deepEqual(a, b, opts);
    }

    if (isBuffer(a)) {
      if (!isBuffer(b)) {
        return false;
      }

      if (a.length !== b.length) return false;

      for (i = 0; i < a.length; i++) {
        if (a[i] !== b[i]) return false;
      }

      return true;
    }

    try {
      var ka = keys$1(a),
          kb = keys$1(b);
    } catch (e) {
      //happens when one is a string literal and the other isn't
      return false;
    } // having the same number of owned properties (keys incorporates
    // hasOwnProperty)


    if (ka.length != kb.length) return false; //the same set of keys (although not necessarily the same order),

    ka.sort();
    kb.sort(); //~~~cheap key test

    for (i = ka.length - 1; i >= 0; i--) {
      if (ka[i] != kb[i]) return false;
    } //equivalent values for every corresponding key, and
    //~~~possibly expensive deep test


    for (i = ka.length - 1; i >= 0; i--) {
      key = ka[i];
      if (!deepEqual(a[key], b[key], opts)) return false;
    }

    return typeof a === typeof b;
  }
});

var FLOAT_EOF = 0xf;
var FLOAT_LOOKUP = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '.', 'E', 'E-', null, '-'];
var FLOAT_ENCODE_LOOKUP = {
  '.': 10,
  'E': 11,
  'E-': 12,
  '-': 14
};

var CFFOperand =
/*#__PURE__*/
function () {
  function CFFOperand() {}

  CFFOperand.decode = function decode(stream, value) {
    if (32 <= value && value <= 246) {
      return value - 139;
    }

    if (247 <= value && value <= 250) {
      return (value - 247) * 256 + stream.readUInt8() + 108;
    }

    if (251 <= value && value <= 254) {
      return -(value - 251) * 256 - stream.readUInt8() - 108;
    }

    if (value === 28) {
      return stream.readInt16BE();
    }

    if (value === 29) {
      return stream.readInt32BE();
    }

    if (value === 30) {
      var str = '';

      while (true) {
        var b = stream.readUInt8();
        var n1 = b >> 4;

        if (n1 === FLOAT_EOF) {
          break;
        }

        str += FLOAT_LOOKUP[n1];
        var n2 = b & 15;

        if (n2 === FLOAT_EOF) {
          break;
        }

        str += FLOAT_LOOKUP[n2];
      }

      return parseFloat(str);
    }

    return null;
  };

  CFFOperand.size = function size(value) {
    // if the value needs to be forced to the largest size (32 bit)
    // e.g. for unknown pointers, set to 32768
    if (value.forceLarge) {
      value = 32768;
    }

    if ((value | 0) !== value) {
      // floating point
      var str = '' + value;
      return 1 + Math.ceil((str.length + 1) / 2);
    } else if (-107 <= value && value <= 107) {
      return 1;
    } else if (108 <= value && value <= 1131 || -1131 <= value && value <= -108) {
      return 2;
    } else if (-32768 <= value && value <= 32767) {
      return 3;
    } else {
      return 5;
    }
  };

  CFFOperand.encode = function encode(stream, value) {
    // if the value needs to be forced to the largest size (32 bit)
    // e.g. for unknown pointers, save the old value and set to 32768
    var val = Number(value);

    if (value.forceLarge) {
      stream.writeUInt8(29);
      return stream.writeInt32BE(val);
    } else if ((val | 0) !== val) {
      // floating point
      stream.writeUInt8(30);
      var str = '' + val;

      for (var i = 0; i < str.length; i += 2) {
        var c1 = str[i];
        var n1 = FLOAT_ENCODE_LOOKUP[c1] || +c1;

        if (i === str.length - 1) {
          var n2 = FLOAT_EOF;
        } else {
          var c2 = str[i + 1];
          var n2 = FLOAT_ENCODE_LOOKUP[c2] || +c2;
        }

        stream.writeUInt8(n1 << 4 | n2 & 15);
      }

      if (n2 !== FLOAT_EOF) {
        return stream.writeUInt8(FLOAT_EOF << 4);
      }
    } else if (-107 <= val && val <= 107) {
      return stream.writeUInt8(val + 139);
    } else if (108 <= val && val <= 1131) {
      val -= 108;
      stream.writeUInt8((val >> 8) + 247);
      return stream.writeUInt8(val & 0xff);
    } else if (-1131 <= val && val <= -108) {
      val = -val - 108;
      stream.writeUInt8((val >> 8) + 251);
      return stream.writeUInt8(val & 0xff);
    } else if (-32768 <= val && val <= 32767) {
      stream.writeUInt8(28);
      return stream.writeInt16BE(val);
    } else {
      stream.writeUInt8(29);
      return stream.writeInt32BE(val);
    }
  };

  return CFFOperand;
}();

var CFFDict =
/*#__PURE__*/
function () {
  function CFFDict(ops) {
    if (ops === void 0) {
      ops = [];
    }

    this.ops = ops;
    this.fields = {};

    for (var _iterator = ops, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
      var _ref;

      if (_isArray) {
        if (_i >= _iterator.length) break;
        _ref = _iterator[_i++];
      } else {
        _i = _iterator.next();
        if (_i.done) break;
        _ref = _i.value;
      }

      var field = _ref;
      var key = Array.isArray(field[0]) ? field[0][0] << 8 | field[0][1] : field[0];
      this.fields[key] = field;
    }
  }

  var _proto = CFFDict.prototype;

  _proto.decodeOperands = function decodeOperands(type, stream, ret, operands) {
    var _this = this;

    if (Array.isArray(type)) {
      return operands.map(function (op, i) {
        return _this.decodeOperands(type[i], stream, ret, [op]);
      });
    } else if (type.decode != null) {
      return type.decode(stream, ret, operands);
    } else {
      switch (type) {
        case 'number':
        case 'offset':
        case 'sid':
          return operands[0];

        case 'boolean':
          return !!operands[0];

        default:
          return operands;
      }
    }
  };

  _proto.encodeOperands = function encodeOperands(type, stream, ctx, operands) {
    var _this2 = this;

    if (Array.isArray(type)) {
      return operands.map(function (op, i) {
        return _this2.encodeOperands(type[i], stream, ctx, op)[0];
      });
    } else if (type.encode != null) {
      return type.encode(stream, operands, ctx);
    } else if (typeof operands === 'number') {
      return [operands];
    } else if (typeof operands === 'boolean') {
      return [+operands];
    } else if (Array.isArray(operands)) {
      return operands;
    } else {
      return [operands];
    }
  };

  _proto.decode = function decode(stream, parent) {
    var end = stream.pos + parent.length;
    var ret = {};
    var operands = []; // define hidden properties

    Object.defineProperties(ret, {
      parent: {
        value: parent
      },
      _startOffset: {
        value: stream.pos
      }
    }); // fill in defaults

    for (var key in this.fields) {
      var field = this.fields[key];
      ret[field[1]] = field[3];
    }

    while (stream.pos < end) {
      var b = stream.readUInt8();

      if (b < 28) {
        if (b === 12) {
          b = b << 8 | stream.readUInt8();
        }

        var _field = this.fields[b];

        if (!_field) {
          throw new Error("Unknown operator " + b);
        }

        var val = this.decodeOperands(_field[2], stream, ret, operands);

        if (val != null) {
          if (val instanceof utils_2) {
            Object.defineProperty(ret, _field[1], val);
          } else {
            ret[_field[1]] = val;
          }
        }

        operands = [];
      } else {
        operands.push(CFFOperand.decode(stream, b));
      }
    }

    return ret;
  };

  _proto.size = function size(dict, parent, includePointers) {
    if (includePointers === void 0) {
      includePointers = true;
    }

    var ctx = {
      parent: parent,
      val: dict,
      pointerSize: 0,
      startOffset: parent.startOffset || 0
    };
    var len = 0;

    for (var k in this.fields) {
      var field = this.fields[k];
      var val = dict[field[1]];

      if (val == null || deepEqual_1(val, field[3])) {
        continue;
      }

      var operands = this.encodeOperands(field[2], null, ctx, val);

      for (var _iterator2 = operands, _isArray2 = Array.isArray(_iterator2), _i2 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
        var _ref2;

        if (_isArray2) {
          if (_i2 >= _iterator2.length) break;
          _ref2 = _iterator2[_i2++];
        } else {
          _i2 = _iterator2.next();
          if (_i2.done) break;
          _ref2 = _i2.value;
        }

        var op = _ref2;
        len += CFFOperand.size(op);
      }

      var key = Array.isArray(field[0]) ? field[0] : [field[0]];
      len += key.length;
    }

    if (includePointers) {
      len += ctx.pointerSize;
    }

    return len;
  };

  _proto.encode = function encode(stream, dict, parent) {
    var ctx = {
      pointers: [],
      startOffset: stream.pos,
      parent: parent,
      val: dict,
      pointerSize: 0
    };
    ctx.pointerOffset = stream.pos + this.size(dict, ctx, false);

    for (var _iterator3 = this.ops, _isArray3 = Array.isArray(_iterator3), _i3 = 0, _iterator3 = _isArray3 ? _iterator3 : _iterator3[Symbol.iterator]();;) {
      var _ref3;

      if (_isArray3) {
        if (_i3 >= _iterator3.length) break;
        _ref3 = _iterator3[_i3++];
      } else {
        _i3 = _iterator3.next();
        if (_i3.done) break;
        _ref3 = _i3.value;
      }

      var field = _ref3;
      var val = dict[field[1]];

      if (val == null || deepEqual_1(val, field[3])) {
        continue;
      }

      var operands = this.encodeOperands(field[2], stream, ctx, val);

      for (var _iterator4 = operands, _isArray4 = Array.isArray(_iterator4), _i4 = 0, _iterator4 = _isArray4 ? _iterator4 : _iterator4[Symbol.iterator]();;) {
        var _ref4;

        if (_isArray4) {
          if (_i4 >= _iterator4.length) break;
          _ref4 = _iterator4[_i4++];
        } else {
          _i4 = _iterator4.next();
          if (_i4.done) break;
          _ref4 = _i4.value;
        }

        var op = _ref4;
        CFFOperand.encode(stream, op);
      }

      var key = Array.isArray(field[0]) ? field[0] : [field[0]];

      for (var _iterator5 = key, _isArray5 = Array.isArray(_iterator5), _i5 = 0, _iterator5 = _isArray5 ? _iterator5 : _iterator5[Symbol.iterator]();;) {
        var _ref5;

        if (_isArray5) {
          if (_i5 >= _iterator5.length) break;
          _ref5 = _iterator5[_i5++];
        } else {
          _i5 = _iterator5.next();
          if (_i5.done) break;
          _ref5 = _i5.value;
        }

        var _op = _ref5;
        stream.writeUInt8(_op);
      }
    }

    var i = 0;

    while (i < ctx.pointers.length) {
      var ptr = ctx.pointers[i++];
      ptr.type.encode(stream, ptr.val, ptr.parent);
    }

    return;
  };

  return CFFDict;
}();

var CFFPointer =
/*#__PURE__*/
function (_r$Pointer) {
  _inheritsLoose(CFFPointer, _r$Pointer);

  function CFFPointer(type, options) {
    if (options === void 0) {
      options = {};
    }

    if (options.type == null) {
      options.type = 'global';
    }

    return _r$Pointer.call(this, null, type, options) || this;
  }

  var _proto = CFFPointer.prototype;

  _proto.decode = function decode(stream, parent, operands) {
    this.offsetType = {
      decode: function decode() {
        return operands[0];
      }
    };
    return _r$Pointer.prototype.decode.call(this, stream, parent, operands);
  };

  _proto.encode = function encode(stream, value, ctx) {
    if (!stream) {
      // compute the size (so ctx.pointerSize is correct)
      this.offsetType = {
        size: function size() {
          return 0;
        }
      };
      this.size(value, ctx);
      return [new Ptr(0)];
    }

    var ptr = null;
    this.offsetType = {
      encode: function encode(stream, val) {
        return ptr = val;
      }
    };

    _r$Pointer.prototype.encode.call(this, stream, value, ctx);

    return [new Ptr(ptr)];
  };

  return CFFPointer;
}(restructure.Pointer);

var Ptr =
/*#__PURE__*/
function () {
  function Ptr(val) {
    this.val = val;
    this.forceLarge = true;
  }

  var _proto2 = Ptr.prototype;

  _proto2.valueOf = function valueOf() {
    return this.val;
  };

  return Ptr;
}();

var CFFBlendOp =
/*#__PURE__*/
function () {
  function CFFBlendOp() {}

  CFFBlendOp.decode = function decode(stream, parent, operands) {
    var numBlends = operands.pop(); // TODO: actually blend. For now just consume the deltas
    // since we don't use any of the values anyway.

    while (operands.length > numBlends) {
      operands.pop();
    }
  };

  return CFFBlendOp;
}();

var CFFPrivateDict = new CFFDict([// key       name                    type                                          default
[6, 'BlueValues', 'delta', null], [7, 'OtherBlues', 'delta', null], [8, 'FamilyBlues', 'delta', null], [9, 'FamilyOtherBlues', 'delta', null], [[12, 9], 'BlueScale', 'number', 0.039625], [[12, 10], 'BlueShift', 'number', 7], [[12, 11], 'BlueFuzz', 'number', 1], [10, 'StdHW', 'number', null], [11, 'StdVW', 'number', null], [[12, 12], 'StemSnapH', 'delta', null], [[12, 13], 'StemSnapV', 'delta', null], [[12, 14], 'ForceBold', 'boolean', false], [[12, 17], 'LanguageGroup', 'number', 0], [[12, 18], 'ExpansionFactor', 'number', 0.06], [[12, 19], 'initialRandomSeed', 'number', 0], [20, 'defaultWidthX', 'number', 0], [21, 'nominalWidthX', 'number', 0], [22, 'vsindex', 'number', 0], [23, 'blend', CFFBlendOp, null], [19, 'Subrs', new CFFPointer(new CFFIndex(), {
  type: 'local'
}), null]]);

// Automatically generated from Appendix A of the CFF specification; do
// not edit. Length should be 391.
var standardStrings = [".notdef", "space", "exclam", "quotedbl", "numbersign", "dollar", "percent", "ampersand", "quoteright", "parenleft", "parenright", "asterisk", "plus", "comma", "hyphen", "period", "slash", "zero", "one", "two", "three", "four", "five", "six", "seven", "eight", "nine", "colon", "semicolon", "less", "equal", "greater", "question", "at", "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z", "bracketleft", "backslash", "bracketright", "asciicircum", "underscore", "quoteleft", "a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s", "t", "u", "v", "w", "x", "y", "z", "braceleft", "bar", "braceright", "asciitilde", "exclamdown", "cent", "sterling", "fraction", "yen", "florin", "section", "currency", "quotesingle", "quotedblleft", "guillemotleft", "guilsinglleft", "guilsinglright", "fi", "fl", "endash", "dagger", "daggerdbl", "periodcentered", "paragraph", "bullet", "quotesinglbase", "quotedblbase", "quotedblright", "guillemotright", "ellipsis", "perthousand", "questiondown", "grave", "acute", "circumflex", "tilde", "macron", "breve", "dotaccent", "dieresis", "ring", "cedilla", "hungarumlaut", "ogonek", "caron", "emdash", "AE", "ordfeminine", "Lslash", "Oslash", "OE", "ordmasculine", "ae", "dotlessi", "lslash", "oslash", "oe", "germandbls", "onesuperior", "logicalnot", "mu", "trademark", "Eth", "onehalf", "plusminus", "Thorn", "onequarter", "divide", "brokenbar", "degree", "thorn", "threequarters", "twosuperior", "registered", "minus", "eth", "multiply", "threesuperior", "copyright", "Aacute", "Acircumflex", "Adieresis", "Agrave", "Aring", "Atilde", "Ccedilla", "Eacute", "Ecircumflex", "Edieresis", "Egrave", "Iacute", "Icircumflex", "Idieresis", "Igrave", "Ntilde", "Oacute", "Ocircumflex", "Odieresis", "Ograve", "Otilde", "Scaron", "Uacute", "Ucircumflex", "Udieresis", "Ugrave", "Yacute", "Ydieresis", "Zcaron", "aacute", "acircumflex", "adieresis", "agrave", "aring", "atilde", "ccedilla", "eacute", "ecircumflex", "edieresis", "egrave", "iacute", "icircumflex", "idieresis", "igrave", "ntilde", "oacute", "ocircumflex", "odieresis", "ograve", "otilde", "scaron", "uacute", "ucircumflex", "udieresis", "ugrave", "yacute", "ydieresis", "zcaron", "exclamsmall", "Hungarumlautsmall", "dollaroldstyle", "dollarsuperior", "ampersandsmall", "Acutesmall", "parenleftsuperior", "parenrightsuperior", "twodotenleader", "onedotenleader", "zerooldstyle", "oneoldstyle", "twooldstyle", "threeoldstyle", "fouroldstyle", "fiveoldstyle", "sixoldstyle", "sevenoldstyle", "eightoldstyle", "nineoldstyle", "commasuperior", "threequartersemdash", "periodsuperior", "questionsmall", "asuperior", "bsuperior", "centsuperior", "dsuperior", "esuperior", "isuperior", "lsuperior", "msuperior", "nsuperior", "osuperior", "rsuperior", "ssuperior", "tsuperior", "ff", "ffi", "ffl", "parenleftinferior", "parenrightinferior", "Circumflexsmall", "hyphensuperior", "Gravesmall", "Asmall", "Bsmall", "Csmall", "Dsmall", "Esmall", "Fsmall", "Gsmall", "Hsmall", "Ismall", "Jsmall", "Ksmall", "Lsmall", "Msmall", "Nsmall", "Osmall", "Psmall", "Qsmall", "Rsmall", "Ssmall", "Tsmall", "Usmall", "Vsmall", "Wsmall", "Xsmall", "Ysmall", "Zsmall", "colonmonetary", "onefitted", "rupiah", "Tildesmall", "exclamdownsmall", "centoldstyle", "Lslashsmall", "Scaronsmall", "Zcaronsmall", "Dieresissmall", "Brevesmall", "Caronsmall", "Dotaccentsmall", "Macronsmall", "figuredash", "hypheninferior", "Ogoneksmall", "Ringsmall", "Cedillasmall", "questiondownsmall", "oneeighth", "threeeighths", "fiveeighths", "seveneighths", "onethird", "twothirds", "zerosuperior", "foursuperior", "fivesuperior", "sixsuperior", "sevensuperior", "eightsuperior", "ninesuperior", "zeroinferior", "oneinferior", "twoinferior", "threeinferior", "fourinferior", "fiveinferior", "sixinferior", "seveninferior", "eightinferior", "nineinferior", "centinferior", "dollarinferior", "periodinferior", "commainferior", "Agravesmall", "Aacutesmall", "Acircumflexsmall", "Atildesmall", "Adieresissmall", "Aringsmall", "AEsmall", "Ccedillasmall", "Egravesmall", "Eacutesmall", "Ecircumflexsmall", "Edieresissmall", "Igravesmall", "Iacutesmall", "Icircumflexsmall", "Idieresissmall", "Ethsmall", "Ntildesmall", "Ogravesmall", "Oacutesmall", "Ocircumflexsmall", "Otildesmall", "Odieresissmall", "OEsmall", "Oslashsmall", "Ugravesmall", "Uacutesmall", "Ucircumflexsmall", "Udieresissmall", "Yacutesmall", "Thornsmall", "Ydieresissmall", "001.000", "001.001", "001.002", "001.003", "Black", "Bold", "Book", "Light", "Medium", "Regular", "Roman", "Semibold"];

var StandardEncoding = ['', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', 'space', 'exclam', 'quotedbl', 'numbersign', 'dollar', 'percent', 'ampersand', 'quoteright', 'parenleft', 'parenright', 'asterisk', 'plus', 'comma', 'hyphen', 'period', 'slash', 'zero', 'one', 'two', 'three', 'four', 'five', 'six', 'seven', 'eight', 'nine', 'colon', 'semicolon', 'less', 'equal', 'greater', 'question', 'at', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'bracketleft', 'backslash', 'bracketright', 'asciicircum', 'underscore', 'quoteleft', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', 'braceleft', 'bar', 'braceright', 'asciitilde', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', 'exclamdown', 'cent', 'sterling', 'fraction', 'yen', 'florin', 'section', 'currency', 'quotesingle', 'quotedblleft', 'guillemotleft', 'guilsinglleft', 'guilsinglright', 'fi', 'fl', '', 'endash', 'dagger', 'daggerdbl', 'periodcentered', '', 'paragraph', 'bullet', 'quotesinglbase', 'quotedblbase', 'quotedblright', 'guillemotright', 'ellipsis', 'perthousand', '', 'questiondown', '', 'grave', 'acute', 'circumflex', 'tilde', 'macron', 'breve', 'dotaccent', 'dieresis', '', 'ring', 'cedilla', '', 'hungarumlaut', 'ogonek', 'caron', 'emdash', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', 'AE', '', 'ordfeminine', '', '', '', '', 'Lslash', 'Oslash', 'OE', 'ordmasculine', '', '', '', '', '', 'ae', '', '', '', 'dotlessi', '', '', 'lslash', 'oslash', 'oe', 'germandbls'];
var ExpertEncoding = ['', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', 'space', 'exclamsmall', 'Hungarumlautsmall', '', 'dollaroldstyle', 'dollarsuperior', 'ampersandsmall', 'Acutesmall', 'parenleftsuperior', 'parenrightsuperior', 'twodotenleader', 'onedotenleader', 'comma', 'hyphen', 'period', 'fraction', 'zerooldstyle', 'oneoldstyle', 'twooldstyle', 'threeoldstyle', 'fouroldstyle', 'fiveoldstyle', 'sixoldstyle', 'sevenoldstyle', 'eightoldstyle', 'nineoldstyle', 'colon', 'semicolon', 'commasuperior', 'threequartersemdash', 'periodsuperior', 'questionsmall', '', 'asuperior', 'bsuperior', 'centsuperior', 'dsuperior', 'esuperior', '', '', 'isuperior', '', '', 'lsuperior', 'msuperior', 'nsuperior', 'osuperior', '', '', 'rsuperior', 'ssuperior', 'tsuperior', '', 'ff', 'fi', 'fl', 'ffi', 'ffl', 'parenleftinferior', '', 'parenrightinferior', 'Circumflexsmall', 'hyphensuperior', 'Gravesmall', 'Asmall', 'Bsmall', 'Csmall', 'Dsmall', 'Esmall', 'Fsmall', 'Gsmall', 'Hsmall', 'Ismall', 'Jsmall', 'Ksmall', 'Lsmall', 'Msmall', 'Nsmall', 'Osmall', 'Psmall', 'Qsmall', 'Rsmall', 'Ssmall', 'Tsmall', 'Usmall', 'Vsmall', 'Wsmall', 'Xsmall', 'Ysmall', 'Zsmall', 'colonmonetary', 'onefitted', 'rupiah', 'Tildesmall', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', 'exclamdownsmall', 'centoldstyle', 'Lslashsmall', '', '', 'Scaronsmall', 'Zcaronsmall', 'Dieresissmall', 'Brevesmall', 'Caronsmall', '', 'Dotaccentsmall', '', '', 'Macronsmall', '', '', 'figuredash', 'hypheninferior', '', '', 'Ogoneksmall', 'Ringsmall', 'Cedillasmall', '', '', '', 'onequarter', 'onehalf', 'threequarters', 'questiondownsmall', 'oneeighth', 'threeeighths', 'fiveeighths', 'seveneighths', 'onethird', 'twothirds', '', '', 'zerosuperior', 'onesuperior', 'twosuperior', 'threesuperior', 'foursuperior', 'fivesuperior', 'sixsuperior', 'sevensuperior', 'eightsuperior', 'ninesuperior', 'zeroinferior', 'oneinferior', 'twoinferior', 'threeinferior', 'fourinferior', 'fiveinferior', 'sixinferior', 'seveninferior', 'eightinferior', 'nineinferior', 'centinferior', 'dollarinferior', 'periodinferior', 'commainferior', 'Agravesmall', 'Aacutesmall', 'Acircumflexsmall', 'Atildesmall', 'Adieresissmall', 'Aringsmall', 'AEsmall', 'Ccedillasmall', 'Egravesmall', 'Eacutesmall', 'Ecircumflexsmall', 'Edieresissmall', 'Igravesmall', 'Iacutesmall', 'Icircumflexsmall', 'Idieresissmall', 'Ethsmall', 'Ntildesmall', 'Ogravesmall', 'Oacutesmall', 'Ocircumflexsmall', 'Otildesmall', 'Odieresissmall', 'OEsmall', 'Oslashsmall', 'Ugravesmall', 'Uacutesmall', 'Ucircumflexsmall', 'Udieresissmall', 'Yacutesmall', 'Thornsmall', 'Ydieresissmall'];

var ISOAdobeCharset = ['.notdef', 'space', 'exclam', 'quotedbl', 'numbersign', 'dollar', 'percent', 'ampersand', 'quoteright', 'parenleft', 'parenright', 'asterisk', 'plus', 'comma', 'hyphen', 'period', 'slash', 'zero', 'one', 'two', 'three', 'four', 'five', 'six', 'seven', 'eight', 'nine', 'colon', 'semicolon', 'less', 'equal', 'greater', 'question', 'at', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'bracketleft', 'backslash', 'bracketright', 'asciicircum', 'underscore', 'quoteleft', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', 'braceleft', 'bar', 'braceright', 'asciitilde', 'exclamdown', 'cent', 'sterling', 'fraction', 'yen', 'florin', 'section', 'currency', 'quotesingle', 'quotedblleft', 'guillemotleft', 'guilsinglleft', 'guilsinglright', 'fi', 'fl', 'endash', 'dagger', 'daggerdbl', 'periodcentered', 'paragraph', 'bullet', 'quotesinglbase', 'quotedblbase', 'quotedblright', 'guillemotright', 'ellipsis', 'perthousand', 'questiondown', 'grave', 'acute', 'circumflex', 'tilde', 'macron', 'breve', 'dotaccent', 'dieresis', 'ring', 'cedilla', 'hungarumlaut', 'ogonek', 'caron', 'emdash', 'AE', 'ordfeminine', 'Lslash', 'Oslash', 'OE', 'ordmasculine', 'ae', 'dotlessi', 'lslash', 'oslash', 'oe', 'germandbls', 'onesuperior', 'logicalnot', 'mu', 'trademark', 'Eth', 'onehalf', 'plusminus', 'Thorn', 'onequarter', 'divide', 'brokenbar', 'degree', 'thorn', 'threequarters', 'twosuperior', 'registered', 'minus', 'eth', 'multiply', 'threesuperior', 'copyright', 'Aacute', 'Acircumflex', 'Adieresis', 'Agrave', 'Aring', 'Atilde', 'Ccedilla', 'Eacute', 'Ecircumflex', 'Edieresis', 'Egrave', 'Iacute', 'Icircumflex', 'Idieresis', 'Igrave', 'Ntilde', 'Oacute', 'Ocircumflex', 'Odieresis', 'Ograve', 'Otilde', 'Scaron', 'Uacute', 'Ucircumflex', 'Udieresis', 'Ugrave', 'Yacute', 'Ydieresis', 'Zcaron', 'aacute', 'acircumflex', 'adieresis', 'agrave', 'aring', 'atilde', 'ccedilla', 'eacute', 'ecircumflex', 'edieresis', 'egrave', 'iacute', 'icircumflex', 'idieresis', 'igrave', 'ntilde', 'oacute', 'ocircumflex', 'odieresis', 'ograve', 'otilde', 'scaron', 'uacute', 'ucircumflex', 'udieresis', 'ugrave', 'yacute', 'ydieresis', 'zcaron'];
var ExpertCharset = ['.notdef', 'space', 'exclamsmall', 'Hungarumlautsmall', 'dollaroldstyle', 'dollarsuperior', 'ampersandsmall', 'Acutesmall', 'parenleftsuperior', 'parenrightsuperior', 'twodotenleader', 'onedotenleader', 'comma', 'hyphen', 'period', 'fraction', 'zerooldstyle', 'oneoldstyle', 'twooldstyle', 'threeoldstyle', 'fouroldstyle', 'fiveoldstyle', 'sixoldstyle', 'sevenoldstyle', 'eightoldstyle', 'nineoldstyle', 'colon', 'semicolon', 'commasuperior', 'threequartersemdash', 'periodsuperior', 'questionsmall', 'asuperior', 'bsuperior', 'centsuperior', 'dsuperior', 'esuperior', 'isuperior', 'lsuperior', 'msuperior', 'nsuperior', 'osuperior', 'rsuperior', 'ssuperior', 'tsuperior', 'ff', 'fi', 'fl', 'ffi', 'ffl', 'parenleftinferior', 'parenrightinferior', 'Circumflexsmall', 'hyphensuperior', 'Gravesmall', 'Asmall', 'Bsmall', 'Csmall', 'Dsmall', 'Esmall', 'Fsmall', 'Gsmall', 'Hsmall', 'Ismall', 'Jsmall', 'Ksmall', 'Lsmall', 'Msmall', 'Nsmall', 'Osmall', 'Psmall', 'Qsmall', 'Rsmall', 'Ssmall', 'Tsmall', 'Usmall', 'Vsmall', 'Wsmall', 'Xsmall', 'Ysmall', 'Zsmall', 'colonmonetary', 'onefitted', 'rupiah', 'Tildesmall', 'exclamdownsmall', 'centoldstyle', 'Lslashsmall', 'Scaronsmall', 'Zcaronsmall', 'Dieresissmall', 'Brevesmall', 'Caronsmall', 'Dotaccentsmall', 'Macronsmall', 'figuredash', 'hypheninferior', 'Ogoneksmall', 'Ringsmall', 'Cedillasmall', 'onequarter', 'onehalf', 'threequarters', 'questiondownsmall', 'oneeighth', 'threeeighths', 'fiveeighths', 'seveneighths', 'onethird', 'twothirds', 'zerosuperior', 'onesuperior', 'twosuperior', 'threesuperior', 'foursuperior', 'fivesuperior', 'sixsuperior', 'sevensuperior', 'eightsuperior', 'ninesuperior', 'zeroinferior', 'oneinferior', 'twoinferior', 'threeinferior', 'fourinferior', 'fiveinferior', 'sixinferior', 'seveninferior', 'eightinferior', 'nineinferior', 'centinferior', 'dollarinferior', 'periodinferior', 'commainferior', 'Agravesmall', 'Aacutesmall', 'Acircumflexsmall', 'Atildesmall', 'Adieresissmall', 'Aringsmall', 'AEsmall', 'Ccedillasmall', 'Egravesmall', 'Eacutesmall', 'Ecircumflexsmall', 'Edieresissmall', 'Igravesmall', 'Iacutesmall', 'Icircumflexsmall', 'Idieresissmall', 'Ethsmall', 'Ntildesmall', 'Ogravesmall', 'Oacutesmall', 'Ocircumflexsmall', 'Otildesmall', 'Odieresissmall', 'OEsmall', 'Oslashsmall', 'Ugravesmall', 'Uacutesmall', 'Ucircumflexsmall', 'Udieresissmall', 'Yacutesmall', 'Thornsmall', 'Ydieresissmall'];
var ExpertSubsetCharset = ['.notdef', 'space', 'dollaroldstyle', 'dollarsuperior', 'parenleftsuperior', 'parenrightsuperior', 'twodotenleader', 'onedotenleader', 'comma', 'hyphen', 'period', 'fraction', 'zerooldstyle', 'oneoldstyle', 'twooldstyle', 'threeoldstyle', 'fouroldstyle', 'fiveoldstyle', 'sixoldstyle', 'sevenoldstyle', 'eightoldstyle', 'nineoldstyle', 'colon', 'semicolon', 'commasuperior', 'threequartersemdash', 'periodsuperior', 'asuperior', 'bsuperior', 'centsuperior', 'dsuperior', 'esuperior', 'isuperior', 'lsuperior', 'msuperior', 'nsuperior', 'osuperior', 'rsuperior', 'ssuperior', 'tsuperior', 'ff', 'fi', 'fl', 'ffi', 'ffl', 'parenleftinferior', 'parenrightinferior', 'hyphensuperior', 'colonmonetary', 'onefitted', 'rupiah', 'centoldstyle', 'figuredash', 'hypheninferior', 'onequarter', 'onehalf', 'threequarters', 'oneeighth', 'threeeighths', 'fiveeighths', 'seveneighths', 'onethird', 'twothirds', 'zerosuperior', 'onesuperior', 'twosuperior', 'threesuperior', 'foursuperior', 'fivesuperior', 'sixsuperior', 'sevensuperior', 'eightsuperior', 'ninesuperior', 'zeroinferior', 'oneinferior', 'twoinferior', 'threeinferior', 'fourinferior', 'fiveinferior', 'sixinferior', 'seveninferior', 'eightinferior', 'nineinferior', 'centinferior', 'dollarinferior', 'periodinferior', 'commainferior'];

// Scripts and Languages #
//########################

var LangSysTable = new restructure.Struct({
  reserved: new restructure.Reserved(restructure.uint16),
  reqFeatureIndex: restructure.uint16,
  featureCount: restructure.uint16,
  featureIndexes: new restructure.Array(restructure.uint16, 'featureCount')
});
var LangSysRecord = new restructure.Struct({
  tag: new restructure.String(4),
  langSys: new restructure.Pointer(restructure.uint16, LangSysTable, {
    type: 'parent'
  })
});
var Script = new restructure.Struct({
  defaultLangSys: new restructure.Pointer(restructure.uint16, LangSysTable),
  count: restructure.uint16,
  langSysRecords: new restructure.Array(LangSysRecord, 'count')
});
var ScriptRecord = new restructure.Struct({
  tag: new restructure.String(4),
  script: new restructure.Pointer(restructure.uint16, Script, {
    type: 'parent'
  })
});
var ScriptList = new restructure.Array(ScriptRecord, restructure.uint16); //#######################
// Features and Lookups #
//#######################

var Feature = new restructure.Struct({
  featureParams: restructure.uint16,
  // pointer
  lookupCount: restructure.uint16,
  lookupListIndexes: new restructure.Array(restructure.uint16, 'lookupCount')
});
var FeatureRecord = new restructure.Struct({
  tag: new restructure.String(4),
  feature: new restructure.Pointer(restructure.uint16, Feature, {
    type: 'parent'
  })
});
var FeatureList = new restructure.Array(FeatureRecord, restructure.uint16);
var LookupFlags = new restructure.Struct({
  markAttachmentType: restructure.uint8,
  flags: new restructure.Bitfield(restructure.uint8, ['rightToLeft', 'ignoreBaseGlyphs', 'ignoreLigatures', 'ignoreMarks', 'useMarkFilteringSet'])
});
function LookupList(SubTable) {
  var Lookup = new restructure.Struct({
    lookupType: restructure.uint16,
    flags: LookupFlags,
    subTableCount: restructure.uint16,
    subTables: new restructure.Array(new restructure.Pointer(restructure.uint16, SubTable), 'subTableCount'),
    markFilteringSet: new restructure.Optional(restructure.uint16, function (t) {
      return t.flags.flags.useMarkFilteringSet;
    })
  });
  return new restructure.LazyArray(new restructure.Pointer(restructure.uint16, Lookup), restructure.uint16);
} //#################
// Coverage Table #
//#################

var RangeRecord = new restructure.Struct({
  start: restructure.uint16,
  end: restructure.uint16,
  startCoverageIndex: restructure.uint16
});
var Coverage = new restructure.VersionedStruct(restructure.uint16, {
  1: {
    glyphCount: restructure.uint16,
    glyphs: new restructure.Array(restructure.uint16, 'glyphCount')
  },
  2: {
    rangeCount: restructure.uint16,
    rangeRecords: new restructure.Array(RangeRecord, 'rangeCount')
  }
}); //#########################
// Class Definition Table #
//#########################

var ClassRangeRecord = new restructure.Struct({
  start: restructure.uint16,
  end: restructure.uint16,
  class: restructure.uint16
});
var ClassDef = new restructure.VersionedStruct(restructure.uint16, {
  1: {
    // Class array
    startGlyph: restructure.uint16,
    glyphCount: restructure.uint16,
    classValueArray: new restructure.Array(restructure.uint16, 'glyphCount')
  },
  2: {
    // Class ranges
    classRangeCount: restructure.uint16,
    classRangeRecord: new restructure.Array(ClassRangeRecord, 'classRangeCount')
  }
}); //###############
// Device Table #
//###############

var Device = new restructure.Struct({
  a: restructure.uint16,
  // startSize for hinting Device, outerIndex for VariationIndex
  b: restructure.uint16,
  // endSize for Device, innerIndex for VariationIndex
  deltaFormat: restructure.uint16
}); //#############################################
// Contextual Substitution/Positioning Tables #
//#############################################

var LookupRecord = new restructure.Struct({
  sequenceIndex: restructure.uint16,
  lookupListIndex: restructure.uint16
});
var Rule = new restructure.Struct({
  glyphCount: restructure.uint16,
  lookupCount: restructure.uint16,
  input: new restructure.Array(restructure.uint16, function (t) {
    return t.glyphCount - 1;
  }),
  lookupRecords: new restructure.Array(LookupRecord, 'lookupCount')
});
var RuleSet = new restructure.Array(new restructure.Pointer(restructure.uint16, Rule), restructure.uint16);
var ClassRule = new restructure.Struct({
  glyphCount: restructure.uint16,
  lookupCount: restructure.uint16,
  classes: new restructure.Array(restructure.uint16, function (t) {
    return t.glyphCount - 1;
  }),
  lookupRecords: new restructure.Array(LookupRecord, 'lookupCount')
});
var ClassSet = new restructure.Array(new restructure.Pointer(restructure.uint16, ClassRule), restructure.uint16);
var Context = new restructure.VersionedStruct(restructure.uint16, {
  1: {
    // Simple context
    coverage: new restructure.Pointer(restructure.uint16, Coverage),
    ruleSetCount: restructure.uint16,
    ruleSets: new restructure.Array(new restructure.Pointer(restructure.uint16, RuleSet), 'ruleSetCount')
  },
  2: {
    // Class-based context
    coverage: new restructure.Pointer(restructure.uint16, Coverage),
    classDef: new restructure.Pointer(restructure.uint16, ClassDef),
    classSetCnt: restructure.uint16,
    classSet: new restructure.Array(new restructure.Pointer(restructure.uint16, ClassSet), 'classSetCnt')
  },
  3: {
    glyphCount: restructure.uint16,
    lookupCount: restructure.uint16,
    coverages: new restructure.Array(new restructure.Pointer(restructure.uint16, Coverage), 'glyphCount'),
    lookupRecords: new restructure.Array(LookupRecord, 'lookupCount')
  }
}); //######################################################
// Chaining Contextual Substitution/Positioning Tables #
//######################################################

var ChainRule = new restructure.Struct({
  backtrackGlyphCount: restructure.uint16,
  backtrack: new restructure.Array(restructure.uint16, 'backtrackGlyphCount'),
  inputGlyphCount: restructure.uint16,
  input: new restructure.Array(restructure.uint16, function (t) {
    return t.inputGlyphCount - 1;
  }),
  lookaheadGlyphCount: restructure.uint16,
  lookahead: new restructure.Array(restructure.uint16, 'lookaheadGlyphCount'),
  lookupCount: restructure.uint16,
  lookupRecords: new restructure.Array(LookupRecord, 'lookupCount')
});
var ChainRuleSet = new restructure.Array(new restructure.Pointer(restructure.uint16, ChainRule), restructure.uint16);
var ChainingContext = new restructure.VersionedStruct(restructure.uint16, {
  1: {
    // Simple context glyph substitution
    coverage: new restructure.Pointer(restructure.uint16, Coverage),
    chainCount: restructure.uint16,
    chainRuleSets: new restructure.Array(new restructure.Pointer(restructure.uint16, ChainRuleSet), 'chainCount')
  },
  2: {
    // Class-based chaining context
    coverage: new restructure.Pointer(restructure.uint16, Coverage),
    backtrackClassDef: new restructure.Pointer(restructure.uint16, ClassDef),
    inputClassDef: new restructure.Pointer(restructure.uint16, ClassDef),
    lookaheadClassDef: new restructure.Pointer(restructure.uint16, ClassDef),
    chainCount: restructure.uint16,
    chainClassSet: new restructure.Array(new restructure.Pointer(restructure.uint16, ChainRuleSet), 'chainCount')
  },
  3: {
    // Coverage-based chaining context
    backtrackGlyphCount: restructure.uint16,
    backtrackCoverage: new restructure.Array(new restructure.Pointer(restructure.uint16, Coverage), 'backtrackGlyphCount'),
    inputGlyphCount: restructure.uint16,
    inputCoverage: new restructure.Array(new restructure.Pointer(restructure.uint16, Coverage), 'inputGlyphCount'),
    lookaheadGlyphCount: restructure.uint16,
    lookaheadCoverage: new restructure.Array(new restructure.Pointer(restructure.uint16, Coverage), 'lookaheadGlyphCount'),
    lookupCount: restructure.uint16,
    lookupRecords: new restructure.Array(LookupRecord, 'lookupCount')
  }
});

var _;
/*******************
 * Variation Store *
 *******************/

var F2DOT14 = new restructure.Fixed(16, 'BE', 14);
var RegionAxisCoordinates = new restructure.Struct({
  startCoord: F2DOT14,
  peakCoord: F2DOT14,
  endCoord: F2DOT14
});
var VariationRegionList = new restructure.Struct({
  axisCount: restructure.uint16,
  regionCount: restructure.uint16,
  variationRegions: new restructure.Array(new restructure.Array(RegionAxisCoordinates, 'axisCount'), 'regionCount')
});
var DeltaSet = new restructure.Struct({
  shortDeltas: new restructure.Array(restructure.int16, function (t) {
    return t.parent.shortDeltaCount;
  }),
  regionDeltas: new restructure.Array(restructure.int8, function (t) {
    return t.parent.regionIndexCount - t.parent.shortDeltaCount;
  }),
  deltas: function deltas(t) {
    return t.shortDeltas.concat(t.regionDeltas);
  }
});
var ItemVariationData = new restructure.Struct({
  itemCount: restructure.uint16,
  shortDeltaCount: restructure.uint16,
  regionIndexCount: restructure.uint16,
  regionIndexes: new restructure.Array(restructure.uint16, 'regionIndexCount'),
  deltaSets: new restructure.Array(DeltaSet, 'itemCount')
});
var ItemVariationStore = new restructure.Struct({
  format: restructure.uint16,
  variationRegionList: new restructure.Pointer(restructure.uint32, VariationRegionList),
  variationDataCount: restructure.uint16,
  itemVariationData: new restructure.Array(new restructure.Pointer(restructure.uint32, ItemVariationData), 'variationDataCount')
});
/**********************
 * Feature Variations *
 **********************/

var ConditionTable = new restructure.VersionedStruct(restructure.uint16, {
  1: (_ = {
    axisIndex: restructure.uint16
  }, _["axisIndex"] = restructure.uint16, _.filterRangeMinValue = F2DOT14, _.filterRangeMaxValue = F2DOT14, _)
});
var ConditionSet = new restructure.Struct({
  conditionCount: restructure.uint16,
  conditionTable: new restructure.Array(new restructure.Pointer(restructure.uint32, ConditionTable), 'conditionCount')
});
var FeatureTableSubstitutionRecord = new restructure.Struct({
  featureIndex: restructure.uint16,
  alternateFeatureTable: new restructure.Pointer(restructure.uint32, Feature, {
    type: 'parent'
  })
});
var FeatureTableSubstitution = new restructure.Struct({
  version: restructure.fixed32,
  substitutionCount: restructure.uint16,
  substitutions: new restructure.Array(FeatureTableSubstitutionRecord, 'substitutionCount')
});
var FeatureVariationRecord = new restructure.Struct({
  conditionSet: new restructure.Pointer(restructure.uint32, ConditionSet, {
    type: 'parent'
  }),
  featureTableSubstitution: new restructure.Pointer(restructure.uint32, FeatureTableSubstitution, {
    type: 'parent'
  })
});
var FeatureVariations = new restructure.Struct({
  majorVersion: restructure.uint16,
  minorVersion: restructure.uint16,
  featureVariationRecordCount: restructure.uint32,
  featureVariationRecords: new restructure.Array(FeatureVariationRecord, 'featureVariationRecordCount')
});

// otherwise delegates to the provided type.

var PredefinedOp =
/*#__PURE__*/
function () {
  function PredefinedOp(predefinedOps, type) {
    this.predefinedOps = predefinedOps;
    this.type = type;
  }

  var _proto = PredefinedOp.prototype;

  _proto.decode = function decode(stream, parent, operands) {
    if (this.predefinedOps[operands[0]]) {
      return this.predefinedOps[operands[0]];
    }

    return this.type.decode(stream, parent, operands);
  };

  _proto.size = function size(value, ctx) {
    return this.type.size(value, ctx);
  };

  _proto.encode = function encode(stream, value, ctx) {
    var index = this.predefinedOps.indexOf(value);

    if (index !== -1) {
      return index;
    }

    return this.type.encode(stream, value, ctx);
  };

  return PredefinedOp;
}();

var CFFEncodingVersion =
/*#__PURE__*/
function (_r$Number) {
  _inheritsLoose(CFFEncodingVersion, _r$Number);

  function CFFEncodingVersion() {
    return _r$Number.call(this, 'UInt8') || this;
  }

  var _proto2 = CFFEncodingVersion.prototype;

  _proto2.decode = function decode(stream) {
    return restructure.uint8.decode(stream) & 0x7f;
  };

  return CFFEncodingVersion;
}(restructure.Number);

var Range1 = new restructure.Struct({
  first: restructure.uint16,
  nLeft: restructure.uint8
});
var Range2 = new restructure.Struct({
  first: restructure.uint16,
  nLeft: restructure.uint16
});
var CFFCustomEncoding = new restructure.VersionedStruct(new CFFEncodingVersion(), {
  0: {
    nCodes: restructure.uint8,
    codes: new restructure.Array(restructure.uint8, 'nCodes')
  },
  1: {
    nRanges: restructure.uint8,
    ranges: new restructure.Array(Range1, 'nRanges') // TODO: supplement?

  }
});
var CFFEncoding = new PredefinedOp([StandardEncoding, ExpertEncoding], new CFFPointer(CFFCustomEncoding, {
  lazy: true
})); // Decodes an array of ranges until the total
// length is equal to the provided length.

var RangeArray =
/*#__PURE__*/
function (_r$Array) {
  _inheritsLoose(RangeArray, _r$Array);

  function RangeArray() {
    return _r$Array.apply(this, arguments) || this;
  }

  var _proto3 = RangeArray.prototype;

  _proto3.decode = function decode(stream, parent) {
    var length = utils_1(this.length, stream, parent);
    var count = 0;
    var res = [];

    while (count < length) {
      var range = this.type.decode(stream, parent);
      range.offset = count;
      count += range.nLeft + 1;
      res.push(range);
    }

    return res;
  };

  return RangeArray;
}(restructure.Array);

var CFFCustomCharset = new restructure.VersionedStruct(restructure.uint8, {
  0: {
    glyphs: new restructure.Array(restructure.uint16, function (t) {
      return t.parent.CharStrings.length - 1;
    })
  },
  1: {
    ranges: new RangeArray(Range1, function (t) {
      return t.parent.CharStrings.length - 1;
    })
  },
  2: {
    ranges: new RangeArray(Range2, function (t) {
      return t.parent.CharStrings.length - 1;
    })
  }
});
var CFFCharset = new PredefinedOp([ISOAdobeCharset, ExpertCharset, ExpertSubsetCharset], new CFFPointer(CFFCustomCharset, {
  lazy: true
}));
var FDRange3 = new restructure.Struct({
  first: restructure.uint16,
  fd: restructure.uint8
});
var FDRange4 = new restructure.Struct({
  first: restructure.uint32,
  fd: restructure.uint16
});
var FDSelect = new restructure.VersionedStruct(restructure.uint8, {
  0: {
    fds: new restructure.Array(restructure.uint8, function (t) {
      return t.parent.CharStrings.length;
    })
  },
  3: {
    nRanges: restructure.uint16,
    ranges: new restructure.Array(FDRange3, 'nRanges'),
    sentinel: restructure.uint16
  },
  4: {
    nRanges: restructure.uint32,
    ranges: new restructure.Array(FDRange4, 'nRanges'),
    sentinel: restructure.uint32
  }
});
var ptr = new CFFPointer(CFFPrivateDict);

var CFFPrivateOp =
/*#__PURE__*/
function () {
  function CFFPrivateOp() {}

  var _proto4 = CFFPrivateOp.prototype;

  _proto4.decode = function decode(stream, parent, operands) {
    parent.length = operands[0];
    return ptr.decode(stream, parent, [operands[1]]);
  };

  _proto4.size = function size(dict, ctx) {
    return [CFFPrivateDict.size(dict, ctx, false), ptr.size(dict, ctx)[0]];
  };

  _proto4.encode = function encode(stream, dict, ctx) {
    return [CFFPrivateDict.size(dict, ctx, false), ptr.encode(stream, dict, ctx)[0]];
  };

  return CFFPrivateOp;
}();

var FontDict = new CFFDict([// key       name                   type(s)                                 default
[18, 'Private', new CFFPrivateOp(), null], [[12, 38], 'FontName', 'sid', null]]);
var CFFTopDict = new CFFDict([// key       name                   type(s)                                 default
[[12, 30], 'ROS', ['sid', 'sid', 'number'], null], [0, 'version', 'sid', null], [1, 'Notice', 'sid', null], [[12, 0], 'Copyright', 'sid', null], [2, 'FullName', 'sid', null], [3, 'FamilyName', 'sid', null], [4, 'Weight', 'sid', null], [[12, 1], 'isFixedPitch', 'boolean', false], [[12, 2], 'ItalicAngle', 'number', 0], [[12, 3], 'UnderlinePosition', 'number', -100], [[12, 4], 'UnderlineThickness', 'number', 50], [[12, 5], 'PaintType', 'number', 0], [[12, 6], 'CharstringType', 'number', 2], [[12, 7], 'FontMatrix', 'array', [0.001, 0, 0, 0.001, 0, 0]], [13, 'UniqueID', 'number', null], [5, 'FontBBox', 'array', [0, 0, 0, 0]], [[12, 8], 'StrokeWidth', 'number', 0], [14, 'XUID', 'array', null], [15, 'charset', CFFCharset, ISOAdobeCharset], [16, 'Encoding', CFFEncoding, StandardEncoding], [17, 'CharStrings', new CFFPointer(new CFFIndex()), null], [18, 'Private', new CFFPrivateOp(), null], [[12, 20], 'SyntheticBase', 'number', null], [[12, 21], 'PostScript', 'sid', null], [[12, 22], 'BaseFontName', 'sid', null], [[12, 23], 'BaseFontBlend', 'delta', null], // CID font specific
[[12, 31], 'CIDFontVersion', 'number', 0], [[12, 32], 'CIDFontRevision', 'number', 0], [[12, 33], 'CIDFontType', 'number', 0], [[12, 34], 'CIDCount', 'number', 8720], [[12, 35], 'UIDBase', 'number', null], [[12, 37], 'FDSelect', new CFFPointer(FDSelect), null], [[12, 36], 'FDArray', new CFFPointer(new CFFIndex(FontDict)), null], [[12, 38], 'FontName', 'sid', null]]);
var VariationStore = new restructure.Struct({
  length: restructure.uint16,
  itemVariationStore: ItemVariationStore
});
var CFF2TopDict = new CFFDict([[[12, 7], 'FontMatrix', 'array', [0.001, 0, 0, 0.001, 0, 0]], [17, 'CharStrings', new CFFPointer(new CFFIndex()), null], [[12, 37], 'FDSelect', new CFFPointer(FDSelect), null], [[12, 36], 'FDArray', new CFFPointer(new CFFIndex(FontDict)), null], [24, 'vstore', new CFFPointer(VariationStore), null], [25, 'maxstack', 'number', 193]]);
var CFFTop = new restructure.VersionedStruct(restructure.fixed16, {
  1: {
    hdrSize: restructure.uint8,
    offSize: restructure.uint8,
    nameIndex: new CFFIndex(new restructure.String('length')),
    topDictIndex: new CFFIndex(CFFTopDict),
    stringIndex: new CFFIndex(new restructure.String('length')),
    globalSubrIndex: new CFFIndex()
  },
  2: {
    hdrSize: restructure.uint8,
    length: restructure.uint16,
    topDict: CFF2TopDict,
    globalSubrIndex: new CFFIndex()
  }
});

var CFFFont =
/*#__PURE__*/
function () {
  function CFFFont(stream) {
    this.stream = stream;
    this.decode();
  }

  CFFFont.decode = function decode(stream) {
    return new CFFFont(stream);
  };

  var _proto = CFFFont.prototype;

  _proto.decode = function decode() {
    var start = this.stream.pos;
    var top = CFFTop.decode(this.stream);

    for (var key in top) {
      var val = top[key];
      this[key] = val;
    }

    if (this.version < 2) {
      if (this.topDictIndex.length !== 1) {
        throw new Error("Only a single font is allowed in CFF");
      }

      this.topDict = this.topDictIndex[0];
    }

    this.isCIDFont = this.topDict.ROS != null;
    return this;
  };

  _proto.string = function string(sid) {
    if (this.version >= 2) {
      return null;
    }

    if (sid < standardStrings.length) {
      return standardStrings[sid];
    }

    return this.stringIndex[sid - standardStrings.length];
  };

  _proto.getCharString = function getCharString(glyph) {
    this.stream.pos = this.topDict.CharStrings[glyph].offset;
    return this.stream.readBuffer(this.topDict.CharStrings[glyph].length);
  };

  _proto.getGlyphName = function getGlyphName(gid) {
    // CFF2 glyph names are in the post table.
    if (this.version >= 2) {
      return null;
    } // CID-keyed fonts don't have glyph names


    if (this.isCIDFont) {
      return null;
    }

    var charset = this.topDict.charset;

    if (Array.isArray(charset)) {
      return charset[gid];
    }

    if (gid === 0) {
      return '.notdef';
    }

    gid -= 1;

    switch (charset.version) {
      case 0:
        return this.string(charset.glyphs[gid]);

      case 1:
      case 2:
        for (var i = 0; i < charset.ranges.length; i++) {
          var range = charset.ranges[i];

          if (range.offset <= gid && gid <= range.offset + range.nLeft) {
            return this.string(range.first + (gid - range.offset));
          }
        }

        break;
    }

    return null;
  };

  _proto.fdForGlyph = function fdForGlyph(gid) {
    if (!this.topDict.FDSelect) {
      return null;
    }

    switch (this.topDict.FDSelect.version) {
      case 0:
        return this.topDict.FDSelect.fds[gid];

      case 3:
      case 4:
        var ranges = this.topDict.FDSelect.ranges;
        var low = 0;
        var high = ranges.length - 1;

        while (low <= high) {
          var mid = low + high >> 1;

          if (gid < ranges[mid].first) {
            high = mid - 1;
          } else if (mid < high && gid > ranges[mid + 1].first) {
            low = mid + 1;
          } else {
            return ranges[mid].fd;
          }
        }

      default:
        throw new Error("Unknown FDSelect version: " + this.topDict.FDSelect.version);
    }
  };

  _proto.privateDictForGlyph = function privateDictForGlyph(gid) {
    if (this.topDict.FDSelect) {
      var fd = this.fdForGlyph(gid);

      if (this.topDict.FDArray[fd]) {
        return this.topDict.FDArray[fd].Private;
      }

      return null;
    }

    if (this.version < 2) {
      return this.topDict.Private;
    }

    return this.topDict.FDArray[0].Private;
  };

  _createClass(CFFFont, [{
    key: "postscriptName",
    get: function get() {
      if (this.version < 2) {
        return this.nameIndex[0];
      }

      return null;
    }
  }, {
    key: "fullName",
    get: function get() {
      return this.string(this.topDict.FullName);
    }
  }, {
    key: "familyName",
    get: function get() {
      return this.string(this.topDict.FamilyName);
    }
  }]);

  return CFFFont;
}();

var VerticalOrigin = new restructure.Struct({
  glyphIndex: restructure.uint16,
  vertOriginY: restructure.int16
});
var VORG = new restructure.Struct({
  majorVersion: restructure.uint16,
  minorVersion: restructure.uint16,
  defaultVertOriginY: restructure.int16,
  numVertOriginYMetrics: restructure.uint16,
  metrics: new restructure.Array(VerticalOrigin, 'numVertOriginYMetrics')
});

var BigMetrics = new restructure.Struct({
  height: restructure.uint8,
  width: restructure.uint8,
  horiBearingX: restructure.int8,
  horiBearingY: restructure.int8,
  horiAdvance: restructure.uint8,
  vertBearingX: restructure.int8,
  vertBearingY: restructure.int8,
  vertAdvance: restructure.uint8
});
var SmallMetrics = new restructure.Struct({
  height: restructure.uint8,
  width: restructure.uint8,
  bearingX: restructure.int8,
  bearingY: restructure.int8,
  advance: restructure.uint8
});
var EBDTComponent = new restructure.Struct({
  glyph: restructure.uint16,
  xOffset: restructure.int8,
  yOffset: restructure.int8
});

var ByteAligned = function ByteAligned() {};

var BitAligned = function BitAligned() {};

var glyph = new restructure.VersionedStruct('version', {
  1: {
    metrics: SmallMetrics,
    data: ByteAligned
  },
  2: {
    metrics: SmallMetrics,
    data: BitAligned
  },
  // format 3 is deprecated
  // format 4 is not supported by Microsoft
  5: {
    data: BitAligned
  },
  6: {
    metrics: BigMetrics,
    data: ByteAligned
  },
  7: {
    metrics: BigMetrics,
    data: BitAligned
  },
  8: {
    metrics: SmallMetrics,
    pad: new restructure.Reserved(restructure.uint8),
    numComponents: restructure.uint16,
    components: new restructure.Array(EBDTComponent, 'numComponents')
  },
  9: {
    metrics: BigMetrics,
    pad: new restructure.Reserved(restructure.uint8),
    numComponents: restructure.uint16,
    components: new restructure.Array(EBDTComponent, 'numComponents')
  },
  17: {
    metrics: SmallMetrics,
    dataLen: restructure.uint32,
    data: new restructure.Buffer('dataLen')
  },
  18: {
    metrics: BigMetrics,
    dataLen: restructure.uint32,
    data: new restructure.Buffer('dataLen')
  },
  19: {
    dataLen: restructure.uint32,
    data: new restructure.Buffer('dataLen')
  }
});

var SBitLineMetrics = new restructure.Struct({
  ascender: restructure.int8,
  descender: restructure.int8,
  widthMax: restructure.uint8,
  caretSlopeNumerator: restructure.int8,
  caretSlopeDenominator: restructure.int8,
  caretOffset: restructure.int8,
  minOriginSB: restructure.int8,
  minAdvanceSB: restructure.int8,
  maxBeforeBL: restructure.int8,
  minAfterBL: restructure.int8,
  pad: new restructure.Reserved(restructure.int8, 2)
});
var CodeOffsetPair = new restructure.Struct({
  glyphCode: restructure.uint16,
  offset: restructure.uint16
});
var IndexSubtable = new restructure.VersionedStruct(restructure.uint16, {
  header: {
    imageFormat: restructure.uint16,
    imageDataOffset: restructure.uint32
  },
  1: {
    offsetArray: new restructure.Array(restructure.uint32, function (t) {
      return t.parent.lastGlyphIndex - t.parent.firstGlyphIndex + 1;
    })
  },
  2: {
    imageSize: restructure.uint32,
    bigMetrics: BigMetrics
  },
  3: {
    offsetArray: new restructure.Array(restructure.uint16, function (t) {
      return t.parent.lastGlyphIndex - t.parent.firstGlyphIndex + 1;
    })
  },
  4: {
    numGlyphs: restructure.uint32,
    glyphArray: new restructure.Array(CodeOffsetPair, function (t) {
      return t.numGlyphs + 1;
    })
  },
  5: {
    imageSize: restructure.uint32,
    bigMetrics: BigMetrics,
    numGlyphs: restructure.uint32,
    glyphCodeArray: new restructure.Array(restructure.uint16, 'numGlyphs')
  }
});
var IndexSubtableArray = new restructure.Struct({
  firstGlyphIndex: restructure.uint16,
  lastGlyphIndex: restructure.uint16,
  subtable: new restructure.Pointer(restructure.uint32, IndexSubtable)
});
var BitmapSizeTable = new restructure.Struct({
  indexSubTableArray: new restructure.Pointer(restructure.uint32, new restructure.Array(IndexSubtableArray, 1), {
    type: 'parent'
  }),
  indexTablesSize: restructure.uint32,
  numberOfIndexSubTables: restructure.uint32,
  colorRef: restructure.uint32,
  hori: SBitLineMetrics,
  vert: SBitLineMetrics,
  startGlyphIndex: restructure.uint16,
  endGlyphIndex: restructure.uint16,
  ppemX: restructure.uint8,
  ppemY: restructure.uint8,
  bitDepth: restructure.uint8,
  flags: new restructure.Bitfield(restructure.uint8, ['horizontal', 'vertical'])
});
var EBLC = new restructure.Struct({
  version: restructure.uint32,
  // 0x00020000
  numSizes: restructure.uint32,
  sizes: new restructure.Array(BitmapSizeTable, 'numSizes')
});

var ImageTable = new restructure.Struct({
  ppem: restructure.uint16,
  resolution: restructure.uint16,
  imageOffsets: new restructure.Array(new restructure.Pointer(restructure.uint32, 'void'), function (t) {
    return t.parent.parent.maxp.numGlyphs + 1;
  })
}); // This is the Apple sbix table, used by the "Apple Color Emoji" font.
// It includes several image tables with images for each bitmap glyph
// of several different sizes.

var sbix = new restructure.Struct({
  version: restructure.uint16,
  flags: new restructure.Bitfield(restructure.uint16, ['renderOutlines']),
  numImgTables: restructure.uint32,
  imageTables: new restructure.Array(new restructure.Pointer(restructure.uint32, ImageTable), 'numImgTables')
});

var LayerRecord = new restructure.Struct({
  gid: restructure.uint16,
  // Glyph ID of layer glyph (must be in z-order from bottom to top).
  paletteIndex: restructure.uint16 // Index value to use in the appropriate palette. This value must

}); // be less than numPaletteEntries in the CPAL table, except for
// the special case noted below. Each palette entry is 16 bits.
// A palette index of 0xFFFF is a special case indicating that
// the text foreground color should be used.

var BaseGlyphRecord = new restructure.Struct({
  gid: restructure.uint16,
  // Glyph ID of reference glyph. This glyph is for reference only
  // and is not rendered for color.
  firstLayerIndex: restructure.uint16,
  // Index (from beginning of the Layer Records) to the layer record.
  // There will be numLayers consecutive entries for this base glyph.
  numLayers: restructure.uint16
});
var COLR = new restructure.Struct({
  version: restructure.uint16,
  numBaseGlyphRecords: restructure.uint16,
  baseGlyphRecord: new restructure.Pointer(restructure.uint32, new restructure.Array(BaseGlyphRecord, 'numBaseGlyphRecords')),
  layerRecords: new restructure.Pointer(restructure.uint32, new restructure.Array(LayerRecord, 'numLayerRecords'), {
    lazy: true
  }),
  numLayerRecords: restructure.uint16
});

var ColorRecord = new restructure.Struct({
  blue: restructure.uint8,
  green: restructure.uint8,
  red: restructure.uint8,
  alpha: restructure.uint8
});
var CPAL = new restructure.VersionedStruct(restructure.uint16, {
  header: {
    numPaletteEntries: restructure.uint16,
    numPalettes: restructure.uint16,
    numColorRecords: restructure.uint16,
    colorRecords: new restructure.Pointer(restructure.uint32, new restructure.Array(ColorRecord, 'numColorRecords')),
    colorRecordIndices: new restructure.Array(restructure.uint16, 'numPalettes')
  },
  0: {},
  1: {
    offsetPaletteTypeArray: new restructure.Pointer(restructure.uint32, new restructure.Array(restructure.uint32, 'numPalettes')),
    offsetPaletteLabelArray: new restructure.Pointer(restructure.uint32, new restructure.Array(restructure.uint16, 'numPalettes')),
    offsetPaletteEntryLabelArray: new restructure.Pointer(restructure.uint32, new restructure.Array(restructure.uint16, 'numPaletteEntries'))
  }
});

var BaseCoord = new restructure.VersionedStruct(restructure.uint16, {
  1: {
    // Design units only
    coordinate: restructure.int16 // X or Y value, in design units

  },
  2: {
    // Design units plus contour point
    coordinate: restructure.int16,
    // X or Y value, in design units
    referenceGlyph: restructure.uint16,
    // GlyphID of control glyph
    baseCoordPoint: restructure.uint16 // Index of contour point on the referenceGlyph

  },
  3: {
    // Design units plus Device table
    coordinate: restructure.int16,
    // X or Y value, in design units
    deviceTable: new restructure.Pointer(restructure.uint16, Device) // Device table for X or Y value

  }
});
var BaseValues = new restructure.Struct({
  defaultIndex: restructure.uint16,
  // Index of default baseline for this script-same index in the BaseTagList
  baseCoordCount: restructure.uint16,
  baseCoords: new restructure.Array(new restructure.Pointer(restructure.uint16, BaseCoord), 'baseCoordCount')
});
var FeatMinMaxRecord = new restructure.Struct({
  tag: new restructure.String(4),
  // 4-byte feature identification tag-must match FeatureTag in FeatureList
  minCoord: new restructure.Pointer(restructure.uint16, BaseCoord, {
    type: 'parent'
  }),
  // May be NULL
  maxCoord: new restructure.Pointer(restructure.uint16, BaseCoord, {
    type: 'parent'
  }) // May be NULL

});
var MinMax = new restructure.Struct({
  minCoord: new restructure.Pointer(restructure.uint16, BaseCoord),
  // May be NULL
  maxCoord: new restructure.Pointer(restructure.uint16, BaseCoord),
  // May be NULL
  featMinMaxCount: restructure.uint16,
  // May be 0
  featMinMaxRecords: new restructure.Array(FeatMinMaxRecord, 'featMinMaxCount') // In alphabetical order

});
var BaseLangSysRecord = new restructure.Struct({
  tag: new restructure.String(4),
  // 4-byte language system identification tag
  minMax: new restructure.Pointer(restructure.uint16, MinMax, {
    type: 'parent'
  })
});
var BaseScript = new restructure.Struct({
  baseValues: new restructure.Pointer(restructure.uint16, BaseValues),
  // May be NULL
  defaultMinMax: new restructure.Pointer(restructure.uint16, MinMax),
  // May be NULL
  baseLangSysCount: restructure.uint16,
  // May be 0
  baseLangSysRecords: new restructure.Array(BaseLangSysRecord, 'baseLangSysCount') // in alphabetical order by BaseLangSysTag

});
var BaseScriptRecord = new restructure.Struct({
  tag: new restructure.String(4),
  // 4-byte script identification tag
  script: new restructure.Pointer(restructure.uint16, BaseScript, {
    type: 'parent'
  })
});
var BaseScriptList = new restructure.Array(BaseScriptRecord, restructure.uint16); // Array of 4-byte baseline identification tags-must be in alphabetical order

var BaseTagList = new restructure.Array(new restructure.String(4), restructure.uint16);
var Axis = new restructure.Struct({
  baseTagList: new restructure.Pointer(restructure.uint16, BaseTagList),
  // May be NULL
  baseScriptList: new restructure.Pointer(restructure.uint16, BaseScriptList)
});
var BASE = new restructure.VersionedStruct(restructure.uint32, {
  header: {
    horizAxis: new restructure.Pointer(restructure.uint16, Axis),
    // May be NULL
    vertAxis: new restructure.Pointer(restructure.uint16, Axis) // May be NULL

  },
  0x00010000: {},
  0x00010001: {
    itemVariationStore: new restructure.Pointer(restructure.uint32, ItemVariationStore)
  }
});

var AttachPoint = new restructure.Array(restructure.uint16, restructure.uint16);
var AttachList = new restructure.Struct({
  coverage: new restructure.Pointer(restructure.uint16, Coverage),
  glyphCount: restructure.uint16,
  attachPoints: new restructure.Array(new restructure.Pointer(restructure.uint16, AttachPoint), 'glyphCount')
});
var CaretValue = new restructure.VersionedStruct(restructure.uint16, {
  1: {
    // Design units only
    coordinate: restructure.int16
  },
  2: {
    // Contour point
    caretValuePoint: restructure.uint16
  },
  3: {
    // Design units plus Device table
    coordinate: restructure.int16,
    deviceTable: new restructure.Pointer(restructure.uint16, Device)
  }
});
var LigGlyph = new restructure.Array(new restructure.Pointer(restructure.uint16, CaretValue), restructure.uint16);
var LigCaretList = new restructure.Struct({
  coverage: new restructure.Pointer(restructure.uint16, Coverage),
  ligGlyphCount: restructure.uint16,
  ligGlyphs: new restructure.Array(new restructure.Pointer(restructure.uint16, LigGlyph), 'ligGlyphCount')
});
var MarkGlyphSetsDef = new restructure.Struct({
  markSetTableFormat: restructure.uint16,
  markSetCount: restructure.uint16,
  coverage: new restructure.Array(new restructure.Pointer(restructure.uint32, Coverage), 'markSetCount')
});
var GDEF = new restructure.VersionedStruct(restructure.uint32, {
  header: {
    glyphClassDef: new restructure.Pointer(restructure.uint16, ClassDef),
    attachList: new restructure.Pointer(restructure.uint16, AttachList),
    ligCaretList: new restructure.Pointer(restructure.uint16, LigCaretList),
    markAttachClassDef: new restructure.Pointer(restructure.uint16, ClassDef)
  },
  0x00010000: {},
  0x00010002: {
    markGlyphSetsDef: new restructure.Pointer(restructure.uint16, MarkGlyphSetsDef)
  },
  0x00010003: {
    markGlyphSetsDef: new restructure.Pointer(restructure.uint16, MarkGlyphSetsDef),
    itemVariationStore: new restructure.Pointer(restructure.uint32, ItemVariationStore)
  }
});

var ValueFormat = new restructure.Bitfield(restructure.uint16, ['xPlacement', 'yPlacement', 'xAdvance', 'yAdvance', 'xPlaDevice', 'yPlaDevice', 'xAdvDevice', 'yAdvDevice']);
var types = {
  xPlacement: restructure.int16,
  yPlacement: restructure.int16,
  xAdvance: restructure.int16,
  yAdvance: restructure.int16,
  xPlaDevice: new restructure.Pointer(restructure.uint16, Device, {
    type: 'global',
    relativeTo: 'rel'
  }),
  yPlaDevice: new restructure.Pointer(restructure.uint16, Device, {
    type: 'global',
    relativeTo: 'rel'
  }),
  xAdvDevice: new restructure.Pointer(restructure.uint16, Device, {
    type: 'global',
    relativeTo: 'rel'
  }),
  yAdvDevice: new restructure.Pointer(restructure.uint16, Device, {
    type: 'global',
    relativeTo: 'rel'
  })
};

var ValueRecord =
/*#__PURE__*/
function () {
  function ValueRecord(key) {
    if (key === void 0) {
      key = 'valueFormat';
    }

    this.key = key;
  }

  var _proto = ValueRecord.prototype;

  _proto.buildStruct = function buildStruct(parent) {
    var struct = parent;

    while (!struct[this.key] && struct.parent) {
      struct = struct.parent;
    }

    if (!struct[this.key]) return;
    var fields = {};

    fields.rel = function () {
      return struct._startOffset;
    };

    var format = struct[this.key];

    for (var key in format) {
      if (format[key]) {
        fields[key] = types[key];
      }
    }

    return new restructure.Struct(fields);
  };

  _proto.size = function size(val, ctx) {
    return this.buildStruct(ctx).size(val, ctx);
  };

  _proto.decode = function decode(stream, parent) {
    var res = this.buildStruct(parent).decode(stream, parent);
    delete res.rel;
    return res;
  };

  return ValueRecord;
}();

var PairValueRecord = new restructure.Struct({
  secondGlyph: restructure.uint16,
  value1: new ValueRecord('valueFormat1'),
  value2: new ValueRecord('valueFormat2')
});
var PairSet = new restructure.Array(PairValueRecord, restructure.uint16);
var Class2Record = new restructure.Struct({
  value1: new ValueRecord('valueFormat1'),
  value2: new ValueRecord('valueFormat2')
});
var Anchor = new restructure.VersionedStruct(restructure.uint16, {
  1: {
    // Design units only
    xCoordinate: restructure.int16,
    yCoordinate: restructure.int16
  },
  2: {
    // Design units plus contour point
    xCoordinate: restructure.int16,
    yCoordinate: restructure.int16,
    anchorPoint: restructure.uint16
  },
  3: {
    // Design units plus Device tables
    xCoordinate: restructure.int16,
    yCoordinate: restructure.int16,
    xDeviceTable: new restructure.Pointer(restructure.uint16, Device),
    yDeviceTable: new restructure.Pointer(restructure.uint16, Device)
  }
});
var EntryExitRecord = new restructure.Struct({
  entryAnchor: new restructure.Pointer(restructure.uint16, Anchor, {
    type: 'parent'
  }),
  exitAnchor: new restructure.Pointer(restructure.uint16, Anchor, {
    type: 'parent'
  })
});
var MarkRecord = new restructure.Struct({
  class: restructure.uint16,
  markAnchor: new restructure.Pointer(restructure.uint16, Anchor, {
    type: 'parent'
  })
});
var MarkArray = new restructure.Array(MarkRecord, restructure.uint16);
var BaseRecord = new restructure.Array(new restructure.Pointer(restructure.uint16, Anchor), function (t) {
  return t.parent.classCount;
});
var BaseArray = new restructure.Array(BaseRecord, restructure.uint16);
var ComponentRecord = new restructure.Array(new restructure.Pointer(restructure.uint16, Anchor), function (t) {
  return t.parent.parent.classCount;
});
var LigatureAttach = new restructure.Array(ComponentRecord, restructure.uint16);
var LigatureArray = new restructure.Array(new restructure.Pointer(restructure.uint16, LigatureAttach), restructure.uint16);
var GPOSLookup = new restructure.VersionedStruct('lookupType', {
  1: new restructure.VersionedStruct(restructure.uint16, {
    // Single Adjustment
    1: {
      // Single positioning value
      coverage: new restructure.Pointer(restructure.uint16, Coverage),
      valueFormat: ValueFormat,
      value: new ValueRecord()
    },
    2: {
      coverage: new restructure.Pointer(restructure.uint16, Coverage),
      valueFormat: ValueFormat,
      valueCount: restructure.uint16,
      values: new restructure.LazyArray(new ValueRecord(), 'valueCount')
    }
  }),
  2: new restructure.VersionedStruct(restructure.uint16, {
    // Pair Adjustment Positioning
    1: {
      // Adjustments for glyph pairs
      coverage: new restructure.Pointer(restructure.uint16, Coverage),
      valueFormat1: ValueFormat,
      valueFormat2: ValueFormat,
      pairSetCount: restructure.uint16,
      pairSets: new restructure.LazyArray(new restructure.Pointer(restructure.uint16, PairSet), 'pairSetCount')
    },
    2: {
      // Class pair adjustment
      coverage: new restructure.Pointer(restructure.uint16, Coverage),
      valueFormat1: ValueFormat,
      valueFormat2: ValueFormat,
      classDef1: new restructure.Pointer(restructure.uint16, ClassDef),
      classDef2: new restructure.Pointer(restructure.uint16, ClassDef),
      class1Count: restructure.uint16,
      class2Count: restructure.uint16,
      classRecords: new restructure.LazyArray(new restructure.LazyArray(Class2Record, 'class2Count'), 'class1Count')
    }
  }),
  3: {
    // Cursive Attachment Positioning
    format: restructure.uint16,
    coverage: new restructure.Pointer(restructure.uint16, Coverage),
    entryExitCount: restructure.uint16,
    entryExitRecords: new restructure.Array(EntryExitRecord, 'entryExitCount')
  },
  4: {
    // MarkToBase Attachment Positioning
    format: restructure.uint16,
    markCoverage: new restructure.Pointer(restructure.uint16, Coverage),
    baseCoverage: new restructure.Pointer(restructure.uint16, Coverage),
    classCount: restructure.uint16,
    markArray: new restructure.Pointer(restructure.uint16, MarkArray),
    baseArray: new restructure.Pointer(restructure.uint16, BaseArray)
  },
  5: {
    // MarkToLigature Attachment Positioning
    format: restructure.uint16,
    markCoverage: new restructure.Pointer(restructure.uint16, Coverage),
    ligatureCoverage: new restructure.Pointer(restructure.uint16, Coverage),
    classCount: restructure.uint16,
    markArray: new restructure.Pointer(restructure.uint16, MarkArray),
    ligatureArray: new restructure.Pointer(restructure.uint16, LigatureArray)
  },
  6: {
    // MarkToMark Attachment Positioning
    format: restructure.uint16,
    mark1Coverage: new restructure.Pointer(restructure.uint16, Coverage),
    mark2Coverage: new restructure.Pointer(restructure.uint16, Coverage),
    classCount: restructure.uint16,
    mark1Array: new restructure.Pointer(restructure.uint16, MarkArray),
    mark2Array: new restructure.Pointer(restructure.uint16, BaseArray)
  },
  7: Context,
  // Contextual positioning
  8: ChainingContext,
  // Chaining contextual positioning
  9: {
    // Extension Positioning
    posFormat: restructure.uint16,
    lookupType: restructure.uint16,
    // cannot also be 9
    extension: new restructure.Pointer(restructure.uint32, GPOSLookup)
  }
}); // Fix circular reference

GPOSLookup.versions[9].extension.type = GPOSLookup;
var GPOS = new restructure.VersionedStruct(restructure.uint32, {
  header: {
    scriptList: new restructure.Pointer(restructure.uint16, ScriptList),
    featureList: new restructure.Pointer(restructure.uint16, FeatureList),
    lookupList: new restructure.Pointer(restructure.uint16, new LookupList(GPOSLookup))
  },
  0x00010000: {},
  0x00010001: {
    featureVariations: new restructure.Pointer(restructure.uint32, FeatureVariations)
  }
}); // export GPOSLookup for JSTF table

var Sequence = new restructure.Array(restructure.uint16, restructure.uint16);
var AlternateSet = Sequence;
var Ligature = new restructure.Struct({
  glyph: restructure.uint16,
  compCount: restructure.uint16,
  components: new restructure.Array(restructure.uint16, function (t) {
    return t.compCount - 1;
  })
});
var LigatureSet = new restructure.Array(new restructure.Pointer(restructure.uint16, Ligature), restructure.uint16);
var GSUBLookup = new restructure.VersionedStruct('lookupType', {
  1: new restructure.VersionedStruct(restructure.uint16, {
    // Single Substitution
    1: {
      coverage: new restructure.Pointer(restructure.uint16, Coverage),
      deltaGlyphID: restructure.int16
    },
    2: {
      coverage: new restructure.Pointer(restructure.uint16, Coverage),
      glyphCount: restructure.uint16,
      substitute: new restructure.LazyArray(restructure.uint16, 'glyphCount')
    }
  }),
  2: {
    // Multiple Substitution
    substFormat: restructure.uint16,
    coverage: new restructure.Pointer(restructure.uint16, Coverage),
    count: restructure.uint16,
    sequences: new restructure.LazyArray(new restructure.Pointer(restructure.uint16, Sequence), 'count')
  },
  3: {
    // Alternate Substitution
    substFormat: restructure.uint16,
    coverage: new restructure.Pointer(restructure.uint16, Coverage),
    count: restructure.uint16,
    alternateSet: new restructure.LazyArray(new restructure.Pointer(restructure.uint16, AlternateSet), 'count')
  },
  4: {
    // Ligature Substitution
    substFormat: restructure.uint16,
    coverage: new restructure.Pointer(restructure.uint16, Coverage),
    count: restructure.uint16,
    ligatureSets: new restructure.LazyArray(new restructure.Pointer(restructure.uint16, LigatureSet), 'count')
  },
  5: Context,
  // Contextual Substitution
  6: ChainingContext,
  // Chaining Contextual Substitution
  7: {
    // Extension Substitution
    substFormat: restructure.uint16,
    lookupType: restructure.uint16,
    // cannot also be 7
    extension: new restructure.Pointer(restructure.uint32, GSUBLookup)
  },
  8: {
    // Reverse Chaining Contextual Single Substitution
    substFormat: restructure.uint16,
    coverage: new restructure.Pointer(restructure.uint16, Coverage),
    backtrackCoverage: new restructure.Array(new restructure.Pointer(restructure.uint16, Coverage), 'backtrackGlyphCount'),
    lookaheadGlyphCount: restructure.uint16,
    lookaheadCoverage: new restructure.Array(new restructure.Pointer(restructure.uint16, Coverage), 'lookaheadGlyphCount'),
    glyphCount: restructure.uint16,
    substitutes: new restructure.Array(restructure.uint16, 'glyphCount')
  }
}); // Fix circular reference

GSUBLookup.versions[7].extension.type = GSUBLookup;
var GSUB = new restructure.VersionedStruct(restructure.uint32, {
  header: {
    scriptList: new restructure.Pointer(restructure.uint16, ScriptList),
    featureList: new restructure.Pointer(restructure.uint16, FeatureList),
    lookupList: new restructure.Pointer(restructure.uint16, new LookupList(GSUBLookup))
  },
  0x00010000: {},
  0x00010001: {
    featureVariations: new restructure.Pointer(restructure.uint32, FeatureVariations)
  }
});

var JstfGSUBModList = new restructure.Array(restructure.uint16, restructure.uint16);
var JstfPriority = new restructure.Struct({
  shrinkageEnableGSUB: new restructure.Pointer(restructure.uint16, JstfGSUBModList),
  shrinkageDisableGSUB: new restructure.Pointer(restructure.uint16, JstfGSUBModList),
  shrinkageEnableGPOS: new restructure.Pointer(restructure.uint16, JstfGSUBModList),
  shrinkageDisableGPOS: new restructure.Pointer(restructure.uint16, JstfGSUBModList),
  shrinkageJstfMax: new restructure.Pointer(restructure.uint16, new LookupList(GPOSLookup)),
  extensionEnableGSUB: new restructure.Pointer(restructure.uint16, JstfGSUBModList),
  extensionDisableGSUB: new restructure.Pointer(restructure.uint16, JstfGSUBModList),
  extensionEnableGPOS: new restructure.Pointer(restructure.uint16, JstfGSUBModList),
  extensionDisableGPOS: new restructure.Pointer(restructure.uint16, JstfGSUBModList),
  extensionJstfMax: new restructure.Pointer(restructure.uint16, new LookupList(GPOSLookup))
});
var JstfLangSys = new restructure.Array(new restructure.Pointer(restructure.uint16, JstfPriority), restructure.uint16);
var JstfLangSysRecord = new restructure.Struct({
  tag: new restructure.String(4),
  jstfLangSys: new restructure.Pointer(restructure.uint16, JstfLangSys)
});
var JstfScript = new restructure.Struct({
  extenderGlyphs: new restructure.Pointer(restructure.uint16, new restructure.Array(restructure.uint16, restructure.uint16)),
  // array of glyphs to extend line length
  defaultLangSys: new restructure.Pointer(restructure.uint16, JstfLangSys),
  langSysCount: restructure.uint16,
  langSysRecords: new restructure.Array(JstfLangSysRecord, 'langSysCount')
});
var JstfScriptRecord = new restructure.Struct({
  tag: new restructure.String(4),
  script: new restructure.Pointer(restructure.uint16, JstfScript, {
    type: 'parent'
  })
});
var JSTF = new restructure.Struct({
  version: restructure.uint32,
  // should be 0x00010000
  scriptCount: restructure.uint16,
  scriptList: new restructure.Array(JstfScriptRecord, 'scriptCount')
});

var VariableSizeNumber =
/*#__PURE__*/
function () {
  function VariableSizeNumber(size) {
    this._size = size;
  }

  var _proto = VariableSizeNumber.prototype;

  _proto.decode = function decode(stream, parent) {
    switch (this.size(0, parent)) {
      case 1:
        return stream.readUInt8();

      case 2:
        return stream.readUInt16BE();

      case 3:
        return stream.readUInt24BE();

      case 4:
        return stream.readUInt32BE();
    }
  };

  _proto.size = function size(val, parent) {
    return utils_1(this._size, null, parent);
  };

  return VariableSizeNumber;
}();

var MapDataEntry = new restructure.Struct({
  entry: new VariableSizeNumber(function (t) {
    return ((t.parent.entryFormat & 0x0030) >> 4) + 1;
  }),
  outerIndex: function outerIndex(t) {
    return t.entry >> (t.parent.entryFormat & 0x000F) + 1;
  },
  innerIndex: function innerIndex(t) {
    return t.entry & (1 << (t.parent.entryFormat & 0x000F) + 1) - 1;
  }
});
var DeltaSetIndexMap = new restructure.Struct({
  entryFormat: restructure.uint16,
  mapCount: restructure.uint16,
  mapData: new restructure.Array(MapDataEntry, 'mapCount')
});
var HVAR = new restructure.Struct({
  majorVersion: restructure.uint16,
  minorVersion: restructure.uint16,
  itemVariationStore: new restructure.Pointer(restructure.uint32, ItemVariationStore),
  advanceWidthMapping: new restructure.Pointer(restructure.uint32, DeltaSetIndexMap),
  LSBMapping: new restructure.Pointer(restructure.uint32, DeltaSetIndexMap),
  RSBMapping: new restructure.Pointer(restructure.uint32, DeltaSetIndexMap)
});

var Signature = new restructure.Struct({
  format: restructure.uint32,
  length: restructure.uint32,
  offset: restructure.uint32
});
var SignatureBlock = new restructure.Struct({
  reserved: new restructure.Reserved(restructure.uint16, 2),
  cbSignature: restructure.uint32,
  // Length (in bytes) of the PKCS#7 packet in pbSignature
  signature: new restructure.Buffer('cbSignature')
});
var DSIG = new restructure.Struct({
  ulVersion: restructure.uint32,
  // Version number of the DSIG table (0x00000001)
  usNumSigs: restructure.uint16,
  // Number of signatures in the table
  usFlag: restructure.uint16,
  // Permission flags
  signatures: new restructure.Array(Signature, 'usNumSigs'),
  signatureBlocks: new restructure.Array(SignatureBlock, 'usNumSigs')
});

var GaspRange = new restructure.Struct({
  rangeMaxPPEM: restructure.uint16,
  // Upper limit of range, in ppem
  rangeGaspBehavior: new restructure.Bitfield(restructure.uint16, [// Flags describing desired rasterizer behavior
  'grayscale', 'gridfit', 'symmetricSmoothing', 'symmetricGridfit' // only in version 1, for ClearType
  ])
});
var gasp = new restructure.Struct({
  version: restructure.uint16,
  // set to 0
  numRanges: restructure.uint16,
  gaspRanges: new restructure.Array(GaspRange, 'numRanges') // Sorted by ppem

});

var DeviceRecord = new restructure.Struct({
  pixelSize: restructure.uint8,
  maximumWidth: restructure.uint8,
  widths: new restructure.Array(restructure.uint8, function (t) {
    return t.parent.parent.maxp.numGlyphs;
  })
}); // The Horizontal Device Metrics table stores integer advance widths scaled to particular pixel sizes

var hdmx = new restructure.Struct({
  version: restructure.uint16,
  numRecords: restructure.int16,
  sizeDeviceRecord: restructure.int32,
  records: new restructure.Array(DeviceRecord, 'numRecords')
});

var KernPair = new restructure.Struct({
  left: restructure.uint16,
  right: restructure.uint16,
  value: restructure.int16
});
var ClassTable = new restructure.Struct({
  firstGlyph: restructure.uint16,
  nGlyphs: restructure.uint16,
  offsets: new restructure.Array(restructure.uint16, 'nGlyphs'),
  max: function max(t) {
    return t.offsets.length && Math.max.apply(Math, t.offsets);
  }
});
var Kern2Array = new restructure.Struct({
  off: function off(t) {
    return t._startOffset - t.parent.parent._startOffset;
  },
  len: function len(t) {
    return ((t.parent.leftTable.max - t.off) / t.parent.rowWidth + 1) * (t.parent.rowWidth / 2);
  },
  values: new restructure.LazyArray(restructure.int16, 'len')
});
var KernSubtable = new restructure.VersionedStruct('format', {
  0: {
    nPairs: restructure.uint16,
    searchRange: restructure.uint16,
    entrySelector: restructure.uint16,
    rangeShift: restructure.uint16,
    pairs: new restructure.Array(KernPair, 'nPairs')
  },
  2: {
    rowWidth: restructure.uint16,
    leftTable: new restructure.Pointer(restructure.uint16, ClassTable, {
      type: 'parent'
    }),
    rightTable: new restructure.Pointer(restructure.uint16, ClassTable, {
      type: 'parent'
    }),
    array: new restructure.Pointer(restructure.uint16, Kern2Array, {
      type: 'parent'
    })
  },
  3: {
    glyphCount: restructure.uint16,
    kernValueCount: restructure.uint8,
    leftClassCount: restructure.uint8,
    rightClassCount: restructure.uint8,
    flags: restructure.uint8,
    kernValue: new restructure.Array(restructure.int16, 'kernValueCount'),
    leftClass: new restructure.Array(restructure.uint8, 'glyphCount'),
    rightClass: new restructure.Array(restructure.uint8, 'glyphCount'),
    kernIndex: new restructure.Array(restructure.uint8, function (t) {
      return t.leftClassCount * t.rightClassCount;
    })
  }
});
var KernTable = new restructure.VersionedStruct('version', {
  0: {
    // Microsoft uses this format
    subVersion: restructure.uint16,
    // Microsoft has an extra sub-table version number
    length: restructure.uint16,
    // Length of the subtable, in bytes
    format: restructure.uint8,
    // Format of subtable
    coverage: new restructure.Bitfield(restructure.uint8, ['horizontal', // 1 if table has horizontal data, 0 if vertical
    'minimum', // If set to 1, the table has minimum values. If set to 0, the table has kerning values.
    'crossStream', // If set to 1, kerning is perpendicular to the flow of the text
    'override' // If set to 1 the value in this table replaces the accumulated value
    ]),
    subtable: KernSubtable,
    padding: new restructure.Reserved(restructure.uint8, function (t) {
      return t.length - t._currentOffset;
    })
  },
  1: {
    // Apple uses this format
    length: restructure.uint32,
    coverage: new restructure.Bitfield(restructure.uint8, [null, null, null, null, null, 'variation', // Set if table has variation kerning values
    'crossStream', // Set if table has cross-stream kerning values
    'vertical' // Set if table has vertical kerning values
    ]),
    format: restructure.uint8,
    tupleIndex: restructure.uint16,
    subtable: KernSubtable,
    padding: new restructure.Reserved(restructure.uint8, function (t) {
      return t.length - t._currentOffset;
    })
  }
});
var kern = new restructure.VersionedStruct(restructure.uint16, {
  0: {
    // Microsoft Version
    nTables: restructure.uint16,
    tables: new restructure.Array(KernTable, 'nTables')
  },
  1: {
    // Apple Version
    reserved: new restructure.Reserved(restructure.uint16),
    // the other half of the version number
    nTables: restructure.uint32,
    tables: new restructure.Array(KernTable, 'nTables')
  }
});

// Records the ppem for each glyph at which the scaling becomes linear again,
// despite instructions effecting the advance width

var LTSH = new restructure.Struct({
  version: restructure.uint16,
  numGlyphs: restructure.uint16,
  yPels: new restructure.Array(restructure.uint8, 'numGlyphs')
});

// NOTE: The PCLT table is strongly discouraged for OpenType fonts with TrueType outlines

var PCLT = new restructure.Struct({
  version: restructure.uint16,
  fontNumber: restructure.uint32,
  pitch: restructure.uint16,
  xHeight: restructure.uint16,
  style: restructure.uint16,
  typeFamily: restructure.uint16,
  capHeight: restructure.uint16,
  symbolSet: restructure.uint16,
  typeface: new restructure.String(16),
  characterComplement: new restructure.String(8),
  fileName: new restructure.String(6),
  strokeWeight: new restructure.String(1),
  widthType: new restructure.String(1),
  serifStyle: restructure.uint8,
  reserved: new restructure.Reserved(restructure.uint8)
});

// sizes. This is needed in order to match font metrics on Windows.

var Ratio = new restructure.Struct({
  bCharSet: restructure.uint8,
  // Character set
  xRatio: restructure.uint8,
  // Value to use for x-Ratio
  yStartRatio: restructure.uint8,
  // Starting y-Ratio value
  yEndRatio: restructure.uint8 // Ending y-Ratio value

});
var vTable = new restructure.Struct({
  yPelHeight: restructure.uint16,
  // yPelHeight to which values apply
  yMax: restructure.int16,
  // Maximum value (in pels) for this yPelHeight
  yMin: restructure.int16 // Minimum value (in pels) for this yPelHeight

});
var VdmxGroup = new restructure.Struct({
  recs: restructure.uint16,
  // Number of height records in this group
  startsz: restructure.uint8,
  // Starting yPelHeight
  endsz: restructure.uint8,
  // Ending yPelHeight
  entries: new restructure.Array(vTable, 'recs') // The VDMX records

});
var VDMX = new restructure.Struct({
  version: restructure.uint16,
  // Version number (0 or 1)
  numRecs: restructure.uint16,
  // Number of VDMX groups present
  numRatios: restructure.uint16,
  // Number of aspect ratio groupings
  ratioRanges: new restructure.Array(Ratio, 'numRatios'),
  // Ratio ranges
  offsets: new restructure.Array(restructure.uint16, 'numRatios'),
  // Offset to the VDMX group for this ratio range
  groups: new restructure.Array(VdmxGroup, 'numRecs') // The actual VDMX groupings

});

var vhea = new restructure.Struct({
  version: restructure.uint16,
  // Version number of the Vertical Header Table
  ascent: restructure.int16,
  // The vertical typographic ascender for this font
  descent: restructure.int16,
  // The vertical typographic descender for this font
  lineGap: restructure.int16,
  // The vertical typographic line gap for this font
  advanceHeightMax: restructure.int16,
  // The maximum advance height measurement found in the font
  minTopSideBearing: restructure.int16,
  // The minimum top side bearing measurement found in the font
  minBottomSideBearing: restructure.int16,
  // The minimum bottom side bearing measurement found in the font
  yMaxExtent: restructure.int16,
  caretSlopeRise: restructure.int16,
  // Caret slope (rise/run)
  caretSlopeRun: restructure.int16,
  caretOffset: restructure.int16,
  // Set value equal to 0 for nonslanted fonts
  reserved: new restructure.Reserved(restructure.int16, 4),
  metricDataFormat: restructure.int16,
  // Set to 0
  numberOfMetrics: restructure.uint16 // Number of advance heights in the Vertical Metrics table

});

var VmtxEntry = new restructure.Struct({
  advance: restructure.uint16,
  // The advance height of the glyph
  bearing: restructure.int16 // The top sidebearing of the glyph

}); // Vertical Metrics Table

var vmtx = new restructure.Struct({
  metrics: new restructure.LazyArray(VmtxEntry, function (t) {
    return t.parent.vhea.numberOfMetrics;
  }),
  bearings: new restructure.LazyArray(restructure.int16, function (t) {
    return t.parent.maxp.numGlyphs - t.parent.vhea.numberOfMetrics;
  })
});

var shortFrac = new restructure.Fixed(16, 'BE', 14);
var Correspondence = new restructure.Struct({
  fromCoord: shortFrac,
  toCoord: shortFrac
});
var Segment = new restructure.Struct({
  pairCount: restructure.uint16,
  correspondence: new restructure.Array(Correspondence, 'pairCount')
});
var avar = new restructure.Struct({
  version: restructure.fixed32,
  axisCount: restructure.uint32,
  segment: new restructure.Array(Segment, 'axisCount')
});

var UnboundedArrayAccessor =
/*#__PURE__*/
function () {
  function UnboundedArrayAccessor(type, stream, parent) {
    this.type = type;
    this.stream = stream;
    this.parent = parent;
    this.base = this.stream.pos;
    this._items = [];
  }

  var _proto = UnboundedArrayAccessor.prototype;

  _proto.getItem = function getItem(index) {
    if (this._items[index] == null) {
      var pos = this.stream.pos;
      this.stream.pos = this.base + this.type.size(null, this.parent) * index;
      this._items[index] = this.type.decode(this.stream, this.parent);
      this.stream.pos = pos;
    }

    return this._items[index];
  };

  _proto.inspect = function inspect() {
    return "[UnboundedArray " + this.type.constructor.name + "]";
  };

  return UnboundedArrayAccessor;
}();

var UnboundedArray =
/*#__PURE__*/
function (_r$Array) {
  _inheritsLoose(UnboundedArray, _r$Array);

  function UnboundedArray(type) {
    return _r$Array.call(this, type, 0) || this;
  }

  var _proto2 = UnboundedArray.prototype;

  _proto2.decode = function decode(stream, parent) {
    return new UnboundedArrayAccessor(this.type, stream, parent);
  };

  return UnboundedArray;
}(restructure.Array);
var LookupTable = function LookupTable(ValueType) {
  if (ValueType === void 0) {
    ValueType = restructure.uint16;
  }

  // Helper class that makes internal structures invisible to pointers
  var Shadow =
  /*#__PURE__*/
  function () {
    function Shadow(type) {
      this.type = type;
    }

    var _proto3 = Shadow.prototype;

    _proto3.decode = function decode(stream, ctx) {
      ctx = ctx.parent.parent;
      return this.type.decode(stream, ctx);
    };

    _proto3.size = function size(val, ctx) {
      ctx = ctx.parent.parent;
      return this.type.size(val, ctx);
    };

    _proto3.encode = function encode(stream, val, ctx) {
      ctx = ctx.parent.parent;
      return this.type.encode(stream, val, ctx);
    };

    return Shadow;
  }();

  ValueType = new Shadow(ValueType);
  var BinarySearchHeader = new restructure.Struct({
    unitSize: restructure.uint16,
    nUnits: restructure.uint16,
    searchRange: restructure.uint16,
    entrySelector: restructure.uint16,
    rangeShift: restructure.uint16
  });
  var LookupSegmentSingle = new restructure.Struct({
    lastGlyph: restructure.uint16,
    firstGlyph: restructure.uint16,
    value: ValueType
  });
  var LookupSegmentArray = new restructure.Struct({
    lastGlyph: restructure.uint16,
    firstGlyph: restructure.uint16,
    values: new restructure.Pointer(restructure.uint16, new restructure.Array(ValueType, function (t) {
      return t.lastGlyph - t.firstGlyph + 1;
    }), {
      type: 'parent'
    })
  });
  var LookupSingle = new restructure.Struct({
    glyph: restructure.uint16,
    value: ValueType
  });
  return new restructure.VersionedStruct(restructure.uint16, {
    0: {
      values: new UnboundedArray(ValueType) // length == number of glyphs maybe?

    },
    2: {
      binarySearchHeader: BinarySearchHeader,
      segments: new restructure.Array(LookupSegmentSingle, function (t) {
        return t.binarySearchHeader.nUnits;
      })
    },
    4: {
      binarySearchHeader: BinarySearchHeader,
      segments: new restructure.Array(LookupSegmentArray, function (t) {
        return t.binarySearchHeader.nUnits;
      })
    },
    6: {
      binarySearchHeader: BinarySearchHeader,
      segments: new restructure.Array(LookupSingle, function (t) {
        return t.binarySearchHeader.nUnits;
      })
    },
    8: {
      firstGlyph: restructure.uint16,
      count: restructure.uint16,
      values: new restructure.Array(ValueType, 'count')
    }
  });
};
function StateTable(entryData, lookupType) {
  if (entryData === void 0) {
    entryData = {};
  }

  if (lookupType === void 0) {
    lookupType = restructure.uint16;
  }

  var entry = Object.assign({
    newState: restructure.uint16,
    flags: restructure.uint16
  }, entryData);
  var Entry = new restructure.Struct(entry);
  var StateArray = new UnboundedArray(new restructure.Array(restructure.uint16, function (t) {
    return t.nClasses;
  }));
  var StateHeader = new restructure.Struct({
    nClasses: restructure.uint32,
    classTable: new restructure.Pointer(restructure.uint32, new LookupTable(lookupType)),
    stateArray: new restructure.Pointer(restructure.uint32, StateArray),
    entryTable: new restructure.Pointer(restructure.uint32, new UnboundedArray(Entry))
  });
  return StateHeader;
} // This is the old version of the StateTable structure

function StateTable1(entryData, lookupType) {
  if (entryData === void 0) {
    entryData = {};
  }

  if (lookupType === void 0) {
    lookupType = restructure.uint16;
  }

  var ClassLookupTable = new restructure.Struct({
    version: function version() {
      return 8;
    },
    // simulate LookupTable
    firstGlyph: restructure.uint16,
    values: new restructure.Array(restructure.uint8, restructure.uint16)
  });
  var entry = Object.assign({
    newStateOffset: restructure.uint16,
    // convert offset to stateArray index
    newState: function newState(t) {
      return (t.newStateOffset - (t.parent.stateArray.base - t.parent._startOffset)) / t.parent.nClasses;
    },
    flags: restructure.uint16
  }, entryData);
  var Entry = new restructure.Struct(entry);
  var StateArray = new UnboundedArray(new restructure.Array(restructure.uint8, function (t) {
    return t.nClasses;
  }));
  var StateHeader1 = new restructure.Struct({
    nClasses: restructure.uint16,
    classTable: new restructure.Pointer(restructure.uint16, ClassLookupTable),
    stateArray: new restructure.Pointer(restructure.uint16, StateArray),
    entryTable: new restructure.Pointer(restructure.uint16, new UnboundedArray(Entry))
  });
  return StateHeader1;
}

var BslnSubtable = new restructure.VersionedStruct('format', {
  0: {
    // Distance-based, no mapping
    deltas: new restructure.Array(restructure.int16, 32)
  },
  1: {
    // Distance-based, with mapping
    deltas: new restructure.Array(restructure.int16, 32),
    mappingData: new LookupTable(restructure.uint16)
  },
  2: {
    // Control point-based, no mapping
    standardGlyph: restructure.uint16,
    controlPoints: new restructure.Array(restructure.uint16, 32)
  },
  3: {
    // Control point-based, with mapping
    standardGlyph: restructure.uint16,
    controlPoints: new restructure.Array(restructure.uint16, 32),
    mappingData: new LookupTable(restructure.uint16)
  }
});
var bsln = new restructure.Struct({
  version: restructure.fixed32,
  format: restructure.uint16,
  defaultBaseline: restructure.uint16,
  subtable: BslnSubtable
});

var Setting = new restructure.Struct({
  setting: restructure.uint16,
  nameIndex: restructure.int16,
  name: function name(t) {
    return t.parent.parent.parent.name.records.fontFeatures[t.nameIndex];
  }
});
var FeatureName = new restructure.Struct({
  feature: restructure.uint16,
  nSettings: restructure.uint16,
  settingTable: new restructure.Pointer(restructure.uint32, new restructure.Array(Setting, 'nSettings'), {
    type: 'parent'
  }),
  featureFlags: new restructure.Bitfield(restructure.uint8, [null, null, null, null, null, null, 'hasDefault', 'exclusive']),
  defaultSetting: restructure.uint8,
  nameIndex: restructure.int16,
  name: function name(t) {
    return t.parent.parent.name.records.fontFeatures[t.nameIndex];
  }
});
var feat = new restructure.Struct({
  version: restructure.fixed32,
  featureNameCount: restructure.uint16,
  reserved1: new restructure.Reserved(restructure.uint16),
  reserved2: new restructure.Reserved(restructure.uint32),
  featureNames: new restructure.Array(FeatureName, 'featureNameCount')
});

var Axis$1 = new restructure.Struct({
  axisTag: new restructure.String(4),
  minValue: restructure.fixed32,
  defaultValue: restructure.fixed32,
  maxValue: restructure.fixed32,
  flags: restructure.uint16,
  nameID: restructure.uint16,
  name: function name(t) {
    return t.parent.parent.name.records.fontFeatures[t.nameID];
  }
});
var Instance = new restructure.Struct({
  nameID: restructure.uint16,
  name: function name(t) {
    return t.parent.parent.name.records.fontFeatures[t.nameID];
  },
  flags: restructure.uint16,
  coord: new restructure.Array(restructure.fixed32, function (t) {
    return t.parent.axisCount;
  }),
  postscriptNameID: new restructure.Optional(restructure.uint16, function (t) {
    return t.parent.instanceSize - t._currentOffset > 0;
  })
});
var fvar = new restructure.Struct({
  version: restructure.fixed32,
  offsetToData: restructure.uint16,
  countSizePairs: restructure.uint16,
  axisCount: restructure.uint16,
  axisSize: restructure.uint16,
  instanceCount: restructure.uint16,
  instanceSize: restructure.uint16,
  axis: new restructure.Array(Axis$1, 'axisCount'),
  instance: new restructure.Array(Instance, 'instanceCount')
});

var shortFrac$1 = new restructure.Fixed(16, 'BE', 14);

var Offset =
/*#__PURE__*/
function () {
  function Offset() {}

  Offset.decode = function decode(stream, parent) {
    // In short format, offsets are multiplied by 2.
    // This doesn't seem to be documented by Apple, but it
    // is implemented this way in Freetype.
    return parent.flags ? stream.readUInt32BE() : stream.readUInt16BE() * 2;
  };

  return Offset;
}();

var gvar = new restructure.Struct({
  version: restructure.uint16,
  reserved: new restructure.Reserved(restructure.uint16),
  axisCount: restructure.uint16,
  globalCoordCount: restructure.uint16,
  globalCoords: new restructure.Pointer(restructure.uint32, new restructure.Array(new restructure.Array(shortFrac$1, 'axisCount'), 'globalCoordCount')),
  glyphCount: restructure.uint16,
  flags: restructure.uint16,
  offsetToData: restructure.uint32,
  offsets: new restructure.Array(new restructure.Pointer(Offset, 'void', {
    relativeTo: 'offsetToData',
    allowNull: false
  }), function (t) {
    return t.glyphCount + 1;
  })
});

var ClassTable$1 = new restructure.Struct({
  length: restructure.uint16,
  coverage: restructure.uint16,
  subFeatureFlags: restructure.uint32,
  stateTable: new StateTable1()
});
var WidthDeltaRecord = new restructure.Struct({
  justClass: restructure.uint32,
  beforeGrowLimit: restructure.fixed32,
  beforeShrinkLimit: restructure.fixed32,
  afterGrowLimit: restructure.fixed32,
  afterShrinkLimit: restructure.fixed32,
  growFlags: restructure.uint16,
  shrinkFlags: restructure.uint16
});
var WidthDeltaCluster = new restructure.Array(WidthDeltaRecord, restructure.uint32);
var ActionData = new restructure.VersionedStruct('actionType', {
  0: {
    // Decomposition action
    lowerLimit: restructure.fixed32,
    upperLimit: restructure.fixed32,
    order: restructure.uint16,
    glyphs: new restructure.Array(restructure.uint16, restructure.uint16)
  },
  1: {
    // Unconditional add glyph action
    addGlyph: restructure.uint16
  },
  2: {
    // Conditional add glyph action
    substThreshold: restructure.fixed32,
    addGlyph: restructure.uint16,
    substGlyph: restructure.uint16
  },
  3: {},
  // Stretch glyph action (no data, not supported by CoreText)
  4: {
    // Ductile glyph action (not supported by CoreText)
    variationAxis: restructure.uint32,
    minimumLimit: restructure.fixed32,
    noStretchValue: restructure.fixed32,
    maximumLimit: restructure.fixed32
  },
  5: {
    // Repeated add glyph action
    flags: restructure.uint16,
    glyph: restructure.uint16
  }
});
var Action = new restructure.Struct({
  actionClass: restructure.uint16,
  actionType: restructure.uint16,
  actionLength: restructure.uint32,
  actionData: ActionData,
  padding: new restructure.Reserved(restructure.uint8, function (t) {
    return t.actionLength - t._currentOffset;
  })
});
var PostcompensationAction = new restructure.Array(Action, restructure.uint32);
var PostCompensationTable = new restructure.Struct({
  lookupTable: new LookupTable(new restructure.Pointer(restructure.uint16, PostcompensationAction))
});
var JustificationTable = new restructure.Struct({
  classTable: new restructure.Pointer(restructure.uint16, ClassTable$1, {
    type: 'parent'
  }),
  wdcOffset: restructure.uint16,
  postCompensationTable: new restructure.Pointer(restructure.uint16, PostCompensationTable, {
    type: 'parent'
  }),
  widthDeltaClusters: new LookupTable(new restructure.Pointer(restructure.uint16, WidthDeltaCluster, {
    type: 'parent',
    relativeTo: 'wdcOffset'
  }))
});
var just = new restructure.Struct({
  version: restructure.uint32,
  format: restructure.uint16,
  horizontal: new restructure.Pointer(restructure.uint16, JustificationTable),
  vertical: new restructure.Pointer(restructure.uint16, JustificationTable)
});

var LigatureData = {
  action: restructure.uint16
};
var ContextualData = {
  markIndex: restructure.uint16,
  currentIndex: restructure.uint16
};
var InsertionData = {
  currentInsertIndex: restructure.uint16,
  markedInsertIndex: restructure.uint16
};
var SubstitutionTable = new restructure.Struct({
  items: new UnboundedArray(new restructure.Pointer(restructure.uint32, new LookupTable()))
});
var SubtableData = new restructure.VersionedStruct('type', {
  0: {
    // Indic Rearrangement Subtable
    stateTable: new StateTable()
  },
  1: {
    // Contextual Glyph Substitution Subtable
    stateTable: new StateTable(ContextualData),
    substitutionTable: new restructure.Pointer(restructure.uint32, SubstitutionTable)
  },
  2: {
    // Ligature subtable
    stateTable: new StateTable(LigatureData),
    ligatureActions: new restructure.Pointer(restructure.uint32, new UnboundedArray(restructure.uint32)),
    components: new restructure.Pointer(restructure.uint32, new UnboundedArray(restructure.uint16)),
    ligatureList: new restructure.Pointer(restructure.uint32, new UnboundedArray(restructure.uint16))
  },
  4: {
    // Non-contextual Glyph Substitution Subtable
    lookupTable: new LookupTable()
  },
  5: {
    // Glyph Insertion Subtable
    stateTable: new StateTable(InsertionData),
    insertionActions: new restructure.Pointer(restructure.uint32, new UnboundedArray(restructure.uint16))
  }
});
var Subtable = new restructure.Struct({
  length: restructure.uint32,
  coverage: restructure.uint24,
  type: restructure.uint8,
  subFeatureFlags: restructure.uint32,
  table: SubtableData,
  padding: new restructure.Reserved(restructure.uint8, function (t) {
    return t.length - t._currentOffset;
  })
});
var FeatureEntry = new restructure.Struct({
  featureType: restructure.uint16,
  featureSetting: restructure.uint16,
  enableFlags: restructure.uint32,
  disableFlags: restructure.uint32
});
var MorxChain = new restructure.Struct({
  defaultFlags: restructure.uint32,
  chainLength: restructure.uint32,
  nFeatureEntries: restructure.uint32,
  nSubtables: restructure.uint32,
  features: new restructure.Array(FeatureEntry, 'nFeatureEntries'),
  subtables: new restructure.Array(Subtable, 'nSubtables')
});
var morx = new restructure.Struct({
  version: restructure.uint16,
  unused: new restructure.Reserved(restructure.uint16),
  nChains: restructure.uint32,
  chains: new restructure.Array(MorxChain, 'nChains')
});

var OpticalBounds = new restructure.Struct({
  left: restructure.int16,
  top: restructure.int16,
  right: restructure.int16,
  bottom: restructure.int16
});
var opbd = new restructure.Struct({
  version: restructure.fixed32,
  format: restructure.uint16,
  lookupTable: new LookupTable(OpticalBounds)
});

var tables = {};
tables.cmap = cmap;
tables.head = head;
tables.hhea = hhea;
tables.hmtx = hmtx;
tables.maxp = maxp;
tables.name = NameTable;
tables['OS/2'] = OS2;
tables.post = post; // TrueType Outlines
tables.fpgm = fpgm;
tables.loca = loca;
tables.prep = prep;
tables['cvt '] = cvt;
tables.glyf = glyf; // PostScript Outlines
tables['CFF '] = CFFFont;
tables['CFF2'] = CFFFont;
tables.VORG = VORG; // Bitmap Glyphs
tables.EBLC = EBLC;
tables.CBLC = tables.EBLC;
tables.sbix = sbix;
tables.COLR = COLR;
tables.CPAL = CPAL; // Advanced OpenType Tables
tables.BASE = BASE;
tables.GDEF = GDEF;
tables.GPOS = GPOS;
tables.GSUB = GSUB;
tables.JSTF = JSTF; // OpenType variations tables
tables.HVAR = HVAR; // Other OpenType Tables
tables.DSIG = DSIG;
tables.gasp = gasp;
tables.hdmx = hdmx;
tables.kern = kern;
tables.LTSH = LTSH;
tables.PCLT = PCLT;
tables.VDMX = VDMX;
tables.vhea = vhea;
tables.vmtx = vmtx; // Apple Advanced Typography Tables
tables.avar = avar;
tables.bsln = bsln;
tables.feat = feat;
tables.fvar = fvar;
tables.gvar = gvar;
tables.just = just;
tables.morx = morx;
tables.opbd = opbd;

var TableEntry = new restructure.Struct({
  tag: new restructure.String(4),
  checkSum: restructure.uint32,
  offset: new restructure.Pointer(restructure.uint32, 'void', {
    type: 'global'
  }),
  length: restructure.uint32
});
var Directory = new restructure.Struct({
  tag: new restructure.String(4),
  numTables: restructure.uint16,
  searchRange: restructure.uint16,
  entrySelector: restructure.uint16,
  rangeShift: restructure.uint16,
  tables: new restructure.Array(TableEntry, 'numTables')
});

Directory.process = function () {
  var tables$$1 = {};

  for (var _iterator = this.tables, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
    var _ref;

    if (_isArray) {
      if (_i >= _iterator.length) break;
      _ref = _iterator[_i++];
    } else {
      _i = _iterator.next();
      if (_i.done) break;
      _ref = _i.value;
    }

    var table = _ref;
    tables$$1[table.tag] = table;
  }

  this.tables = tables$$1;
};

Directory.preEncode = function (stream) {
  var tables$$1 = [];

  for (var tag in this.tables) {
    var table = this.tables[tag];

    if (table) {
      tables$$1.push({
        tag: tag,
        checkSum: 0,
        offset: new restructure.VoidPointer(tables[tag], table),
        length: tables[tag].size(table)
      });
    }
  }

  this.tag = 'true';
  this.numTables = tables$$1.length;
  this.tables = tables$$1;
  var maxExponentFor2 = Math.floor(Math.log(this.numTables) / Math.LN2);
  var maxPowerOf2 = Math.pow(2, maxExponentFor2);
  this.searchRange = maxPowerOf2 * 16;
  this.entrySelector = Math.log(maxPowerOf2) / Math.LN2;
  this.rangeShift = this.numTables * 16 - this.searchRange;
};

function binarySearch(arr, cmp) {
  var min = 0;
  var max = arr.length - 1;

  while (min <= max) {
    var mid = min + max >> 1;
    var res = cmp(arr[mid]);

    if (res < 0) {
      max = mid - 1;
    } else if (res > 0) {
      min = mid + 1;
    } else {
      return mid;
    }
  }

  return -1;
}
function range(index, end) {
  var range = [];

  while (index < end) {
    range.push(index++);
  }

  return range;
}

var _class;
var CmapProcessor = (_class =
/*#__PURE__*/
function () {
  function CmapProcessor(cmapTable) {
    // Attempt to find a Unicode cmap first
    this.encoding = null;
    this.cmap = this.findSubtable(cmapTable, [// 32-bit subtables
    [3, 10], [0, 6], [0, 4], // 16-bit subtables
    [3, 1], [0, 3], [0, 2], [0, 1], [0, 0]]); // If not unicode cmap was found, and iconv-lite is installed,
    // take the first table with a supported encoding.

    if (!this.cmap && lib) {
      for (var _iterator = cmapTable.tables, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
        var _ref;

        if (_isArray) {
          if (_i >= _iterator.length) break;
          _ref = _iterator[_i++];
        } else {
          _i = _iterator.next();
          if (_i.done) break;
          _ref = _i.value;
        }

        var cmap = _ref;
        var encoding = getEncoding(cmap.platformID, cmap.encodingID, cmap.table.language - 1);

        if (lib.encodingExists(encoding)) {
          this.cmap = cmap.table;
          this.encoding = encoding;
        }
      }
    }

    if (!this.cmap) {
      throw new Error("Could not find a supported cmap table");
    }

    this.uvs = this.findSubtable(cmapTable, [[0, 5]]);

    if (this.uvs && this.uvs.version !== 14) {
      this.uvs = null;
    }
  }

  var _proto = CmapProcessor.prototype;

  _proto.findSubtable = function findSubtable(cmapTable, pairs) {
    for (var _iterator2 = pairs, _isArray2 = Array.isArray(_iterator2), _i2 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
      var _ref2;

      if (_isArray2) {
        if (_i2 >= _iterator2.length) break;
        _ref2 = _iterator2[_i2++];
      } else {
        _i2 = _iterator2.next();
        if (_i2.done) break;
        _ref2 = _i2.value;
      }

      var _ref3 = _ref2,
          platformID = _ref3[0],
          encodingID = _ref3[1];

      for (var _iterator3 = cmapTable.tables, _isArray3 = Array.isArray(_iterator3), _i3 = 0, _iterator3 = _isArray3 ? _iterator3 : _iterator3[Symbol.iterator]();;) {
        var _ref4;

        if (_isArray3) {
          if (_i3 >= _iterator3.length) break;
          _ref4 = _iterator3[_i3++];
        } else {
          _i3 = _iterator3.next();
          if (_i3.done) break;
          _ref4 = _i3.value;
        }

        var cmap = _ref4;

        if (cmap.platformID === platformID && cmap.encodingID === encodingID) {
          return cmap.table;
        }
      }
    }

    return null;
  };

  _proto.lookup = function lookup(codepoint, variationSelector) {
    // If there is no Unicode cmap in this font, we need to re-encode
    // the codepoint in the encoding that the cmap supports.
    if (this.encoding) {
      var buf = lib.encode(String.fromCodePoint(codepoint), this.encoding);
      codepoint = 0;

      for (var i = 0; i < buf.length; i++) {
        codepoint = codepoint << 8 | buf[i];
      } // Otherwise, try to get a Unicode variation selector for this codepoint if one is provided.

    } else if (variationSelector) {
      var gid = this.getVariationSelector(codepoint, variationSelector);

      if (gid) {
        return gid;
      }
    }

    var cmap = this.cmap;

    switch (cmap.version) {
      case 0:
        return cmap.codeMap.get(codepoint) || 0;

      case 4:
        {
          var min = 0;
          var max = cmap.segCount - 1;

          while (min <= max) {
            var mid = min + max >> 1;

            if (codepoint < cmap.startCode.get(mid)) {
              max = mid - 1;
            } else if (codepoint > cmap.endCode.get(mid)) {
              min = mid + 1;
            } else {
              var rangeOffset = cmap.idRangeOffset.get(mid);

              var _gid = void 0;

              if (rangeOffset === 0) {
                _gid = codepoint + cmap.idDelta.get(mid);
              } else {
                var index = rangeOffset / 2 + (codepoint - cmap.startCode.get(mid)) - (cmap.segCount - mid);
                _gid = cmap.glyphIndexArray.get(index) || 0;

                if (_gid !== 0) {
                  _gid += cmap.idDelta.get(mid);
                }
              }

              return _gid & 0xffff;
            }
          }

          return 0;
        }

      case 8:
        throw new Error('TODO: cmap format 8');

      case 6:
      case 10:
        return cmap.glyphIndices.get(codepoint - cmap.firstCode) || 0;

      case 12:
      case 13:
        {
          var _min = 0;

          var _max = cmap.nGroups - 1;

          while (_min <= _max) {
            var _mid = _min + _max >> 1;

            var group = cmap.groups.get(_mid);

            if (codepoint < group.startCharCode) {
              _max = _mid - 1;
            } else if (codepoint > group.endCharCode) {
              _min = _mid + 1;
            } else {
              if (cmap.version === 12) {
                return group.glyphID + (codepoint - group.startCharCode);
              } else {
                return group.glyphID;
              }
            }
          }

          return 0;
        }

      case 14:
        throw new Error('TODO: cmap format 14');

      default:
        throw new Error("Unknown cmap format " + cmap.version);
    }
  };

  _proto.getVariationSelector = function getVariationSelector(codepoint, variationSelector) {
    if (!this.uvs) {
      return 0;
    }

    var selectors = this.uvs.varSelectors.toArray();
    var i = binarySearch(selectors, function (x) {
      return variationSelector - x.varSelector;
    });
    var sel = selectors[i];

    if (i !== -1 && sel.defaultUVS) {
      i = binarySearch(sel.defaultUVS, function (x) {
        return codepoint < x.startUnicodeValue ? -1 : codepoint > x.startUnicodeValue + x.additionalCount ? +1 : 0;
      });
    }

    if (i !== -1 && sel.nonDefaultUVS) {
      i = binarySearch(sel.nonDefaultUVS, function (x) {
        return codepoint - x.unicodeValue;
      });

      if (i !== -1) {
        return sel.nonDefaultUVS[i].glyphID;
      }
    }

    return 0;
  };

  _proto.getCharacterSet = function getCharacterSet() {
    var cmap = this.cmap;

    switch (cmap.version) {
      case 0:
        return range(0, cmap.codeMap.length);

      case 4:
        {
          var res = [];
          var endCodes = cmap.endCode.toArray();

          for (var i = 0; i < endCodes.length; i++) {
            var tail = endCodes[i] + 1;
            var start = cmap.startCode.get(i);
            res.push.apply(res, range(start, tail));
          }

          return res;
        }

      case 8:
        throw new Error('TODO: cmap format 8');

      case 6:
      case 10:
        return range(cmap.firstCode, cmap.firstCode + cmap.glyphIndices.length);

      case 12:
      case 13:
        {
          var _res = [];

          for (var _iterator4 = cmap.groups.toArray(), _isArray4 = Array.isArray(_iterator4), _i4 = 0, _iterator4 = _isArray4 ? _iterator4 : _iterator4[Symbol.iterator]();;) {
            var _ref5;

            if (_isArray4) {
              if (_i4 >= _iterator4.length) break;
              _ref5 = _iterator4[_i4++];
            } else {
              _i4 = _iterator4.next();
              if (_i4.done) break;
              _ref5 = _i4.value;
            }

            var group = _ref5;

            _res.push.apply(_res, range(group.startCharCode, group.endCharCode + 1));
          }

          return _res;
        }

      case 14:
        throw new Error('TODO: cmap format 14');

      default:
        throw new Error("Unknown cmap format " + cmap.version);
    }
  };

  _proto.codePointsForGlyph = function codePointsForGlyph(gid) {
    var cmap = this.cmap;

    switch (cmap.version) {
      case 0:
        {
          var res = [];

          for (var i = 0; i < 256; i++) {
            if (cmap.codeMap.get(i) === gid) {
              res.push(i);
            }
          }

          return res;
        }

      case 4:
        {
          var _res2 = [];

          for (var _i5 = 0; _i5 < cmap.segCount; _i5++) {
            var end = cmap.endCode.get(_i5);
            var start = cmap.startCode.get(_i5);
            var rangeOffset = cmap.idRangeOffset.get(_i5);
            var delta = cmap.idDelta.get(_i5);

            for (var c = start; c <= end; c++) {
              var g = 0;

              if (rangeOffset === 0) {
                g = c + delta;
              } else {
                var index = rangeOffset / 2 + (c - start) - (cmap.segCount - _i5);
                g = cmap.glyphIndexArray.get(index) || 0;

                if (g !== 0) {
                  g += delta;
                }
              }

              if (g === gid) {
                _res2.push(c);
              }
            }
          }

          return _res2;
        }

      case 12:
        {
          var _res3 = [];

          for (var _iterator5 = cmap.groups.toArray(), _isArray5 = Array.isArray(_iterator5), _i6 = 0, _iterator5 = _isArray5 ? _iterator5 : _iterator5[Symbol.iterator]();;) {
            var _ref6;

            if (_isArray5) {
              if (_i6 >= _iterator5.length) break;
              _ref6 = _iterator5[_i6++];
            } else {
              _i6 = _iterator5.next();
              if (_i6.done) break;
              _ref6 = _i6.value;
            }

            var group = _ref6;

            if (gid >= group.glyphID && gid <= group.glyphID + (group.endCharCode - group.startCharCode)) {
              _res3.push(group.startCharCode + (gid - group.glyphID));
            }
          }

          return _res3;
        }

      case 13:
        {
          var _res4 = [];

          for (var _iterator6 = cmap.groups.toArray(), _isArray6 = Array.isArray(_iterator6), _i7 = 0, _iterator6 = _isArray6 ? _iterator6 : _iterator6[Symbol.iterator]();;) {
            var _ref7;

            if (_isArray6) {
              if (_i7 >= _iterator6.length) break;
              _ref7 = _iterator6[_i7++];
            } else {
              _i7 = _iterator6.next();
              if (_i7.done) break;
              _ref7 = _i7.value;
            }

            var _group = _ref7;

            if (gid === _group.glyphID) {
              _res4.push.apply(_res4, range(_group.startCharCode, _group.endCharCode + 1));
            }
          }

          return _res4;
        }

      default:
        throw new Error("Unknown cmap format " + cmap.version);
    }
  };

  return CmapProcessor;
}(), (_applyDecoratedDescriptor(_class.prototype, "getCharacterSet", [cache], Object.getOwnPropertyDescriptor(_class.prototype, "getCharacterSet"), _class.prototype), _applyDecoratedDescriptor(_class.prototype, "codePointsForGlyph", [cache], Object.getOwnPropertyDescriptor(_class.prototype, "codePointsForGlyph"), _class.prototype)), _class);

var KernProcessor =
/*#__PURE__*/
function () {
  function KernProcessor(font) {
    this.kern = font.kern;
  }

  var _proto = KernProcessor.prototype;

  _proto.process = function process(glyphs, positions) {
    for (var glyphIndex = 0; glyphIndex < glyphs.length - 1; glyphIndex++) {
      var left = glyphs[glyphIndex].id;
      var right = glyphs[glyphIndex + 1].id;
      positions[glyphIndex].xAdvance += this.getKerning(left, right);
    }
  };

  _proto.getKerning = function getKerning(left, right) {
    var res = 0;

    for (var _iterator = this.kern.tables, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
      var _ref;

      if (_isArray) {
        if (_i >= _iterator.length) break;
        _ref = _iterator[_i++];
      } else {
        _i = _iterator.next();
        if (_i.done) break;
        _ref = _i.value;
      }

      var table = _ref;

      if (table.coverage.crossStream) {
        continue;
      }

      switch (table.version) {
        case 0:
          if (!table.coverage.horizontal) {
            continue;
          }

          break;

        case 1:
          if (table.coverage.vertical || table.coverage.variation) {
            continue;
          }

          break;

        default:
          throw new Error("Unsupported kerning table version " + table.version);
      }

      var val = 0;
      var s = table.subtable;

      switch (table.format) {
        case 0:
          var pairIdx = binarySearch(s.pairs, function (pair) {
            return left - pair.left || right - pair.right;
          });

          if (pairIdx >= 0) {
            val = s.pairs[pairIdx].value;
          }

          break;

        case 2:
          var leftOffset = 0,
              rightOffset = 0;

          if (left >= s.leftTable.firstGlyph && left < s.leftTable.firstGlyph + s.leftTable.nGlyphs) {
            leftOffset = s.leftTable.offsets[left - s.leftTable.firstGlyph];
          } else {
            leftOffset = s.array.off;
          }

          if (right >= s.rightTable.firstGlyph && right < s.rightTable.firstGlyph + s.rightTable.nGlyphs) {
            rightOffset = s.rightTable.offsets[right - s.rightTable.firstGlyph];
          }

          var index = (leftOffset + rightOffset - s.array.off) / 2;
          val = s.array.values.get(index);
          break;

        case 3:
          if (left >= s.glyphCount || right >= s.glyphCount) {
            return 0;
          }

          val = s.kernValue[s.kernIndex[s.leftClass[left] * s.rightClassCount + s.rightClass[right]]];
          break;

        default:
          throw new Error("Unsupported kerning sub-table format " + table.format);
      } // Microsoft supports the override flag, which resets the result
      // Otherwise, the sum of the results from all subtables is returned


      if (table.coverage.override) {
        res = val;
      } else {
        res += val;
      }
    }

    return res;
  };

  return KernProcessor;
}();

var TINF_OK = 0;
var TINF_DATA_ERROR = -3;

function Tree() {
  this.table = new Uint16Array(16);
  /* table of code length counts */

  this.trans = new Uint16Array(288);
  /* code -> symbol translation table */
}

function Data(source, dest) {
  this.source = source;
  this.sourceIndex = 0;
  this.tag = 0;
  this.bitcount = 0;
  this.dest = dest;
  this.destLen = 0;
  this.ltree = new Tree();
  /* dynamic length/symbol tree */

  this.dtree = new Tree();
  /* dynamic distance tree */
}
/* --------------------------------------------------- *
 * -- uninitialized global data (static structures) -- *
 * --------------------------------------------------- */


var sltree = new Tree();
var sdtree = new Tree();
/* extra bits and base tables for length codes */

var length_bits = new Uint8Array(30);
var length_base = new Uint16Array(30);
/* extra bits and base tables for distance codes */

var dist_bits = new Uint8Array(30);
var dist_base = new Uint16Array(30);
/* special ordering of code length codes */

var clcidx = new Uint8Array([16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15]);
/* used by tinf_decode_trees, avoids allocations every call */

var code_tree = new Tree();
var lengths = new Uint8Array(288 + 32);
/* ----------------------- *
 * -- utility functions -- *
 * ----------------------- */

/* build extra bits and base tables */

function tinf_build_bits_base(bits, base, delta, first) {
  var i, sum;
  /* build bits table */

  for (i = 0; i < delta; ++i) {
    bits[i] = 0;
  }

  for (i = 0; i < 30 - delta; ++i) {
    bits[i + delta] = i / delta | 0;
  }
  /* build base table */


  for (sum = first, i = 0; i < 30; ++i) {
    base[i] = sum;
    sum += 1 << bits[i];
  }
}
/* build the fixed huffman trees */


function tinf_build_fixed_trees(lt, dt) {
  var i;
  /* build fixed length tree */

  for (i = 0; i < 7; ++i) {
    lt.table[i] = 0;
  }

  lt.table[7] = 24;
  lt.table[8] = 152;
  lt.table[9] = 112;

  for (i = 0; i < 24; ++i) {
    lt.trans[i] = 256 + i;
  }

  for (i = 0; i < 144; ++i) {
    lt.trans[24 + i] = i;
  }

  for (i = 0; i < 8; ++i) {
    lt.trans[24 + 144 + i] = 280 + i;
  }

  for (i = 0; i < 112; ++i) {
    lt.trans[24 + 144 + 8 + i] = 144 + i;
  }
  /* build fixed distance tree */


  for (i = 0; i < 5; ++i) {
    dt.table[i] = 0;
  }

  dt.table[5] = 32;

  for (i = 0; i < 32; ++i) {
    dt.trans[i] = i;
  }
}
/* given an array of code lengths, build a tree */


var offs = new Uint16Array(16);

function tinf_build_tree(t, lengths, off, num) {
  var i, sum;
  /* clear code length count table */

  for (i = 0; i < 16; ++i) {
    t.table[i] = 0;
  }
  /* scan symbol lengths, and sum code length counts */


  for (i = 0; i < num; ++i) {
    t.table[lengths[off + i]]++;
  }

  t.table[0] = 0;
  /* compute offset table for distribution sort */

  for (sum = 0, i = 0; i < 16; ++i) {
    offs[i] = sum;
    sum += t.table[i];
  }
  /* create code->symbol translation table (symbols sorted by code) */


  for (i = 0; i < num; ++i) {
    if (lengths[off + i]) t.trans[offs[lengths[off + i]]++] = i;
  }
}
/* ---------------------- *
 * -- decode functions -- *
 * ---------------------- */

/* get one bit from source stream */


function tinf_getbit(d) {
  /* check if tag is empty */
  if (!d.bitcount--) {
    /* load next tag */
    d.tag = d.source[d.sourceIndex++];
    d.bitcount = 7;
  }
  /* shift bit out of tag */


  var bit = d.tag & 1;
  d.tag >>>= 1;
  return bit;
}
/* read a num bit value from a stream and add base */


function tinf_read_bits(d, num, base) {
  if (!num) return base;

  while (d.bitcount < 24) {
    d.tag |= d.source[d.sourceIndex++] << d.bitcount;
    d.bitcount += 8;
  }

  var val = d.tag & 0xffff >>> 16 - num;
  d.tag >>>= num;
  d.bitcount -= num;
  return val + base;
}
/* given a data stream and a tree, decode a symbol */


function tinf_decode_symbol(d, t) {
  while (d.bitcount < 24) {
    d.tag |= d.source[d.sourceIndex++] << d.bitcount;
    d.bitcount += 8;
  }

  var sum = 0,
      cur = 0,
      len = 0;
  var tag = d.tag;
  /* get more bits while code value is above sum */

  do {
    cur = 2 * cur + (tag & 1);
    tag >>>= 1;
    ++len;
    sum += t.table[len];
    cur -= t.table[len];
  } while (cur >= 0);

  d.tag = tag;
  d.bitcount -= len;
  return t.trans[sum + cur];
}
/* given a data stream, decode dynamic trees from it */


function tinf_decode_trees(d, lt, dt) {
  var hlit, hdist, hclen;
  var i, num, length;
  /* get 5 bits HLIT (257-286) */

  hlit = tinf_read_bits(d, 5, 257);
  /* get 5 bits HDIST (1-32) */

  hdist = tinf_read_bits(d, 5, 1);
  /* get 4 bits HCLEN (4-19) */

  hclen = tinf_read_bits(d, 4, 4);

  for (i = 0; i < 19; ++i) {
    lengths[i] = 0;
  }
  /* read code lengths for code length alphabet */


  for (i = 0; i < hclen; ++i) {
    /* get 3 bits code length (0-7) */
    var clen = tinf_read_bits(d, 3, 0);
    lengths[clcidx[i]] = clen;
  }
  /* build code length tree */


  tinf_build_tree(code_tree, lengths, 0, 19);
  /* decode code lengths for the dynamic trees */

  for (num = 0; num < hlit + hdist;) {
    var sym = tinf_decode_symbol(d, code_tree);

    switch (sym) {
      case 16:
        /* copy previous code length 3-6 times (read 2 bits) */
        var prev = lengths[num - 1];

        for (length = tinf_read_bits(d, 2, 3); length; --length) {
          lengths[num++] = prev;
        }

        break;

      case 17:
        /* repeat code length 0 for 3-10 times (read 3 bits) */
        for (length = tinf_read_bits(d, 3, 3); length; --length) {
          lengths[num++] = 0;
        }

        break;

      case 18:
        /* repeat code length 0 for 11-138 times (read 7 bits) */
        for (length = tinf_read_bits(d, 7, 11); length; --length) {
          lengths[num++] = 0;
        }

        break;

      default:
        /* values 0-15 represent the actual code lengths */
        lengths[num++] = sym;
        break;
    }
  }
  /* build dynamic trees */


  tinf_build_tree(lt, lengths, 0, hlit);
  tinf_build_tree(dt, lengths, hlit, hdist);
}
/* ----------------------------- *
 * -- block inflate functions -- *
 * ----------------------------- */

/* given a stream and two trees, inflate a block of data */


function tinf_inflate_block_data(d, lt, dt) {
  while (1) {
    var sym = tinf_decode_symbol(d, lt);
    /* check for end of block */

    if (sym === 256) {
      return TINF_OK;
    }

    if (sym < 256) {
      d.dest[d.destLen++] = sym;
    } else {
      var length, dist, offs;
      var i;
      sym -= 257;
      /* possibly get more bits from length code */

      length = tinf_read_bits(d, length_bits[sym], length_base[sym]);
      dist = tinf_decode_symbol(d, dt);
      /* possibly get more bits from distance code */

      offs = d.destLen - tinf_read_bits(d, dist_bits[dist], dist_base[dist]);
      /* copy match */

      for (i = offs; i < offs + length; ++i) {
        d.dest[d.destLen++] = d.dest[i];
      }
    }
  }
}
/* inflate an uncompressed block of data */


function tinf_inflate_uncompressed_block(d) {
  var length, invlength;
  var i;
  /* unread from bitbuffer */

  while (d.bitcount > 8) {
    d.sourceIndex--;
    d.bitcount -= 8;
  }
  /* get length */


  length = d.source[d.sourceIndex + 1];
  length = 256 * length + d.source[d.sourceIndex];
  /* get one's complement of length */

  invlength = d.source[d.sourceIndex + 3];
  invlength = 256 * invlength + d.source[d.sourceIndex + 2];
  /* check length */

  if (length !== (~invlength & 0x0000ffff)) return TINF_DATA_ERROR;
  d.sourceIndex += 4;
  /* copy block */

  for (i = length; i; --i) {
    d.dest[d.destLen++] = d.source[d.sourceIndex++];
  }
  /* make sure we start next block on a byte boundary */


  d.bitcount = 0;
  return TINF_OK;
}
/* inflate stream from source to dest */


function tinf_uncompress(source, dest) {
  var d = new Data(source, dest);
  var bfinal, btype, res;

  do {
    /* read final block flag */
    bfinal = tinf_getbit(d);
    /* read block type (2 bits) */

    btype = tinf_read_bits(d, 2, 0);
    /* decompress block */

    switch (btype) {
      case 0:
        /* decompress uncompressed block */
        res = tinf_inflate_uncompressed_block(d);
        break;

      case 1:
        /* decompress block with fixed huffman trees */
        res = tinf_inflate_block_data(d, sltree, sdtree);
        break;

      case 2:
        /* decompress block with dynamic huffman trees */
        tinf_decode_trees(d, d.ltree, d.dtree);
        res = tinf_inflate_block_data(d, d.ltree, d.dtree);
        break;

      default:
        res = TINF_DATA_ERROR;
    }

    if (res !== TINF_OK) throw new Error('Data error');
  } while (!bfinal);

  if (d.destLen < d.dest.length) {
    if (typeof d.dest.slice === 'function') return d.dest.slice(0, d.destLen);else return d.dest.subarray(0, d.destLen);
  }

  return d.dest;
}
/* -------------------- *
 * -- initialization -- *
 * -------------------- */

/* build fixed huffman trees */


tinf_build_fixed_trees(sltree, sdtree);
/* build extra bits and base tables */

tinf_build_bits_base(length_bits, length_base, 4, 3);
tinf_build_bits_base(dist_bits, dist_base, 2, 1);
/* fix a special case */

length_bits[28] = 0;
length_base[28] = 258;
var tinyInflate = tinf_uncompress;

var UnicodeTrie, inflate;
inflate = tinyInflate;

UnicodeTrie = function () {
  var DATA_BLOCK_LENGTH, DATA_GRANULARITY, DATA_MASK, INDEX_1_OFFSET, INDEX_2_BLOCK_LENGTH, INDEX_2_BMP_LENGTH, INDEX_2_MASK, INDEX_SHIFT, LSCP_INDEX_2_LENGTH, LSCP_INDEX_2_OFFSET, OMITTED_BMP_INDEX_1_LENGTH, SHIFT_1, SHIFT_1_2, SHIFT_2, UTF8_2B_INDEX_2_LENGTH, UTF8_2B_INDEX_2_OFFSET;
  SHIFT_1 = 6 + 5;
  SHIFT_2 = 5;
  SHIFT_1_2 = SHIFT_1 - SHIFT_2;
  OMITTED_BMP_INDEX_1_LENGTH = 0x10000 >> SHIFT_1;
  INDEX_2_BLOCK_LENGTH = 1 << SHIFT_1_2;
  INDEX_2_MASK = INDEX_2_BLOCK_LENGTH - 1;
  INDEX_SHIFT = 2;
  DATA_BLOCK_LENGTH = 1 << SHIFT_2;
  DATA_MASK = DATA_BLOCK_LENGTH - 1;
  LSCP_INDEX_2_OFFSET = 0x10000 >> SHIFT_2;
  LSCP_INDEX_2_LENGTH = 0x400 >> SHIFT_2;
  INDEX_2_BMP_LENGTH = LSCP_INDEX_2_OFFSET + LSCP_INDEX_2_LENGTH;
  UTF8_2B_INDEX_2_OFFSET = INDEX_2_BMP_LENGTH;
  UTF8_2B_INDEX_2_LENGTH = 0x800 >> 6;
  INDEX_1_OFFSET = UTF8_2B_INDEX_2_OFFSET + UTF8_2B_INDEX_2_LENGTH;
  DATA_GRANULARITY = 1 << INDEX_SHIFT;

  function UnicodeTrie(data) {
    var isBuffer, uncompressedLength, view;
    isBuffer = typeof data.readUInt32BE === 'function' && typeof data.slice === 'function';

    if (isBuffer || data instanceof Uint8Array) {
      if (isBuffer) {
        this.highStart = data.readUInt32BE(0);
        this.errorValue = data.readUInt32BE(4);
        uncompressedLength = data.readUInt32BE(8);
        data = data.slice(12);
      } else {
        view = new DataView(data.buffer);
        this.highStart = view.getUint32(0);
        this.errorValue = view.getUint32(4);
        uncompressedLength = view.getUint32(8);
        data = data.subarray(12);
      }

      data = inflate(data, new Uint8Array(uncompressedLength));
      data = inflate(data, new Uint8Array(uncompressedLength));
      this.data = new Uint32Array(data.buffer);
    } else {
      this.data = data.data, this.highStart = data.highStart, this.errorValue = data.errorValue;
    }
  }

  UnicodeTrie.prototype.get = function (codePoint) {
    var index;

    if (codePoint < 0 || codePoint > 0x10ffff) {
      return this.errorValue;
    }

    if (codePoint < 0xd800 || codePoint > 0xdbff && codePoint <= 0xffff) {
      index = (this.data[codePoint >> SHIFT_2] << INDEX_SHIFT) + (codePoint & DATA_MASK);
      return this.data[index];
    }

    if (codePoint <= 0xffff) {
      index = (this.data[LSCP_INDEX_2_OFFSET + (codePoint - 0xd800 >> SHIFT_2)] << INDEX_SHIFT) + (codePoint & DATA_MASK);
      return this.data[index];
    }

    if (codePoint < this.highStart) {
      index = this.data[INDEX_1_OFFSET - OMITTED_BMP_INDEX_1_LENGTH + (codePoint >> SHIFT_1)];
      index = this.data[index + (codePoint >> SHIFT_2 & INDEX_2_MASK)];
      index = (index << INDEX_SHIFT) + (codePoint & DATA_MASK);
      return this.data[index];
    }

    return this.data[this.data.length - DATA_GRANULARITY];
  };

  return UnicodeTrie;
}();

var unicodeTrie = UnicodeTrie;

var base64Arraybuffer = createCommonjsModule(function (module, exports) {
  /*
   * base64-arraybuffer
   * https://github.com/niklasvh/base64-arraybuffer
   *
   * Copyright (c) 2012 Niklas von Hertzen
   * Licensed under the MIT license.
   */
  (function () {

    var chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/"; // Use a lookup table to find the index.

    var lookup = new Uint8Array(256);

    for (var i = 0; i < chars.length; i++) {
      lookup[chars.charCodeAt(i)] = i;
    }

    exports.encode = function (arraybuffer) {
      var bytes = new Uint8Array(arraybuffer),
          i,
          len = bytes.length,
          base64 = "";

      for (i = 0; i < len; i += 3) {
        base64 += chars[bytes[i] >> 2];
        base64 += chars[(bytes[i] & 3) << 4 | bytes[i + 1] >> 4];
        base64 += chars[(bytes[i + 1] & 15) << 2 | bytes[i + 2] >> 6];
        base64 += chars[bytes[i + 2] & 63];
      }

      if (len % 3 === 2) {
        base64 = base64.substring(0, base64.length - 1) + "=";
      } else if (len % 3 === 1) {
        base64 = base64.substring(0, base64.length - 2) + "==";
      }

      return base64;
    };

    exports.decode = function (base64) {
      var bufferLength = base64.length * 0.75,
          len = base64.length,
          i,
          p = 0,
          encoded1,
          encoded2,
          encoded3,
          encoded4;

      if (base64[base64.length - 1] === "=") {
        bufferLength--;

        if (base64[base64.length - 2] === "=") {
          bufferLength--;
        }
      }

      var arraybuffer = new ArrayBuffer(bufferLength),
          bytes = new Uint8Array(arraybuffer);

      for (i = 0; i < len; i += 4) {
        encoded1 = lookup[base64.charCodeAt(i)];
        encoded2 = lookup[base64.charCodeAt(i + 1)];
        encoded3 = lookup[base64.charCodeAt(i + 2)];
        encoded4 = lookup[base64.charCodeAt(i + 3)];
        bytes[p++] = encoded1 << 2 | encoded2 >> 4;
        bytes[p++] = (encoded2 & 15) << 4 | encoded3 >> 2;
        bytes[p++] = (encoded3 & 3) << 6 | encoded4 & 63;
      }

      return arraybuffer;
    };
  })();
});
var base64Arraybuffer_1 = base64Arraybuffer.encode;
var base64Arraybuffer_2 = base64Arraybuffer.decode;

var base64DeflatedData = "eJxtVtty4zYM/Rc/70Ni97LbN1tJ49SXeGI3O93OjgayGBERRXoo0h5Np/9eAKTcpNOHECAFAgfAAZ2/JkcIqnEeVT/55c9JcZx8mnzradk5Wva83clW8bZjraZly8s68lnLZ2y3Nrzla2tedkhL8crGsmVtHXhhLxvLCzvd8N0t3/0my4mvccjCTb5/mhxdV6FF2xQG+j6h3LpQPivna+UVA5lX7qxGWT5joznOQhl34dMQ4KhVXX78fD0e7Z7OyhsYSHt0Acp9rPqjxxPb3rlYGXW1zNsxrBz/61eirNUrb4qiuL3J8jbLaZazLH/I8scsf8ry5yw/Z/klyWn2N83+ptnfNPubZn/T7G+W7WfZfjbaZ//T7H+a/c+yn1n2Mxv9MK5tbAOQfEEPHaQPn7PhlzHBmzGzmzGFaw4j2Nsx2u0I71ZwfezVtba5mlmswEL54vBIrMjYGVtqAFEmtS3x2XWdY6qtISDLhTu5zr0yIx+tVh6DMOjBK9UK504BmZDF4NEYUee+UxaBby9V5ZVwykMlH/dkB6wcNBAsLlErk0PlIedyawO2BrG+U2cyauiLpGYbMKw9RN/FVif1DTxhZT56HNjhATrkyTgoE5uY8rdQg3gmwtIfT9QerSY1QUFJmYEcsFIZxgCWQHEQmp0mZwS2iez9Pmh0p5Q71cW1istfENyaLLkfHhu0wLZPjZaQz9HKhQOlZFyT3EXrOO4iaqzTtwrshXGtdKc4/MbZxpkUf41dFcUMqctcPnUpZROl/ZFiql5li0OKu6Cy5dN9pNpmfQEBWqHJ6ag54JMpC42tFNsD9ZOtHgxw9NTmA75STo1O0HklYjepkSt2l9Q/pJ7YM9IXKe4CuihVH4wLWG6hkZ7uNDR9uQNhQPTQ6+CTrwF0uWaTZ/UGwtvf4DwiL1JenOILKub4RqmgsNzAILy00RiBYBX4csHqcEwVLIhNojyZunwMkEj74Kifx3y6U76Tze9MvfD+vE937wiHl8B7Defsr+/ACgPvTQV9DhaP0Ash6MyOPRRUcxmbk0fHfh67E00XmHLOL4WE3IHpBq+stBkqKq/KHKQiWameo0nLea2HOikbYiODLpdIWmOGk+7fHxeRspDHYqXBOyq5TBInuHcx6FKG9VqirfMfzmg6keiSoMzPqk/T8mjTM4KOOE8dpUv4vx+0oYIx+B5MUP7dCUc7RN9e672Mthm7RYTUHQo1MAHeO0+/Oo7qlp41DW2XhlwDUQZTc3ya/JV2b0LsTTQhfVzpWMOFxlXeM7BBZuCAXkd5s/dY14lnGydGB2iFtHPt+PQr+bZNWaC8PjuIpNpyCUz6IlqFr86z3X0zUPZEgY/9mFsIMtX/7ZOX54B+vOnRBi2uNVx0uaSXmRPdoDxUd/Fk3CC12WNjvzJR03dla1WusIU2Ir/vCi7yX4DQiCMzeOYR/f06+f73P0+ms+k=";

var base64DeflatedTrie = "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";

// we may be running in a browser so we make an Uint8Array

var data = JSON.parse(String.fromCharCode.apply(String, pako.inflate(base64Arraybuffer_2(base64DeflatedData))));
var trieData = pako.inflate(base64Arraybuffer_2(base64DeflatedTrie));
var trie = new unicodeTrie(trieData);

var log2 = Math.log2 || function (n) {
  return Math.log(n) / Math.LN2;
};

var bits = function bits(n) {
  return log2(n) + 1 | 0;
}; // compute the number of bits stored for each field


var CATEGORY_BITS = bits(data.categories.length - 1);
var COMBINING_BITS = bits(data.combiningClasses.length - 1);
var SCRIPT_BITS = bits(data.scripts.length - 1);
var EAW_BITS = bits(data.eaw.length - 1);
var NUMBER_BITS = 10; // compute shift and mask values for each field

var CATEGORY_SHIFT = COMBINING_BITS + SCRIPT_BITS + EAW_BITS + NUMBER_BITS;
var COMBINING_SHIFT = SCRIPT_BITS + EAW_BITS + NUMBER_BITS;
var SCRIPT_SHIFT = EAW_BITS + NUMBER_BITS;
var EAW_SHIFT = NUMBER_BITS;
var CATEGORY_MASK = (1 << CATEGORY_BITS) - 1;
var COMBINING_MASK = (1 << COMBINING_BITS) - 1;
var SCRIPT_MASK = (1 << SCRIPT_BITS) - 1;
var EAW_MASK = (1 << EAW_BITS) - 1;
var NUMBER_MASK = (1 << NUMBER_BITS) - 1;
var getCategory = function getCategory(codePoint) {
  var val = trie.get(codePoint);
  return data.categories[val >> CATEGORY_SHIFT & CATEGORY_MASK];
};
var getCombiningClass = function getCombiningClass(codePoint) {
  var val = trie.get(codePoint);
  return data.combiningClasses[val >> COMBINING_SHIFT & COMBINING_MASK];
};
var getScript = function getScript(codePoint) {
  var val = trie.get(codePoint);
  return data.scripts[val >> SCRIPT_SHIFT & SCRIPT_MASK];
};
var getEastAsianWidth = function getEastAsianWidth(codePoint) {
  var val = trie.get(codePoint);
  return data.eaw[val >> EAW_SHIFT & EAW_MASK];
};
var getNumericValue = function getNumericValue(codePoint) {
  var val = trie.get(codePoint);
  var num = val & NUMBER_MASK;

  if (num === 0) {
    return null;
  } else if (num <= 50) {
    return num - 1;
  } else if (num < 0x1e0) {
    var numerator = (num >> 4) - 12;
    var denominator = (num & 0xf) + 1;
    return numerator / denominator;
  } else if (num < 0x300) {
    val = (num >> 5) - 14;
    var exp = (num & 0x1f) + 2;

    while (exp > 0) {
      val *= 10;
      exp--;
    }

    return val;
  } else {
    val = (num >> 2) - 0xbf;

    var _exp = (num & 3) + 1;

    while (_exp > 0) {
      val *= 60;
      _exp--;
    }

    return val;
  }
};
var isAlphabetic = function isAlphabetic(codePoint) {
  var category = getCategory(codePoint);
  return category === 'Lu' || category === 'Ll' || category === 'Lt' || category === 'Lm' || category === 'Lo' || category === 'Nl';
};
var isDigit = function isDigit(codePoint) {
  return getCategory(codePoint) === 'Nd';
};
var isPunctuation = function isPunctuation(codePoint) {
  var category = getCategory(codePoint);
  return category === 'Pc' || category === 'Pd' || category === 'Pe' || category === 'Pf' || category === 'Pi' || category === 'Po' || category === 'Ps';
};
var isLowerCase = function isLowerCase(codePoint) {
  return getCategory(codePoint) === 'Ll';
};
var isUpperCase = function isUpperCase(codePoint) {
  return getCategory(codePoint) === 'Lu';
};
var isTitleCase = function isTitleCase(codePoint) {
  return getCategory(codePoint) === 'Lt';
};
var isWhiteSpace = function isWhiteSpace(codePoint) {
  var category = getCategory(codePoint);
  return category === 'Zs' || category === 'Zl' || category === 'Zp';
};
var isBaseForm = function isBaseForm(codePoint) {
  var category = getCategory(codePoint);
  return category === 'Nd' || category === 'No' || category === 'Nl' || category === 'Lu' || category === 'Ll' || category === 'Lt' || category === 'Lm' || category === 'Lo' || category === 'Me' || category === 'Mc';
};
var isMark = function isMark(codePoint) {
  var category = getCategory(codePoint);
  return category === 'Mn' || category === 'Me' || category === 'Mc';
};
var unicode = {
  getCategory: getCategory,
  getCombiningClass: getCombiningClass,
  getScript: getScript,
  getEastAsianWidth: getEastAsianWidth,
  getNumericValue: getNumericValue,
  isAlphabetic: isAlphabetic,
  isDigit: isDigit,
  isPunctuation: isPunctuation,
  isLowerCase: isLowerCase,
  isUpperCase: isUpperCase,
  isTitleCase: isTitleCase,
  isWhiteSpace: isWhiteSpace,
  isBaseForm: isBaseForm,
  isMark: isMark
};

/**
 * This class is used when GPOS does not define 'mark' or 'mkmk' features
 * for positioning marks relative to base glyphs. It uses the unicode
 * combining class property to position marks.
 *
 * Based on code from Harfbuzz, thanks!
 * https://github.com/behdad/harfbuzz/blob/master/src/hb-ot-shape-fallback.cc
 */

var UnicodeLayoutEngine =
/*#__PURE__*/
function () {
  function UnicodeLayoutEngine(font) {
    this.font = font;
  }

  var _proto = UnicodeLayoutEngine.prototype;

  _proto.positionGlyphs = function positionGlyphs(glyphs, positions) {
    // find each base + mark cluster, and position the marks relative to the base
    var clusterStart = 0;
    var clusterEnd = 0;

    for (var index = 0; index < glyphs.length; index++) {
      var glyph = glyphs[index];

      if (glyph.isMark) {
        // TODO: handle ligatures
        clusterEnd = index;
      } else {
        if (clusterStart !== clusterEnd) {
          this.positionCluster(glyphs, positions, clusterStart, clusterEnd);
        }

        clusterStart = clusterEnd = index;
      }
    }

    if (clusterStart !== clusterEnd) {
      this.positionCluster(glyphs, positions, clusterStart, clusterEnd);
    }

    return positions;
  };

  _proto.positionCluster = function positionCluster(glyphs, positions, clusterStart, clusterEnd) {
    var base = glyphs[clusterStart];
    var baseBox = base.cbox.copy(); // adjust bounding box for ligature glyphs

    if (base.codePoints.length > 1) {
      // LTR. TODO: RTL support.
      baseBox.minX += (base.codePoints.length - 1) * baseBox.width / base.codePoints.length;
    }

    var xOffset = -positions[clusterStart].xAdvance;
    var yOffset = 0;
    var yGap = this.font.unitsPerEm / 16; // position each of the mark glyphs relative to the base glyph

    for (var index = clusterStart + 1; index <= clusterEnd; index++) {
      var mark = glyphs[index];
      var markBox = mark.cbox;
      var position = positions[index];
      var combiningClass = this.getCombiningClass(mark.codePoints[0]);

      if (combiningClass !== 'Not_Reordered') {
        position.xOffset = position.yOffset = 0; // x positioning

        switch (combiningClass) {
          case 'Double_Above':
          case 'Double_Below':
            // LTR. TODO: RTL support.
            position.xOffset += baseBox.minX - markBox.width / 2 - markBox.minX;
            break;

          case 'Attached_Below_Left':
          case 'Below_Left':
          case 'Above_Left':
            // left align
            position.xOffset += baseBox.minX - markBox.minX;
            break;

          case 'Attached_Above_Right':
          case 'Below_Right':
          case 'Above_Right':
            // right align
            position.xOffset += baseBox.maxX - markBox.width - markBox.minX;
            break;

          default:
            // Attached_Below, Attached_Above, Below, Above, other
            // center align
            position.xOffset += baseBox.minX + (baseBox.width - markBox.width) / 2 - markBox.minX;
        } // y positioning


        switch (combiningClass) {
          case 'Double_Below':
          case 'Below_Left':
          case 'Below':
          case 'Below_Right':
          case 'Attached_Below_Left':
          case 'Attached_Below':
            // add a small gap between the glyphs if they are not attached
            if (combiningClass === 'Attached_Below_Left' || combiningClass === 'Attached_Below') {
              baseBox.minY += yGap;
            }

            position.yOffset = -baseBox.minY - markBox.maxY;
            baseBox.minY += markBox.height;
            break;

          case 'Double_Above':
          case 'Above_Left':
          case 'Above':
          case 'Above_Right':
          case 'Attached_Above':
          case 'Attached_Above_Right':
            // add a small gap between the glyphs if they are not attached
            if (combiningClass === 'Attached_Above' || combiningClass === 'Attached_Above_Right') {
              baseBox.maxY += yGap;
            }

            position.yOffset = baseBox.maxY - markBox.minY;
            baseBox.maxY += markBox.height;
            break;
        }

        position.xAdvance = position.yAdvance = 0;
        position.xOffset += xOffset;
        position.yOffset += yOffset;
      } else {
        xOffset -= position.xAdvance;
        yOffset -= position.yAdvance;
      }
    }

    return;
  };

  _proto.getCombiningClass = function getCombiningClass$$1(codePoint) {
    var combiningClass = unicode.getCombiningClass(codePoint); // Thai / Lao need some per-character work

    if ((codePoint & ~0xff) === 0x0e00) {
      if (combiningClass === 'Not_Reordered') {
        switch (codePoint) {
          case 0x0e31:
          case 0x0e34:
          case 0x0e35:
          case 0x0e36:
          case 0x0e37:
          case 0x0e47:
          case 0x0e4c:
          case 0x0e3d:
          case 0x0e4e:
            return 'Above_Right';

          case 0x0eb1:
          case 0x0eb4:
          case 0x0eb5:
          case 0x0eb6:
          case 0x0eb7:
          case 0x0ebb:
          case 0x0ecc:
          case 0x0ecd:
            return 'Above';

          case 0x0ebc:
            return 'Below';
        }
      } else if (codePoint === 0x0e3a) {
        // virama
        return 'Below_Right';
      }
    }

    switch (combiningClass) {
      // Hebrew
      case 'CCC10': // sheva

      case 'CCC11': // hataf segol

      case 'CCC12': // hataf patah

      case 'CCC13': // hataf qamats

      case 'CCC14': // hiriq

      case 'CCC15': // tsere

      case 'CCC16': // segol

      case 'CCC17': // patah

      case 'CCC18': // qamats

      case 'CCC20': // qubuts

      case 'CCC22':
        // meteg
        return 'Below';

      case 'CCC23':
        // rafe
        return 'Attached_Above';

      case 'CCC24':
        // shin dot
        return 'Above_Right';

      case 'CCC25': // sin dot

      case 'CCC19':
        // holam
        return 'Above_Left';

      case 'CCC26':
        // point varika
        return 'Above';

      case 'CCC21':
        // dagesh
        break;
      // Arabic and Syriac

      case 'CCC27': // fathatan

      case 'CCC28': // dammatan

      case 'CCC30': // fatha

      case 'CCC31': // damma

      case 'CCC33': // shadda

      case 'CCC34': // sukun

      case 'CCC35': // superscript alef

      case 'CCC36':
        // superscript alaph
        return 'Above';

      case 'CCC29': // kasratan

      case 'CCC32':
        // kasra
        return 'Below';
      // Thai

      case 'CCC103':
        // sara u / sara uu
        return 'Below_Right';

      case 'CCC107':
        // mai
        return 'Above_Right';
      // Lao

      case 'CCC118':
        // sign u / sign uu
        return 'Below';

      case 'CCC122':
        // mai
        return 'Above';
      // Tibetan

      case 'CCC129': // sign aa

      case 'CCC132':
        // sign u
        return 'Below';

      case 'CCC130':
        // sign i
        return 'Above';
    }

    return combiningClass;
  };

  return UnicodeLayoutEngine;
}();

/**
 * Represents a glyph bounding box
 */
var BBox =
/*#__PURE__*/
function () {
  function BBox(minX, minY, maxX, maxY) {
    if (minX === void 0) {
      minX = Infinity;
    }

    if (minY === void 0) {
      minY = Infinity;
    }

    if (maxX === void 0) {
      maxX = -Infinity;
    }

    if (maxY === void 0) {
      maxY = -Infinity;
    }

    /**
     * The minimum X position in the bounding box
     * @type {number}
     */
    this.minX = minX;
    /**
     * The minimum Y position in the bounding box
     * @type {number}
     */

    this.minY = minY;
    /**
     * The maxmimum X position in the bounding box
     * @type {number}
     */

    this.maxX = maxX;
    /**
     * The maxmimum Y position in the bounding box
     * @type {number}
     */

    this.maxY = maxY;
  }
  /**
   * The width of the bounding box
   * @type {number}
   */


  var _proto = BBox.prototype;

  _proto.addPoint = function addPoint(x, y) {
    if (Math.abs(x) !== Infinity) {
      if (x < this.minX) {
        this.minX = x;
      }

      if (x > this.maxX) {
        this.maxX = x;
      }
    }

    if (Math.abs(y) !== Infinity) {
      if (y < this.minY) {
        this.minY = y;
      }

      if (y > this.maxY) {
        this.maxY = y;
      }
    }
  };

  _proto.copy = function copy() {
    return new BBox(this.minX, this.minY, this.maxX, this.maxY);
  };

  _createClass(BBox, [{
    key: "width",
    get: function get() {
      return this.maxX - this.minX;
    }
    /**
     * The height of the bounding box
     * @type {number}
     */

  }, {
    key: "height",
    get: function get() {
      return this.maxY - this.minY;
    }
  }]);

  return BBox;
}();

// Data from http://www.microsoft.com/typography/otspec/scripttags.htm
// and http://www.unicode.org/Public/UNIDATA/PropertyValueAliases.txt.

var UNICODE_SCRIPTS = {
  Caucasian_Albanian: 'aghb',
  Arabic: 'arab',
  Imperial_Aramaic: 'armi',
  Armenian: 'armn',
  Avestan: 'avst',
  Balinese: 'bali',
  Bamum: 'bamu',
  Bassa_Vah: 'bass',
  Batak: 'batk',
  Bengali: ['bng2', 'beng'],
  Bopomofo: 'bopo',
  Brahmi: 'brah',
  Braille: 'brai',
  Buginese: 'bugi',
  Buhid: 'buhd',
  Chakma: 'cakm',
  Canadian_Aboriginal: 'cans',
  Carian: 'cari',
  Cham: 'cham',
  Cherokee: 'cher',
  Coptic: 'copt',
  Cypriot: 'cprt',
  Cyrillic: 'cyrl',
  Devanagari: ['dev2', 'deva'],
  Deseret: 'dsrt',
  Duployan: 'dupl',
  Egyptian_Hieroglyphs: 'egyp',
  Elbasan: 'elba',
  Ethiopic: 'ethi',
  Georgian: 'geor',
  Glagolitic: 'glag',
  Gothic: 'goth',
  Grantha: 'gran',
  Greek: 'grek',
  Gujarati: ['gjr2', 'gujr'],
  Gurmukhi: ['gur2', 'guru'],
  Hangul: 'hang',
  Han: 'hani',
  Hanunoo: 'hano',
  Hebrew: 'hebr',
  Hiragana: 'hira',
  Pahawh_Hmong: 'hmng',
  Katakana_Or_Hiragana: 'hrkt',
  Old_Italic: 'ital',
  Javanese: 'java',
  Kayah_Li: 'kali',
  Katakana: 'kana',
  Kharoshthi: 'khar',
  Khmer: 'khmr',
  Khojki: 'khoj',
  Kannada: ['knd2', 'knda'],
  Kaithi: 'kthi',
  Tai_Tham: 'lana',
  Lao: 'lao ',
  Latin: 'latn',
  Lepcha: 'lepc',
  Limbu: 'limb',
  Linear_A: 'lina',
  Linear_B: 'linb',
  Lisu: 'lisu',
  Lycian: 'lyci',
  Lydian: 'lydi',
  Mahajani: 'mahj',
  Mandaic: 'mand',
  Manichaean: 'mani',
  Mende_Kikakui: 'mend',
  Meroitic_Cursive: 'merc',
  Meroitic_Hieroglyphs: 'mero',
  Malayalam: ['mlm2', 'mlym'],
  Modi: 'modi',
  Mongolian: 'mong',
  Mro: 'mroo',
  Meetei_Mayek: 'mtei',
  Myanmar: ['mym2', 'mymr'],
  Old_North_Arabian: 'narb',
  Nabataean: 'nbat',
  Nko: 'nko ',
  Ogham: 'ogam',
  Ol_Chiki: 'olck',
  Old_Turkic: 'orkh',
  Oriya: ['ory2', 'orya'],
  Osmanya: 'osma',
  Palmyrene: 'palm',
  Pau_Cin_Hau: 'pauc',
  Old_Permic: 'perm',
  Phags_Pa: 'phag',
  Inscriptional_Pahlavi: 'phli',
  Psalter_Pahlavi: 'phlp',
  Phoenician: 'phnx',
  Miao: 'plrd',
  Inscriptional_Parthian: 'prti',
  Rejang: 'rjng',
  Runic: 'runr',
  Samaritan: 'samr',
  Old_South_Arabian: 'sarb',
  Saurashtra: 'saur',
  Shavian: 'shaw',
  Sharada: 'shrd',
  Siddham: 'sidd',
  Khudawadi: 'sind',
  Sinhala: 'sinh',
  Sora_Sompeng: 'sora',
  Sundanese: 'sund',
  Syloti_Nagri: 'sylo',
  Syriac: 'syrc',
  Tagbanwa: 'tagb',
  Takri: 'takr',
  Tai_Le: 'tale',
  New_Tai_Lue: 'talu',
  Tamil: ['tml2', 'taml'],
  Tai_Viet: 'tavt',
  Telugu: ['tel2', 'telu'],
  Tifinagh: 'tfng',
  Tagalog: 'tglg',
  Thaana: 'thaa',
  Thai: 'thai',
  Tibetan: 'tibt',
  Tirhuta: 'tirh',
  Ugaritic: 'ugar',
  Vai: 'vai ',
  Warang_Citi: 'wara',
  Old_Persian: 'xpeo',
  Cuneiform: 'xsux',
  Yi: 'yi  ',
  Inherited: 'zinh',
  Common: 'zyyy',
  Unknown: 'zzzz'
};
var OPENTYPE_SCRIPTS = {};

for (var script in UNICODE_SCRIPTS) {
  var tag = UNICODE_SCRIPTS[script];

  if (Array.isArray(tag)) {
    for (var _iterator = tag, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
      var _ref;

      if (_isArray) {
        if (_i >= _iterator.length) break;
        _ref = _iterator[_i++];
      } else {
        _i = _iterator.next();
        if (_i.done) break;
        _ref = _i.value;
      }

      var t = _ref;
      OPENTYPE_SCRIPTS[t] = script;
    }
  } else {
    OPENTYPE_SCRIPTS[tag] = script;
  }
}
function fromOpenType(tag) {
  return OPENTYPE_SCRIPTS[tag];
}
function forString(string) {
  var len = string.length;
  var idx = 0;

  while (idx < len) {
    var code = string.charCodeAt(idx++); // Check if this is a high surrogate

    if (0xd800 <= code && code <= 0xdbff && idx < len) {
      var next = string.charCodeAt(idx); // Check if this is a low surrogate

      if (0xdc00 <= next && next <= 0xdfff) {
        idx++;
        code = ((code & 0x3FF) << 10) + (next & 0x3FF) + 0x10000;
      }
    }

    var _script = unicode.getScript(code);

    if (_script !== 'Common' && _script !== 'Inherited' && _script !== 'Unknown') {
      return UNICODE_SCRIPTS[_script];
    }
  }

  return UNICODE_SCRIPTS.Unknown;
}
function forCodePoints(codePoints) {
  for (var i = 0; i < codePoints.length; i++) {
    var codePoint = codePoints[i];

    var _script2 = unicode.getScript(codePoint);

    if (_script2 !== 'Common' && _script2 !== 'Inherited' && _script2 !== 'Unknown') {
      return UNICODE_SCRIPTS[_script2];
    }
  }

  return UNICODE_SCRIPTS.Unknown;
} // The scripts in this map are written from right to left

var RTL = {
  arab: true,
  // Arabic
  hebr: true,
  // Hebrew
  syrc: true,
  // Syriac
  thaa: true,
  // Thaana
  cprt: true,
  // Cypriot Syllabary
  khar: true,
  // Kharosthi
  phnx: true,
  // Phoenician
  'nko ': true,
  // N'Ko
  lydi: true,
  // Lydian
  avst: true,
  // Avestan
  armi: true,
  // Imperial Aramaic
  phli: true,
  // Inscriptional Pahlavi
  prti: true,
  // Inscriptional Parthian
  sarb: true,
  // Old South Arabian
  orkh: true,
  // Old Turkic, Orkhon Runic
  samr: true,
  // Samaritan
  mand: true,
  // Mandaic, Mandaean
  merc: true,
  // Meroitic Cursive
  mero: true,
  // Meroitic Hieroglyphs
  // Unicode 7.0 (not listed on http://www.microsoft.com/typography/otspec/scripttags.htm)
  mani: true,
  // Manichaean
  mend: true,
  // Mende Kikakui
  nbat: true,
  // Nabataean
  narb: true,
  // Old North Arabian
  palm: true,
  // Palmyrene
  phlp: true // Psalter Pahlavi

};
function direction(script) {
  if (RTL[script]) {
    return 'rtl';
  }

  return 'ltr';
}

/**
 * Represents a run of Glyph and GlyphPosition objects.
 * Returned by the font layout method.
 */

var GlyphRun =
/*#__PURE__*/
function () {
  function GlyphRun(glyphs, features, script, language, direction$$1) {
    /**
     * An array of Glyph objects in the run
     * @type {Glyph[]}
     */
    this.glyphs = glyphs;
    /**
     * An array of GlyphPosition objects for each glyph in the run
     * @type {GlyphPosition[]}
     */

    this.positions = null;
    /**
     * The script that was requested for shaping. This was either passed in or detected automatically.
     * @type {string}
     */

    this.script = script;
    /**
     * The language requested for shaping, as passed in. If `null`, the default language for the
     * script was used.
     * @type {string}
     */

    this.language = language || null;
    /**
     * The direction requested for shaping, as passed in (either ltr or rtl).
     * If `null`, the default direction of the script is used.
     * @type {string}
     */

    this.direction = direction$$1 || direction(script);
    /**
     * The features requested during shaping. This is a combination of user
     * specified features and features chosen by the shaper.
     * @type {object}
     */

    this.features = {}; // Convert features to an object

    if (Array.isArray(features)) {
      for (var _iterator = features, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
        var _ref;

        if (_isArray) {
          if (_i >= _iterator.length) break;
          _ref = _iterator[_i++];
        } else {
          _i = _iterator.next();
          if (_i.done) break;
          _ref = _i.value;
        }

        var tag = _ref;
        this.features[tag] = true;
      }
    } else if (typeof features === 'object') {
      this.features = features;
    }
  }
  /**
   * The total advance width of the run.
   * @type {number}
   */


  _createClass(GlyphRun, [{
    key: "advanceWidth",
    get: function get() {
      var width = 0;

      for (var _iterator2 = this.positions, _isArray2 = Array.isArray(_iterator2), _i2 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
        var _ref2;

        if (_isArray2) {
          if (_i2 >= _iterator2.length) break;
          _ref2 = _iterator2[_i2++];
        } else {
          _i2 = _iterator2.next();
          if (_i2.done) break;
          _ref2 = _i2.value;
        }

        var position = _ref2;
        width += position.xAdvance;
      }

      return width;
    }
    /**
     * The total advance height of the run.
     * @type {number}
     */

  }, {
    key: "advanceHeight",
    get: function get() {
      var height = 0;

      for (var _iterator3 = this.positions, _isArray3 = Array.isArray(_iterator3), _i3 = 0, _iterator3 = _isArray3 ? _iterator3 : _iterator3[Symbol.iterator]();;) {
        var _ref3;

        if (_isArray3) {
          if (_i3 >= _iterator3.length) break;
          _ref3 = _iterator3[_i3++];
        } else {
          _i3 = _iterator3.next();
          if (_i3.done) break;
          _ref3 = _i3.value;
        }

        var position = _ref3;
        height += position.yAdvance;
      }

      return height;
    }
    /**
     * The bounding box containing all glyphs in the run.
     * @type {BBox}
     */

  }, {
    key: "bbox",
    get: function get() {
      var bbox = new BBox();
      var x = 0;
      var y = 0;

      for (var index = 0; index < this.glyphs.length; index++) {
        var glyph = this.glyphs[index];
        var p = this.positions[index];
        var b = glyph.bbox;
        bbox.addPoint(b.minX + x + p.xOffset, b.minY + y + p.yOffset);
        bbox.addPoint(b.maxX + x + p.xOffset, b.maxY + y + p.yOffset);
        x += p.xAdvance;
        y += p.yAdvance;
      }

      return bbox;
    }
  }]);

  return GlyphRun;
}();

/**
 * Represents positioning information for a glyph in a GlyphRun.
 */
var GlyphPosition = function GlyphPosition(xAdvance, yAdvance, xOffset, yOffset) {
  if (xAdvance === void 0) {
    xAdvance = 0;
  }

  if (yAdvance === void 0) {
    yAdvance = 0;
  }

  if (xOffset === void 0) {
    xOffset = 0;
  }

  if (yOffset === void 0) {
    yOffset = 0;
  }

  /**
   * The amount to move the virtual pen in the X direction after rendering this glyph.
   * @type {number}
   */
  this.xAdvance = xAdvance;
  /**
   * The amount to move the virtual pen in the Y direction after rendering this glyph.
   * @type {number}
   */

  this.yAdvance = yAdvance;
  /**
   * The offset from the pen position in the X direction at which to render this glyph.
   * @type {number}
   */

  this.xOffset = xOffset;
  /**
   * The offset from the pen position in the Y direction at which to render this glyph.
   * @type {number}
   */

  this.yOffset = yOffset;
};

// see https://developer.apple.com/fonts/TrueType-Reference-Manual/RM09/AppendixF.html
// and /System/Library/Frameworks/CoreText.framework/Versions/A/Headers/SFNTLayoutTypes.h on a Mac
var features = {
  allTypographicFeatures: {
    code: 0,
    exclusive: false,
    allTypeFeatures: 0
  },
  ligatures: {
    code: 1,
    exclusive: false,
    requiredLigatures: 0,
    commonLigatures: 2,
    rareLigatures: 4,
    // logos: 6
    rebusPictures: 8,
    diphthongLigatures: 10,
    squaredLigatures: 12,
    abbrevSquaredLigatures: 14,
    symbolLigatures: 16,
    contextualLigatures: 18,
    historicalLigatures: 20
  },
  cursiveConnection: {
    code: 2,
    exclusive: true,
    unconnected: 0,
    partiallyConnected: 1,
    cursive: 2
  },
  letterCase: {
    code: 3,
    exclusive: true
  },
  // upperAndLowerCase: 0          # deprecated
  // allCaps: 1                    # deprecated
  // allLowerCase: 2               # deprecated
  // smallCaps: 3                  # deprecated
  // initialCaps: 4                # deprecated
  // initialCapsAndSmallCaps: 5    # deprecated
  verticalSubstitution: {
    code: 4,
    exclusive: false,
    substituteVerticalForms: 0
  },
  linguisticRearrangement: {
    code: 5,
    exclusive: false,
    linguisticRearrangement: 0
  },
  numberSpacing: {
    code: 6,
    exclusive: true,
    monospacedNumbers: 0,
    proportionalNumbers: 1,
    thirdWidthNumbers: 2,
    quarterWidthNumbers: 3
  },
  smartSwash: {
    code: 8,
    exclusive: false,
    wordInitialSwashes: 0,
    wordFinalSwashes: 2,
    // lineInitialSwashes: 4
    // lineFinalSwashes: 6
    nonFinalSwashes: 8
  },
  diacritics: {
    code: 9,
    exclusive: true,
    showDiacritics: 0,
    hideDiacritics: 1,
    decomposeDiacritics: 2
  },
  verticalPosition: {
    code: 10,
    exclusive: true,
    normalPosition: 0,
    superiors: 1,
    inferiors: 2,
    ordinals: 3,
    scientificInferiors: 4
  },
  fractions: {
    code: 11,
    exclusive: true,
    noFractions: 0,
    verticalFractions: 1,
    diagonalFractions: 2
  },
  overlappingCharacters: {
    code: 13,
    exclusive: false,
    preventOverlap: 0
  },
  typographicExtras: {
    code: 14,
    exclusive: false,
    // hyphensToEmDash: 0
    // hyphenToEnDash: 2
    slashedZero: 4
  },
  // formInterrobang: 6
  // smartQuotes: 8
  // periodsToEllipsis: 10
  mathematicalExtras: {
    code: 15,
    exclusive: false,
    // hyphenToMinus: 0
    // asteristoMultiply: 2
    // slashToDivide: 4
    // inequalityLigatures: 6
    // exponents: 8
    mathematicalGreek: 10
  },
  ornamentSets: {
    code: 16,
    exclusive: true,
    noOrnaments: 0,
    dingbats: 1,
    piCharacters: 2,
    fleurons: 3,
    decorativeBorders: 4,
    internationalSymbols: 5,
    mathSymbols: 6
  },
  characterAlternatives: {
    code: 17,
    exclusive: true,
    noAlternates: 0
  },
  // user defined options
  designComplexity: {
    code: 18,
    exclusive: true,
    designLevel1: 0,
    designLevel2: 1,
    designLevel3: 2,
    designLevel4: 3,
    designLevel5: 4
  },
  styleOptions: {
    code: 19,
    exclusive: true,
    noStyleOptions: 0,
    displayText: 1,
    engravedText: 2,
    illuminatedCaps: 3,
    titlingCaps: 4,
    tallCaps: 5
  },
  characterShape: {
    code: 20,
    exclusive: true,
    traditionalCharacters: 0,
    simplifiedCharacters: 1,
    JIS1978Characters: 2,
    JIS1983Characters: 3,
    JIS1990Characters: 4,
    traditionalAltOne: 5,
    traditionalAltTwo: 6,
    traditionalAltThree: 7,
    traditionalAltFour: 8,
    traditionalAltFive: 9,
    expertCharacters: 10,
    JIS2004Characters: 11,
    hojoCharacters: 12,
    NLCCharacters: 13,
    traditionalNamesCharacters: 14
  },
  numberCase: {
    code: 21,
    exclusive: true,
    lowerCaseNumbers: 0,
    upperCaseNumbers: 1
  },
  textSpacing: {
    code: 22,
    exclusive: true,
    proportionalText: 0,
    monospacedText: 1,
    halfWidthText: 2,
    thirdWidthText: 3,
    quarterWidthText: 4,
    altProportionalText: 5,
    altHalfWidthText: 6
  },
  transliteration: {
    code: 23,
    exclusive: true,
    noTransliteration: 0
  },
  // hanjaToHangul: 1
  // hiraganaToKatakana: 2
  // katakanaToHiragana: 3
  // kanaToRomanization: 4
  // romanizationToHiragana: 5
  // romanizationToKatakana: 6
  // hanjaToHangulAltOne: 7
  // hanjaToHangulAltTwo: 8
  // hanjaToHangulAltThree: 9
  annotation: {
    code: 24,
    exclusive: true,
    noAnnotation: 0,
    boxAnnotation: 1,
    roundedBoxAnnotation: 2,
    circleAnnotation: 3,
    invertedCircleAnnotation: 4,
    parenthesisAnnotation: 5,
    periodAnnotation: 6,
    romanNumeralAnnotation: 7,
    diamondAnnotation: 8,
    invertedBoxAnnotation: 9,
    invertedRoundedBoxAnnotation: 10
  },
  kanaSpacing: {
    code: 25,
    exclusive: true,
    fullWidthKana: 0,
    proportionalKana: 1
  },
  ideographicSpacing: {
    code: 26,
    exclusive: true,
    fullWidthIdeographs: 0,
    proportionalIdeographs: 1,
    halfWidthIdeographs: 2
  },
  unicodeDecomposition: {
    code: 27,
    exclusive: false,
    canonicalComposition: 0,
    compatibilityComposition: 2,
    transcodingComposition: 4
  },
  rubyKana: {
    code: 28,
    exclusive: false,
    // noRubyKana: 0     # deprecated - use rubyKanaOff instead
    // rubyKana: 1     # deprecated - use rubyKanaOn instead
    rubyKana: 2
  },
  CJKSymbolAlternatives: {
    code: 29,
    exclusive: true,
    noCJKSymbolAlternatives: 0,
    CJKSymbolAltOne: 1,
    CJKSymbolAltTwo: 2,
    CJKSymbolAltThree: 3,
    CJKSymbolAltFour: 4,
    CJKSymbolAltFive: 5
  },
  ideographicAlternatives: {
    code: 30,
    exclusive: true,
    noIdeographicAlternatives: 0,
    ideographicAltOne: 1,
    ideographicAltTwo: 2,
    ideographicAltThree: 3,
    ideographicAltFour: 4,
    ideographicAltFive: 5
  },
  CJKVerticalRomanPlacement: {
    code: 31,
    exclusive: true,
    CJKVerticalRomanCentered: 0,
    CJKVerticalRomanHBaseline: 1
  },
  italicCJKRoman: {
    code: 32,
    exclusive: false,
    // noCJKItalicRoman: 0     # deprecated - use CJKItalicRomanOff instead
    // CJKItalicRoman: 1     # deprecated - use CJKItalicRomanOn instead
    CJKItalicRoman: 2
  },
  caseSensitiveLayout: {
    code: 33,
    exclusive: false,
    caseSensitiveLayout: 0,
    caseSensitiveSpacing: 2
  },
  alternateKana: {
    code: 34,
    exclusive: false,
    alternateHorizKana: 0,
    alternateVertKana: 2
  },
  stylisticAlternatives: {
    code: 35,
    exclusive: false,
    noStylisticAlternates: 0,
    stylisticAltOne: 2,
    stylisticAltTwo: 4,
    stylisticAltThree: 6,
    stylisticAltFour: 8,
    stylisticAltFive: 10,
    stylisticAltSix: 12,
    stylisticAltSeven: 14,
    stylisticAltEight: 16,
    stylisticAltNine: 18,
    stylisticAltTen: 20,
    stylisticAltEleven: 22,
    stylisticAltTwelve: 24,
    stylisticAltThirteen: 26,
    stylisticAltFourteen: 28,
    stylisticAltFifteen: 30,
    stylisticAltSixteen: 32,
    stylisticAltSeventeen: 34,
    stylisticAltEighteen: 36,
    stylisticAltNineteen: 38,
    stylisticAltTwenty: 40
  },
  contextualAlternates: {
    code: 36,
    exclusive: false,
    contextualAlternates: 0,
    swashAlternates: 2,
    contextualSwashAlternates: 4
  },
  lowerCase: {
    code: 37,
    exclusive: true,
    defaultLowerCase: 0,
    lowerCaseSmallCaps: 1,
    lowerCasePetiteCaps: 2
  },
  upperCase: {
    code: 38,
    exclusive: true,
    defaultUpperCase: 0,
    upperCaseSmallCaps: 1,
    upperCasePetiteCaps: 2
  },
  languageTag: {
    // indices into ltag table
    code: 39,
    exclusive: true
  },
  CJKRomanSpacing: {
    code: 103,
    exclusive: true,
    halfWidthCJKRoman: 0,
    proportionalCJKRoman: 1,
    defaultCJKRoman: 2,
    fullWidthCJKRoman: 3
  }
};

var feature = function feature(name, selector) {
  return [features[name].code, features[name][selector]];
};

var OTMapping = {
  rlig: feature('ligatures', 'requiredLigatures'),
  clig: feature('ligatures', 'contextualLigatures'),
  dlig: feature('ligatures', 'rareLigatures'),
  hlig: feature('ligatures', 'historicalLigatures'),
  liga: feature('ligatures', 'commonLigatures'),
  hist: feature('ligatures', 'historicalLigatures'),
  // ??
  smcp: feature('lowerCase', 'lowerCaseSmallCaps'),
  pcap: feature('lowerCase', 'lowerCasePetiteCaps'),
  frac: feature('fractions', 'diagonalFractions'),
  dnom: feature('fractions', 'diagonalFractions'),
  // ??
  numr: feature('fractions', 'diagonalFractions'),
  // ??
  afrc: feature('fractions', 'verticalFractions'),
  // aalt
  // abvf, abvm, abvs, akhn, blwf, blwm, blws, cfar, cjct, cpsp, falt, isol, jalt, ljmo, mset?
  // ltra, ltrm, nukt, pref, pres, pstf, psts, rand, rkrf, rphf, rtla, rtlm, size, tjmo, tnum?
  // unic, vatu, vhal, vjmo, vpal, vrt2
  // dist -> trak table?
  // kern, vkrn -> kern table
  // lfbd + opbd + rtbd -> opbd table?
  // mark, mkmk -> acnt table?
  // locl -> languageTag + ltag table
  case: feature('caseSensitiveLayout', 'caseSensitiveLayout'),
  // also caseSensitiveSpacing
  ccmp: feature('unicodeDecomposition', 'canonicalComposition'),
  // compatibilityComposition?
  cpct: feature('CJKVerticalRomanPlacement', 'CJKVerticalRomanCentered'),
  // guess..., probably not given below
  valt: feature('CJKVerticalRomanPlacement', 'CJKVerticalRomanCentered'),
  swsh: feature('contextualAlternates', 'swashAlternates'),
  cswh: feature('contextualAlternates', 'contextualSwashAlternates'),
  curs: feature('cursiveConnection', 'cursive'),
  // ??
  c2pc: feature('upperCase', 'upperCasePetiteCaps'),
  c2sc: feature('upperCase', 'upperCaseSmallCaps'),
  init: feature('smartSwash', 'wordInitialSwashes'),
  // ??
  fin2: feature('smartSwash', 'wordFinalSwashes'),
  // ??
  medi: feature('smartSwash', 'nonFinalSwashes'),
  // ??
  med2: feature('smartSwash', 'nonFinalSwashes'),
  // ??
  fin3: feature('smartSwash', 'wordFinalSwashes'),
  // ??
  fina: feature('smartSwash', 'wordFinalSwashes'),
  // ??
  pkna: feature('kanaSpacing', 'proportionalKana'),
  half: feature('textSpacing', 'halfWidthText'),
  // also HalfWidthCJKRoman, HalfWidthIdeographs?
  halt: feature('textSpacing', 'altHalfWidthText'),
  hkna: feature('alternateKana', 'alternateHorizKana'),
  vkna: feature('alternateKana', 'alternateVertKana'),
  // hngl: feature 'transliteration', 'hanjaToHangulSelector' # deprecated
  ital: feature('italicCJKRoman', 'CJKItalicRoman'),
  lnum: feature('numberCase', 'upperCaseNumbers'),
  onum: feature('numberCase', 'lowerCaseNumbers'),
  mgrk: feature('mathematicalExtras', 'mathematicalGreek'),
  // nalt: not enough info. what type of annotation?
  // ornm: ditto, which ornament style?
  calt: feature('contextualAlternates', 'contextualAlternates'),
  // or more?
  vrt2: feature('verticalSubstitution', 'substituteVerticalForms'),
  // oh... below?
  vert: feature('verticalSubstitution', 'substituteVerticalForms'),
  tnum: feature('numberSpacing', 'monospacedNumbers'),
  pnum: feature('numberSpacing', 'proportionalNumbers'),
  sups: feature('verticalPosition', 'superiors'),
  subs: feature('verticalPosition', 'inferiors'),
  ordn: feature('verticalPosition', 'ordinals'),
  pwid: feature('textSpacing', 'proportionalText'),
  hwid: feature('textSpacing', 'halfWidthText'),
  qwid: feature('textSpacing', 'quarterWidthText'),
  // also QuarterWidthNumbers?
  twid: feature('textSpacing', 'thirdWidthText'),
  // also ThirdWidthNumbers?
  fwid: feature('textSpacing', 'proportionalText'),
  //??
  palt: feature('textSpacing', 'altProportionalText'),
  trad: feature('characterShape', 'traditionalCharacters'),
  smpl: feature('characterShape', 'simplifiedCharacters'),
  jp78: feature('characterShape', 'JIS1978Characters'),
  jp83: feature('characterShape', 'JIS1983Characters'),
  jp90: feature('characterShape', 'JIS1990Characters'),
  jp04: feature('characterShape', 'JIS2004Characters'),
  expt: feature('characterShape', 'expertCharacters'),
  hojo: feature('characterShape', 'hojoCharacters'),
  nlck: feature('characterShape', 'NLCCharacters'),
  tnam: feature('characterShape', 'traditionalNamesCharacters'),
  ruby: feature('rubyKana', 'rubyKana'),
  titl: feature('styleOptions', 'titlingCaps'),
  zero: feature('typographicExtras', 'slashedZero'),
  ss01: feature('stylisticAlternatives', 'stylisticAltOne'),
  ss02: feature('stylisticAlternatives', 'stylisticAltTwo'),
  ss03: feature('stylisticAlternatives', 'stylisticAltThree'),
  ss04: feature('stylisticAlternatives', 'stylisticAltFour'),
  ss05: feature('stylisticAlternatives', 'stylisticAltFive'),
  ss06: feature('stylisticAlternatives', 'stylisticAltSix'),
  ss07: feature('stylisticAlternatives', 'stylisticAltSeven'),
  ss08: feature('stylisticAlternatives', 'stylisticAltEight'),
  ss09: feature('stylisticAlternatives', 'stylisticAltNine'),
  ss10: feature('stylisticAlternatives', 'stylisticAltTen'),
  ss11: feature('stylisticAlternatives', 'stylisticAltEleven'),
  ss12: feature('stylisticAlternatives', 'stylisticAltTwelve'),
  ss13: feature('stylisticAlternatives', 'stylisticAltThirteen'),
  ss14: feature('stylisticAlternatives', 'stylisticAltFourteen'),
  ss15: feature('stylisticAlternatives', 'stylisticAltFifteen'),
  ss16: feature('stylisticAlternatives', 'stylisticAltSixteen'),
  ss17: feature('stylisticAlternatives', 'stylisticAltSeventeen'),
  ss18: feature('stylisticAlternatives', 'stylisticAltEighteen'),
  ss19: feature('stylisticAlternatives', 'stylisticAltNineteen'),
  ss20: feature('stylisticAlternatives', 'stylisticAltTwenty')
}; // salt: feature 'stylisticAlternatives', 'stylisticAltOne' # hmm, which one to choose
// Add cv01-cv99 features

for (var i$2 = 1; i$2 <= 99; i$2++) {
  OTMapping["cv" + ("00" + i$2).slice(-2)] = [features.characterAlternatives.code, i$2];
} // create inverse mapping


var AATMapping = {};

for (var ot in OTMapping) {
  var aat = OTMapping[ot];

  if (AATMapping[aat[0]] == null) {
    AATMapping[aat[0]] = {};
  }

  AATMapping[aat[0]][aat[1]] = ot;
} // Maps an array of OpenType features to AAT features
// in the form of {featureType:{featureSetting:true}}


function mapOTToAAT(features) {
  var res = {};

  for (var k in features) {
    var r = void 0;

    if (r = OTMapping[k]) {
      if (res[r[0]] == null) {
        res[r[0]] = {};
      }

      res[r[0]][r[1]] = features[k];
    }
  }

  return res;
} // Maps strings in a [featureType, featureSetting]
// to their equivalent number codes

function mapFeatureStrings(f) {
  var type = f[0],
      setting = f[1];

  if (isNaN(type)) {
    var typeCode = features[type] && features[type].code;
  } else {
    var typeCode = type;
  }

  if (isNaN(setting)) {
    var settingCode = features[type] && features[type][setting];
  } else {
    var settingCode = setting;
  }

  return [typeCode, settingCode];
} // Maps AAT features to an array of OpenType features
// Supports both arrays in the form of [[featureType, featureSetting]]
// and objects in the form of {featureType:{featureSetting:true}}
// featureTypes and featureSettings can be either strings or number codes


function mapAATToOT(features) {
  var res = {};

  if (Array.isArray(features)) {
    for (var k = 0; k < features.length; k++) {
      var r = void 0;
      var f = mapFeatureStrings(features[k]);

      if (r = AATMapping[f[0]] && AATMapping[f[0]][f[1]]) {
        res[r] = true;
      }
    }
  } else if (typeof features === 'object') {
    for (var type in features) {
      var _feature = features[type];

      for (var setting in _feature) {
        var _r = void 0;

        var _f = mapFeatureStrings([type, setting]);

        if (_feature[setting] && (_r = AATMapping[_f[0]] && AATMapping[_f[0]][_f[1]])) {
          res[_r] = true;
        }
      }
    }
  }

  return Object.keys(res);
}

var _class$1;
var AATLookupTable = (_class$1 =
/*#__PURE__*/
function () {
  function AATLookupTable(table) {
    this.table = table;
  }

  var _proto = AATLookupTable.prototype;

  _proto.lookup = function lookup(glyph) {
    switch (this.table.version) {
      case 0:
        // simple array format
        return this.table.values.getItem(glyph);

      case 2: // segment format

      case 4:
        {
          var min = 0;
          var max = this.table.binarySearchHeader.nUnits - 1;

          while (min <= max) {
            var mid = min + max >> 1;
            var seg = this.table.segments[mid]; // special end of search value

            if (seg.firstGlyph === 0xffff) {
              return null;
            }

            if (glyph < seg.firstGlyph) {
              max = mid - 1;
            } else if (glyph > seg.lastGlyph) {
              min = mid + 1;
            } else {
              if (this.table.version === 2) {
                return seg.value;
              } else {
                return seg.values[glyph - seg.firstGlyph];
              }
            }
          }

          return null;
        }

      case 6:
        {
          // lookup single
          var _min = 0;

          var _max = this.table.binarySearchHeader.nUnits - 1;

          while (_min <= _max) {
            var mid = _min + _max >> 1;
            var seg = this.table.segments[mid]; // special end of search value

            if (seg.glyph === 0xffff) {
              return null;
            }

            if (glyph < seg.glyph) {
              _max = mid - 1;
            } else if (glyph > seg.glyph) {
              _min = mid + 1;
            } else {
              return seg.value;
            }
          }

          return null;
        }

      case 8:
        // lookup trimmed
        return this.table.values[glyph - this.table.firstGlyph];

      default:
        throw new Error("Unknown lookup table format: " + this.table.version);
    }
  };

  _proto.glyphsForValue = function glyphsForValue(classValue) {
    var res = [];

    switch (this.table.version) {
      case 2: // segment format

      case 4:
        {
          for (var _iterator = this.table.segments, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
            var _ref;

            if (_isArray) {
              if (_i >= _iterator.length) break;
              _ref = _iterator[_i++];
            } else {
              _i = _iterator.next();
              if (_i.done) break;
              _ref = _i.value;
            }

            var segment = _ref;

            if (this.table.version === 2 && segment.value === classValue) {
              res.push.apply(res, range(segment.firstGlyph, segment.lastGlyph + 1));
            } else {
              for (var index = 0; index < segment.values.length; index++) {
                if (segment.values[index] === classValue) {
                  res.push(segment.firstGlyph + index);
                }
              }
            }
          }

          break;
        }

      case 6:
        {
          // lookup single
          for (var _iterator2 = this.table.segments, _isArray2 = Array.isArray(_iterator2), _i2 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
            var _ref2;

            if (_isArray2) {
              if (_i2 >= _iterator2.length) break;
              _ref2 = _iterator2[_i2++];
            } else {
              _i2 = _iterator2.next();
              if (_i2.done) break;
              _ref2 = _i2.value;
            }

            var _segment = _ref2;

            if (_segment.value === classValue) {
              res.push(_segment.glyph);
            }
          }

          break;
        }

      case 8:
        {
          // lookup trimmed
          for (var i = 0; i < this.table.values.length; i++) {
            if (this.table.values[i] === classValue) {
              res.push(this.table.firstGlyph + i);
            }
          }

          break;
        }

      default:
        throw new Error("Unknown lookup table format: " + this.table.version);
    }

    return res;
  };

  return AATLookupTable;
}(), (_applyDecoratedDescriptor(_class$1.prototype, "glyphsForValue", [cache], Object.getOwnPropertyDescriptor(_class$1.prototype, "glyphsForValue"), _class$1.prototype)), _class$1);

var START_OF_TEXT_STATE = 0;
var END_OF_TEXT_CLASS = 0;
var OUT_OF_BOUNDS_CLASS = 1;
var DELETED_GLYPH_CLASS = 2;
var DONT_ADVANCE = 0x4000;

var AATStateMachine =
/*#__PURE__*/
function () {
  function AATStateMachine(stateTable) {
    this.stateTable = stateTable;
    this.lookupTable = new AATLookupTable(stateTable.classTable);
  }

  var _proto = AATStateMachine.prototype;

  _proto.process = function process(glyphs, reverse, processEntry) {
    var currentState = START_OF_TEXT_STATE; // START_OF_LINE_STATE is used for kashida glyph insertions sometimes I think?

    var index = reverse ? glyphs.length - 1 : 0;
    var dir = reverse ? -1 : 1;

    while (dir === 1 && index <= glyphs.length || dir === -1 && index >= -1) {
      var glyph = null;
      var classCode = OUT_OF_BOUNDS_CLASS;
      var shouldAdvance = true;

      if (index === glyphs.length || index === -1) {
        classCode = END_OF_TEXT_CLASS;
      } else {
        glyph = glyphs[index];

        if (glyph.id === 0xffff) {
          // deleted glyph
          classCode = DELETED_GLYPH_CLASS;
        } else {
          classCode = this.lookupTable.lookup(glyph.id);

          if (classCode == null) {
            classCode = OUT_OF_BOUNDS_CLASS;
          }
        }
      }

      var row = this.stateTable.stateArray.getItem(currentState);
      var entryIndex = row[classCode];
      var entry = this.stateTable.entryTable.getItem(entryIndex);

      if (classCode !== END_OF_TEXT_CLASS && classCode !== DELETED_GLYPH_CLASS) {
        processEntry(glyph, entry, index);
        shouldAdvance = !(entry.flags & DONT_ADVANCE);
      }

      currentState = entry.newState;

      if (shouldAdvance) {
        index += dir;
      }
    }

    return glyphs;
  }
  /**
   * Performs a depth-first traversal of the glyph strings
   * represented by the state machine.
   */
  ;

  _proto.traverse = function traverse(opts, state, visited) {
    if (state === void 0) {
      state = 0;
    }

    if (visited === void 0) {
      visited = new Set();
    }

    if (visited.has(state)) {
      return;
    }

    visited.add(state);
    var _this$stateTable = this.stateTable,
        nClasses = _this$stateTable.nClasses,
        stateArray = _this$stateTable.stateArray,
        entryTable = _this$stateTable.entryTable;
    var row = stateArray.getItem(state); // Skip predefined classes

    for (var classCode = 4; classCode < nClasses; classCode++) {
      var entryIndex = row[classCode];
      var entry = entryTable.getItem(entryIndex); // Try all glyphs in the class

      for (var _iterator = this.lookupTable.glyphsForValue(classCode), _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
        var _ref;

        if (_isArray) {
          if (_i >= _iterator.length) break;
          _ref = _iterator[_i++];
        } else {
          _i = _iterator.next();
          if (_i.done) break;
          _ref = _i.value;
        }

        var glyph = _ref;

        if (opts.enter) {
          opts.enter(glyph, entry);
        }

        if (entry.newState !== 0) {
          this.traverse(opts, entry.newState, visited);
        }

        if (opts.exit) {
          opts.exit(glyph, entry);
        }
      }
    }
  };

  return AATStateMachine;
}();

var _class$2;

var MARK_FIRST = 0x8000;
var MARK_LAST = 0x2000;
var VERB = 0x000F; // contextual substitution and glyph insertion flag

var SET_MARK = 0x8000; // ligature entry flags

var SET_COMPONENT = 0x8000;
var PERFORM_ACTION = 0x2000; // ligature action masks

var LAST_MASK = 0x80000000;
var STORE_MASK = 0x40000000;
var OFFSET_MASK = 0x3FFFFFFF;
var REVERSE_DIRECTION = 0x400000;
var CURRENT_INSERT_BEFORE = 0x0800;
var MARKED_INSERT_BEFORE = 0x0400;
var CURRENT_INSERT_COUNT = 0x03E0;
var MARKED_INSERT_COUNT = 0x001F;
var AATMorxProcessor = (_class$2 =
/*#__PURE__*/
function () {
  function AATMorxProcessor(font) {
    this.processIndicRearragement = this.processIndicRearragement.bind(this);
    this.processContextualSubstitution = this.processContextualSubstitution.bind(this);
    this.processLigature = this.processLigature.bind(this);
    this.processNoncontextualSubstitutions = this.processNoncontextualSubstitutions.bind(this);
    this.processGlyphInsertion = this.processGlyphInsertion.bind(this);
    this.font = font;
    this.morx = font.morx;
    this.inputCache = null;
  } // Processes an array of glyphs and applies the specified features
  // Features should be in the form of {featureType:{featureSetting:true}}


  var _proto = AATMorxProcessor.prototype;

  _proto.process = function process(glyphs, features) {
    if (features === void 0) {
      features = {};
    }

    for (var _iterator = this.morx.chains, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
      var _ref;

      if (_isArray) {
        if (_i >= _iterator.length) break;
        _ref = _iterator[_i++];
      } else {
        _i = _iterator.next();
        if (_i.done) break;
        _ref = _i.value;
      }

      var chain = _ref;
      var flags = chain.defaultFlags; // enable/disable the requested features

      for (var _iterator2 = chain.features, _isArray2 = Array.isArray(_iterator2), _i2 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
        var _ref2;

        if (_isArray2) {
          if (_i2 >= _iterator2.length) break;
          _ref2 = _iterator2[_i2++];
        } else {
          _i2 = _iterator2.next();
          if (_i2.done) break;
          _ref2 = _i2.value;
        }

        var feature = _ref2;
        var f = void 0;

        if ((f = features[feature.featureType]) && f[feature.featureSetting]) {
          flags &= feature.disableFlags;
          flags |= feature.enableFlags;
        }
      }

      for (var _iterator3 = chain.subtables, _isArray3 = Array.isArray(_iterator3), _i3 = 0, _iterator3 = _isArray3 ? _iterator3 : _iterator3[Symbol.iterator]();;) {
        var _ref3;

        if (_isArray3) {
          if (_i3 >= _iterator3.length) break;
          _ref3 = _iterator3[_i3++];
        } else {
          _i3 = _iterator3.next();
          if (_i3.done) break;
          _ref3 = _i3.value;
        }

        var subtable = _ref3;

        if (subtable.subFeatureFlags & flags) {
          this.processSubtable(subtable, glyphs);
        }
      }
    } // remove deleted glyphs


    var index = glyphs.length - 1;

    while (index >= 0) {
      if (glyphs[index].id === 0xffff) {
        glyphs.splice(index, 1);
      }

      index--;
    }

    return glyphs;
  };

  _proto.processSubtable = function processSubtable(subtable, glyphs) {
    this.subtable = subtable;
    this.glyphs = glyphs;

    if (this.subtable.type === 4) {
      this.processNoncontextualSubstitutions(this.subtable, this.glyphs);
      return;
    }

    this.ligatureStack = [];
    this.markedGlyph = null;
    this.firstGlyph = null;
    this.lastGlyph = null;
    this.markedIndex = null;
    var stateMachine = this.getStateMachine(subtable);
    var process = this.getProcessor();
    var reverse = !!(this.subtable.coverage & REVERSE_DIRECTION);
    return stateMachine.process(this.glyphs, reverse, process);
  };

  _proto.getStateMachine = function getStateMachine(subtable) {
    return new AATStateMachine(subtable.table.stateTable);
  };

  _proto.getProcessor = function getProcessor() {
    switch (this.subtable.type) {
      case 0:
        return this.processIndicRearragement;

      case 1:
        return this.processContextualSubstitution;

      case 2:
        return this.processLigature;

      case 4:
        return this.processNoncontextualSubstitutions;

      case 5:
        return this.processGlyphInsertion;

      default:
        throw new Error("Invalid morx subtable type: " + this.subtable.type);
    }
  };

  _proto.processIndicRearragement = function processIndicRearragement(glyph, entry, index) {
    if (entry.flags & MARK_FIRST) {
      this.firstGlyph = index;
    }

    if (entry.flags & MARK_LAST) {
      this.lastGlyph = index;
    }

    reorderGlyphs(this.glyphs, entry.flags & VERB, this.firstGlyph, this.lastGlyph);
  };

  _proto.processContextualSubstitution = function processContextualSubstitution(glyph, entry, index) {
    var subsitutions = this.subtable.table.substitutionTable.items;

    if (entry.markIndex !== 0xffff) {
      var lookup = subsitutions.getItem(entry.markIndex);
      var lookupTable = new AATLookupTable(lookup);
      glyph = this.glyphs[this.markedGlyph];
      var gid = lookupTable.lookup(glyph.id);

      if (gid) {
        this.glyphs[this.markedGlyph] = this.font.getGlyph(gid, glyph.codePoints);
      }
    }

    if (entry.currentIndex !== 0xffff) {
      var _lookup = subsitutions.getItem(entry.currentIndex);

      var _lookupTable = new AATLookupTable(_lookup);

      glyph = this.glyphs[index];

      var gid = _lookupTable.lookup(glyph.id);

      if (gid) {
        this.glyphs[index] = this.font.getGlyph(gid, glyph.codePoints);
      }
    }

    if (entry.flags & SET_MARK) {
      this.markedGlyph = index;
    }
  };

  _proto.processLigature = function processLigature(glyph, entry, index) {
    if (entry.flags & SET_COMPONENT) {
      this.ligatureStack.push(index);
    }

    if (entry.flags & PERFORM_ACTION) {
      var _this$ligatureStack;

      var actions = this.subtable.table.ligatureActions;
      var components = this.subtable.table.components;
      var ligatureList = this.subtable.table.ligatureList;
      var actionIndex = entry.action;
      var last = false;
      var ligatureIndex = 0;
      var codePoints = [];
      var ligatureGlyphs = [];

      while (!last) {
        var _codePoints;

        var componentGlyph = this.ligatureStack.pop();

        (_codePoints = codePoints).unshift.apply(_codePoints, this.glyphs[componentGlyph].codePoints);

        var action = actions.getItem(actionIndex++);
        last = !!(action & LAST_MASK);
        var store = !!(action & STORE_MASK);
        var offset = (action & OFFSET_MASK) << 2 >> 2; // sign extend 30 to 32 bits

        offset += this.glyphs[componentGlyph].id;
        var component = components.getItem(offset);
        ligatureIndex += component;

        if (last || store) {
          var ligatureEntry = ligatureList.getItem(ligatureIndex);
          this.glyphs[componentGlyph] = this.font.getGlyph(ligatureEntry, codePoints);
          ligatureGlyphs.push(componentGlyph);
          ligatureIndex = 0;
          codePoints = [];
        } else {
          this.glyphs[componentGlyph] = this.font.getGlyph(0xffff);
        }
      } // Put ligature glyph indexes back on the stack


      (_this$ligatureStack = this.ligatureStack).push.apply(_this$ligatureStack, ligatureGlyphs);
    }
  };

  _proto.processNoncontextualSubstitutions = function processNoncontextualSubstitutions(subtable, glyphs, index) {
    var lookupTable = new AATLookupTable(subtable.table.lookupTable);

    for (index = 0; index < glyphs.length; index++) {
      var glyph = glyphs[index];

      if (glyph.id !== 0xffff) {
        var gid = lookupTable.lookup(glyph.id);

        if (gid) {
          // 0 means do nothing
          glyphs[index] = this.font.getGlyph(gid, glyph.codePoints);
        }
      }
    }
  };

  _proto._insertGlyphs = function _insertGlyphs(glyphIndex, insertionActionIndex, count, isBefore) {
    var _this$glyphs;

    var insertions = [];

    while (count--) {
      var gid = this.subtable.table.insertionActions.getItem(insertionActionIndex++);
      insertions.push(this.font.getGlyph(gid));
    }

    if (!isBefore) {
      glyphIndex++;
    }

    (_this$glyphs = this.glyphs).splice.apply(_this$glyphs, [glyphIndex, 0].concat(insertions));
  };

  _proto.processGlyphInsertion = function processGlyphInsertion(glyph, entry, index) {
    if (entry.flags & SET_MARK) {
      this.markedIndex = index;
    }

    if (entry.markedInsertIndex !== 0xffff) {
      var count = (entry.flags & MARKED_INSERT_COUNT) >>> 5;
      var isBefore = !!(entry.flags & MARKED_INSERT_BEFORE);

      this._insertGlyphs(this.markedIndex, entry.markedInsertIndex, count, isBefore);
    }

    if (entry.currentInsertIndex !== 0xffff) {
      var _count = (entry.flags & CURRENT_INSERT_COUNT) >>> 5;

      var _isBefore = !!(entry.flags & CURRENT_INSERT_BEFORE);

      this._insertGlyphs(index, entry.currentInsertIndex, _count, _isBefore);
    }
  };

  _proto.getSupportedFeatures = function getSupportedFeatures() {
    var features = [];

    for (var _iterator4 = this.morx.chains, _isArray4 = Array.isArray(_iterator4), _i4 = 0, _iterator4 = _isArray4 ? _iterator4 : _iterator4[Symbol.iterator]();;) {
      var _ref4;

      if (_isArray4) {
        if (_i4 >= _iterator4.length) break;
        _ref4 = _iterator4[_i4++];
      } else {
        _i4 = _iterator4.next();
        if (_i4.done) break;
        _ref4 = _i4.value;
      }

      var chain = _ref4;

      for (var _iterator5 = chain.features, _isArray5 = Array.isArray(_iterator5), _i5 = 0, _iterator5 = _isArray5 ? _iterator5 : _iterator5[Symbol.iterator]();;) {
        var _ref5;

        if (_isArray5) {
          if (_i5 >= _iterator5.length) break;
          _ref5 = _iterator5[_i5++];
        } else {
          _i5 = _iterator5.next();
          if (_i5.done) break;
          _ref5 = _i5.value;
        }

        var feature = _ref5;
        features.push([feature.featureType, feature.featureSetting]);
      }
    }

    return features;
  };

  _proto.generateInputs = function generateInputs(gid) {
    if (!this.inputCache) {
      this.generateInputCache();
    }

    return this.inputCache[gid] || [];
  };

  _proto.generateInputCache = function generateInputCache() {
    this.inputCache = {};

    for (var _iterator6 = this.morx.chains, _isArray6 = Array.isArray(_iterator6), _i6 = 0, _iterator6 = _isArray6 ? _iterator6 : _iterator6[Symbol.iterator]();;) {
      var _ref6;

      if (_isArray6) {
        if (_i6 >= _iterator6.length) break;
        _ref6 = _iterator6[_i6++];
      } else {
        _i6 = _iterator6.next();
        if (_i6.done) break;
        _ref6 = _i6.value;
      }

      var chain = _ref6;
      var flags = chain.defaultFlags;

      for (var _iterator7 = chain.subtables, _isArray7 = Array.isArray(_iterator7), _i7 = 0, _iterator7 = _isArray7 ? _iterator7 : _iterator7[Symbol.iterator]();;) {
        var _ref7;

        if (_isArray7) {
          if (_i7 >= _iterator7.length) break;
          _ref7 = _iterator7[_i7++];
        } else {
          _i7 = _iterator7.next();
          if (_i7.done) break;
          _ref7 = _i7.value;
        }

        var subtable = _ref7;

        if (subtable.subFeatureFlags & flags) {
          this.generateInputsForSubtable(subtable);
        }
      }
    }
  };

  _proto.generateInputsForSubtable = function generateInputsForSubtable(subtable) {
    var _this = this;

    // Currently, only supporting ligature subtables.
    if (subtable.type !== 2) {
      return;
    }

    var reverse = !!(subtable.coverage & REVERSE_DIRECTION);

    if (reverse) {
      throw new Error('Reverse subtable, not supported.');
    }

    this.subtable = subtable;
    this.ligatureStack = [];
    var stateMachine = this.getStateMachine(subtable);
    var process = this.getProcessor();
    var input = [];
    var stack = [];
    this.glyphs = [];
    stateMachine.traverse({
      enter: function enter(glyph, entry) {
        var glyphs = _this.glyphs;
        stack.push({
          glyphs: glyphs.slice(),
          ligatureStack: _this.ligatureStack.slice()
        }); // Add glyph to input and glyphs to process.

        var g = _this.font.getGlyph(glyph);

        input.push(g);
        glyphs.push(input[input.length - 1]); // Process ligature substitution

        process(glyphs[glyphs.length - 1], entry, glyphs.length - 1); // Add input to result if only one matching (non-deleted) glyph remains.

        var count = 0;
        var found = 0;

        for (var i = 0; i < glyphs.length && count <= 1; i++) {
          if (glyphs[i].id !== 0xffff) {
            count++;
            found = glyphs[i].id;
          }
        }

        if (count === 1) {
          var result = input.map(function (g) {
            return g.id;
          });
          var _cache = _this.inputCache[found];

          if (_cache) {
            _cache.push(result);
          } else {
            _this.inputCache[found] = [result];
          }
        }
      },
      exit: function exit() {
        var _stack$pop = stack.pop();

        _this.glyphs = _stack$pop.glyphs;
        _this.ligatureStack = _stack$pop.ligatureStack;
        input.pop();
      }
    });
  };

  return AATMorxProcessor;
}(), (_applyDecoratedDescriptor(_class$2.prototype, "getStateMachine", [cache], Object.getOwnPropertyDescriptor(_class$2.prototype, "getStateMachine"), _class$2.prototype)), _class$2);
// reverse the glyphs inside those ranges if specified
// ranges are in [offset, length] format

function swap$1(glyphs, rangeA, rangeB, reverseA, reverseB) {
  if (reverseA === void 0) {
    reverseA = false;
  }

  if (reverseB === void 0) {
    reverseB = false;
  }

  var end = glyphs.splice(rangeB[0] - (rangeB[1] - 1), rangeB[1]);

  if (reverseB) {
    end.reverse();
  }

  var start = glyphs.splice.apply(glyphs, [rangeA[0], rangeA[1]].concat(end));

  if (reverseA) {
    start.reverse();
  }

  glyphs.splice.apply(glyphs, [rangeB[0] - (rangeA[1] - 1), 0].concat(start));
  return glyphs;
}

function reorderGlyphs(glyphs, verb, firstGlyph, lastGlyph) {

  switch (verb) {
    case 0:
      // no change
      return glyphs;

    case 1:
      // Ax => xA
      return swap$1(glyphs, [firstGlyph, 1], [lastGlyph, 0]);

    case 2:
      // xD => Dx
      return swap$1(glyphs, [firstGlyph, 0], [lastGlyph, 1]);

    case 3:
      // AxD => DxA
      return swap$1(glyphs, [firstGlyph, 1], [lastGlyph, 1]);

    case 4:
      // ABx => xAB
      return swap$1(glyphs, [firstGlyph, 2], [lastGlyph, 0]);

    case 5:
      // ABx => xBA
      return swap$1(glyphs, [firstGlyph, 2], [lastGlyph, 0], true, false);

    case 6:
      // xCD => CDx
      return swap$1(glyphs, [firstGlyph, 0], [lastGlyph, 2]);

    case 7:
      // xCD => DCx
      return swap$1(glyphs, [firstGlyph, 0], [lastGlyph, 2], false, true);

    case 8:
      // AxCD => CDxA
      return swap$1(glyphs, [firstGlyph, 1], [lastGlyph, 2]);

    case 9:
      // AxCD => DCxA
      return swap$1(glyphs, [firstGlyph, 1], [lastGlyph, 2], false, true);

    case 10:
      // ABxD => DxAB
      return swap$1(glyphs, [firstGlyph, 2], [lastGlyph, 1]);

    case 11:
      // ABxD => DxBA
      return swap$1(glyphs, [firstGlyph, 2], [lastGlyph, 1], true, false);

    case 12:
      // ABxCD => CDxAB
      return swap$1(glyphs, [firstGlyph, 2], [lastGlyph, 2]);

    case 13:
      // ABxCD => CDxBA
      return swap$1(glyphs, [firstGlyph, 2], [lastGlyph, 2], true, false);

    case 14:
      // ABxCD => DCxAB
      return swap$1(glyphs, [firstGlyph, 2], [lastGlyph, 2], false, true);

    case 15:
      // ABxCD => DCxBA
      return swap$1(glyphs, [firstGlyph, 2], [lastGlyph, 2], true, true);

    default:
      throw new Error("Unknown verb: " + verb);
  }
}

var AATLayoutEngine =
/*#__PURE__*/
function () {
  function AATLayoutEngine(font) {
    this.font = font;
    this.morxProcessor = new AATMorxProcessor(font);
    this.fallbackPosition = false;
  }

  var _proto = AATLayoutEngine.prototype;

  _proto.substitute = function substitute(glyphRun) {
    // AAT expects the glyphs to be in visual order prior to morx processing,
    // so reverse the glyphs if the script is right-to-left.
    if (glyphRun.direction === 'rtl') {
      glyphRun.glyphs.reverse();
    }

    this.morxProcessor.process(glyphRun.glyphs, mapOTToAAT(glyphRun.features));
  };

  _proto.getAvailableFeatures = function getAvailableFeatures(script, language) {
    return mapAATToOT(this.morxProcessor.getSupportedFeatures());
  };

  _proto.stringsForGlyph = function stringsForGlyph(gid) {
    var glyphStrings = this.morxProcessor.generateInputs(gid);
    var result = new Set();

    for (var _iterator = glyphStrings, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
      var _ref;

      if (_isArray) {
        if (_i >= _iterator.length) break;
        _ref = _iterator[_i++];
      } else {
        _i = _iterator.next();
        if (_i.done) break;
        _ref = _i.value;
      }

      var glyphs = _ref;

      this._addStrings(glyphs, 0, result, '');
    }

    return result;
  };

  _proto._addStrings = function _addStrings(glyphs, index, strings, string) {
    var codePoints = this.font._cmapProcessor.codePointsForGlyph(glyphs[index]);

    for (var _iterator2 = codePoints, _isArray2 = Array.isArray(_iterator2), _i2 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
      var _ref2;

      if (_isArray2) {
        if (_i2 >= _iterator2.length) break;
        _ref2 = _iterator2[_i2++];
      } else {
        _i2 = _iterator2.next();
        if (_i2.done) break;
        _ref2 = _i2.value;
      }

      var codePoint = _ref2;
      var s = string + String.fromCodePoint(codePoint);

      if (index < glyphs.length - 1) {
        this._addStrings(glyphs, index + 1, strings, s);
      } else {
        strings.add(s);
      }
    }
  };

  return AATLayoutEngine;
}();

/**
 * ShapingPlans are used by the OpenType shapers to store which
 * features should by applied, and in what order to apply them.
 * The features are applied in groups called stages. A feature
 * can be applied globally to all glyphs, or locally to only
 * specific glyphs.
 *
 * @private
 */

var ShapingPlan =
/*#__PURE__*/
function () {
  function ShapingPlan(font, script, direction$$1) {
    this.font = font;
    this.script = script;
    this.direction = direction$$1;
    this.stages = [];
    this.globalFeatures = {};
    this.allFeatures = {};
  }
  /**
   * Adds the given features to the last stage.
   * Ignores features that have already been applied.
   */


  var _proto = ShapingPlan.prototype;

  _proto._addFeatures = function _addFeatures(features, global) {
    var stageIndex = this.stages.length - 1;
    var stage = this.stages[stageIndex];

    for (var _iterator = features, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
      var _ref;

      if (_isArray) {
        if (_i >= _iterator.length) break;
        _ref = _iterator[_i++];
      } else {
        _i = _iterator.next();
        if (_i.done) break;
        _ref = _i.value;
      }

      var feature = _ref;

      if (this.allFeatures[feature] == null) {
        stage.push(feature);
        this.allFeatures[feature] = stageIndex;

        if (global) {
          this.globalFeatures[feature] = true;
        }
      }
    }
  }
  /**
   * Add features to the last stage
   */
  ;

  _proto.add = function add(arg, global) {
    if (global === void 0) {
      global = true;
    }

    if (this.stages.length === 0) {
      this.stages.push([]);
    }

    if (typeof arg === 'string') {
      arg = [arg];
    }

    if (Array.isArray(arg)) {
      this._addFeatures(arg, global);
    } else if (typeof arg === 'object') {
      this._addFeatures(arg.global || [], true);

      this._addFeatures(arg.local || [], false);
    } else {
      throw new Error("Unsupported argument to ShapingPlan#add");
    }
  }
  /**
   * Add a new stage
   */
  ;

  _proto.addStage = function addStage(arg, global) {
    if (typeof arg === 'function') {
      this.stages.push(arg, []);
    } else {
      this.stages.push([]);
      this.add(arg, global);
    }
  };

  _proto.setFeatureOverrides = function setFeatureOverrides(features) {
    if (Array.isArray(features)) {
      this.add(features);
    } else if (typeof features === 'object') {
      for (var tag in features) {
        if (features[tag]) {
          this.add(tag);
        } else if (this.allFeatures[tag] != null) {
          var stage = this.stages[this.allFeatures[tag]];
          stage.splice(stage.indexOf(tag), 1);
          delete this.allFeatures[tag];
          delete this.globalFeatures[tag];
        }
      }
    }
  }
  /**
   * Assigns the global features to the given glyphs
   */
  ;

  _proto.assignGlobalFeatures = function assignGlobalFeatures(glyphs) {
    for (var _iterator2 = glyphs, _isArray2 = Array.isArray(_iterator2), _i2 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
      var _ref2;

      if (_isArray2) {
        if (_i2 >= _iterator2.length) break;
        _ref2 = _iterator2[_i2++];
      } else {
        _i2 = _iterator2.next();
        if (_i2.done) break;
        _ref2 = _i2.value;
      }

      var glyph = _ref2;

      for (var feature in this.globalFeatures) {
        glyph.features[feature] = true;
      }
    }
  }
  /**
   * Executes the planned stages using the given OTProcessor
   */
  ;

  _proto.process = function process(processor, glyphs, positions) {
    for (var _iterator3 = this.stages, _isArray3 = Array.isArray(_iterator3), _i3 = 0, _iterator3 = _isArray3 ? _iterator3 : _iterator3[Symbol.iterator]();;) {
      var _ref3;

      if (_isArray3) {
        if (_i3 >= _iterator3.length) break;
        _ref3 = _iterator3[_i3++];
      } else {
        _i3 = _iterator3.next();
        if (_i3.done) break;
        _ref3 = _i3.value;
      }

      var stage = _ref3;

      if (typeof stage === 'function') {
        if (!positions) {
          stage(this.font, glyphs, this);
        }
      } else if (stage.length > 0) {
        processor.applyFeatures(stage, glyphs, positions);
      }
    }
  };

  return ShapingPlan;
}();

var VARIATION_FEATURES = ['rvrn'];
var COMMON_FEATURES = ['ccmp', 'locl', 'rlig', 'mark', 'mkmk'];
var FRACTIONAL_FEATURES = ['frac', 'numr', 'dnom'];
var HORIZONTAL_FEATURES = ['calt', 'clig', 'liga', 'rclt', 'curs', 'kern'];
var DIRECTIONAL_FEATURES = {
  ltr: ['ltra', 'ltrm'],
  rtl: ['rtla', 'rtlm']
};

var DefaultShaper =
/*#__PURE__*/
function () {
  function DefaultShaper() {}

  DefaultShaper.plan = function plan(_plan, glyphs, features) {
    // Plan the features we want to apply
    this.planPreprocessing(_plan);
    this.planFeatures(_plan);
    this.planPostprocessing(_plan, features); // Assign the global features to all the glyphs

    _plan.assignGlobalFeatures(glyphs); // Assign local features to glyphs


    this.assignFeatures(_plan, glyphs);
  };

  DefaultShaper.planPreprocessing = function planPreprocessing(plan) {
    plan.add({
      global: [].concat(VARIATION_FEATURES, DIRECTIONAL_FEATURES[plan.direction]),
      local: FRACTIONAL_FEATURES
    });
  };

  DefaultShaper.planFeatures = function planFeatures(plan) {// Do nothing by default. Let subclasses override this.
  };

  DefaultShaper.planPostprocessing = function planPostprocessing(plan, userFeatures) {
    plan.add([].concat(COMMON_FEATURES, HORIZONTAL_FEATURES));
    plan.setFeatureOverrides(userFeatures);
  };

  DefaultShaper.assignFeatures = function assignFeatures(plan, glyphs) {
    // Enable contextual fractions
    for (var i = 0; i < glyphs.length; i++) {
      var glyph = glyphs[i];

      if (glyph.codePoints[0] === 0x2044) {
        // fraction slash
        var start = i;
        var end = i + 1; // Apply numerator

        while (start > 0 && unicode.isDigit(glyphs[start - 1].codePoints[0])) {
          glyphs[start - 1].features.numr = true;
          glyphs[start - 1].features.frac = true;
          start--;
        } // Apply denominator


        while (end < glyphs.length && unicode.isDigit(glyphs[end].codePoints[0])) {
          glyphs[end].features.dnom = true;
          glyphs[end].features.frac = true;
          end++;
        } // Apply fraction slash


        glyph.features.frac = true;
        i = end - 1;
      }
    }
  };

  return DefaultShaper;
}();

_defineProperty(DefaultShaper, "zeroMarkWidths", 'AFTER_GPOS');

var base64DeflatedTrie$1 = "eJwBpgFZ/gABEAAAAAAAAAAxoAGVAWr+7dgrTwNBFAXg2QdNKYp/QCWSBIMlBI9EIcHgwIEDicQgUQRJUEgcEonC4JDgMJzSmeRmmNnO7s4+KOcmX2a623n27iMdp0qtwhpswFh//ivlDuzCHuzDIRzDKZzDBVzCFVyLdrdwpz9PPIj6o6jTfHuCI+0ZXuC1B/Nqy5uov/dgPkRERERERERETZn8//PRg3kQEREREc2zfHHqbNiMLfR9AvfwCesjpQ5G03M3KLeVUpvD5tb3BVn2+/gSji07jndlJU1+MBj/ORIPed5VjxW57jez5I7S5mpTlmzXp5Bzkmu09ypkfTFU2ds2ZFY9ZB2uvDLnlSh9EeM6kOOFhFyzPFY3Fhz9D7Si3z80JzLRp91uEGH+qsRcXHlSN/eK9sZ1fta4odH1dVc393ztZ93ry+Se/K7r2eabV9kcMvdk329cJRKrrd1PrGdx1znie/8w628rj+vmc9H6mujb974WO2LtcVpxnr5rL7FKU0/0WCHvMGlB/67xzFihcy3zfC/q036euOZn1+37ZZ3w7WMTefcNZvnVvA==";

var trieData$1 = pako.inflate(base64Arraybuffer_2(base64DeflatedTrie$1));
var trie$1 = new unicodeTrie(trieData$1);
var FEATURES = ['isol', 'fina', 'fin2', 'fin3', 'medi', 'med2', 'init'];
var ShapingClasses = {
  Non_Joining: 0,
  Left_Joining: 1,
  Right_Joining: 2,
  Dual_Joining: 3,
  Join_Causing: 3,
  ALAPH: 4,
  'DALATH RISH': 5,
  Transparent: 6
};
var ISOL = 'isol';
var FINA = 'fina';
var FIN2 = 'fin2';
var FIN3 = 'fin3';
var MEDI = 'medi';
var MED2 = 'med2';
var INIT = 'init';
var NONE = null; // Each entry is [prevAction, curAction, nextState]

var STATE_TABLE = [//   Non_Joining,        Left_Joining,       Right_Joining,     Dual_Joining,           ALAPH,            DALATH RISH
// State 0: prev was U,  not willing to join.
[[NONE, NONE, 0], [NONE, ISOL, 2], [NONE, ISOL, 1], [NONE, ISOL, 2], [NONE, ISOL, 1], [NONE, ISOL, 6]], // State 1: prev was R or ISOL/ALAPH,  not willing to join.
[[NONE, NONE, 0], [NONE, ISOL, 2], [NONE, ISOL, 1], [NONE, ISOL, 2], [NONE, FIN2, 5], [NONE, ISOL, 6]], // State 2: prev was D/L in ISOL form,  willing to join.
[[NONE, NONE, 0], [NONE, ISOL, 2], [INIT, FINA, 1], [INIT, FINA, 3], [INIT, FINA, 4], [INIT, FINA, 6]], // State 3: prev was D in FINA form,  willing to join.
[[NONE, NONE, 0], [NONE, ISOL, 2], [MEDI, FINA, 1], [MEDI, FINA, 3], [MEDI, FINA, 4], [MEDI, FINA, 6]], // State 4: prev was FINA ALAPH,  not willing to join.
[[NONE, NONE, 0], [NONE, ISOL, 2], [MED2, ISOL, 1], [MED2, ISOL, 2], [MED2, FIN2, 5], [MED2, ISOL, 6]], // State 5: prev was FIN2/FIN3 ALAPH,  not willing to join.
[[NONE, NONE, 0], [NONE, ISOL, 2], [ISOL, ISOL, 1], [ISOL, ISOL, 2], [ISOL, FIN2, 5], [ISOL, ISOL, 6]], // State 6: prev was DALATH/RISH,  not willing to join.
[[NONE, NONE, 0], [NONE, ISOL, 2], [NONE, ISOL, 1], [NONE, ISOL, 2], [NONE, FIN3, 5], [NONE, ISOL, 6]]];
/**
 * This is a shaper for Arabic, and other cursive scripts.
 * It uses data from ArabicShaping.txt in the Unicode database,
 * compiled to a UnicodeTrie by generate-data.coffee.
 *
 * The shaping state machine was ported from Harfbuzz.
 * https://github.com/behdad/harfbuzz/blob/master/src/hb-ot-shape-complex-arabic.cc
 */

var ArabicShaper =
/*#__PURE__*/
function (_DefaultShaper) {
  _inheritsLoose(ArabicShaper, _DefaultShaper);

  function ArabicShaper() {
    return _DefaultShaper.apply(this, arguments) || this;
  }

  ArabicShaper.planFeatures = function planFeatures(plan) {
    plan.add(['ccmp', 'locl']);

    for (var i = 0; i < FEATURES.length; i++) {
      var feature = FEATURES[i];
      plan.addStage(feature, false);
    }

    plan.addStage('mset');
  };

  ArabicShaper.assignFeatures = function assignFeatures(plan, glyphs) {
    _DefaultShaper.assignFeatures.call(this, plan, glyphs);

    var prev = -1;
    var state = 0;
    var actions = []; // Apply the state machine to map glyphs to features

    for (var i = 0; i < glyphs.length; i++) {
      var curAction = void 0,
          prevAction = void 0;
      var glyph = glyphs[i];
      var type = getShapingClass(glyph.codePoints[0]);

      if (type === ShapingClasses.Transparent) {
        actions[i] = NONE;
        continue;
      }

      var _STATE_TABLE$state$ty = STATE_TABLE[state][type];
      prevAction = _STATE_TABLE$state$ty[0];
      curAction = _STATE_TABLE$state$ty[1];
      state = _STATE_TABLE$state$ty[2];

      if (prevAction !== NONE && prev !== -1) {
        actions[prev] = prevAction;
      }

      actions[i] = curAction;
      prev = i;
    } // Apply the chosen features to their respective glyphs


    for (var index = 0; index < glyphs.length; index++) {
      var feature = void 0;
      var glyph = glyphs[index];

      if (feature = actions[index]) {
        glyph.features[feature] = true;
      }
    }
  };

  return ArabicShaper;
}(DefaultShaper);

function getShapingClass(codePoint) {
  var res = trie$1.get(codePoint);

  if (res) {
    return res - 1;
  }

  var category = unicode.getCategory(codePoint);

  if (category === 'Mn' || category === 'Me' || category === 'Cf') {
    return ShapingClasses.Transparent;
  }

  return ShapingClasses.Non_Joining;
}

var GlyphIterator =
/*#__PURE__*/
function () {
  function GlyphIterator(glyphs, options) {
    this.glyphs = glyphs;
    this.reset(options);
  }

  var _proto = GlyphIterator.prototype;

  _proto.reset = function reset(options, index) {
    if (options === void 0) {
      options = {};
    }

    if (index === void 0) {
      index = 0;
    }

    this.options = options;
    this.flags = options.flags || {};
    this.markAttachmentType = options.markAttachmentType || 0;
    this.index = index;
  };

  _proto.shouldIgnore = function shouldIgnore(glyph) {
    return this.flags.ignoreMarks && glyph.isMark || this.flags.ignoreBaseGlyphs && glyph.isBase || this.flags.ignoreLigatures && glyph.isLigature || this.markAttachmentType && glyph.isMark && glyph.markAttachmentType !== this.markAttachmentType;
  };

  _proto.move = function move(dir) {
    this.index += dir;

    while (0 <= this.index && this.index < this.glyphs.length && this.shouldIgnore(this.glyphs[this.index])) {
      this.index += dir;
    }

    if (0 > this.index || this.index >= this.glyphs.length) {
      return null;
    }

    return this.glyphs[this.index];
  };

  _proto.next = function next() {
    return this.move(+1);
  };

  _proto.prev = function prev() {
    return this.move(-1);
  };

  _proto.peek = function peek(count) {
    if (count === void 0) {
      count = 1;
    }

    var idx = this.index;
    var res = this.increment(count);
    this.index = idx;
    return res;
  };

  _proto.peekIndex = function peekIndex(count) {
    if (count === void 0) {
      count = 1;
    }

    var idx = this.index;
    this.increment(count);
    var res = this.index;
    this.index = idx;
    return res;
  };

  _proto.increment = function increment(count) {
    if (count === void 0) {
      count = 1;
    }

    var dir = count < 0 ? -1 : 1;
    count = Math.abs(count);

    while (count--) {
      this.move(dir);
    }

    return this.glyphs[this.index];
  };

  _createClass(GlyphIterator, [{
    key: "cur",
    get: function get() {
      return this.glyphs[this.index] || null;
    }
  }]);

  return GlyphIterator;
}();

var DEFAULT_SCRIPTS = ['DFLT', 'dflt', 'latn'];

var OTProcessor =
/*#__PURE__*/
function () {
  function OTProcessor(font, table) {
    this.font = font;
    this.table = table;
    this.script = null;
    this.scriptTag = null;
    this.language = null;
    this.languageTag = null;
    this.features = {};
    this.lookups = {}; // Setup variation substitutions

    this.variationsIndex = font._variationProcessor ? this.findVariationsIndex(font._variationProcessor.normalizedCoords) : -1; // initialize to default script + language

    this.selectScript(); // current context (set by applyFeatures)

    this.glyphs = [];
    this.positions = []; // only used by GPOS

    this.ligatureID = 1;
    this.currentFeature = null;
  }

  var _proto = OTProcessor.prototype;

  _proto.findScript = function findScript(script) {
    if (this.table.scriptList == null) {
      return null;
    }

    if (!Array.isArray(script)) {
      script = [script];
    }

    for (var _iterator = script, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
      var _ref;

      if (_isArray) {
        if (_i >= _iterator.length) break;
        _ref = _iterator[_i++];
      } else {
        _i = _iterator.next();
        if (_i.done) break;
        _ref = _i.value;
      }

      var s = _ref;

      for (var _iterator2 = this.table.scriptList, _isArray2 = Array.isArray(_iterator2), _i2 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
        var _ref2;

        if (_isArray2) {
          if (_i2 >= _iterator2.length) break;
          _ref2 = _iterator2[_i2++];
        } else {
          _i2 = _iterator2.next();
          if (_i2.done) break;
          _ref2 = _i2.value;
        }

        var entry = _ref2;

        if (entry.tag === s) {
          return entry;
        }
      }
    }

    return null;
  };

  _proto.selectScript = function selectScript(script, language, direction$$1) {
    var changed = false;
    var entry;

    if (!this.script || script !== this.scriptTag) {
      entry = this.findScript(script);

      if (!entry) {
        entry = this.findScript(DEFAULT_SCRIPTS);
      }

      if (!entry) {
        return this.scriptTag;
      }

      this.scriptTag = entry.tag;
      this.script = entry.script;
      this.language = null;
      this.languageTag = null;
      changed = true;
    }

    if (!direction$$1 || direction$$1 !== this.direction) {
      this.direction = direction$$1 || direction(script);
    }

    if (language && language.length < 4) {
      language += ' '.repeat(4 - language.length);
    }

    if (!language || language !== this.languageTag) {
      this.language = null;

      for (var _iterator3 = this.script.langSysRecords, _isArray3 = Array.isArray(_iterator3), _i3 = 0, _iterator3 = _isArray3 ? _iterator3 : _iterator3[Symbol.iterator]();;) {
        var _ref3;

        if (_isArray3) {
          if (_i3 >= _iterator3.length) break;
          _ref3 = _iterator3[_i3++];
        } else {
          _i3 = _iterator3.next();
          if (_i3.done) break;
          _ref3 = _i3.value;
        }

        var lang = _ref3;

        if (lang.tag === language) {
          this.language = lang.langSys;
          this.languageTag = lang.tag;
          break;
        }
      }

      if (!this.language) {
        this.language = this.script.defaultLangSys;
        this.languageTag = null;
      }

      changed = true;
    } // Build a feature lookup table


    if (changed) {
      this.features = {};

      if (this.language) {
        for (var _iterator4 = this.language.featureIndexes, _isArray4 = Array.isArray(_iterator4), _i4 = 0, _iterator4 = _isArray4 ? _iterator4 : _iterator4[Symbol.iterator]();;) {
          var _ref4;

          if (_isArray4) {
            if (_i4 >= _iterator4.length) break;
            _ref4 = _iterator4[_i4++];
          } else {
            _i4 = _iterator4.next();
            if (_i4.done) break;
            _ref4 = _i4.value;
          }

          var featureIndex = _ref4;
          var record = this.table.featureList[featureIndex];
          var substituteFeature = this.substituteFeatureForVariations(featureIndex);
          this.features[record.tag] = substituteFeature || record.feature;
        }
      }
    }

    return this.scriptTag;
  };

  _proto.lookupsForFeatures = function lookupsForFeatures(userFeatures, exclude) {
    if (userFeatures === void 0) {
      userFeatures = [];
    }

    var lookups = [];

    for (var _iterator5 = userFeatures, _isArray5 = Array.isArray(_iterator5), _i5 = 0, _iterator5 = _isArray5 ? _iterator5 : _iterator5[Symbol.iterator]();;) {
      var _ref5;

      if (_isArray5) {
        if (_i5 >= _iterator5.length) break;
        _ref5 = _iterator5[_i5++];
      } else {
        _i5 = _iterator5.next();
        if (_i5.done) break;
        _ref5 = _i5.value;
      }

      var tag = _ref5;
      var feature = this.features[tag];

      if (!feature) {
        continue;
      }

      for (var _iterator6 = feature.lookupListIndexes, _isArray6 = Array.isArray(_iterator6), _i6 = 0, _iterator6 = _isArray6 ? _iterator6 : _iterator6[Symbol.iterator]();;) {
        var _ref6;

        if (_isArray6) {
          if (_i6 >= _iterator6.length) break;
          _ref6 = _iterator6[_i6++];
        } else {
          _i6 = _iterator6.next();
          if (_i6.done) break;
          _ref6 = _i6.value;
        }

        var lookupIndex = _ref6;

        if (exclude && exclude.indexOf(lookupIndex) !== -1) {
          continue;
        }

        lookups.push({
          feature: tag,
          index: lookupIndex,
          lookup: this.table.lookupList.get(lookupIndex)
        });
      }
    }

    lookups.sort(function (a, b) {
      return a.index - b.index;
    });
    return lookups;
  };

  _proto.substituteFeatureForVariations = function substituteFeatureForVariations(featureIndex) {
    if (this.variationsIndex === -1) {
      return null;
    }

    var record = this.table.featureVariations.featureVariationRecords[this.variationsIndex];
    var substitutions = record.featureTableSubstitution.substitutions;

    for (var _iterator7 = substitutions, _isArray7 = Array.isArray(_iterator7), _i7 = 0, _iterator7 = _isArray7 ? _iterator7 : _iterator7[Symbol.iterator]();;) {
      var _ref7;

      if (_isArray7) {
        if (_i7 >= _iterator7.length) break;
        _ref7 = _iterator7[_i7++];
      } else {
        _i7 = _iterator7.next();
        if (_i7.done) break;
        _ref7 = _i7.value;
      }

      var substitution = _ref7;

      if (substitution.featureIndex === featureIndex) {
        return substitution.alternateFeatureTable;
      }
    }

    return null;
  };

  _proto.findVariationsIndex = function findVariationsIndex(coords) {
    var variations = this.table.featureVariations;

    if (!variations) {
      return -1;
    }

    var records = variations.featureVariationRecords;

    for (var i = 0; i < records.length; i++) {
      var conditions = records[i].conditionSet.conditionTable;

      if (this.variationConditionsMatch(conditions, coords)) {
        return i;
      }
    }

    return -1;
  };

  _proto.variationConditionsMatch = function variationConditionsMatch(conditions, coords) {
    return conditions.every(function (condition) {
      var coord = condition.axisIndex < coords.length ? coords[condition.axisIndex] : 0;
      return condition.filterRangeMinValue <= coord && coord <= condition.filterRangeMaxValue;
    });
  };

  _proto.applyFeatures = function applyFeatures(userFeatures, glyphs, advances) {
    var lookups = this.lookupsForFeatures(userFeatures);
    this.applyLookups(lookups, glyphs, advances);
  };

  _proto.applyLookups = function applyLookups(lookups, glyphs, positions) {
    this.glyphs = glyphs;
    this.positions = positions;
    this.glyphIterator = new GlyphIterator(glyphs);

    for (var _iterator8 = lookups, _isArray8 = Array.isArray(_iterator8), _i8 = 0, _iterator8 = _isArray8 ? _iterator8 : _iterator8[Symbol.iterator]();;) {
      var _ref8;

      if (_isArray8) {
        if (_i8 >= _iterator8.length) break;
        _ref8 = _iterator8[_i8++];
      } else {
        _i8 = _iterator8.next();
        if (_i8.done) break;
        _ref8 = _i8.value;
      }

      var _ref9 = _ref8,
          feature = _ref9.feature,
          lookup = _ref9.lookup;
      this.currentFeature = feature;
      this.glyphIterator.reset(lookup.flags);

      while (this.glyphIterator.index < glyphs.length) {
        if (!(feature in this.glyphIterator.cur.features)) {
          this.glyphIterator.next();
          continue;
        }

        for (var _iterator9 = lookup.subTables, _isArray9 = Array.isArray(_iterator9), _i9 = 0, _iterator9 = _isArray9 ? _iterator9 : _iterator9[Symbol.iterator]();;) {
          var _ref10;

          if (_isArray9) {
            if (_i9 >= _iterator9.length) break;
            _ref10 = _iterator9[_i9++];
          } else {
            _i9 = _iterator9.next();
            if (_i9.done) break;
            _ref10 = _i9.value;
          }

          var table = _ref10;
          var res = this.applyLookup(lookup.lookupType, table);

          if (res) {
            break;
          }
        }

        this.glyphIterator.next();
      }
    }
  };

  _proto.applyLookup = function applyLookup(lookup, table) {
    throw new Error("applyLookup must be implemented by subclasses");
  };

  _proto.applyLookupList = function applyLookupList(lookupRecords) {
    var options = this.glyphIterator.options;
    var glyphIndex = this.glyphIterator.index;

    for (var _iterator10 = lookupRecords, _isArray10 = Array.isArray(_iterator10), _i10 = 0, _iterator10 = _isArray10 ? _iterator10 : _iterator10[Symbol.iterator]();;) {
      var _ref11;

      if (_isArray10) {
        if (_i10 >= _iterator10.length) break;
        _ref11 = _iterator10[_i10++];
      } else {
        _i10 = _iterator10.next();
        if (_i10.done) break;
        _ref11 = _i10.value;
      }

      var lookupRecord = _ref11;
      // Reset flags and find glyph index for this lookup record
      this.glyphIterator.reset(options, glyphIndex);
      this.glyphIterator.increment(lookupRecord.sequenceIndex); // Get the lookup and setup flags for subtables

      var lookup = this.table.lookupList.get(lookupRecord.lookupListIndex);
      this.glyphIterator.reset(lookup.flags, this.glyphIterator.index); // Apply lookup subtables until one matches

      for (var _iterator11 = lookup.subTables, _isArray11 = Array.isArray(_iterator11), _i11 = 0, _iterator11 = _isArray11 ? _iterator11 : _iterator11[Symbol.iterator]();;) {
        var _ref12;

        if (_isArray11) {
          if (_i11 >= _iterator11.length) break;
          _ref12 = _iterator11[_i11++];
        } else {
          _i11 = _iterator11.next();
          if (_i11.done) break;
          _ref12 = _i11.value;
        }

        var table = _ref12;

        if (this.applyLookup(lookup.lookupType, table)) {
          break;
        }
      }
    }

    this.glyphIterator.reset(options, glyphIndex);
    return true;
  };

  _proto.coverageIndex = function coverageIndex(coverage, glyph) {
    if (glyph == null) {
      glyph = this.glyphIterator.cur.id;
    }

    switch (coverage.version) {
      case 1:
        return coverage.glyphs.indexOf(glyph);

      case 2:
        for (var _iterator12 = coverage.rangeRecords, _isArray12 = Array.isArray(_iterator12), _i12 = 0, _iterator12 = _isArray12 ? _iterator12 : _iterator12[Symbol.iterator]();;) {
          var _ref13;

          if (_isArray12) {
            if (_i12 >= _iterator12.length) break;
            _ref13 = _iterator12[_i12++];
          } else {
            _i12 = _iterator12.next();
            if (_i12.done) break;
            _ref13 = _i12.value;
          }

          var range = _ref13;

          if (range.start <= glyph && glyph <= range.end) {
            return range.startCoverageIndex + glyph - range.start;
          }
        }

        break;
    }

    return -1;
  };

  _proto.match = function match(sequenceIndex, sequence, fn, matched) {
    var pos = this.glyphIterator.index;
    var glyph = this.glyphIterator.increment(sequenceIndex);
    var idx = 0;

    while (idx < sequence.length && glyph && fn(sequence[idx], glyph)) {
      if (matched) {
        matched.push(this.glyphIterator.index);
      }

      idx++;
      glyph = this.glyphIterator.next();
    }

    this.glyphIterator.index = pos;

    if (idx < sequence.length) {
      return false;
    }

    return matched || true;
  };

  _proto.sequenceMatches = function sequenceMatches(sequenceIndex, sequence) {
    return this.match(sequenceIndex, sequence, function (component, glyph) {
      return component === glyph.id;
    });
  };

  _proto.sequenceMatchIndices = function sequenceMatchIndices(sequenceIndex, sequence) {
    var _this = this;

    return this.match(sequenceIndex, sequence, function (component, glyph) {
      // If the current feature doesn't apply to this glyph,
      if (!(_this.currentFeature in glyph.features)) {
        return false;
      }

      return component === glyph.id;
    }, []);
  };

  _proto.coverageSequenceMatches = function coverageSequenceMatches(sequenceIndex, sequence) {
    var _this2 = this;

    return this.match(sequenceIndex, sequence, function (coverage, glyph) {
      return _this2.coverageIndex(coverage, glyph.id) >= 0;
    });
  };

  _proto.getClassID = function getClassID(glyph, classDef) {
    switch (classDef.version) {
      case 1:
        // Class array
        var i = glyph - classDef.startGlyph;

        if (i >= 0 && i < classDef.classValueArray.length) {
          return classDef.classValueArray[i];
        }

        break;

      case 2:
        for (var _iterator13 = classDef.classRangeRecord, _isArray13 = Array.isArray(_iterator13), _i13 = 0, _iterator13 = _isArray13 ? _iterator13 : _iterator13[Symbol.iterator]();;) {
          var _ref14;

          if (_isArray13) {
            if (_i13 >= _iterator13.length) break;
            _ref14 = _iterator13[_i13++];
          } else {
            _i13 = _iterator13.next();
            if (_i13.done) break;
            _ref14 = _i13.value;
          }

          var range = _ref14;

          if (range.start <= glyph && glyph <= range.end) {
            return range.class;
          }
        }

        break;
    }

    return 0;
  };

  _proto.classSequenceMatches = function classSequenceMatches(sequenceIndex, sequence, classDef) {
    var _this3 = this;

    return this.match(sequenceIndex, sequence, function (classID, glyph) {
      return classID === _this3.getClassID(glyph.id, classDef);
    });
  };

  _proto.applyContext = function applyContext(table) {
    switch (table.version) {
      case 1:
        var index = this.coverageIndex(table.coverage);

        if (index === -1) {
          return false;
        }

        var set = table.ruleSets[index];

        for (var _iterator14 = set, _isArray14 = Array.isArray(_iterator14), _i14 = 0, _iterator14 = _isArray14 ? _iterator14 : _iterator14[Symbol.iterator]();;) {
          var _ref15;

          if (_isArray14) {
            if (_i14 >= _iterator14.length) break;
            _ref15 = _iterator14[_i14++];
          } else {
            _i14 = _iterator14.next();
            if (_i14.done) break;
            _ref15 = _i14.value;
          }

          var rule = _ref15;

          if (this.sequenceMatches(1, rule.input)) {
            return this.applyLookupList(rule.lookupRecords);
          }
        }

        break;

      case 2:
        if (this.coverageIndex(table.coverage) === -1) {
          return false;
        }

        index = this.getClassID(this.glyphIterator.cur.id, table.classDef);

        if (index === -1) {
          return false;
        }

        set = table.classSet[index];

        for (var _iterator15 = set, _isArray15 = Array.isArray(_iterator15), _i15 = 0, _iterator15 = _isArray15 ? _iterator15 : _iterator15[Symbol.iterator]();;) {
          var _ref16;

          if (_isArray15) {
            if (_i15 >= _iterator15.length) break;
            _ref16 = _iterator15[_i15++];
          } else {
            _i15 = _iterator15.next();
            if (_i15.done) break;
            _ref16 = _i15.value;
          }

          var _rule = _ref16;

          if (this.classSequenceMatches(1, _rule.classes, table.classDef)) {
            return this.applyLookupList(_rule.lookupRecords);
          }
        }

        break;

      case 3:
        if (this.coverageSequenceMatches(0, table.coverages)) {
          return this.applyLookupList(table.lookupRecords);
        }

        break;
    }

    return false;
  };

  _proto.applyChainingContext = function applyChainingContext(table) {
    switch (table.version) {
      case 1:
        var index = this.coverageIndex(table.coverage);

        if (index === -1) {
          return false;
        }

        var set = table.chainRuleSets[index];

        for (var _iterator16 = set, _isArray16 = Array.isArray(_iterator16), _i16 = 0, _iterator16 = _isArray16 ? _iterator16 : _iterator16[Symbol.iterator]();;) {
          var _ref17;

          if (_isArray16) {
            if (_i16 >= _iterator16.length) break;
            _ref17 = _iterator16[_i16++];
          } else {
            _i16 = _iterator16.next();
            if (_i16.done) break;
            _ref17 = _i16.value;
          }

          var rule = _ref17;

          if (this.sequenceMatches(-rule.backtrack.length, rule.backtrack) && this.sequenceMatches(1, rule.input) && this.sequenceMatches(1 + rule.input.length, rule.lookahead)) {
            return this.applyLookupList(rule.lookupRecords);
          }
        }

        break;

      case 2:
        if (this.coverageIndex(table.coverage) === -1) {
          return false;
        }

        index = this.getClassID(this.glyphIterator.cur.id, table.inputClassDef);
        var rules = table.chainClassSet[index];

        if (!rules) {
          return false;
        }

        for (var _iterator17 = rules, _isArray17 = Array.isArray(_iterator17), _i17 = 0, _iterator17 = _isArray17 ? _iterator17 : _iterator17[Symbol.iterator]();;) {
          var _ref18;

          if (_isArray17) {
            if (_i17 >= _iterator17.length) break;
            _ref18 = _iterator17[_i17++];
          } else {
            _i17 = _iterator17.next();
            if (_i17.done) break;
            _ref18 = _i17.value;
          }

          var _rule2 = _ref18;

          if (this.classSequenceMatches(-_rule2.backtrack.length, _rule2.backtrack, table.backtrackClassDef) && this.classSequenceMatches(1, _rule2.input, table.inputClassDef) && this.classSequenceMatches(1 + _rule2.input.length, _rule2.lookahead, table.lookaheadClassDef)) {
            return this.applyLookupList(_rule2.lookupRecords);
          }
        }

        break;

      case 3:
        if (this.coverageSequenceMatches(-table.backtrackGlyphCount, table.backtrackCoverage) && this.coverageSequenceMatches(0, table.inputCoverage) && this.coverageSequenceMatches(table.inputGlyphCount, table.lookaheadCoverage)) {
          return this.applyLookupList(table.lookupRecords);
        }

        break;
    }

    return false;
  };

  return OTProcessor;
}();

var GlyphInfo =
/*#__PURE__*/
function () {
  function GlyphInfo(font, id, codePoints, features) {
    if (codePoints === void 0) {
      codePoints = [];
    }

    this._font = font;
    this.codePoints = codePoints;
    this.id = id;
    this.features = {};

    if (Array.isArray(features)) {
      for (var i = 0; i < features.length; i++) {
        var feature = features[i];
        this.features[feature] = true;
      }
    } else if (typeof features === 'object') {
      Object.assign(this.features, features);
    }

    this.ligatureID = null;
    this.ligatureComponent = null;
    this.isLigated = false;
    this.cursiveAttachment = null;
    this.markAttachment = null;
    this.shaperInfo = null;
    this.substituted = false;
    this.isMultiplied = false;
  }

  var _proto = GlyphInfo.prototype;

  _proto.copy = function copy() {
    return new GlyphInfo(this._font, this.id, this.codePoints, this.features);
  };

  _createClass(GlyphInfo, [{
    key: "id",
    get: function get() {
      return this._id;
    },
    set: function set(id) {
      this._id = id;
      this.substituted = true;
      var GDEF = this._font.GDEF;

      if (GDEF && GDEF.glyphClassDef) {
        // TODO: clean this up
        var classID = OTProcessor.prototype.getClassID(id, GDEF.glyphClassDef);
        this.isBase = classID === 1;
        this.isLigature = classID === 2;
        this.isMark = classID === 3;
        this.markAttachmentType = GDEF.markAttachClassDef ? OTProcessor.prototype.getClassID(id, GDEF.markAttachClassDef) : 0;
      } else {
        this.isMark = this.codePoints.length > 0 && this.codePoints.every(unicode.isMark);
        this.isBase = !this.isMark;
        this.isLigature = this.codePoints.length > 1;
        this.markAttachmentType = 0;
      }
    }
  }]);

  return GlyphInfo;
}();

/**
 * This is a shaper for the Hangul script, used by the Korean language.
 * It does the following:
 *   - decompose if unsupported by the font:
 *     <LV>   -> <L,V>
 *     <LVT>  -> <L,V,T>
 *     <LV,T> -> <L,V,T>
 *
 *   - compose if supported by the font:
 *     <L,V>   -> <LV>
 *     <L,V,T> -> <LVT>
 *     <LV,T>  -> <LVT>
 *
 *   - reorder tone marks (S is any valid syllable):
 *     <S, M> -> <M, S>
 *
 *   - apply ljmo, vjmo, and tjmo OpenType features to decomposed Jamo sequences.
 *
 * This logic is based on the following documents:
 *   - http://www.microsoft.com/typography/OpenTypeDev/hangul/intro.htm
 *   - http://ktug.org/~nomos/harfbuzz-hangul/hangulshaper.pdf
 */

var HangulShaper =
/*#__PURE__*/
function (_DefaultShaper) {
  _inheritsLoose(HangulShaper, _DefaultShaper);

  function HangulShaper() {
    return _DefaultShaper.apply(this, arguments) || this;
  }

  HangulShaper.planFeatures = function planFeatures(plan) {
    plan.add(['ljmo', 'vjmo', 'tjmo'], false);
  };

  HangulShaper.assignFeatures = function assignFeatures(plan, glyphs) {
    var state = 0;
    var i = 0;

    while (i < glyphs.length) {
      var action = void 0;
      var glyph = glyphs[i];
      var code = glyph.codePoints[0];
      var type = getType(code);
      var _STATE_TABLE$state$ty = STATE_TABLE$1[state][type];
      action = _STATE_TABLE$state$ty[0];
      state = _STATE_TABLE$state$ty[1];

      switch (action) {
        case DECOMPOSE:
          // Decompose the composed syllable if it is not supported by the font.
          if (!plan.font.hasGlyphForCodePoint(code)) {
            i = decompose(glyphs, i, plan.font);
          }

          break;

        case COMPOSE:
          // Found a decomposed syllable. Try to compose if supported by the font.
          i = compose(glyphs, i, plan.font);
          break;

        case TONE_MARK:
          // Got a valid syllable, followed by a tone mark. Move the tone mark to the beginning of the syllable.
          reorderToneMark(glyphs, i, plan.font);
          break;

        case INVALID:
          // Tone mark has no valid syllable to attach to, so insert a dotted circle
          i = insertDottedCircle(glyphs, i, plan.font);
          break;
      }

      i++;
    }
  };

  return HangulShaper;
}(DefaultShaper);

_defineProperty(HangulShaper, "zeroMarkWidths", 'NONE');
var HANGUL_BASE = 0xac00;
var HANGUL_END = 0xd7a4;
var HANGUL_COUNT = HANGUL_END - HANGUL_BASE + 1;
var L_BASE = 0x1100; // lead

var V_BASE = 0x1161; // vowel

var T_BASE = 0x11a7; // trail

var L_COUNT = 19;
var V_COUNT = 21;
var T_COUNT = 28;
var L_END = L_BASE + L_COUNT - 1;
var V_END = V_BASE + V_COUNT - 1;
var T_END = T_BASE + T_COUNT - 1;
var DOTTED_CIRCLE = 0x25cc;

var isL = function isL(code) {
  return 0x1100 <= code && code <= 0x115f || 0xa960 <= code && code <= 0xa97c;
};

var isV = function isV(code) {
  return 0x1160 <= code && code <= 0x11a7 || 0xd7b0 <= code && code <= 0xd7c6;
};

var isT = function isT(code) {
  return 0x11a8 <= code && code <= 0x11ff || 0xd7cb <= code && code <= 0xd7fb;
};

var isTone = function isTone(code) {
  return 0x302e <= code && code <= 0x302f;
};

var isLVT = function isLVT(code) {
  return HANGUL_BASE <= code && code <= HANGUL_END;
};

var isLV = function isLV(code) {
  return code - HANGUL_BASE < HANGUL_COUNT && (code - HANGUL_BASE) % T_COUNT === 0;
};

var isCombiningL = function isCombiningL(code) {
  return L_BASE <= code && code <= L_END;
};

var isCombiningV = function isCombiningV(code) {
  return V_BASE <= code && code <= V_END;
};

var isCombiningT = function isCombiningT(code) {
  return 1 <= code && code <= T_END;
}; // Character categories


var X = 0; // Other character

var L = 1; // Leading consonant

var V = 2; // Medial vowel

var T = 3; // Trailing consonant

var LV = 4; // Composed <LV> syllable

var LVT = 5; // Composed <LVT> syllable

var M = 6; // Tone mark
// This function classifies a character using the above categories.

function getType(code) {
  if (isL(code)) {
    return L;
  }

  if (isV(code)) {
    return V;
  }

  if (isT(code)) {
    return T;
  }

  if (isLV(code)) {
    return LV;
  }

  if (isLVT(code)) {
    return LVT;
  }

  if (isTone(code)) {
    return M;
  }

  return X;
} // State machine actions


var NO_ACTION = 0;
var DECOMPOSE = 1;
var COMPOSE = 2;
var TONE_MARK = 4;
var INVALID = 5; // Build a state machine that accepts valid syllables, and applies actions along the way.
// The logic this is implementing is documented at the top of the file.

var STATE_TABLE$1 = [//       X                 L                 V                T                  LV                LVT               M
// State 0: start state
[[NO_ACTION, 0], [NO_ACTION, 1], [NO_ACTION, 0], [NO_ACTION, 0], [DECOMPOSE, 2], [DECOMPOSE, 3], [INVALID, 0]], // State 1: <L>
[[NO_ACTION, 0], [NO_ACTION, 1], [COMPOSE, 2], [NO_ACTION, 0], [DECOMPOSE, 2], [DECOMPOSE, 3], [INVALID, 0]], // State 2: <L,V> or <LV>
[[NO_ACTION, 0], [NO_ACTION, 1], [NO_ACTION, 0], [COMPOSE, 3], [DECOMPOSE, 2], [DECOMPOSE, 3], [TONE_MARK, 0]], // State 3: <L,V,T> or <LVT>
[[NO_ACTION, 0], [NO_ACTION, 1], [NO_ACTION, 0], [NO_ACTION, 0], [DECOMPOSE, 2], [DECOMPOSE, 3], [TONE_MARK, 0]]];

function getGlyph(font, code, features) {
  return new GlyphInfo(font, font.glyphForCodePoint(code).id, [code], features);
}

function decompose(glyphs, i, font) {
  var glyph = glyphs[i];
  var code = glyph.codePoints[0];
  var s = code - HANGUL_BASE;
  var t = T_BASE + s % T_COUNT;
  s = s / T_COUNT | 0;
  var l = L_BASE + s / V_COUNT | 0;
  var v = V_BASE + s % V_COUNT; // Don't decompose if all of the components are not available

  if (!font.hasGlyphForCodePoint(l) || !font.hasGlyphForCodePoint(v) || t !== T_BASE && !font.hasGlyphForCodePoint(t)) {
    return i;
  } // Replace the current glyph with decomposed L, V, and T glyphs,
  // and apply the proper OpenType features to each component.


  var ljmo = getGlyph(font, l, glyph.features);
  ljmo.features.ljmo = true;
  var vjmo = getGlyph(font, v, glyph.features);
  vjmo.features.vjmo = true;
  var insert = [ljmo, vjmo];

  if (t > T_BASE) {
    var tjmo = getGlyph(font, t, glyph.features);
    tjmo.features.tjmo = true;
    insert.push(tjmo);
  }

  glyphs.splice.apply(glyphs, [i, 1].concat(insert));
  return i + insert.length - 1;
}

function compose(glyphs, i, font) {
  var glyph = glyphs[i];
  var code = glyphs[i].codePoints[0];
  var type = getType(code);
  var prev = glyphs[i - 1].codePoints[0];
  var prevType = getType(prev); // Figure out what type of syllable we're dealing with

  var lv, ljmo, vjmo, tjmo;

  if (prevType === LV && type === T) {
    // <LV,T>
    lv = prev;
    tjmo = glyph;
  } else {
    if (type === V) {
      // <L,V>
      ljmo = glyphs[i - 1];
      vjmo = glyph;
    } else {
      // <L,V,T>
      ljmo = glyphs[i - 2];
      vjmo = glyphs[i - 1];
      tjmo = glyph;
    }

    var l = ljmo.codePoints[0];
    var v = vjmo.codePoints[0]; // Make sure L and V are combining characters

    if (isCombiningL(l) && isCombiningV(v)) {
      lv = HANGUL_BASE + ((l - L_BASE) * V_COUNT + (v - V_BASE)) * T_COUNT;
    }
  }

  var t = tjmo && tjmo.codePoints[0] || T_BASE;

  if (lv != null && (t === T_BASE || isCombiningT(t))) {
    var s = lv + (t - T_BASE); // Replace with a composed glyph if supported by the font,
    // otherwise apply the proper OpenType features to each component.

    if (font.hasGlyphForCodePoint(s)) {
      var del = prevType === V ? 3 : 2;
      glyphs.splice(i - del + 1, del, getGlyph(font, s, glyph.features));
      return i - del + 1;
    }
  } // Didn't compose (either a non-combining component or unsupported by font).


  if (ljmo) {
    ljmo.features.ljmo = true;
  }

  if (vjmo) {
    vjmo.features.vjmo = true;
  }

  if (tjmo) {
    tjmo.features.tjmo = true;
  }

  if (prevType === LV) {
    // Sequence was originally <L,V>, which got combined earlier.
    // Either the T was non-combining, or the LVT glyph wasn't supported.
    // Decompose the glyph again and apply OT features.
    decompose(glyphs, i - 1, font);
    return i + 1;
  }

  return i;
}

function getLength(code) {
  switch (getType(code)) {
    case LV:
    case LVT:
      return 1;

    case V:
      return 2;

    case T:
      return 3;
  }
}

function reorderToneMark(glyphs, i, font) {
  var glyph = glyphs[i];
  var code = glyphs[i].codePoints[0]; // Move tone mark to the beginning of the previous syllable, unless it is zero width

  if (font.glyphForCodePoint(code).advanceWidth === 0) {
    return;
  }

  var prev = glyphs[i - 1].codePoints[0];
  var len = getLength(prev);
  glyphs.splice(i, 1);
  return glyphs.splice(i - len, 0, glyph);
}

function insertDottedCircle(glyphs, i, font) {
  var glyph = glyphs[i];
  var code = glyphs[i].codePoints[0];

  if (font.hasGlyphForCodePoint(DOTTED_CIRCLE)) {
    var dottedCircle = getGlyph(font, DOTTED_CIRCLE, glyph.features); // If the tone mark is zero width, insert the dotted circle before, otherwise after

    var idx = font.glyphForCodePoint(code).advanceWidth === 0 ? i : i + 1;
    glyphs.splice(idx, 0, dottedCircle);
    i++;
  }

  return i;
}

var _iterStep = function _iterStep(done, value) {
  return {
    value: value,
    done: !!done
  };
};

var _iterators = {};

var toString$1 = {}.toString;

var _cof = function _cof(it) {
  return toString$1.call(it).slice(8, -1);
};

// eslint-disable-next-line no-prototype-builtins

var _iobject = Object('z').propertyIsEnumerable(0) ? Object : function (it) {
  return _cof(it) == 'String' ? it.split('') : Object(it);
};

// 7.2.1 RequireObjectCoercible(argument)
var _defined = function _defined(it) {
  if (it == undefined) throw TypeError("Can't call method on  " + it);
  return it;
};

var _toIobject = function _toIobject(it) {
  return _iobject(_defined(it));
};

var _library = true;

var _global = createCommonjsModule(function (module) {
  // https://github.com/zloirock/core-js/issues/86#issuecomment-115759028
  var global = module.exports = typeof window != 'undefined' && window.Math == Math ? window : typeof self != 'undefined' && self.Math == Math ? self // eslint-disable-next-line no-new-func
  : Function('return this')();
  if (typeof __g == 'number') __g = global; // eslint-disable-line no-undef
});

var _core = createCommonjsModule(function (module) {
  var core = module.exports = {
    version: '2.5.7'
  };
  if (typeof __e == 'number') __e = core; // eslint-disable-line no-undef
});
var _core_1 = _core.version;

var _aFunction = function _aFunction(it) {
  if (typeof it != 'function') throw TypeError(it + ' is not a function!');
  return it;
};

var _ctx = function _ctx(fn, that, length) {
  _aFunction(fn);
  if (that === undefined) return fn;

  switch (length) {
    case 1:
      return function (a) {
        return fn.call(that, a);
      };

    case 2:
      return function (a, b) {
        return fn.call(that, a, b);
      };

    case 3:
      return function (a, b, c) {
        return fn.call(that, a, b, c);
      };
  }

  return function ()
  /* ...args */
  {
    return fn.apply(that, arguments);
  };
};

var _isObject = function _isObject(it) {
  return typeof it === 'object' ? it !== null : typeof it === 'function';
};

var _anObject = function _anObject(it) {
  if (!_isObject(it)) throw TypeError(it + ' is not an object!');
  return it;
};

var _fails = function _fails(exec) {
  try {
    return !!exec();
  } catch (e) {
    return true;
  }
};

var _descriptors = !_fails(function () {
  return Object.defineProperty({}, 'a', {
    get: function get() {
      return 7;
    }
  }).a != 7;
});

var document = _global.document; // typeof document.createElement is 'object' in old IE

var is = _isObject(document) && _isObject(document.createElement);

var _domCreate = function _domCreate(it) {
  return is ? document.createElement(it) : {};
};

var _ie8DomDefine = !_descriptors && !_fails(function () {
  return Object.defineProperty(_domCreate('div'), 'a', {
    get: function get() {
      return 7;
    }
  }).a != 7;
});

// instead of the ES6 spec version, we didn't implement @@toPrimitive case
// and the second argument - flag - preferred type is a string

var _toPrimitive$1 = function _toPrimitive(it, S) {
  if (!_isObject(it)) return it;
  var fn, val;
  if (S && typeof (fn = it.toString) == 'function' && !_isObject(val = fn.call(it))) return val;
  if (typeof (fn = it.valueOf) == 'function' && !_isObject(val = fn.call(it))) return val;
  if (!S && typeof (fn = it.toString) == 'function' && !_isObject(val = fn.call(it))) return val;
  throw TypeError("Can't convert object to primitive value");
};

var dP = Object.defineProperty;
var f = _descriptors ? Object.defineProperty : function defineProperty(O, P, Attributes) {
  _anObject(O);
  P = _toPrimitive$1(P, true);
  _anObject(Attributes);
  if (_ie8DomDefine) try {
    return dP(O, P, Attributes);
  } catch (e) {
    /* empty */
  }
  if ('get' in Attributes || 'set' in Attributes) throw TypeError('Accessors not supported!');
  if ('value' in Attributes) O[P] = Attributes.value;
  return O;
};
var _objectDp = {
  f: f
};

var _propertyDesc = function _propertyDesc(bitmap, value) {
  return {
    enumerable: !(bitmap & 1),
    configurable: !(bitmap & 2),
    writable: !(bitmap & 4),
    value: value
  };
};

var _hide = _descriptors ? function (object, key, value) {
  return _objectDp.f(object, key, _propertyDesc(1, value));
} : function (object, key, value) {
  object[key] = value;
  return object;
};

var hasOwnProperty$1 = {}.hasOwnProperty;

var _has = function _has(it, key) {
  return hasOwnProperty$1.call(it, key);
};

var PROTOTYPE = 'prototype';

var $export = function $export(type, name, source) {
  var IS_FORCED = type & $export.F;
  var IS_GLOBAL = type & $export.G;
  var IS_STATIC = type & $export.S;
  var IS_PROTO = type & $export.P;
  var IS_BIND = type & $export.B;
  var IS_WRAP = type & $export.W;
  var exports = IS_GLOBAL ? _core : _core[name] || (_core[name] = {});
  var expProto = exports[PROTOTYPE];
  var target = IS_GLOBAL ? _global : IS_STATIC ? _global[name] : (_global[name] || {})[PROTOTYPE];
  var key, own, out;
  if (IS_GLOBAL) source = name;

  for (key in source) {
    // contains in native
    own = !IS_FORCED && target && target[key] !== undefined;
    if (own && _has(exports, key)) continue; // export native or passed

    out = own ? target[key] : source[key]; // prevent global pollution for namespaces

    exports[key] = IS_GLOBAL && typeof target[key] != 'function' ? source[key] // bind timers to global for call from export context
    : IS_BIND && own ? _ctx(out, _global) // wrap global constructors for prevent change them in library
    : IS_WRAP && target[key] == out ? function (C) {
      var F = function F(a, b, c) {
        if (this instanceof C) {
          switch (arguments.length) {
            case 0:
              return new C();

            case 1:
              return new C(a);

            case 2:
              return new C(a, b);
          }

          return new C(a, b, c);
        }

        return C.apply(this, arguments);
      };

      F[PROTOTYPE] = C[PROTOTYPE];
      return F; // make static versions for prototype methods
    }(out) : IS_PROTO && typeof out == 'function' ? _ctx(Function.call, out) : out; // export proto methods to core.%CONSTRUCTOR%.methods.%NAME%

    if (IS_PROTO) {
      (exports.virtual || (exports.virtual = {}))[key] = out; // export proto methods to core.%CONSTRUCTOR%.prototype.%NAME%

      if (type & $export.R && expProto && !expProto[key]) _hide(expProto, key, out);
    }
  }
}; // type bitmap


$export.F = 1; // forced

$export.G = 2; // global

$export.S = 4; // static

$export.P = 8; // proto

$export.B = 16; // bind

$export.W = 32; // wrap

$export.U = 64; // safe

$export.R = 128; // real proto method for `library`

var _export = $export;

var _redefine = _hide;

// 7.1.4 ToInteger
var ceil = Math.ceil;
var floor = Math.floor;

var _toInteger = function _toInteger(it) {
  return isNaN(it = +it) ? 0 : (it > 0 ? floor : ceil)(it);
};

var min = Math.min;

var _toLength = function _toLength(it) {
  return it > 0 ? min(_toInteger(it), 0x1fffffffffffff) : 0; // pow(2, 53) - 1 == 9007199254740991
};

var max = Math.max;
var min$1 = Math.min;

var _toAbsoluteIndex = function _toAbsoluteIndex(index, length) {
  index = _toInteger(index);
  return index < 0 ? max(index + length, 0) : min$1(index, length);
};

// true  -> Array#includes

var _arrayIncludes = function _arrayIncludes(IS_INCLUDES) {
  return function ($this, el, fromIndex) {
    var O = _toIobject($this);
    var length = _toLength(O.length);
    var index = _toAbsoluteIndex(fromIndex, length);
    var value; // Array#includes uses SameValueZero equality algorithm
    // eslint-disable-next-line no-self-compare

    if (IS_INCLUDES && el != el) while (length > index) {
      value = O[index++]; // eslint-disable-next-line no-self-compare

      if (value != value) return true; // Array#indexOf ignores holes, Array#includes - not
    } else for (; length > index; index++) {
      if (IS_INCLUDES || index in O) {
        if (O[index] === el) return IS_INCLUDES || index || 0;
      }
    }
    return !IS_INCLUDES && -1;
  };
};

var _shared = createCommonjsModule(function (module) {
  var SHARED = '__core-js_shared__';
  var store = _global[SHARED] || (_global[SHARED] = {});
  (module.exports = function (key, value) {
    return store[key] || (store[key] = value !== undefined ? value : {});
  })('versions', []).push({
    version: _core.version,
    mode: _library ? 'pure' : 'global',
    copyright: '© 2018 Denis Pushkarev (zloirock.ru)'
  });
});

var id$1 = 0;
var px = Math.random();

var _uid = function _uid(key) {
  return 'Symbol('.concat(key === undefined ? '' : key, ')_', (++id$1 + px).toString(36));
};

var shared = _shared('keys');

var _sharedKey = function _sharedKey(key) {
  return shared[key] || (shared[key] = _uid(key));
};

var arrayIndexOf$1 = _arrayIncludes(false);
var IE_PROTO = _sharedKey('IE_PROTO');

var _objectKeysInternal = function _objectKeysInternal(object, names) {
  var O = _toIobject(object);
  var i = 0;
  var result = [];
  var key;

  for (key in O) {
    if (key != IE_PROTO) _has(O, key) && result.push(key);
  } // Don't enum bug & hidden keys


  while (names.length > i) {
    if (_has(O, key = names[i++])) {
      ~arrayIndexOf$1(result, key) || result.push(key);
    }
  }

  return result;
};

// IE 8- don't enum bug keys
var _enumBugKeys = 'constructor,hasOwnProperty,isPrototypeOf,propertyIsEnumerable,toLocaleString,toString,valueOf'.split(',');

var _objectKeys = Object.keys || function keys(O) {
  return _objectKeysInternal(O, _enumBugKeys);
};

var _objectDps = _descriptors ? Object.defineProperties : function defineProperties(O, Properties) {
  _anObject(O);
  var keys = _objectKeys(Properties);
  var length = keys.length;
  var i = 0;
  var P;

  while (length > i) {
    _objectDp.f(O, P = keys[i++], Properties[P]);
  }

  return O;
};

var document$1 = _global.document;

var _html = document$1 && document$1.documentElement;

var IE_PROTO$1 = _sharedKey('IE_PROTO');

var Empty = function Empty() {
  /* empty */
};

var PROTOTYPE$1 = 'prototype'; // Create object with fake `null` prototype: use iframe Object with cleared prototype

var _createDict = function createDict() {
  // Thrash, waste and sodomy: IE GC bug
  var iframe = _domCreate('iframe');
  var i = _enumBugKeys.length;
  var lt = '<';
  var gt = '>';
  var iframeDocument;
  iframe.style.display = 'none';
  _html.appendChild(iframe);
  iframe.src = 'javascript:'; // eslint-disable-line no-script-url
  // createDict = iframe.contentWindow.Object;
  // html.removeChild(iframe);

  iframeDocument = iframe.contentWindow.document;
  iframeDocument.open();
  iframeDocument.write(lt + 'script' + gt + 'document.F=Object' + lt + '/script' + gt);
  iframeDocument.close();
  _createDict = iframeDocument.F;

  while (i--) {
    delete _createDict[PROTOTYPE$1][_enumBugKeys[i]];
  }

  return _createDict();
};

var _objectCreate = Object.create || function create(O, Properties) {
  var result;

  if (O !== null) {
    Empty[PROTOTYPE$1] = _anObject(O);
    result = new Empty();
    Empty[PROTOTYPE$1] = null; // add "__proto__" for Object.getPrototypeOf polyfill

    result[IE_PROTO$1] = O;
  } else result = _createDict();

  return Properties === undefined ? result : _objectDps(result, Properties);
};

var _wks = createCommonjsModule(function (module) {
  var store = _shared('wks');
  var Symbol = _global.Symbol;
  var USE_SYMBOL = typeof Symbol == 'function';

  var $exports = module.exports = function (name) {
    return store[name] || (store[name] = USE_SYMBOL && Symbol[name] || (USE_SYMBOL ? Symbol : _uid)('Symbol.' + name));
  };

  $exports.store = store;
});

var def = _objectDp.f;
var TAG = _wks('toStringTag');

var _setToStringTag = function _setToStringTag(it, tag, stat) {
  if (it && !_has(it = stat ? it : it.prototype, TAG)) def(it, TAG, {
    configurable: true,
    value: tag
  });
};

var IteratorPrototype = {}; // 25.1.2.1.1 %IteratorPrototype%[@@iterator]()

_hide(IteratorPrototype, _wks('iterator'), function () {
  return this;
});

var _iterCreate = function _iterCreate(Constructor, NAME, next) {
  Constructor.prototype = _objectCreate(IteratorPrototype, {
    next: _propertyDesc(1, next)
  });
  _setToStringTag(Constructor, NAME + ' Iterator');
};

var _toObject = function _toObject(it) {
  return Object(_defined(it));
};

var IE_PROTO$2 = _sharedKey('IE_PROTO');
var ObjectProto = Object.prototype;

var _objectGpo = Object.getPrototypeOf || function (O) {
  O = _toObject(O);
  if (_has(O, IE_PROTO$2)) return O[IE_PROTO$2];

  if (typeof O.constructor == 'function' && O instanceof O.constructor) {
    return O.constructor.prototype;
  }

  return O instanceof Object ? ObjectProto : null;
};

var ITERATOR = _wks('iterator');
var BUGGY = !([].keys && 'next' in [].keys()); // Safari has buggy iterators w/o `next`

var FF_ITERATOR = '@@iterator';
var KEYS = 'keys';
var VALUES = 'values';

var returnThis = function returnThis() {
  return this;
};

var _iterDefine = function _iterDefine(Base, NAME, Constructor, next, DEFAULT, IS_SET, FORCED) {
  _iterCreate(Constructor, NAME, next);

  var getMethod = function getMethod(kind) {
    if (!BUGGY && kind in proto) return proto[kind];

    switch (kind) {
      case KEYS:
        return function keys() {
          return new Constructor(this, kind);
        };

      case VALUES:
        return function values() {
          return new Constructor(this, kind);
        };
    }

    return function entries() {
      return new Constructor(this, kind);
    };
  };

  var TAG = NAME + ' Iterator';
  var DEF_VALUES = DEFAULT == VALUES;
  var VALUES_BUG = false;
  var proto = Base.prototype;
  var $native = proto[ITERATOR] || proto[FF_ITERATOR] || DEFAULT && proto[DEFAULT];
  var $default = $native || getMethod(DEFAULT);
  var $entries = DEFAULT ? !DEF_VALUES ? $default : getMethod('entries') : undefined;
  var $anyNative = NAME == 'Array' ? proto.entries || $native : $native;
  var methods, key, IteratorPrototype; // Fix native

  if ($anyNative) {
    IteratorPrototype = _objectGpo($anyNative.call(new Base()));

    if (IteratorPrototype !== Object.prototype && IteratorPrototype.next) {
      // Set @@toStringTag to native iterators
      _setToStringTag(IteratorPrototype, TAG, true); // fix for some old engines

      if (!_library && typeof IteratorPrototype[ITERATOR] != 'function') _hide(IteratorPrototype, ITERATOR, returnThis);
    }
  } // fix Array#{values, @@iterator}.name in V8 / FF


  if (DEF_VALUES && $native && $native.name !== VALUES) {
    VALUES_BUG = true;

    $default = function values() {
      return $native.call(this);
    };
  } // Define iterator


  if ((!_library || FORCED) && (BUGGY || VALUES_BUG || !proto[ITERATOR])) {
    _hide(proto, ITERATOR, $default);
  } // Plug for library


  _iterators[NAME] = $default;
  _iterators[TAG] = returnThis;

  if (DEFAULT) {
    methods = {
      values: DEF_VALUES ? $default : getMethod(VALUES),
      keys: IS_SET ? $default : getMethod(KEYS),
      entries: $entries
    };
    if (FORCED) for (key in methods) {
      if (!(key in proto)) _redefine(proto, key, methods[key]);
    } else _export(_export.P + _export.F * (BUGGY || VALUES_BUG), NAME, methods);
  }

  return methods;
};

// 22.1.3.13 Array.prototype.keys()
// 22.1.3.29 Array.prototype.values()
// 22.1.3.30 Array.prototype[@@iterator]()


var es6_array_iterator = _iterDefine(Array, 'Array', function (iterated, kind) {
  this._t = _toIobject(iterated); // target

  this._i = 0; // next index

  this._k = kind; // kind
  // 22.1.5.2.1 %ArrayIteratorPrototype%.next()
}, function () {
  var O = this._t;
  var kind = this._k;
  var index = this._i++;

  if (!O || index >= O.length) {
    this._t = undefined;
    return _iterStep(1);
  }

  if (kind == 'keys') return _iterStep(0, index);
  if (kind == 'values') return _iterStep(0, O[index]);
  return _iterStep(0, [index, O[index]]);
}, 'values'); // argumentsList[@@iterator] is %ArrayProto_values% (9.4.4.6, 9.4.4.7)

_iterators.Arguments = _iterators.Array;

var TO_STRING_TAG = _wks('toStringTag');
var DOMIterables = ('CSSRuleList,CSSStyleDeclaration,CSSValueList,ClientRectList,DOMRectList,DOMStringList,' + 'DOMTokenList,DataTransferItemList,FileList,HTMLAllCollection,HTMLCollection,HTMLFormElement,HTMLSelectElement,' + 'MediaList,MimeTypeArray,NamedNodeMap,NodeList,PaintRequestList,Plugin,PluginArray,SVGLengthList,SVGNumberList,' + 'SVGPathSegList,SVGPointList,SVGStringList,SVGTransformList,SourceBufferList,StyleSheetList,TextTrackCueList,' + 'TextTrackList,TouchList').split(',');

for (var i$3 = 0; i$3 < DOMIterables.length; i$3++) {
  var NAME = DOMIterables[i$3];
  var Collection = _global[NAME];
  var proto = Collection && Collection.prototype;
  if (proto && !proto[TO_STRING_TAG]) _hide(proto, TO_STRING_TAG, NAME);
  _iterators[NAME] = _iterators.Array;
}

// false -> String#codePointAt

var _stringAt = function _stringAt(TO_STRING) {
  return function (that, pos) {
    var s = String(_defined(that));
    var i = _toInteger(pos);
    var l = s.length;
    var a, b;
    if (i < 0 || i >= l) return TO_STRING ? '' : undefined;
    a = s.charCodeAt(i);
    return a < 0xd800 || a > 0xdbff || i + 1 === l || (b = s.charCodeAt(i + 1)) < 0xdc00 || b > 0xdfff ? TO_STRING ? s.charAt(i) : a : TO_STRING ? s.slice(i, i + 2) : (a - 0xd800 << 10) + (b - 0xdc00) + 0x10000;
  };
};

var $at = _stringAt(true); // 21.1.3.27 String.prototype[@@iterator]()

_iterDefine(String, 'String', function (iterated) {
  this._t = String(iterated); // target

  this._i = 0; // next index
  // 21.1.5.2.1 %StringIteratorPrototype%.next()
}, function () {
  var O = this._t;
  var index = this._i;
  var point;
  if (index >= O.length) return {
    value: undefined,
    done: true
  };
  point = $at(O, index);
  this._i += point.length;
  return {
    value: point,
    done: false
  };
});

var TAG$1 = _wks('toStringTag'); // ES3 wrong here

var ARG = _cof(function () {
  return arguments;
}()) == 'Arguments'; // fallback for IE11 Script Access Denied error

var tryGet = function tryGet(it, key) {
  try {
    return it[key];
  } catch (e) {
    /* empty */
  }
};

var _classof = function _classof(it) {
  var O, T, B;
  return it === undefined ? 'Undefined' : it === null ? 'Null' // @@toStringTag case
  : typeof (T = tryGet(O = Object(it), TAG$1)) == 'string' ? T // builtinTag case
  : ARG ? _cof(O) // ES3 arguments fallback
  : (B = _cof(O)) == 'Object' && typeof O.callee == 'function' ? 'Arguments' : B;
};

var ITERATOR$1 = _wks('iterator');

var core_isIterable = _core.isIterable = function (it) {
  var O = Object(it);
  return O[ITERATOR$1] !== undefined || '@@iterator' in O // eslint-disable-next-line no-prototype-builtins
  || _iterators.hasOwnProperty(_classof(O));
};

var isIterable = core_isIterable;

var isIterable$1 = createCommonjsModule(function (module) {
  module.exports = {
    "default": isIterable,
    __esModule: true
  };
});
unwrapExports(isIterable$1);

var ITERATOR$2 = _wks('iterator');

var core_getIteratorMethod = _core.getIteratorMethod = function (it) {
  if (it != undefined) return it[ITERATOR$2] || it['@@iterator'] || _iterators[_classof(it)];
};

var core_getIterator = _core.getIterator = function (it) {
  var iterFn = core_getIteratorMethod(it);
  if (typeof iterFn != 'function') throw TypeError(it + ' is not iterable!');
  return _anObject(iterFn.call(it));
};

var getIterator = core_getIterator;

var getIterator$1 = createCommonjsModule(function (module) {
  module.exports = {
    "default": getIterator,
    __esModule: true
  };
});
unwrapExports(getIterator$1);

var slicedToArray = createCommonjsModule(function (module, exports) {

  exports.__esModule = true;

  var _isIterable3 = _interopRequireDefault(isIterable$1);

  var _getIterator3 = _interopRequireDefault(getIterator$1);

  function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
      default: obj
    };
  }

  exports.default = function () {
    function sliceIterator(arr, i) {
      var _arr = [];
      var _n = true;
      var _d = false;
      var _e = undefined;

      try {
        for (var _i = (0, _getIterator3.default)(arr), _s; !(_n = (_s = _i.next()).done); _n = true) {
          _arr.push(_s.value);

          if (i && _arr.length === i) break;
        }
      } catch (err) {
        _d = true;
        _e = err;
      } finally {
        try {
          if (!_n && _i["return"]) _i["return"]();
        } finally {
          if (_d) throw _e;
        }
      }

      return _arr;
    }

    return function (arr, i) {
      if (Array.isArray(arr)) {
        return arr;
      } else if ((0, _isIterable3.default)(Object(arr))) {
        return sliceIterator(arr, i);
      } else {
        throw new TypeError("Invalid attempt to destructure non-iterable instance");
      }
    };
  }();
});
unwrapExports(slicedToArray);

_export(_export.S + _export.F * !_descriptors, 'Object', {
  defineProperty: _objectDp.f
});

var $Object = _core.Object;

var defineProperty = function defineProperty(it, key, desc) {
  return $Object.defineProperty(it, key, desc);
};

var defineProperty$1 = createCommonjsModule(function (module) {
  module.exports = {
    "default": defineProperty,
    __esModule: true
  };
});
unwrapExports(defineProperty$1);

var defineProperty$3 = createCommonjsModule(function (module, exports) {

  exports.__esModule = true;

  var _defineProperty2 = _interopRequireDefault(defineProperty$1);

  function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
      default: obj
    };
  }

  exports.default = function (obj, key, value) {
    if (key in obj) {
      (0, _defineProperty2.default)(obj, key, {
        value: value,
        enumerable: true,
        configurable: true,
        writable: true
      });
    } else {
      obj[key] = value;
    }

    return obj;
  };
});
unwrapExports(defineProperty$3);

var runtime = createCommonjsModule(function (module) {
  /**
   * Copyright (c) 2014-present, Facebook, Inc.
   *
   * This source code is licensed under the MIT license found in the
   * LICENSE file in the root directory of this source tree.
   */
  !function (global) {

    var Op = Object.prototype;
    var hasOwn = Op.hasOwnProperty;
    var undefined; // More compressible than void 0.

    var $Symbol = typeof Symbol === "function" ? Symbol : {};
    var iteratorSymbol = $Symbol.iterator || "@@iterator";
    var asyncIteratorSymbol = $Symbol.asyncIterator || "@@asyncIterator";
    var toStringTagSymbol = $Symbol.toStringTag || "@@toStringTag";
    var runtime = global.regeneratorRuntime;

    if (runtime) {
      {
        // If regeneratorRuntime is defined globally and we're in a module,
        // make the exports object identical to regeneratorRuntime.
        module.exports = runtime;
      } // Don't bother evaluating the rest of this file if the runtime was
      // already defined globally.


      return;
    } // Define the runtime globally (as expected by generated code) as either
    // module.exports (if we're in a module) or a new, empty object.


    runtime = global.regeneratorRuntime = module.exports;

    function wrap(innerFn, outerFn, self, tryLocsList) {
      // If outerFn provided and outerFn.prototype is a Generator, then outerFn.prototype instanceof Generator.
      var protoGenerator = outerFn && outerFn.prototype instanceof Generator ? outerFn : Generator;
      var generator = Object.create(protoGenerator.prototype);
      var context = new Context(tryLocsList || []); // The ._invoke method unifies the implementations of the .next,
      // .throw, and .return methods.

      generator._invoke = makeInvokeMethod(innerFn, self, context);
      return generator;
    }

    runtime.wrap = wrap; // Try/catch helper to minimize deoptimizations. Returns a completion
    // record like context.tryEntries[i].completion. This interface could
    // have been (and was previously) designed to take a closure to be
    // invoked without arguments, but in all the cases we care about we
    // already have an existing method we want to call, so there's no need
    // to create a new function object. We can even get away with assuming
    // the method takes exactly one argument, since that happens to be true
    // in every case, so we don't have to touch the arguments object. The
    // only additional allocation required is the completion record, which
    // has a stable shape and so hopefully should be cheap to allocate.

    function tryCatch(fn, obj, arg) {
      try {
        return {
          type: "normal",
          arg: fn.call(obj, arg)
        };
      } catch (err) {
        return {
          type: "throw",
          arg: err
        };
      }
    }

    var GenStateSuspendedStart = "suspendedStart";
    var GenStateSuspendedYield = "suspendedYield";
    var GenStateExecuting = "executing";
    var GenStateCompleted = "completed"; // Returning this object from the innerFn has the same effect as
    // breaking out of the dispatch switch statement.

    var ContinueSentinel = {}; // Dummy constructor functions that we use as the .constructor and
    // .constructor.prototype properties for functions that return Generator
    // objects. For full spec compliance, you may wish to configure your
    // minifier not to mangle the names of these two functions.

    function Generator() {}

    function GeneratorFunction() {}

    function GeneratorFunctionPrototype() {} // This is a polyfill for %IteratorPrototype% for environments that
    // don't natively support it.


    var IteratorPrototype = {};

    IteratorPrototype[iteratorSymbol] = function () {
      return this;
    };

    var getProto = Object.getPrototypeOf;
    var NativeIteratorPrototype = getProto && getProto(getProto(values([])));

    if (NativeIteratorPrototype && NativeIteratorPrototype !== Op && hasOwn.call(NativeIteratorPrototype, iteratorSymbol)) {
      // This environment has a native %IteratorPrototype%; use it instead
      // of the polyfill.
      IteratorPrototype = NativeIteratorPrototype;
    }

    var Gp = GeneratorFunctionPrototype.prototype = Generator.prototype = Object.create(IteratorPrototype);
    GeneratorFunction.prototype = Gp.constructor = GeneratorFunctionPrototype;
    GeneratorFunctionPrototype.constructor = GeneratorFunction;
    GeneratorFunctionPrototype[toStringTagSymbol] = GeneratorFunction.displayName = "GeneratorFunction"; // Helper for defining the .next, .throw, and .return methods of the
    // Iterator interface in terms of a single ._invoke method.

    function defineIteratorMethods(prototype) {
      ["next", "throw", "return"].forEach(function (method) {
        prototype[method] = function (arg) {
          return this._invoke(method, arg);
        };
      });
    }

    runtime.isGeneratorFunction = function (genFun) {
      var ctor = typeof genFun === "function" && genFun.constructor;
      return ctor ? ctor === GeneratorFunction || // For the native GeneratorFunction constructor, the best we can
      // do is to check its .name property.
      (ctor.displayName || ctor.name) === "GeneratorFunction" : false;
    };

    runtime.mark = function (genFun) {
      if (Object.setPrototypeOf) {
        Object.setPrototypeOf(genFun, GeneratorFunctionPrototype);
      } else {
        genFun.__proto__ = GeneratorFunctionPrototype;

        if (!(toStringTagSymbol in genFun)) {
          genFun[toStringTagSymbol] = "GeneratorFunction";
        }
      }

      genFun.prototype = Object.create(Gp);
      return genFun;
    }; // Within the body of any async function, `await x` is transformed to
    // `yield regeneratorRuntime.awrap(x)`, so that the runtime can test
    // `hasOwn.call(value, "__await")` to determine if the yielded value is
    // meant to be awaited.


    runtime.awrap = function (arg) {
      return {
        __await: arg
      };
    };

    function AsyncIterator(generator) {
      function invoke(method, arg, resolve, reject) {
        var record = tryCatch(generator[method], generator, arg);

        if (record.type === "throw") {
          reject(record.arg);
        } else {
          var result = record.arg;
          var value = result.value;

          if (value && typeof value === "object" && hasOwn.call(value, "__await")) {
            return Promise.resolve(value.__await).then(function (value) {
              invoke("next", value, resolve, reject);
            }, function (err) {
              invoke("throw", err, resolve, reject);
            });
          }

          return Promise.resolve(value).then(function (unwrapped) {
            // When a yielded Promise is resolved, its final value becomes
            // the .value of the Promise<{value,done}> result for the
            // current iteration. If the Promise is rejected, however, the
            // result for this iteration will be rejected with the same
            // reason. Note that rejections of yielded Promises are not
            // thrown back into the generator function, as is the case
            // when an awaited Promise is rejected. This difference in
            // behavior between yield and await is important, because it
            // allows the consumer to decide what to do with the yielded
            // rejection (swallow it and continue, manually .throw it back
            // into the generator, abandon iteration, whatever). With
            // await, by contrast, there is no opportunity to examine the
            // rejection reason outside the generator function, so the
            // only option is to throw it from the await expression, and
            // let the generator function handle the exception.
            result.value = unwrapped;
            resolve(result);
          }, reject);
        }
      }

      var previousPromise;

      function enqueue(method, arg) {
        function callInvokeWithMethodAndArg() {
          return new Promise(function (resolve, reject) {
            invoke(method, arg, resolve, reject);
          });
        }

        return previousPromise = // If enqueue has been called before, then we want to wait until
        // all previous Promises have been resolved before calling invoke,
        // so that results are always delivered in the correct order. If
        // enqueue has not been called before, then it is important to
        // call invoke immediately, without waiting on a callback to fire,
        // so that the async generator function has the opportunity to do
        // any necessary setup in a predictable way. This predictability
        // is why the Promise constructor synchronously invokes its
        // executor callback, and why async functions synchronously
        // execute code before the first await. Since we implement simple
        // async functions in terms of async generators, it is especially
        // important to get this right, even though it requires care.
        previousPromise ? previousPromise.then(callInvokeWithMethodAndArg, // Avoid propagating failures to Promises returned by later
        // invocations of the iterator.
        callInvokeWithMethodAndArg) : callInvokeWithMethodAndArg();
      } // Define the unified helper method that is used to implement .next,
      // .throw, and .return (see defineIteratorMethods).


      this._invoke = enqueue;
    }

    defineIteratorMethods(AsyncIterator.prototype);

    AsyncIterator.prototype[asyncIteratorSymbol] = function () {
      return this;
    };

    runtime.AsyncIterator = AsyncIterator; // Note that simple async functions are implemented on top of
    // AsyncIterator objects; they just return a Promise for the value of
    // the final result produced by the iterator.

    runtime.async = function (innerFn, outerFn, self, tryLocsList) {
      var iter = new AsyncIterator(wrap(innerFn, outerFn, self, tryLocsList));
      return runtime.isGeneratorFunction(outerFn) ? iter // If outerFn is a generator, return the full iterator.
      : iter.next().then(function (result) {
        return result.done ? result.value : iter.next();
      });
    };

    function makeInvokeMethod(innerFn, self, context) {
      var state = GenStateSuspendedStart;
      return function invoke(method, arg) {
        if (state === GenStateExecuting) {
          throw new Error("Generator is already running");
        }

        if (state === GenStateCompleted) {
          if (method === "throw") {
            throw arg;
          } // Be forgiving, per 25.3.3.3.3 of the spec:
          // https://people.mozilla.org/~jorendorff/es6-draft.html#sec-generatorresume


          return doneResult();
        }

        context.method = method;
        context.arg = arg;

        while (true) {
          var delegate = context.delegate;

          if (delegate) {
            var delegateResult = maybeInvokeDelegate(delegate, context);

            if (delegateResult) {
              if (delegateResult === ContinueSentinel) continue;
              return delegateResult;
            }
          }

          if (context.method === "next") {
            // Setting context._sent for legacy support of Babel's
            // function.sent implementation.
            context.sent = context._sent = context.arg;
          } else if (context.method === "throw") {
            if (state === GenStateSuspendedStart) {
              state = GenStateCompleted;
              throw context.arg;
            }

            context.dispatchException(context.arg);
          } else if (context.method === "return") {
            context.abrupt("return", context.arg);
          }

          state = GenStateExecuting;
          var record = tryCatch(innerFn, self, context);

          if (record.type === "normal") {
            // If an exception is thrown from innerFn, we leave state ===
            // GenStateExecuting and loop back for another invocation.
            state = context.done ? GenStateCompleted : GenStateSuspendedYield;

            if (record.arg === ContinueSentinel) {
              continue;
            }

            return {
              value: record.arg,
              done: context.done
            };
          } else if (record.type === "throw") {
            state = GenStateCompleted; // Dispatch the exception by looping back around to the
            // context.dispatchException(context.arg) call above.

            context.method = "throw";
            context.arg = record.arg;
          }
        }
      };
    } // Call delegate.iterator[context.method](context.arg) and handle the
    // result, either by returning a { value, done } result from the
    // delegate iterator, or by modifying context.method and context.arg,
    // setting context.delegate to null, and returning the ContinueSentinel.


    function maybeInvokeDelegate(delegate, context) {
      var method = delegate.iterator[context.method];

      if (method === undefined) {
        // A .throw or .return when the delegate iterator has no .throw
        // method always terminates the yield* loop.
        context.delegate = null;

        if (context.method === "throw") {
          if (delegate.iterator.return) {
            // If the delegate iterator has a return method, give it a
            // chance to clean up.
            context.method = "return";
            context.arg = undefined;
            maybeInvokeDelegate(delegate, context);

            if (context.method === "throw") {
              // If maybeInvokeDelegate(context) changed context.method from
              // "return" to "throw", let that override the TypeError below.
              return ContinueSentinel;
            }
          }

          context.method = "throw";
          context.arg = new TypeError("The iterator does not provide a 'throw' method");
        }

        return ContinueSentinel;
      }

      var record = tryCatch(method, delegate.iterator, context.arg);

      if (record.type === "throw") {
        context.method = "throw";
        context.arg = record.arg;
        context.delegate = null;
        return ContinueSentinel;
      }

      var info = record.arg;

      if (!info) {
        context.method = "throw";
        context.arg = new TypeError("iterator result is not an object");
        context.delegate = null;
        return ContinueSentinel;
      }

      if (info.done) {
        // Assign the result of the finished delegate to the temporary
        // variable specified by delegate.resultName (see delegateYield).
        context[delegate.resultName] = info.value; // Resume execution at the desired location (see delegateYield).

        context.next = delegate.nextLoc; // If context.method was "throw" but the delegate handled the
        // exception, let the outer generator proceed normally. If
        // context.method was "next", forget context.arg since it has been
        // "consumed" by the delegate iterator. If context.method was
        // "return", allow the original .return call to continue in the
        // outer generator.

        if (context.method !== "return") {
          context.method = "next";
          context.arg = undefined;
        }
      } else {
        // Re-yield the result returned by the delegate method.
        return info;
      } // The delegate iterator is finished, so forget it and continue with
      // the outer generator.


      context.delegate = null;
      return ContinueSentinel;
    } // Define Generator.prototype.{next,throw,return} in terms of the
    // unified ._invoke helper method.


    defineIteratorMethods(Gp);
    Gp[toStringTagSymbol] = "Generator"; // A Generator should always return itself as the iterator object when the
    // @@iterator function is called on it. Some browsers' implementations of the
    // iterator prototype chain incorrectly implement this, causing the Generator
    // object to not be returned from this call. This ensures that doesn't happen.
    // See https://github.com/facebook/regenerator/issues/274 for more details.

    Gp[iteratorSymbol] = function () {
      return this;
    };

    Gp.toString = function () {
      return "[object Generator]";
    };

    function pushTryEntry(locs) {
      var entry = {
        tryLoc: locs[0]
      };

      if (1 in locs) {
        entry.catchLoc = locs[1];
      }

      if (2 in locs) {
        entry.finallyLoc = locs[2];
        entry.afterLoc = locs[3];
      }

      this.tryEntries.push(entry);
    }

    function resetTryEntry(entry) {
      var record = entry.completion || {};
      record.type = "normal";
      delete record.arg;
      entry.completion = record;
    }

    function Context(tryLocsList) {
      // The root entry object (effectively a try statement without a catch
      // or a finally block) gives us a place to store values thrown from
      // locations where there is no enclosing try statement.
      this.tryEntries = [{
        tryLoc: "root"
      }];
      tryLocsList.forEach(pushTryEntry, this);
      this.reset(true);
    }

    runtime.keys = function (object) {
      var keys = [];

      for (var key in object) {
        keys.push(key);
      }

      keys.reverse(); // Rather than returning an object with a next method, we keep
      // things simple and return the next function itself.

      return function next() {
        while (keys.length) {
          var key = keys.pop();

          if (key in object) {
            next.value = key;
            next.done = false;
            return next;
          }
        } // To avoid creating an additional object, we just hang the .value
        // and .done properties off the next function object itself. This
        // also ensures that the minifier will not anonymize the function.


        next.done = true;
        return next;
      };
    };

    function values(iterable) {
      if (iterable) {
        var iteratorMethod = iterable[iteratorSymbol];

        if (iteratorMethod) {
          return iteratorMethod.call(iterable);
        }

        if (typeof iterable.next === "function") {
          return iterable;
        }

        if (!isNaN(iterable.length)) {
          var i = -1,
              next = function next() {
            while (++i < iterable.length) {
              if (hasOwn.call(iterable, i)) {
                next.value = iterable[i];
                next.done = false;
                return next;
              }
            }

            next.value = undefined;
            next.done = true;
            return next;
          };

          return next.next = next;
        }
      } // Return an iterator with no values.


      return {
        next: doneResult
      };
    }

    runtime.values = values;

    function doneResult() {
      return {
        value: undefined,
        done: true
      };
    }

    Context.prototype = {
      constructor: Context,
      reset: function reset(skipTempReset) {
        this.prev = 0;
        this.next = 0; // Resetting context._sent for legacy support of Babel's
        // function.sent implementation.

        this.sent = this._sent = undefined;
        this.done = false;
        this.delegate = null;
        this.method = "next";
        this.arg = undefined;
        this.tryEntries.forEach(resetTryEntry);

        if (!skipTempReset) {
          for (var name in this) {
            // Not sure about the optimal order of these conditions:
            if (name.charAt(0) === "t" && hasOwn.call(this, name) && !isNaN(+name.slice(1))) {
              this[name] = undefined;
            }
          }
        }
      },
      stop: function stop() {
        this.done = true;
        var rootEntry = this.tryEntries[0];
        var rootRecord = rootEntry.completion;

        if (rootRecord.type === "throw") {
          throw rootRecord.arg;
        }

        return this.rval;
      },
      dispatchException: function dispatchException(exception) {
        if (this.done) {
          throw exception;
        }

        var context = this;

        function handle(loc, caught) {
          record.type = "throw";
          record.arg = exception;
          context.next = loc;

          if (caught) {
            // If the dispatched exception was caught by a catch block,
            // then let that catch block handle the exception normally.
            context.method = "next";
            context.arg = undefined;
          }

          return !!caught;
        }

        for (var i = this.tryEntries.length - 1; i >= 0; --i) {
          var entry = this.tryEntries[i];
          var record = entry.completion;

          if (entry.tryLoc === "root") {
            // Exception thrown outside of any try block that could handle
            // it, so set the completion value of the entire function to
            // throw the exception.
            return handle("end");
          }

          if (entry.tryLoc <= this.prev) {
            var hasCatch = hasOwn.call(entry, "catchLoc");
            var hasFinally = hasOwn.call(entry, "finallyLoc");

            if (hasCatch && hasFinally) {
              if (this.prev < entry.catchLoc) {
                return handle(entry.catchLoc, true);
              } else if (this.prev < entry.finallyLoc) {
                return handle(entry.finallyLoc);
              }
            } else if (hasCatch) {
              if (this.prev < entry.catchLoc) {
                return handle(entry.catchLoc, true);
              }
            } else if (hasFinally) {
              if (this.prev < entry.finallyLoc) {
                return handle(entry.finallyLoc);
              }
            } else {
              throw new Error("try statement without catch or finally");
            }
          }
        }
      },
      abrupt: function abrupt(type, arg) {
        for (var i = this.tryEntries.length - 1; i >= 0; --i) {
          var entry = this.tryEntries[i];

          if (entry.tryLoc <= this.prev && hasOwn.call(entry, "finallyLoc") && this.prev < entry.finallyLoc) {
            var finallyEntry = entry;
            break;
          }
        }

        if (finallyEntry && (type === "break" || type === "continue") && finallyEntry.tryLoc <= arg && arg <= finallyEntry.finallyLoc) {
          // Ignore the finally entry if control is not jumping to a
          // location outside the try/catch block.
          finallyEntry = null;
        }

        var record = finallyEntry ? finallyEntry.completion : {};
        record.type = type;
        record.arg = arg;

        if (finallyEntry) {
          this.method = "next";
          this.next = finallyEntry.finallyLoc;
          return ContinueSentinel;
        }

        return this.complete(record);
      },
      complete: function complete(record, afterLoc) {
        if (record.type === "throw") {
          throw record.arg;
        }

        if (record.type === "break" || record.type === "continue") {
          this.next = record.arg;
        } else if (record.type === "return") {
          this.rval = this.arg = record.arg;
          this.method = "return";
          this.next = "end";
        } else if (record.type === "normal" && afterLoc) {
          this.next = afterLoc;
        }

        return ContinueSentinel;
      },
      finish: function finish(finallyLoc) {
        for (var i = this.tryEntries.length - 1; i >= 0; --i) {
          var entry = this.tryEntries[i];

          if (entry.finallyLoc === finallyLoc) {
            this.complete(entry.completion, entry.afterLoc);
            resetTryEntry(entry);
            return ContinueSentinel;
          }
        }
      },
      "catch": function _catch(tryLoc) {
        for (var i = this.tryEntries.length - 1; i >= 0; --i) {
          var entry = this.tryEntries[i];

          if (entry.tryLoc === tryLoc) {
            var record = entry.completion;

            if (record.type === "throw") {
              var thrown = record.arg;
              resetTryEntry(entry);
            }

            return thrown;
          }
        } // The context.catch method must only be called with a location
        // argument that corresponds to a known catch block.


        throw new Error("illegal catch attempt");
      },
      delegateYield: function delegateYield(iterable, resultName, nextLoc) {
        this.delegate = {
          iterator: values(iterable),
          resultName: resultName,
          nextLoc: nextLoc
        };

        if (this.method === "next") {
          // Deliberately forget the last sent value so that we don't
          // accidentally pass it on to the delegate.
          this.arg = undefined;
        }

        return ContinueSentinel;
      }
    };
  }( // In sloppy mode, unbound `this` refers to the global object, fallback to
  // Function constructor if we're in global strict mode. That is sadly a form
  // of indirect eval which violates Content Security Policy.
  function () {
    return this;
  }() || Function("return this")());
});

/**
 * Copyright (c) 2014-present, Facebook, Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
// This method of obtaining a reference to the global object needs to be
// kept identical to the way it is obtained in runtime.js

var g = function () {
  return this;
}() || Function("return this")(); // Use `getOwnPropertyNames` because not all browsers support calling
// `hasOwnProperty` on the global `self` object in a worker. See #183.


var hadRuntime = g.regeneratorRuntime && Object.getOwnPropertyNames(g).indexOf("regeneratorRuntime") >= 0; // Save the old regeneratorRuntime in case it needs to be restored later.

var oldRuntime = hadRuntime && g.regeneratorRuntime; // Force reevalutation of runtime.js.

g.regeneratorRuntime = undefined;
var runtimeModule = runtime;

if (hadRuntime) {
  // Restore the original runtime.
  g.regeneratorRuntime = oldRuntime;
} else {
  // Remove the global property added by runtime.js.
  try {
    delete g.regeneratorRuntime;
  } catch (e) {
    g.regeneratorRuntime = undefined;
  }
}

var regenerator = runtimeModule;

var f$1 = _wks;
var _wksExt = {
  f: f$1
};

var iterator = _wksExt.f('iterator');

var iterator$1 = createCommonjsModule(function (module) {
  module.exports = {
    "default": iterator,
    __esModule: true
  };
});
unwrapExports(iterator$1);

var classCallCheck = createCommonjsModule(function (module, exports) {

  exports.__esModule = true;

  exports.default = function (instance, Constructor) {
    if (!(instance instanceof Constructor)) {
      throw new TypeError("Cannot call a class as a function");
    }
  };
});
unwrapExports(classCallCheck);

var createClass = createCommonjsModule(function (module, exports) {

  exports.__esModule = true;

  var _defineProperty2 = _interopRequireDefault(defineProperty$1);

  function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
      default: obj
    };
  }

  exports.default = function () {
    function defineProperties(target, props) {
      for (var i = 0; i < props.length; i++) {
        var descriptor = props[i];
        descriptor.enumerable = descriptor.enumerable || false;
        descriptor.configurable = true;
        if ("value" in descriptor) descriptor.writable = true;
        (0, _defineProperty2.default)(target, descriptor.key, descriptor);
      }
    }

    return function (Constructor, protoProps, staticProps) {
      if (protoProps) defineProperties(Constructor.prototype, protoProps);
      if (staticProps) defineProperties(Constructor, staticProps);
      return Constructor;
    };
  }();
});
unwrapExports(createClass);

function _interopDefault$1(ex) {
  return ex && typeof ex === 'object' && 'default' in ex ? ex['default'] : ex;
}

var _slicedToArray$1 = _interopDefault$1(slicedToArray);

var _getIterator = _interopDefault$1(getIterator$1);

var _defineProperty$1 = _interopDefault$1(defineProperty$3);

var _regeneratorRuntime = _interopDefault$1(regenerator);

var _Symbol$iterator = _interopDefault$1(iterator$1);

var _classCallCheck$1 = _interopDefault$1(classCallCheck);

var _createClass$1 = _interopDefault$1(createClass);

var INITIAL_STATE = 1;
var FAIL_STATE = 0;
/**
 * A StateMachine represents a deterministic finite automaton.
 * It can perform matches over a sequence of values, similar to a regular expression.
 */

var StateMachine = function () {
  function StateMachine(dfa) {
    _classCallCheck$1(this, StateMachine);

    this.stateTable = dfa.stateTable;
    this.accepting = dfa.accepting;
    this.tags = dfa.tags;
  }
  /**
   * Returns an iterable object that yields pattern matches over the input sequence.
   * Matches are of the form [startIndex, endIndex, tags].
   */


  _createClass$1(StateMachine, [{
    key: 'match',
    value: function match(str) {
      var self = this;
      return _defineProperty$1({}, _Symbol$iterator, _regeneratorRuntime.mark(function _callee() {
        var state, startRun, lastAccepting, lastState, p, c;
        return _regeneratorRuntime.wrap(function _callee$(_context) {
          while (1) {
            switch (_context.prev = _context.next) {
              case 0:
                state = INITIAL_STATE;
                startRun = null;
                lastAccepting = null;
                lastState = null;
                p = 0;

              case 5:
                if (!(p < str.length)) {
                  _context.next = 21;
                  break;
                }

                c = str[p];
                lastState = state;
                state = self.stateTable[state][c];

                if (!(state === FAIL_STATE)) {
                  _context.next = 15;
                  break;
                }

                if (!(startRun != null && lastAccepting != null && lastAccepting >= startRun)) {
                  _context.next = 13;
                  break;
                }

                _context.next = 13;
                return [startRun, lastAccepting, self.tags[lastState]];

              case 13:
                // reset the state as if we started over from the initial state
                state = self.stateTable[INITIAL_STATE][c];
                startRun = null;

              case 15:
                // start a run if not in the failure state
                if (state !== FAIL_STATE && startRun == null) {
                  startRun = p;
                } // if accepting, mark the potential match end


                if (self.accepting[state]) {
                  lastAccepting = p;
                } // reset the state to the initial state if we get into the failure state


                if (state === FAIL_STATE) {
                  state = INITIAL_STATE;
                }

              case 18:
                p++;
                _context.next = 5;
                break;

              case 21:
                if (!(startRun != null && lastAccepting != null && lastAccepting >= startRun)) {
                  _context.next = 24;
                  break;
                }

                _context.next = 24;
                return [startRun, lastAccepting, self.tags[state]];

              case 24:
              case 'end':
                return _context.stop();
            }
          }
        }, _callee, this);
      }));
    }
    /**
     * For each match over the input sequence, action functions matching
     * the tag definitions in the input pattern are called with the startIndex,
     * endIndex, and sub-match sequence.
     */

  }, {
    key: 'apply',
    value: function apply(str, actions) {
      var _iteratorNormalCompletion = true;
      var _didIteratorError = false;
      var _iteratorError = undefined;

      try {
        for (var _iterator = _getIterator(this.match(str)), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
          var _step$value = _slicedToArray$1(_step.value, 3);

          var start = _step$value[0];
          var end = _step$value[1];
          var tags = _step$value[2];
          var _iteratorNormalCompletion2 = true;
          var _didIteratorError2 = false;
          var _iteratorError2 = undefined;

          try {
            for (var _iterator2 = _getIterator(tags), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
              var tag = _step2.value;

              if (typeof actions[tag] === 'function') {
                actions[tag](start, end, str.slice(start, end + 1));
              }
            }
          } catch (err) {
            _didIteratorError2 = true;
            _iteratorError2 = err;
          } finally {
            try {
              if (!_iteratorNormalCompletion2 && _iterator2.return) {
                _iterator2.return();
              }
            } finally {
              if (_didIteratorError2) {
                throw _iteratorError2;
              }
            }
          }
        }
      } catch (err) {
        _didIteratorError = true;
        _iteratorError = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion && _iterator.return) {
            _iterator.return();
          }
        } finally {
          if (_didIteratorError) {
            throw _iteratorError;
          }
        }
      }
    }
  }]);

  return StateMachine;
}();

var dfa = StateMachine;

// Cateories used in the OpenType spec:
// https://www.microsoft.com/typography/otfntdev/devanot/shaping.aspx
var CATEGORIES = {
  X: 1 << 0,
  C: 1 << 1,
  V: 1 << 2,
  N: 1 << 3,
  H: 1 << 4,
  ZWNJ: 1 << 5,
  ZWJ: 1 << 6,
  M: 1 << 7,
  SM: 1 << 8,
  VD: 1 << 9,
  A: 1 << 10,
  Placeholder: 1 << 11,
  Dotted_Circle: 1 << 12,
  RS: 1 << 13,
  // Register Shifter, used in Khmer OT spec.
  Coeng: 1 << 14,
  // Khmer-style Virama.
  Repha: 1 << 15,
  // Atomically-encoded logical or visual repha.
  Ra: 1 << 16,
  CM: 1 << 17,
  // Consonant-Medial.
  Symbol: 1 << 18 // Avagraha, etc that take marks (SM,A,VD).

}; // Visual positions in a syllable from left to right.

var POSITIONS = {
  Start: 1 << 0,
  Ra_To_Become_Reph: 1 << 1,
  Pre_M: 1 << 2,
  Pre_C: 1 << 3,
  Base_C: 1 << 4,
  After_Main: 1 << 5,
  Above_C: 1 << 6,
  Before_Sub: 1 << 7,
  Below_C: 1 << 8,
  After_Sub: 1 << 9,
  Before_Post: 1 << 10,
  Post_C: 1 << 11,
  After_Post: 1 << 12,
  Final_C: 1 << 13,
  SMVD: 1 << 14,
  End: 1 << 15
};
var CONSONANT_FLAGS = CATEGORIES.C | CATEGORIES.Ra | CATEGORIES.CM | CATEGORIES.V | CATEGORIES.Placeholder | CATEGORIES.Dotted_Circle;
var JOINER_FLAGS = CATEGORIES.ZWJ | CATEGORIES.ZWNJ;
var HALANT_OR_COENG_FLAGS = CATEGORIES.H | CATEGORIES.Coeng;
var INDIC_CONFIGS = {
  Default: {
    hasOldSpec: false,
    virama: 0,
    basePos: 'Last',
    rephPos: POSITIONS.Before_Post,
    rephMode: 'Implicit',
    blwfMode: 'Pre_And_Post'
  },
  Devanagari: {
    hasOldSpec: true,
    virama: 0x094D,
    basePos: 'Last',
    rephPos: POSITIONS.Before_Post,
    rephMode: 'Implicit',
    blwfMode: 'Pre_And_Post'
  },
  Bengali: {
    hasOldSpec: true,
    virama: 0x09CD,
    basePos: 'Last',
    rephPos: POSITIONS.After_Sub,
    rephMode: 'Implicit',
    blwfMode: 'Pre_And_Post'
  },
  Gurmukhi: {
    hasOldSpec: true,
    virama: 0x0A4D,
    basePos: 'Last',
    rephPos: POSITIONS.Before_Sub,
    rephMode: 'Implicit',
    blwfMode: 'Pre_And_Post'
  },
  Gujarati: {
    hasOldSpec: true,
    virama: 0x0ACD,
    basePos: 'Last',
    rephPos: POSITIONS.Before_Post,
    rephMode: 'Implicit',
    blwfMode: 'Pre_And_Post'
  },
  Oriya: {
    hasOldSpec: true,
    virama: 0x0B4D,
    basePos: 'Last',
    rephPos: POSITIONS.After_Main,
    rephMode: 'Implicit',
    blwfMode: 'Pre_And_Post'
  },
  Tamil: {
    hasOldSpec: true,
    virama: 0x0BCD,
    basePos: 'Last',
    rephPos: POSITIONS.After_Post,
    rephMode: 'Implicit',
    blwfMode: 'Pre_And_Post'
  },
  Telugu: {
    hasOldSpec: true,
    virama: 0x0C4D,
    basePos: 'Last',
    rephPos: POSITIONS.After_Post,
    rephMode: 'Explicit',
    blwfMode: 'Post_Only'
  },
  Kannada: {
    hasOldSpec: true,
    virama: 0x0CCD,
    basePos: 'Last',
    rephPos: POSITIONS.After_Post,
    rephMode: 'Implicit',
    blwfMode: 'Post_Only'
  },
  Malayalam: {
    hasOldSpec: true,
    virama: 0x0D4D,
    basePos: 'Last',
    rephPos: POSITIONS.After_Main,
    rephMode: 'Log_Repha',
    blwfMode: 'Pre_And_Post'
  },
  // Handled by UniversalShaper
  // Sinhala: {
  //   hasOldSpec: false,
  //   virama: 0x0DCA,
  //   basePos: 'Last_Sinhala',
  //   rephPos: POSITIONS.After_Main,
  //   rephMode: 'Explicit',
  //   blwfMode: 'Pre_And_Post'
  // },
  Khmer: {
    hasOldSpec: false,
    virama: 0x17D2,
    basePos: 'First',
    rephPos: POSITIONS.Ra_To_Become_Reph,
    rephMode: 'Vis_Repha',
    blwfMode: 'Pre_And_Post'
  }
}; // Additional decompositions that aren't in Unicode

var INDIC_DECOMPOSITIONS = {
  // Khmer
  0x17BE: [0x17C1, 0x17BE],
  0x17BF: [0x17C1, 0x17BF],
  0x17C0: [0x17C1, 0x17C0],
  0x17C4: [0x17C1, 0x17C4],
  0x17C5: [0x17C1, 0x17C5]
};

var base64DeflatedIndicMachine = "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";

var base64DeflatedUseData = "eJzNmFtv2zYUx7+LnvnA+6Vva4Z0K5BsWLYOmBEEsqMF3mw5kJUOQ9HvvnNIWaJsz7oWMFiT5k+Hfx2eQ9JMvySrtMxedsU62yfvFslPCUl+vP8e6gf4fHgPFX5u76C6+fAR6k933y0/+/bnfYlt1Q29m7v3m398t8iwCb0fvH34flONP3TxPb/g+37DF1Wid2H4bTC9DfA2jHioBB4Or6qM//j9/qNvsPbVJ9S+x/dj5Yc9kuQ5W+22r7v9ulzvcpj0l4QramDy0AjCpZNgBN9tjRRziKz1yFpDuKPcI8dqZI0NiEdWApCgDAdCI4mgVAfkAlIN4rSxEgwRkxIRk5wIplBecO0R14IIbpVHhnlkaIR4hHRAskG6QipG9WhJ/WjhLIGKBiQCchGStZVkfqCSGpGSMCslaEA2QpWVi1FjqmjDlQ+AFt5DrRmB75gYYbT3xFjwxGjMgnDKWzkFMXFKBmQipANCTyQkA5C2AbkGedOKaxq4ixBrUMvUNS/ySDufD2go0U6rgERAPEI6IBkhE5CKkAvINsgJ6ZkTBqDjpqKqoramhkqfYmwZgdrHC3s6pjbYWik8tRKplbyiMqbiQHXLFj0zjDoMMbaaQB10oecqair6lST7Es6dX9PlJoNnC0pmKY9kwQnsFSKhKCDgBrHEEUYJY4RxwgSB/QW7jml4zAyB1OMTRzglnBHOvRIXocGPxArmN9lJrmsvRw3HMybunYpxB5WAf7BdIAywJOEsgd0EW9sQgYucyGAJaYMHcK5ICBZESwVaeYiGvm9mmHaTm6Z3yE1sweTxhOOMdKnRI7WT6A0ROywb7xbv9K1aLV1TdT2n2kvuzKqmYVHTy2u6fqW6MO0gO+u8p+kxezRwYOlcL9NWX8c5881SArBvDHseidNdlbYljN1ewtINDAI7nFadUTij2VUGHj6dcuz8g3FSA1ZlbDFhN48rylfs4gk7MqVzuDdNqf17PNWrGaUuTzC6I8RG4vj9rYtBp+CJ3nEZJhffYby47nSxvrt0Tlo0K2keybN3LdrzqlUZma4gBO3ZozBNU15OemfpsZAmLsw5bsOTUiTU2BwFWcVbTkO3n9NKjPFa9nb7nHBXGXqmdOod/6BN0xq2pmKjOc+R0Rn5/1/ZoyvZbH8ojL3qnRUeX05uGKOFZHd6jvbjvMf82M1+Vn1kDOqIPpIkXa2y13KdvyTvFn+mm31GyuJtdDVd4doqCFGZvuD/nC8g9Mmy2P2d5U+rzdu+zIoE0Tp/zl4zqPKyxff/bpe7TRuVaf6cFs8teEbyGtC1+DEe5W/brEg3XWZD8tczpVcIP6+LdJs+QXe7ztMyuybfvgUcv0SALbPi6a/dOodmcsBOXvH49T/dPl6+";

var base64DeflatedTrie$2 = "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";

// we may be running in a browser so we make an Uint8Array

var indicMachine = JSON.parse(String.fromCharCode.apply(String, pako.inflate(base64Arraybuffer_2(base64DeflatedIndicMachine))));
var useData = JSON.parse(String.fromCharCode.apply(String, pako.inflate(base64Arraybuffer_2(base64DeflatedUseData))));
var trieData$2 = pako.inflate(base64Arraybuffer_2(base64DeflatedTrie$2));
var decompositions = useData.decompositions;
var trie$2 = new unicodeTrie(trieData$2);
var stateMachine = new dfa(indicMachine);
/**
 * The IndicShaper supports indic scripts e.g. Devanagari, Kannada, etc.
 * Based on code from Harfbuzz: https://github.com/behdad/harfbuzz/blob/master/src/hb-ot-shape-complex-indic.cc
 */

var IndicShaper =
/*#__PURE__*/
function (_DefaultShaper) {
  _inheritsLoose(IndicShaper, _DefaultShaper);

  function IndicShaper() {
    return _DefaultShaper.apply(this, arguments) || this;
  }

  IndicShaper.planFeatures = function planFeatures(plan) {
    plan.addStage(setupSyllables);
    plan.addStage(['locl', 'ccmp']);
    plan.addStage(initialReordering);
    plan.addStage('nukt');
    plan.addStage('akhn');
    plan.addStage('rphf', false);
    plan.addStage('rkrf');
    plan.addStage('pref', false);
    plan.addStage('blwf', false);
    plan.addStage('abvf', false);
    plan.addStage('half', false);
    plan.addStage('pstf', false);
    plan.addStage('vatu');
    plan.addStage('cjct');
    plan.addStage('cfar', false);
    plan.addStage(finalReordering);
    plan.addStage({
      local: ['init'],
      global: ['pres', 'abvs', 'blws', 'psts', 'haln', 'dist', 'abvm', 'blwm', 'calt', 'clig']
    }); // Setup the indic config for the selected script

    plan.unicodeScript = fromOpenType(plan.script);
    plan.indicConfig = INDIC_CONFIGS[plan.unicodeScript] || INDIC_CONFIGS.Default;
    plan.isOldSpec = plan.indicConfig.hasOldSpec && plan.script[plan.script.length - 1] !== '2'; // TODO: turn off kern (Khmer) and liga features.
  };

  IndicShaper.assignFeatures = function assignFeatures(plan, glyphs) {
    var _loop = function _loop(i) {
      var codepoint = glyphs[i].codePoints[0];
      var d = INDIC_DECOMPOSITIONS[codepoint] || decompositions[codepoint];

      if (d) {
        var decomposed = d.map(function (c) {
          var g = plan.font.glyphForCodePoint(c);
          return new GlyphInfo(plan.font, g.id, [c], glyphs[i].features);
        });
        glyphs.splice.apply(glyphs, [i, 1].concat(decomposed));
      }
    };

    // Decompose split matras
    // TODO: do this in a more general unicode normalizer
    for (var i = glyphs.length - 1; i >= 0; i--) {
      _loop(i);
    }
  };

  return IndicShaper;
}(DefaultShaper);

_defineProperty(IndicShaper, "zeroMarkWidths", 'NONE');

function indicCategory(glyph) {
  return trie$2.get(glyph.codePoints[0]) >> 8;
}

function indicPosition(glyph) {
  return 1 << (trie$2.get(glyph.codePoints[0]) & 0xff);
}

var IndicInfo = function IndicInfo(category, position, syllableType, syllable) {
  this.category = category;
  this.position = position;
  this.syllableType = syllableType;
  this.syllable = syllable;
};

function setupSyllables(font, glyphs) {
  var syllable = 0;
  var last = 0;

  for (var _iterator = stateMachine.match(glyphs.map(indicCategory)), _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
    var _ref;

    if (_isArray) {
      if (_i >= _iterator.length) break;
      _ref = _iterator[_i++];
    } else {
      _i = _iterator.next();
      if (_i.done) break;
      _ref = _i.value;
    }

    var _ref2 = _ref,
        start = _ref2[0],
        end = _ref2[1],
        tags = _ref2[2];

    if (start > last) {
      ++syllable;

      for (var _i2 = last; _i2 < start; _i2++) {
        glyphs[_i2].shaperInfo = new IndicInfo(CATEGORIES.X, POSITIONS.End, 'non_indic_cluster', syllable);
      }
    }

    ++syllable; // Create shaper info

    for (var _i3 = start; _i3 <= end; _i3++) {
      glyphs[_i3].shaperInfo = new IndicInfo(1 << indicCategory(glyphs[_i3]), indicPosition(glyphs[_i3]), tags[0], syllable);
    }

    last = end + 1;
  }

  if (last < glyphs.length) {
    ++syllable;

    for (var i = last; i < glyphs.length; i++) {
      glyphs[i].shaperInfo = new IndicInfo(CATEGORIES.X, POSITIONS.End, 'non_indic_cluster', syllable);
    }
  }
}

function isConsonant(glyph) {
  return glyph.shaperInfo.category & CONSONANT_FLAGS;
}

function isJoiner(glyph) {
  return glyph.shaperInfo.category & JOINER_FLAGS;
}

function isHalantOrCoeng(glyph) {
  return glyph.shaperInfo.category & HALANT_OR_COENG_FLAGS;
}

function wouldSubstitute(glyphs, feature) {
  for (var _iterator2 = glyphs, _isArray2 = Array.isArray(_iterator2), _i4 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
    var _glyph$features;

    var _ref3;

    if (_isArray2) {
      if (_i4 >= _iterator2.length) break;
      _ref3 = _iterator2[_i4++];
    } else {
      _i4 = _iterator2.next();
      if (_i4.done) break;
      _ref3 = _i4.value;
    }

    var glyph = _ref3;
    glyph.features = (_glyph$features = {}, _glyph$features[feature] = true, _glyph$features);
  }

  var GSUB = glyphs[0]._font._layoutEngine.engine.GSUBProcessor;
  GSUB.applyFeatures([feature], glyphs);
  return glyphs.length === 1;
}

function consonantPosition(font, consonant, virama) {
  var glyphs = [virama, consonant, virama];

  if (wouldSubstitute(glyphs.slice(0, 2), 'blwf') || wouldSubstitute(glyphs.slice(1, 3), 'blwf')) {
    return POSITIONS.Below_C;
  } else if (wouldSubstitute(glyphs.slice(0, 2), 'pstf') || wouldSubstitute(glyphs.slice(1, 3), 'pstf')) {
    return POSITIONS.Post_C;
  } else if (wouldSubstitute(glyphs.slice(0, 2), 'pref') || wouldSubstitute(glyphs.slice(1, 3), 'pref')) {
    return POSITIONS.Post_C;
  }

  return POSITIONS.Base_C;
}

function initialReordering(font, glyphs, plan) {
  var indicConfig = plan.indicConfig;
  var features = font._layoutEngine.engine.GSUBProcessor.features;
  var dottedCircle = font.glyphForCodePoint(0x25cc).id;
  var virama = font.glyphForCodePoint(indicConfig.virama).id;

  if (virama) {
    var info = new GlyphInfo(font, virama, [indicConfig.virama]);

    for (var i = 0; i < glyphs.length; i++) {
      if (glyphs[i].shaperInfo.position === POSITIONS.Base_C) {
        glyphs[i].shaperInfo.position = consonantPosition(font, glyphs[i].copy(), info);
      }
    }
  }

  for (var start = 0, end = nextSyllable(glyphs, 0); start < glyphs.length; start = end, end = nextSyllable(glyphs, start)) {
    var _glyphs$start$shaperI = glyphs[start].shaperInfo,
        category = _glyphs$start$shaperI.category,
        syllableType = _glyphs$start$shaperI.syllableType;

    if (syllableType === 'symbol_cluster' || syllableType === 'non_indic_cluster') {
      continue;
    }

    if (syllableType === 'broken_cluster' && dottedCircle) {
      var g = new GlyphInfo(font, dottedCircle, [0x25cc]);
      g.shaperInfo = new IndicInfo(1 << indicCategory(g), indicPosition(g), glyphs[start].shaperInfo.syllableType, glyphs[start].shaperInfo.syllable); // Insert after possible Repha.

      var _i5 = start;

      while (_i5 < end && glyphs[_i5].shaperInfo.category === CATEGORIES.Repha) {
        _i5++;
      }

      glyphs.splice(_i5++, 0, g);
      end++;
    } // 1. Find base consonant:
    //
    // The shaping engine finds the base consonant of the syllable, using the
    // following algorithm: starting from the end of the syllable, move backwards
    // until a consonant is found that does not have a below-base or post-base
    // form (post-base forms have to follow below-base forms), or that is not a
    // pre-base reordering Ra, or arrive at the first consonant. The consonant
    // stopped at will be the base.


    var base = end;
    var limit = start;
    var hasReph = false; // If the syllable starts with Ra + Halant (in a script that has Reph)
    // and has more than one consonant, Ra is excluded from candidates for
    // base consonants.

    if (indicConfig.rephPos !== POSITIONS.Ra_To_Become_Reph && features.rphf && start + 3 <= end && (indicConfig.rephMode === 'Implicit' && !isJoiner(glyphs[start + 2]) || indicConfig.rephMode === 'Explicit' && glyphs[start + 2].shaperInfo.category === CATEGORIES.ZWJ)) {
      // See if it matches the 'rphf' feature.
      var _g = [glyphs[start].copy(), glyphs[start + 1].copy(), glyphs[start + 2].copy()];

      if (wouldSubstitute(_g.slice(0, 2), 'rphf') || indicConfig.rephMode === 'Explicit' && wouldSubstitute(_g, 'rphf')) {
        limit += 2;

        while (limit < end && isJoiner(glyphs[limit])) {
          limit++;
        }

        base = start;
        hasReph = true;
      }
    } else if (indicConfig.rephMode === 'Log_Repha' && glyphs[start].shaperInfo.category === CATEGORIES.Repha) {
      limit++;

      while (limit < end && isJoiner(glyphs[limit])) {
        limit++;
      }

      base = start;
      hasReph = true;
    }

    switch (indicConfig.basePos) {
      case 'Last':
        {
          // starting from the end of the syllable, move backwards
          var _i6 = end;
          var seenBelow = false;

          do {
            var _info = glyphs[--_i6].shaperInfo; // until a consonant is found

            if (isConsonant(glyphs[_i6])) {
              // that does not have a below-base or post-base form
              // (post-base forms have to follow below-base forms),
              if (_info.position !== POSITIONS.Below_C && (_info.position !== POSITIONS.Post_C || seenBelow)) {
                base = _i6;
                break;
              } // or that is not a pre-base reordering Ra,
              //
              // IMPLEMENTATION NOTES:
              //
              // Our pre-base reordering Ra's are marked POS_POST_C, so will be skipped
              // by the logic above already.
              //
              // or arrive at the first consonant. The consonant stopped at will
              // be the base.


              if (_info.position === POSITIONS.Below_C) {
                seenBelow = true;
              }

              base = _i6;
            } else if (start < _i6 && _info.category === CATEGORIES.ZWJ && glyphs[_i6 - 1].shaperInfo.category === CATEGORIES.H) {
              // A ZWJ after a Halant stops the base search, and requests an explicit
              // half form.
              // A ZWJ before a Halant, requests a subjoined form instead, and hence
              // search continues.  This is particularly important for Bengali
              // sequence Ra,H,Ya that should form Ya-Phalaa by subjoining Ya.
              break;
            }
          } while (_i6 > limit);

          break;
        }

      case 'First':
        {
          // The first consonant is always the base.
          base = start; // Mark all subsequent consonants as below.

          for (var _i7 = base + 1; _i7 < end; _i7++) {
            if (isConsonant(glyphs[_i7])) {
              glyphs[_i7].shaperInfo.position = POSITIONS.Below_C;
            }
          }
        }
    } // If the syllable starts with Ra + Halant (in a script that has Reph)
    // and has more than one consonant, Ra is excluded from candidates for
    // base consonants.
    //
    //  Only do this for unforced Reph. (ie. not for Ra,H,ZWJ)


    if (hasReph && base === start && limit - base <= 2) {
      hasReph = false;
    } // 2. Decompose and reorder Matras:
    //
    // Each matra and any syllable modifier sign in the cluster are moved to the
    // appropriate position relative to the consonant(s) in the cluster. The
    // shaping engine decomposes two- or three-part matras into their constituent
    // parts before any repositioning. Matra characters are classified by which
    // consonant in a conjunct they have affinity for and are reordered to the
    // following positions:
    //
    //   o Before first half form in the syllable
    //   o After subjoined consonants
    //   o After post-form consonant
    //   o After main consonant (for above marks)
    //
    // IMPLEMENTATION NOTES:
    //
    // The normalize() routine has already decomposed matras for us, so we don't
    // need to worry about that.
    // 3.  Reorder marks to canonical order:
    //
    // Adjacent nukta and halant or nukta and vedic sign are always repositioned
    // if necessary, so that the nukta is first.
    //
    // IMPLEMENTATION NOTES:
    //
    // We don't need to do this: the normalize() routine already did this for us.
    // Reorder characters


    for (var _i8 = start; _i8 < base; _i8++) {
      var _info2 = glyphs[_i8].shaperInfo;
      _info2.position = Math.min(POSITIONS.Pre_C, _info2.position);
    }

    if (base < end) {
      glyphs[base].shaperInfo.position = POSITIONS.Base_C;
    } // Mark final consonants.  A final consonant is one appearing after a matra,
    // like in Khmer.


    for (var _i9 = base + 1; _i9 < end; _i9++) {
      if (glyphs[_i9].shaperInfo.category === CATEGORIES.M) {
        for (var j = _i9 + 1; j < end; j++) {
          if (isConsonant(glyphs[j])) {
            glyphs[j].shaperInfo.position = POSITIONS.Final_C;
            break;
          }
        }

        break;
      }
    } // Handle beginning Ra


    if (hasReph) {
      glyphs[start].shaperInfo.position = POSITIONS.Ra_To_Become_Reph;
    } // For old-style Indic script tags, move the first post-base Halant after
    // last consonant.
    //
    // Reports suggest that in some scripts Uniscribe does this only if there
    // is *not* a Halant after last consonant already (eg. Kannada), while it
    // does it unconditionally in other scripts (eg. Malayalam).  We don't
    // currently know about other scripts, so we single out Malayalam for now.
    //
    // Kannada test case:
    // U+0C9A,U+0CCD,U+0C9A,U+0CCD
    // With some versions of Lohit Kannada.
    // https://bugs.freedesktop.org/show_bug.cgi?id=59118
    //
    // Malayalam test case:
    // U+0D38,U+0D4D,U+0D31,U+0D4D,U+0D31,U+0D4D
    // With lohit-ttf-20121122/Lohit-Malayalam.ttf


    if (plan.isOldSpec) {
      var disallowDoubleHalants = plan.unicodeScript !== 'Malayalam';

      for (var _i10 = base + 1; _i10 < end; _i10++) {
        if (glyphs[_i10].shaperInfo.category === CATEGORIES.H) {
          var _j = void 0;

          for (_j = end - 1; _j > _i10; _j--) {
            if (isConsonant(glyphs[_j]) || disallowDoubleHalants && glyphs[_j].shaperInfo.category === CATEGORIES.H) {
              break;
            }
          }

          if (glyphs[_j].shaperInfo.category !== CATEGORIES.H && _j > _i10) {
            // Move Halant to after last consonant.
            var t = glyphs[_i10];
            glyphs.splice.apply(glyphs, [_i10, 0].concat(glyphs.splice(_i10 + 1, _j - _i10)));
            glyphs[_j] = t;
          }

          break;
        }
      }
    } // Attach misc marks to previous char to move with them.


    var lastPos = POSITIONS.Start;

    for (var _i11 = start; _i11 < end; _i11++) {
      var _info3 = glyphs[_i11].shaperInfo;

      if (_info3.category & (JOINER_FLAGS | CATEGORIES.N | CATEGORIES.RS | CATEGORIES.CM | HALANT_OR_COENG_FLAGS & _info3.category)) {
        _info3.position = lastPos;

        if (_info3.category === CATEGORIES.H && _info3.position === POSITIONS.Pre_M) {
          // Uniscribe doesn't move the Halant with Left Matra.
          // TEST: U+092B,U+093F,U+094DE
          // We follow.  This is important for the Sinhala
          // U+0DDA split matra since it decomposes to U+0DD9,U+0DCA
          // where U+0DD9 is a left matra and U+0DCA is the virama.
          // We don't want to move the virama with the left matra.
          // TEST: U+0D9A,U+0DDA
          for (var _j2 = _i11; _j2 > start; _j2--) {
            if (glyphs[_j2 - 1].shaperInfo.position !== POSITIONS.Pre_M) {
              _info3.position = glyphs[_j2 - 1].shaperInfo.position;
              break;
            }
          }
        }
      } else if (_info3.position !== POSITIONS.SMVD) {
        lastPos = _info3.position;
      }
    } // For post-base consonants let them own anything before them
    // since the last consonant or matra.


    var last = base;

    for (var _i12 = base + 1; _i12 < end; _i12++) {
      if (isConsonant(glyphs[_i12])) {
        for (var _j3 = last + 1; _j3 < _i12; _j3++) {
          if (glyphs[_j3].shaperInfo.position < POSITIONS.SMVD) {
            glyphs[_j3].shaperInfo.position = glyphs[_i12].shaperInfo.position;
          }
        }

        last = _i12;
      } else if (glyphs[_i12].shaperInfo.category === CATEGORIES.M) {
        last = _i12;
      }
    }

    var arr = glyphs.slice(start, end);
    arr.sort(function (a, b) {
      return a.shaperInfo.position - b.shaperInfo.position;
    });
    glyphs.splice.apply(glyphs, [start, arr.length].concat(arr)); // Find base again

    for (var _i13 = start; _i13 < end; _i13++) {
      if (glyphs[_i13].shaperInfo.position === POSITIONS.Base_C) {
        base = _i13;
        break;
      }
    } // Setup features now
    // Reph


    for (var _i14 = start; _i14 < end && glyphs[_i14].shaperInfo.position === POSITIONS.Ra_To_Become_Reph; _i14++) {
      glyphs[_i14].features.rphf = true;
    } // Pre-base


    var blwf = !plan.isOldSpec && indicConfig.blwfMode === 'Pre_And_Post';

    for (var _i15 = start; _i15 < base; _i15++) {
      glyphs[_i15].features.half = true;

      if (blwf) {
        glyphs[_i15].features.blwf = true;
      }
    } // Post-base


    for (var _i16 = base + 1; _i16 < end; _i16++) {
      glyphs[_i16].features.abvf = true;
      glyphs[_i16].features.pstf = true;
      glyphs[_i16].features.blwf = true;
    }

    if (plan.isOldSpec && plan.unicodeScript === 'Devanagari') {
      // Old-spec eye-lash Ra needs special handling.  From the
      // spec:
      //
      // "The feature 'below-base form' is applied to consonants
      // having below-base forms and following the base consonant.
      // The exception is vattu, which may appear below half forms
      // as well as below the base glyph. The feature 'below-base
      // form' will be applied to all such occurrences of Ra as well."
      //
      // Test case: U+0924,U+094D,U+0930,U+094d,U+0915
      // with Sanskrit 2003 font.
      //
      // However, note that Ra,Halant,ZWJ is the correct way to
      // request eyelash form of Ra, so we wouldbn't inhibit it
      // in that sequence.
      //
      // Test case: U+0924,U+094D,U+0930,U+094d,U+200D,U+0915
      for (var _i17 = start; _i17 + 1 < base; _i17++) {
        if (glyphs[_i17].shaperInfo.category === CATEGORIES.Ra && glyphs[_i17 + 1].shaperInfo.category === CATEGORIES.H && (_i17 + 1 === base || glyphs[_i17 + 2].shaperInfo.category === CATEGORIES.ZWJ)) {
          glyphs[_i17].features.blwf = true;
          glyphs[_i17 + 1].features.blwf = true;
        }
      }
    }

    var prefLen = 2;

    if (features.pref && base + prefLen < end) {
      // Find a Halant,Ra sequence and mark it for pre-base reordering processing.
      for (var _i18 = base + 1; _i18 + prefLen - 1 < end; _i18++) {
        var _g2 = [glyphs[_i18].copy(), glyphs[_i18 + 1].copy()];

        if (wouldSubstitute(_g2, 'pref')) {
          for (var _j4 = 0; _j4 < prefLen; _j4++) {
            glyphs[_i18++].features.pref = true;
          } // Mark the subsequent stuff with 'cfar'.  Used in Khmer.
          // Read the feature spec.
          // This allows distinguishing the following cases with MS Khmer fonts:
          // U+1784,U+17D2,U+179A,U+17D2,U+1782
          // U+1784,U+17D2,U+1782,U+17D2,U+179A


          if (features.cfar) {
            for (; _i18 < end; _i18++) {
              glyphs[_i18].features.cfar = true;
            }
          }

          break;
        }
      }
    } // Apply ZWJ/ZWNJ effects


    for (var _i19 = start + 1; _i19 < end; _i19++) {
      if (isJoiner(glyphs[_i19])) {
        var nonJoiner = glyphs[_i19].shaperInfo.category === CATEGORIES.ZWNJ;
        var _j5 = _i19;

        do {
          _j5--; // ZWJ/ZWNJ should disable CJCT.  They do that by simply
          // being there, since we don't skip them for the CJCT
          // feature (ie. F_MANUAL_ZWJ)
          // A ZWNJ disables HALF.

          if (nonJoiner) {
            delete glyphs[_j5].features.half;
          }
        } while (_j5 > start && !isConsonant(glyphs[_j5]));
      }
    }
  }
}

function finalReordering(font, glyphs, plan) {
  var indicConfig = plan.indicConfig;
  var features = font._layoutEngine.engine.GSUBProcessor.features;

  for (var start = 0, end = nextSyllable(glyphs, 0); start < glyphs.length; start = end, end = nextSyllable(glyphs, start)) {
    // 4. Final reordering:
    //
    // After the localized forms and basic shaping forms GSUB features have been
    // applied (see below), the shaping engine performs some final glyph
    // reordering before applying all the remaining font features to the entire
    // cluster.
    var tryPref = !!features.pref; // Find base again

    var base = start;

    for (; base < end; base++) {
      if (glyphs[base].shaperInfo.position >= POSITIONS.Base_C) {
        if (tryPref && base + 1 < end) {
          for (var i = base + 1; i < end; i++) {
            if (glyphs[i].features.pref) {
              if (!(glyphs[i].substituted && glyphs[i].isLigated && !glyphs[i].isMultiplied)) {
                // Ok, this was a 'pref' candidate but didn't form any.
                // Base is around here...
                base = i;

                while (base < end && isHalantOrCoeng(glyphs[base])) {
                  base++;
                }

                glyphs[base].shaperInfo.position = POSITIONS.BASE_C;
                tryPref = false;
              }

              break;
            }
          }
        } // For Malayalam, skip over unformed below- (but NOT post-) forms.


        if (plan.unicodeScript === 'Malayalam') {
          for (var _i20 = base + 1; _i20 < end; _i20++) {
            while (_i20 < end && isJoiner(glyphs[_i20])) {
              _i20++;
            }

            if (_i20 === end || !isHalantOrCoeng(glyphs[_i20])) {
              break;
            }

            _i20++; // Skip halant.

            while (_i20 < end && isJoiner(glyphs[_i20])) {
              _i20++;
            }

            if (_i20 < end && isConsonant(glyphs[_i20]) && glyphs[_i20].shaperInfo.position === POSITIONS.Below_C) {
              base = _i20;
              glyphs[base].shaperInfo.position = POSITIONS.Base_C;
            }
          }
        }

        if (start < base && glyphs[base].shaperInfo.position > POSITIONS.Base_C) {
          base--;
        }

        break;
      }
    }

    if (base === end && start < base && glyphs[base - 1].shaperInfo.category === CATEGORIES.ZWJ) {
      base--;
    }

    if (base < end) {
      while (start < base && glyphs[base].shaperInfo.category & (CATEGORIES.N | HALANT_OR_COENG_FLAGS)) {
        base--;
      }
    } // o Reorder matras:
    //
    // If a pre-base matra character had been reordered before applying basic
    // features, the glyph can be moved closer to the main consonant based on
    // whether half-forms had been formed. Actual position for the matra is
    // defined as “after last standalone halant glyph, after initial matra
    // position and before the main consonant”. If ZWJ or ZWNJ follow this
    // halant, position is moved after it.
    //


    if (start + 1 < end && start < base) {
      // Otherwise there can't be any pre-base matra characters.
      // If we lost track of base, alas, position before last thingy.
      var newPos = base === end ? base - 2 : base - 1; // Malayalam / Tamil do not have "half" forms or explicit virama forms.
      // The glyphs formed by 'half' are Chillus or ligated explicit viramas.
      // We want to position matra after them.

      if (plan.unicodeScript !== 'Malayalam' && plan.unicodeScript !== 'Tamil') {
        while (newPos > start && !(glyphs[newPos].shaperInfo.category & (CATEGORIES.M | HALANT_OR_COENG_FLAGS))) {
          newPos--;
        } // If we found no Halant we are done.
        // Otherwise only proceed if the Halant does
        // not belong to the Matra itself!


        if (isHalantOrCoeng(glyphs[newPos]) && glyphs[newPos].shaperInfo.position !== POSITIONS.Pre_M) {
          // If ZWJ or ZWNJ follow this halant, position is moved after it.
          if (newPos + 1 < end && isJoiner(glyphs[newPos + 1])) {
            newPos++;
          }
        } else {
          newPos = start; // No move.
        }
      }

      if (start < newPos && glyphs[newPos].shaperInfo.position !== POSITIONS.Pre_M) {
        // Now go see if there's actually any matras...
        for (var _i21 = newPos; _i21 > start; _i21--) {
          if (glyphs[_i21 - 1].shaperInfo.position === POSITIONS.Pre_M) {
            var oldPos = _i21 - 1;

            if (oldPos < base && base <= newPos) {
              // Shouldn't actually happen.
              base--;
            }

            var tmp = glyphs[oldPos];
            glyphs.splice.apply(glyphs, [oldPos, 0].concat(glyphs.splice(oldPos + 1, newPos - oldPos)));
            glyphs[newPos] = tmp;
            newPos--;
          }
        }
      }
    } // o Reorder reph:
    //
    // Reph’s original position is always at the beginning of the syllable,
    // (i.e. it is not reordered at the character reordering stage). However,
    // it will be reordered according to the basic-forms shaping results.
    // Possible positions for reph, depending on the script, are; after main,
    // before post-base consonant forms, and after post-base consonant forms.
    // Two cases:
    //
    // - If repha is encoded as a sequence of characters (Ra,H or Ra,H,ZWJ), then
    //   we should only move it if the sequence ligated to the repha form.
    //
    // - If repha is encoded separately and in the logical position, we should only
    //   move it if it did NOT ligate.  If it ligated, it's probably the font trying
    //   to make it work without the reordering.


    if (start + 1 < end && glyphs[start].shaperInfo.position === POSITIONS.Ra_To_Become_Reph && glyphs[start].shaperInfo.category === CATEGORIES.Repha !== (glyphs[start].isLigated && !glyphs[start].isMultiplied)) {
      var newRephPos = void 0;
      var rephPos = indicConfig.rephPos;
      var found = false; // 1. If reph should be positioned after post-base consonant forms,
      //    proceed to step 5.

      if (rephPos !== POSITIONS.After_Post) {
        //  2. If the reph repositioning class is not after post-base: target
        //     position is after the first explicit halant glyph between the
        //     first post-reph consonant and last main consonant. If ZWJ or ZWNJ
        //     are following this halant, position is moved after it. If such
        //     position is found, this is the target position. Otherwise,
        //     proceed to the next step.
        //
        //     Note: in old-implementation fonts, where classifications were
        //     fixed in shaping engine, there was no case where reph position
        //     will be found on this step.
        newRephPos = start + 1;

        while (newRephPos < base && !isHalantOrCoeng(glyphs[newRephPos])) {
          newRephPos++;
        }

        if (newRephPos < base && isHalantOrCoeng(glyphs[newRephPos])) {
          // ->If ZWJ or ZWNJ are following this halant, position is moved after it.
          if (newRephPos + 1 < base && isJoiner(glyphs[newRephPos + 1])) {
            newRephPos++;
          }

          found = true;
        } // 3. If reph should be repositioned after the main consonant: find the
        //    first consonant not ligated with main, or find the first
        //    consonant that is not a potential pre-base reordering Ra.


        if (!found && rephPos === POSITIONS.After_Main) {
          newRephPos = base;

          while (newRephPos + 1 < end && glyphs[newRephPos + 1].shaperInfo.position <= POSITIONS.After_Main) {
            newRephPos++;
          }

          found = newRephPos < end;
        } // 4. If reph should be positioned before post-base consonant, find
        //    first post-base classified consonant not ligated with main. If no
        //    consonant is found, the target position should be before the
        //    first matra, syllable modifier sign or vedic sign.
        //
        // This is our take on what step 4 is trying to say (and failing, BADLY).


        if (!found && rephPos === POSITIONS.After_Sub) {
          newRephPos = base;

          while (newRephPos + 1 < end && !(glyphs[newRephPos + 1].shaperInfo.position & (POSITIONS.Post_C | POSITIONS.After_Post | POSITIONS.SMVD))) {
            newRephPos++;
          }

          found = newRephPos < end;
        }
      } //  5. If no consonant is found in steps 3 or 4, move reph to a position
      //     immediately before the first post-base matra, syllable modifier
      //     sign or vedic sign that has a reordering class after the intended
      //     reph position. For example, if the reordering position for reph
      //     is post-main, it will skip above-base matras that also have a
      //     post-main position.


      if (!found) {
        // Copied from step 2.
        newRephPos = start + 1;

        while (newRephPos < base && !isHalantOrCoeng(glyphs[newRephPos])) {
          newRephPos++;
        }

        if (newRephPos < base && isHalantOrCoeng(glyphs[newRephPos])) {
          // ->If ZWJ or ZWNJ are following this halant, position is moved after it.
          if (newRephPos + 1 < base && isJoiner(glyphs[newRephPos + 1])) {
            newRephPos++;
          }

          found = true;
        }
      } // 6. Otherwise, reorder reph to the end of the syllable.


      if (!found) {
        newRephPos = end - 1;

        while (newRephPos > start && glyphs[newRephPos].shaperInfo.position === POSITIONS.SMVD) {
          newRephPos--;
        } // If the Reph is to be ending up after a Matra,Halant sequence,
        // position it before that Halant so it can interact with the Matra.
        // However, if it's a plain Consonant,Halant we shouldn't do that.
        // Uniscribe doesn't do this.
        // TEST: U+0930,U+094D,U+0915,U+094B,U+094D


        if (isHalantOrCoeng(glyphs[newRephPos])) {
          for (var _i22 = base + 1; _i22 < newRephPos; _i22++) {
            if (glyphs[_i22].shaperInfo.category === CATEGORIES.M) {
              newRephPos--;
            }
          }
        }
      }

      var reph = glyphs[start];
      glyphs.splice.apply(glyphs, [start, 0].concat(glyphs.splice(start + 1, newRephPos - start)));
      glyphs[newRephPos] = reph;

      if (start < base && base <= newRephPos) {
        base--;
      }
    } // o Reorder pre-base reordering consonants:
    //
    // If a pre-base reordering consonant is found, reorder it according to
    // the following rules:


    if (tryPref && base + 1 < end) {
      for (var _i23 = base + 1; _i23 < end; _i23++) {
        if (glyphs[_i23].features.pref) {
          // 1. Only reorder a glyph produced by substitution during application
          //    of the <pref> feature. (Note that a font may shape a Ra consonant with
          //    the feature generally but block it in certain contexts.)
          // Note: We just check that something got substituted.  We don't check that
          // the <pref> feature actually did it...
          //
          // Reorder pref only if it ligated.
          if (glyphs[_i23].isLigated && !glyphs[_i23].isMultiplied) {
            // 2. Try to find a target position the same way as for pre-base matra.
            //    If it is found, reorder pre-base consonant glyph.
            //
            // 3. If position is not found, reorder immediately before main
            //    consonant.
            var _newPos = base; // Malayalam / Tamil do not have "half" forms or explicit virama forms.
            // The glyphs formed by 'half' are Chillus or ligated explicit viramas.
            // We want to position matra after them.

            if (plan.unicodeScript !== 'Malayalam' && plan.unicodeScript !== 'Tamil') {
              while (_newPos > start && !(glyphs[_newPos - 1].shaperInfo.category & (CATEGORIES.M | HALANT_OR_COENG_FLAGS))) {
                _newPos--;
              } // In Khmer coeng model, a H,Ra can go *after* matras.  If it goes after a
              // split matra, it should be reordered to *before* the left part of such matra.


              if (_newPos > start && glyphs[_newPos - 1].shaperInfo.category === CATEGORIES.M) {
                var _oldPos2 = _i23;

                for (var j = base + 1; j < _oldPos2; j++) {
                  if (glyphs[j].shaperInfo.category === CATEGORIES.M) {
                    _newPos--;
                    break;
                  }
                }
              }
            }

            if (_newPos > start && isHalantOrCoeng(glyphs[_newPos - 1])) {
              // -> If ZWJ or ZWNJ follow this halant, position is moved after it.
              if (_newPos < end && isJoiner(glyphs[_newPos])) {
                _newPos++;
              }
            }

            var _oldPos = _i23;
            var _tmp = glyphs[_oldPos];
            glyphs.splice.apply(glyphs, [_newPos + 1, 0].concat(glyphs.splice(_newPos, _oldPos - _newPos)));
            glyphs[_newPos] = _tmp;

            if (_newPos <= base && base < _oldPos) {
              base++;
            }
          }

          break;
        }
      }
    } // Apply 'init' to the Left Matra if it's a word start.


    if (glyphs[start].shaperInfo.position === POSITIONS.Pre_M && (!start || !/Cf|Mn/.test(unicode.getCategory(glyphs[start - 1].codePoints[0])))) {
      glyphs[start].features.init = true;
    }
  }
}

function nextSyllable(glyphs, start) {
  if (start >= glyphs.length) return start;
  var syllable = glyphs[start].shaperInfo.syllable;

  while (++start < glyphs.length && glyphs[start].shaperInfo.syllable === syllable) {
  }

  return start;
}

var base64DeflatedTrie$3 = "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";

// we may be running in a browser so we make an Uint8Array

var useData$1 = JSON.parse(String.fromCharCode.apply(String, pako.inflate(base64Arraybuffer_2(base64DeflatedUseData))));
var trieData$3 = pako.inflate(base64Arraybuffer_2(base64DeflatedTrie$3));
var categories = useData$1.categories,
    decompositions$1 = useData$1.decompositions;
var trie$3 = new unicodeTrie(trieData$3);
var stateMachine$1 = new dfa(useData$1);
/**
 * This shaper is an implementation of the Universal Shaping Engine, which
 * uses Unicode data to shape a number of scripts without a dedicated shaping engine.
 * See https://www.microsoft.com/typography/OpenTypeDev/USE/intro.htm.
 */

var UniversalShaper =
/*#__PURE__*/
function (_DefaultShaper) {
  _inheritsLoose(UniversalShaper, _DefaultShaper);

  function UniversalShaper() {
    return _DefaultShaper.apply(this, arguments) || this;
  }

  UniversalShaper.planFeatures = function planFeatures(plan) {
    plan.addStage(setupSyllables$1); // Default glyph pre-processing group

    plan.addStage(['locl', 'ccmp', 'nukt', 'akhn']); // Reordering group

    plan.addStage(clearSubstitutionFlags);
    plan.addStage(['rphf'], false);
    plan.addStage(recordRphf);
    plan.addStage(clearSubstitutionFlags);
    plan.addStage(['pref']);
    plan.addStage(recordPref); // Orthographic unit shaping group

    plan.addStage(['rkrf', 'abvf', 'blwf', 'half', 'pstf', 'vatu', 'cjct']);
    plan.addStage(reorder); // Topographical features
    // Scripts that need this are handled by the Arabic shaper, not implemented here for now.
    // plan.addStage(['isol', 'init', 'medi', 'fina', 'med2', 'fin2', 'fin3'], false);
    // Standard topographic presentation and positional feature application

    plan.addStage(['abvs', 'blws', 'pres', 'psts', 'dist', 'abvm', 'blwm']);
  };

  UniversalShaper.assignFeatures = function assignFeatures(plan, glyphs) {
    var _loop = function _loop(i) {
      var codepoint = glyphs[i].codePoints[0];

      if (decompositions$1[codepoint]) {
        var decomposed = decompositions$1[codepoint].map(function (c) {
          var g = plan.font.glyphForCodePoint(c);
          return new GlyphInfo(plan.font, g.id, [c], glyphs[i].features);
        });
        glyphs.splice.apply(glyphs, [i, 1].concat(decomposed));
      }
    };

    // Decompose split vowels
    // TODO: do this in a more general unicode normalizer
    for (var i = glyphs.length - 1; i >= 0; i--) {
      _loop(i);
    }
  };

  return UniversalShaper;
}(DefaultShaper);

_defineProperty(UniversalShaper, "zeroMarkWidths", 'BEFORE_GPOS');

function useCategory(glyph) {
  return trie$3.get(glyph.codePoints[0]);
}

var USEInfo = function USEInfo(category, syllableType, syllable) {
  this.category = category;
  this.syllableType = syllableType;
  this.syllable = syllable;
};

function setupSyllables$1(font, glyphs) {
  var syllable = 0;

  for (var _iterator = stateMachine$1.match(glyphs.map(useCategory)), _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
    var _ref;

    if (_isArray) {
      if (_i >= _iterator.length) break;
      _ref = _iterator[_i++];
    } else {
      _i = _iterator.next();
      if (_i.done) break;
      _ref = _i.value;
    }

    var _ref2 = _ref,
        start = _ref2[0],
        end = _ref2[1],
        tags = _ref2[2];
    ++syllable; // Create shaper info

    for (var i = start; i <= end; i++) {
      glyphs[i].shaperInfo = new USEInfo(categories[useCategory(glyphs[i])], tags[0], syllable);
    } // Assign rphf feature


    var limit = glyphs[start].shaperInfo.category === 'R' ? 1 : Math.min(3, end - start);

    for (var _i2 = start; _i2 < start + limit; _i2++) {
      glyphs[_i2].features.rphf = true;
    }
  }
}

function clearSubstitutionFlags(font, glyphs) {
  for (var _iterator2 = glyphs, _isArray2 = Array.isArray(_iterator2), _i3 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
    var _ref3;

    if (_isArray2) {
      if (_i3 >= _iterator2.length) break;
      _ref3 = _iterator2[_i3++];
    } else {
      _i3 = _iterator2.next();
      if (_i3.done) break;
      _ref3 = _i3.value;
    }

    var glyph = _ref3;
    glyph.substituted = false;
  }
}

function recordRphf(font, glyphs) {
  for (var _iterator3 = glyphs, _isArray3 = Array.isArray(_iterator3), _i4 = 0, _iterator3 = _isArray3 ? _iterator3 : _iterator3[Symbol.iterator]();;) {
    var _ref4;

    if (_isArray3) {
      if (_i4 >= _iterator3.length) break;
      _ref4 = _iterator3[_i4++];
    } else {
      _i4 = _iterator3.next();
      if (_i4.done) break;
      _ref4 = _i4.value;
    }

    var glyph = _ref4;

    if (glyph.substituted && glyph.features.rphf) {
      // Mark a substituted repha.
      glyph.shaperInfo.category = 'R';
    }
  }
}

function recordPref(font, glyphs) {
  for (var _iterator4 = glyphs, _isArray4 = Array.isArray(_iterator4), _i5 = 0, _iterator4 = _isArray4 ? _iterator4 : _iterator4[Symbol.iterator]();;) {
    var _ref5;

    if (_isArray4) {
      if (_i5 >= _iterator4.length) break;
      _ref5 = _iterator4[_i5++];
    } else {
      _i5 = _iterator4.next();
      if (_i5.done) break;
      _ref5 = _i5.value;
    }

    var glyph = _ref5;

    if (glyph.substituted) {
      // Mark a substituted pref as VPre, as they behave the same way.
      glyph.shaperInfo.category = 'VPre';
    }
  }
}

function reorder(font, glyphs) {
  var dottedCircle = font.glyphForCodePoint(0x25cc).id;

  for (var start = 0, end = nextSyllable$1(glyphs, 0); start < glyphs.length; start = end, end = nextSyllable$1(glyphs, start)) {
    var i = void 0,
        j = void 0;
    var info = glyphs[start].shaperInfo;
    var type = info.syllableType; // Only a few syllable types need reordering.

    if (type !== 'virama_terminated_cluster' && type !== 'standard_cluster' && type !== 'broken_cluster') {
      continue;
    } // Insert a dotted circle glyph in broken clusters.


    if (type === 'broken_cluster' && dottedCircle) {
      var g = new GlyphInfo(font, dottedCircle, [0x25cc]);
      g.shaperInfo = info; // Insert after possible Repha.

      for (i = start; i < end && glyphs[i].shaperInfo.category === 'R'; i++) {
      }

      glyphs.splice(++i, 0, g);
      end++;
    } // Move things forward.


    if (info.category === 'R' && end - start > 1) {
      // Got a repha. Reorder it to after first base, before first halant.
      for (i = start + 1; i < end; i++) {
        info = glyphs[i].shaperInfo;

        if (isBase(info) || isHalant(glyphs[i])) {
          // If we hit a halant, move before it; otherwise it's a base: move to it's
          // place, and shift things in between backward.
          if (isHalant(glyphs[i])) {
            i--;
          }

          glyphs.splice.apply(glyphs, [start, 0].concat(glyphs.splice(start + 1, i - start), [glyphs[i]]));
          break;
        }
      }
    } // Move things back.


    for (i = start, j = end; i < end; i++) {
      info = glyphs[i].shaperInfo;

      if (isBase(info) || isHalant(glyphs[i])) {
        // If we hit a halant, move after it; otherwise it's a base: move to it's
        // place, and shift things in between backward.
        j = isHalant(glyphs[i]) ? i + 1 : i;
      } else if ((info.category === 'VPre' || info.category === 'VMPre') && j < i) {
        glyphs.splice.apply(glyphs, [j, 1, glyphs[i]].concat(glyphs.splice(j, i - j)));
      }
    }
  }
}

function nextSyllable$1(glyphs, start) {
  if (start >= glyphs.length) return start;
  var syllable = glyphs[start].shaperInfo.syllable;

  while (++start < glyphs.length && glyphs[start].shaperInfo.syllable === syllable) {
  }

  return start;
}

function isHalant(glyph) {
  return glyph.shaperInfo.category === 'H' && !glyph.isLigated;
}

function isBase(info) {
  return info.category === 'B' || info.category === 'GB';
}

var SHAPERS = {
  arab: ArabicShaper,
  // Arabic
  mong: ArabicShaper,
  // Mongolian
  syrc: ArabicShaper,
  // Syriac
  'nko ': ArabicShaper,
  // N'Ko
  phag: ArabicShaper,
  // Phags Pa
  mand: ArabicShaper,
  // Mandaic
  mani: ArabicShaper,
  // Manichaean
  phlp: ArabicShaper,
  // Psalter Pahlavi
  hang: HangulShaper,
  // Hangul
  bng2: IndicShaper,
  // Bengali
  beng: IndicShaper,
  // Bengali
  dev2: IndicShaper,
  // Devanagari
  deva: IndicShaper,
  // Devanagari
  gjr2: IndicShaper,
  // Gujarati
  gujr: IndicShaper,
  // Gujarati
  guru: IndicShaper,
  // Gurmukhi
  gur2: IndicShaper,
  // Gurmukhi
  knda: IndicShaper,
  // Kannada
  knd2: IndicShaper,
  // Kannada
  mlm2: IndicShaper,
  // Malayalam
  mlym: IndicShaper,
  // Malayalam
  ory2: IndicShaper,
  // Oriya
  orya: IndicShaper,
  // Oriya
  taml: IndicShaper,
  // Tamil
  tml2: IndicShaper,
  // Tamil
  telu: IndicShaper,
  // Telugu
  tel2: IndicShaper,
  // Telugu
  khmr: IndicShaper,
  // Khmer
  bali: UniversalShaper,
  // Balinese
  batk: UniversalShaper,
  // Batak
  brah: UniversalShaper,
  // Brahmi
  bugi: UniversalShaper,
  // Buginese
  buhd: UniversalShaper,
  // Buhid
  cakm: UniversalShaper,
  // Chakma
  cham: UniversalShaper,
  // Cham
  dupl: UniversalShaper,
  // Duployan
  egyp: UniversalShaper,
  // Egyptian Hieroglyphs
  gran: UniversalShaper,
  // Grantha
  hano: UniversalShaper,
  // Hanunoo
  java: UniversalShaper,
  // Javanese
  kthi: UniversalShaper,
  // Kaithi
  kali: UniversalShaper,
  // Kayah Li
  khar: UniversalShaper,
  // Kharoshthi
  khoj: UniversalShaper,
  // Khojki
  sind: UniversalShaper,
  // Khudawadi
  lepc: UniversalShaper,
  // Lepcha
  limb: UniversalShaper,
  // Limbu
  mahj: UniversalShaper,
  // Mahajani
  // mand: UniversalShaper, // Mandaic
  // mani: UniversalShaper, // Manichaean
  mtei: UniversalShaper,
  // Meitei Mayek
  modi: UniversalShaper,
  // Modi
  // mong: UniversalShaper, // Mongolian
  // 'nko ': UniversalShaper, // N’Ko
  hmng: UniversalShaper,
  // Pahawh Hmong
  // phag: UniversalShaper, // Phags-pa
  // phlp: UniversalShaper, // Psalter Pahlavi
  rjng: UniversalShaper,
  // Rejang
  saur: UniversalShaper,
  // Saurashtra
  shrd: UniversalShaper,
  // Sharada
  sidd: UniversalShaper,
  // Siddham
  sinh: UniversalShaper,
  // Sinhala
  sund: UniversalShaper,
  // Sundanese
  sylo: UniversalShaper,
  // Syloti Nagri
  tglg: UniversalShaper,
  // Tagalog
  tagb: UniversalShaper,
  // Tagbanwa
  tale: UniversalShaper,
  // Tai Le
  lana: UniversalShaper,
  // Tai Tham
  tavt: UniversalShaper,
  // Tai Viet
  takr: UniversalShaper,
  // Takri
  tibt: UniversalShaper,
  // Tibetan
  tfng: UniversalShaper,
  // Tifinagh
  tirh: UniversalShaper,
  // Tirhuta
  latn: DefaultShaper,
  // Latin
  DFLT: DefaultShaper // Default

};
function choose(script) {
  if (!Array.isArray(script)) {
    script = [script];
  }

  for (var _iterator = script, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
    var _ref;

    if (_isArray) {
      if (_i >= _iterator.length) break;
      _ref = _iterator[_i++];
    } else {
      _i = _iterator.next();
      if (_i.done) break;
      _ref = _i.value;
    }

    var s = _ref;
    var shaper = SHAPERS[s];

    if (shaper) {
      return shaper;
    }
  }

  return DefaultShaper;
}

var GSUBProcessor =
/*#__PURE__*/
function (_OTProcessor) {
  _inheritsLoose(GSUBProcessor, _OTProcessor);

  function GSUBProcessor() {
    return _OTProcessor.apply(this, arguments) || this;
  }

  var _proto = GSUBProcessor.prototype;

  _proto.applyLookup = function applyLookup(lookupType, table) {
    var _this = this;

    switch (lookupType) {
      case 1:
        {
          // Single Substitution
          var index = this.coverageIndex(table.coverage);

          if (index === -1) {
            return false;
          }

          var glyph = this.glyphIterator.cur;

          switch (table.version) {
            case 1:
              glyph.id = glyph.id + table.deltaGlyphID & 0xffff;
              break;

            case 2:
              glyph.id = table.substitute.get(index);
              break;
          }

          return true;
        }

      case 2:
        {
          // Multiple Substitution
          var _index = this.coverageIndex(table.coverage);

          if (_index !== -1) {
            var _this$glyphs;

            var sequence = table.sequences.get(_index);
            this.glyphIterator.cur.id = sequence[0];
            this.glyphIterator.cur.ligatureComponent = 0;
            var features = this.glyphIterator.cur.features;
            var curGlyph = this.glyphIterator.cur;
            var replacement = sequence.slice(1).map(function (gid, i) {
              var glyph = new GlyphInfo(_this.font, gid, undefined, features);
              glyph.shaperInfo = curGlyph.shaperInfo;
              glyph.isLigated = curGlyph.isLigated;
              glyph.ligatureComponent = i + 1;
              glyph.substituted = true;
              glyph.isMultiplied = true;
              return glyph;
            });

            (_this$glyphs = this.glyphs).splice.apply(_this$glyphs, [this.glyphIterator.index + 1, 0].concat(replacement));

            return true;
          }

          return false;
        }

      case 3:
        {
          // Alternate Substitution
          var _index2 = this.coverageIndex(table.coverage);

          if (_index2 !== -1) {
            var USER_INDEX = 0; // TODO

            this.glyphIterator.cur.id = table.alternateSet.get(_index2)[USER_INDEX];
            return true;
          }

          return false;
        }

      case 4:
        {
          // Ligature Substitution
          var _index3 = this.coverageIndex(table.coverage);

          if (_index3 === -1) {
            return false;
          }

          for (var _iterator = table.ligatureSets.get(_index3), _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
            var _ref;

            if (_isArray) {
              if (_i >= _iterator.length) break;
              _ref = _iterator[_i++];
            } else {
              _i = _iterator.next();
              if (_i.done) break;
              _ref = _i.value;
            }

            var ligature = _ref;
            var matched = this.sequenceMatchIndices(1, ligature.components);

            if (!matched) {
              continue;
            }

            var _curGlyph = this.glyphIterator.cur; // Concatenate all of the characters the new ligature will represent

            var characters = _curGlyph.codePoints.slice();

            for (var _iterator2 = matched, _isArray2 = Array.isArray(_iterator2), _i2 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
              var _ref2;

              if (_isArray2) {
                if (_i2 >= _iterator2.length) break;
                _ref2 = _iterator2[_i2++];
              } else {
                _i2 = _iterator2.next();
                if (_i2.done) break;
                _ref2 = _i2.value;
              }

              var _index4 = _ref2;
              characters.push.apply(characters, this.glyphs[_index4].codePoints);
            } // Create the replacement ligature glyph


            var ligatureGlyph = new GlyphInfo(this.font, ligature.glyph, characters, _curGlyph.features);
            ligatureGlyph.shaperInfo = _curGlyph.shaperInfo;
            ligatureGlyph.isLigated = true;
            ligatureGlyph.substituted = true; // From Harfbuzz:
            // - If it *is* a mark ligature, we don't allocate a new ligature id, and leave
            //   the ligature to keep its old ligature id.  This will allow it to attach to
            //   a base ligature in GPOS.  Eg. if the sequence is: LAM,LAM,SHADDA,FATHA,HEH,
            //   and LAM,LAM,HEH for a ligature, they will leave SHADDA and FATHA with a
            //   ligature id and component value of 2.  Then if SHADDA,FATHA form a ligature
            //   later, we don't want them to lose their ligature id/component, otherwise
            //   GPOS will fail to correctly position the mark ligature on top of the
            //   LAM,LAM,HEH ligature. See https://bugzilla.gnome.org/show_bug.cgi?id=676343
            //
            // - If a ligature is formed of components that some of which are also ligatures
            //   themselves, and those ligature components had marks attached to *their*
            //   components, we have to attach the marks to the new ligature component
            //   positions!  Now *that*'s tricky!  And these marks may be following the
            //   last component of the whole sequence, so we should loop forward looking
            //   for them and update them.
            //
            //   Eg. the sequence is LAM,LAM,SHADDA,FATHA,HEH, and the font first forms a
            //   'calt' ligature of LAM,HEH, leaving the SHADDA and FATHA with a ligature
            //   id and component == 1.  Now, during 'liga', the LAM and the LAM-HEH ligature
            //   form a LAM-LAM-HEH ligature.  We need to reassign the SHADDA and FATHA to
            //   the new ligature with a component value of 2.
            //
            //   This in fact happened to a font...  See https://bugzilla.gnome.org/show_bug.cgi?id=437633

            var isMarkLigature = _curGlyph.isMark;

            for (var i = 0; i < matched.length && isMarkLigature; i++) {
              isMarkLigature = this.glyphs[matched[i]].isMark;
            }

            ligatureGlyph.ligatureID = isMarkLigature ? null : this.ligatureID++;
            var lastLigID = _curGlyph.ligatureID;
            var lastNumComps = _curGlyph.codePoints.length;
            var curComps = lastNumComps;
            var idx = this.glyphIterator.index + 1; // Set ligatureID and ligatureComponent on glyphs that were skipped in the matched sequence.
            // This allows GPOS to attach marks to the correct ligature components.

            for (var _iterator3 = matched, _isArray3 = Array.isArray(_iterator3), _i3 = 0, _iterator3 = _isArray3 ? _iterator3 : _iterator3[Symbol.iterator]();;) {
              var _ref3;

              if (_isArray3) {
                if (_i3 >= _iterator3.length) break;
                _ref3 = _iterator3[_i3++];
              } else {
                _i3 = _iterator3.next();
                if (_i3.done) break;
                _ref3 = _i3.value;
              }

              var matchIndex = _ref3;

              // Don't assign new ligature components for mark ligatures (see above)
              if (isMarkLigature) {
                idx = matchIndex;
              } else {
                while (idx < matchIndex) {
                  var ligatureComponent = curComps - lastNumComps + Math.min(this.glyphs[idx].ligatureComponent || 1, lastNumComps);
                  this.glyphs[idx].ligatureID = ligatureGlyph.ligatureID;
                  this.glyphs[idx].ligatureComponent = ligatureComponent;
                  idx++;
                }
              }

              lastLigID = this.glyphs[idx].ligatureID;
              lastNumComps = this.glyphs[idx].codePoints.length;
              curComps += lastNumComps;
              idx++; // skip base glyph
            } // Adjust ligature components for any marks following


            if (lastLigID && !isMarkLigature) {
              for (var _i4 = idx; _i4 < this.glyphs.length; _i4++) {
                if (this.glyphs[_i4].ligatureID === lastLigID) {
                  var ligatureComponent = curComps - lastNumComps + Math.min(this.glyphs[_i4].ligatureComponent || 1, lastNumComps);
                  this.glyphs[_i4].ligatureComponent = ligatureComponent;
                } else {
                  break;
                }
              }
            } // Delete the matched glyphs, and replace the current glyph with the ligature glyph


            for (var _i5 = matched.length - 1; _i5 >= 0; _i5--) {
              this.glyphs.splice(matched[_i5], 1);
            }

            this.glyphs[this.glyphIterator.index] = ligatureGlyph;
            return true;
          }

          return false;
        }

      case 5:
        // Contextual Substitution
        return this.applyContext(table);

      case 6:
        // Chaining Contextual Substitution
        return this.applyChainingContext(table);

      case 7:
        // Extension Substitution
        return this.applyLookup(table.lookupType, table.extension);

      default:
        throw new Error("GSUB lookupType " + lookupType + " is not supported");
    }
  };

  return GSUBProcessor;
}(OTProcessor);

var GPOSProcessor =
/*#__PURE__*/
function (_OTProcessor) {
  _inheritsLoose(GPOSProcessor, _OTProcessor);

  function GPOSProcessor() {
    return _OTProcessor.apply(this, arguments) || this;
  }

  var _proto = GPOSProcessor.prototype;

  _proto.applyPositionValue = function applyPositionValue(sequenceIndex, value) {
    var position = this.positions[this.glyphIterator.peekIndex(sequenceIndex)];

    if (value.xAdvance != null) {
      position.xAdvance += value.xAdvance;
    }

    if (value.yAdvance != null) {
      position.yAdvance += value.yAdvance;
    }

    if (value.xPlacement != null) {
      position.xOffset += value.xPlacement;
    }

    if (value.yPlacement != null) {
      position.yOffset += value.yPlacement;
    } // Adjustments for font variations


    var variationProcessor = this.font._variationProcessor;
    var variationStore = this.font.GDEF && this.font.GDEF.itemVariationStore;

    if (variationProcessor && variationStore) {
      if (value.xPlaDevice) {
        position.xOffset += variationProcessor.getDelta(variationStore, value.xPlaDevice.a, value.xPlaDevice.b);
      }

      if (value.yPlaDevice) {
        position.yOffset += variationProcessor.getDelta(variationStore, value.yPlaDevice.a, value.yPlaDevice.b);
      }

      if (value.xAdvDevice) {
        position.xAdvance += variationProcessor.getDelta(variationStore, value.xAdvDevice.a, value.xAdvDevice.b);
      }

      if (value.yAdvDevice) {
        position.yAdvance += variationProcessor.getDelta(variationStore, value.yAdvDevice.a, value.yAdvDevice.b);
      }
    } // TODO: device tables

  };

  _proto.applyLookup = function applyLookup(lookupType, table) {
    switch (lookupType) {
      case 1:
        {
          // Single positioning value
          var index = this.coverageIndex(table.coverage);

          if (index === -1) {
            return false;
          }

          switch (table.version) {
            case 1:
              this.applyPositionValue(0, table.value);
              break;

            case 2:
              this.applyPositionValue(0, table.values.get(index));
              break;
          }

          return true;
        }

      case 2:
        {
          // Pair Adjustment Positioning
          var nextGlyph = this.glyphIterator.peek();

          if (!nextGlyph) {
            return false;
          }

          var _index = this.coverageIndex(table.coverage);

          if (_index === -1) {
            return false;
          }

          switch (table.version) {
            case 1:
              // Adjustments for glyph pairs
              var set = table.pairSets.get(_index);

              for (var _iterator = set, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
                var _ref;

                if (_isArray) {
                  if (_i >= _iterator.length) break;
                  _ref = _iterator[_i++];
                } else {
                  _i = _iterator.next();
                  if (_i.done) break;
                  _ref = _i.value;
                }

                var _pair = _ref;

                if (_pair.secondGlyph === nextGlyph.id) {
                  this.applyPositionValue(0, _pair.value1);
                  this.applyPositionValue(1, _pair.value2);
                  return true;
                }
              }

              return false;

            case 2:
              // Class pair adjustment
              var class1 = this.getClassID(this.glyphIterator.cur.id, table.classDef1);
              var class2 = this.getClassID(nextGlyph.id, table.classDef2);

              if (class1 === -1 || class2 === -1) {
                return false;
              }

              var pair = table.classRecords.get(class1).get(class2);
              this.applyPositionValue(0, pair.value1);
              this.applyPositionValue(1, pair.value2);
              return true;
          }
        }

      case 3:
        {
          // Cursive Attachment Positioning
          var nextIndex = this.glyphIterator.peekIndex();
          var _nextGlyph = this.glyphs[nextIndex];

          if (!_nextGlyph) {
            return false;
          }

          var curRecord = table.entryExitRecords[this.coverageIndex(table.coverage)];

          if (!curRecord || !curRecord.exitAnchor) {
            return false;
          }

          var nextRecord = table.entryExitRecords[this.coverageIndex(table.coverage, _nextGlyph.id)];

          if (!nextRecord || !nextRecord.entryAnchor) {
            return false;
          }

          var entry = this.getAnchor(nextRecord.entryAnchor);
          var exit = this.getAnchor(curRecord.exitAnchor);
          var cur = this.positions[this.glyphIterator.index];
          var next = this.positions[nextIndex];

          switch (this.direction) {
            case 'ltr':
              cur.xAdvance = exit.x + cur.xOffset;
              var d = entry.x + next.xOffset;
              next.xAdvance -= d;
              next.xOffset -= d;
              break;

            case 'rtl':
              d = exit.x + cur.xOffset;
              cur.xAdvance -= d;
              cur.xOffset -= d;
              next.xAdvance = entry.x + next.xOffset;
              break;
          }

          if (this.glyphIterator.flags.rightToLeft) {
            this.glyphIterator.cur.cursiveAttachment = nextIndex;
            cur.yOffset = entry.y - exit.y;
          } else {
            _nextGlyph.cursiveAttachment = this.glyphIterator.index;
            cur.yOffset = exit.y - entry.y;
          }

          return true;
        }

      case 4:
        {
          // Mark to base positioning
          var markIndex = this.coverageIndex(table.markCoverage);

          if (markIndex === -1) {
            return false;
          } // search backward for a base glyph


          var baseGlyphIndex = this.glyphIterator.index;

          while (--baseGlyphIndex >= 0 && (this.glyphs[baseGlyphIndex].isMark || this.glyphs[baseGlyphIndex].ligatureComponent > 0)) {
          }

          if (baseGlyphIndex < 0) {
            return false;
          }

          var baseIndex = this.coverageIndex(table.baseCoverage, this.glyphs[baseGlyphIndex].id);

          if (baseIndex === -1) {
            return false;
          }

          var markRecord = table.markArray[markIndex];
          var baseAnchor = table.baseArray[baseIndex][markRecord.class];
          this.applyAnchor(markRecord, baseAnchor, baseGlyphIndex);
          return true;
        }

      case 5:
        {
          // Mark to ligature positioning
          var _markIndex = this.coverageIndex(table.markCoverage);

          if (_markIndex === -1) {
            return false;
          } // search backward for a base glyph


          var _baseGlyphIndex = this.glyphIterator.index;

          while (--_baseGlyphIndex >= 0 && this.glyphs[_baseGlyphIndex].isMark) {
          }

          if (_baseGlyphIndex < 0) {
            return false;
          }

          var ligIndex = this.coverageIndex(table.ligatureCoverage, this.glyphs[_baseGlyphIndex].id);

          if (ligIndex === -1) {
            return false;
          }

          var ligAttach = table.ligatureArray[ligIndex];
          var markGlyph = this.glyphIterator.cur;
          var ligGlyph = this.glyphs[_baseGlyphIndex];
          var compIndex = ligGlyph.ligatureID && ligGlyph.ligatureID === markGlyph.ligatureID && markGlyph.ligatureComponent > 0 ? Math.min(markGlyph.ligatureComponent, ligGlyph.codePoints.length) - 1 : ligGlyph.codePoints.length - 1;
          var _markRecord = table.markArray[_markIndex];
          var _baseAnchor = ligAttach[compIndex][_markRecord.class];
          this.applyAnchor(_markRecord, _baseAnchor, _baseGlyphIndex);
          return true;
        }

      case 6:
        {
          // Mark to mark positioning
          var mark1Index = this.coverageIndex(table.mark1Coverage);

          if (mark1Index === -1) {
            return false;
          } // get the previous mark to attach to


          var prevIndex = this.glyphIterator.peekIndex(-1);
          var prev = this.glyphs[prevIndex];

          if (!prev || !prev.isMark) {
            return false;
          }

          var _cur = this.glyphIterator.cur; // The following logic was borrowed from Harfbuzz

          var good = false;

          if (_cur.ligatureID === prev.ligatureID) {
            if (!_cur.ligatureID) {
              // Marks belonging to the same base
              good = true;
            } else if (_cur.ligatureComponent === prev.ligatureComponent) {
              // Marks belonging to the same ligature component
              good = true;
            }
          } else {
            // If ligature ids don't match, it may be the case that one of the marks
            // itself is a ligature, in which case match.
            if (_cur.ligatureID && !_cur.ligatureComponent || prev.ligatureID && !prev.ligatureComponent) {
              good = true;
            }
          }

          if (!good) {
            return false;
          }

          var mark2Index = this.coverageIndex(table.mark2Coverage, prev.id);

          if (mark2Index === -1) {
            return false;
          }

          var _markRecord2 = table.mark1Array[mark1Index];
          var _baseAnchor2 = table.mark2Array[mark2Index][_markRecord2.class];
          this.applyAnchor(_markRecord2, _baseAnchor2, prevIndex);
          return true;
        }

      case 7:
        // Contextual positioning
        return this.applyContext(table);

      case 8:
        // Chaining contextual positioning
        return this.applyChainingContext(table);

      case 9:
        // Extension positioning
        return this.applyLookup(table.lookupType, table.extension);

      default:
        throw new Error("Unsupported GPOS table: " + lookupType);
    }
  };

  _proto.applyAnchor = function applyAnchor(markRecord, baseAnchor, baseGlyphIndex) {
    var baseCoords = this.getAnchor(baseAnchor);
    var markCoords = this.getAnchor(markRecord.markAnchor);
    var basePos = this.positions[baseGlyphIndex];
    var markPos = this.positions[this.glyphIterator.index];
    markPos.xOffset = baseCoords.x - markCoords.x;
    markPos.yOffset = baseCoords.y - markCoords.y;
    this.glyphIterator.cur.markAttachment = baseGlyphIndex;
  };

  _proto.getAnchor = function getAnchor(anchor) {
    // TODO: contour point, device tables
    var x = anchor.xCoordinate;
    var y = anchor.yCoordinate; // Adjustments for font variations

    var variationProcessor = this.font._variationProcessor;
    var variationStore = this.font.GDEF && this.font.GDEF.itemVariationStore;

    if (variationProcessor && variationStore) {
      if (anchor.xDeviceTable) {
        x += variationProcessor.getDelta(variationStore, anchor.xDeviceTable.a, anchor.xDeviceTable.b);
      }

      if (anchor.yDeviceTable) {
        y += variationProcessor.getDelta(variationStore, anchor.yDeviceTable.a, anchor.yDeviceTable.b);
      }
    }

    return {
      x: x,
      y: y
    };
  };

  _proto.applyFeatures = function applyFeatures(userFeatures, glyphs, advances) {
    _OTProcessor.prototype.applyFeatures.call(this, userFeatures, glyphs, advances);

    for (var i = 0; i < this.glyphs.length; i++) {
      this.fixCursiveAttachment(i);
    }

    this.fixMarkAttachment();
  };

  _proto.fixCursiveAttachment = function fixCursiveAttachment(i) {
    var glyph = this.glyphs[i];

    if (glyph.cursiveAttachment != null) {
      var j = glyph.cursiveAttachment;
      glyph.cursiveAttachment = null;
      this.fixCursiveAttachment(j);
      this.positions[i].yOffset += this.positions[j].yOffset;
    }
  };

  _proto.fixMarkAttachment = function fixMarkAttachment() {
    for (var i = 0; i < this.glyphs.length; i++) {
      var glyph = this.glyphs[i];

      if (glyph.markAttachment != null) {
        var j = glyph.markAttachment;
        this.positions[i].xOffset += this.positions[j].xOffset;
        this.positions[i].yOffset += this.positions[j].yOffset;

        if (this.direction === 'ltr') {
          for (var k = j; k < i; k++) {
            this.positions[i].xOffset -= this.positions[k].xAdvance;
            this.positions[i].yOffset -= this.positions[k].yAdvance;
          }
        } else {
          for (var _k = j + 1; _k < i + 1; _k++) {
            this.positions[i].xOffset += this.positions[_k].xAdvance;
            this.positions[i].yOffset += this.positions[_k].yAdvance;
          }
        }
      }
    }
  };

  return GPOSProcessor;
}(OTProcessor);

var OTLayoutEngine =
/*#__PURE__*/
function () {
  function OTLayoutEngine(font) {
    this.font = font;
    this.glyphInfos = null;
    this.plan = null;
    this.GSUBProcessor = null;
    this.GPOSProcessor = null;
    this.fallbackPosition = true;

    if (font.GSUB) {
      this.GSUBProcessor = new GSUBProcessor(font, font.GSUB);
    }

    if (font.GPOS) {
      this.GPOSProcessor = new GPOSProcessor(font, font.GPOS);
    }
  }

  var _proto = OTLayoutEngine.prototype;

  _proto.setup = function setup(glyphRun) {
    var _this = this;

    // Map glyphs to GlyphInfo objects so data can be passed between
    // GSUB and GPOS without mutating the real (shared) Glyph objects.
    this.glyphInfos = glyphRun.glyphs.map(function (glyph) {
      return new GlyphInfo(_this.font, glyph.id, [].concat(glyph.codePoints));
    }); // Select a script based on what is available in GSUB/GPOS.

    var script = null;

    if (this.GPOSProcessor) {
      script = this.GPOSProcessor.selectScript(glyphRun.script, glyphRun.language, glyphRun.direction);
    }

    if (this.GSUBProcessor) {
      script = this.GSUBProcessor.selectScript(glyphRun.script, glyphRun.language, glyphRun.direction);
    } // Choose a shaper based on the script, and setup a shaping plan.
    // This determines which features to apply to which glyphs.


    this.shaper = choose(script);
    this.plan = new ShapingPlan(this.font, script, glyphRun.direction);
    this.shaper.plan(this.plan, this.glyphInfos, glyphRun.features); // Assign chosen features to output glyph run

    for (var key in this.plan.allFeatures) {
      glyphRun.features[key] = true;
    }
  };

  _proto.substitute = function substitute(glyphRun) {
    var _this2 = this;

    if (this.GSUBProcessor) {
      this.plan.process(this.GSUBProcessor, this.glyphInfos); // Map glyph infos back to normal Glyph objects

      glyphRun.glyphs = this.glyphInfos.map(function (glyphInfo) {
        return _this2.font.getGlyph(glyphInfo.id, glyphInfo.codePoints);
      });
    }
  };

  _proto.position = function position(glyphRun) {
    if (this.shaper.zeroMarkWidths === 'BEFORE_GPOS') {
      this.zeroMarkAdvances(glyphRun.positions);
    }

    if (this.GPOSProcessor) {
      this.plan.process(this.GPOSProcessor, this.glyphInfos, glyphRun.positions);
    }

    if (this.shaper.zeroMarkWidths === 'AFTER_GPOS') {
      this.zeroMarkAdvances(glyphRun.positions);
    } // Reverse the glyphs and positions if the script is right-to-left


    if (glyphRun.direction === 'rtl') {
      glyphRun.glyphs.reverse();
      glyphRun.positions.reverse();
    }

    return this.GPOSProcessor && this.GPOSProcessor.features;
  };

  _proto.zeroMarkAdvances = function zeroMarkAdvances(positions) {
    for (var i = 0; i < this.glyphInfos.length; i++) {
      if (this.glyphInfos[i].isMark) {
        positions[i].xAdvance = 0;
        positions[i].yAdvance = 0;
      }
    }
  };

  _proto.cleanup = function cleanup() {
    this.glyphInfos = null;
    this.plan = null;
    this.shaper = null;
  };

  _proto.getAvailableFeatures = function getAvailableFeatures(script, language) {
    var features = [];

    if (this.GSUBProcessor) {
      this.GSUBProcessor.selectScript(script, language);
      features.push.apply(features, Object.keys(this.GSUBProcessor.features));
    }

    if (this.GPOSProcessor) {
      this.GPOSProcessor.selectScript(script, language);
      features.push.apply(features, Object.keys(this.GPOSProcessor.features));
    }

    return features;
  };

  return OTLayoutEngine;
}();

var LayoutEngine =
/*#__PURE__*/
function () {
  function LayoutEngine(font) {
    this.font = font;
    this.unicodeLayoutEngine = null;
    this.kernProcessor = null; // Choose an advanced layout engine. We try the AAT morx table first since more
    // scripts are currently supported because the shaping logic is built into the font.

    if (this.font.morx) {
      this.engine = new AATLayoutEngine(this.font);
    } else if (this.font.GSUB || this.font.GPOS) {
      this.engine = new OTLayoutEngine(this.font);
    }
  }

  var _proto = LayoutEngine.prototype;

  _proto.layout = function layout(string, features, script, language, direction$$1) {
    // Make the features parameter optional
    if (typeof features === 'string') {
      direction$$1 = language;
      language = script;
      script = features;
      features = [];
    } // Map string to glyphs if needed


    if (typeof string === 'string') {
      // Attempt to detect the script from the string if not provided.
      if (script == null) {
        script = forString(string);
      }

      var glyphs = this.font.glyphsForString(string);
    } else {
      // Attempt to detect the script from the glyph code points if not provided.
      if (script == null) {
        var codePoints = [];

        for (var _iterator = string, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
          var _ref;

          if (_isArray) {
            if (_i >= _iterator.length) break;
            _ref = _iterator[_i++];
          } else {
            _i = _iterator.next();
            if (_i.done) break;
            _ref = _i.value;
          }

          var glyph = _ref;
          codePoints.push.apply(codePoints, glyph.codePoints);
        }

        script = forCodePoints(codePoints);
      }

      var glyphs = string;
    }

    var glyphRun = new GlyphRun(glyphs, features, script, language, direction$$1); // Return early if there are no glyphs

    if (glyphs.length === 0) {
      glyphRun.positions = [];
      return glyphRun;
    } // Setup the advanced layout engine


    if (this.engine && this.engine.setup) {
      this.engine.setup(glyphRun);
    } // Substitute and position the glyphs


    this.substitute(glyphRun);
    this.position(glyphRun);
    this.hideDefaultIgnorables(glyphRun.glyphs, glyphRun.positions); // Let the layout engine clean up any state it might have

    if (this.engine && this.engine.cleanup) {
      this.engine.cleanup();
    }

    return glyphRun;
  };

  _proto.substitute = function substitute(glyphRun) {
    // Call the advanced layout engine to make substitutions
    if (this.engine && this.engine.substitute) {
      this.engine.substitute(glyphRun);
    }
  };

  _proto.position = function position(glyphRun) {
    // Get initial glyph positions
    glyphRun.positions = glyphRun.glyphs.map(function (glyph) {
      return new GlyphPosition(glyph.advanceWidth);
    });
    var positioned = null; // Call the advanced layout engine. Returns the features applied.

    if (this.engine && this.engine.position) {
      positioned = this.engine.position(glyphRun);
    } // if there is no GPOS table, use unicode properties to position marks.


    if (!positioned && (!this.engine || this.engine.fallbackPosition)) {
      if (!this.unicodeLayoutEngine) {
        this.unicodeLayoutEngine = new UnicodeLayoutEngine(this.font);
      }

      this.unicodeLayoutEngine.positionGlyphs(glyphRun.glyphs, glyphRun.positions);
    } // if kerning is not supported by GPOS, do kerning with the TrueType/AAT kern table


    if ((!positioned || !positioned.kern) && glyphRun.features.kern !== false && this.font.kern) {
      if (!this.kernProcessor) {
        this.kernProcessor = new KernProcessor(this.font);
      }

      this.kernProcessor.process(glyphRun.glyphs, glyphRun.positions);
      glyphRun.features.kern = true;
    }
  };

  _proto.hideDefaultIgnorables = function hideDefaultIgnorables(glyphs, positions) {
    var space = this.font.glyphForCodePoint(0x20);

    for (var i = 0; i < glyphs.length; i++) {
      if (this.isDefaultIgnorable(glyphs[i].codePoints[0])) {
        glyphs[i] = space;
        positions[i].xAdvance = 0;
        positions[i].yAdvance = 0;
      }
    }
  };

  _proto.isDefaultIgnorable = function isDefaultIgnorable(ch) {
    // From DerivedCoreProperties.txt in the Unicode database,
    // minus U+115F, U+1160, U+3164 and U+FFA0, which is what
    // Harfbuzz and Uniscribe do.
    var plane = ch >> 16;

    if (plane === 0) {
      // BMP
      switch (ch >> 8) {
        case 0x00:
          return ch === 0x00AD;

        case 0x03:
          return ch === 0x034F;

        case 0x06:
          return ch === 0x061C;

        case 0x17:
          return 0x17B4 <= ch && ch <= 0x17B5;

        case 0x18:
          return 0x180B <= ch && ch <= 0x180E;

        case 0x20:
          return 0x200B <= ch && ch <= 0x200F || 0x202A <= ch && ch <= 0x202E || 0x2060 <= ch && ch <= 0x206F;

        case 0xFE:
          return 0xFE00 <= ch && ch <= 0xFE0F || ch === 0xFEFF;

        case 0xFF:
          return 0xFFF0 <= ch && ch <= 0xFFF8;

        default:
          return false;
      }
    } else {
      // Other planes
      switch (plane) {
        case 0x01:
          return 0x1BCA0 <= ch && ch <= 0x1BCA3 || 0x1D173 <= ch && ch <= 0x1D17A;

        case 0x0E:
          return 0xE0000 <= ch && ch <= 0xE0FFF;

        default:
          return false;
      }
    }
  };

  _proto.getAvailableFeatures = function getAvailableFeatures(script, language) {
    var features = [];

    if (this.engine) {
      features.push.apply(features, this.engine.getAvailableFeatures(script, language));
    }

    if (this.font.kern && features.indexOf('kern') === -1) {
      features.push('kern');
    }

    return features;
  };

  _proto.stringsForGlyph = function stringsForGlyph(gid) {
    var result = new Set();

    var codePoints = this.font._cmapProcessor.codePointsForGlyph(gid);

    for (var _iterator2 = codePoints, _isArray2 = Array.isArray(_iterator2), _i2 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
      var _ref2;

      if (_isArray2) {
        if (_i2 >= _iterator2.length) break;
        _ref2 = _iterator2[_i2++];
      } else {
        _i2 = _iterator2.next();
        if (_i2.done) break;
        _ref2 = _i2.value;
      }

      var codePoint = _ref2;
      result.add(String.fromCodePoint(codePoint));
    }

    if (this.engine && this.engine.stringsForGlyph) {
      for (var _iterator3 = this.engine.stringsForGlyph(gid), _isArray3 = Array.isArray(_iterator3), _i3 = 0, _iterator3 = _isArray3 ? _iterator3 : _iterator3[Symbol.iterator]();;) {
        var _ref3;

        if (_isArray3) {
          if (_i3 >= _iterator3.length) break;
          _ref3 = _iterator3[_i3++];
        } else {
          _i3 = _iterator3.next();
          if (_i3.done) break;
          _ref3 = _i3.value;
        }

        var string = _ref3;
        result.add(string);
      }
    }

    return Array.from(result);
  };

  return LayoutEngine;
}();

var SVG_COMMANDS = {
  moveTo: 'M',
  lineTo: 'L',
  quadraticCurveTo: 'Q',
  bezierCurveTo: 'C',
  closePath: 'Z'
};
/**
 * Path objects are returned by glyphs and represent the actual
 * vector outlines for each glyph in the font. Paths can be converted
 * to SVG path data strings, or to functions that can be applied to
 * render the path to a graphics context.
 */

var Path =
/*#__PURE__*/
function () {
  function Path() {
    this.commands = [];
    this._bbox = null;
    this._cbox = null;
  }
  /**
   * Compiles the path to a JavaScript function that can be applied with
   * a graphics context in order to render the path.
   * @return {string}
   */


  var _proto = Path.prototype;

  _proto.toFunction = function toFunction() {
    var _this = this;

    return function (ctx) {
      return _this.commands.forEach(function (c) {
        return ctx[c.command].apply(ctx, c.args);
      });
    };
  }
  /**
   * Converts the path to an SVG path data string
   * @return {string}
   */
  ;

  _proto.toSVG = function toSVG() {
    var cmds = this.commands.map(function (c) {
      var args = c.args.map(function (arg) {
        return Math.round(arg * 100) / 100;
      });
      return "" + SVG_COMMANDS[c.command] + args.join(' ');
    });
    return cmds.join('');
  }
  /**
   * Gets the "control box" of a path.
   * This is like the bounding box, but it includes all points including
   * control points of bezier segments and is much faster to compute than
   * the real bounding box.
   * @type {BBox}
   */
  ;

  /**
   * Applies a mapping function to each point in the path.
   * @param {function} fn
   * @return {Path}
   */
  _proto.mapPoints = function mapPoints(fn) {
    var path = new Path();

    for (var _iterator = this.commands, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
      var _ref;

      if (_isArray) {
        if (_i >= _iterator.length) break;
        _ref = _iterator[_i++];
      } else {
        _i = _iterator.next();
        if (_i.done) break;
        _ref = _i.value;
      }

      var c = _ref;
      var args = [];

      for (var i = 0; i < c.args.length; i += 2) {
        var _fn = fn(c.args[i], c.args[i + 1]),
            x = _fn[0],
            y = _fn[1];

        args.push(x, y);
      }

      path[c.command].apply(path, args);
    }

    return path;
  }
  /**
   * Transforms the path by the given matrix.
   */
  ;

  _proto.transform = function transform(m0, m1, m2, m3, m4, m5) {
    return this.mapPoints(function (x, y) {
      x = m0 * x + m2 * y + m4;
      y = m1 * x + m3 * y + m5;
      return [x, y];
    });
  }
  /**
   * Translates the path by the given offset.
   */
  ;

  _proto.translate = function translate(x, y) {
    return this.transform(1, 0, 0, 1, x, y);
  }
  /**
   * Rotates the path by the given angle (in radians).
   */
  ;

  _proto.rotate = function rotate(angle) {
    var cos = Math.cos(angle);
    var sin = Math.sin(angle);
    return this.transform(cos, sin, -sin, cos, 0, 0);
  }
  /**
   * Scales the path.
   */
  ;

  _proto.scale = function scale(scaleX, scaleY) {
    if (scaleY === void 0) {
      scaleY = scaleX;
    }

    return this.transform(scaleX, 0, 0, scaleY, 0, 0);
  };

  _createClass(Path, [{
    key: "cbox",
    get: function get() {
      if (!this._cbox) {
        var cbox = new BBox();

        for (var _iterator2 = this.commands, _isArray2 = Array.isArray(_iterator2), _i2 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
          var _ref2;

          if (_isArray2) {
            if (_i2 >= _iterator2.length) break;
            _ref2 = _iterator2[_i2++];
          } else {
            _i2 = _iterator2.next();
            if (_i2.done) break;
            _ref2 = _i2.value;
          }

          var command = _ref2;

          for (var i = 0; i < command.args.length; i += 2) {
            cbox.addPoint(command.args[i], command.args[i + 1]);
          }
        }

        this._cbox = Object.freeze(cbox);
      }

      return this._cbox;
    }
    /**
     * Gets the exact bounding box of the path by evaluating curve segments.
     * Slower to compute than the control box, but more accurate.
     * @type {BBox}
     */

  }, {
    key: "bbox",
    get: function get() {
      if (this._bbox) {
        return this._bbox;
      }

      var bbox = new BBox();
      var cx = 0,
          cy = 0;

      var f = function f(t) {
        return Math.pow(1 - t, 3) * p0[i] + 3 * Math.pow(1 - t, 2) * t * p1[i] + 3 * (1 - t) * Math.pow(t, 2) * p2[i] + Math.pow(t, 3) * p3[i];
      };

      for (var _iterator3 = this.commands, _isArray3 = Array.isArray(_iterator3), _i3 = 0, _iterator3 = _isArray3 ? _iterator3 : _iterator3[Symbol.iterator]();;) {
        var _ref3;

        if (_isArray3) {
          if (_i3 >= _iterator3.length) break;
          _ref3 = _iterator3[_i3++];
        } else {
          _i3 = _iterator3.next();
          if (_i3.done) break;
          _ref3 = _i3.value;
        }

        var c = _ref3;

        switch (c.command) {
          case 'moveTo':
          case 'lineTo':
            var _c$args = c.args,
                x = _c$args[0],
                y = _c$args[1];
            bbox.addPoint(x, y);
            cx = x;
            cy = y;
            break;

          case 'quadraticCurveTo':
          case 'bezierCurveTo':
            if (c.command === 'quadraticCurveTo') {
              // http://fontforge.org/bezier.html
              var _c$args2 = c.args,
                  qp1x = _c$args2[0],
                  qp1y = _c$args2[1],
                  p3x = _c$args2[2],
                  p3y = _c$args2[3];
              var cp1x = cx + 2 / 3 * (qp1x - cx); // CP1 = QP0 + 2/3 * (QP1-QP0)

              var cp1y = cy + 2 / 3 * (qp1y - cy);
              var cp2x = p3x + 2 / 3 * (qp1x - p3x); // CP2 = QP2 + 2/3 * (QP1-QP2)

              var cp2y = p3y + 2 / 3 * (qp1y - p3y);
            } else {
              var _c$args3 = c.args,
                  cp1x = _c$args3[0],
                  cp1y = _c$args3[1],
                  cp2x = _c$args3[2],
                  cp2y = _c$args3[3],
                  p3x = _c$args3[4],
                  p3y = _c$args3[5];
            } // http://blog.hackers-cafe.net/2009/06/how-to-calculate-bezier-curves-bounding.html


            bbox.addPoint(p3x, p3y);
            var p0 = [cx, cy];
            var p1 = [cp1x, cp1y];
            var p2 = [cp2x, cp2y];
            var p3 = [p3x, p3y];

            for (var i = 0; i <= 1; i++) {
              var b = 6 * p0[i] - 12 * p1[i] + 6 * p2[i];
              var a = -3 * p0[i] + 9 * p1[i] - 9 * p2[i] + 3 * p3[i];
              c = 3 * p1[i] - 3 * p0[i];

              if (a === 0) {
                if (b === 0) {
                  continue;
                }

                var t = -c / b;

                if (0 < t && t < 1) {
                  if (i === 0) {
                    bbox.addPoint(f(t), bbox.maxY);
                  } else if (i === 1) {
                    bbox.addPoint(bbox.maxX, f(t));
                  }
                }

                continue;
              }

              var b2ac = Math.pow(b, 2) - 4 * c * a;

              if (b2ac < 0) {
                continue;
              }

              var t1 = (-b + Math.sqrt(b2ac)) / (2 * a);

              if (0 < t1 && t1 < 1) {
                if (i === 0) {
                  bbox.addPoint(f(t1), bbox.maxY);
                } else if (i === 1) {
                  bbox.addPoint(bbox.maxX, f(t1));
                }
              }

              var t2 = (-b - Math.sqrt(b2ac)) / (2 * a);

              if (0 < t2 && t2 < 1) {
                if (i === 0) {
                  bbox.addPoint(f(t2), bbox.maxY);
                } else if (i === 1) {
                  bbox.addPoint(bbox.maxX, f(t2));
                }
              }
            }

            cx = p3x;
            cy = p3y;
            break;
        }
      }

      return this._bbox = Object.freeze(bbox);
    }
  }]);

  return Path;
}();
var _arr = ['moveTo', 'lineTo', 'quadraticCurveTo', 'bezierCurveTo', 'closePath'];

var _loop = function _loop() {
  var command = _arr[_i4];

  Path.prototype[command] = function () {
    for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
      args[_key] = arguments[_key];
    }

    this._bbox = this._cbox = null;
    this.commands.push({
      command: command,
      args: args
    });
    return this;
  };
};

for (var _i4 = 0; _i4 < _arr.length; _i4++) {
  _loop();
}

var StandardNames = ['.notdef', '.null', 'nonmarkingreturn', 'space', 'exclam', 'quotedbl', 'numbersign', 'dollar', 'percent', 'ampersand', 'quotesingle', 'parenleft', 'parenright', 'asterisk', 'plus', 'comma', 'hyphen', 'period', 'slash', 'zero', 'one', 'two', 'three', 'four', 'five', 'six', 'seven', 'eight', 'nine', 'colon', 'semicolon', 'less', 'equal', 'greater', 'question', 'at', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'bracketleft', 'backslash', 'bracketright', 'asciicircum', 'underscore', 'grave', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', 'braceleft', 'bar', 'braceright', 'asciitilde', 'Adieresis', 'Aring', 'Ccedilla', 'Eacute', 'Ntilde', 'Odieresis', 'Udieresis', 'aacute', 'agrave', 'acircumflex', 'adieresis', 'atilde', 'aring', 'ccedilla', 'eacute', 'egrave', 'ecircumflex', 'edieresis', 'iacute', 'igrave', 'icircumflex', 'idieresis', 'ntilde', 'oacute', 'ograve', 'ocircumflex', 'odieresis', 'otilde', 'uacute', 'ugrave', 'ucircumflex', 'udieresis', 'dagger', 'degree', 'cent', 'sterling', 'section', 'bullet', 'paragraph', 'germandbls', 'registered', 'copyright', 'trademark', 'acute', 'dieresis', 'notequal', 'AE', 'Oslash', 'infinity', 'plusminus', 'lessequal', 'greaterequal', 'yen', 'mu', 'partialdiff', 'summation', 'product', 'pi', 'integral', 'ordfeminine', 'ordmasculine', 'Omega', 'ae', 'oslash', 'questiondown', 'exclamdown', 'logicalnot', 'radical', 'florin', 'approxequal', 'Delta', 'guillemotleft', 'guillemotright', 'ellipsis', 'nonbreakingspace', 'Agrave', 'Atilde', 'Otilde', 'OE', 'oe', 'endash', 'emdash', 'quotedblleft', 'quotedblright', 'quoteleft', 'quoteright', 'divide', 'lozenge', 'ydieresis', 'Ydieresis', 'fraction', 'currency', 'guilsinglleft', 'guilsinglright', 'fi', 'fl', 'daggerdbl', 'periodcentered', 'quotesinglbase', 'quotedblbase', 'perthousand', 'Acircumflex', 'Ecircumflex', 'Aacute', 'Edieresis', 'Egrave', 'Iacute', 'Icircumflex', 'Idieresis', 'Igrave', 'Oacute', 'Ocircumflex', 'apple', 'Ograve', 'Uacute', 'Ucircumflex', 'Ugrave', 'dotlessi', 'circumflex', 'tilde', 'macron', 'breve', 'dotaccent', 'ring', 'cedilla', 'hungarumlaut', 'ogonek', 'caron', 'Lslash', 'lslash', 'Scaron', 'scaron', 'Zcaron', 'zcaron', 'brokenbar', 'Eth', 'eth', 'Yacute', 'yacute', 'Thorn', 'thorn', 'minus', 'multiply', 'onesuperior', 'twosuperior', 'threesuperior', 'onehalf', 'onequarter', 'threequarters', 'franc', 'Gbreve', 'gbreve', 'Idotaccent', 'Scedilla', 'scedilla', 'Cacute', 'cacute', 'Ccaron', 'ccaron', 'dcroat'];

var _class$3;
/**
 * Glyph objects represent a glyph in the font. They have various properties for accessing metrics and
 * the actual vector path the glyph represents, and methods for rendering the glyph to a graphics context.
 *
 * You do not create glyph objects directly. They are created by various methods on the font object.
 * There are several subclasses of the base Glyph class internally that may be returned depending
 * on the font format, but they all inherit from this class.
 */

var Glyph = (_class$3 =
/*#__PURE__*/
function () {
  function Glyph(id, codePoints, font) {
    /**
     * The glyph id in the font
     * @type {number}
     */
    this.id = id;
    /**
     * An array of unicode code points that are represented by this glyph.
     * There can be multiple code points in the case of ligatures and other glyphs
     * that represent multiple visual characters.
     * @type {number[]}
     */

    this.codePoints = codePoints;
    this._font = font; // TODO: get this info from GDEF if available

    this.isMark = this.codePoints.length > 0 && this.codePoints.every(unicode.isMark);
    this.isLigature = this.codePoints.length > 1;
  }

  var _proto = Glyph.prototype;

  _proto._getPath = function _getPath() {
    return new Path();
  };

  _proto._getCBox = function _getCBox() {
    return this.path.cbox;
  };

  _proto._getBBox = function _getBBox() {
    return this.path.bbox;
  };

  _proto._getTableMetrics = function _getTableMetrics(table) {
    if (this.id < table.metrics.length) {
      return table.metrics.get(this.id);
    }

    var metric = table.metrics.get(table.metrics.length - 1);
    var res = {
      advance: metric ? metric.advance : 0,
      bearing: table.bearings.get(this.id - table.metrics.length) || 0
    };
    return res;
  };

  _proto._getMetrics = function _getMetrics(cbox) {
    if (this._metrics) {
      return this._metrics;
    }

    var _this$_getTableMetric = this._getTableMetrics(this._font.hmtx),
        advanceWidth = _this$_getTableMetric.advance,
        leftBearing = _this$_getTableMetric.bearing; // For vertical metrics, use vmtx if available, or fall back to global data from OS/2 or hhea


    if (this._font.vmtx) {
      var _this$_getTableMetric2 = this._getTableMetrics(this._font.vmtx),
          advanceHeight = _this$_getTableMetric2.advance,
          topBearing = _this$_getTableMetric2.bearing;
    } else {
      var os2;

      if (typeof cbox === 'undefined' || cbox === null) {
        cbox = this.cbox;
      }

      if ((os2 = this._font['OS/2']) && os2.version > 0) {
        var advanceHeight = Math.abs(os2.typoAscender - os2.typoDescender);
        var topBearing = os2.typoAscender - cbox.maxY;
      } else {
        var hhea = this._font.hhea;
        var advanceHeight = Math.abs(hhea.ascent - hhea.descent);
        var topBearing = hhea.ascent - cbox.maxY;
      }
    }

    if (this._font._variationProcessor && this._font.HVAR) {
      advanceWidth += this._font._variationProcessor.getAdvanceAdjustment(this.id, this._font.HVAR);
    }

    return this._metrics = {
      advanceWidth: advanceWidth,
      advanceHeight: advanceHeight,
      leftBearing: leftBearing,
      topBearing: topBearing
    };
  }
  /**
   * The glyph’s control box.
   * This is often the same as the bounding box, but is faster to compute.
   * Because of the way bezier curves are defined, some of the control points
   * can be outside of the bounding box. Where `bbox` takes this into account,
   * `cbox` does not. Thus, cbox is less accurate, but faster to compute.
   * See [here](http://www.freetype.org/freetype2/docs/glyphs/glyphs-6.html#section-2)
   * for a more detailed description.
   *
   * @type {BBox}
   */
  ;

  /**
   * Returns a path scaled to the given font size.
   * @param {number} size
   * @return {Path}
   */
  _proto.getScaledPath = function getScaledPath(size) {
    var scale = 1 / this._font.unitsPerEm * size;
    return this.path.scale(scale);
  }
  /**
   * The glyph's advance width.
   * @type {number}
   */
  ;

  _proto._getName = function _getName() {
    var post = this._font.post;

    if (!post) {
      return null;
    }

    switch (post.version) {
      case 1:
        return StandardNames[this.id];

      case 2:
        var id = post.glyphNameIndex[this.id];

        if (id < StandardNames.length) {
          return StandardNames[id];
        }

        return post.names[id - StandardNames.length];

      case 2.5:
        return StandardNames[this.id + post.offsets[this.id]];

      case 4:
        return String.fromCharCode(post.map[this.id]);
    }
  }
  /**
   * The glyph's name
   * @type {string}
   */
  ;

  /**
   * Renders the glyph to the given graphics context, at the specified font size.
   * @param {CanvasRenderingContext2d} ctx
   * @param {number} size
   */
  _proto.render = function render(ctx, size) {
    ctx.save();
    var scale = 1 / this._font.head.unitsPerEm * size;
    ctx.scale(scale, scale);
    var fn = this.path.toFunction();
    fn(ctx);
    ctx.fill();
    ctx.restore();
  };

  _createClass(Glyph, [{
    key: "cbox",
    get: function get() {
      return this._getCBox();
    }
    /**
     * The glyph’s bounding box, i.e. the rectangle that encloses the
     * glyph outline as tightly as possible.
     * @type {BBox}
     */

  }, {
    key: "bbox",
    get: function get() {
      return this._getBBox();
    }
    /**
     * A vector Path object representing the glyph outline.
     * @type {Path}
     */

  }, {
    key: "path",
    get: function get() {
      // Cache the path so we only decode it once
      // Decoding is actually performed by subclasses
      return this._getPath();
    }
  }, {
    key: "advanceWidth",
    get: function get() {
      return this._getMetrics().advanceWidth;
    }
    /**
     * The glyph's advance height.
     * @type {number}
     */

  }, {
    key: "advanceHeight",
    get: function get() {
      return this._getMetrics().advanceHeight;
    }
  }, {
    key: "ligatureCaretPositions",
    get: function get() {}
  }, {
    key: "name",
    get: function get() {
      return this._getName();
    }
  }]);

  return Glyph;
}(), (_applyDecoratedDescriptor(_class$3.prototype, "cbox", [cache], Object.getOwnPropertyDescriptor(_class$3.prototype, "cbox"), _class$3.prototype), _applyDecoratedDescriptor(_class$3.prototype, "bbox", [cache], Object.getOwnPropertyDescriptor(_class$3.prototype, "bbox"), _class$3.prototype), _applyDecoratedDescriptor(_class$3.prototype, "path", [cache], Object.getOwnPropertyDescriptor(_class$3.prototype, "path"), _class$3.prototype), _applyDecoratedDescriptor(_class$3.prototype, "advanceWidth", [cache], Object.getOwnPropertyDescriptor(_class$3.prototype, "advanceWidth"), _class$3.prototype), _applyDecoratedDescriptor(_class$3.prototype, "advanceHeight", [cache], Object.getOwnPropertyDescriptor(_class$3.prototype, "advanceHeight"), _class$3.prototype), _applyDecoratedDescriptor(_class$3.prototype, "name", [cache], Object.getOwnPropertyDescriptor(_class$3.prototype, "name"), _class$3.prototype)), _class$3);

var GlyfHeader = new restructure.Struct({
  numberOfContours: restructure.int16,
  // if negative, this is a composite glyph
  xMin: restructure.int16,
  yMin: restructure.int16,
  xMax: restructure.int16,
  yMax: restructure.int16
}); // Flags for simple glyphs

var ON_CURVE = 1 << 0;
var X_SHORT_VECTOR = 1 << 1;
var Y_SHORT_VECTOR = 1 << 2;
var REPEAT = 1 << 3;
var SAME_X = 1 << 4;
var SAME_Y = 1 << 5; // Flags for composite glyphs

var ARG_1_AND_2_ARE_WORDS = 1 << 0;
var WE_HAVE_A_SCALE = 1 << 3;
var MORE_COMPONENTS = 1 << 5;
var WE_HAVE_AN_X_AND_Y_SCALE = 1 << 6;
var WE_HAVE_A_TWO_BY_TWO = 1 << 7;
var WE_HAVE_INSTRUCTIONS = 1 << 8;

var Point =
/*#__PURE__*/
function () {
  function Point(onCurve, endContour, x, y) {
    if (x === void 0) {
      x = 0;
    }

    if (y === void 0) {
      y = 0;
    }

    this.onCurve = onCurve;
    this.endContour = endContour;
    this.x = x;
    this.y = y;
  }

  var _proto = Point.prototype;

  _proto.copy = function copy() {
    return new Point(this.onCurve, this.endContour, this.x, this.y);
  };

  return Point;
}(); // Represents a component in a composite glyph

var Component = function Component(glyphID, dx, dy) {
  this.glyphID = glyphID;
  this.dx = dx;
  this.dy = dy;
  this.pos = 0;
  this.scaleX = this.scaleY = 1;
  this.scale01 = this.scale10 = 0;
};
/**
 * Represents a TrueType glyph.
 */


var TTFGlyph =
/*#__PURE__*/
function (_Glyph) {
  _inheritsLoose(TTFGlyph, _Glyph);

  function TTFGlyph() {
    return _Glyph.apply(this, arguments) || this;
  }

  var _proto2 = TTFGlyph.prototype;

  // Parses just the glyph header and returns the bounding box
  _proto2._getCBox = function _getCBox(internal) {
    // We need to decode the glyph if variation processing is requested,
    // so it's easier just to recompute the path's cbox after decoding.
    if (this._font._variationProcessor && !internal) {
      return this.path.cbox;
    }

    var stream = this._font._getTableStream('glyf');

    stream.pos += this._font.loca.offsets[this.id];
    var glyph = GlyfHeader.decode(stream);
    var cbox = new BBox(glyph.xMin, glyph.yMin, glyph.xMax, glyph.yMax);
    return Object.freeze(cbox);
  } // Parses a single glyph coordinate
  ;

  _proto2._parseGlyphCoord = function _parseGlyphCoord(stream, prev, short, same) {
    if (short) {
      var val = stream.readUInt8();

      if (!same) {
        val = -val;
      }

      val += prev;
    } else {
      if (same) {
        var val = prev;
      } else {
        var val = prev + stream.readInt16BE();
      }
    }

    return val;
  } // Decodes the glyph data into points for simple glyphs,
  // or components for composite glyphs
  ;

  _proto2._decode = function _decode() {
    var glyfPos = this._font.loca.offsets[this.id];
    var nextPos = this._font.loca.offsets[this.id + 1]; // Nothing to do if there is no data for this glyph

    if (glyfPos === nextPos) {
      return null;
    }

    var stream = this._font._getTableStream('glyf');

    stream.pos += glyfPos;
    var startPos = stream.pos;
    var glyph = GlyfHeader.decode(stream);

    if (glyph.numberOfContours > 0) {
      this._decodeSimple(glyph, stream);
    } else if (glyph.numberOfContours < 0) {
      this._decodeComposite(glyph, stream, startPos);
    }

    return glyph;
  };

  _proto2._decodeSimple = function _decodeSimple(glyph, stream) {
    // this is a simple glyph
    glyph.points = [];
    var endPtsOfContours = new restructure.Array(restructure.uint16, glyph.numberOfContours).decode(stream);
    glyph.instructions = new restructure.Array(restructure.uint8, restructure.uint16).decode(stream);
    var flags = [];
    var numCoords = endPtsOfContours[endPtsOfContours.length - 1] + 1;

    while (flags.length < numCoords) {
      var flag = stream.readUInt8();
      flags.push(flag); // check for repeat flag

      if (flag & REPEAT) {
        var count = stream.readUInt8();

        for (var j = 0; j < count; j++) {
          flags.push(flag);
        }
      }
    }

    for (var i = 0; i < flags.length; i++) {
      var flag = flags[i];
      var point = new Point(!!(flag & ON_CURVE), endPtsOfContours.indexOf(i) >= 0, 0, 0);
      glyph.points.push(point);
    }

    var px = 0;

    for (var i = 0; i < flags.length; i++) {
      var flag = flags[i];
      glyph.points[i].x = px = this._parseGlyphCoord(stream, px, flag & X_SHORT_VECTOR, flag & SAME_X);
    }

    var py = 0;

    for (var i = 0; i < flags.length; i++) {
      var flag = flags[i];
      glyph.points[i].y = py = this._parseGlyphCoord(stream, py, flag & Y_SHORT_VECTOR, flag & SAME_Y);
    }

    if (this._font._variationProcessor) {
      var points = glyph.points.slice();
      points.push.apply(points, this._getPhantomPoints(glyph));

      this._font._variationProcessor.transformPoints(this.id, points);

      glyph.phantomPoints = points.slice(-4);
    }

    return;
  };

  _proto2._decodeComposite = function _decodeComposite(glyph, stream, offset) {
    if (offset === void 0) {
      offset = 0;
    }

    // this is a composite glyph
    glyph.components = [];
    var haveInstructions = false;
    var flags = MORE_COMPONENTS;

    while (flags & MORE_COMPONENTS) {
      flags = stream.readUInt16BE();
      var gPos = stream.pos - offset;
      var glyphID = stream.readUInt16BE();

      if (!haveInstructions) {
        haveInstructions = (flags & WE_HAVE_INSTRUCTIONS) !== 0;
      }

      if (flags & ARG_1_AND_2_ARE_WORDS) {
        var dx = stream.readInt16BE();
        var dy = stream.readInt16BE();
      } else {
        var dx = stream.readInt8();
        var dy = stream.readInt8();
      }

      var component = new Component(glyphID, dx, dy);
      component.pos = gPos;

      if (flags & WE_HAVE_A_SCALE) {
        // fixed number with 14 bits of fraction
        component.scaleX = component.scaleY = (stream.readUInt8() << 24 | stream.readUInt8() << 16) / 1073741824;
      } else if (flags & WE_HAVE_AN_X_AND_Y_SCALE) {
        component.scaleX = (stream.readUInt8() << 24 | stream.readUInt8() << 16) / 1073741824;
        component.scaleY = (stream.readUInt8() << 24 | stream.readUInt8() << 16) / 1073741824;
      } else if (flags & WE_HAVE_A_TWO_BY_TWO) {
        component.scaleX = (stream.readUInt8() << 24 | stream.readUInt8() << 16) / 1073741824;
        component.scale01 = (stream.readUInt8() << 24 | stream.readUInt8() << 16) / 1073741824;
        component.scale10 = (stream.readUInt8() << 24 | stream.readUInt8() << 16) / 1073741824;
        component.scaleY = (stream.readUInt8() << 24 | stream.readUInt8() << 16) / 1073741824;
      }

      glyph.components.push(component);
    }

    if (this._font._variationProcessor) {
      var points = [];

      for (var j = 0; j < glyph.components.length; j++) {
        var component = glyph.components[j];
        points.push(new Point(true, true, component.dx, component.dy));
      }

      points.push.apply(points, this._getPhantomPoints(glyph));

      this._font._variationProcessor.transformPoints(this.id, points);

      glyph.phantomPoints = points.splice(-4, 4);

      for (var i = 0; i < points.length; i++) {
        var point = points[i];
        glyph.components[i].dx = point.x;
        glyph.components[i].dy = point.y;
      }
    }

    return haveInstructions;
  };

  _proto2._getPhantomPoints = function _getPhantomPoints(glyph) {
    var cbox = this._getCBox(true);

    if (this._metrics == null) {
      this._metrics = Glyph.prototype._getMetrics.call(this, cbox);
    }

    var _this$_metrics = this._metrics,
        advanceWidth = _this$_metrics.advanceWidth,
        advanceHeight = _this$_metrics.advanceHeight,
        leftBearing = _this$_metrics.leftBearing,
        topBearing = _this$_metrics.topBearing;
    return [new Point(false, true, glyph.xMin - leftBearing, 0), new Point(false, true, glyph.xMin - leftBearing + advanceWidth, 0), new Point(false, true, 0, glyph.yMax + topBearing), new Point(false, true, 0, glyph.yMax + topBearing + advanceHeight)];
  } // Decodes font data, resolves composite glyphs, and returns an array of contours
  ;

  _proto2._getContours = function _getContours() {
    var glyph = this._decode();

    if (!glyph) {
      return [];
    }

    var points = [];

    if (glyph.numberOfContours < 0) {
      // resolve composite glyphs
      for (var _iterator = glyph.components, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
        var _ref;

        if (_isArray) {
          if (_i >= _iterator.length) break;
          _ref = _iterator[_i++];
        } else {
          _i = _iterator.next();
          if (_i.done) break;
          _ref = _i.value;
        }

        var component = _ref;

        var _contours = this._font.getGlyph(component.glyphID)._getContours();

        for (var i = 0; i < _contours.length; i++) {
          var contour = _contours[i];

          for (var j = 0; j < contour.length; j++) {
            var _point = contour[j];
            var x = _point.x * component.scaleX + _point.y * component.scale01 + component.dx;
            var y = _point.y * component.scaleY + _point.x * component.scale10 + component.dy;
            points.push(new Point(_point.onCurve, _point.endContour, x, y));
          }
        }
      }
    } else {
      points = glyph.points || [];
    } // Recompute and cache metrics if we performed variation processing, and don't have an HVAR table


    if (glyph.phantomPoints && !this._font.directory.tables.HVAR) {
      this._metrics.advanceWidth = glyph.phantomPoints[1].x - glyph.phantomPoints[0].x;
      this._metrics.advanceHeight = glyph.phantomPoints[3].y - glyph.phantomPoints[2].y;
      this._metrics.leftBearing = glyph.xMin - glyph.phantomPoints[0].x;
      this._metrics.topBearing = glyph.phantomPoints[2].y - glyph.yMax;
    }

    var contours = [];
    var cur = [];

    for (var k = 0; k < points.length; k++) {
      var point = points[k];
      cur.push(point);

      if (point.endContour) {
        contours.push(cur);
        cur = [];
      }
    }

    return contours;
  };

  _proto2._getMetrics = function _getMetrics() {
    if (this._metrics) {
      return this._metrics;
    }

    var cbox = this._getCBox(true);

    _Glyph.prototype._getMetrics.call(this, cbox);

    if (this._font._variationProcessor && !this._font.HVAR) {
      // No HVAR table, decode the glyph. This triggers recomputation of metrics.
      this.path;
    }

    return this._metrics;
  } // Converts contours to a Path object that can be rendered
  ;

  _proto2._getPath = function _getPath() {
    var contours = this._getContours();

    var path = new Path();

    for (var i = 0; i < contours.length; i++) {
      var contour = contours[i];
      var firstPt = contour[0];
      var lastPt = contour[contour.length - 1];
      var start = 0;

      if (firstPt.onCurve) {
        // The first point will be consumed by the moveTo command, so skip in the loop
        var curvePt = null;
        start = 1;
      } else {
        if (lastPt.onCurve) {
          // Start at the last point if the first point is off curve and the last point is on curve
          firstPt = lastPt;
        } else {
          // Start at the middle if both the first and last points are off curve
          firstPt = new Point(false, false, (firstPt.x + lastPt.x) / 2, (firstPt.y + lastPt.y) / 2);
        }

        var curvePt = firstPt;
      }

      path.moveTo(firstPt.x, firstPt.y);

      for (var j = start; j < contour.length; j++) {
        var pt = contour[j];
        var prevPt = j === 0 ? firstPt : contour[j - 1];

        if (prevPt.onCurve && pt.onCurve) {
          path.lineTo(pt.x, pt.y);
        } else if (prevPt.onCurve && !pt.onCurve) {
          var curvePt = pt;
        } else if (!prevPt.onCurve && !pt.onCurve) {
          var midX = (prevPt.x + pt.x) / 2;
          var midY = (prevPt.y + pt.y) / 2;
          path.quadraticCurveTo(prevPt.x, prevPt.y, midX, midY);
          var curvePt = pt;
        } else if (!prevPt.onCurve && pt.onCurve) {
          path.quadraticCurveTo(curvePt.x, curvePt.y, pt.x, pt.y);
          var curvePt = null;
        } else {
          throw new Error("Unknown TTF path state");
        }
      } // Connect the first and last points


      if (curvePt) {
        path.quadraticCurveTo(curvePt.x, curvePt.y, firstPt.x, firstPt.y);
      }

      path.closePath();
    }

    return path;
  };

  return TTFGlyph;
}(Glyph);

/**
 * Represents an OpenType PostScript glyph, in the Compact Font Format.
 */

var CFFGlyph =
/*#__PURE__*/
function (_Glyph) {
  _inheritsLoose(CFFGlyph, _Glyph);

  function CFFGlyph() {
    return _Glyph.apply(this, arguments) || this;
  }

  var _proto = CFFGlyph.prototype;

  _proto._getName = function _getName() {
    if (this._font.CFF2) {
      return _Glyph.prototype._getName.call(this);
    }

    return this._font['CFF '].getGlyphName(this.id);
  };

  _proto.bias = function bias(s) {
    if (s.length < 1240) {
      return 107;
    } else if (s.length < 33900) {
      return 1131;
    } else {
      return 32768;
    }
  };

  _proto._getPath = function _getPath() {
    var stream = this._font.stream;
    var pos = stream.pos;
    var cff = this._font.CFF2 || this._font['CFF '];
    var str = cff.topDict.CharStrings[this.id];
    var end = str.offset + str.length;
    stream.pos = str.offset;
    var path = new Path();
    var stack = [];
    var trans = [];
    var width = null;
    var nStems = 0;
    var x = 0,
        y = 0;
    var usedGsubrs;
    var usedSubrs;
    var open = false;
    this._usedGsubrs = usedGsubrs = {};
    this._usedSubrs = usedSubrs = {};
    var gsubrs = cff.globalSubrIndex || [];
    var gsubrsBias = this.bias(gsubrs);
    var privateDict = cff.privateDictForGlyph(this.id);
    var subrs = privateDict.Subrs || [];
    var subrsBias = this.bias(subrs);
    var vstore = cff.topDict.vstore && cff.topDict.vstore.itemVariationStore;
    var vsindex = privateDict.vsindex;
    var variationProcessor = this._font._variationProcessor;

    function checkWidth() {
      if (width == null) {
        width = stack.shift() + privateDict.nominalWidthX;
      }
    }

    function parseStems() {
      if (stack.length % 2 !== 0) {
        checkWidth();
      }

      nStems += stack.length >> 1;
      return stack.length = 0;
    }

    function moveTo(x, y) {
      if (open) {
        path.closePath();
      }

      path.moveTo(x, y);
      open = true;
    }

    var parse = function parse() {
      while (stream.pos < end) {
        var op = stream.readUInt8();

        if (op < 32) {
          switch (op) {
            case 1: // hstem

            case 3: // vstem

            case 18: // hstemhm

            case 23:
              // vstemhm
              parseStems();
              break;

            case 4:
              // vmoveto
              if (stack.length > 1) {
                checkWidth();
              }

              y += stack.shift();
              moveTo(x, y);
              break;

            case 5:
              // rlineto
              while (stack.length >= 2) {
                x += stack.shift();
                y += stack.shift();
                path.lineTo(x, y);
              }

              break;

            case 6: // hlineto

            case 7:
              // vlineto
              var phase = op === 6;

              while (stack.length >= 1) {
                if (phase) {
                  x += stack.shift();
                } else {
                  y += stack.shift();
                }

                path.lineTo(x, y);
                phase = !phase;
              }

              break;

            case 8:
              // rrcurveto
              while (stack.length > 0) {
                var c1x = x + stack.shift();
                var c1y = y + stack.shift();
                var c2x = c1x + stack.shift();
                var c2y = c1y + stack.shift();
                x = c2x + stack.shift();
                y = c2y + stack.shift();
                path.bezierCurveTo(c1x, c1y, c2x, c2y, x, y);
              }

              break;

            case 10:
              // callsubr
              var index = stack.pop() + subrsBias;
              var subr = subrs[index];

              if (subr) {
                usedSubrs[index] = true;
                var p = stream.pos;
                var e = end;
                stream.pos = subr.offset;
                end = subr.offset + subr.length;
                parse();
                stream.pos = p;
                end = e;
              }

              break;

            case 11:
              // return
              if (cff.version >= 2) {
                break;
              }

              return;

            case 14:
              // endchar
              if (cff.version >= 2) {
                break;
              }

              if (stack.length > 0) {
                checkWidth();
              }

              if (open) {
                path.closePath();
                open = false;
              }

              break;

            case 15:
              {
                // vsindex
                if (cff.version < 2) {
                  throw new Error('vsindex operator not supported in CFF v1');
                }

                vsindex = stack.pop();
                break;
              }

            case 16:
              {
                // blend
                if (cff.version < 2) {
                  throw new Error('blend operator not supported in CFF v1');
                }

                if (!variationProcessor) {
                  throw new Error('blend operator in non-variation font');
                }

                var blendVector = variationProcessor.getBlendVector(vstore, vsindex);
                var numBlends = stack.pop();
                var numOperands = numBlends * blendVector.length;
                var delta = stack.length - numOperands;
                var base = delta - numBlends;

                for (var i = 0; i < numBlends; i++) {
                  var sum = stack[base + i];

                  for (var j = 0; j < blendVector.length; j++) {
                    sum += blendVector[j] * stack[delta++];
                  }

                  stack[base + i] = sum;
                }

                while (numOperands--) {
                  stack.pop();
                }

                break;
              }

            case 19: // hintmask

            case 20:
              // cntrmask
              parseStems();
              stream.pos += nStems + 7 >> 3;
              break;

            case 21:
              // rmoveto
              if (stack.length > 2) {
                checkWidth();
              }

              x += stack.shift();
              y += stack.shift();
              moveTo(x, y);
              break;

            case 22:
              // hmoveto
              if (stack.length > 1) {
                checkWidth();
              }

              x += stack.shift();
              moveTo(x, y);
              break;

            case 24:
              // rcurveline
              while (stack.length >= 8) {
                var c1x = x + stack.shift();
                var c1y = y + stack.shift();
                var c2x = c1x + stack.shift();
                var c2y = c1y + stack.shift();
                x = c2x + stack.shift();
                y = c2y + stack.shift();
                path.bezierCurveTo(c1x, c1y, c2x, c2y, x, y);
              }

              x += stack.shift();
              y += stack.shift();
              path.lineTo(x, y);
              break;

            case 25:
              // rlinecurve
              while (stack.length >= 8) {
                x += stack.shift();
                y += stack.shift();
                path.lineTo(x, y);
              }

              var c1x = x + stack.shift();
              var c1y = y + stack.shift();
              var c2x = c1x + stack.shift();
              var c2y = c1y + stack.shift();
              x = c2x + stack.shift();
              y = c2y + stack.shift();
              path.bezierCurveTo(c1x, c1y, c2x, c2y, x, y);
              break;

            case 26:
              // vvcurveto
              if (stack.length % 2) {
                x += stack.shift();
              }

              while (stack.length >= 4) {
                c1x = x;
                c1y = y + stack.shift();
                c2x = c1x + stack.shift();
                c2y = c1y + stack.shift();
                x = c2x;
                y = c2y + stack.shift();
                path.bezierCurveTo(c1x, c1y, c2x, c2y, x, y);
              }

              break;

            case 27:
              // hhcurveto
              if (stack.length % 2) {
                y += stack.shift();
              }

              while (stack.length >= 4) {
                c1x = x + stack.shift();
                c1y = y;
                c2x = c1x + stack.shift();
                c2y = c1y + stack.shift();
                x = c2x + stack.shift();
                y = c2y;
                path.bezierCurveTo(c1x, c1y, c2x, c2y, x, y);
              }

              break;

            case 28:
              // shortint
              stack.push(stream.readInt16BE());
              break;

            case 29:
              // callgsubr
              index = stack.pop() + gsubrsBias;
              subr = gsubrs[index];

              if (subr) {
                usedGsubrs[index] = true;
                var p = stream.pos;
                var e = end;
                stream.pos = subr.offset;
                end = subr.offset + subr.length;
                parse();
                stream.pos = p;
                end = e;
              }

              break;

            case 30: // vhcurveto

            case 31:
              // hvcurveto
              phase = op === 31;

              while (stack.length >= 4) {
                if (phase) {
                  c1x = x + stack.shift();
                  c1y = y;
                  c2x = c1x + stack.shift();
                  c2y = c1y + stack.shift();
                  y = c2y + stack.shift();
                  x = c2x + (stack.length === 1 ? stack.shift() : 0);
                } else {
                  c1x = x;
                  c1y = y + stack.shift();
                  c2x = c1x + stack.shift();
                  c2y = c1y + stack.shift();
                  x = c2x + stack.shift();
                  y = c2y + (stack.length === 1 ? stack.shift() : 0);
                }

                path.bezierCurveTo(c1x, c1y, c2x, c2y, x, y);
                phase = !phase;
              }

              break;

            case 12:
              op = stream.readUInt8();

              switch (op) {
                case 3:
                  // and
                  var a = stack.pop();
                  var b = stack.pop();
                  stack.push(a && b ? 1 : 0);
                  break;

                case 4:
                  // or
                  a = stack.pop();
                  b = stack.pop();
                  stack.push(a || b ? 1 : 0);
                  break;

                case 5:
                  // not
                  a = stack.pop();
                  stack.push(a ? 0 : 1);
                  break;

                case 9:
                  // abs
                  a = stack.pop();
                  stack.push(Math.abs(a));
                  break;

                case 10:
                  // add
                  a = stack.pop();
                  b = stack.pop();
                  stack.push(a + b);
                  break;

                case 11:
                  // sub
                  a = stack.pop();
                  b = stack.pop();
                  stack.push(a - b);
                  break;

                case 12:
                  // div
                  a = stack.pop();
                  b = stack.pop();
                  stack.push(a / b);
                  break;

                case 14:
                  // neg
                  a = stack.pop();
                  stack.push(-a);
                  break;

                case 15:
                  // eq
                  a = stack.pop();
                  b = stack.pop();
                  stack.push(a === b ? 1 : 0);
                  break;

                case 18:
                  // drop
                  stack.pop();
                  break;

                case 20:
                  // put
                  var val = stack.pop();
                  var idx = stack.pop();
                  trans[idx] = val;
                  break;

                case 21:
                  // get
                  idx = stack.pop();
                  stack.push(trans[idx] || 0);
                  break;

                case 22:
                  // ifelse
                  var s1 = stack.pop();
                  var s2 = stack.pop();
                  var v1 = stack.pop();
                  var v2 = stack.pop();
                  stack.push(v1 <= v2 ? s1 : s2);
                  break;

                case 23:
                  // random
                  stack.push(Math.random());
                  break;

                case 24:
                  // mul
                  a = stack.pop();
                  b = stack.pop();
                  stack.push(a * b);
                  break;

                case 26:
                  // sqrt
                  a = stack.pop();
                  stack.push(Math.sqrt(a));
                  break;

                case 27:
                  // dup
                  a = stack.pop();
                  stack.push(a, a);
                  break;

                case 28:
                  // exch
                  a = stack.pop();
                  b = stack.pop();
                  stack.push(b, a);
                  break;

                case 29:
                  // index
                  idx = stack.pop();

                  if (idx < 0) {
                    idx = 0;
                  } else if (idx > stack.length - 1) {
                    idx = stack.length - 1;
                  }

                  stack.push(stack[idx]);
                  break;

                case 30:
                  // roll
                  var n = stack.pop();

                  var _j = stack.pop();

                  if (_j >= 0) {
                    while (_j > 0) {
                      var t = stack[n - 1];

                      for (var _i = n - 2; _i >= 0; _i--) {
                        stack[_i + 1] = stack[_i];
                      }

                      stack[0] = t;
                      _j--;
                    }
                  } else {
                    while (_j < 0) {
                      var t = stack[0];

                      for (var _i2 = 0; _i2 <= n; _i2++) {
                        stack[_i2] = stack[_i2 + 1];
                      }

                      stack[n - 1] = t;
                      _j++;
                    }
                  }

                  break;

                case 34:
                  // hflex
                  c1x = x + stack.shift();
                  c1y = y;
                  c2x = c1x + stack.shift();
                  c2y = c1y + stack.shift();
                  var c3x = c2x + stack.shift();
                  var c3y = c2y;
                  var c4x = c3x + stack.shift();
                  var c4y = c3y;
                  var c5x = c4x + stack.shift();
                  var c5y = c4y;
                  var c6x = c5x + stack.shift();
                  var c6y = c5y;
                  x = c6x;
                  y = c6y;
                  path.bezierCurveTo(c1x, c1y, c2x, c2y, c3x, c3y);
                  path.bezierCurveTo(c4x, c4y, c5x, c5y, c6x, c6y);
                  break;

                case 35:
                  // flex
                  var pts = [];

                  for (var _i3 = 0; _i3 <= 5; _i3++) {
                    x += stack.shift();
                    y += stack.shift();
                    pts.push(x, y);
                  }

                  path.bezierCurveTo.apply(path, pts.slice(0, 6));
                  path.bezierCurveTo.apply(path, pts.slice(6));
                  stack.shift(); // fd

                  break;

                case 36:
                  // hflex1
                  c1x = x + stack.shift();
                  c1y = y + stack.shift();
                  c2x = c1x + stack.shift();
                  c2y = c1y + stack.shift();
                  c3x = c2x + stack.shift();
                  c3y = c2y;
                  c4x = c3x + stack.shift();
                  c4y = c3y;
                  c5x = c4x + stack.shift();
                  c5y = c4y + stack.shift();
                  c6x = c5x + stack.shift();
                  c6y = c5y;
                  x = c6x;
                  y = c6y;
                  path.bezierCurveTo(c1x, c1y, c2x, c2y, c3x, c3y);
                  path.bezierCurveTo(c4x, c4y, c5x, c5y, c6x, c6y);
                  break;

                case 37:
                  // flex1
                  var startx = x;
                  var starty = y;
                  pts = [];

                  for (var _i4 = 0; _i4 <= 4; _i4++) {
                    x += stack.shift();
                    y += stack.shift();
                    pts.push(x, y);
                  }

                  if (Math.abs(x - startx) > Math.abs(y - starty)) {
                    // horizontal
                    x += stack.shift();
                    y = starty;
                  } else {
                    x = startx;
                    y += stack.shift();
                  }

                  pts.push(x, y);
                  path.bezierCurveTo.apply(path, pts.slice(0, 6));
                  path.bezierCurveTo.apply(path, pts.slice(6));
                  break;

                default:
                  throw new Error("Unknown op: 12 " + op);
              }

              break;

            default:
              throw new Error("Unknown op: " + op);
          }
        } else if (op < 247) {
          stack.push(op - 139);
        } else if (op < 251) {
          var b1 = stream.readUInt8();
          stack.push((op - 247) * 256 + b1 + 108);
        } else if (op < 255) {
          var b1 = stream.readUInt8();
          stack.push(-(op - 251) * 256 - b1 - 108);
        } else {
          stack.push(stream.readInt32BE() / 65536);
        }
      }
    };

    parse();

    if (open) {
      path.closePath();
    }

    return path;
  };

  return CFFGlyph;
}(Glyph);

var SBIXImage = new restructure.Struct({
  originX: restructure.uint16,
  originY: restructure.uint16,
  type: new restructure.String(4),
  data: new restructure.Buffer(function (t) {
    return t.parent.buflen - t._currentOffset;
  })
});
/**
 * Represents a color (e.g. emoji) glyph in Apple's SBIX format.
 */

var SBIXGlyph =
/*#__PURE__*/
function (_TTFGlyph) {
  _inheritsLoose(SBIXGlyph, _TTFGlyph);

  function SBIXGlyph() {
    return _TTFGlyph.apply(this, arguments) || this;
  }

  var _proto = SBIXGlyph.prototype;

  /**
   * Returns an object representing a glyph image at the given point size.
   * The object has a data property with a Buffer containing the actual image data,
   * along with the image type, and origin.
   *
   * @param {number} size
   * @return {object}
   */
  _proto.getImageForSize = function getImageForSize(size) {
    for (var i = 0; i < this._font.sbix.imageTables.length; i++) {
      var table = this._font.sbix.imageTables[i];

      if (table.ppem >= size) {
        break;
      }
    }

    var offsets = table.imageOffsets;
    var start = offsets[this.id];
    var end = offsets[this.id + 1];

    if (start === end) {
      return null;
    }

    this._font.stream.pos = start;
    return SBIXImage.decode(this._font.stream, {
      buflen: end - start
    });
  };

  _proto.render = function render(ctx, size) {
    var img = this.getImageForSize(size);

    if (img != null) {
      var scale = size / this._font.unitsPerEm;
      ctx.image(img.data, {
        height: size,
        x: img.originX,
        y: (this.bbox.minY - img.originY) * scale
      });
    }

    if (this._font.sbix.flags.renderOutlines) {
      _TTFGlyph.prototype.render.call(this, ctx, size);
    }
  };

  return SBIXGlyph;
}(TTFGlyph);

var COLRLayer = function COLRLayer(glyph, color) {
  this.glyph = glyph;
  this.color = color;
};
/**
 * Represents a color (e.g. emoji) glyph in Microsoft's COLR format.
 * Each glyph in this format contain a list of colored layers, each
 * of which  is another vector glyph.
 */


var COLRGlyph =
/*#__PURE__*/
function (_Glyph) {
  _inheritsLoose(COLRGlyph, _Glyph);

  function COLRGlyph() {
    return _Glyph.apply(this, arguments) || this;
  }

  var _proto = COLRGlyph.prototype;

  _proto._getBBox = function _getBBox() {
    var bbox = new BBox();

    for (var i = 0; i < this.layers.length; i++) {
      var layer = this.layers[i];
      var b = layer.glyph.bbox;
      bbox.addPoint(b.minX, b.minY);
      bbox.addPoint(b.maxX, b.maxY);
    }

    return bbox;
  }
  /**
   * Returns an array of objects containing the glyph and color for
   * each layer in the composite color glyph.
   * @type {object[]}
   */
  ;

  _proto.render = function render(ctx, size) {
    for (var _iterator = this.layers, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
      var _ref;

      if (_isArray) {
        if (_i >= _iterator.length) break;
        _ref = _iterator[_i++];
      } else {
        _i = _iterator.next();
        if (_i.done) break;
        _ref = _i.value;
      }

      var _ref2 = _ref,
          glyph = _ref2.glyph,
          color = _ref2.color;
      ctx.fillColor([color.red, color.green, color.blue], color.alpha / 255 * 100);
      glyph.render(ctx, size);
    }

    return;
  };

  _createClass(COLRGlyph, [{
    key: "layers",
    get: function get() {
      var cpal = this._font.CPAL;
      var colr = this._font.COLR;
      var low = 0;
      var high = colr.baseGlyphRecord.length - 1;

      while (low <= high) {
        var mid = low + high >> 1;
        var rec = colr.baseGlyphRecord[mid];

        if (this.id < rec.gid) {
          high = mid - 1;
        } else if (this.id > rec.gid) {
          low = mid + 1;
        } else {
          var baseLayer = rec;
          break;
        }
      } // if base glyph not found in COLR table,
      // default to normal glyph from glyf or CFF


      if (baseLayer == null) {
        var g = this._font._getBaseGlyph(this.id);

        var color = {
          red: 0,
          green: 0,
          blue: 0,
          alpha: 255
        };
        return [new COLRLayer(g, color)];
      } // otherwise, return an array of all the layers


      var layers = [];

      for (var i = baseLayer.firstLayerIndex; i < baseLayer.firstLayerIndex + baseLayer.numLayers; i++) {
        var rec = colr.layerRecords[i];
        var color = cpal.colorRecords[rec.paletteIndex];

        var g = this._font._getBaseGlyph(rec.gid);

        layers.push(new COLRLayer(g, color));
      }

      return layers;
    }
  }]);

  return COLRGlyph;
}(Glyph);

var TUPLES_SHARE_POINT_NUMBERS = 0x8000;
var TUPLE_COUNT_MASK = 0x0fff;
var EMBEDDED_TUPLE_COORD = 0x8000;
var INTERMEDIATE_TUPLE = 0x4000;
var PRIVATE_POINT_NUMBERS = 0x2000;
var TUPLE_INDEX_MASK = 0x0fff;
var POINTS_ARE_WORDS = 0x80;
var POINT_RUN_COUNT_MASK = 0x7f;
var DELTAS_ARE_ZERO = 0x80;
var DELTAS_ARE_WORDS = 0x40;
var DELTA_RUN_COUNT_MASK = 0x3f;
/**
 * This class is transforms TrueType glyphs according to the data from
 * the Apple Advanced Typography variation tables (fvar, gvar, and avar).
 * These tables allow infinite adjustments to glyph weight, width, slant,
 * and optical size without the designer needing to specify every exact style.
 *
 * Apple's documentation for these tables is not great, so thanks to the
 * Freetype project for figuring much of this out.
 *
 * @private
 */

var GlyphVariationProcessor =
/*#__PURE__*/
function () {
  function GlyphVariationProcessor(font, coords) {
    this.font = font;
    this.normalizedCoords = this.normalizeCoords(coords);
    this.blendVectors = new Map();
  }

  var _proto = GlyphVariationProcessor.prototype;

  _proto.normalizeCoords = function normalizeCoords(coords) {
    // the default mapping is linear along each axis, in two segments:
    // from the minValue to defaultValue, and from defaultValue to maxValue.
    var normalized = [];

    for (var i = 0; i < this.font.fvar.axis.length; i++) {
      var axis = this.font.fvar.axis[i];

      if (coords[i] < axis.defaultValue) {
        normalized.push((coords[i] - axis.defaultValue + Number.EPSILON) / (axis.defaultValue - axis.minValue + Number.EPSILON));
      } else {
        normalized.push((coords[i] - axis.defaultValue + Number.EPSILON) / (axis.maxValue - axis.defaultValue + Number.EPSILON));
      }
    } // if there is an avar table, the normalized value is calculated
    // by interpolating between the two nearest mapped values.


    if (this.font.avar) {
      for (var i = 0; i < this.font.avar.segment.length; i++) {
        var segment = this.font.avar.segment[i];

        for (var j = 0; j < segment.correspondence.length; j++) {
          var pair = segment.correspondence[j];

          if (j >= 1 && normalized[i] < pair.fromCoord) {
            var prev = segment.correspondence[j - 1];
            normalized[i] = ((normalized[i] - prev.fromCoord) * (pair.toCoord - prev.toCoord) + Number.EPSILON) / (pair.fromCoord - prev.fromCoord + Number.EPSILON) + prev.toCoord;
            break;
          }
        }
      }
    }

    return normalized;
  };

  _proto.transformPoints = function transformPoints(gid, glyphPoints) {
    if (!this.font.fvar || !this.font.gvar) {
      return;
    }

    var gvar = this.font.gvar;

    if (gid >= gvar.glyphCount) {
      return;
    }

    var offset = gvar.offsets[gid];

    if (offset === gvar.offsets[gid + 1]) {
      return;
    } // Read the gvar data for this glyph


    var stream = this.font.stream;
    stream.pos = offset;

    if (stream.pos >= stream.length) {
      return;
    }

    var tupleCount = stream.readUInt16BE();
    var offsetToData = offset + stream.readUInt16BE();

    if (tupleCount & TUPLES_SHARE_POINT_NUMBERS) {
      var here = stream.pos;
      stream.pos = offsetToData;
      var sharedPoints = this.decodePoints();
      offsetToData = stream.pos;
      stream.pos = here;
    }

    var origPoints = glyphPoints.map(function (pt) {
      return pt.copy();
    });
    tupleCount &= TUPLE_COUNT_MASK;

    for (var i = 0; i < tupleCount; i++) {
      var tupleDataSize = stream.readUInt16BE();
      var tupleIndex = stream.readUInt16BE();

      if (tupleIndex & EMBEDDED_TUPLE_COORD) {
        var tupleCoords = [];

        for (var a = 0; a < gvar.axisCount; a++) {
          tupleCoords.push(stream.readInt16BE() / 16384);
        }
      } else {
        if ((tupleIndex & TUPLE_INDEX_MASK) >= gvar.globalCoordCount) {
          throw new Error('Invalid gvar table');
        }

        var tupleCoords = gvar.globalCoords[tupleIndex & TUPLE_INDEX_MASK];
      }

      if (tupleIndex & INTERMEDIATE_TUPLE) {
        var startCoords = [];

        for (var _a = 0; _a < gvar.axisCount; _a++) {
          startCoords.push(stream.readInt16BE() / 16384);
        }

        var endCoords = [];

        for (var _a2 = 0; _a2 < gvar.axisCount; _a2++) {
          endCoords.push(stream.readInt16BE() / 16384);
        }
      } // Get the factor at which to apply this tuple


      var factor = this.tupleFactor(tupleIndex, tupleCoords, startCoords, endCoords);

      if (factor === 0) {
        offsetToData += tupleDataSize;
        continue;
      }

      var here = stream.pos;
      stream.pos = offsetToData;

      if (tupleIndex & PRIVATE_POINT_NUMBERS) {
        var points = this.decodePoints();
      } else {
        var points = sharedPoints;
      } // points.length = 0 means there are deltas for all points


      var nPoints = points.length === 0 ? glyphPoints.length : points.length;
      var xDeltas = this.decodeDeltas(nPoints);
      var yDeltas = this.decodeDeltas(nPoints);

      if (points.length === 0) {
        // all points
        for (var _i = 0; _i < glyphPoints.length; _i++) {
          var point = glyphPoints[_i];
          point.x += Math.round(xDeltas[_i] * factor);
          point.y += Math.round(yDeltas[_i] * factor);
        }
      } else {
        var outPoints = origPoints.map(function (pt) {
          return pt.copy();
        });
        var hasDelta = glyphPoints.map(function () {
          return false;
        });

        for (var _i2 = 0; _i2 < points.length; _i2++) {
          var idx = points[_i2];

          if (idx < glyphPoints.length) {
            var _point = outPoints[idx];
            hasDelta[idx] = true;
            _point.x += Math.round(xDeltas[_i2] * factor);
            _point.y += Math.round(yDeltas[_i2] * factor);
          }
        }

        this.interpolateMissingDeltas(outPoints, origPoints, hasDelta);

        for (var _i3 = 0; _i3 < glyphPoints.length; _i3++) {
          var deltaX = outPoints[_i3].x - origPoints[_i3].x;
          var deltaY = outPoints[_i3].y - origPoints[_i3].y;
          glyphPoints[_i3].x += deltaX;
          glyphPoints[_i3].y += deltaY;
        }
      }

      offsetToData += tupleDataSize;
      stream.pos = here;
    }
  };

  _proto.decodePoints = function decodePoints() {
    var stream = this.font.stream;
    var count = stream.readUInt8();

    if (count & POINTS_ARE_WORDS) {
      count = (count & POINT_RUN_COUNT_MASK) << 8 | stream.readUInt8();
    }

    var points = new Uint16Array(count);
    var i = 0;
    var point = 0;

    while (i < count) {
      var run = stream.readUInt8();
      var runCount = (run & POINT_RUN_COUNT_MASK) + 1;
      var fn = run & POINTS_ARE_WORDS ? stream.readUInt16 : stream.readUInt8;

      for (var j = 0; j < runCount && i < count; j++) {
        point += fn.call(stream);
        points[i++] = point;
      }
    }

    return points;
  };

  _proto.decodeDeltas = function decodeDeltas(count) {
    var stream = this.font.stream;
    var i = 0;
    var deltas = new Int16Array(count);

    while (i < count) {
      var run = stream.readUInt8();
      var runCount = (run & DELTA_RUN_COUNT_MASK) + 1;

      if (run & DELTAS_ARE_ZERO) {
        i += runCount;
      } else {
        var fn = run & DELTAS_ARE_WORDS ? stream.readInt16BE : stream.readInt8;

        for (var j = 0; j < runCount && i < count; j++) {
          deltas[i++] = fn.call(stream);
        }
      }
    }

    return deltas;
  };

  _proto.tupleFactor = function tupleFactor(tupleIndex, tupleCoords, startCoords, endCoords) {
    var normalized = this.normalizedCoords;
    var gvar = this.font.gvar;
    var factor = 1;

    for (var i = 0; i < gvar.axisCount; i++) {
      if (tupleCoords[i] === 0) {
        continue;
      }

      if (normalized[i] === 0) {
        return 0;
      }

      if ((tupleIndex & INTERMEDIATE_TUPLE) === 0) {
        if (normalized[i] < Math.min(0, tupleCoords[i]) || normalized[i] > Math.max(0, tupleCoords[i])) {
          return 0;
        }

        factor = (factor * normalized[i] + Number.EPSILON) / (tupleCoords[i] + Number.EPSILON);
      } else {
        if (normalized[i] < startCoords[i] || normalized[i] > endCoords[i]) {
          return 0;
        } else if (normalized[i] < tupleCoords[i]) {
          factor = factor * (normalized[i] - startCoords[i] + Number.EPSILON) / (tupleCoords[i] - startCoords[i] + Number.EPSILON);
        } else {
          factor = factor * (endCoords[i] - normalized[i] + Number.EPSILON) / (endCoords[i] - tupleCoords[i] + Number.EPSILON);
        }
      }
    }

    return factor;
  } // Interpolates points without delta values.
  // Needed for the Ø and Q glyphs in Skia.
  // Algorithm from Freetype.
  ;

  _proto.interpolateMissingDeltas = function interpolateMissingDeltas(points, inPoints, hasDelta) {
    if (points.length === 0) {
      return;
    }

    var point = 0;

    while (point < points.length) {
      var firstPoint = point; // find the end point of the contour

      var endPoint = point;
      var pt = points[endPoint];

      while (!pt.endContour) {
        pt = points[++endPoint];
      } // find the first point that has a delta


      while (point <= endPoint && !hasDelta[point]) {
        point++;
      }

      if (point > endPoint) {
        continue;
      }

      var firstDelta = point;
      var curDelta = point;
      point++;

      while (point <= endPoint) {
        // find the next point with a delta, and interpolate intermediate points
        if (hasDelta[point]) {
          this.deltaInterpolate(curDelta + 1, point - 1, curDelta, point, inPoints, points);
          curDelta = point;
        }

        point++;
      } // shift contour if we only have a single delta


      if (curDelta === firstDelta) {
        this.deltaShift(firstPoint, endPoint, curDelta, inPoints, points);
      } else {
        // otherwise, handle the remaining points at the end and beginning of the contour
        this.deltaInterpolate(curDelta + 1, endPoint, curDelta, firstDelta, inPoints, points);

        if (firstDelta > 0) {
          this.deltaInterpolate(firstPoint, firstDelta - 1, curDelta, firstDelta, inPoints, points);
        }
      }

      point = endPoint + 1;
    }
  };

  _proto.deltaInterpolate = function deltaInterpolate(p1, p2, ref1, ref2, inPoints, outPoints) {
    if (p1 > p2) {
      return;
    }

    var iterable = ['x', 'y'];

    for (var i = 0; i < iterable.length; i++) {
      var k = iterable[i];

      if (inPoints[ref1][k] > inPoints[ref2][k]) {
        var p = ref1;
        ref1 = ref2;
        ref2 = p;
      }

      var in1 = inPoints[ref1][k];
      var in2 = inPoints[ref2][k];
      var out1 = outPoints[ref1][k];
      var out2 = outPoints[ref2][k]; // If the reference points have the same coordinate but different
      // delta, inferred delta is zero.  Otherwise interpolate.

      if (in1 !== in2 || out1 === out2) {
        var scale = in1 === in2 ? 0 : (out2 - out1) / (in2 - in1);

        for (var _p = p1; _p <= p2; _p++) {
          var out = inPoints[_p][k];

          if (out <= in1) {
            out += out1 - in1;
          } else if (out >= in2) {
            out += out2 - in2;
          } else {
            out = out1 + (out - in1) * scale;
          }

          outPoints[_p][k] = out;
        }
      }
    }
  };

  _proto.deltaShift = function deltaShift(p1, p2, ref, inPoints, outPoints) {
    var deltaX = outPoints[ref].x - inPoints[ref].x;
    var deltaY = outPoints[ref].y - inPoints[ref].y;

    if (deltaX === 0 && deltaY === 0) {
      return;
    }

    for (var p = p1; p <= p2; p++) {
      if (p !== ref) {
        outPoints[p].x += deltaX;
        outPoints[p].y += deltaY;
      }
    }
  };

  _proto.getAdvanceAdjustment = function getAdvanceAdjustment(gid, table) {
    var outerIndex, innerIndex;

    if (table.advanceWidthMapping) {
      var idx = gid;

      if (idx >= table.advanceWidthMapping.mapCount) {
        idx = table.advanceWidthMapping.mapCount - 1;
      }

      var entryFormat = table.advanceWidthMapping.entryFormat;
      var _table$advanceWidthMa = table.advanceWidthMapping.mapData[idx];
      outerIndex = _table$advanceWidthMa.outerIndex;
      innerIndex = _table$advanceWidthMa.innerIndex;
    } else {
      outerIndex = 0;
      innerIndex = gid;
    }

    return this.getDelta(table.itemVariationStore, outerIndex, innerIndex);
  } // See pseudo code from `Font Variations Overview'
  // in the OpenType specification.
  ;

  _proto.getDelta = function getDelta(itemStore, outerIndex, innerIndex) {
    if (outerIndex >= itemStore.itemVariationData.length) {
      return 0;
    }

    var varData = itemStore.itemVariationData[outerIndex];

    if (innerIndex >= varData.deltaSets.length) {
      return 0;
    }

    var deltaSet = varData.deltaSets[innerIndex];
    var blendVector = this.getBlendVector(itemStore, outerIndex);
    var netAdjustment = 0;

    for (var master = 0; master < varData.regionIndexCount; master++) {
      netAdjustment += deltaSet.deltas[master] * blendVector[master];
    }

    return netAdjustment;
  };

  _proto.getBlendVector = function getBlendVector(itemStore, outerIndex) {
    var varData = itemStore.itemVariationData[outerIndex];

    if (this.blendVectors.has(varData)) {
      return this.blendVectors.get(varData);
    }

    var normalizedCoords = this.normalizedCoords;
    var blendVector = []; // outer loop steps through master designs to be blended

    for (var master = 0; master < varData.regionIndexCount; master++) {
      var scalar = 1;
      var regionIndex = varData.regionIndexes[master];
      var axes = itemStore.variationRegionList.variationRegions[regionIndex]; // inner loop steps through axes in this region

      for (var j = 0; j < axes.length; j++) {
        var axis = axes[j];
        var axisScalar = void 0; // compute the scalar contribution of this axis
        // ignore invalid ranges

        if (axis.startCoord > axis.peakCoord || axis.peakCoord > axis.endCoord) {
          axisScalar = 1;
        } else if (axis.startCoord < 0 && axis.endCoord > 0 && axis.peakCoord !== 0) {
          axisScalar = 1; // peak of 0 means ignore this axis
        } else if (axis.peakCoord === 0) {
          axisScalar = 1; // ignore this region if coords are out of range
        } else if (normalizedCoords[j] < axis.startCoord || normalizedCoords[j] > axis.endCoord) {
          axisScalar = 0; // calculate a proportional factor
        } else {
          if (normalizedCoords[j] === axis.peakCoord) {
            axisScalar = 1;
          } else if (normalizedCoords[j] < axis.peakCoord) {
            axisScalar = (normalizedCoords[j] - axis.startCoord + Number.EPSILON) / (axis.peakCoord - axis.startCoord + Number.EPSILON);
          } else {
            axisScalar = (axis.endCoord - normalizedCoords[j] + Number.EPSILON) / (axis.endCoord - axis.peakCoord + Number.EPSILON);
          }
        } // take product of all the axis scalars


        scalar *= axisScalar;
      }

      blendVector[master] = scalar;
    }

    this.blendVectors.set(varData, blendVector);
    return blendVector;
  };

  return GlyphVariationProcessor;
}();

var clone_1 = createCommonjsModule(function (module) {
  var clone = function () {
    /**
     * Clones (copies) an Object using deep copying.
     *
     * This function supports circular references by default, but if you are certain
     * there are no circular references in your object, you can save some CPU time
     * by calling clone(obj, false).
     *
     * Caution: if `circular` is false and `parent` contains circular references,
     * your program may enter an infinite loop and crash.
     *
     * @param `parent` - the object to be cloned
     * @param `circular` - set to true if the object to be cloned may contain
     *    circular references. (optional - true by default)
     * @param `depth` - set to a number if the object is only to be cloned to
     *    a particular depth. (optional - defaults to Infinity)
     * @param `prototype` - sets the prototype to be used when cloning an object.
     *    (optional - defaults to parent prototype).
    */

    function clone(parent, circular, depth, prototype) {
      var filter;

      if (typeof circular === 'object') {
        depth = circular.depth;
        prototype = circular.prototype;
        filter = circular.filter;
        circular = circular.circular;
      } // maintain two arrays for circular references, where corresponding parents
      // and children have the same index


      var allParents = [];
      var allChildren = [];
      var useBuffer = typeof Buffer != 'undefined';
      if (typeof circular == 'undefined') circular = true;
      if (typeof depth == 'undefined') depth = Infinity; // recurse this function so we don't reset allParents and allChildren

      function _clone(parent, depth) {
        // cloning null always returns null
        if (parent === null) return null;
        if (depth == 0) return parent;
        var child;
        var proto;

        if (typeof parent != 'object') {
          return parent;
        }

        if (clone.__isArray(parent)) {
          child = [];
        } else if (clone.__isRegExp(parent)) {
          child = new RegExp(parent.source, __getRegExpFlags(parent));
          if (parent.lastIndex) child.lastIndex = parent.lastIndex;
        } else if (clone.__isDate(parent)) {
          child = new Date(parent.getTime());
        } else if (useBuffer && isBuffer(parent)) {
          if (Buffer.allocUnsafe) {
            // Node.js >= 4.5.0
            child = Buffer.allocUnsafe(parent.length);
          } else {
            // Older Node.js versions
            child = new Buffer(parent.length);
          }

          parent.copy(child);
          return child;
        } else {
          if (typeof prototype == 'undefined') {
            proto = Object.getPrototypeOf(parent);
            child = Object.create(proto);
          } else {
            child = Object.create(prototype);
            proto = prototype;
          }
        }

        if (circular) {
          var index = allParents.indexOf(parent);

          if (index != -1) {
            return allChildren[index];
          }

          allParents.push(parent);
          allChildren.push(child);
        }

        for (var i in parent) {
          var attrs;

          if (proto) {
            attrs = Object.getOwnPropertyDescriptor(proto, i);
          }

          if (attrs && attrs.set == null) {
            continue;
          }

          child[i] = _clone(parent[i], depth - 1);
        }

        return child;
      }

      return _clone(parent, depth);
    }
    /**
     * Simple flat clone using prototype, accepts only objects, usefull for property
     * override on FLAT configuration object (no nested props).
     *
     * USE WITH CAUTION! This may not behave as you wish if you do not know how this
     * works.
     */


    clone.clonePrototype = function clonePrototype(parent) {
      if (parent === null) return null;

      var c = function c() {};

      c.prototype = parent;
      return new c();
    }; // private utility functions


    function __objToStr(o) {
      return Object.prototype.toString.call(o);
    }
    clone.__objToStr = __objToStr;

    function __isDate(o) {
      return typeof o === 'object' && __objToStr(o) === '[object Date]';
    }
    clone.__isDate = __isDate;

    function __isArray(o) {
      return typeof o === 'object' && __objToStr(o) === '[object Array]';
    }
    clone.__isArray = __isArray;

    function __isRegExp(o) {
      return typeof o === 'object' && __objToStr(o) === '[object RegExp]';
    }
    clone.__isRegExp = __isRegExp;

    function __getRegExpFlags(re) {
      var flags = '';
      if (re.global) flags += 'g';
      if (re.ignoreCase) flags += 'i';
      if (re.multiline) flags += 'm';
      return flags;
    }
    clone.__getRegExpFlags = __getRegExpFlags;
    return clone;
  }();

  if (module.exports) {
    module.exports = clone;
  }
});

var Subset =
/*#__PURE__*/
function () {
  function Subset(font) {
    this.font = font;
    this.glyphs = [];
    this.mapping = {}; // always include the missing glyph

    this.includeGlyph(0);
  }

  var _proto = Subset.prototype;

  _proto.includeGlyph = function includeGlyph(glyph) {
    if (typeof glyph === 'object') {
      glyph = glyph.id;
    }

    if (this.mapping[glyph] == null) {
      this.glyphs.push(glyph);
      this.mapping[glyph] = this.glyphs.length - 1;
    }

    return this.mapping[glyph];
  };

  _proto.encodeStream = function encodeStream() {
    var _this = this;

    var s = new restructure.EncodeStream();
    nextTick(function () {
      _this.encode(s);

      return s.end();
    });
    return s;
  };

  return Subset;
}();

var ON_CURVE$1 = 1 << 0;
var X_SHORT_VECTOR$1 = 1 << 1;
var Y_SHORT_VECTOR$1 = 1 << 2;
var REPEAT$1 = 1 << 3;
var SAME_X$1 = 1 << 4;
var SAME_Y$1 = 1 << 5;

var Point$1 =
/*#__PURE__*/
function () {
  function Point() {}

  Point.size = function size(val) {
    return val >= 0 && val <= 255 ? 1 : 2;
  };

  Point.encode = function encode(stream, value) {
    if (value >= 0 && value <= 255) {
      stream.writeUInt8(value);
    } else {
      stream.writeInt16BE(value);
    }
  };

  return Point;
}();

var Glyf = new restructure.Struct({
  numberOfContours: restructure.int16,
  // if negative, this is a composite glyph
  xMin: restructure.int16,
  yMin: restructure.int16,
  xMax: restructure.int16,
  yMax: restructure.int16,
  endPtsOfContours: new restructure.Array(restructure.uint16, 'numberOfContours'),
  instructions: new restructure.Array(restructure.uint8, restructure.uint16),
  flags: new restructure.Array(restructure.uint8, 0),
  xPoints: new restructure.Array(Point$1, 0),
  yPoints: new restructure.Array(Point$1, 0)
});
/**
 * Encodes TrueType glyph outlines
 */

var TTFGlyphEncoder =
/*#__PURE__*/
function () {
  function TTFGlyphEncoder() {}

  var _proto = TTFGlyphEncoder.prototype;

  _proto.encodeSimple = function encodeSimple(path, instructions) {
    if (instructions === void 0) {
      instructions = [];
    }

    var endPtsOfContours = [];
    var xPoints = [];
    var yPoints = [];
    var flags = [];
    var same = 0;
    var lastX = 0,
        lastY = 0,
        lastFlag = 0;
    var pointCount = 0;

    for (var i = 0; i < path.commands.length; i++) {
      var c = path.commands[i];

      for (var j = 0; j < c.args.length; j += 2) {
        var x = c.args[j];
        var y = c.args[j + 1];
        var flag = 0; // If the ending point of a quadratic curve is the midpoint
        // between the control point and the control point of the next
        // quadratic curve, we can omit the ending point.

        if (c.command === 'quadraticCurveTo' && j === 2) {
          var next = path.commands[i + 1];

          if (next && next.command === 'quadraticCurveTo') {
            var midX = (lastX + next.args[0]) / 2;
            var midY = (lastY + next.args[1]) / 2;

            if (x === midX && y === midY) {
              continue;
            }
          }
        } // All points except control points are on curve.


        if (!(c.command === 'quadraticCurveTo' && j === 0)) {
          flag |= ON_CURVE$1;
        }

        flag = this._encodePoint(x, lastX, xPoints, flag, X_SHORT_VECTOR$1, SAME_X$1);
        flag = this._encodePoint(y, lastY, yPoints, flag, Y_SHORT_VECTOR$1, SAME_Y$1);

        if (flag === lastFlag && same < 255) {
          flags[flags.length - 1] |= REPEAT$1;
          same++;
        } else {
          if (same > 0) {
            flags.push(same);
            same = 0;
          }

          flags.push(flag);
          lastFlag = flag;
        }

        lastX = x;
        lastY = y;
        pointCount++;
      }

      if (c.command === 'closePath') {
        endPtsOfContours.push(pointCount - 1);
      }
    } // Close the path if the last command didn't already


    if (path.commands.length > 1 && path.commands[path.commands.length - 1].command !== 'closePath') {
      endPtsOfContours.push(pointCount - 1);
    }

    var bbox = path.bbox;
    var glyf = {
      numberOfContours: endPtsOfContours.length,
      xMin: bbox.minX,
      yMin: bbox.minY,
      xMax: bbox.maxX,
      yMax: bbox.maxY,
      endPtsOfContours: endPtsOfContours,
      instructions: instructions,
      flags: flags,
      xPoints: xPoints,
      yPoints: yPoints
    };
    var size = Glyf.size(glyf);
    var tail = 4 - size % 4;
    var stream = new restructure.EncodeStream(size + tail);
    Glyf.encode(stream, glyf); // Align to 4-byte length

    if (tail !== 0) {
      stream.fill(0, tail);
    }

    return stream.buffer;
  };

  _proto._encodePoint = function _encodePoint(value, last, points, flag, shortFlag, sameFlag) {
    var diff = value - last;

    if (value === last) {
      flag |= sameFlag;
    } else {
      if (-255 <= diff && diff <= 255) {
        flag |= shortFlag;

        if (diff < 0) {
          diff = -diff;
        } else {
          flag |= sameFlag;
        }
      }

      points.push(diff);
    }

    return flag;
  };

  return TTFGlyphEncoder;
}();

var TTFSubset =
/*#__PURE__*/
function (_Subset) {
  _inheritsLoose(TTFSubset, _Subset);

  function TTFSubset(font) {
    var _this;

    _this = _Subset.call(this, font) || this;
    _this.glyphEncoder = new TTFGlyphEncoder();
    return _this;
  }

  var _proto = TTFSubset.prototype;

  _proto._addGlyph = function _addGlyph(gid) {
    var glyph = this.font.getGlyph(gid);

    var glyf = glyph._decode(); // get the offset to the glyph from the loca table


    var curOffset = this.font.loca.offsets[gid];
    var nextOffset = this.font.loca.offsets[gid + 1];

    var stream = this.font._getTableStream('glyf');

    stream.pos += curOffset;
    var buffer = stream.readBuffer(nextOffset - curOffset); // if it is a compound glyph, include its components

    if (glyf && glyf.numberOfContours < 0) {
      buffer = new Buffer(buffer);

      for (var _iterator = glyf.components, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
        var _ref;

        if (_isArray) {
          if (_i >= _iterator.length) break;
          _ref = _iterator[_i++];
        } else {
          _i = _iterator.next();
          if (_i.done) break;
          _ref = _i.value;
        }

        var component = _ref;
        gid = this.includeGlyph(component.glyphID);
        buffer.writeUInt16BE(gid, component.pos);
      }
    } else if (glyf && this.font._variationProcessor) {
      // If this is a TrueType variation glyph, re-encode the path
      buffer = this.glyphEncoder.encodeSimple(glyph.path, glyf.instructions);
    }

    this.glyf.push(buffer);
    this.loca.offsets.push(this.offset);
    this.hmtx.metrics.push({
      advance: glyph.advanceWidth,
      bearing: glyph._getMetrics().leftBearing
    });
    this.offset += buffer.length;
    return this.glyf.length - 1;
  };

  _proto.encode = function encode(stream) {
    // tables required by PDF spec:
    //   head, hhea, loca, maxp, cvt , prep, glyf, hmtx, fpgm
    //
    // additional tables required for standalone fonts:
    //   name, cmap, OS/2, post
    this.glyf = [];
    this.offset = 0;
    this.loca = {
      offsets: []
    };
    this.hmtx = {
      metrics: [],
      bearings: []
    }; // include all the glyphs
    // not using a for loop because we need to support adding more
    // glyphs to the array as we go, and CoffeeScript caches the length.

    var i = 0;

    while (i < this.glyphs.length) {
      this._addGlyph(this.glyphs[i++]);
    }

    var maxp = clone_1(this.font.maxp);
    maxp.numGlyphs = this.glyf.length;
    this.loca.offsets.push(this.offset);
    tables.loca.preEncode.call(this.loca);
    var head = clone_1(this.font.head);
    head.indexToLocFormat = this.loca.version;
    var hhea = clone_1(this.font.hhea);
    hhea.numberOfMetrics = this.hmtx.metrics.length; // map = []
    // for index in [0...256]
    //     if index < @numGlyphs
    //         map[index] = index
    //     else
    //         map[index] = 0
    //
    // cmapTable =
    //     version: 0
    //     length: 262
    //     language: 0
    //     codeMap: map
    //
    // cmap =
    //     version: 0
    //     numSubtables: 1
    //     tables: [
    //         platformID: 1
    //         encodingID: 0
    //         table: cmapTable
    //     ]
    // TODO: subset prep, cvt, fpgm?

    Directory.encode(stream, {
      tables: {
        head: head,
        hhea: hhea,
        loca: this.loca,
        maxp: maxp,
        'cvt ': this.font['cvt '],
        prep: this.font.prep,
        glyf: this.glyf,
        hmtx: this.hmtx,
        fpgm: this.font.fpgm // name: clone @font.name
        // 'OS/2': clone @font['OS/2']
        // post: clone @font.post
        // cmap: cmap

      }
    });
  };

  return TTFSubset;
}(Subset);

var CFFSubset =
/*#__PURE__*/
function (_Subset) {
  _inheritsLoose(CFFSubset, _Subset);

  function CFFSubset(font) {
    var _this;

    _this = _Subset.call(this, font) || this;
    _this.cff = _this.font['CFF '];

    if (!_this.cff) {
      throw new Error('Not a CFF Font');
    }

    return _this;
  }

  var _proto = CFFSubset.prototype;

  _proto.subsetCharstrings = function subsetCharstrings() {
    this.charstrings = [];
    var gsubrs = {};

    for (var _iterator = this.glyphs, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
      var _ref;

      if (_isArray) {
        if (_i >= _iterator.length) break;
        _ref = _iterator[_i++];
      } else {
        _i = _iterator.next();
        if (_i.done) break;
        _ref = _i.value;
      }

      var gid = _ref;
      this.charstrings.push(this.cff.getCharString(gid));
      var glyph = this.font.getGlyph(gid);
      var path = glyph.path; // this causes the glyph to be parsed

      for (var subr in glyph._usedGsubrs) {
        gsubrs[subr] = true;
      }
    }

    this.gsubrs = this.subsetSubrs(this.cff.globalSubrIndex, gsubrs);
  };

  _proto.subsetSubrs = function subsetSubrs(subrs, used) {
    var res = [];

    for (var i = 0; i < subrs.length; i++) {
      var subr = subrs[i];

      if (used[i]) {
        this.cff.stream.pos = subr.offset;
        res.push(this.cff.stream.readBuffer(subr.length));
      } else {
        res.push(new Buffer([11])); // return
      }
    }

    return res;
  };

  _proto.subsetFontdict = function subsetFontdict(topDict) {
    topDict.FDArray = [];
    topDict.FDSelect = {
      version: 0,
      fds: []
    };
    var used_fds = {};
    var used_subrs = [];

    for (var _iterator2 = this.glyphs, _isArray2 = Array.isArray(_iterator2), _i2 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
      var _ref2;

      if (_isArray2) {
        if (_i2 >= _iterator2.length) break;
        _ref2 = _iterator2[_i2++];
      } else {
        _i2 = _iterator2.next();
        if (_i2.done) break;
        _ref2 = _i2.value;
      }

      var gid = _ref2;
      var fd = this.cff.fdForGlyph(gid);

      if (fd == null) {
        continue;
      }

      if (!used_fds[fd]) {
        topDict.FDArray.push(Object.assign({}, this.cff.topDict.FDArray[fd]));
        used_subrs.push({});
      }

      used_fds[fd] = true;
      topDict.FDSelect.fds.push(topDict.FDArray.length - 1);
      var glyph = this.font.getGlyph(gid);
      var path = glyph.path; // this causes the glyph to be parsed

      for (var subr in glyph._usedSubrs) {
        used_subrs[used_subrs.length - 1][subr] = true;
      }
    }

    for (var i = 0; i < topDict.FDArray.length; i++) {
      var dict = topDict.FDArray[i];
      delete dict.FontName;

      if (dict.Private && dict.Private.Subrs) {
        dict.Private = Object.assign({}, dict.Private);
        dict.Private.Subrs = this.subsetSubrs(dict.Private.Subrs, used_subrs[i]);
      }
    }

    return;
  };

  _proto.createCIDFontdict = function createCIDFontdict(topDict) {
    var used_subrs = {};

    for (var _iterator3 = this.glyphs, _isArray3 = Array.isArray(_iterator3), _i3 = 0, _iterator3 = _isArray3 ? _iterator3 : _iterator3[Symbol.iterator]();;) {
      var _ref3;

      if (_isArray3) {
        if (_i3 >= _iterator3.length) break;
        _ref3 = _iterator3[_i3++];
      } else {
        _i3 = _iterator3.next();
        if (_i3.done) break;
        _ref3 = _i3.value;
      }

      var gid = _ref3;
      var glyph = this.font.getGlyph(gid);
      var path = glyph.path; // this causes the glyph to be parsed

      for (var subr in glyph._usedSubrs) {
        used_subrs[subr] = true;
      }
    }

    var privateDict = Object.assign({}, this.cff.topDict.Private);

    if (this.cff.topDict.Private && this.cff.topDict.Private.Subrs) {
      privateDict.Subrs = this.subsetSubrs(this.cff.topDict.Private.Subrs, used_subrs);
    }

    topDict.FDArray = [{
      Private: privateDict
    }];
    return topDict.FDSelect = {
      version: 3,
      nRanges: 1,
      ranges: [{
        first: 0,
        fd: 0
      }],
      sentinel: this.charstrings.length
    };
  };

  _proto.addString = function addString(string) {
    if (!string) {
      return null;
    }

    if (!this.strings) {
      this.strings = [];
    }

    this.strings.push(string);
    return standardStrings.length + this.strings.length - 1;
  };

  _proto.encode = function encode(stream) {
    this.subsetCharstrings();
    var charset = {
      version: this.charstrings.length > 255 ? 2 : 1,
      ranges: [{
        first: 1,
        nLeft: this.charstrings.length - 2
      }]
    };
    var topDict = Object.assign({}, this.cff.topDict);
    topDict.Private = null;
    topDict.charset = charset;
    topDict.Encoding = null;
    topDict.CharStrings = this.charstrings;
    var _arr = ['version', 'Notice', 'Copyright', 'FullName', 'FamilyName', 'Weight', 'PostScript', 'BaseFontName', 'FontName'];

    for (var _i4 = 0; _i4 < _arr.length; _i4++) {
      var key = _arr[_i4];
      topDict[key] = this.addString(this.cff.string(topDict[key]));
    }

    topDict.ROS = [this.addString('Adobe'), this.addString('Identity'), 0];
    topDict.CIDCount = this.charstrings.length;

    if (this.cff.isCIDFont) {
      this.subsetFontdict(topDict);
    } else {
      this.createCIDFontdict(topDict);
    }

    var top = {
      version: 1,
      hdrSize: this.cff.hdrSize,
      offSize: this.cff.length,
      header: this.cff.header,
      nameIndex: [this.cff.postscriptName],
      topDictIndex: [topDict],
      stringIndex: this.strings,
      globalSubrIndex: this.gsubrs
    };
    CFFTop.encode(stream, top);
  };

  return CFFSubset;
}(Subset);

var _class$4;
/**
 * This is the base class for all SFNT-based font formats in fontkit.
 * It supports TrueType, and PostScript glyphs, and several color glyph formats.
 */

var TTFFont = (_class$4 =
/*#__PURE__*/
function () {
  TTFFont.probe = function probe(buffer) {
    var format = buffer.toString('ascii', 0, 4);
    return format === 'true' || format === 'OTTO' || format === String.fromCharCode(0, 1, 0, 0);
  };

  function TTFFont(stream, variationCoords) {
    if (variationCoords === void 0) {
      variationCoords = null;
    }

    this.stream = stream;
    this.variationCoords = variationCoords;
    this._directoryPos = this.stream.pos;
    this._tables = {};
    this._glyphs = {};

    this._decodeDirectory(); // define properties for each table to lazily parse


    for (var tag in this.directory.tables) {
      var table = this.directory.tables[tag];

      if (tables[tag] && table.length > 0) {
        Object.defineProperty(this, tag, {
          get: this._getTable.bind(this, table)
        });
      }
    }
  }

  var _proto = TTFFont.prototype;

  _proto._getTable = function _getTable(table) {
    if (!(table.tag in this._tables)) {
      try {
        this._tables[table.tag] = this._decodeTable(table);
      } catch (e) {
        if (fontkit.logErrors) {
          console.error("Error decoding table " + table.tag);
          console.error(e.stack);
        }
      }
    }

    return this._tables[table.tag];
  };

  _proto._getTableStream = function _getTableStream(tag) {
    var table = this.directory.tables[tag];

    if (table) {
      this.stream.pos = table.offset;
      return this.stream;
    }

    return null;
  };

  _proto._decodeDirectory = function _decodeDirectory() {
    return this.directory = Directory.decode(this.stream, {
      _startOffset: 0
    });
  };

  _proto._decodeTable = function _decodeTable(table) {
    var pos = this.stream.pos;

    var stream = this._getTableStream(table.tag);

    var result = tables[table.tag].decode(stream, this, table.length);
    this.stream.pos = pos;
    return result;
  }
  /**
   * The unique PostScript name for this font
   * @type {string}
   */
  ;

  /**
   * Gets a string from the font's `name` table
   * `lang` is a BCP-47 language code.
   * @return {string}
   */
  _proto.getName = function getName(key, lang) {
    if (lang === void 0) {
      lang = 'en';
    }

    var record = this.name.records[key];

    if (record) {
      return record[lang];
    }

    return null;
  }
  /**
   * The font's full name, e.g. "Helvetica Bold"
   * @type {string}
   */
  ;

  /**
   * Returns whether there is glyph in the font for the given unicode code point.
   *
   * @param {number} codePoint
   * @return {boolean}
   */
  _proto.hasGlyphForCodePoint = function hasGlyphForCodePoint(codePoint) {
    return !!this._cmapProcessor.lookup(codePoint);
  }
  /**
   * Maps a single unicode code point to a Glyph object.
   * Does not perform any advanced substitutions (there is no context to do so).
   *
   * @param {number} codePoint
   * @return {Glyph}
   */
  ;

  _proto.glyphForCodePoint = function glyphForCodePoint(codePoint) {
    return this.getGlyph(this._cmapProcessor.lookup(codePoint), [codePoint]);
  }
  /**
   * Returns an array of Glyph objects for the given string.
   * This is only a one-to-one mapping from characters to glyphs.
   * For most uses, you should use font.layout (described below), which
   * provides a much more advanced mapping supporting AAT and OpenType shaping.
   *
   * @param {string} string
   * @return {Glyph[]}
   */
  ;

  _proto.glyphsForString = function glyphsForString(string) {
    var glyphs = [];
    var len = string.length;
    var idx = 0;
    var last = -1;
    var state = -1;

    while (idx <= len) {
      var code = 0;
      var nextState = 0;

      if (idx < len) {
        // Decode the next codepoint from UTF 16
        code = string.charCodeAt(idx++);

        if (0xd800 <= code && code <= 0xdbff && idx < len) {
          var next = string.charCodeAt(idx);

          if (0xdc00 <= next && next <= 0xdfff) {
            idx++;
            code = ((code & 0x3ff) << 10) + (next & 0x3ff) + 0x10000;
          }
        } // Compute the next state: 1 if the next codepoint is a variation selector, 0 otherwise.


        nextState = 0xfe00 <= code && code <= 0xfe0f || 0xe0100 <= code && code <= 0xe01ef ? 1 : 0;
      } else {
        idx++;
      }

      if (state === 0 && nextState === 1) {
        // Variation selector following normal codepoint.
        glyphs.push(this.getGlyph(this._cmapProcessor.lookup(last, code), [last, code]));
      } else if (state === 0 && nextState === 0) {
        // Normal codepoint following normal codepoint.
        glyphs.push(this.glyphForCodePoint(last));
      }

      last = code;
      state = nextState;
    }

    return glyphs;
  };

  /**
   * Returns a GlyphRun object, which includes an array of Glyphs and GlyphPositions for the given string.
   *
   * @param {string} string
   * @param {string[]} [userFeatures]
   * @param {string} [script]
   * @param {string} [language]
   * @param {string} [direction]
   * @return {GlyphRun}
   */
  _proto.layout = function layout(string, userFeatures, script, language, direction) {
    return this._layoutEngine.layout(string, userFeatures, script, language, direction);
  }
  /**
   * Returns an array of strings that map to the given glyph id.
   * @param {number} gid - glyph id
   */
  ;

  _proto.stringsForGlyph = function stringsForGlyph(gid) {
    return this._layoutEngine.stringsForGlyph(gid);
  }
  /**
   * An array of all [OpenType feature tags](https://www.microsoft.com/typography/otspec/featuretags.htm)
   * (or mapped AAT tags) supported by the font.
   * The features parameter is an array of OpenType feature tags to be applied in addition to the default set.
   * If this is an AAT font, the OpenType feature tags are mapped to AAT features.
   *
   * @type {string[]}
   */
  ;

  _proto.getAvailableFeatures = function getAvailableFeatures(script, language) {
    return this._layoutEngine.getAvailableFeatures(script, language);
  };

  _proto._getBaseGlyph = function _getBaseGlyph(glyph, characters) {
    if (characters === void 0) {
      characters = [];
    }

    if (!this._glyphs[glyph]) {
      if (this.directory.tables.glyf) {
        this._glyphs[glyph] = new TTFGlyph(glyph, characters, this);
      } else if (this.directory.tables['CFF '] || this.directory.tables.CFF2) {
        this._glyphs[glyph] = new CFFGlyph(glyph, characters, this);
      }
    }

    return this._glyphs[glyph] || null;
  }
  /**
   * Returns a glyph object for the given glyph id.
   * You can pass the array of code points this glyph represents for
   * your use later, and it will be stored in the glyph object.
   *
   * @param {number} glyph
   * @param {number[]} characters
   * @return {Glyph}
   */
  ;

  _proto.getGlyph = function getGlyph(glyph, characters) {
    if (characters === void 0) {
      characters = [];
    }

    if (!this._glyphs[glyph]) {
      if (this.directory.tables.sbix) {
        this._glyphs[glyph] = new SBIXGlyph(glyph, characters, this);
      } else if (this.directory.tables.COLR && this.directory.tables.CPAL) {
        this._glyphs[glyph] = new COLRGlyph(glyph, characters, this);
      } else {
        this._getBaseGlyph(glyph, characters);
      }
    }

    return this._glyphs[glyph] || null;
  }
  /**
   * Returns a Subset for this font.
   * @return {Subset}
   */
  ;

  _proto.createSubset = function createSubset() {
    if (this.directory.tables['CFF ']) {
      return new CFFSubset(this);
    }

    return new TTFSubset(this);
  }
  /**
   * Returns an object describing the available variation axes
   * that this font supports. Keys are setting tags, and values
   * contain the axis name, range, and default value.
   *
   * @type {object}
   */
  ;

  /**
   * Returns a new font with the given variation settings applied.
   * Settings can either be an instance name, or an object containing
   * variation tags as specified by the `variationAxes` property.
   *
   * @param {object} settings
   * @return {TTFFont}
   */
  _proto.getVariation = function getVariation(settings) {
    if (!(this.directory.tables.fvar && (this.directory.tables.gvar && this.directory.tables.glyf || this.directory.tables.CFF2))) {
      throw new Error('Variations require a font with the fvar, gvar and glyf, or CFF2 tables.');
    }

    if (typeof settings === 'string') {
      settings = this.namedVariations[settings];
    }

    if (typeof settings !== 'object') {
      throw new Error('Variation settings must be either a variation name or settings object.');
    } // normalize the coordinates


    var coords = this.fvar.axis.map(function (axis, i) {
      var axisTag = axis.axisTag.trim();

      if (axisTag in settings) {
        return Math.max(axis.minValue, Math.min(axis.maxValue, settings[axisTag]));
      } else {
        return axis.defaultValue;
      }
    });
    var stream = new restructure.DecodeStream(this.stream.buffer);
    stream.pos = this._directoryPos;
    var font = new TTFFont(stream, coords);
    font._tables = this._tables;
    return font;
  };

  // Standardized format plugin API
  _proto.getFont = function getFont(name) {
    return this.getVariation(name);
  };

  _createClass(TTFFont, [{
    key: "postscriptName",
    get: function get() {
      var name = this.name.records.postscriptName;

      if (name) {
        var lang = Object.keys(name)[0];
        return name[lang];
      }

      return null;
    }
  }, {
    key: "fullName",
    get: function get() {
      return this.getName('fullName');
    }
    /**
     * The font's family name, e.g. "Helvetica"
     * @type {string}
     */

  }, {
    key: "familyName",
    get: function get() {
      return this.getName('fontFamily');
    }
    /**
     * The font's sub-family, e.g. "Bold".
     * @type {string}
     */

  }, {
    key: "subfamilyName",
    get: function get() {
      return this.getName('fontSubfamily');
    }
    /**
     * The font's copyright information
     * @type {string}
     */

  }, {
    key: "copyright",
    get: function get() {
      return this.getName('copyright');
    }
    /**
     * The font's version number
     * @type {string}
     */

  }, {
    key: "version",
    get: function get() {
      return this.getName('version');
    }
    /**
     * The font’s [ascender](https://en.wikipedia.org/wiki/Ascender_(typography))
     * @type {number}
     */

  }, {
    key: "ascent",
    get: function get() {
      return this.hhea.ascent;
    }
    /**
     * The font’s [descender](https://en.wikipedia.org/wiki/Descender)
     * @type {number}
     */

  }, {
    key: "descent",
    get: function get() {
      return this.hhea.descent;
    }
    /**
     * The amount of space that should be included between lines
     * @type {number}
     */

  }, {
    key: "lineGap",
    get: function get() {
      return this.hhea.lineGap;
    }
    /**
     * The offset from the normal underline position that should be used
     * @type {number}
     */

  }, {
    key: "underlinePosition",
    get: function get() {
      return this.post.underlinePosition;
    }
    /**
     * The weight of the underline that should be used
     * @type {number}
     */

  }, {
    key: "underlineThickness",
    get: function get() {
      return this.post.underlineThickness;
    }
    /**
     * If this is an italic font, the angle the cursor should be drawn at to match the font design
     * @type {number}
     */

  }, {
    key: "italicAngle",
    get: function get() {
      return this.post.italicAngle;
    }
    /**
     * The height of capital letters above the baseline.
     * See [here](https://en.wikipedia.org/wiki/Cap_height) for more details.
     * @type {number}
     */

  }, {
    key: "capHeight",
    get: function get() {
      var os2 = this['OS/2'];
      return os2 ? os2.capHeight : this.ascent;
    }
    /**
     * The height of lower case letters in the font.
     * See [here](https://en.wikipedia.org/wiki/X-height) for more details.
     * @type {number}
     */

  }, {
    key: "xHeight",
    get: function get() {
      var os2 = this['OS/2'];
      return os2 ? os2.xHeight : 0;
    }
    /**
     * The number of glyphs in the font.
     * @type {number}
     */

  }, {
    key: "numGlyphs",
    get: function get() {
      return this.maxp.numGlyphs;
    }
    /**
     * The size of the font’s internal coordinate grid
     * @type {number}
     */

  }, {
    key: "unitsPerEm",
    get: function get() {
      return this.head.unitsPerEm;
    }
    /**
     * The font’s bounding box, i.e. the box that encloses all glyphs in the font.
     * @type {BBox}
     */

  }, {
    key: "bbox",
    get: function get() {
      return Object.freeze(new BBox(this.head.xMin, this.head.yMin, this.head.xMax, this.head.yMax));
    }
  }, {
    key: "_cmapProcessor",
    get: function get() {
      return new CmapProcessor(this.cmap);
    }
    /**
     * An array of all of the unicode code points supported by the font.
     * @type {number[]}
     */

  }, {
    key: "characterSet",
    get: function get() {
      return this._cmapProcessor.getCharacterSet();
    }
  }, {
    key: "_layoutEngine",
    get: function get() {
      return new LayoutEngine(this);
    }
  }, {
    key: "availableFeatures",
    get: function get() {
      return this._layoutEngine.getAvailableFeatures();
    }
  }, {
    key: "variationAxes",
    get: function get() {
      var res = {};

      if (!this.fvar) {
        return res;
      }

      for (var _iterator = this.fvar.axis, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
        var _ref;

        if (_isArray) {
          if (_i >= _iterator.length) break;
          _ref = _iterator[_i++];
        } else {
          _i = _iterator.next();
          if (_i.done) break;
          _ref = _i.value;
        }

        var axis = _ref;
        res[axis.axisTag.trim()] = {
          name: axis.name.en,
          min: axis.minValue,
          default: axis.defaultValue,
          max: axis.maxValue
        };
      }

      return res;
    }
    /**
     * Returns an object describing the named variation instances
     * that the font designer has specified. Keys are variation names
     * and values are the variation settings for this instance.
     *
     * @type {object}
     */

  }, {
    key: "namedVariations",
    get: function get() {
      var res = {};

      if (!this.fvar) {
        return res;
      }

      for (var _iterator2 = this.fvar.instance, _isArray2 = Array.isArray(_iterator2), _i2 = 0, _iterator2 = _isArray2 ? _iterator2 : _iterator2[Symbol.iterator]();;) {
        var _ref2;

        if (_isArray2) {
          if (_i2 >= _iterator2.length) break;
          _ref2 = _iterator2[_i2++];
        } else {
          _i2 = _iterator2.next();
          if (_i2.done) break;
          _ref2 = _i2.value;
        }

        var instance = _ref2;
        var settings = {};

        for (var i = 0; i < this.fvar.axis.length; i++) {
          var axis = this.fvar.axis[i];
          settings[axis.axisTag.trim()] = instance.coord[i];
        }

        res[instance.name.en] = settings;
      }

      return res;
    }
  }, {
    key: "_variationProcessor",
    get: function get() {
      if (!this.fvar) {
        return null;
      }

      var variationCoords = this.variationCoords; // Ignore if no variation coords and not CFF2

      if (!variationCoords && !this.CFF2) {
        return null;
      }

      if (!variationCoords) {
        variationCoords = this.fvar.axis.map(function (axis) {
          return axis.defaultValue;
        });
      }

      return new GlyphVariationProcessor(this, variationCoords);
    }
  }]);

  return TTFFont;
}(), (_applyDecoratedDescriptor(_class$4.prototype, "bbox", [cache], Object.getOwnPropertyDescriptor(_class$4.prototype, "bbox"), _class$4.prototype), _applyDecoratedDescriptor(_class$4.prototype, "_cmapProcessor", [cache], Object.getOwnPropertyDescriptor(_class$4.prototype, "_cmapProcessor"), _class$4.prototype), _applyDecoratedDescriptor(_class$4.prototype, "characterSet", [cache], Object.getOwnPropertyDescriptor(_class$4.prototype, "characterSet"), _class$4.prototype), _applyDecoratedDescriptor(_class$4.prototype, "_layoutEngine", [cache], Object.getOwnPropertyDescriptor(_class$4.prototype, "_layoutEngine"), _class$4.prototype), _applyDecoratedDescriptor(_class$4.prototype, "variationAxes", [cache], Object.getOwnPropertyDescriptor(_class$4.prototype, "variationAxes"), _class$4.prototype), _applyDecoratedDescriptor(_class$4.prototype, "namedVariations", [cache], Object.getOwnPropertyDescriptor(_class$4.prototype, "namedVariations"), _class$4.prototype), _applyDecoratedDescriptor(_class$4.prototype, "_variationProcessor", [cache], Object.getOwnPropertyDescriptor(_class$4.prototype, "_variationProcessor"), _class$4.prototype)), _class$4);

var WOFFDirectoryEntry = new restructure.Struct({
  tag: new restructure.String(4),
  offset: new restructure.Pointer(restructure.uint32, 'void', {
    type: 'global'
  }),
  compLength: restructure.uint32,
  length: restructure.uint32,
  origChecksum: restructure.uint32
});
var WOFFDirectory = new restructure.Struct({
  tag: new restructure.String(4),
  // should be 'wOFF'
  flavor: restructure.uint32,
  length: restructure.uint32,
  numTables: restructure.uint16,
  reserved: new restructure.Reserved(restructure.uint16),
  totalSfntSize: restructure.uint32,
  majorVersion: restructure.uint16,
  minorVersion: restructure.uint16,
  metaOffset: restructure.uint32,
  metaLength: restructure.uint32,
  metaOrigLength: restructure.uint32,
  privOffset: restructure.uint32,
  privLength: restructure.uint32,
  tables: new restructure.Array(WOFFDirectoryEntry, 'numTables')
});

WOFFDirectory.process = function () {
  var tables$$1 = {};

  for (var _iterator = this.tables, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
    var _ref;

    if (_isArray) {
      if (_i >= _iterator.length) break;
      _ref = _iterator[_i++];
    } else {
      _i = _iterator.next();
      if (_i.done) break;
      _ref = _i.value;
    }

    var table = _ref;
    tables$$1[table.tag] = table;
  }

  this.tables = tables$$1;
};

var WOFFFont =
/*#__PURE__*/
function (_TTFFont) {
  _inheritsLoose(WOFFFont, _TTFFont);

  function WOFFFont() {
    return _TTFFont.apply(this, arguments) || this;
  }

  WOFFFont.probe = function probe(buffer) {
    return buffer.toString('ascii', 0, 4) === 'wOFF';
  };

  var _proto = WOFFFont.prototype;

  _proto._decodeDirectory = function _decodeDirectory() {
    this.directory = WOFFDirectory.decode(this.stream, {
      _startOffset: 0
    });
  };

  _proto._getTableStream = function _getTableStream(tag) {
    var table = this.directory.tables[tag];

    if (table) {
      this.stream.pos = table.offset;

      if (table.compLength < table.length) {
        this.stream.pos += 2; // skip deflate header

        var outBuffer = new Buffer(table.length);
        var buf = tinyInflate(this.stream.readBuffer(table.compLength - 2), outBuffer);
        return new restructure.DecodeStream(buf);
      } else {
        return this.stream;
      }
    }

    return null;
  };

  return WOFFFont;
}(TTFFont);

function BrotliInput(buffer) {
  this.buffer = buffer;
  this.pos = 0;
}

BrotliInput.prototype.read = function (buf, i, count) {
  if (this.pos + count > this.buffer.length) {
    count = this.buffer.length - this.pos;
  }

  for (var p = 0; p < count; p++) {
    buf[i + p] = this.buffer[this.pos + p];
  }

  this.pos += count;
  return count;
};

var BrotliInput_1 = BrotliInput;

function BrotliOutput(buf) {
  this.buffer = buf;
  this.pos = 0;
}

BrotliOutput.prototype.write = function (buf, count) {
  if (this.pos + count > this.buffer.length) throw new Error('Output buffer is not large enough');
  this.buffer.set(buf.subarray(0, count), this.pos);
  this.pos += count;
  return count;
};

var BrotliOutput_1 = BrotliOutput;
var streams$1 = {
  BrotliInput: BrotliInput_1,
  BrotliOutput: BrotliOutput_1
};

/* Copyright 2013 Google Inc. All Rights Reserved.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

   Bit reading helpers
*/
var BROTLI_READ_SIZE = 4096;
var BROTLI_IBUF_SIZE = 2 * BROTLI_READ_SIZE + 32;
var BROTLI_IBUF_MASK = 2 * BROTLI_READ_SIZE - 1;
var kBitMask = new Uint32Array([0, 1, 3, 7, 15, 31, 63, 127, 255, 511, 1023, 2047, 4095, 8191, 16383, 32767, 65535, 131071, 262143, 524287, 1048575, 2097151, 4194303, 8388607, 16777215]);
/* Input byte buffer, consist of a ringbuffer and a "slack" region where */

/* bytes from the start of the ringbuffer are copied. */

function BrotliBitReader(input) {
  this.buf_ = new Uint8Array(BROTLI_IBUF_SIZE);
  this.input_ = input;
  /* input callback */

  this.reset();
}

BrotliBitReader.READ_SIZE = BROTLI_READ_SIZE;
BrotliBitReader.IBUF_MASK = BROTLI_IBUF_MASK;

BrotliBitReader.prototype.reset = function () {
  this.buf_ptr_ = 0;
  /* next input will write here */

  this.val_ = 0;
  /* pre-fetched bits */

  this.pos_ = 0;
  /* byte position in stream */

  this.bit_pos_ = 0;
  /* current bit-reading position in val_ */

  this.bit_end_pos_ = 0;
  /* bit-reading end position from LSB of val_ */

  this.eos_ = 0;
  /* input stream is finished */

  this.readMoreInput();

  for (var i = 0; i < 4; i++) {
    this.val_ |= this.buf_[this.pos_] << 8 * i;
    ++this.pos_;
  }

  return this.bit_end_pos_ > 0;
};
/* Fills up the input ringbuffer by calling the input callback.

   Does nothing if there are at least 32 bytes present after current position.

   Returns 0 if either:
    - the input callback returned an error, or
    - there is no more input and the position is past the end of the stream.

   After encountering the end of the input stream, 32 additional zero bytes are
   copied to the ringbuffer, therefore it is safe to call this function after
   every 32 bytes of input is read.
*/


BrotliBitReader.prototype.readMoreInput = function () {
  if (this.bit_end_pos_ > 256) {
    return;
  } else if (this.eos_) {
    if (this.bit_pos_ > this.bit_end_pos_) throw new Error('Unexpected end of input ' + this.bit_pos_ + ' ' + this.bit_end_pos_);
  } else {
    var dst = this.buf_ptr_;
    var bytes_read = this.input_.read(this.buf_, dst, BROTLI_READ_SIZE);

    if (bytes_read < 0) {
      throw new Error('Unexpected end of input');
    }

    if (bytes_read < BROTLI_READ_SIZE) {
      this.eos_ = 1;
      /* Store 32 bytes of zero after the stream end. */

      for (var p = 0; p < 32; p++) {
        this.buf_[dst + bytes_read + p] = 0;
      }
    }

    if (dst === 0) {
      /* Copy the head of the ringbuffer to the slack region. */
      for (var p = 0; p < 32; p++) {
        this.buf_[(BROTLI_READ_SIZE << 1) + p] = this.buf_[p];
      }

      this.buf_ptr_ = BROTLI_READ_SIZE;
    } else {
      this.buf_ptr_ = 0;
    }

    this.bit_end_pos_ += bytes_read << 3;
  }
};
/* Guarantees that there are at least 24 bits in the buffer. */


BrotliBitReader.prototype.fillBitWindow = function () {
  while (this.bit_pos_ >= 8) {
    this.val_ >>>= 8;
    this.val_ |= this.buf_[this.pos_ & BROTLI_IBUF_MASK] << 24;
    ++this.pos_;
    this.bit_pos_ = this.bit_pos_ - 8 >>> 0;
    this.bit_end_pos_ = this.bit_end_pos_ - 8 >>> 0;
  }
};
/* Reads the specified number of bits from Read Buffer. */


BrotliBitReader.prototype.readBits = function (n_bits) {
  if (32 - this.bit_pos_ < n_bits) {
    this.fillBitWindow();
  }

  var val = this.val_ >>> this.bit_pos_ & kBitMask[n_bits];
  this.bit_pos_ += n_bits;
  return val;
};

var bit_reader = BrotliBitReader;

var byteLength_1 = byteLength$1;
var toByteArray_1 = toByteArray$1;
var fromByteArray_1 = fromByteArray$1;
var lookup$1 = [];
var revLookup$1 = [];
var Arr$1 = typeof Uint8Array !== 'undefined' ? Uint8Array : Array;
var code = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';

for (var i$4 = 0, len = code.length; i$4 < len; ++i$4) {
  lookup$1[i$4] = code[i$4];
  revLookup$1[code.charCodeAt(i$4)] = i$4;
} // Support decoding URL-safe base64 strings, as Node.js does.
// See: https://en.wikipedia.org/wiki/Base64#URL_applications


revLookup$1['-'.charCodeAt(0)] = 62;
revLookup$1['_'.charCodeAt(0)] = 63;

function getLens(b64) {
  var len = b64.length;

  if (len % 4 > 0) {
    throw new Error('Invalid string. Length must be a multiple of 4');
  } // Trim off extra bytes after placeholder bytes are found
  // See: https://github.com/beatgammit/base64-js/issues/42


  var validLen = b64.indexOf('=');
  if (validLen === -1) validLen = len;
  var placeHoldersLen = validLen === len ? 0 : 4 - validLen % 4;
  return [validLen, placeHoldersLen];
} // base64 is 4/3 + up to two characters of the original data


function byteLength$1(b64) {
  var lens = getLens(b64);
  var validLen = lens[0];
  var placeHoldersLen = lens[1];
  return (validLen + placeHoldersLen) * 3 / 4 - placeHoldersLen;
}

function _byteLength(b64, validLen, placeHoldersLen) {
  return (validLen + placeHoldersLen) * 3 / 4 - placeHoldersLen;
}

function toByteArray$1(b64) {
  var tmp;
  var lens = getLens(b64);
  var validLen = lens[0];
  var placeHoldersLen = lens[1];
  var arr = new Arr$1(_byteLength(b64, validLen, placeHoldersLen));
  var curByte = 0; // if there are placeholders, only get up to the last complete 4 chars

  var len = placeHoldersLen > 0 ? validLen - 4 : validLen;

  for (var i = 0; i < len; i += 4) {
    tmp = revLookup$1[b64.charCodeAt(i)] << 18 | revLookup$1[b64.charCodeAt(i + 1)] << 12 | revLookup$1[b64.charCodeAt(i + 2)] << 6 | revLookup$1[b64.charCodeAt(i + 3)];
    arr[curByte++] = tmp >> 16 & 0xFF;
    arr[curByte++] = tmp >> 8 & 0xFF;
    arr[curByte++] = tmp & 0xFF;
  }

  if (placeHoldersLen === 2) {
    tmp = revLookup$1[b64.charCodeAt(i)] << 2 | revLookup$1[b64.charCodeAt(i + 1)] >> 4;
    arr[curByte++] = tmp & 0xFF;
  }

  if (placeHoldersLen === 1) {
    tmp = revLookup$1[b64.charCodeAt(i)] << 10 | revLookup$1[b64.charCodeAt(i + 1)] << 4 | revLookup$1[b64.charCodeAt(i + 2)] >> 2;
    arr[curByte++] = tmp >> 8 & 0xFF;
    arr[curByte++] = tmp & 0xFF;
  }

  return arr;
}

function tripletToBase64$1(num) {
  return lookup$1[num >> 18 & 0x3F] + lookup$1[num >> 12 & 0x3F] + lookup$1[num >> 6 & 0x3F] + lookup$1[num & 0x3F];
}

function encodeChunk$1(uint8, start, end) {
  var tmp;
  var output = [];

  for (var i = start; i < end; i += 3) {
    tmp = (uint8[i] << 16 & 0xFF0000) + (uint8[i + 1] << 8 & 0xFF00) + (uint8[i + 2] & 0xFF);
    output.push(tripletToBase64$1(tmp));
  }

  return output.join('');
}

function fromByteArray$1(uint8) {
  var tmp;
  var len = uint8.length;
  var extraBytes = len % 3; // if we have 1 byte left, pad 2 bytes

  var parts = [];
  var maxChunkLength = 16383; // must be multiple of 3
  // go through the array every three bytes, we'll deal with trailing stuff later

  for (var i = 0, len2 = len - extraBytes; i < len2; i += maxChunkLength) {
    parts.push(encodeChunk$1(uint8, i, i + maxChunkLength > len2 ? len2 : i + maxChunkLength));
  } // pad the end with zeros, but make sure to not forget the extra bytes


  if (extraBytes === 1) {
    tmp = uint8[len - 1];
    parts.push(lookup$1[tmp >> 2] + lookup$1[tmp << 4 & 0x3F] + '==');
  } else if (extraBytes === 2) {
    tmp = (uint8[len - 2] << 8) + uint8[len - 1];
    parts.push(lookup$1[tmp >> 10] + lookup$1[tmp >> 4 & 0x3F] + lookup$1[tmp << 2 & 0x3F] + '=');
  }

  return parts.join('');
}

var base64Js = {
  byteLength: byteLength_1,
  toByteArray: toByteArray_1,
  fromByteArray: fromByteArray_1
};

var dictionary_bin = "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