"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const PDFRef_1 = (0, tslib_1.__importDefault)(require("../objects/PDFRef"));
const CharCodes_1 = (0, tslib_1.__importDefault)(require("../syntax/CharCodes"));
const utils_1 = require("../../utils");
/**
 * Entries should be added using the [[addEntry]] and [[addDeletedEntry]]
 * methods **in order of ascending object number**.
 */
class PDFCrossRefSection {
    constructor(firstEntry) {
        this.subsections = firstEntry ? [[firstEntry]] : [];
        this.chunkIdx = 0;
        this.chunkLength = firstEntry ? 1 : 0;
    }
    addEntry(ref, offset) {
        this.append({ ref, offset, deleted: false });
    }
    addDeletedEntry(ref, nextFreeObjectNumber) {
        this.append({ ref, offset: nextFreeObjectNumber, deleted: true });
    }
    toString() {
        let section = `xref\n`;
        for (let rangeIdx = 0, rangeLen = this.subsections.length; rangeIdx < rangeLen; rangeIdx++) {
            const range = this.subsections[rangeIdx];
            section += `${range[0].ref.objectNumber} ${range.length}\n`;
            for (let entryIdx = 0, entryLen = range.length; entryIdx < entryLen; entryIdx++) {
                const entry = range[entryIdx];
                section += (0, utils_1.padStart)(String(entry.offset), 10, '0');
                section += ' ';
                section += (0, utils_1.padStart)(String(entry.ref.generationNumber), 5, '0');
                section += ' ';
                section += entry.deleted ? 'f' : 'n';
                section += ' \n';
            }
        }
        return section;
    }
    sizeInBytes() {
        let size = 5;
        for (let idx = 0, len = this.subsections.length; idx < len; idx++) {
            const subsection = this.subsections[idx];
            const subsectionLength = subsection.length;
            const [firstEntry] = subsection;
            size += 2;
            size += String(firstEntry.ref.objectNumber).length;
            size += String(subsectionLength).length;
            size += 20 * subsectionLength;
        }
        return size;
    }
    copyBytesInto(buffer, offset) {
        const initialOffset = offset;
        buffer[offset++] = CharCodes_1.default.x;
        buffer[offset++] = CharCodes_1.default.r;
        buffer[offset++] = CharCodes_1.default.e;
        buffer[offset++] = CharCodes_1.default.f;
        buffer[offset++] = CharCodes_1.default.Newline;
        offset += this.copySubsectionsIntoBuffer(this.subsections, buffer, offset);
        return offset - initialOffset;
    }
    copySubsectionsIntoBuffer(subsections, buffer, offset) {
        const initialOffset = offset;
        const length = subsections.length;
        for (let idx = 0; idx < length; idx++) {
            const subsection = this.subsections[idx];
            const firstObjectNumber = String(subsection[0].ref.objectNumber);
            offset += (0, utils_1.copyStringIntoBuffer)(firstObjectNumber, buffer, offset);
            buffer[offset++] = CharCodes_1.default.Space;
            const rangeLength = String(subsection.length);
            offset += (0, utils_1.copyStringIntoBuffer)(rangeLength, buffer, offset);
            buffer[offset++] = CharCodes_1.default.Newline;
            offset += this.copyEntriesIntoBuffer(subsection, buffer, offset);
        }
        return offset - initialOffset;
    }
    copyEntriesIntoBuffer(entries, buffer, offset) {
        const length = entries.length;
        for (let idx = 0; idx < length; idx++) {
            const entry = entries[idx];
            const entryOffset = (0, utils_1.padStart)(String(entry.offset), 10, '0');
            offset += (0, utils_1.copyStringIntoBuffer)(entryOffset, buffer, offset);
            buffer[offset++] = CharCodes_1.default.Space;
            const entryGen = (0, utils_1.padStart)(String(entry.ref.generationNumber), 5, '0');
            offset += (0, utils_1.copyStringIntoBuffer)(entryGen, buffer, offset);
            buffer[offset++] = CharCodes_1.default.Space;
            buffer[offset++] = entry.deleted ? CharCodes_1.default.f : CharCodes_1.default.n;
            buffer[offset++] = CharCodes_1.default.Space;
            buffer[offset++] = CharCodes_1.default.Newline;
        }
        return 20 * length;
    }
    append(currEntry) {
        if (this.chunkLength === 0) {
            this.subsections.push([currEntry]);
            this.chunkIdx = 0;
            this.chunkLength = 1;
            return;
        }
        const chunk = this.subsections[this.chunkIdx];
        const prevEntry = chunk[this.chunkLength - 1];
        if (currEntry.ref.objectNumber - prevEntry.ref.objectNumber > 1) {
            this.subsections.push([currEntry]);
            this.chunkIdx += 1;
            this.chunkLength = 1;
        }
        else {
            chunk.push(currEntry);
            this.chunkLength += 1;
        }
    }
}
PDFCrossRefSection.create = () => new PDFCrossRefSection({
    ref: PDFRef_1.default.of(0, 65535),
    offset: 0,
    deleted: true,
});
PDFCrossRefSection.createEmpty = () => new PDFCrossRefSection();
exports.default = PDFCrossRefSection;
//# sourceMappingURL=PDFCrossRefSection.js.map