"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const PDFObject_1 = (0, tslib_1.__importDefault)(require("./PDFObject"));
const CharCodes_1 = (0, tslib_1.__importDefault)(require("../syntax/CharCodes"));
const utils_1 = require("../../utils");
const errors_1 = require("../errors");
class PDFString extends PDFObject_1.default {
    constructor(value) {
        super();
        this.value = value;
    }
    asBytes() {
        const bytes = [];
        let octal = '';
        let escaped = false;
        const pushByte = (byte) => {
            if (byte !== undefined)
                bytes.push(byte);
            escaped = false;
        };
        for (let idx = 0, len = this.value.length; idx < len; idx++) {
            const char = this.value[idx];
            const byte = (0, utils_1.toCharCode)(char);
            const nextChar = this.value[idx + 1];
            if (!escaped) {
                if (byte === CharCodes_1.default.BackSlash)
                    escaped = true;
                else
                    pushByte(byte);
            }
            else {
                if (byte === CharCodes_1.default.Newline)
                    pushByte();
                else if (byte === CharCodes_1.default.CarriageReturn)
                    pushByte();
                else if (byte === CharCodes_1.default.n)
                    pushByte(CharCodes_1.default.Newline);
                else if (byte === CharCodes_1.default.r)
                    pushByte(CharCodes_1.default.CarriageReturn);
                else if (byte === CharCodes_1.default.t)
                    pushByte(CharCodes_1.default.Tab);
                else if (byte === CharCodes_1.default.b)
                    pushByte(CharCodes_1.default.Backspace);
                else if (byte === CharCodes_1.default.f)
                    pushByte(CharCodes_1.default.FormFeed);
                else if (byte === CharCodes_1.default.LeftParen)
                    pushByte(CharCodes_1.default.LeftParen);
                else if (byte === CharCodes_1.default.RightParen)
                    pushByte(CharCodes_1.default.RightParen);
                else if (byte === CharCodes_1.default.Backspace)
                    pushByte(CharCodes_1.default.BackSlash);
                else if (byte >= CharCodes_1.default.Zero && byte <= CharCodes_1.default.Seven) {
                    octal += char;
                    if (octal.length === 3 || !(nextChar >= '0' && nextChar <= '7')) {
                        pushByte(parseInt(octal, 8));
                        octal = '';
                    }
                }
                else {
                    pushByte(byte);
                }
            }
        }
        return new Uint8Array(bytes);
    }
    decodeText() {
        const bytes = this.asBytes();
        if ((0, utils_1.hasUtf16BOM)(bytes))
            return (0, utils_1.utf16Decode)(bytes);
        return (0, utils_1.pdfDocEncodingDecode)(bytes);
    }
    decodeDate() {
        const text = this.decodeText();
        const date = (0, utils_1.parseDate)(text);
        if (!date)
            throw new errors_1.InvalidPDFDateStringError(text);
        return date;
    }
    asString() {
        return this.value;
    }
    clone() {
        return PDFString.of(this.value);
    }
    toString() {
        return `(${this.value})`;
    }
    sizeInBytes() {
        return this.value.length + 2;
    }
    copyBytesInto(buffer, offset) {
        buffer[offset++] = CharCodes_1.default.LeftParen;
        offset += (0, utils_1.copyStringIntoBuffer)(this.value, buffer, offset);
        buffer[offset++] = CharCodes_1.default.RightParen;
        return this.value.length + 2;
    }
}
// The PDF spec allows newlines and parens to appear directly within a literal
// string. These character _may_ be escaped. But they do not _have_ to be. So
// for simplicity, we will not bother escaping them.
PDFString.of = (value) => new PDFString(value);
PDFString.fromDate = (date) => {
    const year = (0, utils_1.padStart)(String(date.getUTCFullYear()), 4, '0');
    const month = (0, utils_1.padStart)(String(date.getUTCMonth() + 1), 2, '0');
    const day = (0, utils_1.padStart)(String(date.getUTCDate()), 2, '0');
    const hours = (0, utils_1.padStart)(String(date.getUTCHours()), 2, '0');
    const mins = (0, utils_1.padStart)(String(date.getUTCMinutes()), 2, '0');
    const secs = (0, utils_1.padStart)(String(date.getUTCSeconds()), 2, '0');
    return new PDFString(`D:${year}${month}${day}${hours}${mins}${secs}Z`);
};
exports.default = PDFString;
//# sourceMappingURL=PDFString.js.map