"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const errors_1 = require("../errors");
const CharCodes_1 = (0, tslib_1.__importDefault)(require("../syntax/CharCodes"));
const Numeric_1 = require("../syntax/Numeric");
const Whitespace_1 = require("../syntax/Whitespace");
const utils_1 = require("../../utils");
const { Newline, CarriageReturn } = CharCodes_1.default;
// TODO: Throw error if eof is reached before finishing object parse...
class BaseParser {
    constructor(bytes, capNumbers = false) {
        this.bytes = bytes;
        this.capNumbers = capNumbers;
    }
    parseRawInt() {
        let value = '';
        while (!this.bytes.done()) {
            const byte = this.bytes.peek();
            if (!Numeric_1.IsDigit[byte])
                break;
            value += (0, utils_1.charFromCode)(this.bytes.next());
        }
        const numberValue = Number(value);
        if (!value || !isFinite(numberValue)) {
            throw new errors_1.NumberParsingError(this.bytes.position(), value);
        }
        return numberValue;
    }
    // TODO: Maybe handle exponential format?
    // TODO: Compare performance of string concatenation to charFromCode(...bytes)
    parseRawNumber() {
        let value = '';
        // Parse integer-part, the leading (+ | - | . | 0-9)
        while (!this.bytes.done()) {
            const byte = this.bytes.peek();
            if (!Numeric_1.IsNumeric[byte])
                break;
            value += (0, utils_1.charFromCode)(this.bytes.next());
            if (byte === CharCodes_1.default.Period)
                break;
        }
        // Parse decimal-part, the trailing (0-9)
        while (!this.bytes.done()) {
            const byte = this.bytes.peek();
            if (!Numeric_1.IsDigit[byte])
                break;
            value += (0, utils_1.charFromCode)(this.bytes.next());
        }
        const numberValue = Number(value);
        if (!value || !isFinite(numberValue)) {
            throw new errors_1.NumberParsingError(this.bytes.position(), value);
        }
        if (numberValue > Number.MAX_SAFE_INTEGER) {
            if (this.capNumbers) {
                const msg = `Parsed number that is too large for some PDF readers: ${value}, using Number.MAX_SAFE_INTEGER instead.`;
                console.warn(msg);
                return Number.MAX_SAFE_INTEGER;
            }
            else {
                const msg = `Parsed number that is too large for some PDF readers: ${value}, not capping.`;
                console.warn(msg);
            }
        }
        return numberValue;
    }
    skipWhitespace() {
        while (!this.bytes.done() && Whitespace_1.IsWhitespace[this.bytes.peek()]) {
            this.bytes.next();
        }
    }
    skipLine() {
        while (!this.bytes.done()) {
            const byte = this.bytes.peek();
            if (byte === Newline || byte === CarriageReturn)
                return;
            this.bytes.next();
        }
    }
    skipComment() {
        if (this.bytes.peek() !== CharCodes_1.default.Percent)
            return false;
        while (!this.bytes.done()) {
            const byte = this.bytes.peek();
            if (byte === Newline || byte === CarriageReturn)
                return true;
            this.bytes.next();
        }
        return true;
    }
    skipWhitespaceAndComments() {
        this.skipWhitespace();
        while (this.skipComment())
            this.skipWhitespace();
    }
    matchKeyword(keyword) {
        const initialOffset = this.bytes.offset();
        for (let idx = 0, len = keyword.length; idx < len; idx++) {
            if (this.bytes.done() || this.bytes.next() !== keyword[idx]) {
                this.bytes.moveTo(initialOffset);
                return false;
            }
        }
        return true;
    }
}
exports.default = BaseParser;
//# sourceMappingURL=BaseParser.js.map