"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const errors_1 = require("../errors");
const PDFArray_1 = (0, tslib_1.__importDefault)(require("../objects/PDFArray"));
const PDFName_1 = (0, tslib_1.__importDefault)(require("../objects/PDFName"));
const PDFNumber_1 = (0, tslib_1.__importDefault)(require("../objects/PDFNumber"));
const PDFRef_1 = (0, tslib_1.__importDefault)(require("../objects/PDFRef"));
const ByteStream_1 = (0, tslib_1.__importDefault)(require("./ByteStream"));
class PDFXRefStreamParser {
    constructor(rawStream) {
        this.alreadyParsed = false;
        this.dict = rawStream.dict;
        this.bytes = ByteStream_1.default.fromPDFRawStream(rawStream);
        this.context = this.dict.context;
        const Size = this.dict.lookup(PDFName_1.default.of('Size'), PDFNumber_1.default);
        const Index = this.dict.lookup(PDFName_1.default.of('Index'));
        if (Index instanceof PDFArray_1.default) {
            this.subsections = [];
            for (let idx = 0, len = Index.size(); idx < len; idx += 2) {
                const firstObjectNumber = Index.lookup(idx + 0, PDFNumber_1.default).asNumber();
                const length = Index.lookup(idx + 1, PDFNumber_1.default).asNumber();
                this.subsections.push({ firstObjectNumber, length });
            }
        }
        else {
            this.subsections = [{ firstObjectNumber: 0, length: Size.asNumber() }];
        }
        const W = this.dict.lookup(PDFName_1.default.of('W'), PDFArray_1.default);
        this.byteWidths = [-1, -1, -1];
        for (let idx = 0, len = W.size(); idx < len; idx++) {
            this.byteWidths[idx] = W.lookup(idx, PDFNumber_1.default).asNumber();
        }
    }
    parseIntoContext() {
        if (this.alreadyParsed) {
            throw new errors_1.ReparseError('PDFXRefStreamParser', 'parseIntoContext');
        }
        this.alreadyParsed = true;
        this.context.trailerInfo = {
            Root: this.dict.get(PDFName_1.default.of('Root')),
            Encrypt: this.dict.get(PDFName_1.default.of('Encrypt')),
            Info: this.dict.get(PDFName_1.default.of('Info')),
            ID: this.dict.get(PDFName_1.default.of('ID')),
        };
        const entries = this.parseEntries();
        // for (let idx = 0, len = entries.length; idx < len; idx++) {
        // const entry = entries[idx];
        // if (entry.deleted) this.context.delete(entry.ref);
        // }
        return entries;
    }
    parseEntries() {
        const entries = [];
        const [typeFieldWidth, offsetFieldWidth, genFieldWidth] = this.byteWidths;
        for (let subsectionIdx = 0, subsectionLen = this.subsections.length; subsectionIdx < subsectionLen; subsectionIdx++) {
            const { firstObjectNumber, length } = this.subsections[subsectionIdx];
            for (let objIdx = 0; objIdx < length; objIdx++) {
                let type = 0;
                for (let idx = 0, len = typeFieldWidth; idx < len; idx++) {
                    type = (type << 8) | this.bytes.next();
                }
                let offset = 0;
                for (let idx = 0, len = offsetFieldWidth; idx < len; idx++) {
                    offset = (offset << 8) | this.bytes.next();
                }
                let generationNumber = 0;
                for (let idx = 0, len = genFieldWidth; idx < len; idx++) {
                    generationNumber = (generationNumber << 8) | this.bytes.next();
                }
                // When the `type` field is absent, it defaults to 1
                if (typeFieldWidth === 0)
                    type = 1;
                const objectNumber = firstObjectNumber + objIdx;
                const entry = {
                    ref: PDFRef_1.default.of(objectNumber, generationNumber),
                    offset,
                    deleted: type === 0,
                    inObjectStream: type === 2,
                };
                entries.push(entry);
            }
        }
        return entries;
    }
}
PDFXRefStreamParser.forStream = (rawStream) => new PDFXRefStreamParser(rawStream);
exports.default = PDFXRefStreamParser;
//# sourceMappingURL=PDFXRefStreamParser.js.map