"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const PDFName_1 = (0, tslib_1.__importDefault)(require("../objects/PDFName"));
const PDFNumber_1 = (0, tslib_1.__importDefault)(require("../objects/PDFNumber"));
const PDFFlateStream_1 = (0, tslib_1.__importDefault)(require("./PDFFlateStream"));
const CharCodes_1 = (0, tslib_1.__importDefault)(require("../syntax/CharCodes"));
const utils_1 = require("../../utils");
class PDFObjectStream extends PDFFlateStream_1.default {
    constructor(context, objects, encode = true) {
        super(context.obj({}), encode);
        this.objects = objects;
        this.offsets = this.computeObjectOffsets();
        this.offsetsString = this.computeOffsetsString();
        this.dict.set(PDFName_1.default.of('Type'), PDFName_1.default.of('ObjStm'));
        this.dict.set(PDFName_1.default.of('N'), PDFNumber_1.default.of(this.objects.length));
        this.dict.set(PDFName_1.default.of('First'), PDFNumber_1.default.of(this.offsetsString.length));
    }
    getObjectsCount() {
        return this.objects.length;
    }
    clone(context) {
        return PDFObjectStream.withContextAndObjects(context || this.dict.context, this.objects.slice(), this.encode);
    }
    getContentsString() {
        let value = this.offsetsString;
        for (let idx = 0, len = this.objects.length; idx < len; idx++) {
            const [, object] = this.objects[idx];
            value += `${object}\n`;
        }
        return value;
    }
    getUnencodedContents() {
        const buffer = new Uint8Array(this.getUnencodedContentsSize());
        let offset = (0, utils_1.copyStringIntoBuffer)(this.offsetsString, buffer, 0);
        for (let idx = 0, len = this.objects.length; idx < len; idx++) {
            const [, object] = this.objects[idx];
            offset += object.copyBytesInto(buffer, offset);
            buffer[offset++] = CharCodes_1.default.Newline;
        }
        return buffer;
    }
    getUnencodedContentsSize() {
        return (this.offsetsString.length +
            (0, utils_1.last)(this.offsets)[1] +
            (0, utils_1.last)(this.objects)[1].sizeInBytes() +
            1);
    }
    computeOffsetsString() {
        let offsetsString = '';
        for (let idx = 0, len = this.offsets.length; idx < len; idx++) {
            const [objectNumber, offset] = this.offsets[idx];
            offsetsString += `${objectNumber} ${offset} `;
        }
        return offsetsString;
    }
    computeObjectOffsets() {
        let offset = 0;
        const offsets = new Array(this.objects.length);
        for (let idx = 0, len = this.objects.length; idx < len; idx++) {
            const [ref, object] = this.objects[idx];
            offsets[idx] = [ref.objectNumber, offset];
            offset += object.sizeInBytes() + 1; // '\n'
        }
        return offsets;
    }
}
PDFObjectStream.withContextAndObjects = (context, objects, encode = true) => new PDFObjectStream(context, objects, encode);
exports.default = PDFObjectStream;
//# sourceMappingURL=PDFObjectStream.js.map