"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const maths_1 = require("../maths");
const GraphElement_1 = (0, tslib_1.__importDefault)(require("./GraphElement"));
const Point_1 = (0, tslib_1.__importDefault)(require("./Point"));
const Segment_1 = (0, tslib_1.__importDefault)(require("./Segment"));
class Ellipse extends GraphElement_1.default {
    constructor(A = new Point_1.default(), B = new Point_1.default(), C = new Point_1.default()) {
        super();
        this.A = A;
        this.B = B;
        this.C = C;
    }
    center() {
        const center = this.axis().middle();
        return center;
    }
    axis() {
        const axis = new Segment_1.default(this.A, this.B);
        return axis;
    }
    a() {
        const axis = this.axis();
        return Math.max(axis.length() / 2, axis.distance(this.C));
    }
    b() {
        const axis = this.axis();
        return Math.min(axis.length() / 2, axis.distance(this.C));
    }
    rotation() {
        const axis = this.axis();
        return axis.length() / 2 > axis.distance(this.C)
            ? (0, maths_1.orientation)(axis.dirVect())
            : (0, maths_1.orientation)((0, maths_1.orthogonal)(axis.dirVect()));
    }
    getSize() {
        return { width: 2 * this.a(), height: 2 * this.b() };
    }
    isEqual(element) {
        if (!(element instanceof Ellipse))
            return false;
        const a = this.a();
        const b = this.b();
        const rotation = this.rotation();
        const eltA = element.a();
        const eltB = element.b();
        const eltRotation = element.rotation();
        // If the main axis is the same on both ellipse
        if (eltA < eltB === a < b) {
            // The rotation is equivalent module PI as the element is symetrical
            return ((0, maths_1.isEqual)(eltA, a) &&
                (0, maths_1.isEqual)(eltB, b) &&
                (0, maths_1.isEqual)(rotation + (Math.PI % Math.PI), eltRotation + (Math.PI % Math.PI)));
        }
        // If the small axis is different
        else {
            // We add a rotation of PI / 2 to emulate the fact that the main axis are actually orthogonal
            return ((0, maths_1.isEqual)(eltA, b) &&
                (0, maths_1.isEqual)(eltB, a) &&
                (0, maths_1.isEqual)(rotation + (Math.PI % Math.PI), eltRotation + (((3 * Math.PI) / 2) % Math.PI)));
        }
    }
    includes(P) {
        const { x, y } = P.toCoords();
        const { x: cx, y: cy } = this.center().toCoords();
        const teta = this.rotation();
        return (0, maths_1.isEqual)(Math.pow(((x - cx) * Math.cos(teta) + (y - cy) * Math.sin(teta)) / this.a(), 2) +
            Math.pow(((x - cx) * Math.sin(teta) - (y - cy) * Math.cos(teta)) / this.b(), 2), 1);
    }
    orthoProjection(P) {
        // We will consider that the parametric projection is a correct approximation of the distance for the current case, even if it is not orthogonal
        const C = this.center();
        const axis = this.axis();
        const CP = (0, maths_1.vector)(C, P);
        const teta = (0, maths_1.angle)(axis.dirVect(), (0, maths_1.vector)(C, P));
        const ray = this.polarRay(teta);
        if ((0, maths_1.distance)(P, this.center()) < ray)
            return P;
        const vect = (0, maths_1.times)((0, maths_1.unitVector)(CP), ray);
        return new Point_1.default(this.center().plus(vect).toCoords());
    }
    polarRay(teta) {
        const a = this.a();
        const b = this.b();
        const excentricity = Math.sqrt(Math.abs(a * a - b * b)) / Math.max(a, b);
        return (Math.min(a, b) / Math.sqrt(1 - Math.pow(excentricity * Math.cos(teta), 2)));
    }
}
exports.default = Ellipse;
//# sourceMappingURL=Ellipse.js.map