"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const maths_1 = require("../maths");
const GraphElement_1 = (0, tslib_1.__importDefault)(require("./GraphElement"));
const Line_1 = (0, tslib_1.__importDefault)(require("./Line"));
const Point_1 = (0, tslib_1.__importDefault)(require("./Point"));
class Segment extends GraphElement_1.default {
    constructor(A = new Point_1.default(), B = new Point_1.default()) {
        super();
        this.A = A;
        this.B = B;
    }
    origin() {
        return this.A;
    }
    destination() {
        return this.B;
    }
    dirVect() {
        return (0, maths_1.vector)(this.origin(), this.destination());
    }
    length() {
        return (0, maths_1.distance)(this.destination(), this.origin());
    }
    isEqual(element) {
        if (!(element instanceof Segment))
            return false;
        const o = this.origin();
        const dest = this.destination();
        const oE = element.origin();
        const destE = element.destination();
        return (element instanceof Segment &&
            ((o.isEqual(oE) && dest.isEqual(destE)) ||
                (o.isEqual(destE) && dest.isEqual(oE))));
    }
    /** Returns an equivalent line object */
    getLine() {
        const line = new Line_1.default(this.origin(), this.destination());
        return line;
    }
    includes(P) {
        const vect = this.dirVect();
        const otherVect = (0, maths_1.vector)(this.origin(), P);
        // The vectors are not even colinear
        if (!(0, maths_1.isColinear)(vect, otherVect))
            return false;
        // The point is behind the origin
        else if ((0, maths_1.scalar)(vect, otherVect) < 0)
            return false;
        // The point is after the destination
        else if ((0, maths_1.norm)(vect) < (0, maths_1.norm)(otherVect))
            return false;
        else
            return true;
    }
    middle() {
        const mid = new Point_1.default((0, maths_1.plus)(this.origin().toCoords(), (0, maths_1.times)(this.dirVect(), 0.5)));
        return mid;
    }
    orthoProjection(P) {
        const H = this.getLine().orthoProjection(P);
        const vect = this.dirVect();
        const origin = this.origin().toCoords();
        const destination = this.destination().toCoords();
        const otherVect = (0, maths_1.vector)(this.origin(), H);
        // The point is before the origin
        if ((0, maths_1.scalar)(vect, otherVect) < 0)
            return new Point_1.default(origin);
        // The point is after the destination
        else if ((0, maths_1.norm)(vect) < (0, maths_1.norm)(otherVect))
            return new Point_1.default(destination);
        // The point is within the segment
        else
            return H;
    }
}
exports.default = Segment;
Segment.type = 'Segment';
//# sourceMappingURL=Segment.js.map