import { asNumber, asPDFName, asPDFNumber } from './objects.js';
import { degreesToRadians } from './rotations.js';
import { PDFOperator, PDFOperatorNames as Ops, } from '../core/index.js';
/* ==================== Clipping Path Operators ==================== */
export const clip = () => PDFOperator.of(Ops.ClipNonZero);
export const clipEvenOdd = () => PDFOperator.of(Ops.ClipEvenOdd);
/* ==================== Graphics State Operators ==================== */
const { cos, sin, tan } = Math;
export const concatTransformationMatrix = (a, b, c, d, e, f) => PDFOperator.of(Ops.ConcatTransformationMatrix, [
    asPDFNumber(a),
    asPDFNumber(b),
    asPDFNumber(c),
    asPDFNumber(d),
    asPDFNumber(e),
    asPDFNumber(f),
]);
export const translate = (xPos, yPos) => concatTransformationMatrix(1, 0, 0, 1, xPos, yPos);
export const scale = (xPos, yPos) => concatTransformationMatrix(xPos, 0, 0, yPos, 0, 0);
export const rotateRadians = (angle) => concatTransformationMatrix(cos(asNumber(angle)), sin(asNumber(angle)), -sin(asNumber(angle)), cos(asNumber(angle)), 0, 0);
export const rotateDegrees = (angle) => rotateRadians(degreesToRadians(asNumber(angle)));
export const skewRadians = (xSkewAngle, ySkewAngle) => concatTransformationMatrix(1, tan(asNumber(xSkewAngle)), tan(asNumber(ySkewAngle)), 1, 0, 0);
export const skewDegrees = (xSkewAngle, ySkewAngle) => skewRadians(degreesToRadians(asNumber(xSkewAngle)), degreesToRadians(asNumber(ySkewAngle)));
export const setDashPattern = (dashArray, dashPhase) => PDFOperator.of(Ops.SetLineDashPattern, [
    `[${dashArray.map(asPDFNumber).join(' ')}]`,
    asPDFNumber(dashPhase),
]);
export const restoreDashPattern = () => setDashPattern([], 0);
export var LineCapStyle;
(function (LineCapStyle) {
    LineCapStyle[LineCapStyle["Butt"] = 0] = "Butt";
    LineCapStyle[LineCapStyle["Round"] = 1] = "Round";
    LineCapStyle[LineCapStyle["Projecting"] = 2] = "Projecting";
})(LineCapStyle || (LineCapStyle = {}));
export const setLineCap = (style) => PDFOperator.of(Ops.SetLineCapStyle, [asPDFNumber(style)]);
export var LineJoinStyle;
(function (LineJoinStyle) {
    LineJoinStyle[LineJoinStyle["Miter"] = 0] = "Miter";
    LineJoinStyle[LineJoinStyle["Round"] = 1] = "Round";
    LineJoinStyle[LineJoinStyle["Bevel"] = 2] = "Bevel";
})(LineJoinStyle || (LineJoinStyle = {}));
export const setLineJoin = (style) => PDFOperator.of(Ops.SetLineJoinStyle, [asPDFNumber(style)]);
export const setGraphicsState = (state) => PDFOperator.of(Ops.SetGraphicsStateParams, [asPDFName(state)]);
export const pushGraphicsState = () => PDFOperator.of(Ops.PushGraphicsState);
export const popGraphicsState = () => PDFOperator.of(Ops.PopGraphicsState);
export const setLineWidth = (width) => PDFOperator.of(Ops.SetLineWidth, [asPDFNumber(width)]);
/* ==================== Path Construction Operators ==================== */
export const appendBezierCurve = (x1, y1, x2, y2, x3, y3) => PDFOperator.of(Ops.AppendBezierCurve, [
    asPDFNumber(x1),
    asPDFNumber(y1),
    asPDFNumber(x2),
    asPDFNumber(y2),
    asPDFNumber(x3),
    asPDFNumber(y3),
]);
export const appendQuadraticCurve = (x1, y1, x2, y2) => PDFOperator.of(Ops.CurveToReplicateInitialPoint, [
    asPDFNumber(x1),
    asPDFNumber(y1),
    asPDFNumber(x2),
    asPDFNumber(y2),
]);
export const closePath = () => PDFOperator.of(Ops.ClosePath);
export const moveTo = (xPos, yPos) => PDFOperator.of(Ops.MoveTo, [asPDFNumber(xPos), asPDFNumber(yPos)]);
export const lineTo = (xPos, yPos) => PDFOperator.of(Ops.LineTo, [asPDFNumber(xPos), asPDFNumber(yPos)]);
/**
 * @param xPos x coordinate for the lower left corner of the rectangle
 * @param yPos y coordinate for the lower left corner of the rectangle
 * @param width width of the rectangle
 * @param height height of the rectangle
 */
export const rectangle = (xPos, yPos, width, height) => PDFOperator.of(Ops.AppendRectangle, [
    asPDFNumber(xPos),
    asPDFNumber(yPos),
    asPDFNumber(width),
    asPDFNumber(height),
]);
/**
 * @param xPos x coordinate for the lower left corner of the square
 * @param yPos y coordinate for the lower left corner of the square
 * @param size width and height of the square
 */
export const square = (xPos, yPos, size) => rectangle(xPos, yPos, size, size);
/* ==================== Path Painting Operators ==================== */
export const stroke = () => PDFOperator.of(Ops.StrokePath);
export var FillRule;
(function (FillRule) {
    FillRule["NonZero"] = "f";
    FillRule["EvenOdd"] = "f*";
})(FillRule || (FillRule = {}));
export const fill = () => PDFOperator.of(Ops.FillNonZero);
export const fillEvenOdd = () => PDFOperator.of(Ops.FillEvenOdd);
export const fillAndStroke = () => PDFOperator.of(Ops.FillNonZeroAndStroke);
export const endPath = () => PDFOperator.of(Ops.EndPath);
/* ==================== Text Positioning Operators ==================== */
export const nextLine = () => PDFOperator.of(Ops.NextLine);
export const moveText = (x, y) => PDFOperator.of(Ops.MoveText, [asPDFNumber(x), asPDFNumber(y)]);
/* ==================== Text Showing Operators ==================== */
export const showText = (text) => PDFOperator.of(Ops.ShowText, [text]);
/* ==================== Text State Operators ==================== */
export const beginText = () => PDFOperator.of(Ops.BeginText);
export const endText = () => PDFOperator.of(Ops.EndText);
export const setFontAndSize = (name, size) => PDFOperator.of(Ops.SetFontAndSize, [asPDFName(name), asPDFNumber(size)]);
export const setCharacterSpacing = (spacing) => PDFOperator.of(Ops.SetCharacterSpacing, [asPDFNumber(spacing)]);
export const setWordSpacing = (spacing) => PDFOperator.of(Ops.SetWordSpacing, [asPDFNumber(spacing)]);
/** @param squeeze horizontal character spacing */
export const setCharacterSqueeze = (squeeze) => PDFOperator.of(Ops.SetTextHorizontalScaling, [asPDFNumber(squeeze)]);
export const setLineHeight = (lineHeight) => PDFOperator.of(Ops.SetTextLineHeight, [asPDFNumber(lineHeight)]);
export const setTextRise = (rise) => PDFOperator.of(Ops.SetTextRise, [asPDFNumber(rise)]);
export var TextRenderingMode;
(function (TextRenderingMode) {
    TextRenderingMode[TextRenderingMode["Fill"] = 0] = "Fill";
    TextRenderingMode[TextRenderingMode["Outline"] = 1] = "Outline";
    TextRenderingMode[TextRenderingMode["FillAndOutline"] = 2] = "FillAndOutline";
    TextRenderingMode[TextRenderingMode["Invisible"] = 3] = "Invisible";
    TextRenderingMode[TextRenderingMode["FillAndClip"] = 4] = "FillAndClip";
    TextRenderingMode[TextRenderingMode["OutlineAndClip"] = 5] = "OutlineAndClip";
    TextRenderingMode[TextRenderingMode["FillAndOutlineAndClip"] = 6] = "FillAndOutlineAndClip";
    TextRenderingMode[TextRenderingMode["Clip"] = 7] = "Clip";
})(TextRenderingMode || (TextRenderingMode = {}));
export const setTextRenderingMode = (mode) => PDFOperator.of(Ops.SetTextRenderingMode, [asPDFNumber(mode)]);
export const setTextMatrix = (a, b, c, d, e, f) => PDFOperator.of(Ops.SetTextMatrix, [
    asPDFNumber(a),
    asPDFNumber(b),
    asPDFNumber(c),
    asPDFNumber(d),
    asPDFNumber(e),
    asPDFNumber(f),
]);
export const rotateAndSkewTextRadiansAndTranslate = (rotationAngle, xSkewAngle, ySkewAngle, x, y) => setTextMatrix(cos(asNumber(rotationAngle)), sin(asNumber(rotationAngle)) + tan(asNumber(xSkewAngle)), -sin(asNumber(rotationAngle)) + tan(asNumber(ySkewAngle)), cos(asNumber(rotationAngle)), x, y);
export const rotateAndSkewTextDegreesAndTranslate = (rotationAngle, xSkewAngle, ySkewAngle, x, y) => rotateAndSkewTextRadiansAndTranslate(degreesToRadians(asNumber(rotationAngle)), degreesToRadians(asNumber(xSkewAngle)), degreesToRadians(asNumber(ySkewAngle)), x, y);
/* ==================== XObject Operator ==================== */
export const drawObject = (name) => PDFOperator.of(Ops.DrawObject, [asPDFName(name)]);
/* ==================== Color Operators ==================== */
export const setFillingGrayscaleColor = (gray) => PDFOperator.of(Ops.NonStrokingColorGray, [asPDFNumber(gray)]);
export const setStrokingGrayscaleColor = (gray) => PDFOperator.of(Ops.StrokingColorGray, [asPDFNumber(gray)]);
export const setFillingRgbColor = (red, green, blue) => PDFOperator.of(Ops.NonStrokingColorRgb, [
    asPDFNumber(red),
    asPDFNumber(green),
    asPDFNumber(blue),
]);
export const setStrokingRgbColor = (red, green, blue) => PDFOperator.of(Ops.StrokingColorRgb, [
    asPDFNumber(red),
    asPDFNumber(green),
    asPDFNumber(blue),
]);
export const setFillingCmykColor = (cyan, magenta, yellow, key) => PDFOperator.of(Ops.NonStrokingColorCmyk, [
    asPDFNumber(cyan),
    asPDFNumber(magenta),
    asPDFNumber(yellow),
    asPDFNumber(key),
]);
export const setStrokingCmykColor = (cyan, magenta, yellow, key) => PDFOperator.of(Ops.StrokingColorCmyk, [
    asPDFNumber(cyan),
    asPDFNumber(magenta),
    asPDFNumber(yellow),
    asPDFNumber(key),
]);
/* ==================== Marked Content Operators ==================== */
export const beginMarkedContent = (tag) => PDFOperator.of(Ops.BeginMarkedContent, [asPDFName(tag)]);
export const endMarkedContent = () => PDFOperator.of(Ops.EndMarkedContent);
//# sourceMappingURL=operators.js.map