import PDFAcroTerminal from './PDFAcroTerminal.js';
import PDFHexString from '../objects/PDFHexString.js';
import PDFString from '../objects/PDFString.js';
import PDFArray from '../objects/PDFArray.js';
import PDFName from '../objects/PDFName.js';
import { AcroChoiceFlags } from './flags.js';
import { InvalidAcroFieldValueError, MultiSelectValueError } from '../errors.js';
class PDFAcroChoice extends PDFAcroTerminal {
    setValues(values) {
        if (this.hasFlag(AcroChoiceFlags.Combo) &&
            !this.hasFlag(AcroChoiceFlags.Edit) &&
            !this.valuesAreValid(values)) {
            throw new InvalidAcroFieldValueError();
        }
        if (values.length === 0) {
            this.dict.delete(PDFName.of('V'));
        }
        if (values.length === 1) {
            this.dict.set(PDFName.of('V'), values[0]);
        }
        if (values.length > 1) {
            if (!this.hasFlag(AcroChoiceFlags.MultiSelect)) {
                throw new MultiSelectValueError();
            }
            this.dict.set(PDFName.of('V'), this.dict.context.obj(values));
        }
        this.updateSelectedIndices(values);
    }
    valuesAreValid(values) {
        const options = this.getOptions();
        for (let idx = 0, len = values.length; idx < len; idx++) {
            const val = values[idx].decodeText();
            if (!options.find((o) => val === (o.display || o.value).decodeText())) {
                return false;
            }
        }
        return true;
    }
    updateSelectedIndices(values) {
        if (values.length > 1) {
            const indices = new Array(values.length);
            const options = this.getOptions();
            for (let idx = 0, len = values.length; idx < len; idx++) {
                const val = values[idx].decodeText();
                indices[idx] = options.findIndex((o) => val === (o.display || o.value).decodeText());
            }
            this.dict.set(PDFName.of('I'), this.dict.context.obj(indices.sort()));
        }
        else {
            this.dict.delete(PDFName.of('I'));
        }
    }
    getValues() {
        const v = this.V();
        if (v instanceof PDFString || v instanceof PDFHexString)
            return [v];
        if (v instanceof PDFArray) {
            const values = [];
            for (let idx = 0, len = v.size(); idx < len; idx++) {
                const value = v.lookup(idx);
                if (value instanceof PDFString || value instanceof PDFHexString) {
                    values.push(value);
                }
            }
            return values;
        }
        return [];
    }
    Opt() {
        return this.dict.lookupMaybe(PDFName.of('Opt'), PDFString, PDFHexString, PDFArray);
    }
    setOptions(options) {
        const newOpt = new Array(options.length);
        for (let idx = 0, len = options.length; idx < len; idx++) {
            const { value, display } = options[idx];
            newOpt[idx] = this.dict.context.obj([value, display || value]);
        }
        this.dict.set(PDFName.of('Opt'), this.dict.context.obj(newOpt));
    }
    getOptions() {
        const Opt = this.Opt();
        // Not supposed to happen - Opt _should_ always be `PDFArray | undefined`
        if (Opt instanceof PDFString || Opt instanceof PDFHexString) {
            return [{ value: Opt, display: Opt }];
        }
        if (Opt instanceof PDFArray) {
            const res = [];
            for (let idx = 0, len = Opt.size(); idx < len; idx++) {
                const item = Opt.lookup(idx);
                // If `item` is a string, use that as both the export and text value
                if (item instanceof PDFString || item instanceof PDFHexString) {
                    res.push({ value: item, display: item });
                }
                // If `item` is an array of one, treat it the same as just a string,
                // if it's an array of two then `item[0]` is the export value and
                // `item[1]` is the text value
                if (item instanceof PDFArray) {
                    if (item.size() > 0) {
                        const first = item.lookup(0, PDFString, PDFHexString);
                        const second = item.lookupMaybe(1, PDFString, PDFHexString);
                        res.push({ value: first, display: second || first });
                    }
                }
            }
            return res;
        }
        return [];
    }
}
export default PDFAcroChoice;
//# sourceMappingURL=PDFAcroChoice.js.map